/*
*
*				gcurve.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Propagation algorithm for curve hypersurface boundaries in three
*	space dimensions.
*/

#if defined(THREED)

#include <gdecs/gdecs.h>



struct _O_BOND
{
  	BOND 		*old_bond;
 	BOND_TRI 	**_btris;
  	float		*angle;
  	int		*orientation; /*TODO change to type ORIENTATION */
  	int		num_btris;
};
typedef struct _O_BOND O_BOND;

LOCAL  	O_BOND  *make_o_bond(BOND*);
LOCAL  	int     attached_hsbdry_propagate(Front*,POINTER,CURVE*,CURVE*,float);
LOCAL	void	difference3d(float*,float*,float*);
LOCAL	void	vector_scale3d(float*,float);

EXPORT  void	g_curve_propagate_3d(
	Front		*front,
	POINTER 	wave,
	CURVE		*oldc,
	CURVE		*newc,
	float		dt)
{
	DEBUG_ENTER(g_curve_propagate_3d)

        switch (hsbdry_type(oldc))
	{
	case ATTACHED_B_HSBDRY:
	    attached_hsbdry_propagate(front,wave,oldc,newc,dt);
	    break;
	default:
	    (void) printf("oldc = %p ",oldc); 
	    print_hsbdry_type("Unable to process HSBDRY type ",
			      hsbdry_type(oldc),"\n",oldc->interface);
	    break;
	}

	DEBUG_LEAVE(g_curve_propagate_3d)
	return;
} 		/*end g_curve_propagate_3d*/

LOCAL 	int 	attached_hsbdry_propagate(
	Front		*fr,
	POINTER		wave,
	CURVE		*oldc,
	CURVE		*newc,
	float		dt)
{
	O_BOND		*ordered_bond;
	BOND   		*b_old, *b_new;
	BOND_TRI 	*bt, *this_bt, *other_bt;
	TRI		*tri;
	SURFACE   	*surf, *this_surf, *other_surf;
	POINT		*p_old, *p_new;
	COMPONENT	comp;
	int		this_wave, other_wave;
	int		status = FUNCTION_SUCCEEDED;
	int		this_orient, other_orient;
	int    		i, j, b_cnt, num_btris = 0;
	float		V[3];
	Locstate	this_state, other_state;
	Locstate	first_state, last_state;
	size_t		sizest = fr->sizest;
	static Locstate temp_state = NULL;
	DEBUG_ENTER(attached_hsbdry_propagate)

	if (temp_state == NULL)
	    alloc_state(fr->interf,&temp_state,sizest);

	b_old = oldc->first;
	b_new = newc->first;
	b_cnt = 0;

	while (Btris(b_old)[num_btris])
	    num_btris++;

	while (b_old != NULL) 
	{

	    p_old = b_old->start;
	    p_new = b_new->start;

	    for (i = 0; i < num_btris; i++)
	    {
	        bt = Btris(b_old)[i];
		tri = bt->tri;
		surf = bt->surface;

		/* force normal3d to compute new normal	*
		 * for each tri in b_tris list 		*/

		normal_at_point(p_old)[0] = HUGE_VAL; 
		point_propagate(fr,wave,p_old,p_new,tri,surf,dt,V);


		Coords(p_new)[0] = Coords(p_old)[0];
		Coords(p_new)[1] = Coords(p_old)[1];
		Coords(p_new)[2] = Coords(p_old)[2];
	    }


	    ordered_bond = make_o_bond(b_new);

	    for (i = 0; i < num_btris; i++)
	    {
	        this_bt = ordered_bond->_btris[i];
		this_surf = this_bt->surface;
		this_orient = ordered_bond->orientation[i];

		comp = (ordered_bond->orientation[i] == +1) ? 
		    positive_component(this_surf) : 
		    negative_component(this_surf);

		if (is_excluded_comp(comp,this_surf->interface)) 
		{

		    continue;
		}

		j = (i == ordered_bond->num_btris - 1) ? 0 : i+1;
		other_bt = ordered_bond->_btris[j];
		other_surf = other_bt->surface;
		other_orient = ordered_bond->orientation[j];

		this_wave = wave_type(this_surf);
		other_wave = wave_type(other_surf);
		

	        if (this_wave == SUBDOMAIN_BOUNDARY ||
		    other_wave == SUBDOMAIN_BOUNDARY)
		{

		    continue;
		}

		if (this_wave == PASSIVE_BOUNDARY ||
		    other_wave == PASSIVE_BOUNDARY)
		{
		    continue;
		}

		this_state = (this_orient == +1) ?
		    right_start_btri_state(this_bt) :
		    left_start_btri_state(this_bt);
		other_state = (other_orient == +1) ? 
		    left_start_btri_state(other_bt) :
		    right_start_btri_state(other_bt);

		if ((wave_type(this_surf)  >= FIRST_PHYSICS_WAVE_TYPE) &&
		    (wave_type(other_surf) <  FIRST_PHYSICS_WAVE_TYPE))                    
		{


		    assign(other_state,this_state,sizest);
		}

		else if ((wave_type(other_surf) >= FIRST_PHYSICS_WAVE_TYPE) &&
			 (wave_type(this_surf)  <  FIRST_PHYSICS_WAVE_TYPE))
		{


	            assign(this_state,other_state,sizest);
		}

		else
		{
		    interpolate_states(fr,0.5,0.5,Coords(p_new),other_state,
				       Coords(p_new),this_state,temp_state);
		    assign(other_state,temp_state,sizest);
		    assign(this_state,temp_state,sizest);
		}
	    }


            b_cnt++;
	    b_old = b_old->next;
	    b_new = b_new->next;
	}


        for (i = 0; i < num_btris; i++)
	{
	    first_state = right_start_btri_state(Btris(newc->first)[i]);
	    last_state = right_end_btri_state(Btris(newc->last)[i]);
	    assign(last_state,first_state,sizest);

	    first_state = left_start_btri_state(Btris(newc->first)[i]);
	    last_state = left_end_btri_state(Btris(newc->last)[i]);
	    assign(last_state,first_state,sizest);
	}


	DEBUG_LEAVE(attached_hsbdry_propagate)
        return status;
}		/*end attached_hsbdry_propagate*/

LOCAL  	O_BOND 	*make_o_bond(
	BOND		*b)
{
	float		x_axis[3], y_axis[3], z_axis[3];
	float		vector_in_xz_plane[3], vector[3];
	float		*origin = Coords(b->start);
	float		magx, magy;
	float		x_proj, y_proj;
	int		i, j_start, j_end, other;
	int		num_btris = 0;
	int		not_ordered;
	TRI		*tri;
	O_BOND		*ordered_bond;
	static int  	third_point[3][3] = 
	{
	    {-1, 2, 1},
	    { 2,-1, 0},
	    { 1, 0,-1}
	};

	static int 	orient_map[3][3] =
	{
	    { 0, 1,-1},
	    {-1, 0, 1},
	    { 1,-1, 0}
	};

	DEBUG_ENTER(make_o_bond)

	while (Btris(b)[num_btris])
	    num_btris++;

	ordered_bond = (O_BOND*) Store(sizeof(O_BOND));
	ordered_bond->_btris = (BOND_TRI**) Store(num_btris*sizeof(BOND_TRI*));
	ordered_bond->angle = (float*) Store(num_btris*sizeof(float));
	ordered_bond->orientation = (int*) Store(num_btris*sizeof(int));

	for (i = 0; i < num_btris; i++)
	    ordered_bond->_btris[i] = Btris(b)[i];
	ordered_bond->num_btris = num_btris;
	ordered_bond->old_bond = b;

	/* Construct local coordinate system. *
	 * x_axis defined to be zero radians  *
	 *  and in plane of first tri,        *
	 * z_axis defined to be bond b,       *
	 * y_axis = z_axis X x_axis           */
	
	tri = Btris(b)[0]->tri;  
	j_start = j_end = 0;
	
	while (Point_of_tri(tri)[j_start] != b->start && j_start < 3)
	    j_start++;
	while (Point_of_tri(tri)[j_end] != b->end && j_start < 3)
	    j_end++;
	other = third_point[j_start][j_end];

	difference3d(Coords(b->end),origin,z_axis);
	difference3d(Coords(Point_of_tri(tri)[other]),origin,
		     vector_in_xz_plane);
	magy = vector_product(z_axis,vector_in_xz_plane,y_axis,3);
	magx = vector_product(y_axis,z_axis,x_axis,3);
	vector_scale3d(y_axis,(1.0/magy));
	vector_scale3d(x_axis,(1.0/magx));


	for (i = 0; i < num_btris; i++)
	{
	    tri = Btris(b)[i]->tri;
	    j_start = j_end = 0;
	    
	    while (Point_of_tri(tri)[j_start] != b->start && j_start < 3)
	        j_start++;
	    while (Point_of_tri(tri)[j_end] != b->end && j_start < 3)
	        j_end++;
	    ordered_bond->orientation[i] = orient_map[j_start][j_end];
	    other = third_point[j_start][j_end];
	    
	    if (ordered_bond->orientation[i] == 0)
	    {
	        screen("unable to orient tri wrt bond in order_bond_btris()\n");
		clean_up(ERROR);
	    }

	    difference3d(Coords(Point_of_tri(tri)[other]),origin,vector);

	    x_proj = scalar_product(vector,x_axis,3);
	    y_proj = scalar_product(vector,y_axis,3);
      
	    if (sqr(x_proj) + sqr(y_proj) <= .000001) /*TOLERANCE*/
	    {
	        screen("ERROR in order_bond_btris(), degenerate TRI\n");
		print_tri(tri,Btris(b)[i]->surface->interface);
		clean_up(ERROR);
	    }

	    ordered_bond->angle[i] = normalized_angle(atan2(y_proj,x_proj));
	}

	ordered_bond->angle[0] = 0.0;
	not_ordered = YES;
	while (not_ordered)  /* TODO: this should be rewritten using qsort() */
        {
	    not_ordered = NO;
	    for (i = 0; i < num_btris - 1; i++)
	        if (ordered_bond->angle[i+1] < ordered_bond->angle[i])
		{
		    float ftmp;
		    int   itmp;
		    BOND_TRI *btmp;

		    ftmp = ordered_bond->angle[i];
		    btmp = ordered_bond->_btris[i];
		    itmp = ordered_bond->orientation[i];

		    ordered_bond->angle[i] = ordered_bond->angle[i+1];
		    ordered_bond->_btris[i] = ordered_bond->_btris[i+1];
		    ordered_bond->orientation[i] = 
		        ordered_bond->orientation[i+1];

		    ordered_bond->angle[i+1] = ftmp;
		    ordered_bond->_btris[i+1] = btmp;
		    ordered_bond->orientation[i+1] = itmp;

		    not_ordered = YES;
		} 
	}
        DEBUG_LEAVE(make_o_bond)
	return ordered_bond;
}
		/*end make_o_bond*/

LOCAL	void	difference3d(
	float *a,
	float *b,
	float *c)
{
	c[0] = a[0] - b[0];
	c[1] = a[1] - b[1];
	c[2] = a[2] - b[2];
}		/*end difference3d*/

LOCAL	void	vector_scale3d(
	float *a,
	float s)
{
	a[0] *= s;
	a[1] *= s;
	a[2] *= s;
}		/*end vector_scale3d*/


#endif /* defined(THREED) */
