#ifdef MHD
/*
*				muserintfc.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*			User Supplied Operations
*	for gas dynamical specific interface operations for MHD runs.
*/

#include <gdecs/gdecs.h>
#include <ellip/edecs/euserint.h>
#include <ellip/eellip/eellipprotos.h>

	/* LOCAL function declarations */
LOCAL	HYPER_SURF      *g_make_hypersurface(COMPONENT,COMPONENT);
LOCAL	INTERFACE	*g_copy_interface(INTERFACE*);
LOCAL	INTERFACE	*g_receive_interface(int);
LOCAL	void	g_reconstruct_interface_pointers(INTERFACE*,struct Table*,
						 POINTER*,POINTER*);
LOCAL	void	g_reconstruct_point_pointers(POINT*,INTERFACE*,INTERFACE*,
					     POINTER*,POINTER*,int);
LOCAL	void 	g_send_interface(INTERFACE*,int);
LOCAL	void	g_fset_hyper_surf_color(FILE*,HYPER_SURF*);
LOCAL	void	g_user_copy_hyper_surf(HYPER_SURF*,HYPER_SURF*);
LOCAL	void	g_user_fprint_interface(FILE*,INTERFACE*);
LOCAL	void	g_user_make_interface(INTERFACE*);
LOCAL	void	reconstruct_gas_params(Locstate,INTERFACE*,INTERFACE*,POINTER*,
				       POINTER*,int);
LOCAL	void	set_params_list_for_interface(const INTERFACE*);

#if defined(TWOD) || defined(THREED)
LOCAL	CURVE	*m_copy_curve(CURVE*,NODE*,NODE*);
LOCAL	CURVE	*m_make_curve(COMPONENT,COMPONENT,NODE*,NODE*);
LOCAL	NODE	*g_copy_node(NODE*);
LOCAL	NODE	*g_make_node(POINT*);
LOCAL	bool	g_user_join_curves(CURVE*,CURVE*,CURVE*);
LOCAL	bool	g_user_split_curve(int,POINT*,BOND*,CURVE*,CURVE**);
LOCAL	void	g_invert_curve(CURVE*);
LOCAL	void	g_reconstruct_bond_pointers(BOND*,INTERFACE*,INTERFACE*,
					    POINTER*,POINTER*,int);
LOCAL	void	g_reconstruct_curve_pointers(CURVE*,INTERFACE*,INTERFACE*,
					     POINTER*,POINTER*,int);
LOCAL	void	g_reconstruct_node_pointers(NODE*,INTERFACE*,INTERFACE*,
					    POINTER*,POINTER*,int);
LOCAL	void	g_reverse_curve(CURVE*);
LOCAL	void	g_user_fprint_curve(FILE*,CURVE*);
#endif /* defined(TWOD) || defined(THREED) */
#if defined(ONED)
LOCAL	void	g_user_fprint_point(FILE*,POINT*);
#endif /* defined(ONED) */
#if defined(TWOD)
LOCAL	CURVE	*g_attach_curve_to_node(CURVE*,POINT*,BOND*,NODE*);
#endif /* defined(TWOD) */
#if defined(THREED)
LOCAL	SURFACE	*g_copy_surface(SURFACE*,CURVE**,CURVE**,bool);
LOCAL	SURFACE *g_join_surfaces(CURVE*);
LOCAL	void	g_user_fprint_surface(FILE*,SURFACE*);
#endif /* defined(THREED) */


EXPORT	void m_set_interface_hooks(
	int		dim)
{
	I_USER_INTERFACE	*iuh = i_user_hook(dim);
	F_USER_INTERFACE	*fuh = f_user_hook(dim);

	e_set_interface_hooks(dim,0);
	iuh->size_interface = sizeof(G_INTERFACE);
	iuh->size_hyper_surf = sizeof(G_HYPER_SURF);
	iuh->size_curve = sizeof(G_CURVE);
	iuh->size_node = sizeof(G_NODE);
	iuh->_user_make_interface = g_user_make_interface;
	iuh->_copy_interface = g_copy_interface;
	iuh->_receive_interface = g_receive_interface;
	iuh->_user_fprint_interface = g_user_fprint_interface;
	iuh->_Point = e_Point;
	iuh->_send_interface = g_send_interface;
	iuh->_reconstruct_interface_pointers = g_reconstruct_interface_pointers;
	iuh->_reconstruct_point_pointers = g_reconstruct_point_pointers;
	fuh->_wave_type_as_string = g_wave_type_as_string;
	fuh->_fprint_hsbdry_type = g_fprint_hsbdry_type;
	fuh->_fprint_state_data = g_fprint_state_data;
	iuh->_fset_hyper_surf_color = g_fset_hyper_surf_color;
	iuh->_make_hypersurface = g_make_hypersurface;
	iuh->_user_copy_hyper_surf = g_user_copy_hyper_surf;
	iuh->_copy_point = e_copy_point;
	iuh->_Static_point = e_Static_point;
	iuh->_Bond = e_Bond;	
	switch (dim)
	{
#if defined(ONED)
	case 1:
	    iuh->_user_fprint_point = g_user_fprint_point;
	    iuh->_reflect_point = g_reflect_point;
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    iuh->_make_node = g_make_node;
	    iuh->_copy_node = g_copy_node;
	    iuh->_user_fprint_node = g_user_fprint_node;
	    iuh->_make_curve = m_make_curve;
	    iuh->_copy_curve = m_copy_curve;
	    iuh->_user_fprint_curve = g_user_fprint_curve;
	    iuh->_user_split_curve = g_user_split_curve;
	    iuh->_user_join_curves = g_user_join_curves;
	    iuh->_reconstruct_node_pointers = g_reconstruct_node_pointers;
	    iuh->_reconstruct_bond_pointers = g_reconstruct_bond_pointers;
	    iuh->_reconstruct_curve_pointers = g_reconstruct_curve_pointers;
	    iuh->_invert_curve = g_invert_curve;
	    iuh->_reverse_curve = g_reverse_curve;
	    iuh->_attach_curve_to_node = g_attach_curve_to_node;
	    iuh->_reflect_node = g_reflect_node2d;
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    iuh->_make_node = g_make_node;
	    iuh->_copy_node = g_copy_node;
	    iuh->_user_fprint_node = g_user_fprint_node;
	    iuh->_make_curve = m_make_curve;
	    iuh->_copy_curve = m_copy_curve;
	    iuh->_user_fprint_curve = g_user_fprint_curve;
	    iuh->_user_split_curve = g_user_split_curve;
	    iuh->_user_join_curves = g_user_join_curves;
	    iuh->_reconstruct_node_pointers = g_reconstruct_node_pointers;
	    iuh->_reconstruct_bond_pointers = g_reconstruct_bond_pointers;
	    iuh->_reconstruct_curve_pointers = g_reconstruct_curve_pointers;
	    iuh->_invert_curve = g_invert_curve;
	    iuh->_reverse_curve = g_reverse_curve;
	    iuh->_copy_surface = g_copy_surface;
	    iuh->_join_surfaces = g_join_surfaces;
	    iuh->_consistent_interface = g_consistent_interface;
	    iuh->_user_fprint_surface = g_user_fprint_surface;
	    break;
#endif /* defined(THREED) */
	}
}		/*end m_set_interface_hooks*/

LOCAL	void g_user_make_interface(
	INTERFACE	*intfc)
{
	G_USER_INTERFACE *guh = g_user_hook(intfc->dim);

	f_user_make_interface(intfc);

	g_user_interface(intfc) = *guh;
	num_layers(intfc) = 0;
}		/*end g_user_make_interface*/

LOCAL	HYPER_SURF *g_make_hypersurface(
	COMPONENT neg_comp,
	COMPONENT pos_comp)
{
	HYPER_SURF *hs;
	hs = f_make_hypersurface(neg_comp,pos_comp);
	no_slip(hs) = NO;
	adherence_coeff(hs) = 0.0;
	return hs;
}		/*end g_make_hypersurface*/

LOCAL	void g_user_copy_hyper_surf(
	HYPER_SURF	*new_hs,
	HYPER_SURF	*old_hs)
{
	f_user_copy_hyper_surf(new_hs,old_hs);
	no_slip(new_hs) = no_slip(old_hs);
	adherence_coeff(new_hs) = adherence_coeff(old_hs);
}		/*end g_user_copy_hyper_surf*/


LOCAL	INTERFACE *g_receive_interface(
	int		src_id)
{
	set_size_of_intfc_state(g_sizest());
	return i_receive_interface(src_id);
}		/*end g_receive_interface*/

LOCAL	INTERFACE *g_copy_interface(
	INTERFACE	*intfc)
{
	INTERFACE	*new_intfc = f_copy_interface(intfc);

	if (new_intfc == NULL) return new_intfc;

	g_user_interface(new_intfc) = g_user_interface(intfc);
	num_layers(new_intfc) = num_layers(intfc);
	return new_intfc;
}		/*end g_copy_interface*/


LOCAL	void g_user_fprint_interface(
	FILE		*file,
	INTERFACE	*infc)
{
	f_user_fprint_interface(file,infc);
	(void) foutput(file);
	(void) fprintf(file,"INTERFACE TYPE FOR INTERFACE %llu\n",
		       interface_number(infc));
	switch (interface_type(infc))
	{
	case PHYSICAL_INTERFACE:
	    (void) fprintf(file,"Interface type = %d PHYSICAL_INTERFACE\n",
	    	           interface_type(infc));
	    if (infc->hss != NULL)
	    {
	    	g_fprint_Gas_param_list(file,infc);
	    	g_fprint_Dirichlet_bdry_states(file,infc);
	    }
#if defined(TWOD)
	    g_fprint_RP_DATA_at_nodes(file,infc);
#endif /* defined(TWOD) */
	    (void) foutput(file);
	    (void) fprintf(file,"STRATIFIED STATE FUNCTION\n");
	    (void) fprintf(file,"stratified_state = %s\n",
	    	           g_user_interface(infc).stratified_state_name);
#if defined(TWOD)
	    g_fprint_ContactWallNodeParams(file,infc);
#endif /* defined(TWOD) */
	    break;
#if defined(TWOD)
	case EOS_INTERFACE:
	    (void) fprintf(file,"Interface type = %d EOS_INTERFACE\n",
	    	           interface_type(infc));
	    break;
#endif /* defined(TWOD) */
	default:
	    (void) fprintf(file,"Interface type = %d -- ** UNKNOWN **\n",
	    	           interface_type(infc));
	}
}		/*end g_user_fprint_interface*/


LOCAL	void	g_send_interface(
	INTERFACE       *intfc,
	int             dst_id)
{
	Gas_param **sav_list = gas_params_list(intfc);

	set_params_list_for_interface(intfc);
	i_send_interface(intfc,dst_id);
	gas_params_list(intfc) = sav_list;
}


/*
*			set_params_list_for_interface():
*
*	Copy params list into g_user_interface
*/

LOCAL	void	set_params_list_for_interface(
	const INTERFACE	*_intfc)
{
	INTERFACE *intfc=const_cast<INTERFACE*>(_intfc);
	G_USER_INTERFACE	*guh = g_user_hook(intfc->dim);
	int			i;
	int			num_params = guh->num_params;
	Gas_param		**params_list = guh->params_list;
	INTERFACE		*sav_intfc = current_interface();

	num_gas_params(intfc) = num_params;
	if (num_params == 0)
	{
	    gas_params_list(intfc) = NULL;
	    return;
	}

	set_current_interface(intfc);
	gas_params_list(intfc) =
	    (Gas_param**)Store(num_params*sizeof(Gas_param*));
	for (i = 0; i < num_params; ++i)
	    gas_params_list(intfc)[i] = params_list[i];
	set_current_interface(sav_intfc);

}	/*end set_params_list_for_interface*/


LOCAL	void g_reconstruct_interface_pointers(
	INTERFACE	*nintfc,
	struct Table	*otbl,
	POINTER		*ocad,
	POINTER		*ncad)
{
	f_reconstruct_interface_pointers(nintfc,otbl,ocad,ncad);
}		/*end g_reconstruct_interface_pointers*/

LOCAL	void g_reconstruct_point_pointers(
	POINT		*p,
	INTERFACE	*nintfc,
	INTERFACE	*ointfc,
	POINTER		*ocad,
	POINTER		*ncad,
	int		nchks)
{
	f_reconstruct_point_pointers(p,nintfc,ointfc,ocad,ncad,nchks);
#if defined(THREED)
	if ((nintfc->dim == 3) && (sorted(p) == YES)) /*Already done*/
	    return;
#endif /* defined(THREED) */
	if (size_of_state(nintfc) != 0)
	{
	    reconstruct_gas_params(left_state(p),nintfc,ointfc,
				   ocad,ncad,nchks);
	    reconstruct_gas_params(right_state(p),nintfc,ointfc,
				   ocad,ncad,nchks);
#if defined(THREED)
	    if (nintfc->dim == 3)
		sorted(p) = YES;
#endif /* defined(THREED) */
	}
}		/*end g_reconstruct_point_pointers*/



/*ARGSUSED*/
LOCAL	void	reconstruct_gas_params(
	Locstate	state,
	INTERFACE	*nintfc,
	INTERFACE	*ointfc,
	POINTER		*ocad,
	POINTER		*ncad,
	int		nchks)
{
	Gas_param	**new_list, **old_list;
	int		i, num_params;

	debug_print("reconstruct","Entered reconstruct_gas_params()\n");

	if (state == NULL || Params(state) == NULL)
	{
	    debug_print("reconstruct","Left reconstruct_gas_params()\n");
	    return;
	}
	num_params = return_params_list(&new_list);
	old_list = (Gas_param **) new_address(nintfc,gas_params_list(ointfc),
					      ocad,ncad,nchks);

	for (i = 0; i < num_params; ++i)
	{
	    if (Params(state) == old_list[i])
	    {
	        Params(state) = new_list[i];
	        break;
	    }
	}
	debug_print("reconstruct","Left reconstruct_gas_params()\n");
}		/*end reconstruct_gas_params*/

#if defined(TWOD) || defined(THREED)
LOCAL	CURVE* m_make_curve(
	COMPONENT	left_c,
	COMPONENT	right_c,
	NODE		*start,
	NODE		*end)
{
	CURVE		*curve = e_make_curve(left_c,right_c,start,end);

	if (curve == NULL)
	    return curve;

#if defined(TWOD)
	if (curve->interface->dim == 2)
	{
	    INTERFACE *intfc = curve->interface;
	    zero_scalar(left_start_state(curve),size_of_state(intfc));
	    zero_scalar(right_start_state(curve),size_of_state(intfc));
	    zero_scalar(left_end_state(curve),size_of_state(intfc));
	    zero_scalar(right_end_state(curve),size_of_state(intfc));
	    start_status(curve) = ERROR;
	    end_status(curve) = ERROR;
	    layer_index(Hyper_surf(curve)) = 0;
	    surface_tension(curve) = 0.0;
	}
#endif /* defined(TWOD) */
	return curve;
}		/*end m_make_curve*/

LOCAL	CURVE	*m_copy_curve(
	CURVE		*curve,
	NODE		*start,
	NODE		*end)
{
	CURVE		*new_curve = e_copy_curve(curve, start, end);

	if (new_curve == NULL)
	    return new_curve;
#if defined(TWOD)
	if (curve->interface->dim == 2)
	{
	    start_status(new_curve) = start_status(curve);
	    end_status(new_curve) = end_status(curve);
	    surface_tension(new_curve) = surface_tension(curve);
	    layer_index(Hyper_surf(new_curve)) = layer_index(Hyper_surf(curve));
	}
#endif /* defined(TWOD) */
	return new_curve;
}		/*end m_copy_curve*/


/* ARGSUSED */
LOCAL	void g_user_fprint_curve(
	FILE		*file,
	CURVE		*curve)
{
	f_user_fprint_curve(file,curve);
#if defined(TWOD)
	if (curve->interface->dim == 2)
	{
	    fprint_curve_status(file,"\tcurve->start_status = ",
				start_status(curve));
	    fprint_curve_status(file,"\tcurve->end_status = ",
			        end_status(curve));
	    (void) fprintf(file,"\tcurve->surface_tension = ");
	    if (is_binary_output() == YES)
	    {
	    	(void) fprintf(file,"\f%c",1);
	    	(void) fwrite((const void *) &surface_tension(curve),
			      FLOAT,1,file);
		(void) fprintf(file,"\n");
	    }
	    else
	    	(void) fprintf(file,"%"FFMT"\n",surface_tension(curve));
	    (void) fprintf(file,"\tcurve->layer_index = %d\n",
	    		   layer_index(Hyper_surf(curve)));
	    if (wave_type(curve) == NEUMANN_BOUNDARY)
	    {
		(void) fprintf(file,"\tno slip = %s\n",
			       y_or_n(no_slip(Hyper_surf(curve))));
	    	if (no_slip(Hyper_surf(curve)) == YES)
		{
		    (void) fprintf(file,"\tadherence coefficient = ");
		    if (is_binary_output() == YES)
		    {
		    	(void) fprintf(file,"\f%c",1);
			(void) fwrite((const void *) &adherence_coeff(
				Hyper_surf(curve)),FLOAT,1,file);
		    	(void) fprintf(file,"\n");
		    }
		    else
		    	(void) fprintf(file,"%"FFMT"\n",adherence_coeff(
				Hyper_surf(curve)));
		}
	    }
	    (void) fprintf(file,"\n");
	}
#endif /* defined(TWOD) */
}		/*end g_user_fprint_curve*/

LOCAL	bool	g_user_split_curve(
	int		is_a_node,
	POINT		*p,
	BOND		*bond,
	CURVE		*curve,
	CURVE		**curves)
{
	if (f_user_split_curve(is_a_node,p,bond,curve,curves) != YES)
	    return NO;

#if defined(TWOD)
	if (curve->interface->dim == 2)
	{
	    HYPER_SURF	*hs = Hyper_surf(curve);
	    INTERFACE	*intfc = curve->interface;
	    start_status(curves[0]) = start_status(curve);
	    end_status(curves[1]) = end_status(curve);
	    /* Default status at new node */
	    if (wave_type(curve) < FIRST_PHYSICS_WAVE_TYPE)
	    {
	    	end_status(curves[0])   = FIXED;
	    	start_status(curves[1]) = FIXED;
	    }
	    else
	    {
	    	end_status(curves[0])   = INCIDENT;
	    	start_status(curves[1]) = INCIDENT;
	    }
	    surface_tension(curves[0]) = surface_tension(curve);
	    surface_tension(curves[1]) = surface_tension(curve);
	    if (is_subdomain_boundary(hs))
	    {
		size_t sizest = size_of_state(intfc);
	    	obstacle_state(intfc,left_end_state(curves[0]),sizest);
	    	obstacle_state(intfc,right_end_state(curves[0]),sizest);
	    	obstacle_state(intfc,left_start_state(curves[1]),sizest);
	    	obstacle_state(intfc,right_start_state(curves[1]),sizest);
	    }
	    layer_index(Hyper_surf(curves[0])) = layer_index(hs);
	    layer_index(Hyper_surf(curves[1])) = layer_index(hs);
	    return YES;
	}
#endif /* defined(TWOD) */
	return YES;
}		/*end g_user_split_curve*/

LOCAL	bool g_user_join_curves(
	CURVE		*curve,
	CURVE		*curve1,
	CURVE		*curve2)
{
	if (f_user_join_curves(curve,curve1,curve2) != YES)
	    return NO;
#if defined(TWOD)
	if (curve->interface->dim == 2)
	{
	    HYPER_SURF *hs = Hyper_surf(curve);
	    HYPER_SURF *hs1 = Hyper_surf(curve1);
	    HYPER_SURF *hs2 = Hyper_surf(curve2);
	    start_status(curve) = start_status(curve1);
	    end_status(curve) = end_status(curve2);
	    surface_tension(curve) = 0.5*(surface_tension(curve1) +
				          surface_tension(curve2));
	    if (layer_index(hs1) == layer_index(hs2))
	        layer_index(hs) = layer_index(hs1);
	}
#endif /* defined(TWOD) */
	return YES;
}		/*end g_user_join_curves*/

LOCAL	NODE *g_make_node(
	POINT		*posn)
{
	NODE		*newnod = f_make_node(posn);

#if defined(TWOD)
	if (newnod->interface->dim == 2)
	{
	    static ADJUST_ANGLE_DATA dflt_adjust_ang = {
							 0.0,
							 g_adjust_angle_len
						       };

	    size_t sizest = size_of_state(newnod->interface);
	    Rp_data(newnod) = allocate_RP_DATA_structure(sizest,YES,GAS_STATE);
	    adjust_angle(newnod) = dflt_adjust_ang;
	}
#endif /* defined(TWOD) */

	return newnod;
}		/*end g_make_node*/

LOCAL	NODE *g_copy_node(
	NODE		*old_node)
{
	NODE		*new_node = f_copy_node(old_node);

	if (new_node == NULL)
	    return new_node;

#if defined(TWOD)
	if ((new_node->interface->dim == 2) &&
	    (interface_type(new_node->interface) == PHYSICAL_INTERFACE) &&
	    (copy_intfc_states() == YES))
		copy_RP_DATA_structure(Rp_data(new_node),Rp_data(old_node));
	adjust_angle(new_node) = adjust_angle(old_node);
	adjust_len(new_node) = 0.0;
#endif /* defined(TWOD) */
	return new_node;
}		/*end g_copy_node*/


LOCAL	void g_reconstruct_bond_pointers(
	BOND		*b,
	INTERFACE	*nintfc,
	INTERFACE	*ointfc,
	POINTER		*ocad,
	POINTER		*ncad,
	int		nchks)
{
#if defined(THREED)
	BOND_TRI		**btris;
#endif /* defined(THREED) */

	f_reconstruct_bond_pointers(b,nintfc,ointfc,ocad,ncad,nchks);

#if defined(THREED)
	if (size_of_state(nintfc) == 0)
	    return;
	for (btris = Btris(b); btris && *btris; ++btris)
	{
		reconstruct_gas_params(left_start_btri_state(*btris),
				       nintfc,ointfc,ocad,ncad,nchks);
		reconstruct_gas_params(right_start_btri_state(*btris),
				       nintfc,ointfc,ocad,ncad,nchks);
		reconstruct_gas_params(left_end_btri_state(*btris),
				       nintfc,ointfc,ocad,ncad,nchks);
		reconstruct_gas_params(right_end_btri_state(*btris),
				       nintfc,ointfc,ocad,ncad,nchks);
	}
#endif /* defined(THREED) */
}		/*end g_reconstruct_bond_pointers*/

LOCAL	void g_reconstruct_node_pointers(
	NODE		*n,
	INTERFACE	*nintfc,
	INTERFACE	*ointfc,
	POINTER		*ocad,
	POINTER		*ncad,
	int		nchks)
{
	debug_print("reconstruct","Entered g_user_reconstruct_node_pointers()\n");

	f_reconstruct_node_pointers(n,nintfc,ointfc,ocad,ncad,nchks);
#if defined(TWOD)
	if (Rp_data(n) != NULL)
	{
		int i;
		Rp_data(n) = (RP_DATA *) new_address(nintfc,Rp_data(n),ocad,
						     ncad,nchks);

		for (i = 0; i < MAX_N_CURVES; ++i)
		{
			Rp_data(n)->state[i] =
			    (Locstate) new_address(nintfc,Rp_data(n)->state[i],
						   ocad,ncad,nchks);
			reconstruct_gas_params(Rp_data(n)->state[i],
					       nintfc,ointfc,ocad,ncad,nchks);
		}
	}
#endif /* defined(TWOD) */
	debug_print("reconstruct","Left g_user_reconstruct_node_pointers()\n");
}		/*end g_reconstruct_node_pointers*/

LOCAL	void g_reconstruct_curve_pointers(
	CURVE		*c,
	INTERFACE	*nintfc,
	INTERFACE	*ointfc,
	POINTER		*ocad,
	POINTER		*ncad,
	int		nchks)
{
	debug_print("reconstruct","Entered g_reconstruct_curve_pointers()\n");

	f_reconstruct_curve_pointers(c,nintfc,ointfc,ocad,ncad,nchks);
#if defined(TWOD)
	if ((nintfc->dim == 2) && (size_of_state(nintfc) != 0))
	{
		reconstruct_gas_params(left_start_state(c),
				       nintfc,ointfc,ocad,ncad,nchks);
		reconstruct_gas_params(right_start_state(c),
				       nintfc,ointfc,ocad,ncad,nchks);
		reconstruct_gas_params(left_end_state(c),
				       nintfc,ointfc,ocad,ncad,nchks);
		reconstruct_gas_params(right_end_state(c),
				       nintfc,ointfc,ocad,ncad,nchks);
	}
#endif /* defined(TWOD) */
	debug_print("reconstruct","Left g_reconstruct_curve_pointers()\n");
}		/*end g_reconstruct_curve_pointers*/

LOCAL	void	g_invert_curve(
	CURVE		*c)
{
	f_invert_curve(c);
#if defined(TWOD)
	if (c->interface->dim == 2)
	{
	    wave_type(c) = opposite_wave_type(wave_type(c));
	}
#endif /* defined(TWOD) */
}		/*end g_invert_curve*/

LOCAL	void	g_reverse_curve(
	CURVE		*c)
{
	f_reverse_curve(c);
#if defined(TWOD)
	if (c->interface->dim == 2)
	{
	    int status;
	    status = start_status(c);
	    start_status(c) = end_status(c);
	    end_status(c) = status;
	}
#endif /* defined(TWOD) */
}		/*end g_reverse_curve*/
#endif /* defined(TWOD) || defined(THREED) */


#if defined(TWOD)
LOCAL	CURVE *g_attach_curve_to_node(
	CURVE		*c1,
	POINT		*p,
	BOND		*b,
	NODE		*n)
{
	CURVE		*c2;

	c2 = f_attach_curve_to_node(c1,p,b,n);
	node_type(n) = ERROR;
	return c2;
}		/*end g_attach_curve_to_node*/
#endif /* defined(TWOD) */

#if defined(ONED)
LOCAL	void g_user_fprint_point(
	FILE		*file,
	POINT		*point)
{
	if (point->interface != NULL)
	{
	    fprint_wave_type(file,"\t\twave_type = ",wave_type(point),"\n",
	    	                  Hyper_surf(point)->interface);
	    (void) fprintf(file,"\tlayer_index = %d\n",
				layer_index(Hyper_surf(point)));
	}
	(void) fprintf(file,"\t\tLeft state:\n");
	g_fprint_state(file,left_state(point));
	(void) fprintf(file,"\t\tRight state:\n");
	g_fprint_state(file,right_state(point));
}		/*end g_fprint_point*/
#endif /* defined(ONED) */


#if defined(THREED)
LOCAL	SURFACE	*g_copy_surface(
	SURFACE	*surf,
	CURVE	**pos,
	CURVE	**neg,
	bool copy_tris)
{
	SURFACE	*new_surf = f_copy_surface(surf,pos,neg,copy_tris);

	if (new_surf == NULL)
	    return new_surf;

	surface_tension(new_surf) = surface_tension(surf);
	layer_index(Hyper_surf(new_surf)) = layer_index(Hyper_surf(surf));
	return new_surf;
}		/*end g_copy_surface*/

LOCAL	SURFACE *g_join_surfaces(
	CURVE *c)
{
	SURFACE *news, *sn, *sp;
	float   stn, stp;
	int     lin, lip;

	if (!find_surfaces_to_join_at_curve(c,&sn,&sp))
	    return NULL;
	stn = surface_tension(sn);
	stp = surface_tension(sp);
	lin = layer_index(Hyper_surf(sn));
	lip = layer_index(Hyper_surf(sp));
	news = f_join_surfaces(c);
	if (sn != sp)
	{
	    surface_tension(news) = 0.5*(stn+stp);
	    if (lin == lip)
	        layer_index(Hyper_surf(news)) = lin;
	}
	return news;
}		/*end g_join_surfaces*/

LOCAL	void	g_user_fprint_surface(
	FILE		*file,
	SURFACE		*s)
{
	f_user_fprint_surface(file,s);
	if (wave_type(s) == NEUMANN_BOUNDARY)
	{
	    (void) fprintf(file,"\tno slip = %s\n",
			   y_or_n(no_slip(Hyper_surf(s))));
	    if (no_slip(Hyper_surf(s)) == YES)
	    {
		(void) fprintf(file,"\tadherence coefficient = ");
		if (is_binary_output() == YES)
		{
		    (void) fprintf(file,"\f%c",1);
		    (void) fwrite((const void *) &adherence_coeff(
				Hyper_surf(s)),FLOAT,1,file);
		    (void) fprintf(file,"\n");
		}
		else
		    (void) fprintf(file,"%"FFMT"\n",adherence_coeff(
				Hyper_surf(s)));
	    }
	}
}		/*end g_user_fprint_surface*/
#endif /* defined(THREED) */

#include <plotdecs.h>

LOCAL	void	g_fset_hyper_surf_color(
	FILE            *file,
	HYPER_SURF	*hs)
{
	int		w_type;

	w_type = wave_type(hs);
	if (w_type >= FIRST_VECTOR_PHYSICS_WAVE_TYPE)
	{
	    if (is_shock_wave(w_type))
	    	fset_color_from_table(file,5);
	    else if (is_rarefaction_leading_edge(w_type))
	    	fset_color_from_table(file,3);
	    else if (is_rarefaction_trailing_edge(w_type))
	    	fset_color_from_table(file,4);
	}
	else
	    f_fset_hyper_surf_color(file,hs);
}		/*end g_fset_hyper_surf_color*/
#endif
