/*
*
*				gimksurf.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	This file contains functions for the construction of non-trivial
*	geometric curves needed for the initialization of an interface.
*	
*	The following exported functions are contained in this file.
*
*		make_random_surface()
*		g_make_ellipsoid()
*/

#if defined(THREED)

#include <ginit/ginit.h>

LOCAL	float   adjust_for_z_grid_spacing(float,float,float);
LOCAL	float	pert_height(float*,RECT_GRID*,SINE_PERT*,int);
LOCAL	float   spherical_harmonics(double,double,FOURIER_POLY*);
LOCAL   float   random_pert_func(POINTER,float*);
LOCAL	void    perturb_ellipsoid(SURFACE*,ELLIPSOID*);

/*
*		make_random_surface():
*
*	Make_random_surface constructs the initial contact wave
*	for the "random surface" (statistics of fingers) problem.
*
*/

struct _RANDOM_PARAMS
{
        RECT_GRID *gr;
        SINE_PERT *pert;
};
typedef struct _RANDOM_PARAMS RANDOM_PARAMS;

EXPORT void make_random_surface(
	int w_type,
	Front *front,
	SINE_PERT *RS_b,
	COMPONENT compb,
	COMPONENT compa,
	float surf_ten,
	int dim)
{
        INTERFACE          *intfc = front->interf;
        RECT_GRID          *gr = computational_grid(intfc);
        SINE_PERT          *pert = Sine_pert(comp_type(compb));
        SURFACE            *surf;
        RANDOM_PARAMS rand_params;

        rand_params.gr = gr;
        rand_params.pert = pert;
#if !defined(USE_OVERTURE)
        gr = computational_grid(intfc);
#else  /* if !defined(USE_OVERTURE) */
        {
            RECT_GRID fine_grid, *tmpgr;
            int i, refine;
            refine = 1;
            tmpgr = computational_grid(intfc);
            for(i = 1; i < front->NumberOfLevels; i++)
                refine *= 2;  /* should be adjustable */
            for(i = 0; i < tmpgr->dim; i++)
            {
                fine_grid.gmax[i] = refine*tmpgr->gmax[i];
                fine_grid.lbuf[i] = refine*tmpgr->lbuf[i];
                fine_grid.ubuf[i] = refine*tmpgr->ubuf[i];
            }
            set_rect_grid(tmpgr->L, tmpgr->U, tmpgr->GL, tmpgr->GU,
                fine_grid.lbuf, fine_grid.ubuf, fine_grid.gmax,
                tmpgr->dim, &tmpgr->Remap, &fine_grid);
            gr = &fine_grid;
        }
#endif /* if !defined(USE_OVERTURE) */
        make_comp2_surface(gr,compb,compa,random_pert_func,
                        (POINTER)&rand_params,&surf);
        wave_type(surf) = w_type;

        reset_normal_on_intfc(intfc);
        surface_tension(surf) = surf_ten;
        install_subdomain_bdry_curves(intfc);
        reset_intfc_num_points(intfc);
}		/*end make_random_surface*/

LOCAL float random_pert_func(
        POINTER func_params,
        float *coords)
{
        RANDOM_PARAMS *rand_params = (RANDOM_PARAMS*)func_params;
        RECT_GRID *gr = rand_params->gr;
        SINE_PERT *pert = rand_params->pert;
        int dim = gr->dim;
        float dist;

        dist = coords[dim-1] - pert_height(coords,gr,pert,dim);
        return dist;
}       /* end random_pert_func */

EXPORT HYPER_SURF       *g_make_ellipsoid(
	ELLIPSOID       *ellip,
	COMPONENT       compin,
	COMPONENT       compout,
	Front           *front)
{
	SURFACE         *s;
        RECT_GRID       *rgr = front->rect_grid;
	ELLIP_PARAMS	ep;

	ep.cen = ellip->cen;	ep.rad = ellip->rad;
	make_comp2_surface(rgr,compin,compout,ellipsoid_func,
			(POINTER)&ep,&s);
        ellip->hs = Hyper_surf(s);

        perturb_ellipsoid(s,ellip);
        reset_normal_on_intfc(front->interf);
        wave_type(s) = ellip->wv_type;
        surface_tension(s) = ellip->surf_tension;
        layer_index(ellip->hs) = ellip->layer_index;

        install_subdomain_bdry_curves(front->interf);
        reset_intfc_num_points(front->interf);
	untracked_hyper_surf(Hyper_surf(s)) = ellip->untracked;
        return Hyper_surf(s);
}		/* end g_make_ellipsoid */

EXPORT	void	g_make_ellip_region_boundaries3d(
	ELLIPSOID *ellip,
	Front     *front)
{
	bool make_bdry;
	int   i, dim = front->rect_grid->dim;

	if (dim != 3)
	{
	    screen("ERROR in g_make_ellip_region_boundaries3d(), "
		   "invalid dimension %d\n",dim);
	    clean_up(ERROR);
	}

	/* Check for the need to make region boundaries */
	make_bdry = NO;
	for (i = 0; i < 4; i++)
	{
	    if (ellip->btype[i] != UNKNOWN_BOUNDARY_TYPE)
	        make_bdry = YES;
	    if (ellip->obtype[i] != UNKNOWN_BOUNDARY_TYPE)
	        make_bdry = YES;
	}
	if (make_bdry == NO)
	    return;

	screen("ERROR in g_make_ellip_region_boundaries3d(), "
	       "function not implemented\n");
	clean_up(ERROR);
}		/*end g_make_ellip_region_boundaries3d*/

LOCAL	float	pert_height(
	float     *coords,
	RECT_GRID *gr,
	SINE_PERT *pert,
	int       dim)
{
	float *GL = gr->GL;
	float *GU = gr->GU;
	float h = gr->h[dim-1];
	float L = gr->VL[dim-1] + 0.5*h;
	int   i;
	float height;
	float crds[3];

	for (i = 0; i < dim-1; i++)
	{
	    crds[i] = coords[i];
	    if (pert->pert_bdry_type[i] == PERIODIC)
	    {
		while (crds[i] < GL[i])
		    crds[i] += (GU[i] - GL[i]);
		while (crds[i] > GU[i])
		    crds[i] -= (GU[i] - GL[i]);
	    }
	    else if (pert->pert_bdry_type[i] == SYMMETRIC)
	    {
		if (crds[i] < GL[i])
		    crds[i] = 2.0*GL[i] - coords[i];
		if (crds[i] > GU[i])
		    crds[i] = 2.0*GU[i] - coords[i];
	    }
	}
	height = pert_interface(pert,crds,0.0,dim);
	return adjust_for_z_grid_spacing(height,L,h);
}		/*end pert_height */

LOCAL	float adjust_for_z_grid_spacing(
	float height,
	float L,
	float h)
{
	float              k, delta;
	static const float htol = 0.004;/*TOLERANCE*/

	k = floor((height - L)/h);
	delta = (height - (L + k*h))/h;
	if (delta < htol)
	    height = L + (k + htol)*h;
	else if (1.0 - delta < htol)
	    height = L + (k + 1 - htol)*h;
	return height;
}	/* end adjust_for_z_grid_spacing */


LOCAL	void perturb_ellipsoid(
	SURFACE *s,
	ELLIPSOID *ellip)
{
	TRI          *tri;
	POINT        *p;
	int          i, j, k, l;
	float        phi, theta;
	float        x[3], xp[3], rr;
	float	     *cen = ellip->cen, **Q = ellip->Q;
	FOURIER_POLY *fpoly = ellip->fpoly;
	float	     Dr[3];
	float	     er[3];

	if (fpoly == NULL)
	    return;

	for (tri = first_tri(s); !at_end_of_tri_list(tri,s);
			tri = tri->next)
	{
	    for (i = 0; i < 3; i++)
	    {
		p = Point_of_tri(tri)[i];
		sorted(p) = NO;
	    }
	}
	for (tri = first_tri(s); !at_end_of_tri_list(tri,s);
			tri = tri->next)
	{
	    for (i = 0; i < 3; i++)
	    {
		p = Point_of_tri(tri)[i];
		if (sorted(p))
		    continue;
		for (k = 0; k < 3; k++)
		    x[k] = Coords(p)[k] - cen[k];
		for (k = 0; k < 3; k++)
		    for(xp[k] = 0, l = 0; l < 3; l++)
			xp[k] += Q[l][k]*x[l];
		rr = mag_vector(xp,3);
		for (k = 0; k < 3; k++)
		    er[k] = xp[k]/rr;
		theta = atan2(er[1],er[0]);
		phi = acos(er[2]);
		rr += spherical_harmonics(phi,theta,fpoly);
		for (i = 0; i < 3; i++)
		    Dr[i] = rr*er[i];
		for (i = 0; i < 3; i++)
		{
		    Coords(p)[i] = cen[i];
		    for (j = 0; j < 3; j++)
			Coords(p)[i] += Q[i][j]*Dr[j];
		}
		sorted(p) = YES;
	    }
	}
}	/* end perturb_ellipsoid */

LOCAL	float spherical_harmonics(
	double       phi,
	double       theta,
	FOURIER_POLY *fpoly)
{
	int 	k,l,m;
	int 	N = fpoly->num_modes;
	float   *phase = fpoly->phase;
	float   **nu = fpoly->nu;
	float   *A = fpoly->A;
	float 	z = 0.0;

	for (k = 0; k < N; k++)
	{
	    l = irint(nu[k][0]);
	    m = irint(nu[k][1]);
	    z += A[k]*SphericalHarmonic_s(l,m,phi,theta,phase[k]);
	}
	return z;
}	/* end spherical_harmonics */
#endif /* defined(THREED) */
