/*
*				gijet.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Initialized an injection jet.
*/

#if defined(TWOD) || defined(THREED)
#define DEBUG_STRING    "init_jet"
#include <ginit/ginit.h>

	/* LOCAL Function Declarations */
#if defined(TWOD)
LOCAL	float eval_poly(float,const float*,int);
LOCAL	void	init_injection_inlet_jet2d(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	init_fuel_injection_jet2d(INIT_DATA*,INIT_PHYSICS*);
LOCAL	void	insert_elliptical_curve_section(BOND*,CURVE*,float,float,float*,
	                                        float*,float,Front*);
LOCAL	void	insert_linear_section(BOND*,CURVE*,float*,float*,Front*);
LOCAL	void	insert_polynomial_curve_section(BOND*,CURVE*,float,float,
	                                        float*,int,Front*);
#endif /* defined(TWOD) */
#if defined(THREED)
LOCAL	void	init_injection_inlet_jet3d(INIT_DATA*,INIT_PHYSICS*);
LOCAL   void 	map_rectangle_to_circle(float*,float*,float,float,float,float);
LOCAL	void	make_jet_surface(Front*,float,float,int,float,
				 COMPONENT,COMPONENT,COMPONENT);
LOCAL 	void  	reset_surface_sort_status(SURFACE*);
LOCAL 	void  	spherical_surface(SURFACE*,float,float,float*,RECT_GRID*);
LOCAL 	void 	surfaces_on_bonds_of_curve(INTERFACE*);
#endif /* defined(THREED) */

EXPORT  void init_fuel_injection_jet(
        INIT_DATA       *init,
        INIT_PHYSICS    *ip)
{
	switch (ip->root->front->rect_grid->dim)
	{
#if defined(TWOD)
	case 2:
	    init_fuel_injection_jet2d(init,ip);
	    break;
#endif /* defined(TWOD) */
	default:
	    screen("ERROR in init_fuel_injection_jet(), "
		   "unsupported or invalid dimension %d\n",
		   ip->root->front->rect_grid->dim);
	    clean_up(ERROR);
	    break;
	}
}	/* end init_fuel_injection_jet */

/*
*		init_fuel_injection_jet2d():
*
*	Initializes a high pressure chamber for the injection jet.
*
*
*       |           
*       |           
*       |<-n_wu------------------------------>2_________________________
*	|                  _                  /#########################|
*	|                  |                 /##########################|
*	|                 n_hu              /###########################|
*       |                  |               /############################|
*       |                  -              /#############################|
*       |<-n_wm------------------------->/##############################|
*       |                  _             \##############################|
*       |                  |              \#############################|
*	|                 nh_l             \############################|
*       |                  |                \###########################|
*       |                  -                 \##########################|
*	|<-nw_l------------------------------>3          |##############|
*	|                                            _   |##############|
*	|                                            |   |##############|
*	|                                            |   |##############|1
*	|                                           c_h  |##############|  _
*	|                                            |   |##############|  |
*	|                                            |   |##############|  |
*	|                                            -  4|##############|  |
*	|                                           _    |##############|  |
*	|                                           |    |##############|  |
*	|                                           |    |##############|  |
*	|                                           |    |##############|  |
*	|                                          h_i   |##############|  h_o
*	|                                           |    |##############|  |
*	|                                           |    |##############|  |
*	|                                           |    |##############|  |
*	|<-------------------------r_i--------------|--->|##############|  |
*	|                                           |    |##############|  |
*	|                                           |    |##############|  |
*	|                                           |    |<-----w_w---->|  |
*	|                                           |    |##############|  |
*	|___________________________________________-____|##############|  -
*
*	Points 1 and 2, and/or points 3 and 4 may be optionally connected
*	by an elliptical section to create a curved outer (respectively inner)
*	boundary cap.
*/

LOCAL	void init_fuel_injection_jet2d(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Front      *front = ip->root->front;
	INTERFACE  *intfc = front->interf;
	RECT_GRID  *gr = front->rect_grid;
	BOND       *b;
	CURVE      *wall, *inwall, *jet;
	CURVE      **curves;
	POINT      *p;
	NODE       *ns, *ne;
	Gas_param  *paramsa, *paramsb;
	float      cen[2], rad[2];
	float      alpha;
	float      *L = gr->GL;
	float      *U = gr->GU;
	float	   *h = gr->h;
	float	   coords[MAXD];
	float      nor[3];
	float      y0, y1, y0p, y1p, x0, x1, c[4];
	float      theta;
	char 	   s[1024];
	int        i, dim = gr->dim;
	float      r_i, w_w, h_i, h_o, nw_l, c_h, nw_m, nh_l, nw_u, nh_u;
	float      nh_max;
	float      tol = MIN_SC_SEP(intfc);
	bool       crop_x_bdry = NO;

	debug_print("jet","Entered init_fuel_injection_jet2d()\n");

	set_obstacle_comp_type(comp_type(COMPOBST),front);

	screen("\nFuel injection jet enters from below.\n");
	(void) sprintf(s,
            "The jet is initialized as a half jet centered "
            "about r = %g of the computational domain. ",L[0]);
	screen_print_long_string(s);

	i = 0;
	screen("The fuel chamber and nozzle may be entered in "
	       "one of three ways\n"
	       "\t(1) A preset geometry consisting of a fuel chamber,\n"
	       "\t    an elliptical cross section cap, an a Lavel nozzle\n"
	       "\t(2) Direct entry of the chamber wall and nozzle shape\n"
	       "\t(3) Read the coordinates of the chamber wall and nozzle\n"
	       "\t    from a file\n");
	screen("Enter choice (default = %d): ",i);
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    if ((sscanf(s,"%d",&i) != 1) || (i < 1) || (i > 3))
	    {
	        screen("ERROR in init_fuel_injection_jet2d(), "
		       "invalid choice = %d\n",i);
	        clean_up(ERROR);
	    }
	}
	    
	if ((i == 2) || (i == 3))
	{
	    float *x, *y;
	    int   N;

	    screen("The chamber wall is assumed to be oriented so that the\n"
		   "\tright side of the curve bounding the chamber wall is\n"
		   "\tadjacent to the interior of the computational domain.\n");
	    if (i == 2)
	    {
		screen("Enter the number of points on the curve: ");
		if ((Scanf("%d",&N) != 1) || (N <= 0))
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid number of points\n");
	            clean_up(ERROR);
		}
		vector(&x,N,FLOAT);
		vector(&y,N,FLOAT);
		screen("\nEnter %d ordered pairs for the coordinates of the "
		       "chamber wall.\n",N);
		for (i = 0; i < N; ++i)
		{
		    screen("\t: ");
		    if ((Scanf("%f %f\n",x+i,y+i) != 2) ||
			    (x[i] < L[0]) || (y[i] < L[1]) ||
			    (U[0] < x[i]) || (U[1] < y[i]))
		    {
	                screen("ERROR in init_fuel_injection_jet2d(), "
		               "invalid input of coordinate pair\n");
	                clean_up(ERROR);
		    }
		}
	    }
	    else
	    {
	        FILE *file;
		float tmp;
		screen("The point data for the chamber wall should consist of\n"
		       "\tprecisely N pairs of ASCII numbers. The number of\n"
		       "\tpoints in the wall is obtained counting the number\n"
		       "\tpairs in the file\n");
		screen("Enter the file name for the chamber wall data: ");
		(void) Gets(s);
		if ((s[0] == '\0') || ((file = fopen(s,"r")) == NULL))
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "can't open %s\n",s);
	            clean_up(ERROR);
		}
		for (N = 0; (fscan_float(file,&tmp) == 1); ++N);
		if (N%2)
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid (odd) number of points in %s\n",s);
	            clean_up(ERROR);
		}
		N /= 2;
		vector(&x,N,FLOAT);
		vector(&y,N,FLOAT);
		rewind(file);
		for (i = 0; i < N; ++i)
		{
		    (void) fscan_float(file,x+i);
		    (void) fscan_float(file,y+i);
		}
		(void) fclose(file);
	    }
	    coords[0] = x[0];
	    coords[1] = y[0];
	    ns = make_node(Point(coords));
	    set_is_bdry(ns);
	    node_type(ns) = FIXED_NODE;
	    coords[0] = x[N-1];
	    coords[1] = y[N-1];
	    ne = make_node(Point(coords));
	    set_is_bdry(ne);
	    node_type(ne) = FIXED_NODE;
	    wall = make_curve(COMPOBST,COMPA,ns,ne);
	    wave_type(wall) = NEUMANN_BOUNDARY;
	    start_status(wall) = end_status(wall) = FIXED;
	    for (i = 1; i < (N-1); ++i)
	    {
	        coords[0] = x[i];
	        coords[1] = y[i];
		insert_point_in_bond(Point(coords),wall->last,wall);
	    }
	    free_these(2,x,y);

	    screen("Redistribute the input curve (default = yes): ");
	    (void) Gets(s);
	    if (strncasecmp(s,"n",1) != 0)
	    {
		if (!equi_curve_redistribute(front,wall,YES))
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
			   "redistribution of wall boundary failed\n");
		    clean_up(ERROR);
		}
	    }
	}
	else
	{
	    screen("Enter the inner radius of the fuel container: ");
	    (void) Scanf("%f\n",&r_i);
	    if ((r_i <= 0.0) || (U[0] < (L[0] + r_i)))
	    {
	        screen("ERROR in init_fuel_injection_jet2d(), "
		       "invalid inner radius r_i = %"FFMT"\n",r_i);
	        (void) printf("L[0] = %"FFMT", U[0] = %"FFMT"\n",L[0],U[0]);
	        clean_up(ERROR);
	    }

	    screen("Enter the width of the fuel container wall: ");
            (void) Scanf("%f\n",&w_w);
	    if (w_w <= 0.0)
	    {
	        screen("ERROR in init_fuel_injection_jet2d(), "
		       "invalid container width = %"FFMT"\n",w_w);
	        clean_up(ERROR);
	    }
   
	    h_i = 0.0;
	    screen("Enter the height of the interior vertical wall\n"
		   "\tof the fuel chamber (default = %g): ",h_i);
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
	        sscan_float(s,&h_i);
	        if ((h_i < 0.0) || (U[1] < (L[1] + h_i)))
	        {
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid interior wall height h_i = %"FFMT"\n",h_i);
	            (void) printf("L[1] = %"FFMT", U[1] = %"FFMT"\n",L[1],U[1]);
	            clean_up(ERROR);
	        }
	    }

	    h_o = h_i;
	    screen("Enter the height of the exterior vertical wall\n"
		   "\tof the fuel chamber (default = %g): ",h_o);
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
	        sscan_float(s,&h_o);
	        if ((h_o < 0.0) || (U[1] < (L[1] + h_o)))
	        {
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid exterior wall height h_o = %"FFMT"\n",h_o);
	            (void) printf("L[1] = %"FFMT", U[1] = %"FFMT"\n",L[1],U[1]);
	            clean_up(ERROR);
	        }
	    }

	    c_h = 0.0;
	    screen("Enter the height of the cap on the fuel chamber\n"
	           "\t(default or zero = no cap): ");
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
	        if ((sscan_float(s,&c_h) != 1) || (c_h < 0.0) ||
		        (U[1] < (L[1] + h_i + c_h) ))
	        {
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid cap height c_h = %"FFMT"\n",c_h);
	            (void) printf("L[1] = %"FFMT", U[1] = %"FFMT"\n",L[1],U[1]);
	            clean_up(ERROR);
	        }
	    }
	    screen("Enter the width of the lower (inlet) "
	           "section of the nozzle: ");
	    (void) Scanf("%f\n",&nw_l);
	    if ((nw_l <= 0.0) || (r_i < nw_l) || (U[0] < (L[0] + nw_l)))
	    {
	        screen("ERROR in init_fuel_injection_jet2d(), "
		       "invalid lower inlet width nw_l = %"FFMT"\n",nw_l);
	        (void) printf("L[0] = %"FFMT", U[0] = %"FFMT"\n",L[0],U[0]);
	        clean_up(ERROR);
	    }
	    nw_u = nw_m = nw_l;
    
	    nh_l = 0.0;
	    screen("Enter the height of the lower section of the nozzle\n"
	           "\t(default or zero = no lower section): ");
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
	        if ((sscan_float(s,&nh_l) != 1) || (nh_l < 0.0) ||
		        (U[1] < (L[1] + h_i + c_h + nh_l) ))
	        {
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid lower section nozzle "
			   "height nh_l = %"FFMT"\n",nh_l);
	            (void) printf("L[1] = %"FFMT", U[1] = %"FFMT"\n",L[1],U[1]);
	            clean_up(ERROR);
	        }
	    }
	    if (nh_l > 0.0)
	    {
	        screen("Enter the width of the middle inlet of the nozzle\n"
	               "\t(default = %g): ",nw_m);
	        (void) Gets(s);
	        if (s[0] != '\0')
	        {
	            if ((sscan_float(s,&nw_m) != 1) || (nw_m < 0.0) ||
		            ((r_i + w_w) < nw_m))
	            {
	                screen("ERROR in init_fuel_injection_jet2d(), invalid "
			        "middle inlet width nw_m = %"FFMT"\n",
		               nw_m);
	                clean_up(ERROR);
	            }
	        }
	    }

	    nh_u = 0.0;
	    screen("Enter the height of the upper section of the nozzle\n"
	           "\t(default or zero = no upper section): ");
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
	        if ((sscan_float(s,&nh_u) != 1) || (nh_u < 0.0) ||
		        (U[1] < (L[1] + h_i + c_h + nh_l + nh_u) ))
	        {
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid upper section nozzle "
			   "height nh_u = %"FFMT"\n",
		           nh_u);
	            (void) printf("L[1] = %"FFMT", U[1] = %"FFMT"\n",L[1],U[1]);
	            clean_up(ERROR);
	        }
	    }
	    if (nh_u > 0.0)
	    {
	        screen("Enter the width of the upper outlet of the nozzle\n"
	               "\t(default = %g): ",nw_u);
	        (void) Gets(s);
	        if (s[0] != '\0')
	        {
	            if ((sscan_float(s,&nw_u) != 1) || (nw_u < 0.0) ||
		            ((r_i + w_w) < nw_u))
	            {
	                screen("ERROR in init_fuel_injection_jet2d(), "
		               "invalid oulet section "
			       "nozzle width nw_u = %"FFMT"\n",
		               nw_u);
	                clean_up(ERROR);
	            }
	        }
	    }
	    coords[0] = L[0] + r_i + w_w;
	    coords[1] = L[1];
	    ns = make_node(Point(coords));
	    set_is_bdry(ns);
	    node_type(ns) = FIXED_NODE;
	    coords[0] = L[0] + r_i;
	    coords[1] = L[1];
	    ne = make_node(Point(coords));
	    set_is_bdry(ne);
	    node_type(ne) = FIXED_NODE;
	    wall = make_curve(COMPOBST,COMPA,ns,ne);
	    wave_type(wall) = NEUMANN_BOUNDARY;
	    start_status(wall) = end_status(wall) = FIXED;
	    if (h_o > 0.0)
	    {
	        coords[0] = L[0] + r_i + w_w;
	        coords[1] = L[1] + h_o;
		insert_linear_section(wall->last,wall,Coords(ns->posn),
			              coords,front);
	    }
	    if (h_o < (h_i + c_h + nh_l + nh_u))
	    {
		cen[0] = L[0] + nw_u;
		cen[1] = L[1] + h_o;
		rad[0] = w_w + r_i - nw_u;
		rad[1] = h_i + c_h + nh_l + nh_u - h_o;
                insert_elliptical_curve_section(wall->last,wall,0.0,0.5*PI,
			                        cen,rad,0.0,front);
	    }
	    else
	    {
	        coords[0] = L[0] + nw_u;
	        coords[1] = L[1] + h_i + c_h + nh_l + nh_u;
		insert_linear_section(wall->last,wall,Coords(wall->last->start),
			              coords,front);
	    }
	    if (nh_u > 0.0)
	    {
	        coords[0] = L[0] + nw_m;
	        coords[1] = L[1] + h_i + c_h + nh_l;
		insert_linear_section(wall->last,wall,Coords(wall->last->start),
			              coords,front);
		if (nh_l > 0.0)
		{
	            coords[0] = L[0] + nw_l;
	            coords[1] = L[1] + h_i + c_h;
		    insert_linear_section(wall->last,wall,
			                  Coords(wall->last->start),
					  coords,front);
		}
	    }
	    else
	    {
	        coords[0] = L[0] + nw_l;
	        coords[1] = L[1] + h_i + c_h;
		insert_linear_section(wall->last,wall,Coords(wall->last->start),
			              coords,front);
	    }
	    if (c_h > 0.0)
	    {
		cen[0] = L[0] + nw_l;
		cen[1] = L[1] + h_i;
		rad[0] = r_i - nw_l;
		rad[1] = c_h;
                insert_elliptical_curve_section(wall->last,wall,0.5*PI,0.0,
			                        cen,rad,0.0,front);
	    }
	    insert_linear_section(wall->last,wall,Coords(wall->last->start),
			          Coords(wall->last->end),front);
	}
	never_redistribute(wall) = YES;

	/* Clip the nozzle wall at the upper x-boundary if necessary */

	if (U[0] < Coords(wall->start->posn)[0])
	{
	    ORIENTATION orient = POSITIVE_ORIENTATION;
	    
	    for (b = wall->first; b != NULL; b = b->next)
	    {
		if (Coords(b->end)[0] < U[0])
		    break;
	    }
	    if (b == NULL)
	    {
	        screen("ERROR in init_fuel_injection_jet2d(), "
		       "invalid nozzle, all points are out of bounds\n");
	        clean_up(ERROR);
	    }
	    alpha = (U[0] - Coords(b->start)[0])/
		    (Coords(b->end)[0] - Coords(b->start)[0]);
	    coords[0] = U[0];
	    coords[1] = (1.0 - alpha)*Coords(b->start)[1] +
		        alpha*Coords(b->end)[1];
	    ns = wall->start;
	    Coords(ns->posn)[0] = coords[0];
	    Coords(ns->posn)[1] = coords[1];
	    set_bond_length(wall->first,dim);
	    p = b->end;
	    while(Point_adjacent_to_node(wall,orient) != p)
		(void) delete_point_adjacent_to_node(front,wall,orient);
	    crop_x_bdry = YES;
	}

	nh_max = Coords(wall->start->posn)[1];
	for (b = wall->first; b != NULL; b = b->next)
	    if (nh_max < Coords(b->end)[1])
		nh_max = Coords(b->end)[1];

	coords[1] = nh_max;
	screen("Enter the height at which the jet connects "
	       "to the nozzle wall\n\t(default = %g): ",nh_max-L[1]);
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    if ((sscan_float(s,coords+1) != 1) || (nh_max < coords[1]))
	    {
	        screen("ERROR in init_fuel_injection_jet2d(), "
		       "invalid jet height, height %g too high\n",coords[1]);
	        clean_up(ERROR);
	    }
	    coords[1] += L[1];
	}

	for (b = wall->last; b != NULL; b = b->prev)
	{
	    if (Coords(b->start)[1] > nh_max - tol*h[1])
		break;
	}
	alpha = (coords[1] - Coords(b->start)[1])/
	        (Coords(b->end)[1] - Coords(b->start)[1]);
	coords[0] = (1.0 - alpha)*Coords(b->start)[0] + alpha*Coords(b->end)[0];
	if (_scaled_separation(coords,Coords(b->start),h,dim) < tol)
	    p = b->start;
	else if (_scaled_separation(coords,Coords(b->end),h,dim) < tol)
	{
	    p = b->end;
	    if (b->next == NULL)
		p = b->start;
	    else
	        b = b->next;
	}
	else
	{
	    p = Point(coords);
	    insert_point_in_bond(p,b,wall);
	    b = b->next;
	}
	normal(p,Hyper_surf_element(b),Hyper_surf(wall),nor,front);
	curves = split_curve(p,b,wall,COMPOBST,COMPB,COMPOBST,COMPA);
	inwall = curves[1];
	no_slip(Hyper_surf(inwall)) = NO; /*default*/
	screen("The inner injector wall is set as a slip Nuemann ");
	screen("boundary.\nFor no slip Neumann boundary, type 'ns'.\n");
	screen("Enter choice: ");
	(void) Gets(s);
	if (s[0] == 'n' || s[0] == 'N')
	{
	    float adherence_coeff;
	    no_slip(Hyper_surf(inwall)) = YES; 
	    adherence_coeff = 1.0;
	    screen("Enter the adherence coefficient\n");
	    screen("\tadherence_coeff = 0.0 for fully slippery\n");
	    screen("\tadherence_coeff = 1.0 for fully non-slippery\n");
	    screen("\tdefault = %g (non-slippery)\n",adherence_coeff);
	    screen("Enter the value of adherence coefficient: ");
	    (void) Gets(s);
	    sscan_float(s,&adherence_coeff);
	    if (adherence_coeff < 0.0 || adherence_coeff > 1.0)
	    {
		screen("ERROR in init_fuel_injection_jet2d(), "
                       "invalid input of partial slip parameter\n");
                clean_up(ERROR);
	    }
	    if(adherence_coeff == 0.0)
	        no_slip(Hyper_surf(inwall)) = NO; /* turn back to slip */
	    else
	        adherence_coeff(inwall) = adherence_coeff;
	}
	wall = NULL;

	ns = inwall->start;
	node_type(ns) = NEUMANN_NODE;
	screen("Select nozzle-jet node type, choices are\n"
	       "\tNeumann node (n, default)\n"
	       "\tAttached B node (a)\n"
	       "Enter choice: ");
	Gets(s);
	if (s[0] == 'n' || s[0] == 'N')
	    node_type(ns) = NEUMANN_NODE;
	else if (s[0] == 'a' || s[0] == 'A')
	    node_type(ns) = ATTACHED_B_NODE;


	if (nor[0] != 0.0)
	{
	    x1 = Coords(ns->posn)[0];
	    y1 = Coords(ns->posn)[1];
	    y1p = nor[1]/nor[0];

	    screen("Enter the tangent angle of jet surface\n"
		   "\tat the nozzle wall (default = %g): ",
		   degrees(atan2(-nor[1],-nor[0])));
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
		if (sscan_float(s,&theta) != 1)
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid input of jet tangent at nozzle\n");
		    clean_up(ERROR);
		}
		y1p = tan(radians(theta));
	    }

	    x0 = L[0];
	    y0 = y1 - 0.5*y1p*(x1 - x0);
	    y0p = 0.0;

	    screen("Enter the change in height of the jet from the nozzle\n"
		   "\tto the interior wall or symmetry axis (default = %g): ",
		   y0-y1);
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
		if (sscan_float(s,&y0) != 1)
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid input of jet height at interior boundary\n");
		    clean_up(ERROR);
		}
		y0 += y1;
	    }
	    screen("Enter the tangent angle of jet surface at the "
		   "interior boundary\n\t(default = %g): ",atan(y0p));
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
		if (sscan_float(s,&theta) != 1)
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid input of jet tangent at nozzle\n");
		    clean_up(ERROR);
		}
		y0p = tan(radians(theta));
	    }
	    coords[0] = x0;
	    coords[1] = y0;
	    ne = make_node(Point(coords));
	    set_is_bdry(ne);
	    jet = make_curve(COMPA,COMPB,ns,ne);
	    c[0] = ( (x0*x0*x0*y1 - x1*x1*x1*y0) +
		     3.0*x0*x1*(x1*y0-x0*y1) +
		     x0*x1*(x1*x1*y0p - x0*x0*y1p) +
		     x0*x0*x1*x1*(y1p - y0p) ) / ((x0-x1)*(x0-x1)*(x0-x1));
	    c[1] = ( (x0*x0*x0*y1p - x1*x1*x1*y0p) +
		     2.0*x0*x1*(x0*y0p - x1*y1p) +
		     x0*x1*(x0*y1p - x1*y0p)) / ((x0-x1)*(x0-x1)*(x0-x1));
	    c[2] = ( (x1*x1*y1p - x0*x0*y0p) +
		     2.0*(x1*x1*y0p - x0*x0*y1p) +
		     3.0*(x0*y0 - x1*y1) +
		     x0*x1*(y1p - y0p) +
		     3.0*(x1*y0 - x0*y1) ) / ((x0-x1)*(x0-x1)*(x0-x1));
	    c[3] = ( 2.0*(y1 - y0) + (x0 - x1)*(y0p + y1p) ) /
		     ((x0-x1)*(x0-x1)*(x0-x1));
            insert_polynomial_curve_section(jet->first,jet,x1,x0,c,3,front);
	}
	else
	{
	    cen[0] = L[0];
	    cen[1] = Coords(ns->posn)[1];
	    rad[0] = Coords(ns->posn)[0];
	    rad[1] = 2.0*h[1];
	    screen("Enter the change in height of the jet from the nozzle "
		   "to the interior wall or symmetry axis (default = %g): ",
		   rad[1]);
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
		if ((sscan_float(s,rad+1) != 1) || (rad[1] < 0.0))
		{
	            screen("ERROR in init_fuel_injection_jet2d(), "
		           "invalid input of jet height at interior boundary\n");
		    clean_up(ERROR);
		}
	    }
	    coords[0] = cen[0];
	    coords[1] = cen[1] + rad[1];
	    ne = make_node(Point(coords));
	    set_is_bdry(ne);
	    jet = make_curve(COMPA,COMPB,ns,ne);
            insert_elliptical_curve_section(jet->first,jet,0.0,0.5*PI,
			                    cen,rad,0.0,front);
	}
	start_status(jet) = INCIDENT;
	end_status(jet) = INCIDENT;
	wave_type(jet) = CONTACT;

	(void) prompt_for_eos_params(init,ip,YES,"");
	paramsa = prompt_for_eos_params(init,ip,YES,"\n\tfor the injected gas");
	paramsb = prompt_for_eos_params(init,ip,YES,"\n\tfor the ambient gas");
	prompt_for_ambient_state(comp_type(COMPA),paramsa,
                                 " for the injected gas",front,init);
	prompt_for_ambient_state(comp_type(COMPB),paramsb,
                                 " for the ambient gas",front,init);
	surface_tension(jet) =  prompt_for_surface_tension(CONTACT,
						           "for the jet ");

	rect_boundary_type(intfc,1,0) = MIXED_TYPE_BOUNDARY;
	if (crop_x_bdry)
	  rect_boundary_type(intfc,0,1) = MIXED_TYPE_BOUNDARY;
	

	debug_print("jet","Left init_fuel_injection_jet2d()\n");
}	/* end init_fuel_injection_jet2d */

EXPORT	void init_injection_inlet_jet(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	switch (ip->root->front->rect_grid->dim)
	{
#if defined(TWOD)
	case 2:
	    init_injection_inlet_jet2d(init,ip);
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    init_injection_inlet_jet3d(init,ip);
	    break;
#endif /* defined(THREED) */
	default:
	    screen("ERROR in init_injection_inlet_jet(), "
		   "unsupported or invalid dimension %d\n",
		   ip->root->front->rect_grid->dim);
	    clean_up(ERROR);
	    break;
	}
}		/*end init_injection_inlet_jet*/

#if defined(TWOD)

		/* possible types of jet curves */

enum _JET_TYPE {
	FULL_JET = 1,
	HALF_JET = 2
};
typedef enum _JET_TYPE JET_TYPE;


LOCAL	void init_injection_inlet_jet2d(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	COMPONENT	ecomp;
	CURVE		*cur, *inlet;
	ELLIPSOID	Ellip;
	HYPER_SURF	*jet_hs;
	Front		*front = ip->root->front;
	INTERFACE	*intfc = front->interf;
	Locstate	si;
	NODE		*ns, *ne;
	NODE		*ins, *ine;
	RECT_GRID	*gr = front->rect_grid;
	char		s[1024];
	float		A, beta, beta_max, v[MAXD];
	float		theta0;
	float		iradius, imid, ilower, iupper;
	float		jradius, jmid, jlower, jupper;
	float		*L = front->rect_grid->GL, *U = front->rect_grid->GU;
	float		coords[3];
	float		ci;
	JET_TYPE	jet;
	bool		attached_jet;
	int		i, dim = intfc->dim;
	Gas_param	*params, *params2;

	debug_print("jet","Entered init_injection_inlet_jet2d()\n");

	if (dim != 2)
	{
	    screen("ERROR in init_injection_inlet_jet2d(), "
		   "unsupported dim %d\n",dim);
	    clean_up(ERROR);
	    debug_print("jet","Left init_injection_inlet_jet2d()\n");
	    return;
	}

	iradius = 0.1*(U[0] - L[0]);
	imid = 0.5*(U[0] + L[0]);
	switch (gr->Remap.remap)
	{
	case IDENTITY_REMAP:
	    jet = FULL_JET;
	    ilower = imid - iradius;
	    iupper = imid + iradius;
	    break;
	case CYLINDRICAL_REMAP:
	    jet = HALF_JET;
	    ilower = RL_eff(init);
	    iupper = ilower + 2.0*iradius;
	    break;
	default:
	    screen("ERROR in init_injection_inlet_jet2d(), "
		   "unsupported remap\n");
	    clean_up(ERROR);
	    debug_print("jet","Left init_injection_inlet_jet2d()\n");
	    return;
	}
	imid = 0.5*(ilower + iupper);

	screen("\nJet enters from below.\n");
	(void) sprintf(s,
	    "The jet may be initialized as either a half jet centered "
	    "about %s = %g, or a full inlet along the lower boundary "
	    "of the computational domain\n",
	    (gr->Remap.remap == CYLINDRICAL_REMAP) ? "r" : "x",L[0]);
	screen_print_long_string(s);
	screen("Enter the jet type (dflt = %s): ",
	       (jet == FULL_JET) ? "full" : "half");
	(void) Gets(s);
	if ((s[0] == 'F') || (s[0] == 'f'))
	    jet = FULL_JET;
	else if ((s[0] == 'H') || (s[0] == 'h'))
	    jet = HALF_JET;
	switch (jet)
	{
	case FULL_JET:
	    screen("Enter the lower coordinate of the inlet (dflt = %g): ",
		   ilower);
	    (void) Gets(s);
	    if (s[0] != '\0')
		(void) sscan_float(s,&ilower);
	    screen("Enter the upper coordinate of the inlet (dflt = %g): ",
		   iupper);
	    (void) Gets(s);
	    if (s[0] != '\0')
		(void) sscan_float(s,&iupper);
	    if (iupper < ilower)
	    {
	        screen("ERROR in init_injection_inlet_jet2d(), invalid "
		       "input, upper coords less than lower coords\n");
	        clean_up(ERROR);
	        debug_print("jet","Left init_injection_inlet_jet2d()\n");
	        return;
	    }
	    imid   = 0.5*(ilower + iupper);
	    iradius = 0.5*(iupper - ilower);
	    break;

	case HALF_JET:
	    screen("Enter the radius of the inlet (dflt = %g): ",iradius);
	    (void) Gets(s);
	    if (s[0] != '\0')
		(void) sscan_float(s,&iradius);
	    iupper = ilower + iradius;
	    imid = ilower;
	    break;
	}

	attached_jet = YES;
	screen("Do you wish the jet to remain attached to the inlet edge? "
	       "(dflt = %s): ",y_or_n(attached_jet));
	Gets(s);
	if (s[0] == 'y' || s[0] == 'Y')
	    attached_jet = YES;
	if (s[0] == 'n' || s[0] == 'N')
	    attached_jet = NO;

	jradius = iradius;
	jlower = ilower;
	jupper = iupper;
	jmid = imid;
	if (attached_jet == NO)
	{
	    switch (jet)
	    {
	    case FULL_JET:
	        screen("Enter the lower coordinate (<= %g) of the "
		       "jet (dflt = %g): ",ilower,jlower);
	        (void) Gets(s);
	        if (s[0] != '\0')
		    (void) sscan_float(s,&jlower);
		if (jlower > ilower)
		{
	            screen("ERROR in init_injection_inlet_jet2d(), invalid "
		           "input, left jet edge greater "
			   "than left inlet edge\n");
	            clean_up(ERROR);
		}
	        screen("Enter the upper coordinate (>= %g) of the "
		       "jet (dflt = %g): ",iupper,jupper);
	        (void) Gets(s);
	        if (s[0] != '\0')
		    (void) sscan_float(s,&jupper);
	        if (jupper < iupper)
	        {
	            screen("ERROR in init_injection_inlet_jet2d(), invalid "
		           "input, right jet edge less "
			   "than right inlet edge\n");
	            clean_up(ERROR);
	            debug_print("jet","Left init_injection_inlet_jet2d()\n");
	            return;
	        }
	        jmid   = 0.5*(jlower + jupper);
	        jradius = 0.5*(jupper - jlower);
	        break;

	    case HALF_JET:
	        screen("Enter the radius the jet (>= %g) (dflt = %g): ",
		       iradius,jradius);
	        (void) Gets(s);
	        if (s[0] != '\0')
		    (void) sscan_float(s,&jradius);
	        if (jradius < iradius)
	        {
	            screen("ERROR in init_injection_inlet_jet2d(), invalid "
		           "input, jet radius less "
			   "than inlet radius\n");
	            clean_up(ERROR);
	            debug_print("jet","Left init_injection_inlet_jet2d()\n");
	            return;
	        }
	        jupper = jlower + jradius;
	        jmid = jlower;
	        break;
	    }
	}

	set_default_ellipsoid_structure(&Ellip,dim);
	Ellip.cen[0] = jmid;
	Ellip.cen[1] = L[1];
	Ellip.rad[0] = jradius;
	Ellip.rad[1] = A = jradius;;
	Ellip.ThetaS[0] = 0.0;
	Ellip.ThetaE[0] = (jet == FULL_JET) ? PI : 0.5*PI;
	Ellip.closed = NO;
	Ellip.nor_orient = POSITIVE_ORIENTATION;
	Ellip.compin = COMPB;
	Ellip.compout = COMPA;
	Ellip.fpoly = NULL;
	Ellip.hs = NULL;
	Ellip.surf_tension = 0.0;
	Ellip.wv_type = CONTACT;
	Ellip.dim = dim;
	Ellip.untracked = NO;
	Ellip.layer_index = ++num_layers(intfc);

	screen("Enter the initial height of the jet (dflt = %g): ",A);
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscan_float(s,&A);
	beta_max = atan(jradius/A);
	beta = 0.0;
	screen("Enter the angle in degrees that the jet makes with the "
	       "vertical on\n\t"
	       "its right hand side, the absolute value of "
	       "this angle must be less than\n\t"
	       "%g degrees, (dflt = %g): ",degrees(beta_max),degrees(beta));
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscan_float(s,&beta);
	beta = radians(beta);
	if (beta > beta_max)
	{
	    screen("ERROR in init_injection_inlet_jet2d(), invalid "
		   "input, angle of jet edge with vertical too large\n");
	    clean_up(ERROR);
	    debug_print("jet","Left init_injection_inlet_jet2d()\n");
	    return;
	}
	theta0 = asin((A/jradius)*tan(beta));
	Ellip.rad[0] = jradius/cos(theta0);
	Ellip.rad[1] = A/(1.0 - sin(theta0));
	Ellip.cen[1] = L[1] - Ellip.rad[1]*sin(theta0);
	Ellip.ThetaS[0] = theta0;
	Ellip.ThetaE[0] = (jet == FULL_JET) ? PI - theta0 : 0.5*PI;

	(void) prompt_for_eos_params(init,ip,YES,"");
	params = prompt_for_eos_params(init,ip,YES,"\n\tfor the ambient gas");
	params2 = prompt_for_eos_params(init,ip,YES,"\n\tfor the injected gas");
	prompt_for_ambient_state(comp_type(COMPA),params,
		                 " for the ambient gas",front,init);
	prompt_for_ambient_state(comp_type(COMPB),params2,
		                 " for the injected gas",front,init);
	si = Ambient(comp_type(COMPB));
	for (i = 0; i < dim; ++i)
	    v[i] = vel(i,si);
	if ((gr->Remap.remap == CYLINDRICAL_REMAP) && (jet == HALF_JET))
	{
	    if (v[0] != 0.0)
	    {
		screen("ERROR in init_injection_inlet_jet2d(), "
		       "a cylindrical half jet with vr != 0\n");
		clean_up(ERROR);
	    }
	}
	if (v[1] <= 0.0)
	{
	    (void) printf("WARNING in init_injection_inlet_jet2d(), "
			  "negative injection velocity\n");
	}
	ci = sound_speed(si);
	if (ci > fabs(v[1]))
	{
	    (void) printf("WARNING in init_injection_inlet_jet2d(), "
			  "subsonic flow at inlet\n");
	}

	Ellip.surf_tension = prompt_for_surface_tension(CONTACT,"for the jet ");

	if (debugging("jet"))
	{
	    (void) printf("Ellip\n");
	    print_ellipsoid(&Ellip,intfc);
	}
	jet_hs = make_ellipsoid(&Ellip,COMPB,COMPA,front);
	set_is_bdry(Curve_of_hs(jet_hs)->start);
	set_is_bdry(Curve_of_hs(jet_hs)->end);
	ecomp = exterior_component(intfc);
	if (attached_jet == YES)
	{
	    if (jet == FULL_JET)
	    {
	        ins = Curve_of_hs(jet_hs)->start;
		ine = Curve_of_hs(jet_hs)->end;
		node_type(ins) = node_type(ine) = ATTACHED_B_NODE;
	    }
	    else
	    {
		node_type(Curve_of_hs(jet_hs)->start) = ATTACHED_B_NODE;
		node_type(Curve_of_hs(jet_hs)->end) = NEUMANN_NODE;
	        ins = Curve_of_hs(jet_hs)->start;
	        coords[0] = ilower;
	        coords[1] = L[1];
	        ine = make_node(Point(coords));
	        node_type(ine) = FIXED_NODE;
	        set_is_bdry(ine);
	    }
	}
	else
	{
	    node_type(Curve_of_hs(jet_hs)->start) = NEUMANN_NODE;
	    node_type(Curve_of_hs(jet_hs)->end) = NEUMANN_NODE;

	    coords[0] = iupper;
	    coords[1] = L[1];
	    ins = make_node(Point(coords));
	    set_is_bdry(ins);
	    node_type(ins) = FIXED_NODE;

	    coords[0] = ilower;
	    coords[1] = L[1];
	    ine = make_node(Point(coords));
	    set_is_bdry(ine);
	    node_type(ine) = FIXED_NODE;

	    ns = Curve_of_hs(jet_hs)->start;
	    ne = ins;
	    cur = make_curve(ecomp,Ellip.compin,ns,ne);
	    set_is_bdry(cur);
	    start_status(cur) = end_status(cur) = FIXED;
	    wave_type(cur) = NEUMANN_BOUNDARY;

	    if (jet == FULL_JET)
	    {
	        ns = ine;
	        ne = Curve_of_hs(jet_hs)->end;
	        cur = make_curve(ecomp,Ellip.compin,ns,ne);
	        set_is_bdry(cur);
	        start_status(cur) = end_status(cur) = FIXED;
	        wave_type(cur) = NEUMANN_BOUNDARY;

	        ns = Curve_of_hs(jet_hs)->end;
	        coords[0] = RL_eff(init);
	        coords[1] = L[1];
	        ne = make_node(Point(coords));
	        set_is_bdry(ne);
	        node_type(ns) = FIXED_NODE;
	        cur = make_curve(ecomp,Ellip.compout,ns,ne);
	        set_is_bdry(cur);
	        start_status(cur) = end_status(cur) = FIXED;
	        wave_type(cur) = NEUMANN_BOUNDARY;
	    }
	}

	coords[0] = U[0];
	coords[1] = L[1];
	ns = make_node(Point(coords));
	set_is_bdry(ns);
	node_type(ns) = FIXED_NODE;
	ne = Curve_of_hs(jet_hs)->start;
	cur = make_curve(ecomp,Ellip.compout,ns,ne);
	set_is_bdry(cur);
	start_status(cur) = end_status(cur) = FIXED;
	wave_type(cur) = NEUMANN_BOUNDARY;

	inlet = make_curve(ecomp,Ellip.compin,ins,ine);
	set_is_bdry(inlet);
	wave_type(inlet) = DIRICHLET_BOUNDARY;
	bstate_index(inlet) = prompt_for_boundary_state(wave_type(inlet),
							"inlet",NULL,
						        Ellip.compin,
						        -1,Hyper_surf(inlet),
						        init,ip);

	rect_boundary_type(intfc,1,0) = MIXED_TYPE_BOUNDARY;
	if (debugging("jet"))
	{
	    (void) printf("Interface after init_injection_inlet_jet2d()\n");
	    print_interface(intfc);
	}
	debug_print("jet","Left init_injection_inlet_jet2d()\n");
}		/*end init_injection_inlet_jet2d*/
#endif /* defined(TWOD) */

#if defined(THREED)


LOCAL	void init_injection_inlet_jet3d(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	Front	  *front = ip->root->front;
	float	  radius, height;
	float	  surf_ten;
	Gas_param *params, *params2;
	COMPONENT JET_EXTERIOR = exterior_component(front->interf);
	COMPONENT JET_INLET    = FIRST_DYNAMIC_COMPONENT;
	COMPONENT JET_AMBIENT  = JET_INLET + 1;
	DEBUG_ENTER(init_injection_inlet_jet3d)

	screen("\nJet enters from below.\n");
	screen("\tInitial jet surface is semi-ellipsoid\n"
	       "\tEnter the height of the initial jet perturbation: ");
	(void) Scanf("%f\n",&height);
	screen("\tEnter the radius of the jet inlet orifice: ");
	(void) Scanf("%f\n",&radius);  

	(void) prompt_for_eos_params(init,ip,YES,"");
	params = prompt_for_eos_params(init,ip,YES,"\n\tfor the ambient gas");
	params2 = prompt_for_eos_params(init,ip,YES,"\n\tfor the injected gas");
	prompt_for_ambient_state(comp_type(JET_AMBIENT),params,
				 " for the ambient gas",front,init);
	prompt_for_ambient_state(comp_type(JET_INLET),params2,
				 " for the jet",front,init);

	surf_ten = prompt_for_surface_tension(CONTACT,"for the jet ");
	make_jet_surface(front,height,radius,BOTTOM,surf_ten,
			 JET_EXTERIOR,JET_INLET,JET_AMBIENT);

	DEBUG_LEAVE(init_injection_inlet_jet3d)
}		/*end init_injection_inlet_jet3d*/

/*
*		make_jet_surface():
*
*	make_jet_surface constructs the initial contact wave
*	for an injection jet.
*
*       Author: J.D. Pinezich 9/3/98
*	Rewritten by: X. L. Li 4/4/03
*
*
*/

LOCAL 	void 	make_jet_surface( 
     	Front	     *front,
	float	     height,
	float 	     radius,
	int	     top_or_bottom,
	float 	     surf_ten,
	COMPONENT    JET_EXTERIOR,
	COMPONENT    JET_INLET,
	COMPONENT    JET_AMBIENT)
{
	BOND	   *b;
	INTERFACE  *intfc = front->interf;
	NODE	   *nodes[8];
	CURVE 	   *curves[8];
	CURVE      *inlet_boundary;
	POINT	   *pmin,*pmel,*pmeu,*pmax;
	POINT 	   *p;
	SURFACE	   *s[3], *surf_annulus, *surf_orifice, *surf_cap;
	SURFACE	   **surfs;
	float	   coords[MAXD], midpoint[MAXD];
	float	   x_o[2],y_o[2],x_i[2],y_i[2];
	float	   edge;
	float      X0, Y0, X, Y, l0, lmax;
	int	   i, j, dim = intfc->dim;
	const int  index[] = { 0, 1, 3, 2 };
	RECT_GRID  *comp_grid = front->rect_grid;
	RECT_GRID  dual_grid;
	float	   *dual_VL, *dual_VU;
	float      *L = comp_grid->GL, *U = comp_grid->GU;
	float      *h = comp_grid->h, hx = h[0], hy = h[1];
	int	   *dual_gmax,i_edge[2];
	bool	   odd_gmax[2] = {YES,YES};
	PLANE_PARAMS plane_params;
	ELLIP_PARAMS ellipsoid_params;
	float N[3],P[3],cen[3],rad[3];
	DEBUG_ENTER(make_jet_surface)

	if (top_or_bottom == BOTTOM)
	{
	    midpoint[2] = comp_grid->L[2];
	    height = fabs(height);
	}
	else
	{
	    midpoint[2]  = comp_grid->U[2];
	    height = -fabs(height);
	}

	/* make x and y coordinates */

	/*
	*  	y_o[1] N3------------c2------------N2    
	*		|                          |    
	*               |                          |    
	*       y_i[1] 	|       N7---c6----N6      |    
	*		|        |         |       |    
	*              c3       c7         c5      c1    
	*               |        |         |       |    
	*       y_i[0]  |       N4---c4----N5      |    
	*               |                          |    
	*               |                          |    
	*       y_o[0] N0------------c0------------N1   
	*                                               
	*            x_o[0]   x_i[0]     x_i[1]  x_o[1] 
	*/

	if (debugging("mk_jet_surf"))
	{
	    (void) printf("comp_grid:\n");
	    print_RECT_GRID_structure(comp_grid);
	}


	/* make surfaces using components :    		  *
	 *                                                *
	 *                                                *
	 *                   JET_AMBIENT                  *
	 *                                                *
	 *                        /\                      *
	 *                       /||\                     *
	 *                        ||                      *
	 *                        ||                      *
	 *                                                *
	 *                     -------                    *
	 *                   /         \                  *
	 *                  /           \                 *
	 *                 /             \                *
	 *                |   JET_INLET   |               *
	 *                |               |               *
	 *   -------------------------------------------- *
	 *                                                *
	 *                  JET_EXTERIOR                  *
	 *                                                */


	N[0] = N[1] = 0,0; 	N[2] = 1.0;
	rad[0] = rad[1] = radius;
	rad[2] = height;
	for (i = 0; i < 3; ++i)	
	{
	    P[i] = comp_grid->L[i];
	    cen[i] = comp_grid->L[i] + 0.5*(comp_grid->U[i] 
	    			- comp_grid->L[i]);
	}
	plane_params.N = N;
	plane_params.P = P;
	ellipsoid_params.cen = cen;
	ellipsoid_params.rad = rad;

	/*TMP*/
	printf("Calling make_comp3_surfaces()\n");
	printf("N = %f %f %f\n",N[0],N[1],N[2]);
	printf("P = %f %f %f\n",P[0],P[1],P[2]);
	printf("cen = %f %f %f\n",cen[0],cen[1],cen[2]);
	printf("rad = %f %f %f\n",rad[0],rad[1],rad[2]);
	make_comp3_surfaces(comp_grid,JET_EXTERIOR,JET_INLET,JET_AMBIENT,
			plane_func,(POINTER)&plane_params,
			ellipsoid_func,(POINTER)&ellipsoid_params,
			&surfs,&inlet_boundary);
	/*
	surf_annulus = make_surface(JET_EXTERIOR,JET_AMBIENT,NULL,NULL);
	wave_type(surf_annulus) = NEUMANN_BOUNDARY;
	set_is_bdry(surf_annulus);

	for (i = 0; i < 4; ++i)
	    install_curve_in_surface_bdry(surf_annulus,curves[i],
					  POSITIVE_ORIENTATION);
	for (i = 4; i < 8; ++i)
	    install_curve_in_surface_bdry(surf_annulus,curves[i],
					  NEGATIVE_ORIENTATION);

	pmin = nodes[0]->posn;
	pmel = nodes[4]->posn;
	pmeu = nodes[6]->posn;
	pmax = nodes[2]->posn;
	planar_hole_surface_triangulation(surf_annulus,front->rect_grid,
					  pmin,pmel,pmeu,pmax);

	for (i = 0; i < 4; ++i)
	{
	    set_bdry_side(Boundary(curves[i]),0,top_or_bottom);
	    set_bdry_side(Boundary(curves[i]),1,top_or_bottom);
	}
	  
	surf_cap = make_surface(JET_INLET,JET_AMBIENT,NULL,NULL);
	wave_type(surf_cap) = CONTACT;
	surface_tension(surf_cap) = surf_ten;

	for (i = 4; i < 8; ++i)
	    install_curve_in_surface_bdry(surf_cap,curves[i],
					  POSITIVE_ORIENTATION);

	oblique_planar_surface_triangulation(surf_cap,front->rect_grid);

	surf_orifice = make_surface(JET_EXTERIOR,JET_INLET,NULL,NULL);
	*/
	set_is_bdry(surf_orifice);

	for (i = 4; i < 8; ++i)
	    install_curve_in_surface_bdry(surf_orifice,curves[i],
					  POSITIVE_ORIENTATION);

	oblique_planar_surface_triangulation(surf_orifice,front->rect_grid);

	if (debugging("mk_jet_surf"))
	    summarize_interface("mk_jet_surf","square_hole",intfc,
				XY_PLANE,"make_jet_surface","square_hole");

	/*
	*  Remap square regions to circular regions correspond to the
	*  desired geometry.
	*/

	for (i = 5; i < 8; ++i)
	{
	    CURVE *curve1, *curve2;

	    curve1 = nodes[i]->in_curves[0];
	    curve2 = nodes[i]->out_curves[0];
	    inlet_boundary = join_curves(curve1,curve2,NO_COMP,NO_COMP,NULL);
	    if (inlet_boundary == NULL)
	    {
		screen("ERROR in make_jet_surface(), join_curves failed\n");
		clean_up(ERROR);
	    }
	    (void) delete_node(nodes[i]);
	    nodes[i] = NULL;
	}
	for (i = 4; i < 8; ++i)
	    curves[i] = NULL;

	s[0] = surf_annulus;
	s[1] = surf_orifice;
	s[2] = surf_cap;
	for (i = 0; i < 3; ++i)
	    reset_surface_sort_status(s[i]);

	p = inlet_boundary->first->start;
	map_rectangle_to_circle(Coords(p),midpoint,X0,Y0,radius,lmax);
	sorted(p) = YES;
	for (b = inlet_boundary->first; b != inlet_boundary->last; b = b->next)
	{
	    p = b->end;
	    map_rectangle_to_circle(Coords(p),midpoint,X0,Y0,radius,lmax);
	    set_bond_length(b,dim);
	    sorted(p) = YES;
	}
	set_bond_length(inlet_boundary->last,dim);

	for (i = 0; i < 3; ++i)
	{
	    TRI *t;
	    for (t = first_tri(s[i]); !at_end_of_tri_list(t,s[i]);
		 t = t->next)
	    {
	        for (j = 0; j < 3; ++j)
	        {
		    p = Point_of_tri(t)[j];
	            if (sorted(p) == NO)
		    {
	                map_rectangle_to_circle(Coords(p),midpoint,X0,Y0,
						radius,lmax);
	                sorted(p) = YES;
		    }
	        }
	    }
	}
        reset_surface_sort_status(surf_cap);
	spherical_surface(surf_cap,height,radius,midpoint,comp_grid);
	reset_normal_on_intfc(intfc);

	interface_reconstructed(intfc) = NO;

	if (debugging("mk_jet_surf"))
	    summarize_interface("mk_jet_surf","round_hole",intfc,
				XY_PLANE,"make_jet_surface","round_hole");

	if (debugging("mk_jet_surf"))
	    surfaces_on_bonds_of_curve(intfc);

	if (top_or_bottom == BOTTOM)
            rect_boundary_type(intfc,2,0) = MIXED_TYPE_BOUNDARY;
	else
            rect_boundary_type(intfc,2,1) = MIXED_TYPE_BOUNDARY;

	if (!i_consistent_interface(intfc))
	{
	    screen("ERROR in make_jet_surface(), inconsistent interface\n");
	    clean_up(ERROR);
	}

	if (debugging("mk_jet_surf"))
	    points_of_interface(intfc);

	DEBUG_LEAVE(make_jet_surface)
}		/*end make_jet_surface*/

LOCAL   void 	spherical_surface(
	SURFACE   *s,
	float 	  height,
	float 	  r_circle,
	float 	  *m,
	RECT_GRID *comp_grid)
{
	TRI         *t;
	POINT       *p;
	int         j;
	float       x, y, r;
	float       *h = comp_grid->h;
	float       rtol, ztol;
	static const float tolerance = 0.01; /*TOLERANCE*/

	rtol = tolerance*hypot(h[0],h[1]);
	ztol = tolerance*h[2];
	for (t = first_tri(s); !at_end_of_tri_list(t,s); t = t->next)
	{
	    for (j = 0; j < 3; ++j)
	    {
		p = Point_of_tri(t)[j];
	        if (sorted(p) == NO)
		{
		    x = Coords(p)[0];
		    y = Coords(p)[1];
		    r = hypot(x-m[0],y-m[1]);
		    if (fabs(r - r_circle) < rtol)
		        Coords(p)[2] = m[2];
		    else
		    {
			float ratio = r/r_circle;
			Coords(p)[2] = (ratio < 1.0) ?
		                       height*sqrt(1.0 - sqr(ratio)) : m[2];
			if (fabs(Coords(p)[2] - m[2]) < ztol)
			    Coords(p)[2] = m[2];
		    }
		    sorted(p) = YES;
		}
	    }
	}
} 		/*end spherical_surface*/


LOCAL   void 	reset_surface_sort_status(
	SURFACE		*s)
{
	TRI *t;

	for (t = first_tri(s); !at_end_of_tri_list(t,s); t = t->next)
	{
	    sorted(Point_of_tri(t)[0]) = NO;
	    sorted(Point_of_tri(t)[1]) = NO;
	    sorted(Point_of_tri(t)[2]) = NO;
	}
}       	/*end reset_surface_sort_status*/

/*                                                                      *
 *	int   surfaces_on_bonds_of_curve()                              *
 *                                                                      *
 *      loop through all bonds in interface                             *
 *      printout surfaces of curve_of_bond and surfaces of bond         * 
 *      return 0 if order of surface storage for curves agrees with     *
 *      order for bonds.                                                *
 *                                                                      *
 *      TODO: fully implement                                           */

LOCAL   void	surfaces_on_bonds_of_curve(
	INTERFACE 	*intfc)
{
  	BOND 		*b;
	CURVE 		*c;
	SURFACE 	*s;
	BOND_TRI	*t;
	int 		j, i = 0;

	(void) next_bond(intfc,NULL,NULL);
	(void) printf(" bond#   bondX  curve#   curveX   curve surfaces     "
		      "::      bond surfaces\n");
	while (next_bond(intfc,&b,&c))
	{
	    (void) printf("%d  %llu %5d  %llu   : ",i,bond_number(b,intfc),
			  index_of_pointer((POINTER*)intfc->curves,c),
			  curve_number(c));
	    j = 0; 
	    while ((c)->pos_surfaces != NULL &&
		   (s = (c)->pos_surfaces[j++]) != NULL)
	        (void) printf(" [+]%llu ",surface_number(s));
	    j = 0;
	    while ((c)->neg_surfaces != NULL &&
		   (s = (c)->neg_surfaces[j++]) != NULL)
	        (void) printf(" [-]%llu ",surface_number(s));
	    (void) printf(" :: ");
	    j = 0;
	    while (Btris(b) != NULL && (t = Btris(b)[j++]) != NULL)
	        (void) printf(" %llu ",
			      surface_number(Surface_of_tri(t->tri)));
	    (void) printf("\n");
	    ++i;
	}
} 		/*end surfaces_on_bonds_of_curve*/

/*
*			map_rectangle_to_circle():
*
*	Defines a planar transformation with the following properties.
*
*	1. Maps concentric rectangles inside the rectangle R0 of center m and
*          half side lengths X0 and Y0 to concentric circles centered at m
*          inside the circle C0 of radius r0.  R0 is mapped to C0.
*
*	2.  For region outside R0 the mapping transforms continuously from
*           the above mapping to the identity mapping. This is accomplished by
*           linearly interpolating between the map in (1) and the identity
*           mapping in the region outside R0 and inside Rmax,  where Rmax
*           is the rectangle concentric and similar to R0 with half diagonal
*	    length lmax.
*
*	3. Outside of Rmax the mapping is an identity.
*
*	Input:
*	        c    = coordinates to be transformed.
*               m    = center of the transformation.
*               X0   = half X width of the rectangle R0.
*               Y0   = half Y height of the rectangle R0.
*	        r0   = radius of image of the region defined in 1.
*	        lmax = half diagonal length of the rectangle at which the
*                      tranformation becomes the identity. 
*
*	Output:
*	        c   = transformed coordinates.
*
*/

LOCAL   void 	map_rectangle_to_circle( 
	float *c,
	float *m,
	float X0,
	float Y0,
	float r0,
	float lmax)
{
	float        r, g, alpha;
	float        x, y;
	float        ax, ay;
	float        l, l0;

	l0 = hypot(X0,Y0);

  	x = c[0] - m[0];
	y = c[1] - m[1];
	r = hypot(x,y);

	ax = fabs(x)/X0;
	ay = fabs(y)/Y0;
	l = l0*max(ax,ay);

	g = l*r0/l0;
	if (l <= l0)
	    alpha = (r > 0.0) ? g/r : 1.0;
	else if ((l0 < l) && (l <= lmax))
	{
	    float t = (l - l0)/(lmax - l0);
	    alpha = (1.0 - t)*g/r + t;
	}
	else
	    alpha = 1.0;

	c[0] = alpha*x + m[0];
	c[1] = alpha*y + m[1];

  	return;
}    		/*end map_rectangle_to_circle*/
#endif /* defined(THREED) */

#if defined(TWOD)
/*
*		insert_linear_section():
*
*	Inserts a linear section running from start to end into the bond b.
*/

LOCAL	void	insert_linear_section(
	BOND  *b,
	CURVE *c,
	float *start,
	float *end,
	Front *fr)
{
	RECT_GRID *gr = fr->rect_grid;
	float     *h = gr->h;
	float     L, ds;
	float     coords[2];
	float     space;
	int       i, N;
	int       dim = gr->dim;
	float     tol = MIN_SC_SEP(fr->interf);

	space = Front_spacing(fr,
	    	    (wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
	    		VECTOR_WAVE : GENERAL_WAVE);

	L = _scaled_separation(start,end,h,dim);
	N = irint(L/space);
	ds = L/N;

	if (_scaled_separation(start,Coords(b->start),h,dim) > tol)
	{
	    insert_point_in_bond(Point(start),b,c);
	    b = b->next;
	}
	for (i = 1; i < N; ++i)
	{
	    float alpha;
	    alpha = i*ds/L;
	    coords[0] = (1.0-alpha)*start[0] + alpha*end[0];
	    coords[1] = (1.0-alpha)*start[1] + alpha*end[1];
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}
	if (_scaled_separation(end,Coords(b->end),h,dim) > tol)
	{
	    insert_point_in_bond(Point(end),b,c);
	    b = b->next;
	}
}		/*end insert_linear_section*/

/*
*		insert_elliptical_curve_section():
*
*	Inserts an elliptical section with center cen and axes with
*	radii rad rotated by angle phi, into the bond b of curve c.
*	The inserted points are chosen to be separated by appropriate
*	front spacing factor in the scaled metric.
*/

LOCAL	const float *el_rad, *el_h;
LOCAL	float el_phi;
LOCAL	float el_len(float,POINTER);
#if defined(__cplusplus)
extern "C" {    
#endif /* defined(__cplusplus) */
    LOCAL   LSODE_FUNC  iel_len;
#if defined(__cplusplus)
}                          
#endif /* defined(__cplusplus) */

LOCAL	void	insert_elliptical_curve_section(
	BOND  *b,
	CURVE *c,
	float ThetaS,
	float ThetaE,
	float *cen,
	float *rad,
	float phi,
	Front *fr)
{
	RECT_GRID         *gr = fr->rect_grid;

#if defined(USE_OVERTURE)
        float             h[MAXD];  
        int               amr_refinecoeff = 1;
#else /* if defined(USE_OVERTURE) */ 
	float             *h = gr->h;
#endif /* if defined(USE_OVERTURE) */

	float             L;
	float             epsabs, epsrel;
	float             abserr;
	float             space;
	float             *theta;
	float             ds;
	float             cp, sp, coords[2];
	float             x, y;
	int               i, N, dim = gr->dim;
	int               neval;
	int               istate;
	QUADRATURE_STATUS ier;
	float             tol = MIN_SC_SEP(fr->interf);

#if defined(USE_OVERTURE)
        if(fr->NumberOfLevels != 0)
            amr_refinecoeff = (int)pow(2, fr->NumberOfLevels-1);
        for(i = 0; i < dim; i++)
            h[i] = gr->h[i]/amr_refinecoeff;
#endif /* if defined(USE_OVERTURE) */

	space = Front_spacing(fr,
	    	    (wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
	    		VECTOR_WAVE : GENERAL_WAVE);

	el_rad = rad; el_h = h;
	el_phi = phi;

	/* Compute the arc length in the scaled metric of the curve
	 * section to be inserted.
	 */
	epsrel = 1.0e-6;
	epsabs = sqrt((rad[0]/h[0])*(rad[1]/h[1]))*epsrel;
	L = dqng(el_len,NULL,ThetaS,ThetaE,
		      epsabs,epsrel,&abserr,&neval,&ier);
	switch (ier)
	{
	case INVALID_EPSILON:
	    screen("ERROR in insert_elliptical_curve_section(), "
		   "invalid epsilons\n"
		   "epsabs = %"FFMT", epsrel = %"FFMT"\n",epsabs,epsrel);
	    clean_up(ERROR);
	    break;
	case INACCURATE_INTEGRAL:
	    if ((fabs(abserr) > 20.0*epsabs) && (fabs(abserr) > 20.0*epsrel*L))
	    {
		/*
		 * Don't print a warning is we are close to satifying the
		 * error requirements
		 */
	        (void) printf("WARNING in insert_elliptical_curve_section(), "
			      "inaccurate result\n \tneval = %d, "
			      "abserr = %"FFMT" result = %"FFMT"\n",
			      neval,abserr,L);
	    }
	    break;
	case ACCURATE_INTEGRAL:
	default:
	    break;
	}
	N = irint(fabs(L)/space);
	if (N < 1)	/*Nothing to do, the elliptical section would be */
	    return;	/* too short */

	vector(&theta,N+1,FLOAT);

	ds = L/N;

	/*
	*  Integrate with respect to arc length to find angles with
	*  eqi-distance separation in the scaled metric along the ellipse
	*/
	theta[0] = ThetaS;
	istate = 1;
	if ((!ode_solver(iel_len,theta,0.0,ds,N,epsrel,
			 epsrel*fabs(ThetaE-ThetaS),
		         abserr,&istate,NULL)) || (istate != 2))
	{
	    screen("ERROR in insert_elliptical_curve_section() can't solve ODE\n"
	           "error flag = %d\n",istate);
	    clean_up(ERROR);
	}
	theta[N] = ThetaE;


	cp = cos(phi); sp = sin(phi);
	x = rad[0]*cos(theta[0]-phi);
	y = rad[1]*sin(theta[0]-phi);
	coords[0] = cen[0] + x*cp + y*sp;
	coords[1] = cen[1] - x*sp + y*cp;
	if (_scaled_separation(coords,Coords(b->start),h,dim) > tol)
	{
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}
	for (i = 1; i < N; ++i)
	{
	    x = rad[0]*cos(theta[i]-phi);
	    y = rad[1]*sin(theta[i]-phi);
	    coords[0] = cen[0] + x*cp + y*sp;
	    coords[1] = cen[1] - x*sp + y*cp;
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}
	x = rad[0]*cos(theta[N]-phi);
	y = rad[1]*sin(theta[N]-phi);
	coords[0] = cen[0] + x*cp + y*sp;
	coords[1] = cen[1] - x*sp + y*cp;
	if (_scaled_separation(coords,Coords(b->end),h,dim) > tol)
	{
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}

	free(theta);
}		/*end insert_elliptical_curve_section*/

/*ARGSUSED*/
LOCAL	float el_len(
	float theta,
	POINTER prms)
{
	IMPORT const float *el_rad, *el_h;
	IMPORT float el_phi;
	float dx, dy;
	float dx1, dy1;
	float c, s;

	dx1 = -el_rad[0]*sin(theta-el_phi);
	dy1 =  el_rad[1]*cos(theta-el_phi);
	c = cos(el_phi);
	s = sin(el_phi);
	dx =  dx1*c + dy1*s;
	dy = -dx1*s + dy1*c;
	return hypot(dx/el_h[0],dy/el_h[1]);
}		/*end el_len*/

#if defined(__cplusplus)
extern "C" {    
#endif /* defined(__cplusplus) */

/*ARGSUSED*/
LOCAL	void iel_len(
	int		*neq,
	float		*x,
	float		*y,
	float		*yp)
{
	*yp = 1.0/el_len(*y,NULL);
}		/*end iel_len*/

#if defined(__cplusplus)
}                          
#endif /* defined(__cplusplus) */

/*
*		insert_polynomial_curve_section():
*
*	Inserts a polynomial curve y = p[x] into the curve c inside bond b.
*/

LOCAL	const float *pc, *poly_h;
LOCAL	int   d;

LOCAL	float poly_len(float,POINTER);
#if defined(__cplusplus)
extern "C" {    
#endif /* defined(__cplusplus) */
    LOCAL   LSODE_FUNC  ipoly_len;
#if defined(__cplusplus)
}                          
#endif /* defined(__cplusplus) */

LOCAL	void	insert_polynomial_curve_section(
	BOND  *b,
	CURVE *c,
	float xs,
	float xe,
	float *coef,
	int   n,
	Front *fr)
{
	RECT_GRID         *gr = fr->rect_grid;

#if defined(USE_OVERTURE)
        float             h[MAXD];
        int               amr_refinecoeff = 1;
#else /* if defined(USE_OVERTURE) */
        float             *h = gr->h;
#endif /* if defined(USE_OVERTURE) */

	float             L;
	float             epsabs, epsrel;
	float             abserr;
	float             space;
	float             *x;
	float             ds;
	float             coords[2];
	int               i, N, dim = gr->dim;
	int               neval;
	int               istate;
	QUADRATURE_STATUS ier;
	float             tol = MIN_SC_SEP(fr->interf);

#if defined(USE_OVERTURE)
        if(fr->NumberOfLevels != 0)
            amr_refinecoeff = (int)pow(2, fr->NumberOfLevels-1);
        for(i = 0; i < dim; i++)
            h[i] = gr->h[i]/amr_refinecoeff;
#endif /* if defined(USE_OVERTURE) */

	space = Front_spacing(fr,
	    	    (wave_type(c) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE) ?
	    		VECTOR_WAVE : GENERAL_WAVE);

	pc = coef;
	poly_h = h;
	d = n;

	/* Compute the arc length in the scaled metric of the curve
	 * section to be inserted.
	 */
	epsrel = 1.0e-6;
	epsabs = fabs(xs - xe)*epsrel;
	L = dqng(poly_len,NULL,xs,xe,
		      epsabs,epsrel,&abserr,&neval,&ier);
	switch (ier)
	{
	case INVALID_EPSILON:
	    screen("ERROR in insert_polynomial_curve_section(), "
		   "invalid epsilons\n"
		   "epsabs = %"FFMT", epsrel = %"FFMT"\n",epsabs,epsrel);
	    clean_up(ERROR);
	    break;
	case INACCURATE_INTEGRAL:
	    if ((fabs(abserr) > 20.0*epsabs) && (fabs(abserr) > 20.0*epsrel*L))
	    {
		/*
		 * Don't print a warning is we are close to satifying the
		 * error requirements
		 */
	        (void) printf("WARNING in insert_polynomial_curve_section(), "
			      "inaccurate result\n \tneval = %d, "
			      "abserr = %"FFMT" result = %"FFMT"\n",
			      neval,abserr,L);
	    }
	    break;
	case ACCURATE_INTEGRAL:
	default:
	    break;
	}
	N = irint(fabs(L)/space);
	if (N < 1)	/*Nothing to do, the elliptical section would be */
	    return;	/* too short */

	vector(&x,N+1,FLOAT);

	ds = L/N;

	/*
	*  Integrate with respect to arc length to find angles with
	*  eqi-distance separation in the scaled metric along the ellipse
	*/
	x[0] = xs;
	istate = 1;
	if ((!ode_solver(ipoly_len,x,xs,ds,N,epsrel,
			 epsrel*fabs(xs-xe),
		         abserr,&istate,NULL)) || (istate != 2))
	{
	    screen("ERROR in insert_polynomial_curve_section() can't solve ODE\n"
	           "error flag = %d\n",istate);
	    clean_up(ERROR);
	}
	x[N] = xe;


	coords[0] = xs;
	coords[1] = eval_poly(coords[0],coef,n);
	if (_scaled_separation(coords,Coords(b->start),h,dim) > tol)
	{
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}
	for (i = 1; i < N; ++i)
	{
	    coords[0] = x[i];
	    coords[1] = eval_poly(coords[0],coef,n);
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}
	coords[0] = x[N];
	coords[1] = eval_poly(coords[0],coef,n);
	if (_scaled_separation(coords,Coords(b->end),h,dim) > tol)
	{
	    insert_point_in_bond(Point(coords),b,c);
	    b = b->next;
	}

	free(x);
}		/*end insert_polynomial_curve_section*/

/*ARGSUSED*/
LOCAL	float poly_len(
	float x,
	POINTER prms)
{
	IMPORT	const float *pc, *poly_h;
	float dp;
	int   i;

	dp = 0.0;
	for (i=d-1; i >= 0; --i)
	    dp = (i+1)*pc[i+1] + x*dp;

	return hypot(1.0/poly_h[0],dp/poly_h[1]);
}		/*end poly_len*/

#if defined(__cplusplus)
extern "C" {    
#endif /* defined(__cplusplus) */

/*ARGSUSED*/
LOCAL	void ipoly_len(
	int		*neq,
	float		*x,
	float		*y,
	float		*yp)
{
	*yp = 1.0/poly_len(*y,NULL);
}		/*end ipoly_len*/
#if defined(__cplusplus)
}                          
#endif /* defined(__cplusplus) */

LOCAL	float eval_poly(
	float       x,
	const float *coef,
	int         n)
{
    	float p;
	int   i;

	p = 0.0;
	for (i=n; i >= 0; --i)
	    p = coef[i] + x*p;
	return p;
}

#endif /* defined(TWOD) */

#endif /* defined(TWOD) || defined(THREED) */
