#ifdef MHD
/*
*				gicrp.c:
*
*	Copyright 2000 by The University at Stony Brook, All rights reserved.
*
*	Contains initialization routines for cylindrical Riemann
*	problems in gas dynamics.
*
*/

#include <ginit/ginit.h>

struct __CYLINDRICAL {
	CYLINDER            *cylinder;
	RANDOM_STATE	    *rstate;
	Locstate	    state;
	Locstate	    wkstate[2];
	float               weight[MAXD];
	float		    r0;
	STRATIFICATION_TYPE stratification_type;
	bool		    untracked_shock;    /* flag for untracked shock */
	Locstate	    utsb_state;		/* behind untracked shock state */
	CYLINDER	    *uts_cylinder;	/* ellipsoidal front of 
						   untracked shock */
};
typedef struct __CYLINDRICAL _CYLINDRICAL;
						 
enum _CYLINDRICAL_REGION_TYPE {
    AMBIENT_CYLINDRICAL_REGION                           = 0,
    RANDOM_AMBIENT_CYLINDRICAL_REGION                    = 1,
    CYLINDRICAL_VELOCITY_REGION                          = 2,
    RANDOM_CYLINDRICAL_VELOCITY_REGION                   = 3,
    RADIAL_STRATIFIED_CYLINDRICAL_VELOCITY_REGION        = 4,
    RANDOM_RADIAL_STRATIFIED_CYLINDRICAL_VELOCITY_REGION = 5
};
typedef enum _CYLINDRICAL_REGION_TYPE CYLINDRICAL_REGION_TYPE;

	/* LOCAL Function Prototypes */

LOCAL	CYLINDER *allocate_cylinder(CYLINDER*,int);
LOCAL	bool	prompt_for_cylindrical_region_type(const INIT_PHYSICS*,
					       	const INIT_DATA*,
						CYLINDRICAL_REGION_TYPE*);
LOCAL	float	mean_pressure_for_region(COMP_TYPE*);
LOCAL	void	init_crp_in_to_out_states(INIT_PHYSICS*,INIT_DATA*,float*,
					   COMPONENT*,int*,Gas_param**,
					   CYLINDER**,int);
LOCAL	void	init_crp_states(INIT_PHYSICS*,INIT_DATA*,float*,COMPONENT*,
				 int*,Gas_param**,CYLINDER**,int);
LOCAL	void	prompt_for_cylindrical_state(INIT_PHYSICS*,COMPONENT,
					    CYLINDER*,Gas_param*,
					    const char*,INIT_DATA*);
LOCAL	void	set_default_cylinder_structure(CYLINDER*,int);
LOCAL	void	init_rp_params(INIT_DATA*,INIT_PHYSICS*,
				 Gas_param**,const char*,int*,int);


/*
*	The various regions in this initialization are referred to
*	by indices as indicated in the table below.
*
*	Inside the inner cylinder		0
*	Between inner and outer cylinders	1
*	Outside the outer cylinder		2
*/

EXPORT	void init_cyl_riem_prob(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip)
{
	COMPONENT	min_comp = FIRST_DYNAMIC_COMPONENT;
	Front		*front = ip->root->front;
	INTERFACE	*intfc = front->interf, temp_intfc;
	char		s[Gets_BUF_SIZE];
	const char	*cylname;
	char		where[256];
	CYLINDER	**cylinder;
	float		upper_height = 0.0, lower_height = 0.0;
	float		*L = front->rect_grid->L;
	float		*h = front->rect_grid->h;
	float		dflt_rad;
	float           dflt_length;
	float		*pr_region;
	int		i, k, dim = front->interf->dim;
	int		num_cylinder;
	int		*wv_type;
	COMPONENT	*comps;
	Gas_param	**prms;

	if (dim < 2)
	{
	    screen("ERROR in init_cyl_riem_prob(), dim < 2 not supported\n");
	    clean_up(ERROR);
	    return;
	}

        init_magnetic_field(ip);

	screen("\n Initialization of a %dD cylinder\n\n",dim);

	cylname = "cylinder";
	num_cylinder = 1;
	screen("Enter the number (>= 1) of %s waves (dflt = %d): ",
				"cylindrical",num_cylinder);
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscanf(s,"%d",&num_cylinder);
	if (num_cylinder < 1)
	{
	    screen("ERROR in init_cyl_riem_prob(), no waves specified\n");
	    clean_up(ERROR);
	    return;
	}
	vector(&comps,num_cylinder+3,sizeof(COMPONENT));
	vector(&prms,num_cylinder+3,sizeof(Gas_param*));
	vector(&cylinder,num_cylinder,sizeof(CYLINDER*));
	vector(&wv_type,num_cylinder+3,INT);
	vector(&pr_region,num_cylinder+3,FLOAT);

	comps[0] = min_comp;
	for (i = 1; i < num_cylinder+1; i++)
	    comps[i] = comps[i-1] + 1;
	comps[num_cylinder+1] = comps[num_cylinder+2] = comps[num_cylinder];
	for (i = 0; i < num_cylinder+3; i++)
	{
	    wv_type[i] = UNKNOWN_WAVE_TYPE;
	    pr_region[i] = -HUGE_VAL;
	}

	    /* Choose and locate tracked waves */

	(void) sprintf(where,"%s",(num_cylinder > 1) ? "inner" : "");
	
	switch (dim)
	{
#if defined TWOD
	case 2:
	    cylinder[0] = prompt_for_cylinder(front,"inner",NULL,1,1,0,ip);
	    break;
#endif /* defined TWOD */
#if defined THREED
	case 3:
	    cylinder[0] = prompt_for_cylinder(front,"inner",NULL,1,1,0,ip);
            break;
#endif /* defined THREED */
	}
	wv_type[0] = cylinder[0]->wv_type;
	if ((wv_type[0] < FIRST_PHYSICS_WAVE_TYPE) && (num_cylinder > 1))
	{
	    (void)SetConstantFlowRegion(comps[0],return_obst_state(),intfc);
	    set_obstacle_comp_type(comp_type(comps[0]),front);
	}
	for (i = 1; i < num_cylinder; i++)
	{
	    char	mesg[80];

	    (void) sprintf(mesg,"%s %d",cylname,i);
	    	dflt_rad = cylinder[i-1]->rad + 0.5*h[0];
		dflt_length = cylinder[i-1]->length + 0.5*h[0];
		switch (dim)
            {
#if defined TWOD
            case 2:
	        cylinder[i] = prompt_for_cylinder(front,mesg,
		              cylinder[i-1]->cen,dflt_rad,dflt_length,0,ip);
                break;
#endif /* defined TWOD */
#if defined THREED
            case 3:
	        cylinder[i] = prompt_for_cylinder(front,mesg,
		              cylinder[i-1]->cen,dflt_rad,dflt_length,0,ip);
                break;
#endif /* defined THREED */
            }
	    wv_type[i] = cylinder[i]->wv_type;
	    if (wv_type[i] < FIRST_PHYSICS_WAVE_TYPE)
	    {
	    	num_cylinder = i+1;
	    	(void)SetConstantFlowRegion(comps[i+1],return_obst_state(),
					    intfc);
	    	set_obstacle_comp_type(comp_type(comps[i+1]),front);
	    	break;
	    }
	}


	upper_height += L[dim-1];
	lower_height += L[dim-1];


		/* EOS for cylindrical regions */

	init_rp_params(init,ip,prms,cylname,wv_type,num_cylinder);

	init_crp_states(ip,init,pr_region,comps,wv_type,prms,cylinder,num_cylinder);
	screen("\n");

	/* NOTE:  assumes explosion or implosion into vr = 0 */
	/* TODO: revise radial_analysis so shock is not assumed
	*  to propagate first. */

	for (i = 0; i < num_cylinder; i++)
	{
	    HYPER_SURF *hs;

	    if (is_scalar_wave(cylinder[i]->wv_type))
		cylinder[i]->layer_index = ++num_layers(front->interf);

	    switch (dim)
            {
#if defined TWOD
            case 2:
		hs = make_2d_cylinder(cylinder[i],comps[i],comps[i+1],front,YES,YES);
                break;
#endif /* defined TWOD */
#if defined THREED
            case 3:
		hs = g_make_cylinder(cylinder[i],comps[i],comps[i+1],front);
                break;
#endif /* defined THREED */
            }
	    if (wv_type[i] < FIRST_PHYSICS_WAVE_TYPE)
	    {	
	    	(void) sprintf(s,"%s %d",cylname,i);
	    	bstate_index(hs) = prompt_for_boundary_state(wave_type(hs),
				       s,NULL,positive_component(hs),
				       -1,hs,init,ip);
	    }
	}

		
	if (wv_type[num_cylinder-1] >= FIRST_PHYSICS_WAVE_TYPE)
	{
	    if (wv_type[num_cylinder] != UNKNOWN_WAVE_TYPE) 
	    {
	    	make_plane_wave(upper_height,comps[num_cylinder+1],
	    		        comps[num_cylinder],pr_region[num_cylinder+1],
	    		        pr_region[num_cylinder],
	    		        wv_type[num_cylinder],front->rect_grid);
	    	screen("Is the flow above the upper plane wave "
	    	       "constant (dflt = no): ");
	    	(void) Gets(s);
	    	if (s[0] == 'y' || s[0] == 'Y')
	    	    (void)SetConstantFlowRegion(comps[num_cylinder+1],
	    			Ambient(comp_type(comps[num_cylinder+1])),intfc);
	    }
	    if (wv_type[num_cylinder+1] != UNKNOWN_WAVE_TYPE) 
	    {
	    	make_plane_wave(lower_height,comps[num_cylinder],
	    		        comps[num_cylinder+2],pr_region[num_cylinder],
			        pr_region[num_cylinder+2],
			        wv_type[num_cylinder+1],front->rect_grid);
		screen("Is the flow below the lower plane wave "
		       "constant (dflt = no): ");
		(void) Gets(s);
		if (s[0] == 'y' || s[0] == 'Y')
		    (void)SetConstantFlowRegion(comps[num_cylinder+2],
		    		Ambient(comp_type(comps[num_cylinder+2])),intfc);
	    }
	}
	free_these(5,comps,cylinder,wv_type,prms,pr_region);
}		/*end init_cyl_riem_prob*/

LOCAL	void init_crp_states(
	INIT_PHYSICS	*ip,
	INIT_DATA	*init,
	float		*pr_region,
	COMPONENT	*comps,
	int		*wv_type,
	Gas_param	**prms,
	CYLINDER	**cylinder,
	int		num_cylinder)
{
	Front		*front = ip->root->front;
	int		i, dim = front->rect_grid->dim;
	float		nor[MAXD];
	char		s[Gets_BUF_SIZE];

	screen("Initialize states from inner to outer \n "); 
	init_crp_in_to_out_states(ip,init,pr_region,comps,wv_type,
			               prms,cylinder,num_cylinder);


        pr_region[num_cylinder+1] = pr_region[num_cylinder];
        pr_region[num_cylinder+2] = pr_region[num_cylinder];
}		/*end init_elrp_states*/

LOCAL	void init_crp_in_to_out_states(
	INIT_PHYSICS	*ip,
	INIT_DATA	*init,
	float		*pr_region,
	COMPONENT	*comps,
	int		*wv_type,
	Gas_param	**prms,
	CYLINDER	**cylinder,
	int		num_cylinder)
{
	CYLINDER  *cyl;
	Front	   *front = ip->root->front;
	INTERFACE  *intfc = front->interf;
	int	   i, dim = front->rect_grid->dim;
	int	   istart;
	const char *cylname = "cylinder";
	char	   where[256];
	char	   s[Gets_BUF_SIZE];

	if (prms[0] == Params(return_obst_state()))
	{
	    pr_region[0] = 0.0;
	    (void)SetConstantFlowRegion(comps[0],return_obst_state(),intfc);
	    set_obstacle_comp_type(comp_type(comps[0]),front);
	    (void) sprintf(where," outside the %s%s",
	                   (num_cylinder > 1) ? "inner " : "",cylname);
	    prompt_for_cylindrical_state(ip,comps[1],cylinder[0],
				        prms[1],where,init);
	    if (num_cylinder == 1)
		return;
	    istart = 2;
	}
	else
	{
	    (void) sprintf(where," inside the %s%s",
			   (num_cylinder > 1) ? "inner " : "",cylname);
	    prompt_for_cylindrical_state(ip,comps[0],cylinder[0],
					prms[0],where,init);
	    pr_region[0] = mean_pressure_for_region(comp_type(comps[0]));
	    istart = 1;
	}
		
	for (i = istart; i < num_cylinder; i++)
	{
	    cyl = cylinder[i];
	    (void) sprintf(where," between the %d-%s && the %d-%s %ss",
		           i-1,ordinal_suffix(i-1),i,ordinal_suffix(i),cylname);
	    screen("Type 'y' to enter an arbitrary state ");
	    screen("for the region\n\t%s: ",where);
	    (void) Gets(s);
	    if (s[0] == 'y' || s[0] == 'Y')
	    {
	        prompt_for_cylindrical_state(ip,comps[i],cyl,prms[i],where,init);
		pr_region[i] = mean_pressure_for_region(comp_type(comps[i]));
	    }
	}
	if (prms[num_cylinder] == Params(return_obst_state()))
	{
	    pr_region[0] = 0.0;
	    (void)SetConstantFlowRegion(comps[num_cylinder],
					return_obst_state(),intfc);
	    set_obstacle_comp_type(comp_type(comps[num_cylinder]),front);
	}
	else
	{
	    i = num_cylinder;
	    cyl = cylinder[i-1];
	    (void) sprintf(where," outside the %s%s",
			   (num_cylinder > 1) ? "outer " : "",cylname);
	    screen("Type 'y' to enter an arbitrary state ");
	    screen("for the region\n\t%s: ",where);
	    (void) Gets(s);
	    if (s[0] == 'y' || s[0] == 'Y')
	    {
	        prompt_for_cylindrical_state(ip,comps[i],cyl,prms[i],where,init);
		pr_region[i] = mean_pressure_for_region(comp_type(comps[i]));
	    }
	}
}		/*end init_crp_in_to_out_states*/

LOCAL	float	mean_pressure_for_region(
	COMP_TYPE	*ct)
{
	switch (ct->type)
	{
	case AMBIENT:
	    return pressure(Ambient(ct));
	case RANDOM_REGION:
	    return pressure(Mean(Random_state(ct)));
	case CYLINDRICAL:
	default:
	    screen("ERROR in mean_pressure_for_region(), "
		   "invalid component type\n");
	    clean_up(ERROR);
	}
	return -HUGE_VAL;
}		/*end mean_pressure_for_region*/


EXPORT  CYLINDER *prompt_for_cylinder(
	Front		*front,
	const char	*mesg,
	float		*dflt_center,
	float		dflt_radius,
	float           dflt_length,
	int		flag,
	INIT_PHYSICS	*ip)
{
	IMPORT	bool suppress_prompts;
	CYLINDER	*cylinder = NULL;
	CYLINDER	Cylinder;
	char	    	s[Gets_BUF_SIZE];
	int	    	dim = front->rect_grid->dim, i;
	char	    	*cylname;
	const char      *in_out;
	int  A[2], B[2];

	cylname = "cylinder";


	set_default_cylinder_structure(&Cylinder,dim);
	Cylinder.nor_orient = POSITIVE_ORIENTATION;

	screen("The %s %s can be chosen to be a ",mesg,cylname);
	screen("closed (full) (C, F, default), or half (H) %s\n",cylname);
	screen("Enter choice: ");
	(void) Gets(s);
	switch (s[0])
	{
	case 'h':
	case 'H':
	    Cylinder.ThetaS = 0.0;
	    Cylinder.ThetaE = PI;
	    Cylinder.closed = NO;
	    break;
	case 'f':
	case 'F':
	case 'c':
	case 'C':
	default:
	    Cylinder.ThetaS = 0.0;
	    Cylinder.ThetaE = 2*PI;
	    Cylinder.closed = YES;
	    break;
	}
	screen("This %s has been oriented in the counter clockwise direction\n"
	       "so that the forward moving normal directions point towards\n"
	       "the outside of the %s.\n",cylname,cylname);

	screen("Enter the %d coordinates of the center of the %s %s",
	       dim,mesg,cylname);
	if (dflt_center != NULL)
	{
	    char *dflt = "\n\t(dflt = ";
	    print_general_vector(dflt,dflt_center,dim,"): ");
	    if (suppress_prompts == NO)
	        fprint_general_vector(stderr,dflt,dflt_center,dim,"): ");
	}
	else
	    screen(": ");
	read_vector(Cylinder.cen,dflt_center,dim);

	screen("Enter the radius of the %s %s,\n",mesg,cylname);
        screen("(dflt = 1): ");
	(void) Gets(s);
	if (s[0] != 0)
	  {
	   (void) sscan_float(s,&Cylinder.rad);
	  }
	screen("Enter the semi-length of the %s %s",mesg,cylname);
        screen("(dflt = 1): ");
	(void) Gets(s);
	if (s[0] != 0)
	  {
	   (void) sscan_float(s,&Cylinder.length);
	  }
	screen("Type 'y' if the %s %s is to be perturbed by surface waves: ",mesg,cylname);
	(void) Gets(s);
	if (s[0] == 'Y' || s[0] == 'y')
	{
	  Cylinder.perturbed = YES;
	  screen("Enter coef. of random perturbation of coords (<< 1 || 0): ");
	  (void) Gets(s);
	  (void) sscan_float(s,&Cylinder.rand_coef);
	  
	  screen("Enter the number of the surface waves: ");
	  (void) Gets(s);
	  (void) sscanf(s,"%d",&Cylinder.waves);
	  
	  screen("Enter the amplitudes and the wave numbers\n");
	  screen("(here wave number = number of waves on the length of the jet)\n");
	  
	  for (i = 0; i < Cylinder.waves; i++)
	    {
	      screen("Enter the amplitude for the wave %d: ", i);
	      (void) Gets(s);
	      (void) sscan_float(s,&Cylinder.A_wave[i]);
	      
	      screen("Enter the wave number for the wave %d: ", i);
	      (void) Gets(s);
	      (void) sscan_float(s,&Cylinder.wave_number[i]);
	    }
	}
	screen("Type 'y' if edges of the %s %s are to be perturbed by 
               surface waves: ",mesg,cylname);
	(void) Gets(s);
	if (s[0] == 'Y' || s[0] == 'y')
	{
	  Cylinder.perturbed_ed = YES;

	  screen("Enter coef. of random perturbation of coords (<< 1 || 0): ");
	  (void) Gets(s);
	  (void) sscan_float(s,&Cylinder.rand_coef_ed);

	  screen("Enter the number of the surface waves: ");
	  (void) Gets(s);
	  (void) sscanf(s,"%d",&Cylinder.waves_ed);
	  
	  screen("Enter the amplitudes and the wave numbers\n");
	  screen("(here wave number = number of waves on the length of the jet edge)\n");
	  
	  for (i = 0; i < Cylinder.waves_ed; i++)
	    {
	      screen("Enter the amplitude for the wave %d: ", i);
	      (void) Gets(s);
	      (void) sscan_float(s,&Cylinder.A_wave_ed[i]);
	      
	      screen("Enter the wave number for the wave %d: ", i);
	      (void) Gets(s);
	      (void) sscan_float(s,&Cylinder.wave_number_ed[i]);
	    }
	}
	cylinder = allocate_cylinder(&Cylinder,0);
	/*
	if ( flag & ALL_CONTACT)
	    cylinder->wv_type = CONTACT;
	else       
	*/
	in_out = (cylinder->nor_orient == POSITIVE_ORIENTATION) ?
	  "outward" : "inward";
	(void) sprintf(s, "for the %s %s (normal/forward direction = %s)",
	    	       mesg, cylname,in_out);
	cylinder->wv_type = prompt_for_wave_type(s,front->interf,ip);
	

	(void) sprintf(s,"for the %s cylindrical curve ",mesg);
	cylinder->surf_tension =
	    prompt_for_surface_tension(cylinder->wv_type,s);

	cylinder->untracked = NO;
	screen("Type y to turn off tracking for the %s %s: ", mesg, cylname);
	(void) Gets(s);
	if ( s[0] == 'y' || s[0] == 'Y' )
	    cylinder->untracked = YES;
/*
	screen("\n Type y to initialize the external energy deposition: ");
	(void) Gets(s);
	if (s[0] == 'y' || s[0] == 'Y')
	  init_energy_deposition();
*/	 
	return cylinder;
}	/*end prompt_for_cylinder*/

LOCAL void prompt_for_cylindrical_state(
	INIT_PHYSICS	*ip,
	COMPONENT	comp,
	CYLINDER	*cylinder,
	Gas_param	*params,
	const char	*message,
	INIT_DATA	*init)
{
	CYLINDRICAL_REGION_TYPE ert;
	bool		       random;
	char		       s[Gets_BUF_SIZE];
	Front		       *front = ip->root->front;
	COMP_TYPE	       *ctype = comp_type(comp);
	_CYLINDRICAL	       *cyl;
	float		       rho,pr,vr, maxr;
	int		       i, dim = cylinder->dim;

	screen("Initialize the cylindrical state%s -\n",message);

	random = prompt_for_cylindrical_region_type(ip,init,&ert);

	if ((ert == AMBIENT_CYLINDRICAL_REGION) ||
	    (ert == RANDOM_AMBIENT_CYLINDRICAL_REGION))
	{
	    set_up_ambient_elliptic_region(random,ctype,params,
					   message,front,init);
	    return;
	}

}		/*end prompt_for_cylindrical_state*/


LOCAL	bool	prompt_for_cylindrical_region_type(
	const INIT_PHYSICS           *ip,
	const INIT_DATA              *init,
	CYLINDRICAL_REGION_TYPE       *ert)
{
	GRAVITY *grav_data = gravity_data(init);
	bool random = NO;
	bool allow_radial_stratified;
	char    s[Gets_BUF_SIZE];

	zero_scalar(s,Gets_BUF_SIZE);
	*ert = CYLINDRICAL_VELOCITY_REGION;

	allow_radial_stratified =
	    ((problem_type(ip) == RADIAL_RAYLEIGH_TAYLOR) &&
	     (grav_data->type == RADIAL_GRAVITY)) ? YES : NO;

	screen("The choices for flow initialization types are\n");
	screen("\tAmbient region (constant initial conditions) (A%s)\n",
	       (*ert == AMBIENT_CYLINDRICAL_REGION) ? ", default" : "");
	screen("\tRandom perturbation of an ambient region (RA%s)\n",
	       (*ert == RANDOM_AMBIENT_CYLINDRICAL_REGION) ? ", default" : "");
	screen("\tCylindrical region "
	       "(constant thermodynamics, radial velocity (E%s)\n",
	       (*ert == CYLINDRICAL_VELOCITY_REGION) ? ", default" : "");
	screen("\tRandom perturbation of cylindrical region (RE%s)\n",
	       (*ert == RANDOM_CYLINDRICAL_VELOCITY_REGION) ? ", default" : "");
	if (allow_radial_stratified == YES)
	{
	    screen("\tRadial gravity stratified region (RGSR%s)\n",
		   (*ert == RADIAL_STRATIFIED_CYLINDRICAL_VELOCITY_REGION) ?
		       ", default" : "");
	    screen("\tRandom perturbation of radial gravity "
		   "stratified region (RRGSR%s)\n",
		   (*ert==RANDOM_RADIAL_STRATIFIED_CYLINDRICAL_VELOCITY_REGION) ?
		   ", default" : "");
	}
	screen("Enter choice: ");
	(void) Gets(s);
	if (strncasecmp(s,"A", 1) == 0)
	{

	  /* init_elrp_out_to_in() was here before */
        *ert = AMBIENT_CYLINDRICAL_REGION;
	    random = NO;
	}
	else if (strncasecmp(s,"RA",2) == 0)
	{
	    *ert = RANDOM_AMBIENT_CYLINDRICAL_REGION;
	    random = YES;
	}
	else if (strncasecmp(s,"E", 1) == 0)
	{
	    *ert = CYLINDRICAL_VELOCITY_REGION;
	    random = NO;
	}
	else if (strncasecmp(s,"RE", 2) == 0)
	{
	    *ert = RANDOM_CYLINDRICAL_VELOCITY_REGION;
	    random = YES;
	}
	if (allow_radial_stratified == YES)
	{
	    if (strncasecmp(s,"RGSR", 4) == 0)
	    {
	        *ert = RADIAL_STRATIFIED_CYLINDRICAL_VELOCITY_REGION;
	        random = NO;
	    }
	    else if (strncasecmp(s,"RRGSR", 5) == 0)
	    {
	        *ert = RANDOM_RADIAL_STRATIFIED_CYLINDRICAL_VELOCITY_REGION;
	        random = YES;
	    }
	}
	return random;
}		/*end prompt_for_cylindrical_region_type*/

                                 
LOCAL	CYLINDER        *allocate_cylinder(
	CYLINDER	*template_cylinder,
	int		nmodes)
{
	CYLINDER	*cylinder;
	size_t		size, naE = 0;
	int		i, j, dim = template_cylinder->dim;

	if (nmodes > 0)
	{
	    size_t	naFP, naA, naP, naNUP, naNUS;
	    naE = num_aligns(sizeof(CYLINDER));
	    naFP = num_aligns(sizeof(FOURIER_POLY));
	    naA = num_aligns(nmodes*FLOAT);
	    naP = num_aligns(nmodes*FLOAT);
	    naNUP = num_aligns(nmodes*sizeof(float*));
	    naNUS = num_aligns(nmodes*(dim-1)*FLOAT);
	    size = sizeof(ALIGN)*(naE + naFP + naA + naP + naNUP + naNUS);
	}
	else
	    size = sizeof(CYLINDER);

	scalar(&cylinder,size);

	set_default_cylinder_structure(cylinder,dim);

	for (i = 0; i < 3; i++)
	{
	    cylinder->cen[i] = template_cylinder->cen[i];
	    for (j = 0; j < 3; j++)
	    	cylinder->Q[i][j] = template_cylinder->Q[i][j];
	}

	for (i = 0; i < 10; i++)
	{
	    cylinder->A_wave[i] = template_cylinder->A_wave[i];
	    cylinder->wave_number[i] = template_cylinder->wave_number[i];
	    cylinder->A_wave_ed[i] = template_cylinder->A_wave_ed[i];
	    cylinder->wave_number_ed[i] = template_cylinder->wave_number_ed[i];
	}

        cylinder->ThetaS = template_cylinder->ThetaS;
        cylinder->ThetaE = template_cylinder->ThetaE;
	cylinder->rad = template_cylinder->rad;
	cylinder->length = template_cylinder->length;

	cylinder->closed = template_cylinder->closed;
	cylinder->perturbed = template_cylinder->perturbed;
	cylinder->perturbed_ed = template_cylinder->perturbed_ed;
	cylinder->nor_orient = template_cylinder->nor_orient;
	cylinder->compin = template_cylinder->compin;
	cylinder->compout = template_cylinder->compout;
	cylinder->rand_coef = template_cylinder->rand_coef;
	cylinder->rand_coef_ed = template_cylinder->rand_coef_ed;
	cylinder->waves = template_cylinder->waves;
	cylinder->waves_ed = template_cylinder->waves_ed;
	cylinder->hs = template_cylinder->hs;
	cylinder->surf_tension = template_cylinder->surf_tension;
	cylinder->wv_type = template_cylinder->wv_type;
	cylinder->dim = template_cylinder->dim;
	cylinder->untracked = template_cylinder->untracked;
	cylinder->layer_index = 0;
	
	return cylinder;
}		/*end allocate_cylinder*/


LOCAL	void	set_default_cylinder_structure(
	CYLINDER	*cylinder,
	int		dim)
{
	int		i, j;

	zero_scalar(cylinder,sizeof(CYLINDER));
	cylinder->Q = cylinder->Qrows;
	for (i = 0; i < 3; i++)
	    cylinder->Q[i] = &cylinder->Qstore[i][0];

	for (i = 0; i < 3; i++)
	{
	    cylinder->cen[i] = 0.0;
	    for (j = 0; j < 3; j++)
	    	cylinder->Q[i][j] = (i == j) ? 1.0 : 0.0;
	}

        cylinder->rad = 1.0;
        cylinder->length = 1.0;

	cylinder->dim = dim;
}		/*end set_default_cylinder_structure*/


LOCAL	void init_rp_params(
	INIT_DATA	*init,
	INIT_PHYSICS	*ip,
	Gas_param	**prms,
	const char	*name,
	int		*wv_type,
	int		num_obj)
{
	int	j;
	char	mesg[80];
	char	s[Gets_BUF_SIZE];

	(void) prompt_for_eos_params(init,ip,YES,"");
	if (wv_type[0] < FIRST_PHYSICS_WAVE_TYPE)
	{
	    if (num_obj > 1)
	        prms[0] = Params(return_obst_state());
	    else
	    {
		screen("Is the region inside the %s active (dflt = y): ",
		       name);
		(void) Gets(s);
		if ((s[0] == 'N') || (s[0] == 'n'))
		{
	            (void) sprintf(mesg,"\n\tfor the region outside the %s",
			           name);
		    prms[0] = Params(return_obst_state());
	            prms[1] = prompt_for_eos_params(init,ip,YES,mesg);
		}
		else
		{
	            (void) sprintf(mesg,"\n\tfor the region inside the %s",
			           name);
	            prms[0] = prompt_for_eos_params(init,ip,YES,mesg);
		    prms[1] = Params(return_obst_state());
		}
	    }
	}
	else
	{
	    (void) sprintf(mesg,"\n\tfor the region inside the %s%s",
	                   (num_obj > 1) ? "inner " : "",name);
	    prms[0] = prompt_for_eos_params(init,ip,YES,mesg);
	}
	for (j = 1; j < 2; j++)
	{
	    if (is_scalar_wave(wv_type[j-1]) ||
		(wv_type[j-1] < FIRST_PHYSICS_WAVE_TYPE))
	    {
	        (void) sprintf(mesg,"\n\tfor the region between the "
				    "%d-%s and %d-%s %ss",
				    j-1,ordinal_suffix(j-1),
				    j,ordinal_suffix(j),name);
	        prms[j] = prompt_for_eos_params(init,ip,YES,mesg);
	    }
	    else
	        prms[j] = prms[j-1];
	}
	if ((wv_type[num_obj-1]<FIRST_PHYSICS_WAVE_TYPE) && (num_obj>1))
	    prms[num_obj] = Params(return_obst_state());
	else if (is_scalar_wave(wv_type[num_obj-1]))
	{
	    (void) sprintf(mesg,"\n\tfor the region outside the %s%s",
			   (num_obj > 1) ? "outer " : "",name);
	}
	else
	    prms[num_obj] = prms[num_obj-1];
	screen("\n");
}		/*end init_rp_params*/

#endif
