/*
*                               gvisc.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*       The difference between Euler and Navier-Stokes equations are the
*       viscosity and heat conduction source terms. 
*
*       The viscosity terms involve velocity Laplace for incompressible
*       fluid and velocity cross derivative terms for compressible fluid.
*       The heat conduction follows Fourier Law; the heat flux is proportional 
*       to temperaure gradient.  We assume the molecular viscosity coefficient
*       mu and the second viscosity coefficient lambda have the relation      
*       lambda = -2/3 mu by Stokes' hypothesis to simplify the equations.
*       
*       This file provides the finite difference loop to calculate the Laplace
*       and cross derivative terms of velocity by using the state data from
*       Euler equation solver.  Also the heat conduction term of temperature
*       gradient.
*                                        
*/

#include <ghyp/ghyp.h>
#include <ghyp/ghypprotos.h>
#include <gdecs/gdecs.h>
#include <gdecs/geos.h>
#include <gdecs/gstate.h>

#define  twothird  0.666666666666666666666667     /* value of (2.0/3.0) */

typedef struct {
        int             npts;  /* n point stencil, not used now */
        float           dt;
#if defined ONED
        int             **icoords1d;
        Locstate        *sts1d;
#endif /* ONED */
#if defined TWOD
        int             ***icoords2d;
        Locstate        **sts2d;
#endif /* TWOD */
#if defined THREED
        int             ****icoords3d;
        Locstate        ***sts3d;
#endif /* THREED */
        Locstate        ans;
        Front           *fr, *newfr;
        struct _Wave    *wave, *newwave;
} Pstencil;    /* stencil for parabolic solvers */

	/* LOCAL Function Declarations */
LOCAL  bool    g_compute_NS_terms(Locstate);
LOCAL  float   Fourier_heat_conduction(Pstencil*);
LOCAL  void    g_ns_soln(Pstencil*);
LOCAL  void    fill_2d_9pt_Pstencil(int,int,Pstencil*,Front*,Wave*);
LOCAL  int     g_neumann_bdry_state_beta(float*,COMPONENT,POINT*,
			HYPER_SURF*,Front*,POINTER,Locstate);

#if defined(CONSERVATIVE_ALG)
LOCAL  int     contrl_vol_parab_driver(float,float*,Wave*,Front*);  
LOCAL  void    vol_top_stat_with_tri(Wave*,Front*);
LOCAL  void    contrl_vol_parab_npt(float,Front*,Wave*,Wave*); 
LOCAL  int     vol_in_ns_sten_region(CSG_Solid*,RECT_GRID*,int*,int*,int*,int*,int*); 
#endif /* if defined(CONSERVATIVE_ALG) */

/*
*			parab_driver():
*/

EXPORT	int parab_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front)
{
	Wave		*newwave;
	int		i, dim = front->interf->dim;
	int             *iperm; /* permutation of {0,...,dim-1} */
	static char	warn[] = "WARNING in parab_driver()";
	static char	err[] = "ERROR in parab_driver()";
	DEBUG_ENTER(parab_driver)

	debug_print("parab","Entered parab_driver()\n");

#if defined(CONSERVATIVE_ALG)
        if(wave_tri_soln(wave)->tri_grid->Volume.blk != NULL)
        {
            return contrl_vol_parab_driver(dt,dt_frac,wave,front); 
        }
#endif /* if defined(CONSERVATIVE_ALG) */

	debug_front("front","in parabolic solver",front);
	if( debugging("parab_states") )
	{
	    (void) printf("States before calling parab_solver:\n\n");
	    (void) printf("Front\n");
	    graph_front_states(front);
	    (void) printf("wave %p\n",(POINTER)wave);
	    (*wave->show_wave_states)(wave);
	}

			/* parabolic solver */

	iperm = set_iperm(front->step,dim);

		/* Initialize Intermediate Storage for States */

	newwave = copy_wave(wave);
	assign_wave_parameters(newwave,wave);
	if( !copy_hyp_solution_function(wave,newwave) )
	{
	    screen("%s, copy_hyp_solution_function() failed\n",err);
	    free_wave(newwave);
	    DEBUG_LEAVE(parab_driver)
	    return ERROR_IN_STEP;
	}

	start_clock("parab_solver");

        parab_npt(dt,front,wave,newwave);

	start_clock("scatter_states");

	iperm = set_iperm(front->step,dim);
        for (i = 0; i < dim; i++)
        {
            if (!scatter_states(newwave,front,iperm,i))
            {
                screen("scatter_states() failed in direction\n",i);
                clean_up(ERROR);
            }
        }

	stop_clock("scatter_states");

		/* Copy updated wave */

	assign_copy_wave_pointers(wave,newwave);
	free_wave(newwave);

	stop_clock("parab_solver");
	if( debugging("parab_states") )
	{
	    if (wave->show_tri_soln)
		(*wave->show_tri_soln)(front,wave);
	    (*wave->show_wave_states)(wave);
	}

	debug_print("parab","Left parab_driver()\n");
	DEBUG_LEAVE(parab_driver)
	return GOOD_STEP;
}		/*end parab_driver*/

        /* for parabolic solver */

EXPORT	void    parab_npt(
        float   dt,
        Front   *front,
        Wave    *wv,
        Wave    *newwv)
{
        RECT_GRID       *rgr = wv->rect_grid;
        int             dim = rgr->dim;
        int             *gmax = rgr->gmax;
        int             i,j,k;
        int             icoords[MAXD];
        static Pstencil *nsten;
        int             npts,mid;
        Locstate        ans;
        int             imin[MAXD],imax[MAXD];
	int		*iperm;
	size_t		sizest = front->sizest;

	debug_print("parab","Entered parab_npt()\n");
        if (nsten == NULL)
        {
            stat_scalar(&nsten,sizeof(Pstencil));
            switch (dim)
            {
            case 1:
                matrix(&nsten->icoords1d,3,1,INT);
                vector(&nsten->sts1d,3,sizeof(Locstate));
                break;
            case 2:
                tri_array(&nsten->icoords2d,3,3,2,INT);
                matrix(&nsten->sts2d,3,3,sizeof(Locstate));
                for (i = 0; i < 3; i++)
                {
                    for (j = 0; j < 3; j++)
                    {
                        alloc_state(front->interf,&nsten->sts2d[i][j],
                                front->sizest);
                    }
                }
                break;
            case 3:
                quad_array(&nsten->icoords3d,3,3,3,3,INT);
                tri_array(&nsten->sts3d,3,3,3,sizeof(Locstate));
                break;
            }
        }
        nsten->fr = front;
        nsten->wave = wv;
        nsten->newwave = newwv;
        nsten->npts = npts = wv->npts_sten;
        mid = (int) (npts/2);

        for (i = 0; i < dim; i++)
        {
            imin[i] = 0;        imax[i] = gmax[i];
        }
        switch (dim)
        {
#if defined ONED
        case 1:
        {
            int i0;

            for (i = imin[0]; i < imax[0]; i++)
            {
                icoords[0] = i;
		assign(Rect_state(icoords,newwv),
		       Rect_state(icoords,wv),sizest);
		/*
                for (i0 = 0; i0 < 3; i0++)
                {
                    nsten->icoords1d[i0][0] = i + i0 - 1;
                }
                for (i0 = 0; i0 < 3; i0++)
                {
                    nsten->sts1d[i0] =
                        Rect_state(nsten->icoords1d[i0],wv);
                }
                icoords[0] = i;
                nsten->ans = Rect_state(icoords,newwv);
		*/
            }
        }
        break;

#endif /* defined ONED */
#if defined TWOD
        case 2:
        {
	    int i0, i1, ixl, ixr, iy, icl[2], icr[2];

	    float *h, *L, *U, coordsl[2], coordsr[2], *coords_l, *coords_r; 

            for (j = imin[1]; j < imax[1]; j++)
            {
                icoords[1] = j;
                for (i = imin[0]; i < imax[0]; i++)
                {
                    icoords[0] = i;
		    assign(Rect_state(icoords,newwv),
			   Rect_state(icoords,wv),sizest);
		    if(g_compute_NS_terms(Rect_state(icoords,wv)))
		    {
                    	fill_2d_9pt_Pstencil(i,j,nsten,front,wv);
                    	nsten->ans = Rect_state(icoords,newwv);
		    	g_ns_soln(nsten);
		    }
                }
            }
        }
        break;

#endif /* defined TWOD */
#if defined THREED
        case 3:
        {
            int i0, i1, i2;

            for (i = imin[0]; i < imax[0]; i++)
            {
              icoords[0] = i;
              for (j = imin[1]; j < imax[1]; j++)
              {
                 icoords[1] = j;
                 for (k = imin[2]; k < imax[2]; k++)
                 {
                    icoords[2] = k;
		    assign(Rect_state(icoords,newwv),
		       	   Rect_state(icoords,wv),sizest);
		    /*
                    for (i0 = 0; i0 < 3; i0++)
                    {
                       for (i1 = 0; i1 < 3; i1++)
                       {
                          for (i2 = 0; i2 < 3; i2++)
                          {
                            nsten->icoords3d[i0][i1][i2][0] = i + i0 - 1;
                            nsten->icoords3d[i0][i1][i2][1] = j + i1 - 1;
                            nsten->icoords3d[i0][i1][i2][2] = k + i2 - 1;
                          }
                       }
                    }
                    for (i0 = 0; i0 < 3; i0++)
                    {
                       for (i1 = 0; i1 < 3; i1++)
                       {
                          for (i2 = 0; i2 < 3; i2++)
                          {
                            nsten->sts3d[i0][i1][i2] =
                                Rect_state(nsten->icoords3d[i0][i1][i2],wv);
                          }
                       }
                    }
                    nsten->ans = Rect_state(icoords,newwv);
		    */
                 }
              }
            }
        }
        break;

#endif /* defined THREED */
        }
	debug_print("parab","Left parab_npt()\n");
}	/* end parab_npt */


LOCAL void fill_2d_9pt_Pstencil(
        int             i,
        int             j,
        Pstencil        *nsten,
	Front		*front,
        Wave            *wv)
{
        static int      ii, jj, icoords[2];
        static COMPONENT **comp, cc;
        static Locstate st1, st2, st3, st4, st5, st6, st7, st8;
        size_t          sizest = nsten->fr->sizest;
	int             st_typ;
	Locstate	state;
	
        if (st1 == NULL)   
        {
	    alloc_state(nsten->fr->interf,&st1,sizest);
	    alloc_state(nsten->fr->interf,&st2,sizest);
	    alloc_state(nsten->fr->interf,&st3,sizest);
	    alloc_state(nsten->fr->interf,&st4,sizest);
	    alloc_state(nsten->fr->interf,&st5,sizest);
	    alloc_state(nsten->fr->interf,&st6,sizest);
	    alloc_state(nsten->fr->interf,&st7,sizest);
	    alloc_state(nsten->fr->interf,&st8,sizest);
        }

        icoords[0] = i;
        icoords[1] = j;


        if (comp == NULL)
            stat_matrix(&comp,3,3,sizeof(COMPONENT));

        cc = Rect_comp(icoords,wv);

        for (jj = 0; jj < 3; jj++)
        {
            for (ii = 0; ii < 3; ii++)
            {
                nsten->icoords2d[ii][jj][0] = i + ii - 1;
                nsten->icoords2d[ii][jj][1] = j + jj - 1;
                comp[ii][jj] = Rect_comp(nsten->icoords2d[ii][jj],wv);
		state = nsten->sts2d[ii][jj];
                if (comp[ii][jj] == cc)
		{
		    assign(state,Rect_state(nsten->icoords2d[ii][jj],wv),
		    		sizest);
		}
                else
		{
		    RECT_GRID *gr = computational_grid(front->interf);
		    float coords[2] = {
			cell_center(nsten->icoords2d[ii][jj][0],0,gr),
			cell_center(nsten->icoords2d[ii][jj][1],1,gr)
		    };
		    HYPER_SURF *hs=0;		    

		    if (!is_excluded_comp(comp[ii][jj],front->interf))
		    {
			hyp_solution(coords,cc,hs,UNKNOWN_SIDE,front,
					wv,state,NULL);
		    }
		    else	
		    {
			HYPER_SURF_ELEMENT *hse = 0;
			float coords_on[2], t[2];

                        if(debugging("woh"))
                        {
                            static float alpha=0.0;
                            if(debugging("woh") && alpha==0.0)
                            {
                                CURVE **c;
                                for(c=front->interf->curves;*c && c;++c)
                                    alpha = max(adherence_coeff(*c),alpha);
                            }
                            if( (i==wv->rect_grid->gmax[0]-1 && j==0
                                 && ii==2 && jj ==0) ||
                                (i==wv->rect_grid->gmax[0]-1 &&
                                 j==wv->rect_grid->gmax[1]-1 &&
                                 ii==2 && jj==2))
                            {
                                int i, dim = front->rect_grid->dim;
                                int st_type;
                                float nor[3]={-1,0,0};
                                float vn;

                                assign(state,nsten->sts2d[1][jj],front->sizest);
                                st_type=state_type(state);
                                if(st_type != TGAS_STATE)
                                    set_state(state,TGAS_STATE,state);
                                vn = scalar_product(Vel(state),nor,dim);
                                for (i = 0; i < dim; ++i)
                                    Vel(state)[i] = (1-2*alpha)*Vel(state)[i];

                                zero_normal_velocity(state,nor,dim);
                                for (i = 0; i < dim; ++i)
                                    Vel(state)[i] += - vn * nor[i];
                                if(st_type != TGAS_STATE)
                                    set_state(state,st_type,state);
                                continue;
                            }
                        }

			/* Call nearest_interface_point() to find hs */

			if (!nearest_interface_point(coords,cc,front->interf,
				     	NO_SUBDOMAIN,NULL,coords_on,
					t,&hse,&hs))
			{
			    screen("ERROR in fill_2d_9pt_Pstencil(), "
				   "can't find nearest interface point\n");
			    clear_state(front->interf,state,front->sizest);
			    clean_up(ERROR);
			}
		    
			switch (wave_type(hs))
			{

			case CONTACT:
			    if(debugging("print_wave_type"))
			    {
			        printf("CONTACT in fill_2d_9pt_stencil_new(), "
				     "cc = %d, comp[%d][%d] = %d,"
				     " coord = (%g,%g)\n" ,cc,ii,jj,
				     comp[ii][jj],coords[0],coords[1]);
			    }
			    hyp_solution(coords,cc,hs,UNKNOWN_SIDE,front,
			    		wv,state,NULL);
			    break;

			case NEUMANN_BOUNDARY:
			    g_neumann_bdry_state_beta(coords,cc,0,hs,front,
			    		(POINTER)wv,state);
			    break;

			case DIRICHLET_BOUNDARY:    /* far field conditions */
			    evaluate_dirichlet_boundary_state(coords,hs,
			    		front,wv,state);
			    break;
			case PASSIVE_BOUNDARY:			
			case SUBDOMAIN_BOUNDARY:
			default:
			    screen("ERROR in fill_2d_9pt_stencil_new(), "
				   "unknown boundary type\n");
			    screen("\twave type = %s\n"
				   ,wave_type_as_string(wave_type(hs)
							,hs->interface));
			    screen("\n\tcoord = (%g,%g)\n",coords[0],coords[1]);
			    clean_up(ERROR);
			    break;
			}
		    }
		}
	    }
	}
}       /* end fill_2d_9pt_stencil_new */
LOCAL  void  g_ns_soln(
	Pstencil        *nsten) 
{
	RECT_GRID       *gr = nsten->wave->rect_grid;
	int             dim = gr->dim; 
	float		dt = nsten->fr->dt;
	float           mu;    /* NS viscosity coefficient */
	float           kappa; /* NS thermal conduction coeff */
	Locstate	ans = nsten->ans;
	float           *coords;
	int             i,icoords[MAXD];
	
/*
 *      The state array sts3d[3][3][3] is a moving cube in the general three
 *      dimensional space, but also could be applied in 2D and 1D space.
 *      For the first, second, and cross derivatives of velocity and Laplacian,
 *      three point stencil will produce second order accuracy computation.
 *      s[1][1][1] is the center of the cube, which is moving through all
 *      computational grids to get all velocity derivatives of all grids.
 */

	if (is_obstacle_state(ans))
	  return;

	mu = shear_viscosity(ans);
	kappa = heat_coeff(ans);

	for (i = 0; i < dim; i++)
	{
	    icoords[i] = nsten->icoords2d[1][1][i];
	}
	coords = Rect_coords(icoords,nsten->wave);
	set_max_viscosity(mu/density(ans),ans,coords,nsten->wave);
	
	switch (dim)
        {
#if defined ONED
        case 1:
        {
            float           ux;    /* first derivative of vel u */
            float           uxx;   /* second derivative of vel u */
	    float		dh[1];

            dh[0] = nsten->wave->rect_grid->h[0];

            ux = (vel(0,nsten->sts1d[2]) - vel(0,nsten->sts1d[0]))/(2.0*dh[0]);

            uxx = (vel(0,nsten->sts1d[2]) - 2.0*vel(0,nsten->sts1d[1])
                   + vel(0,nsten->sts1d[0]))/(dh[0]*dh[0]);

            Mom(ans)[0] += dt * mu *(twothird*(2.0*uxx));

	    Energy(ans) += dt * (kappa*Fourier_heat_conduction(nsten)
				 + mu*twothird*2.0*(sqr(ux) + 
				   vel(0,nsten->sts1d[1])*uxx));
	}
	break;
	 
#endif /* defined ONED */
#if defined TWOD
	case 2:
	{
	    float           ux, uy;   /* first derivative of vel u */
            float           vx, vy;   /* first derivative of vel v */
            float           uxy;      /* cross derivative of vel u */
            float           vxy;      /* cross derivative of vel v */
            float           uxx, uyy; /* second derivative of vel u */
            float           vxx, vyy; /* second derivative of vel v */
 	    float           dh[2];
 	    float           invs_dh[2];
 	    float           invsq_dh[2];
 	    float           cross_dh;
      
            dh[0] = nsten->wave->rect_grid->h[0];

	    invs_dh[0] = 1.0/(2.0*dh[0]);     /* inverse of 2.0*dh[0] */
    	
	    invsq_dh[0] = 1.0/(sqr(dh[0])); 

            dh[1] = nsten->wave->rect_grid->h[1];

	    invs_dh[1] = 1.0/(2.0*dh[1]);     /* inverse of 2.0*dh[1] */
	
	    invsq_dh[1] = 1.0/(sqr(dh[1]));

	    cross_dh = 1.0/(4.0*dh[0]*dh[1]); 


            if(debugging("ns_soln"))
	    {
	        (void) printf("sts2d[0][0] = %d\n",nsten->sts2d[0][0]);
                verbose_print_state("sts2d[0][0]",nsten->sts2d[0][0]);
	        (void) printf("sts2d[1][0] = %d\n",nsten->sts2d[1][0]);
                verbose_print_state("sts2d[1][0]",nsten->sts2d[1][0]);
	        (void) printf("sts2d[2][0] = %d\n",nsten->sts2d[2][0]);
	        (void) printf("icoords2d[2][0]=(%d,%d)\n",
			      nsten->icoords2d[2][0][0],
			      nsten->icoords2d[2][0][1]);
                verbose_print_state("sts2d[2][0]",nsten->sts2d[2][0]);
	        (void) printf("sts2d[0][1] = %d\n",nsten->sts2d[0][1]);
                verbose_print_state("sts2d[0][1]",nsten->sts2d[0][1]);
	        (void) printf("sts2d[1][1] = %d\n",nsten->sts2d[1][1]);
                verbose_print_state("sts2d[1][1]",nsten->sts2d[1][1]);
	        (void) printf("sts2d[2][1] = %d\n",nsten->sts2d[2][1]);
                verbose_print_state("sts2d[2][1]",nsten->sts2d[2][1]);
	        (void) printf("sts2d[0][2] = %d\n",nsten->sts2d[0][2]);
                verbose_print_state("sts2d[0][2]",nsten->sts2d[0][2]);
	        (void) printf("sts2d[1][2] = %d\n",nsten->sts2d[1][2]);
                verbose_print_state("sts2d[1][2]",nsten->sts2d[1][2]);
	        (void) printf("sts2d[2][2] = %d\n",nsten->sts2d[2][2]);
                verbose_print_state("sts2d[2][2]",nsten->sts2d[2][2]);
	    }
            /*** velocity_first_derivative ***/

            ux = (vel(0,nsten->sts2d[2][1]) - vel(0,nsten->sts2d[0][1]))*invs_dh[0];

            uy = (vel(0,nsten->sts2d[1][2]) - vel(0,nsten->sts2d[1][0]))*invs_dh[1];

            vx = (vel(1,nsten->sts2d[2][1]) - vel(1,nsten->sts2d[0][1]))*invs_dh[0];

            vy = (vel(1,nsten->sts2d[1][2]) - vel(1,nsten->sts2d[1][0]))*invs_dh[1];

	    if(debugging("ns_soln"))
                (void) printf("ux = %lf, uy = %lf, vx = %lf, vy = %lf\n",ux,uy,vx,vy); 
    
	    /*** velocity_cross_derivative ***/

            uxy = (vel(0,nsten->sts2d[2][2]) - vel(0,nsten->sts2d[2][0])
                  -vel(0,nsten->sts2d[0][2]) + vel(0,nsten->sts2d[0][0]))
                  *cross_dh;

            vxy = (vel(1,nsten->sts2d[2][2]) - vel(1,nsten->sts2d[2][0])
                  -vel(1,nsten->sts2d[0][2]) + vel(1,nsten->sts2d[0][0]))
                  *cross_dh;
	
	    if(debugging("ns_soln"))
                (void) printf("uxy = %lf, vxy = %lf\n",uxy,vxy); 
    
            /* uxy = vxy = 0.0;  Temporary setting */

	    /*** velocity_second_derivative ***/ 
            uxx = (vel(0,nsten->sts2d[2][1]) - 2.0*vel(0,nsten->sts2d[1][1])
                   + vel(0,nsten->sts2d[0][1]))*invsq_dh[0];

            uyy = (vel(0,nsten->sts2d[1][2]) - 2.0*vel(0,nsten->sts2d[1][1])
                   + vel(0,nsten->sts2d[1][0]))*invsq_dh[1];

            vxx = (vel(1,nsten->sts2d[2][1]) - 2.0*vel(1,nsten->sts2d[1][1])
                   + vel(1,nsten->sts2d[0][1]))*invsq_dh[0];

            vyy = (vel(1,nsten->sts2d[1][2]) - 2.0*vel(1,nsten->sts2d[1][1])
                   + vel(1,nsten->sts2d[1][0]))*invsq_dh[1];
	
	    if (debugging("ns_soln"))
	    {
                (void) printf("uxx = %lf, uyy = %lf, vxx = %lf, vyy = %lf\n",
				    uxx,uyy,vxx,vyy); 
	    }

            Mom(ans)[0] += dt * mu * (twothird*(2.0*uxx - vxy) + (uyy + vxy));

	    if(debugging("ns_soln"))
                (void) printf("Mom(ans)[0] adds %lf\n",
			    dt*mu*(2/3*(2*uxx-vxy) + (uyy+vxy)));  

            Mom(ans)[1] += dt * mu * (twothird*(2.0*vyy - uxy) + (vxx + uxy));

	    if(debugging("ns_soln"))
                (void) printf("Mom(ans)[1] adds %lf\n",
			      dt*mu*(2/3*(2*vyy-uxy) + (vxx+uxy)));  


	    Energy(ans) += dt * (kappa*Fourier_heat_conduction(nsten) 
			 + mu*(twothird*(ux*(2.0*ux - vy) + vy*(2.0*vy - ux)) 
			 + sqr(uy+vx)
			 + vel(0,nsten->sts2d[1][1])*(twothird*(2.0*uxx - vxy)
			 + (uyy + vxy)) 
			 + vel(1,nsten->sts2d[1][1])*(twothird*(2.0*vyy - uxy)
			 + (vxx + uxy))));

	    if(debugging("ns_soln"))
	    {
        	(void) printf("txx + tyy = %lf\n",
			      Fourier_heat_conduction(nsten));

        	(void) printf("Energy(ans) adds %lf\n",
		         dt * (kappa*Fourier_heat_conduction(nsten)
                         + mu*(2/3*(ux*(2*ux - vy) + vy*(2*vy - ux))
                         + sqr(uy+vx)
                         + vel(0,nsten->sts2d[1][1])*(2/3*(2*uxx - vxy)
                         + (uyy + vxy))
                         + vel(1,nsten->sts2d[1][1])*(2/3*(2*vyy - uxy)
                         + (vxx + uxy))
                         ))); 
                verbose_print_state("sts2d[0][0]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[1][0]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[2][0]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[0][1]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[1][1]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[2][1]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[0][2]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[1][2]",nsten->sts2d[1][1]); 
                verbose_print_state("sts2d[2][2]",nsten->sts2d[1][1]); 
                verbose_print_state("NS_ans",ans); 
	    }
	    /* Add 1/r terms for cylindrical geometry */
	    /* : d/d(phi) = 0, all phi-component = 0  */
	    if (gr->Remap.remap == CYLINDRICAL_REMAP)
	    {
	      	float r = coords[0];
	      	float u = vel(0,nsten->sts2d[1][1]);
              	float v = vel(1,nsten->sts2d[1][1]);
	      	Mom(ans)[0] += dt*mu*2*twothird/r*(ux - u/r);
	      	Mom(ans)[1] += dt*mu/r*(vx + uy/3);
              	Energy(ans) += dt*mu*(4*(u*ux/r  -  u*u/(r*r))/3
                                    + v*uy/(3*r) + v*vx/r);
	    }
	}
	break;
	
#endif /* defined TWOD */
#if defined THREED
	case 3:
	{
	    float           ux, uy, uz;      /* first derivative of vel u */
	    float           vx, vy, vz;      /* first derivative of vel v */
	    float           wx, wy, wz;      /* first derivative of vel w */
	    float           uxy, uyz, uzx;   /* cross derivative of vel u */
	    float           vxy, vyz, vzx;   /* cross derivative of vel v */
	    float           wxy, wyz, wzx;   /* cross derivative of vel w */
	    float           uxx, uyy, uzz;   /* second derivative of vel u */
	    float           vxx, vyy, vzz;   /* second derivative of vel v */
	    float           wxx, wyy, wzz;   /* second derivative of vel w */
	    float           dh[3];

	    /*** velocity_first_derivative ***/


	    ux = (vel(0,nsten->sts3d[2][1][1]) - vel(0,nsten->sts3d[0][1][1]))
	         /(2.0*dh[0]);

	    uy = (vel(0,nsten->sts3d[1][2][1]) - vel(0,nsten->sts3d[1][0][1]))
	         /(2.0*dh[1]);

	    uz = (vel(0,nsten->sts3d[1][1][2]) - vel(0,nsten->sts3d[1][1][0]))
	         /(2.0*dh[2]);

	    vx = (vel(1,nsten->sts3d[2][1][1]) - vel(1,nsten->sts3d[0][1][1]))
	         /(2.0*dh[0]);

	    vy = (vel(1,nsten->sts3d[1][2][1]) - vel(1,nsten->sts3d[1][0][1]))
	         /(2.0*dh[1]);

	    vz = (vel(1,nsten->sts3d[1][1][2]) - vel(1,nsten->sts3d[1][1][0]))
	         /(2.0*dh[2]);

	    wx = (vel(2,nsten->sts3d[2][1][1]) - vel(2,nsten->sts3d[0][1][1]))
	         /(2.0*dh[0]);

	    wy = (vel(2,nsten->sts3d[1][2][1]) - vel(2,nsten->sts3d[1][0][1]))
	         /(2.0*dh[1]);

	    wz = (vel(2,nsten->sts3d[1][1][2]) - vel(2,nsten->sts3d[1][1][0]))
	         /(2.0*dh[2]);


	    /*** velocity_cross_derivative ***/


	    uxy = (vel(0,nsten->sts3d[2][2][1]) - vel(0,nsten->sts3d[0][2][1])
	          -vel(0,nsten->sts3d[2][0][1]) + vel(0,nsten->sts3d[0][0][1]))/
	          (4.0*dh[0]*dh[1]);

	    uyz = (vel(0,nsten->sts3d[1][2][2]) - vel(0,nsten->sts3d[1][0][2])  
	          -vel(0,nsten->sts3d[1][2][0]) + vel(0,nsten->sts3d[1][0][0]))/ 
	          (4.0*dh[1]*dh[2]);

	    uzx = (vel(0,nsten->sts3d[2][1][2]) - vel(0,nsten->sts3d[2][1][0])  
	          -vel(0,nsten->sts3d[0][1][2]) + vel(0,nsten->sts3d[0][1][0]))/ 
	          (4.0*dh[2]*dh[0]);

	    vxy = (vel(1,nsten->sts3d[2][2][1]) - vel(1,nsten->sts3d[0][2][1])  
	          -vel(1,nsten->sts3d[2][0][1]) + vel(1,nsten->sts3d[0][0][1]))/   
	          (4.0*dh[0]*dh[1]);

	    vyz = (vel(1,nsten->sts3d[1][2][2]) - vel(1,nsten->sts3d[1][0][2])  
	          -vel(1,nsten->sts3d[1][2][0]) + vel(1,nsten->sts3d[1][0][0]))/   
	          (4.0*dh[1]*dh[2]);

	    vzx = (vel(1,nsten->sts3d[2][1][2]) - vel(1,nsten->sts3d[2][1][0])  
	          -vel(1,nsten->sts3d[0][1][2]) + vel(1,nsten->sts3d[0][1][0]))/   
	          (4.0*dh[2]*dh[0]);

	    wxy = (vel(2,nsten->sts3d[2][2][1]) - vel(2,nsten->sts3d[0][2][1])  
	          -vel(2,nsten->sts3d[2][0][1]) + vel(2,nsten->sts3d[0][0][1]))/   
	          (4.0*dh[0]*dh[1]);

	    wyz = (vel(2,nsten->sts3d[1][2][2]) - vel(2,nsten->sts3d[1][0][2])  
	          -vel(2,nsten->sts3d[1][2][0]) + vel(2,nsten->sts3d[1][0][0]))/   
	          (4.0*dh[1]*dh[2]);

	    wzx = (vel(2,nsten->sts3d[2][1][2]) - vel(2,nsten->sts3d[2][1][0])  
	          -vel(2,nsten->sts3d[0][1][2]) + vel(2,nsten->sts3d[0][1][0]))/   
	          (4.0*dh[2]*dh[0]);
	    
	    /* Temporary setting 
	    uxy = uyz = uzx = vxy = vyz = vzx = wxy = wyz = wzx = 0.0; */
	    /*** velocity_second_derivative ***/


	    uxx = (vel(0,nsten->sts3d[2][1][1]) - 2.0*vel(0,nsten->sts3d[1][1][1]) 
	               + vel(0,nsten->sts3d[0][1][1]))/(dh[0]*dh[0]);

	    uyy = (vel(0,nsten->sts3d[1][2][1]) - 2.0*vel(0,nsten->sts3d[1][1][1]) 
	               + vel(0,nsten->sts3d[1][0][1]))/(dh[1]*dh[1]);

	    uzz = (vel(0,nsten->sts3d[1][1][2]) - 2.0*vel(0,nsten->sts3d[1][1][1]) 
	               + vel(0,nsten->sts3d[1][1][0]))/(dh[2]*dh[2]);

	    vxx = (vel(1,nsten->sts3d[2][1][1]) - 2.0*vel(1,nsten->sts3d[1][1][1]) 
	               + vel(1,nsten->sts3d[0][1][1]))/(dh[0]*dh[0]);

	    vyy = (vel(1,nsten->sts3d[1][2][1]) - 2.0*vel(1,nsten->sts3d[1][1][1]) 
	               + vel(1,nsten->sts3d[1][0][1]))/(dh[1]*dh[1]);

	    vzz = (vel(1,nsten->sts3d[1][1][2]) - 2.0*vel(1,nsten->sts3d[1][1][1]) 
	               + vel(1,nsten->sts3d[1][1][0]))/(dh[2]*dh[2]);

	    wxx = (vel(2,nsten->sts3d[2][1][1]) - 2.0*vel(2,nsten->sts3d[1][1][1]) 
	               + vel(2,nsten->sts3d[0][1][1]))/(dh[0]*dh[0]);

	    wyy = (vel(2,nsten->sts3d[1][2][1]) - 2.0*vel(2,nsten->sts3d[1][1][1]) 
	               + vel(2,nsten->sts3d[1][0][1]))/(dh[1]*dh[1]);
	
	    wzz = (vel(2,nsten->sts3d[1][1][2]) - 2.0*vel(2,nsten->sts3d[1][1][1]) 
	               + vel(2,nsten->sts3d[1][1][0]))/(dh[2]*dh[2]);


	    Mom(ans)[0] += dt * mu * (twothird*(2.0*uxx - vxy - wzx)
 		     + (uyy + vxy) + (uzz + wzx));

	    Mom(ans)[1] += dt * mu * (twothird*(2.0*vyy - uxy - wyz)
	    	     + (vxx + uxy) + (vzz + wyz));

	    Mom(ans)[2] += dt * mu * (twothird*(2.0*wzz - uzx - vyz)
	    	     + (wxx + uzx) + (wyy + vyz));

	    Energy(ans) += dt * (kappa*Fourier_heat_conduction(nsten) 
	    		       + mu*(twothird*(ux*(2.0*ux - vy - wz)
	    				       + vy*(2.0*vy - ux - wz)
	    				       + wz*(2.0*wz - ux - vy))
	    			     + sqr(uy+vx) + sqr(wx+uz) + sqr(vz+wy)
	    			     + vel(0,nsten->sts3d[1][1][1])
	    			     *(twothird*(2.0*uxx - vxy - wzx) 
	    			       + (uyy + vxy) + (uzz + wzx))
	    			     + vel(1,nsten->sts3d[1][1][1])
	    			     *(twothird*(2.0*vyy - uxy - wyz) 
	    			       + (vxx + uxy) + (vzz + wyz))
	    			     + vel(2,nsten->sts3d[1][1][1])
	    			     *(twothird*(2.0*wzz - uzx - vyz) 
	    			       + (wxx + uzx) + (wyy + vyz))
	    			     ));
	}
	break;
	 
#endif /* defined THREED */
	}
        /*printf("Leaving ns_soln()\n");*/
}


/*
 *      Fourier heat conduction is used in the energy equation in Navier-
 *	Stokes equations, and in fact is a Laplacian of temperature T. 
 *
*/

LOCAL float Fourier_heat_conduction(
	    Pstencil         *nsten)
{
	RECT_GRID       *gr = nsten->wave->rect_grid;
	int             dim = gr->dim;
	
        switch (dim)
        {
#if defined ONED
        case 1:
        { 
            float           txx;   /* second derivative of temp t */
	    float           dh[1];
	
	dh[0] = nsten->wave->rect_grid->h[0];

        txx = (temperature(nsten->sts1d[2]) - 2.0*temperature(nsten->sts1d[1])
               + temperature(nsten->sts2d[0]))/(dh[0]*dh[0]);

        return txx;
	}
	break;
         
#endif /* defined ONED */
#if defined TWOD
        case 2:
	{
            float           txx, tyy;   /* second derivative of temp t */
	    float           dh[2];

	dh[0] = nsten->wave->rect_grid->h[0];
	dh[1] = nsten->wave->rect_grid->h[1];

        txx = (temperature(nsten->sts2d[2][1]) 
               - 2.0*temperature(nsten->sts2d[1][1])
               + temperature(nsten->sts2d[0][1]))/(dh[0]*dh[0]);
	
        tyy = (temperature(nsten->sts2d[1][2]) 
               - 2.0*temperature(nsten->sts2d[1][1])
               + temperature(nsten->sts2d[1][0]))/(dh[1]*dh[1]);

        return (txx + tyy);
	}
	break;
         
#endif /* defined TWOD */
#if defined THREED
        case 3:
	{
	    float           txx, tyy, tzz;   /* second derivative of temp t */
	    float           dh[3];

	dh[0] = nsten->wave->rect_grid->h[0];
	dh[1] = nsten->wave->rect_grid->h[1];
	dh[2] = nsten->wave->rect_grid->h[2];

        txx = (temperature(nsten->sts3d[1][1][2]) 
               - 2.0*temperature(nsten->sts3d[1][1][1]) 
	       + temperature(nsten->sts3d[1][1][0]))/(dh[0]*dh[0]);

	tyy = (temperature(nsten->sts3d[1][2][1])
               - 2.0*temperature(nsten->sts3d[1][1][1]) 
	       + temperature(nsten->sts3d[1][0][1]))/(dh[1]*dh[1]);

	tzz = (temperature(nsten->sts3d[2][1][1])
               - 2.0*temperature(nsten->sts3d[1][1][1]) 
	       + temperature(nsten->sts3d[0][1][1]))/(dh[2]*dh[2]);

 	return (txx + tyy + tzz);
	}
	break;
         
#endif /* defined THREED */
        }
}

LOCAL  bool  g_compute_NS_terms(
	      Locstate        state) 
{
  	if ( (Params(state) == NULL) ||
       	    (Params(state)->eos->_compute_ns_terms == NO))
            return NO;
  	else if (Params(state)->eos->_compute_ns_terms == YES)
       	    return YES;
  	return NO;
}

	/* Use only in parabolic step */
LOCAL	int g_neumann_bdry_state_beta(
	float		*coords,
	COMPONENT	int_comp,
	POINT		*pt,
	HYPER_SURF	*Nbdry,
	Front		*front,
	POINTER		p2wave,
	Locstate	state)
{
	Wave		*wave = (Wave*)p2wave;
	float		nor[MAXD];
	float		coords_ref[MAXD];
	float		coords_tmp[MAXD];

	int		i, dim = front->rect_grid->dim;

	float           tmp_vel[MAXD], vn;

	/* coords_ref is the location of the reflection of the point coords
	   across the Neumann boundary. */

	if (debugging("no_nbdry") 
	    || !reflect_pt_about_Nbdry(coords,coords_ref,nor,
					  int_comp,Nbdry,front))
	    return NO;

	/* Get the state at coords_ref. */
	
	hyp_solution(coords_ref,int_comp,Nbdry,UNKNOWN_SIDE,
		     front,wave,state,NULL);

	set_state(state,TGAS_STATE,state);
	vn = 0.0;
	for (i = 0; i < dim; i ++)
	{
	    tmp_vel[i] = Vel(state)[i];
	    vn += tmp_vel[i] * nor[i];
	}

	if (no_slip(Nbdry))
        {
	    float alpha = adherence_coeff(Nbdry);
            for (i = 0; i < dim; i ++)
	    	Vel(state)[i] = (1-2*alpha)*Vel(state)[i];
	}

	zero_normal_velocity(state,nor,dim);
	for (i = 0; i < dim; i ++)
	    Vel(state)[i] += - vn * nor[i];
	set_state(state,GAS_STATE,state);
	return YES;
}		/*end g_neumann_bdry_state_beta*/


#if defined(CONSERVATIVE_ALG)
LOCAL	int contrl_vol_parab_driver(
	float		dt,
	float		*dt_frac,
	Wave		*wave,
	Front		*front)
{
	Wave		*newwave;
	int		i, dim = front->interf->dim;
        int             nstorage;  
	int             *iperm; /* permutation of {0,...,dim-1} */
        size_t          sizest = wave->sizest; 
        byte            *old_st_store, *new_st_store;
	static char	warn[] = "WARNING in contrl_vol_parab_driver()";
	static char	err[] = "ERROR in contrl_vol_parab_driver()";

	debug_print("parab","Entered contrl_vol_parab_driver()\n");

	if( debugging("parab_states") )
	{
	    (void) printf("States before calling contrl_vol_parab_solver:\n\n");
	    (void) printf("Front\n");
	    graph_front_states(front);
	    (void) printf("wave %p\n",(POINTER)wave);
	    (*wave->show_wave_states)(wave);
	}

			/* parabolic solver */

	iperm = set_iperm(front->step,dim);

		/* Initialize Intermediate Storage for States */

	newwave = copy_wave(wave);
	assign_wave_parameters(newwave,wave);
	if( !copy_hyp_solution_function(wave,newwave) )
	{
	    screen("%s, copy_hyp_solution_function() failed\n",err);
	    free_wave(newwave);
	    DEBUG_LEAVE(contrl_vol_parab_driver)
	    return ERROR_IN_STEP;
	}

        nstorage = init_vol_state_storage(wave_tri_soln(newwave)->tri_grid,
                      front->interf,sizest);
        new_st_store = wave_tri_soln(newwave)->tri_grid->vol_rect_state_storage;
        old_st_store = wave_tri_soln(wave)->tri_grid->vol_rect_state_storage;
        for(i = 0; i < nstorage; i++)
        {
            assign(new_st_store, old_st_store, sizest); 
            new_st_store += sizest;
            old_st_store += sizest;
        } 
        wave_tri_soln(newwave)->tri_grid->Volume.blk =
            wave_tri_soln(wave)->tri_grid->Volume.blk;
        vol_top_stat_with_tri(newwave,front);

	start_clock("contrl_vol_parab_solver");

        parab_npt(dt,front,wave,newwave);
        contrl_vol_parab_npt(dt,front,wave,newwave);

	start_clock("scatter_states");

	iperm = set_iperm(front->step,dim);
        for (i = 0; i < dim; i++)
        {
            if (!scatter_states(newwave,front,iperm,i))
            {
                screen("scatter_states() failed in direction\n",i);
                clean_up(ERROR);
            }
        }
        (*newwave->_scatter_frac_cell_states)(newwave, front);

	stop_clock("scatter_states");

		/* Copy updated wave */

        free_frac_state_storage(wave_tri_soln(wave)->tri_grid); 
        /*
        wave_tri_soln(wave)->tri_grid->vol_states = 
            wave_tri_soln(newwave)->tri_grid->vol_states;
        wave_tri_soln(wave)->tri_grid->vol_rect_state_storage = 
            wave_tri_soln(newwave)->tri_grid->vol_rect_state_storage;
        */

	assign_copy_wave_pointers(wave,newwave);
	free_wave(newwave);

	stop_clock("contrl_vol_parab_solver");
	if( debugging("parab_states") )
	{
	    if (wave->show_tri_soln)
		(*wave->show_tri_soln)(front,wave);
	    (*wave->show_wave_states)(wave);
	}

	debug_print("parab","Left contrl_vol_parab_driver()\n");
	return GOOD_STEP;
}		/*end contrl_vol_parab_driver*/

LOCAL void vol_top_stat_with_tri(
        Wave         *wv,
        Front        *fr)
{
        TRI_GRID      *tg = wave_tri_soln(wv)->tri_grid;
        int            i, j, smin[MAXD], smax[MAXD];
        int            ic[MAXD];
        CSG_BLK_CRX    *blk_crx;
        CSG_Solid      *s;


        for (i = 0; i < 2; i++)
        {
            smin[i] = 0;
            smax[i] = wv->rect_grid->gmax[i];
            smin[i] -= wv->rect_grid->lbuf[i];
            smax[i] += wv->rect_grid->ubuf[i];
        }

        for(i= smin[0]; i < smax[0]; i++)
            for(j = smin[1]; j < smax[1]; j++)
        {
            ic[0] = i; ic[1] = j;
            blk_crx = Comp_blk(ic,tg->Volume.blk,tg);
            if(blk_crx)
            {
                s = blk_crx->s;
                while(s)
                {
                    top_and_btm_stat_with_tri(ic,s,tg,NULL,2,fr->print_state);
                    s = s->nexts;
                }
            }
        }
}

LOCAL void    contrl_vol_parab_npt(
        float   dt,
        Front   *front,
        Wave    *wv,
        Wave    *newwv)
{
        RECT_GRID       *rgr = wv->rect_grid;
        int             dim = rgr->dim;
        int             *gmax = rgr->gmax;
        int             i,j,k;
        int             icoords[MAXD], ic[MAXD];
        static Pstencil *nsten;
        int             npts,mid;
        Locstate        ans;
        int             imin[MAXD],imax[MAXD];
        int             *iperm;
        size_t          sizest = front->sizest;
        TRI_GRID        *tg = wave_tri_soln(wv)->tri_grid;
        CSG_BLK_CRX     *blk_crx;
        CSG_Solid       *s;
        int             lbuffed[MAXD], ubuffed[MAXD];

        for (i = 0; i < dim; ++i)
        {
            imin[i] = 0;
            imax[i] = gmax[i];
            imin[i] -= rgr->lbuf[i];
            imax[i] += rgr->ubuf[i];
            if(rgr->lbuf[i] != 0)
                lbuffed[i] = YES;
            else
                lbuffed[i] = NO;
            if(rgr->ubuf[i] != 0)
                ubuffed[i] = YES;
            else
                ubuffed[i] = NO;
        }

        if (nsten == NULL)
        {
            stat_scalar(&nsten,sizeof(Pstencil));
            switch (dim)
            {
            case 2:
                tri_array(&nsten->icoords2d,3,3,2,INT);
                matrix(&nsten->sts2d,3,3,sizeof(Locstate));
                for (i = 0; i < 3; i++)
                {
                    for (j = 0; j < 3; j++)
                    {
                        alloc_state(front->interf,&nsten->sts2d[i][j],
                                front->sizest);
                    }
                }
                break;
            default:
               printf("ERROR contrl_vol_parab_npt,\n");
               printf("Implement %D case \n", dim); 
               clean_up(ERROR); 
            }
        }
        nsten->fr = front;
        nsten->wave = wv;
        nsten->newwave = newwv;
        nsten->npts = npts = wv->npts_sten;
        mid = (int) (npts/2);


        for (j = imin[1]; j < imax[1]; j++)
        {
            icoords[1] = j;
            for (i = imin[0]; i < imax[0]; i++)
            {
                icoords[0] = i;
                blk_crx = Comp_blk(icoords,tg->Volume.blk,tg);
                if(blk_crx)
                {
                    s = blk_crx->s;
                    while(s)
                    {
                        if(vol_in_ns_sten_region(s,rgr,imin,imax,lbuffed,ubuffed,ic))
                        {
                            assign(Rect_state(ic,newwv),
                               Rect_state(ic,wv),sizest);
                            if(g_compute_NS_terms(Rect_state(ic,wv)))
                            {
                                fill_2d_9pt_Pstencil(ic[0],ic[1],nsten,front,wv);
                                nsten->ans = Rect_state(ic,newwv);
                                g_ns_soln(nsten);
                                update_top_face_state(sizest,s,nsten->ans);
                            }
                            // else update_top_face_state(sizest,s,Rect_state(ic,wv));
                        } 
                        s = s->nexts;
                    }
                }
            }
        }
}

LOCAL int vol_in_ns_sten_region(
        CSG_Solid       *s,
        RECT_GRID       *rgr,
        int             *imin,
        int             *imax,
        int             *lbuffed,
        int             *ubuffed,
        int             *ic)
{
        CSG_Face     *f;
        int          i, where[3], t;
        float        t_centroid[3] = {0.0, 0.0, 0.0};        
        float        t_area = 0.0;

        f = s->sfaces;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 1)
                {
                    for(i = 0; i < 3; i++)
                        t_centroid[i] += f->centroid[i]*f->area;
                    t_area += f->area;
                    t++;
                }
            }
            f = f->nextf;
        }

        if(t != 0)
        {
            for(i = 0; i < 3; i++)
                t_centroid[i] /= t_area;
        }

        if(rect_in_which(t_centroid,ic, rgr) == FUNCTION_FAILED)
            return NO; 

        if(ic[0] < imin[0]+1 && lbuffed[0] == YES)
            return NO; 
        if(ic[1] < imin[1]+1 && lbuffed[1] == YES)
            return NO; 
        if(ic[0] >= imax[0] && ubuffed[0] == YES)
            return NO; 
        if(ic[1] >= imax[1] && ubuffed[1] == YES)
            return NO; 

        return YES; 
}

#endif /* if defined(CONSERVATIVE_ALG) */
