/*
*                               gunsplithypsub.c
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*/

#include <ghyp/ghyp.h>
#include <gdecs/vecdecs.h>

#if defined(CONSERVATIVE_ALG) && defined(TWOD)

LOCAL int debug_flag = NO; 
#define CHECK_FOR_BAD_STATES

LOCAL  void    unsplit_2d_MUSCL_in_dir(int,int*,int*,Wave*,Wave*,Front*,Front*,Stencil*,
                           Tan_stencil*,int,int,Vec_Gas*,Vec_Gas**,Vec_Src*,
                           float,float*,int,int);
LOCAL  TIME_DYNAMICS  g_load_unsplit_state_vectors(int,int*,Vec_Gas*,int,int,
                                           Wave*,Wave*,int*,int);
LOCAL  void    unsplit_g_assign_wave_state_vectors(int,int*,Wave*,Wave*,Vec_Gas*,
                              int,int,int*,int); 
LOCAL  void    g_load_muscl_flux2(int,int,int,int*,float**,Vec_Gas*,MUSCL_FLUX*,Vec_Muscl*);  
LOCAL  void    g_unsplit_muscl_flux_vectors(int,int,int,int*,float**,float*,MUSCL_FLUX*,
                                     Vec_Muscl*);
LOCAL  void    ggodunov_flux(Locstate,Locstate,int,float*,float*);
LOCAL  void    apply_limiter2(float**,int,int,Vec_Muscl*);
LOCAL  void    unsplit_left_multiply_state_by_matrix(float**,float***,float**,int,int,int); 
LOCAL  void    g_load_newucon_state_data(int,int*,int,int,int,Stencil*,Wave*,Wave*,Vec_Muscl*,
                        int*,int);
LOCAL  Vec_Muscl *g_unsplit_muscl_alloc_wk_space(Muscl_Opts*,int,int,Vec_Muscl*);
LOCAL  float   *g_set_unsplit_vmuscl_vec_gas_storage(float**,Vec_Gas*,Vec_Muscl*); 

LOCAL  void    g_2dunsplit_add_art_visc2(int,int,Vec_Muscl*); 



EXPORT void g_unsplit_muscl_alloc_phys_vecs(
        Wave            *wave,
        int             vs)
{
        RECT_GRID *gr = wave->rect_grid;
        int       dim = gr->dim;

        g_wave_vgas(wave) = g_alloc_vgas(g_wave_vgas(wave),vs,dim);
        g_wave_Tvgas(wave)[0] = g_alloc_vgas(g_wave_Tvgas(wave)[0],vs,dim);
        g_wave_Tvgas(wave)[1] = g_alloc_vgas(g_wave_Tvgas(wave)[1],vs,dim);
        g_wave_vsrc(wave) = muscl_alloc_vsrc(g_wave_vsrc(wave),vs,wave);

}               /*end g_unsplit_muscl_alloc_phys_vecs*/

EXPORT void g_unsplit_muscl_free_phys_vecs(
        Wave            *wave)
{
        g_free_vgas(g_wave_vgas(wave));
        g_wave_vgas(wave) = NULL;

        g_free_vgas(g_wave_Tvgas(wave)[0]);
        g_wave_Tvgas(wave)[0] = NULL;
        g_free_vgas(g_wave_Tvgas(wave)[1]);
        g_wave_Tvgas(wave)[1] = NULL;

        g_free_vsrc(g_wave_vsrc(wave));
        g_wave_vsrc(wave) = NULL;
}               /*end g_muscl_free_phys_vecs*/

/*
*                         ghyp_unsplit_vec()
*
*       The subroutine first loads the state variables into vst, the source
*       variables to src, then passes to the oned_interior_scheme().
*/

/*ARGSUSED*/
EXPORT  void ghyp_unsplit_vec(
        int             swp_num,
        int             *iperm,
        float           *dir,
        Wave            *wv,
        Wave            *newwv,
        Front           *fr,
        Front           *newfr,
        int             *icoords,
        int             imin,
        int             imax,
        float           dt,
        float           *dh)
{
        Vec_Gas         *vst = g_wave_vgas(wv);
        Vec_Gas         **Tvst = g_wave_Tvgas(wv);
        Vec_Src         *src = g_wave_vsrc(wv);
        int             idir = iperm[swp_num];
        int             dim = wv->rect_grid->dim;
        int             nrad = vsten_radius(wv);
        int             vsize, i, icrds[MAXD];
        int             j, idirs[MAXD];  
        static float    **Q = NULL;
#if defined(ROTATIONAL_SYMMETRY)
        static  float   alpha;
        static  bool    first = YES;
#endif /* defined(ROTATIONAL_SYMMETRY) */

#if defined(TIME_HYPVEC)
        start_clock("ghyp_unsplit_vec");
#endif /* defined(TIME_HYPVEC) */

        debug_print("ghyp_unsplit_vec",
              "Entered ghyp_unsplit_vec(), dir = %d ic[%d, %d]\n",
              idir, icoords[0], icoords[1]);

#if defined(ROTATIONAL_SYMMETRY)
        if (first == YES)
        {
            first = NO;
            alpha = rotational_symmetry();
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        vsize = imax - imin;
        if (vsize < wv->npts_vsten)
        {
            debug_print("ghyp_unsplit_vec",
                  "Left ghyp_unsplit_vec(), dir = %d ic[%d, %d]\n",
                  idir, icoords[0], icoords[1]);
             return;
        }

        // g_load_state_vectors;  
        if (g_load_unsplit_state_vectors(swp_num,iperm,vst,0,vsize,
                   wv,newwv,icoords,imin) == CONSTANT_IN_TIME)
        {
#if defined(TIME_HYPVEC)
            stop_clock("ghyp_unsplit_vec");
#endif /* defined(TIME_HYPVEC) */
            debug_print("ghyp_unsplit_vec",
                  "Left ghyp_unsplit_vec(), dir = %d ic[%d, %d]\n",
                  idir, icoords[0], icoords[1]);
            return;
        }

        for (i = 0; i < dim; ++i)
        {
            idirs[i] = iperm[(i+swp_num)%dim];
            icrds[i] = icoords[i];  
        }
        if (Q == NULL)
            matrix(&Q,3,3,FLOAT);
        Q[0][0] = Q[0][1] = Q[0][2] = 0.0;
        Q[1][0] = Q[1][1] = Q[1][2] = 0.0;
        Q[2][0] = Q[2][1] = Q[2][2] = 0.0;
        for (i = 0; i < dim; ++i)
            Q[i][idirs[i]] = 1.0;
        for (; i < 3; ++i)
            Q[i][i] = 1.0;
        vst->Q = (const float* const*)Q;

        icrds[idirs[1]] = icoords[idirs[1]] - 1;  
        if (g_load_unsplit_state_vectors(swp_num,iperm,Tvst[0],0,vsize,
                        wv,newwv,icrds,imin) == CONSTANT_IN_TIME)
        {
#if defined(TIME_HYPVEC)
            stop_clock("ghyp_unsplit_vec");
#endif /* defined(TIME_HYPVEC) */
            debug_print("ghyp_unsplit_vec",
                  "Left ghyp_unsplit_vec(), dir = %d ic[%d, %d]\n",
                  idir, icoords[0], icoords[1]);
            return;
        }
 
        icrds[idirs[1]] = icoords[idirs[1]] + 1;  
        if (g_load_unsplit_state_vectors(swp_num,iperm,Tvst[1],0,vsize,
                      wv,newwv,icrds,imin) == CONSTANT_IN_TIME)
        {
#if defined(TIME_HYPVEC)
            stop_clock("ghyp_unsplit_vec");
#endif /* defined(TIME_HYPVEC) */
            debug_print("ghyp_unsplit_vec",
                  "Left ghyp_unsplit_vec(), dir = %d ic[%d, %d]\n",
                  idir, icoords[0], icoords[1]);
            return;
        }

#if defined(ROTATIONAL_SYMMETRY)
        if ( (alpha > 0.0) && (iperm[swp_num] == 0))
        {
            RECT_GRID *rgr = wv->rect_grid;
            float     *radii = src->radii;
            int       j;

            src->rmin = pos_radius(0.0,rgr);
            for (j = 0; j < vsize; j++)
                radii[j] = pos_radius(cell_center(j+imin,0,rgr),rgr);
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        unsplit_2d_MUSCL_in_dir(swp_num,iperm,icoords,wv,newwv,fr,newfr,
                 NULL,NULL,0,vsize,vst,Tvst,src,dt,dh,dim,imin);  

        unsplit_g_assign_wave_state_vectors(swp_num,iperm,wv,newwv,vst,nrad,
                                  vsize-nrad,icoords,imin);

        debug_print("ghyp_unsplit_vec",
              "Left ghyp_unsplit_vec(), dir = %d ic[%d, %d]\n",
              idir, icoords[0], icoords[1]);
}


/*
*                       unsplit_g_assign_wave_state_vectors():
*
*       See comments preceding g_load_state_vectors.  This is the function that
*       loads the Vec_Gas back into the wave structure after the vector
*       solver.
*       Also, we need to make sure we don't reset obstacle states (see
*       g_init_obstacle_states() ).
*       The difference between this function and g_assign_wave_state_vectors()
*       is that the Mom is not permuted in this call. 
*/

LOCAL  void unsplit_g_assign_wave_state_vectors(
        int             swp_num,
        int             *iperm,
        Wave            *wv,
        Wave            *newwv,
        Vec_Gas         *vst,
        int             imin,
        int             imax,
        int             *icoords,
        int             pbuf)
{
        int             i, k;
        int             dim = wv->rect_grid->dim;
        Locstate        state;
        int             idirs[MAXD];  

        debug_print("ghyp_unsplit_vec",
          "Entered unsplit_g_assign_wave_state_vectors(),"
          " dir = %d ic[%d, %d]\n",
          iperm[swp_num], icoords[0], icoords[1]);

        for (i = 0; i < dim; ++i)
            idirs[i] = iperm[(i+swp_num)%dim];

        for (i = imin; i < imax; ++i)
        {
            icoords[idirs[0]] = i + pbuf;
            if (is_obstacle_state(Rect_state(icoords,wv)))
                g_obstacle_state(Rect_state(icoords,newwv),wv->sizest);
            else
            {
                state = Rect_state(icoords,newwv);
                Dens(state) = vst->rho[i];
                for (k = 0; k < dim; ++k)
                    Mom(state)[k] = vst->m[k][i];
                Energy(state) = vst->en_den[i];
                Set_params(state,vst->state[i]);
                set_type_of_state(state,GAS_STATE);
#if defined(CHECK_FOR_BAD_STATES)
                if (debugging("bad_state") &&
                    is_bad_state(state,YES,"unsplit_g_assign_wave_state_vectors"))
                {
                    int j, icrds[3];
                    bool   regular_stencil;
                    COMPONENT comp0, comp;

                    comp0 = Rect_comp(icoords,wv);
                    for (j = 0; j < dim; ++j)
                        icrds[j] = icoords[j];
                    regular_stencil = NO;
                    for (j = 1; j <= wv->sten_rad; ++j)
                    {
                        icrds[idirs[0]] = icoords[idirs[0]] - j;
                        comp = Rect_comp(icrds,wv);
                        if (comp != comp0)
                        {
                            regular_stencil = NO;
                            break;
                        }
                        icrds[idirs[0]] = icoords[idirs[0]] + j;
                        comp = Rect_comp(icrds,wv);
                        if (comp != comp0)
                        {
                            regular_stencil = NO;
                            break;
                        }
                    }
                    if (regular_stencil == YES)
                    {
                        screen("ERROR in unsplit_g_assign_wave_state_vectors(), "
                               "bad state found on regular stencil\n");
                        (void) printf("bad state - ");
                        /*
                        fprint_raw_gas_data(stdout,state,
                                            current_interface()->dim);
                        */
                        g_verbose_print_state(state); 
                        print_int_vector("icoords = ",icoords,dim,"\n");
                        print_general_vector("coords = ",
                                             Rect_coords(icoords,wv),dim,"\n");
                        clean_up(ERROR);
                    }
                }
#endif /* defined(CHECK_FOR_BAD_STATES) */
            }
        }

        debug_print("ghyp_unsplit_vec",
          "Left unsplit_g_assign_wave_state_vectors(),"
          " dir = %d ic[%d, %d]\n",
          iperm[swp_num], icoords[0], icoords[1]);
}


/*
*                       g_load_unsplit_state_vectors():
*
*       This function loads the conservative state variables from the wave into
*       a Vec_Gas suitable for use by the vector solvers.  This function is
*       currently used only by the vector solvers, thus we make the assumption
*       that imin == 0.  This means there are offset artificial states
*       on either end of the Vec_Gas -- 0 to (offset-1) and (imax-offset) to
*       (imax-1).
*
*       To simply load a Vec_Gas from arbitray imin to arbitray imax,
*       just pass offset = 0.
*/

LOCAL TIME_DYNAMICS g_load_unsplit_state_vectors(
        int             swp_num,
        int             *iperm,
        Vec_Gas         *vst,
        int             imin,
        int             imax,
        Wave            *wv,
        Wave            *newwv,
        int             *icrds,
        int             pbuf)
{
        Locstate        rst;
        Locstate        *state = vst->state;
        float           **coords = vst->coords;
        float           *rho = vst->rho;
        float           *en_den = vst->en_den;
        float           *m[MAXD];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        float           *vacuum_dens = vst->vacuum_dens;
        float           *min_pressure = vst->min_pressure;
        float           *min_energy = vst->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        int             i, j;
        int             dim = wv->rect_grid->dim;
        int             idirs[MAXD];
        int             icoords[MAXD]; 
        // static float    **Q = NULL;

        clear_Vec_Gas_set_flags(vst);
        for (j = 0; j < dim; ++j)
        {
            idirs[j] = iperm[(j+swp_num)%dim];
            m[j] = vst->m[j];
            icoords[j] = icrds[j]; 
        }
        /*
        if (Q == NULL)
            matrix(&Q,3,3,FLOAT);
        Q[0][0] = Q[0][1] = Q[0][2] = 0.0;
        Q[1][0] = Q[1][1] = Q[1][2] = 0.0;
        Q[2][0] = Q[2][1] = Q[2][2] = 0.0;
        for (i = 0; i < dim; ++i)
            Q[i][idirs[i]] = 1.0;
        for (; i < 3; ++i)
            Q[i][i] = 1.0;
        vst->Q = (const float* const*)Q;
        */
        for (i = imin; i < imax; ++i)
        {
            icoords[idirs[0]] = i + pbuf;
            state[i] = rst = Rect_state(icoords,wv);
            coords[i] = Rect_coords(icoords,wv);
            rho[i] = Dens(rst);
            en_den[i] = Energy(rst);
            for (j = 0; j < dim; ++j)
                m[j][i] = Mom(rst)[j];   // not permute with sweep direction. 
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            if (Params(rst) != NULL)
            {
                vacuum_dens[i] = Vacuum_dens(rst);
                min_pressure[i] = Min_pressure(rst);
                min_energy[i] = Min_energy(rst);
            }

#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        }
        Vec_Gas_field_set(vst,state) = YES;
        Vec_Gas_field_set(vst,coords) = YES;
        Vec_Gas_field_set(vst,rho) = YES;
        Vec_Gas_field_set(vst,en_den) = YES;
        Vec_Gas_field_set(vst,m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        Vec_Gas_field_set(vst,min_pressure) = YES;
        Vec_Gas_field_set(vst,min_energy) = YES;
        Vec_Gas_field_set(vst,vacuum_dens) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        set_params_jumps(vst,imin,imax);
        if ((Params(state[0]) == NULL) && (vst->nprms <= 1))
        {
            int nrad = vsten_radius(wv);
            copy_states_to_new_time_level(idirs,wv,newwv,imin+nrad,
                                          imax-nrad,icoords,pbuf);
            return CONSTANT_IN_TIME;
        }
        g_init_obstacle_states(vst,imin,dim);
        return DYNAMIC_IN_TIME;
}               /*end g_load_unsplit_state_vectors*/




LOCAL void unsplit_2d_MUSCL_in_dir(
        int             swp_num,
        int             *iperm,
        int             *icoords,
        Wave            *wave,
        Wave            *new_swap_wv,
        Front           *fr,
        Front           *newfr,
        Stencil         *sten,
        Tan_stencil     *tsten,
        int             offset,
        int             vsize,
        Vec_Gas         *vst,
        Vec_Gas         **Tvst,
        Vec_Src         *src,
        float           dt,
        float           *dn,
        int             dim,
        int             pbuf)
{

        Vec_Muscl       *vmuscl;
        float           dtdni;
        float           **ucon, **pucon, **F, **source;
        float           *uconk, *puconk, *Fk, *sourcek;
        int             idirs[MAXD];  
        int             start, end;
        int             j, k, kmax;
        int             sten_rad;
        float           *J,*J_half;
        Muscl_Opts      *MOpts;
        float           *pM,*Fluxk,*cur_srck;
        float           **Flux, **cur_src;
        float           radl;
        RECT_GRID       *r_grid = wave->rect_grid;

        MOpts = muscl_options();
        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];
        dtdni = dt/dn[idirs[0]];

        vmuscl = g_load_unsplit_muscl_state_data(MOpts,swp_num,iperm,fr,wave,sten,tsten,
                                 vst,Tvst,src,offset,vsize,dim,dt,dn[idirs[0]]);

        g_load_newucon_state_data(swp_num,iperm,offset,vsize,dim,sten,wave,new_swap_wv,
               vmuscl,icoords,pbuf);

        sten_rad = vmuscl->sten_rad;

        /* compute the eigenvalues, eigenvectors and max wavespeed*/
        g_compute_2dunsplit_eigens(0,vsize,swp_num,iperm,dim,vmuscl);

        start = sten_rad-1;
        end = vsize-start;

        /* compute the coefficients for Lapidus and slope limiting viscosity */
        // This function is not fully implemented.  
        g_compute_2dunsplit_art_visc_coefs(start,end,swp_num,iperm,dim,vmuscl);

        /* compute the linear reconstruction of the state variables */
        g_2dunsplit_bct_linear_reconstructor(start,end,vmuscl);

        /* Evolve for half time step.  The returned data uL and uR
         * are these evolved states on the two sides of each mesh edge.
         * They will be put in the Riemann solver. */
        start = sten_rad;
        end = vsize - sten_rad + 1;
  
        if(sten != NULL)
        {
            if(debug_flag == YES)
            {
                // add_to_debug("N_half_step");
                // add_to_debug("T_half_step");
            }
        } 
        
        /* Half time step expansion in normal dir. */
        g_2dunsplit_N_half_step(start,end,dt,dn,swp_num,iperm,dim,vmuscl);

        /* Add the constribution from another direction */ 
        start = sten_rad - 1;
        end = vsize - sten_rad + 1;
        g_2dunsplit_T_half_step(start,end,dt,dn,swp_num,iperm,dim,vmuscl);

        if(sten != NULL)
        {
            if(debug_flag == YES)
            { 
                // remove_from_debug("N_half_step");
                // remove_from_debug("T_half_step");
            }
        } 

        start = sten_rad;
        end = vsize - sten_rad + 1;
        g_unsplit_muscl_exact_rsolver(start,end,swp_num,iperm,vmuscl->uL,
                  &vmuscl->VLst,vmuscl->uR,&vmuscl->VRst,vmuscl->uM,
                  &vmuscl->VMst,&vmuscl->Flux,vmuscl);

        if (vmuscl->avisc)
            g_2dunsplit_add_art_visc2(start,end,vmuscl);

        /* Set conservative source terms */
        sten_rad = vmuscl->sten_rad;
        start = sten_rad;
        end = vsize - sten_rad;
        g_unsplit_cons_src(start,end,swp_num,iperm,tsten,vmuscl);
        /* Compute the cell average of the approximate solution 
           for the next time step */
        kmax    = dim+2;
        ucon    = vmuscl->ucon;
        pucon    = vmuscl->pucon;
        F       = vmuscl->Flux.F;
        J       = vmuscl->J + offset;
        J_half  = vmuscl->J_half +offset;
        source  = vmuscl->source;

        matrix(&cur_src,kmax,vsize,FLOAT);
        pM    = vmuscl->pM;

        for (j = start; j < end; ++j)
        {
            cur_src[0][j] = 0.0;
            for (k = 0; k < dim; ++k)
                cur_src[vmuscl->index.v[k]][j] = 0.0;
            cur_src[vmuscl->index.energy][j] = 0.0;
        }
#if defined(ROTATIONAL_SYMMETRY)
        if(tsten == NULL)
        {
            for (j = start; j < end; ++j)
            {
                if (r_grid->Remap.remap == CYLINDRICAL_REMAP )
                {
                    if (iperm[swp_num]==0)
                        cur_src[vmuscl->index.v[0]][j] = (pM[j]+pM[j+1])/2.0;
                }
                if (r_grid->Remap.remap == SPHERICAL_REMAP )
                {
                    radl = sqrt(J[j]);
                    cur_src[2][j] = 2.0*radl*(pM[j]+pM[j+1])/2.0;
                }
            }
        }
        else
        {
            /* Tangential sweep */
            const float* const *Q = vmuscl->Q;

            for (j = start; j < end; ++j)
            {
                if (r_grid->Remap.remap == CYLINDRICAL_REMAP)
                    cur_src[2][j] = 0.5*(pM[j]+pM[j+1])*Q[0][0];
                if (r_grid->Remap.remap == SPHERICAL_REMAP)
                {
                    radl = sqrt(J[j]);
                    cur_src[2][j] = 2.0*radl*0.5*(pM[j]+pM[j+1])*Q[0][0];
                }
            }
        }
#endif /* if defined(ROTATIONAL_SYMMETRY) */

        if (source != NULL)
        {
            //TMP
            if(debug_flag == YES)
            {
                for(j = start; j < end; j++)
                    printf("[%d] pucon[] %g %g %g %g\n", j, 
                        pucon[0][j], pucon[1][j], pucon[2][j], pucon[3][j]); 
                for(j = start; j <= end; j++)
                    printf("[%d]  F[]    %g %g %g %g\n", 
                        j, F[0][j], F[1][j], F[2][j], F[3][j]); 
                for(j = start; j < end; j++)
                    printf("[%d] cur_src[] %g %g %g %g\n", j, 
                       cur_src[0][j], cur_src[1][j], cur_src[2][j], cur_src[3][j]); 
                    
            }
 
            for (k = 0; k < kmax; ++k)
            {
                puconk = pucon[k];
                uconk = ucon[k];
                cur_srck = cur_src[k];
                Fk = F[k];
                sourcek = source[k];
                for (j = start; j < end; ++j)
                {
                    uconk[j] = puconk[j] + dtdni*(J_half[j]*Fk[j] 
                                - J_half[j+1]*Fk[j+1])/J[j]
                                + dt*cur_srck[j]/J[j] + dt*sourcek[j];
                }
            }
        }
        else
        {
            for (k = 0; k < kmax; ++k)
            {
                puconk = pucon[k];
                uconk = ucon[k];
                cur_srck = cur_src[k];
                Fk = F[k];
                for (j = start; j < end; ++j)
                {
                    uconk[j] = puconk[j] + dtdni*(J_half[j]*Fk[j] 
                               - J_half[j+1]*Fk[j+1])/J[j]; 
                }
            }
        }

        if(sten != NULL)
        {
            /*
            if(sten->icoords[0][0] == 20 && sten->icoords[0][1] == 20)
            {
                printf("In dir[%d] flux is:\n", iperm[swp_num]); 
                printf("Left side %17.15f, %17.15f, %17.15f, %17.15f\n",
                         F[0][start], F[1][start], F[2][start], F[3][start]); 
                printf("mid state rho, v, e, P[%g, %g,  %g, %g, %g]\n",
                     vmuscl->VMst.rho[start], vmuscl->VMst.v[0][start], 
                     vmuscl->VMst.v[1][start], vmuscl->VMst.e[start], 
                     pM[start]); 
                printf("Right side %g, %g, %g, %g\n",
                         F[0][end], F[1][end], F[2][end], F[3][end]); 
                printf("mid state rho, v, e, P[%g, %g,  %g, %g, %g]\n",
                     vmuscl->VMst.rho[end], vmuscl->VMst.v[0][end], 
                     vmuscl->VMst.v[1][end], vmuscl->VMst.e[end], 
                     pM[end]); 
                printf("Dens Fl-Fr = %g, dtdni*(Fl-Fr) = %g dtdni = %g J_half [%g, %g]\n", 
                       F[0][start] - F[0][end], 
                       dtdni*(J_half[start]*F[0][start] - J_half[end]*F[0][end]), 
                       dtdni, J_half[start], J_half[end]); 
                printf("Old ucon[0] %g, new ucon[0] %g\n", 
                            pucon[0][start], ucon[0][start]); 
                if(swp_num == 1)
                    g_print_state(Rect_state(sten->icoords[0],new_swap_wv)); 
            }
            */
        }

        if(sten != NULL && debugging("check_mass_conservation"))
        { 
            if(sten->comp[0] == 2)
            {
                static int first = YES, step = 0; 
                static float  btm_dens_flux_sum = 0.0; 
                static float  l_dens_flux_sum = 0.0, r_dens_flux_sum = 0.0; 
                if(first)
                {
                    first = NO;
                    step = sten->fr->step;
                }
                if(sten->fr->step != step)
                {
                    printf("At step %d btm_dens_flux %15.12f"
                        " l_dens_flux %15.12f r_dens_flux %15.12f total %15.12f\n",
                        step, btm_dens_flux_sum, l_dens_flux_sum, r_dens_flux_sum,
                        btm_dens_flux_sum + l_dens_flux_sum - r_dens_flux_sum); 
                    btm_dens_flux_sum = 0.0; 
                    l_dens_flux_sum = r_dens_flux_sum = 0.0; 
                    step = sten->fr->step;
                }
                else
                {
                    if(iperm[swp_num] == 1 && sten->icoords[0][1] == 0 && 
                       (sten->icoords[0][0] >= 0 && sten->icoords[0][0] <= 39))
                    {
                        Fk = F[0];
                        btm_dens_flux_sum += dt*dn[1]*Fk[start]; 
                    }
                    if(iperm[swp_num] == 0 && sten->icoords[0][0] == 0)
                    {
                        Fk = F[0];
                        l_dens_flux_sum += dt*dn[0]*Fk[start]; 
                    }
                    if(iperm[swp_num] == 0 && sten->icoords[0][0] == 39)
                    {
                        Fk = F[0];
                        r_dens_flux_sum += dt*dn[0]*Fk[end]; 
                    }
                }
            }
        }

        free(cur_src);  
}

EXPORT void    g_unsplit_muscl_exact_rsolver(
        int        start,
        int        end,
        int        swp_num,
        int        *iperm,
        float      **uL,
        Vec_Gas    *vlst,
        float      **uR,
        Vec_Gas    *vrst,
        float      **uM,
        Vec_Gas    *vmst,
        MUSCL_FLUX *Flux,
        Vec_Muscl  *vmuscl)
{
        static Locstate sl = NULL, sr = NULL, smid = NULL;
        static float    dirs[3][3] = { 1.0, 0.0, 0.0,
                                       0.0, 1.0, 0.0,
                                       0.0, 0.0, 1.0};
        float           *dir;  
        int             idirs[MAXD];  
        Locstate        *state;
        float           *rhol, *rhor;
        float           *el, *er;
        float           *ml[3], *mr[3];
        int             dim;
        int             i,j;

        debug_print("rsolve","Entered g_unsplit_muscl_exact_rsolver()\n");

        dim = vmuscl->dim;
        state = vmuscl->vst->state + vmuscl->offset;

        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];
        dir = dirs[idirs[0]];  

        if (sl == NULL)
        {
            size_t      sizest = vmuscl->sizest;
            g_alloc_state(&sl,sizest);
            g_alloc_state(&sr,sizest);
            g_alloc_state(&smid,sizest);
        }

        rhol = uL[vmuscl->index.density];
        el = uL[vmuscl->index.energy];
        rhor = uR[vmuscl->index.density];
        er = uR[vmuscl->index.energy];
        for (i = 0; i < dim; ++i)
        {
            ml[i] = uL[vmuscl->index.v[i]];
            mr[i] = uR[vmuscl->index.v[i]];
        }

        for (j = start; j < end; ++j)
        {
            if (is_obstacle_state(state[j]))
                continue;
            Dens(sl) = rhol[j];
            Dens(sr) = rhor[j];
            Energy(sl) = el[j];
            Energy(sr) = er[j];
            for (i = 0; i < dim; ++i)
            {
                Mom(sl)[i] = ml[i][j];
                Mom(sr)[i] = mr[i][j];
            }
            Set_params(sl,state[j]);
            set_type_of_state(sl,GAS_STATE);
            Set_params(sr,state[j]);
            set_type_of_state(sr,GAS_STATE);
           
            riemann_solution(0.0,dir,sl,sr,smid,EGAS_STATE);

            vmst->rho[j] = Dens(smid);
            vmst->e[j] = Energy(smid);
            for (i = 0; i < dim; ++i)
                vmst->v[i][j] = Vel(smid)[i];

            if(debug_flag)
            {
                printf("sl and sr state for midst [%d]\n", j);
                g_verbose_print_state(sl); 
                g_verbose_print_state(sr); 
                printf("mid state :"); 
                g_verbose_print_state(smid); 
            } 
        } 
        Vec_Gas_field_set(vmst,rho) = YES;
        Vec_Gas_field_set(vmst,e) = YES;
        Vec_Gas_field_set(vmst,v) = YES;
        if (Flux != NULL)
            g_load_muscl_flux2(start,end,swp_num,iperm,uM,vmst,Flux,vmuscl);

        debug_print("rsolve","Left g_unsplit_muscl_exact_rsolver()\n");
}

/*        g_load_muscl_flux2(), 
 * a copy of g_load_muscl_flux except
 *  NO art. viscosity is added here.
 * Need to implement.
 */
LOCAL  void g_load_muscl_flux2(
        int        start,
        int        end,
        int        swp_num,
        int        *iperm,
        float      **uM,
        Vec_Gas    *vmst,
        MUSCL_FLUX *Flux,
        Vec_Muscl  *vmuscl)
{

        Locstate *state;
        float    *pM;
        int      nprms, *prms_jmp;
        int      i, j, k;
        int      dim;

        debug_print("mflux","Entered g_load_muscl_flux()\n");

        if (Flux == NULL)
        {
            debug_print("mflux","Left g_load_muscl_flux()\n");
            return;
        }

        /*
        if (vmuscl->avisc)
            add_art_visc1(start,end,uM,vmuscl);
        */

        dim = vmuscl->dim;
        pM = vmst->p;
        load_pressure(vmst,start,end-start);

        /*
        if(debug_flag)
        {
            for(j = start; j < end; j++)
            {
                if(j == 2 || j == 3)
                    printf("g_load_muscl_flux2 [%d] mid P %g\n",
                      j, pM[j]);  
            }
        }
        */

        nprms = vmst->nprms;
        prms_jmp = vmst->prms_jmp;
        state = vmuscl->vst->state + vmuscl->offset;
        for (k = 0; k < nprms; ++k)
        {
            if (is_obstacle_state(state[prms_jmp[k]]))
            {
                for (j = prms_jmp[k]; j < prms_jmp[k+1]; ++j)
                {
                    for (i = 0; i < dim; ++i)
                        uM[i+1][j] = 0.0;  // Let V be 0
                }
            }
        }

        /* compute the flux  */
        //flux_vectors(start,end,uM,pM,Flux,vmuscl);
        g_unsplit_muscl_flux_vectors(start,end,swp_num,iperm,uM,pM,Flux,vmuscl); 

        debug_print("mflux","Left g_load_muscl_flux()\n");
}

EXPORT void g_2dunsplit_T_half_step(
        int             start,
        int             end,
        float           dt,
        float           *dn, 
        int             swp_num,
        int             *iperm,
        int             dim, 
        Vec_Muscl       *vmuscl)
{
        int             k, j, idirs[MAXD];
        float           dir[MAXD];  
        Vec_Gas         *vst = vmuscl->vst;
        Vec_Gas         **Tvst = vmuscl->Tvst;
        Locstate        s1, s2, s3;
        float           fluxl[MAXD+2], fluxr[MAXD+2]; 
        float           **uL, **uR;
        float           *uk, *uLk, *uRk;
        
        for (j = 0; j < dim; ++j)
        {
            idirs[j] = iperm[(j+swp_num+1)%dim];
            dir[j] = (float)iperm[(j+swp_num+1)%dim]; 
        }

        if(idirs[0] == 0)
        {
            dir[0] = 1.0;
            dir[1] = 0.0;
        }
        if(idirs[0] == 1)
        {
            dir[0] = 0.0;
            dir[1] = 1.0;
        }

        uL = vmuscl->uL;
        uR = vmuscl->uR;
        for(j = start; j < end; j++)
        {
            s1 = Tvst[0]->state[j]; 
            s2 = vst->state[j]; 
            s3 = Tvst[1]->state[j]; 

            ggodunov_flux(s2, s3, dim, dir, fluxr); 
            ggodunov_flux(s1, s2, dim, dir, fluxl); 
            for(k = 0; k < dim+2; k++)
            {
                uLk = uL[k];
                uRk = uR[k];
                if((j + 1) != end)
                {
                    // uLk[j+1] -= 0.5*dt*(fluxr[k] - fluxl[k]);
                    uLk[j+1] -= 0.5*dt/dn[idirs[0]]*(fluxr[k] - fluxl[k]);
                } 
                if(j != start)
                {
                    // uRk[j] -= 0.5*dt*(fluxr[k] - fluxl[k]); 
                    uRk[j] -= 0.5*dt/dn[idirs[0]]*(fluxr[k] - fluxl[k]); 
                }
            }
        }

#if defined(DEBUG_MUSCL)
        if(debugging("T_half_step"))
        {
            printf("After added the trans. T_half_step, wave\n"); 
            g_printout_vec_data("Left state on the edge",
                                uL[vmuscl->index.density],
                                uL[vmuscl->index.energy],
                                uL+vmuscl->index.v[0],dim,start+1,end,"cons");
            g_printout_vec_data("Right state on the edge",
                                uR[vmuscl->index.density],
                                uR[vmuscl->index.energy],
                                uR+vmuscl->index.v[0],dim,start+1,end,"cons");

        }
#endif /* #if defined(DEBUG_MUSCL) */


}

LOCAL void ggodunov_flux(
        Locstate        s1,
        Locstate        s2,
        int             dim,
        float           *dir,
        float           *flux)
{
        int             i;   
        static size_t   sizest;
        static Locstate ans = NULL; 
        float           rm, em, um[MAXD];   /* mid of dens, mom, en, velo 1D */
        float           vtan1[MAXD+1], vtan2[MAXD+1];

        if(NULL == ans)
        {
            sizest = Params(s2)->sizest;
            (*Params(s2)->_alloc_state)(&ans, max(sizeof(VGas),sizest));
            set_type_of_state(ans,GAS_STATE); 
        }

        riemann_solution(0.0,dir,s1,s2,ans,GAS_STATE);

        rm = Dens(ans);
        em = Energy(ans);
        for(i = 0; i < dim; i++)
            um[i] = Mom(ans)[i]/rm;  

        for(i = 0; i < dim + 2; i++)
            flux[i] = 0.0; 
        for(i = 0; i < dim; i++) 
        {
            flux[0] += rm*um[i]*dir[i];  
            flux[dim+1] += (em + pressure(ans))*um[i]*dir[i];   
        }
        if(dim == 2)
        {
            flux[1] = (rm*um[0]*um[0]+pressure(ans))*dir[0] + 
                       rm*um[0]*um[1]*dir[1]; 
            flux[2] = rm*um[0]*um[1]*dir[0] + 
                      (rm*um[1]*um[1]+pressure(ans))*dir[1]; 
        }
        
}

/* The expansion is in the conservative variables */
/* Not done yet, Need to transfer from conservative variables 
 * to rho, e, v.  
 */
EXPORT void g_2dunsplit_N_half_step(
        int             start,
        int             end,
        float           dt,
        float           *dn, 
        int             swp_num,
        int             *iperm,
        int             dim, 
        Vec_Muscl       *vmuscl)
{
        Vec_Eigen       *vegn;
        int             i, j, k, idirs[MAXD];
        int             kmax;
        float           **ucon, **source, **dq, **uL, **uR;
        float           **right_wvs, **left_wvs, **source_sum;
        float           ***l, ***r;
        float           **lambda;
        float           **sgnl, **sgnr;
        float           *ssk;
        float           *rk0, *rk1, *rk2, *rk3;
        float           *ws0, *ws1, *ws2, *ws3, *wsk;
        float           *lambdak, *lambda1;
        float           *dqk;
        float           *sgnrk, *sgnr1, *sgnlk, *sgnl1;
        float           *uk, *uLk, *uRk;
        float           dtdni;
        static Locstate tmpst = NULL;

        kmax =          dim+2;
        ucon =          vmuscl->ucon;
        source =        vmuscl->source;
        dq =            vmuscl->dq;
        uL =            vmuscl->uL;
        uR =            vmuscl->uR;
        right_wvs =     vmuscl->right_wvs;
        left_wvs =      vmuscl->left_wvs;
        source_sum =    vmuscl->source_sum;
        vegn =          &vmuscl->Vegn;
        l =             vegn->l;
        r =             vegn->r;
        lambda =        vegn->lambda;
        sgnl =          vegn->sgnl;
        sgnr =          vegn->sgnr;

        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];

        dtdni = dt/dn[idirs[0]];

        /* add up source term contributions for half time-step */
        /* For the current implementation, source correspond to 
         * the eqn with conservative variables, namely, rho, Mom[0,1,2], E. 
         */
        if ((source != NULL) && (vmuscl->src != NULL))
        {
            // unsplit_left_multiply_state_by_matrix(source_sum,l,
            //       source,dim,start-1,end);
            for (k = 0; k < kmax; ++k)
            {
                ssk = source_sum[k];
                for (j = start - 1; j < end; ++j)
                {
                    ssk[j] = source[k][j]*0.5 * dt;
                }
            }
            if(debug_flag == YES)
            {
                printf("g_2dunsplit_N_half_step, add source\n");
                for(j = start - 1; j < end; ++j)
                {
                    for(k = 0; k < kmax; k++)
                        printf("src%d[%d] = %g ", k, j, source_sum[k][j]); 
                    printf("\n"); 
                }
            } 
        }
        else
        {
            for (k = 0; k < kmax; ++k)
            {
                ssk = source_sum[k];
                for (j = start - 1; j < end; ++j)
                    ssk[j] = 0.0;
            }
        }
        /* add up contributions from right moving waves */
        /* First add the wave contribution from the reference states and source */
        /* Then add the wave contribution from all wave families */

        if(debug_flag == YES)
        {
            printf("Updating UL, UR, lambda dq:\n");
            for(j = start-1; j < end; j++)
            { 
                printf("[%d] ", j); 
                for(k = 0; k < kmax; ++k)
                    printf("lambda[%d] = %g, ", k, lambda[k][j]); 
                printf("\n"); 
                for(k = 0; k < kmax; ++k)
                    printf("dq[%d] = %g, ", k, dq[k][j]); 
                printf("\n"); 
            }
            printf("dt = %g\n", dt); 
        }

        for (k = 0; k < kmax; ++k)
        {
            wsk = right_wvs[k];
            ssk = source_sum[k];
            lambdak = lambda[k];
            dqk = dq[k];
            sgnrk = sgnr[k];

            for (j = start; j < end; ++j)
            {
                wsk[j] = 0.5*(1.0-dtdni*max(lambda[3][j-1], 0.0))*dqk[j-1];
                if(lambdak[j-1] > 0.0)
                    wsk[j] += 0.5*dtdni*(lambda[3][j-1] - lambdak[j-1])*dqk[j-1];
            }
        }

        /* compute uL */
        for (k = 0; k < kmax; ++k)
        {
            rk0 = r[k][0];  rk1 = r[k][1];  rk2 = r[k][2]; rk3 = r[k][3]; 
            ws0 = right_wvs[0]; ws1 = right_wvs[1]; ws2 = right_wvs[2];
            ws3 = right_wvs[3];  
            uk = ucon[k];
            uLk = uL[k];
            ssk = source_sum[k];
            for (j = start; j < end; ++j)
            {
                uLk[j] = uk[j-1] +
                    (ws0[j]*rk0[j-1] + ws1[j]*rk1[j-1] + ws2[j]*rk2[j-1] + ws3[j]*rk3[j-1])
                    + ssk[j-1];
            }
        }

        Vec_Gas_field_set(&vmuscl->VLst,rho) = YES;
        Vec_Gas_field_set(&vmuscl->VLst,en_den) = YES;
        Vec_Gas_field_set(&vmuscl->VLst,m) = YES;

        /* add up contributions from left moving waves */
        /* First add the wave contribution from the reference states and source */
        /* Then add add the wave contribution from all wave families */
        for (k = 0; k < kmax; ++k)
        {
            wsk = left_wvs[k];
            ssk = source_sum[k];
            lambdak = lambda[k];
            dqk = dq[k];
            sgnlk = sgnl[k];
            for (j = start; j < end; ++j)
            {
                wsk[j] = 0.5*(-1.0 - dtdni*min(lambda[0][j], 0.0))*dqk[j];
                if(lambdak[j] < 0.0)
                    wsk[j] += 0.5*dtdni*(lambda[0][j] - lambdak[j])*dqk[j];
            }
        }

        /* compute uR */
        for (k = 0; k < kmax; ++k)
        {
            rk0 = r[k][0]; rk1 = r[k][1]; rk2 = r[k][2]; rk3 = r[k][3];  
            ws0 = left_wvs[0]; ws1 = left_wvs[1]; ws2 = left_wvs[2]; ws3 = left_wvs[3]; 
            uk = ucon[k];
            uRk = uR[k];
            ssk = source_sum[k];
            for (j = start; j < end; ++j)
            {
                uRk[j] = uk[j] + 
                   (ws0[j]*rk0[j]+ws1[j]*rk1[j]+ws2[j]*rk2[j] + ws3[j]*rk3[j]) + ssk[j];
            }
        }

        Vec_Gas_field_set(&vmuscl->VRst,rho) = YES;
        Vec_Gas_field_set(&vmuscl->VRst,en_den) = YES;
        Vec_Gas_field_set(&vmuscl->VRst,m) = YES;

#if defined(DEBUG_MUSCL)
        if(debugging("N_half_step"))
        {
            (void) printf(" Solution before consv half step\n");
            /***
            for (j = start; j < end; ++j)
            {
                (void) printf("%-4d",j);
                //(void) printf(" %-14g",ucon[vmuscl->index.density][j]);
                (void) printf(" %-17.15f",ucon[vmuscl->index.density][j]);
                //(void) printf(" %-14g",ucon[vmuscl->index.energy][j]);
                (void) printf(" %-17.15f",ucon[vmuscl->index.energy][j]);
                for (k = 0; k < dim; ++k)
                //    (void) printf(" %-14g",ucon[vmuscl->index.v[k]][j]);
                    (void) printf(" %-17.15f",ucon[vmuscl->index.v[k]][j]);
                (void) printf("\n");
            }
            (void) printf("\n");
            ***/
            g_printout_vec_data("Left state on the edge",
                                uL[vmuscl->index.density],
                                uL[vmuscl->index.energy],
                                uL+vmuscl->index.v[0],dim,start,end,"cons");
            g_printout_vec_data("Right state on the edge",
                                uR[vmuscl->index.density],
                                uR[vmuscl->index.energy],
                                uR+vmuscl->index.v[0],dim,start,end,"cons");
        }
#endif /* #if defined(DEBUG_MUSCL) */
}

LOCAL   void    unsplit_left_multiply_state_by_matrix(
        float           **q,
        float           ***l,
        float           **u,
        int             dim,
        int             start,
        int             end)
{
        float           *u0, *u1, *u2, *u3, *uk;
        float           *lk0, *lk1, *lk2, *lk3;
        float           *qk;
        int             j, k;
        int             kmax = 2+dim;

        u0 = u[0]; u1 = u[1]; u2 = u[2]; u3= u[3]; 
        for (k = 0; k < kmax; ++k)
        {
            lk0 = l[k][0]; lk1 = l[k][1]; lk2 = l[k][2]; lk3 = l[k][3]; 
            qk = q[k];
            for (j = start; j < end; ++j)
                qk[j] = lk0[j]*u0[j] + lk1[j]*u1[j] + lk2[j]*u2[j] + lk3[j]*u3[j];
        }
}               /*end unsplit_left_multiply_state_by_matrix*/


/*  
 *    g_2dunsplit_bct_linear_reconstructor() 
 * Follows Multidimensional Upwind Methods
 * for Hyperbolic Conservation Laws. J.C.P. 87, 171-200 (1990)
 * Expand the conservative variables. 
 */
EXPORT void g_2dunsplit_bct_linear_reconstructor(
        int             start,
        int             end,
        Vec_Muscl       *vmuscl)
{

        int             dim, j, k;   /* j is the mesh index */
        int             kmax;
        float           **ucon, **backward, **central, **forward;
        float           ***l;
        float           *bk, *ck, *fk;
        float           *u0, *u1, *u2, *u3, *uk;
        float           *lk0, *lk1, *lk2, *lk3;

        // might want to use limiting coeffs. here

        dim =           vmuscl->dim;  
        kmax =          dim+2;
        ucon =          vmuscl->ucon;
        l =             vmuscl->Vegn.l;
        backward =      vmuscl->backward;
        central =       vmuscl->central;
        forward =       vmuscl->forward;

        u0 = ucon[0]; u1 = ucon[1]; u2 = ucon[2]; u3 = ucon[3];  

        if(debug_flag == YES)
        {
            printf("g_2dunsplit_bct_linear_reconstructor\n"); 
            for (j = start; j < end; ++j)
            {
                for (k = 0; k < kmax; ++k)
                {
                    printf("ucon%d[%d] = %g ", 
                          k, j, ucon[k][j]);
                }
                printf("\n"); 
            }

            for (j = start; j < end; ++j)
            {
                printf("%d L is\n", j); 
                for(k = 0; k < kmax; k++)
                    printf("l%d = %g %g %g %g\n", 
                     k, l[k][0][j], l[k][1][j], l[k][2][j], l[k][3][j]); 
            }
        }

        for (k = 0; k < kmax; ++k)
        {
            bk = backward[k]; ck = central[k]; fk = forward[k];
            lk0 = l[k][0]; lk1 = l[k][1]; lk2 = l[k][2]; lk3 = l[k][3]; 
            for (j = start; j < end; ++j)
            {
                bk[j] = ((u0[j] - u0[j-1]) * lk0[j] +
                         (u1[j] - u1[j-1]) * lk1[j] +
                         (u2[j] - u2[j-1]) * lk2[j] + 
                         (u3[j] - u3[j-1]) * lk3[j]);
                ck[j] = 0.5 * (
                        (u0[j+1] - u0[j-1]) * lk0[j] +
                        (u1[j+1] - u1[j-1]) * lk1[j] +
                        (u2[j+1] - u2[j-1]) * lk2[j] + 
                        (u3[j+1] - u3[j-1]) * lk3[j]);
                fk[j] = ((u0[j+1] - u0[j]) * lk0[j] +
                         (u1[j+1] - u1[j]) * lk1[j] +
                         (u2[j+1] - u2[j]) * lk2[j] +
                         (u3[j+1] - u3[j]) * lk3[j]);
                if(debug_flag == YES && k == 1)
                {
                    printf("At J[%d] bk, ck, fk, %g %g %g\n", j, bk[j], ck[j], fk[j]); 
                }
            }
        }

        apply_limiter2(vmuscl->dq,start,end,vmuscl);

}

LOCAL   void    apply_limiter2(
        float           **slope,
        int             start,
        int             end,
        Vec_Muscl       *vmuscl)
{
        int             dim;
        int             j, k;    /* j is the mesh index */
        int             kmax;
        float           **backward, **central, **forward;
        float           *bk, *ck, *fk;
        float           *slopek;
        float           sign, dl, abk, afk, ack;
        float           fact = 1.0; // 2.0 is van leer

        dim =           vmuscl->dim;
        kmax =          dim+2;
        backward =      vmuscl->backward;
        central =       vmuscl->central;
        forward =       vmuscl->forward;

#if defined(DEBUG_MUSCL)
        if (debugging("lcnst"))
        {
            (void) printf("\n");
            g_printout_vec_data("Backward differences",backward[0],backward[1],
                                backward+2,dim,start,end,"muncons");
            g_printout_vec_data("Central differences",central[0],central[1],
                                central+2,dim,start,end,"muncons");
            g_printout_vec_data("Forward differences",forward[0],forward[1],
                                forward+2,dim,start,end,"muncons");
        }
#endif /* defined(DEBUG_MUSCL) */

        // compute_slope_limiting_coeffs(start,end,vmuscl);

        /* Compute the standard Van Leer slope limiter */

        for (k = 0; k < kmax; ++k)
        {
            bk = backward[k]; ck = central[k]; fk = forward[k];
            slopek = slope[k];
            for (j = start; j < end; ++j)
            {
                if (bk[j]*fk[j] > 0.0)
                {
                    abk = fabs(bk[j]);
                    afk = fabs(fk[j]);
                    dl = fact*min(abk,afk);
                    sign = (ck[j] == 0.0) ? 0.0 : (ck[j] > 0.0) ? 1.0 : -1.0;
                    ack = fabs(ck[j]);
                    // slopek[j] = sign*min(ack,dl);
                    slopek[j] = sign*min(abk,afk); /* mini_mod() */
                }
                else
                {
                    slopek[j] = 0.0;
                }
            }
        }

        /*
        if (vmuscl->msf != NULL)
        {
            float *chi = vmuscl->msf->chi;
            for (k = 0; k < kmax; ++k)
            {
                slopek = slope[k];
                for (j = start; j < end; ++j)
                    slopek[j] *= chi[j];
            }
        }
        */
}  /* end of apply_limiter2 */

/*  g_compute_2dunsplit_art_visc_coefs ()
 *
 *     WARNING: need to impliment ompute_lapidus_art_visc_coefs
 *     for 2d unsplit solver.
 */

EXPORT void g_compute_2dunsplit_art_visc_coefs(
        int             start,
        int             end,
        int             swp_num,
        int             *iperm,
        int             dim, 
        Vec_Muscl       *vmuscl)
{
        int             i, j, k, idirs[MAXD];  
        Vec_Avisc       *avisc = vmuscl->avisc;
        float           *g0;
        Locstate        *state;
        Gas_param       *prms;
        float           **lambda;
        float           dlambda, max_dlambda;
        int             negn;

        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];


        if (avisc == NULL)
            return;
        if (avisc->use_lapidus)
        {
            printf("WARNING: need to impliment ompute_lapidus_art_visc_coefs\n");
            printf("For 2d unsplit solver\n"); 
            // g_compute_lapidus_art_visc_coefs(start,end,vmuscl);
        }
        else
        {
            g0 = avisc->g[0];
            for (j = start; j < end; ++j)
                g0[j] = 0.0;
        }
        state = vmuscl->vst->state + vmuscl->offset;

        g0 = avisc->g[0];
        if (avisc->use_linear)
        {
            float *c = vmuscl->vst->c;
            float *vn = vmuscl->vst->v[idirs[0]];
            prms = NULL;
            for (j = start; j < end; ++j)
            {
                if (Params(state[j]) != prms)
                    prms = Params(state[j]);
                if (prms != NULL)
                {
                    float sp = (fabs(vn[j]) + c[j])*prms->avisc.sp_coef;;
                    g0[j] += 2.0*sp*prms->avisc.linear_visc_coef;
                }
            }
        }
        if (avisc->use_upwind)
        {
            lambda = vmuscl->Vegn.lambda;
            negn = vmuscl->Vegn.negn;
            prms = NULL;
            for (j = start; j < end; ++j)
            {
                if (Params(state[j]) != prms)
                    prms = Params(state[j]);
                if (prms == NULL)
                    continue;
                max_dlambda = 0.0;
                for (i = 0; i < negn; ++i)
                {
                    dlambda = lambda[i][j] - lambda[i][j+1];
                    if (dlambda > max_dlambda)
                        max_dlambda = dlambda;
                }
                g0[j] += 2.0*prms->avisc.upwind_visc_coef*max_dlambda;
            }
        }

}

/*
*                        g_compute_2dunsplit_eigens():
*
*       Calculate the eigenvalues and eigenvectors of the linearized gas
*       dynamical equations in the normal/tangential direction.
*       The state vector is u = (rho, Mom0, Mom1, Mom2, E), which are to be
*       interpreted as cell averages and with all
*       vectors written in these coordinates.  
*/

EXPORT void g_compute_2dunsplit_eigens(
        int             start,
        int             end,
        int             swp_num,
        int             *iperm,
        int             dim, 
        Vec_Muscl       *vmuscl)
{
        int             j, k,  idirs[MAXD];  
        float           b1, b2;
        float           q_sqr;
        float           u, v;
        float           A, B;
        float           u_hat, v_hat;
        float           H;
        Vec_Eigen  *vegn = &vmuscl->Vegn;
        Vec_Gas    *vst = vmuscl->vst;
        int        offset = vmuscl->offset;
        float      **lambda = vegn->lambda;
        float      ***l = vegn->l, ***r = vegn->r;
        float      *rho = vst->rho + offset;
        float      *en_den = vst->en_den + offset;  
        float      *v0 = vst->v[0] + offset;
        float      *v1 = vst->v[1] + offset;
        float      *vN;
        float      *c = vst->c + offset;
        float      *p = vst->p + offset;
        float      *GAM = vst->GAM + offset;
        float      *c2 = vst->c2 + offset;
        float      **sgnl = vegn->sgnl;
        float      **sgnr = vegn->sgnr;
        float      speed;
        float      *sgnlk, *sgnrk;
        float      *lambdak;
        float      *lambda0, *lambda1, *lambda2, *lambda3;
        float      *r00, *r01, *r02, *r03;
        float      *r10, *r11, *r12, *r13;
        float      *r20, *r21, *r22, *r23;
        float      *r30, *r31, *r32, *r33;

        float      *l00, *l01, *l02, *l03;
        float      *l10, *l11, *l12, *l13;
        float      *l20, *l21, *l22, *l23;
        float      *l30, *l31, *l32, *l33;
        Locstate   *state = vst->state + offset;


        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];

        if(idirs[0] == 0)
        {
            A = 1.0; B = 0.0;
            vN = vst->v[0] + offset; 
        }
        else
        {
            A = 0.0; B = 1.0;
            vN = vst->v[1] + offset; 
        }

        lambda0 = lambda[0]; lambda1 = lambda[1]; lambda2 = lambda[2]; lambda3 = lambda[3]; 
 
        r00 = r[0][0]; r01 = r[0][1]; r02 = r[0][2]; r03 = r[0][3]; 
        r10 = r[1][0]; r11 = r[1][1]; r12 = r[1][2]; r13 = r[1][3]; 
        r20 = r[2][0]; r21 = r[2][1]; r22 = r[2][2]; r23 = r[2][3]; 
        r30 = r[3][0]; r31 = r[3][1]; r32 = r[3][2]; r33 = r[3][3]; 

        l00 = l[0][0]; l01 = l[0][1]; l02 = l[0][2]; l03 = l[0][3]; 
        l10 = l[1][0]; l11 = l[1][1]; l12 = l[1][2]; l13 = l[1][3]; 
        l20 = l[2][0]; l21 = l[2][1]; l22 = l[2][2]; l23 = l[2][3]; 
        l30 = l[3][0]; l31 = l[3][1]; l32 = l[3][2]; l33 = l[3][3]; 

        for (j = start; j < end; ++j)
        {
            lambda0[j] = vN[j] - c[j];
            lambda1[j] = vN[j];
            lambda2[j] = vN[j];
            lambda3[j] = vN[j] + c[j];

            q_sqr = sqr(v0[j])+sqr(v1[j]);
            u_hat = A*v0[j]+B*v1[j];
            v_hat = A*v1[j]-B*v0[j];
            H = (en_den[j]+p[j])/rho[j];
            b1 = GAM[j]/sqr(c[j]);
            b2 = 1.0 + b1*q_sqr-b1*H;  

            r00[j] = 1.0;
            r10[j] = v0[j] - A*c[j];
            r20[j] = v1[j] - B*c[j];
            r30[j] = H - u_hat*c[j]; 

            r01[j] = 1.0;
            r11[j] = v0[j];
            r21[j] = v1[j];
            r31[j] =  H-1.0/b1;  

            r02[j] = 0.0;
            r12[j] = B;
            r22[j] = -A;
            r32[j] = -v_hat;

            r03[j] = 1.0;
            r13[j] = v0[j] + A*c[j];
            r23[j] = v1[j] + B*c[j];
            r33[j] = H + u_hat*c[j];

            l00[j] = b2/2.0+u_hat/(2.0*c[j]);
            l01[j] = -b1*v0[j]/2.0-A/(2.0*c[j]);
            l02[j] = -b1*v1[j]/2.0-B/(2.0*c[j]);
            l03[j] = b1/2.0; 

            l10[j] = 1-b2;
            l11[j] = b1*v0[j];
            l12[j] = b1*v1[j];
            l13[j] = -b1;

            l20[j] = v_hat;
            l21[j] = B;
            l22[j] = -A;
            l23[j] = 0.0; 

            l30[j] = b2/2.0-u_hat/(2.0*c[j]);
            l31[j] = -b1*v0[j]/2.0+A/(2.0*c[j]);
            l32[j] = -b1*v1[j]/2.0+B/(2.0*c[j]);
            l33[j] = b1/2.0; 

            if(debug_flag == YES)
            {
                int   kk; 
                float sum;  
                // printf("[%d] b2 %g, u_hat %g, c %g, v0 %g, v1 %g, b1 %g, GAM %g A %g, B %g\n", 
                //    j, b2, u_hat, c[j], v0[j], v1[j], b1, GAM[j], A , B); 
                for(kk = 0; kk < 4; kk++)
                {
                    sum = l[kk][0][j]*r[0][kk][j] + 
                          l[kk][1][j]*r[1][kk][j] +
                          l[kk][2][j]*r[2][kk][j] +
                          l[kk][3][j]*r[3][kk][j];
                    if(fabs(1.0-sum) > 0.00001)
                    {
                        printf("ERROR: g_compute_2dunsplit_eigens\n"); 
                        printf("%d cell %d componenet at dir %d sum = %g\n",
                           j, kk, idirs[0], sum); 
                        clean_up(ERROR);   
                    }
                }
            } 
        }         

        for (j = start; j < end; ++j)
        {
            if (Params(state[j]) != NULL)
            {
                speed = Params(state[j])->avisc.sp_coef*(fabs(vN[j]) + c[j]);
                if (vmuscl->wave != NULL)
                {
                    set_max_wave_speed(vmuscl->idir,speed,vst->state[j+offset],
                                       vst->coords[j+offset],vmuscl->wave);
                }
                else
                {
                    int    k, dim = vmuscl->dim;

                    for (k = 0; k < dim; ++k)
                    {
                        set_max_front_speed(k,vmuscl->Q[0][k]*speed,
                                            vst->state[j+offset],
                                            vst->coords[j+offset],
                                            vmuscl->front);
                    }
                    set_max_front_speed(dim,speed/vmuscl->dn,
                                        vst->state[j+offset],
                                        vst->coords[j+offset],
                                        vmuscl->front);
                }
            }
        }


        for (k = 0; k < 3; ++k)
        {
            sgnlk = sgnl[k]; sgnrk = sgnr[k]; lambdak = lambda[k];
            for (j = start; j < end; ++j)
            {
                if (lambdak[j] > 0.0)
                {
                    sgnlk[j] = 0.0;
                    sgnrk[j] = 1.0;
                }
                else
                {
                    sgnlk[j] = 1.0;
                    sgnrk[j] = 0.0;
                }
            }
        }

}

LOCAL void g_load_newucon_state_data(
        int             swp_num,
        int             *iperm,
        int             offset,
        int             vsize,
        int             dim,
        Stencil         *sten, 
        Wave            *wave,
        Wave            *newwave,
        Vec_Muscl       *vmuscl,
        int             *icoords,
        int             pbuf)
{
        int             start, end;
        int             j, k, endpt;
        int             sten_rad;
        int             ic[MAXD], idirs[MAXD]; 
        float           **pucon;
        Vec_Gas         *vst;  
        Locstate        rst;

        sten_rad = vmuscl->sten_rad;
        start = sten_rad;
        end = vsize - sten_rad;
        pucon = vmuscl->pucon;  
        vst = vmuscl->vst;
        for(k = 0; k < dim; k++)
        {
            ic[k] = icoords[k];  
            idirs[k] = iperm[(k+swp_num)%dim];
        }

        if(sten == NULL)
        {
            if(swp_num == 0)
            {
                for (j = 0; j < vsize; ++j)
                {
                    pucon[vmuscl->index.density][j] = vst->rho[j]; 
                    pucon[vmuscl->index.energy][j] = vst->en_den[j];
                    for(k = 0; k < dim; k++)
                        pucon[vmuscl->index.v[k]][j] = vst->m[k][j];  
                }
            }
            else  
            {
                for(j = 0; j < vsize; ++j)
                {
                    ic[idirs[0]] = j + pbuf;
                    rst = Rect_state(ic,newwave);
                    pucon[vmuscl->index.density][j] = Dens(rst); 
                    pucon[vmuscl->index.energy][j] = Energy(rst);
                    for(k = 0; k < dim; k++)
                        pucon[vmuscl->index.v[k]][j] = Mom(rst)[k]; 
                }
            }
        }
        else
        {
            if(swp_num == 0)
            {
                for (j = 0; j < vsize; ++j)
                {
                    pucon[vmuscl->index.density][j] = vst->rho[j]; 
                    pucon[vmuscl->index.energy][j] = vst->en_den[j];
                    for(k = 0; k < dim; k++)
                        pucon[vmuscl->index.v[k]][j] = vst->m[k][j];  
                }
            }
            else
            {
                int  ictmp; 
                ictmp = ic[idirs[0]]; 
                endpt = (int)(sten->npts/2);  

                rst = Rect_state(ic,newwave);
                pucon[vmuscl->index.density][endpt] = Dens(rst); 
                pucon[vmuscl->index.energy][endpt] = Energy(rst);
                for(k = 0; k < dim; k++)
                    pucon[vmuscl->index.v[k]][endpt] = Mom(rst)[k]; 
                /*
                for(j = -endpt; j <= endpt; ++j)
                {
                    ic[idirs[0]] = ictmp + j;
                    rst = Rect_state(ic,newwave);
                    pucon[vmuscl->index.density][j+endpt] = Dens(rst); 
                    pucon[vmuscl->index.energy][j+endpt] = Energy(rst);
                    for(k = 0; k < dim; k++)
                        pucon[vmuscl->index.v[k]][j+endpt] = Mom(rst)[k]; 
                }
                */
            }
        }
}

EXPORT   Vec_Muscl *g_init_unsplit_muscl_state_data(
        Muscl_Opts      *mopts,
        int             swp_num,
        int             *iperm,
        Front           *fr,
        Wave            *wave,
        Stencil         *sten,
        Tan_stencil     *tsten,
        Vec_Gas         *vst,
        Vec_Gas         **Tvst,
        Vec_Src         *src,
        int             offset,
        int             vsize,
        int             dim,
        float           dt,
        float           dn)
{
        static Vec_Muscl *vmuscl = NULL;
        float            **u, **ucon, **source;
        const float* const *Q = vst->Q;
        bool             is_src;
        int              i, j;

        if (vmuscl == NULL || vsize > vmuscl->max_vsize || dim > vmuscl->dim)
        {
            vmuscl = g_muscl_free_wk_space(vmuscl);
            vmuscl = g_unsplit_muscl_alloc_wk_space(mopts,vsize,dim,vmuscl);
        }
        vmuscl->sizest = fr->sizest;
        vmuscl->idir = (iperm != NULL) ? iperm[swp_num] : dim;
        vmuscl->Q = Q;
        vmuscl->dt = dt;
        vmuscl->dn = dn;
        vmuscl->front = fr;
        vmuscl->wave = wave;
        if (wave != NULL)
            vmuscl->sten_rad = wave->npts_sten/2;
        else if (tsten != NULL)
            vmuscl->sten_rad = tsten->npts/2;
        else
        {
            vmuscl->sten_rad = 2;
            screen("ERROR in g_load_muscl_state_data(), can't determine "
                   "stencil size\n");
            clean_up(ERROR);
        }
        vmuscl->vst = vst;
        vmuscl->src = src;
        vmuscl->offset = offset;        vmuscl->vsize = vsize;
        vmuscl->Tvst[0] = Tvst[0]; vmuscl->Tvst[1] = Tvst[1]; 

        g_load_VGas_state_vectors(offset,vsize,vst,dim);

        clear_Vec_Gas_set_flags(&vmuscl->VMst);
        clear_Vec_Gas_set_flags(&vmuscl->VLst);
        clear_Vec_Gas_set_flags(&vmuscl->VRst);
        vmuscl->VMst.Q = vmuscl->VLst.Q = vmuscl->VRst.Q = Q;

        /* Set params jump points for vec gas states */
        copy_vec_state_params(&vmuscl->VMst,vst,offset,vsize);
        copy_vec_state_params(&vmuscl->VRst,vst,offset,vsize);
        copy_vec_state_params(&vmuscl->VLst,vst,offset-1,vsize);

        /*put the state variables and source terms in a form more
          easily usable by the subroutines*/
        u = vmuscl->u;   
        ucon = vmuscl->ucon;  
        u[vmuscl->index.density] = ucon[0] = vst->rho + offset;
        u[vmuscl->index.energy] = vst->e + offset;
        ucon[vmuscl->index.energy] = vst->en_den + offset; 
        for (i = 0; i < dim; ++i)
        {
            u[vmuscl->index.v[i]] = vst->v[i] + offset;
            ucon[vmuscl->index.v[i]] = vst->m[i] + offset; 
        }

        if ((src != NULL) && ((source = vmuscl->source) != NULL))
        {
            source[0] = src->mass + offset;
            for (i=0; i < dim; ++i)
                source[vmuscl->index.v[i]] = src->mom[i] + offset;
            source[vmuscl->index.energy] = src->energy + offset;
        }

        if (vmuscl->gravity != NULL)
        {
            float time = fr->time + 0.5*dt;
            float **coords = vst->coords + offset;
            float **g = vmuscl->gravity + offset;

            if (tsten == NULL)
            {
                for (j = 0; j < vsize; ++j)
                {
                    for(i = 0; i < dim; i++)
                        g[i][j] = gravity(coords[j],time)[i];
                }
            }
            else
            {
                printf("ERROR  g_init_unsplit_muscl_state_data,\n");
                printf("Implement for gravity for tangent sweep\n");
                clean_up(ERROR); 
            }
        }

        return vmuscl;  
}  /*end g_init_unsplit_muscl_state_data*/ 


EXPORT   Vec_Muscl *g_load_unsplit_muscl_state_data(
        Muscl_Opts      *mopts,
        int             swp_num,
        int             *iperm,
        Front           *fr,
        Wave            *wave,
        Stencil         *sten,
        Tan_stencil     *tsten,
        Vec_Gas         *vst,
        Vec_Gas         **Tvst,
        Vec_Src         *src,
        int             offset,
        int             vsize,
        int             dim,
        float           dt,
        float           dn)
{
        Vec_Muscl   *vmuscl = NULL;
        float       **u, **ucon, **source;
        int         i, j;
        bool        is_src;

        vmuscl = g_init_unsplit_muscl_state_data(mopts, swp_num, iperm, 
                    fr, wave, sten, tsten, vst, Tvst, src, offset,
                     vsize, dim, dt, dn); 

        g_compute_Jacobi(swp_num,iperm,0,vsize,sten,NULL,vmuscl,wave,fr,vst,offset);

        is_src = g_unsplit_muscl_load_source_vectors(swp_num,iperm,tsten,
                 vmuscl->gravity,vst,src,vmuscl->J,offset,vsize,fr->rect_grid);

        vmuscl->is_src = is_src;

        u = vmuscl->u;
        ucon = vmuscl->ucon;
        u[vmuscl->index.density] = ucon[0] = vst->rho + offset;
        u[vmuscl->index.energy] = vst->e + offset;
        ucon[vmuscl->index.energy] = vst->en_den + offset;
        for (i = 0; i < dim; ++i)
        {
            u[vmuscl->index.v[i]] = vst->v[i] + offset;
            ucon[vmuscl->index.v[i]] = vst->m[i] + offset;
        }

#if defined(DEBUG_MUSCL)
        if (debugging("load_Vgas"))
        {
            (void) printf("All quantities before MUSCL computation:\n");

            g_printout_vec_data("The state variables",
                                u[vmuscl->index.density],
                                u[vmuscl->index.energy],
                                u+vmuscl->index.v[0],
                                dim,0,vsize,"muncons");
            g_printout_vec_data("The conserved variables",
                                ucon[0],ucon[vmuscl->index.energy],
                                ucon+vmuscl->index.v[0],dim,0,vsize,"cons");
            if (is_src)
            {
                g_printout_vec_data("The source terms",
                                    source[0],source[1],source+2,dim,1,
                                    vsize-1,"src");
            }

            (void) printf("Miscelaneous quantities:\n");
            (void) printf("%-4s %-14s %-14s %-14s\n","n","pressure",
                          "sound speed","Grun. coef");
            for (i = offset; i < vsize; ++i)
                (void) printf("%-4d %-14g %-14g %-14g\n",
                              i,vst->p[i],vst->c[i],vst->GAM[i]);
            (void) printf("\n");
        }
#endif /* defined(DEBUG_MUSCL) */
        return vmuscl;
}

/****
EXPORT   Vec_Muscl *g_load_unsplit_muscl_state_data(
        Muscl_Opts      *mopts,
        int             swp_num,
        int             *iperm,
        Front           *fr,
        Wave            *wave,
        Stencil         *sten,
        Tan_stencil     *tsten,
        Vec_Gas         *vst,
        Vec_Gas         **Tvst,
        Vec_Src         *src,
        int             offset,
        int             vsize,
        int             dim,
        float           dt,
        float           dn)
{
        static Vec_Muscl *vmuscl = NULL;
        float            **u, **ucon, **source;
        const float* const *Q = vst->Q;
        bool             is_src;
        int              i, j;

        if (vmuscl == NULL || vsize > vmuscl->max_vsize || dim > vmuscl->dim)
        {
            vmuscl = g_muscl_free_wk_space(vmuscl);
            vmuscl = g_unsplit_muscl_alloc_wk_space(mopts,vsize,dim,vmuscl);
        }
        vmuscl->sizest = fr->sizest;
        vmuscl->idir = (iperm != NULL) ? iperm[swp_num] : dim;
        vmuscl->Q = Q;
        vmuscl->dt = dt;
        vmuscl->dn = dn;
        vmuscl->front = fr;
        vmuscl->wave = wave;
        if (wave != NULL)
            vmuscl->sten_rad = wave->npts_sten/2;
        else if (tsten != NULL)
            vmuscl->sten_rad = tsten->npts/2;
        else
        {
            vmuscl->sten_rad = 2;
            screen("ERROR in g_load_muscl_state_data(), can't determine "
                   "stencil size\n");
            clean_up(ERROR);
        }
        vmuscl->vst = vst;
        vmuscl->src = src;
        vmuscl->offset = offset;        vmuscl->vsize = vsize;
        vmuscl->Tvst[0] = Tvst[0]; vmuscl->Tvst[1] = Tvst[1]; 

        g_load_VGas_state_vectors(offset,vsize,vst,dim);

        clear_Vec_Gas_set_flags(&vmuscl->VMst);
        clear_Vec_Gas_set_flags(&vmuscl->VLst);
        clear_Vec_Gas_set_flags(&vmuscl->VRst);
        vmuscl->VMst.Q = vmuscl->VLst.Q = vmuscl->VRst.Q = Q;

        // gvol_compute_Jacobi(swp_num,iperm,0,vsize,vmuscl,wave,fr,vst,offset);
        g_compute_Jacobi(swp_num,iperm,0,vsize,sten,NULL,vmuscl,wave,fr,vst,offset);

        // Set params jump points for vec gas states 
        copy_vec_state_params(&vmuscl->VMst,vst,offset,vsize);
        copy_vec_state_params(&vmuscl->VRst,vst,offset,vsize);
        copy_vec_state_params(&vmuscl->VLst,vst,offset-1,vsize);

        //  put the state variables and source terms in a form more
        //  easily usable by the subroutines
        u = vmuscl->u;   
        ucon = vmuscl->ucon;  
        u[vmuscl->index.density] = ucon[0] = vst->rho + offset;
        u[vmuscl->index.energy] = vst->e + offset;
        ucon[vmuscl->index.energy] = vst->en_den + offset; 
        for (i = 0; i < dim; ++i)
        {
            u[vmuscl->index.v[i]] = vst->v[i] + offset;
            ucon[vmuscl->index.v[i]] = vst->m[i] + offset; 
        }

        if ((src != NULL) && ((source = vmuscl->source) != NULL))
        {
            source[0] = src->mass + offset;
            for (i=0; i < dim; ++i)
                source[vmuscl->index.v[i]] = src->mom[i] + offset;
            source[vmuscl->index.energy] = src->energy + offset;
        }

        if (vmuscl->gravity != NULL)
        {
            float time = fr->time + 0.5*dt;
            float **coords = vst->coords + offset;
            float **g = vmuscl->gravity + offset;

            if (tsten == NULL)
            {
                for (j = 0; j < vsize; ++j)
                {
                    for(i = 0; i < dim; i++)
                        g[i][j] = gravity(coords[j],time)[i];
                }
            }
            else
            {
                printf("ERROR  g_load_unsplit_muscl_state_data,\n");
                printf("Implement for gravity for tangent sweep\n");
                clean_up(ERROR); 
            }
        }

        // The unsplit src is different from split src. 
        // Check if the gravity is correctly added.
        is_src = g_unsplit_muscl_load_source_vectors(swp_num,iperm,tsten,
                 vmuscl->gravity,vst,src,vmuscl->J,offset,vsize,fr->rect_grid);
        vmuscl->is_src = is_src;

#if defined(DEBUG_MUSCL)
        if (debugging("load_Vgas"))
        {
            (void) printf("All quantities before MUSCL computation:\n");

            g_printout_vec_data("The state variables",
                                u[vmuscl->index.density],
                                u[vmuscl->index.energy],
                                u+vmuscl->index.v[0],
                                dim,0,vsize,"muncons");
            g_printout_vec_data("The conserved variables",
                                ucon[0],ucon[vmuscl->index.energy],
                                ucon+vmuscl->index.v[0],dim,0,vsize,"cons");
            if (is_src)
            {
                g_printout_vec_data("The source terms",
                                    source[0],source[1],source+2,dim,1,
                                    vsize-1,"src");
            }

            (void) printf("Miscelaneous quantities:\n");
            (void) printf("%-4s %-14s %-14s %-14s\n","n","pressure",
                          "sound speed","Grun. coef");
            for (i = offset; i < vsize; ++i)
                (void) printf("%-4d %-14g %-14g %-14g\n",
                              i,vst->p[i],vst->c[i],vst->GAM[i]);
            (void) printf("\n");
        }
#endif // defined(DEBUG_MUSCL) 
        return vmuscl;  
}  // end g_load_unsplit_muscl_state_data
****/

EXPORT bool g_unsplit_muscl_load_source_vectors(
        int                swp_num,
        int                *iperm,
        Tan_stencil        *tsten,
        float              **grav,
        Vec_Gas            *vst,
        Vec_Src            *src,
        float              *Jacobi, 
        int                offset,
        int                vs,
        RECT_GRID          *gr)
{
        int                i, j;
        int                dim = gr->dim;
        const float* const *Q = vst->Q;
        float              *rho_src;
        float              *e_src;
        float              *v_src[SMAXD];
        float              *v_src0, *v_srci;
        static  bool       first = YES;
        float              *v0, *rho, *E, *mom[MAXD];
#if defined(ROTATIONAL_SYMMETRY)
        float           *J; 
        float           *pr;
        static  float   alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */

        if (src == NULL)
            return NO;

        rho_src = src->mass + offset;
        e_src = src->energy + offset;
        for (i = 0; i < dim; ++i)
            v_src[i] = src->mom[i] + offset;

        if (first == YES)
        {
            first = NO;
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        for (j = 0; j < vs; ++j)
        {
            rho_src[j] = 0.0;
            e_src[j] = 0.0;
        }
        for (i = 0; i < dim; ++i)
        {
            v_srci = v_src[i];
            for (j = 0; j < vs; ++j)
                v_srci[j] = 0.0;
        }
 
        rho = vst->rho + offset; 
        for (i = 0; i < dim; ++i)
            mom[i] = vst->m[i] + offset; 

        // should add gravity . 
        if (grav != NULL)
        {
            float  **g = grav + offset;
            for(i = 0; i < dim; i++)
            { 
                v_srci = v_src[i];
                for (j = 0; j < vs; ++j)
                    v_srci[j] += rho[j]*g[i][j];
            }
            for (j = 0; j < vs; ++j)
            {
                for(i = 0; i < dim; i++)
                    e_src[j] += mom[i][j]*g[i][j];
            }
        }

        if (tsten == NULL)
        {
            /* Rect sweep */
#if defined(ROTATIONAL_SYMMETRY)
            // Should not depend on the swp_dir ???? for the unsplitting scheme.
            // if ((alpha > 0.0) && (iperm[swp_num]==0)) 
            if ((alpha > 0.0)) 
            {
                /* Include cylindrical source terms */

                float *radii = src->radii + offset;
                float rmin = src->rmin;
                float *m[SMAXD];

                J = Jacobi + offset;
                for(i = 0; i < dim; i++)
                    m[i] = vst->m[i] + offset;
                pr = vst->p + offset;
                rho = vst->rho + offset;
                E = vst->en_den + offset;  
                v0 = vst->v[0] + offset;

                if(debug_flag == YES)
                {
                    printf("print Jacob. Mom[0] for g_unsplit_muscl_load_source_vectors\n");
                    for (j = 0; j < vs; ++j)
                        printf("Jacob[%d] = %g, Mom[0][%d] = %g v0 = %g\n",
                              j, J[j], j, m[0][j], v0[j]); 
                }

                for (j = 0; j < vs; ++j)
                {
                    // what about the gravity ??? on E and Mom. 
                    rho_src[j] -= alpha*m[0][j]/J[j];
                    e_src[j] -= alpha*(pr[j] + E[j])*v0[j]/J[j];  
                    for (i = 0; i < dim; ++i)
                    {
                        v_srci = v_src[i];
                        v_srci[j] -= alpha*m[i][j]*v0[j]/J[j];  
                    }
                }
            }
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }
        else
        {
            /* Tangential sweep */

            printf("ERROR g_unsplit_muscl_load_source_vectors\n"); 
            printf("Entered tangent swp, not implement adding source\n");  
            clean_up(ERROR);  
#if defined(ROTATIONAL_SYMMETRY)
            /* For cylindrical coordinates the mass equation for the
            *  tangential sweep becomes
            *
            *   drho/dt + d(rho*vtan)/dr = - rho*vtan/r * dr/dtan
            *
            *  where dr/dtan is the change in radius with respect to
            *  changes in the tangential coordinate.  If we let
            *  (r, z) = tT + nN, i.e. represent r and z by a combination
            *  of vectors T & N tangent and normal to the curve, dr/dtan
            *  becomes d(tT[0])/dt = T[0] ( = Q[0][0] )
            */

            if (alpha > 0.0)
            {
                float rmin, rad;
                float *mom0;
                POINT **pt;

                mom0 = vst->m[0];
                pt = tsten->p;
                pr = vst->p;
                rho = vst->rho;
                v0 = vst->v[0];

                rmin = src->rmin;
                for (j = 0; j < vs; ++j)
                {
                    rad = pos_radius(Coords(pt[j-2])[0],gr);
                    if (fabs(rad) > fabs(rmin))
                    {
                        rho_src[j] -= alpha*mom0[j]*Q[0][0]/rad;
                        e_src[j] -= alpha*Q[0][0]*v0[j]*pr[j]/(rho[j]*rad);
                    }
                }
            }
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        return YES;  
}


/* Note here, the index is in the order of rho, mom, engery 
 * which is different from the split solver */
LOCAL   Vec_Muscl *g_unsplit_muscl_alloc_wk_space(
        Muscl_Opts *mopts,
        int        vsize,
        int        dim,
        Vec_Muscl  *vmuscl)
{
        float            ***worksp;
        AVISC            Avisc;
        Vec_Eigen        *vegn;
        int              nfloats = mopts->nfloats;
        int              negn = 4;
        int              worksp_len = mopts->worksp_len;
        int              i, nvar_u;

        vmuscl = alloc_Vec_Muscl(vmuscl);
        vegn = &vmuscl->Vegn;

        vmuscl->Opts = *mopts;
        vmuscl->dim = dim;

        nvar_u = 0;
        vmuscl->index.density = nvar_u++;
        for (i = 0; i < dim; ++i)
            vmuscl->index.v[i] = nvar_u++;
        vmuscl->index.energy = nvar_u++;
        vmuscl->nvar_u = nvar_u;

        zero_scalar(vegn,sizeof(Vec_Eigen));
        vegn->negn = negn;
        set_no_alloc(vegn,vegn);
        MATRIX(vegn,lambda,negn,vsize,FLOAT);
        MATRIX(vegn,sgnl,negn,vsize,FLOAT);
        MATRIX(vegn,sgnr,negn,vsize,FLOAT);
        MATRIX(vegn,sgnm,negn,vsize,FLOAT);
        TRI_ARRAY(vegn,l,negn,negn,vsize,FLOAT);
        TRI_ARRAY(vegn,r,negn,negn,vsize,FLOAT);

        if (is_gravity() == YES)
        {
            // VECTOR(vmuscl,grav,vsize,FLOAT);
            set_no_alloc(vmuscl,grav);
            MATRIX(vmuscl,gravity,dim,vsize,FLOAT); 
        }

        /* Jacobi is alloced here */
        VECTOR(vmuscl,J,vsize,FLOAT);
        VECTOR(vmuscl,J_half,vsize,FLOAT);

        TRI_ARRAY(vmuscl,worksp,worksp_len,nfloats,vsize,FLOAT);
        worksp = vmuscl->worksp;

        VECTOR(vmuscl,u,nfloats,sizeof(float*));
        VECTOR(vmuscl,ucon,nfloats,sizeof(float*));
        /* Different from split MUSCL, ucon now 
         * have their own storage.  
        VECTOR(vmuscl,ucon,nfloats,sizeof(float*));
        MATRIX(vmuscl,ucon,nfloats,vsize,FLOAT);
        */
        /* nucon: swp_num = 0, nucon = wave state;
         *        swp_num = 1, nucon = newwave state;  
         */
        MATRIX(vmuscl,pucon,nfloats,vsize,FLOAT);

        if (source_terms_exist() == YES)
            VECTOR(vmuscl,source,nfloats,sizeof(float*));

        use_artificial_dissipation(&Avisc);
        if (use_lapidus_artificial_viscosity(Avisc))
        {
            scalar(&vmuscl->avisc,sizeof(Vec_Avisc));
            set_alloc(vmuscl->avisc,avisc);
            MATRIX(vmuscl->avisc,g,3,vsize,FLOAT);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,cs_ave,worksp[0][0]);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,c_ave,worksp[0][1]);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,vn_ave,worksp[0][2]);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,b,worksp[1]);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,visc,worksp[2][0]);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,uconM,worksp[1]);
            set_no_alloc(vmuscl->avisc,mdlambda);
        }
        else if (use_upwind_artificial_viscosity(Avisc) ||
                 use_linear_artificial_viscosity(Avisc))
        {
            scalar(&vmuscl->avisc,sizeof(Vec_Avisc));
            set_alloc(vmuscl->avisc,avisc);

            MATRIX(vmuscl->avisc,g,1,vsize,sizeof(float));

            set_no_alloc(vmuscl->avisc,cs_ave);
            set_no_alloc(vmuscl->avisc,c_ave);
            set_no_alloc(vmuscl->avisc,vn_ave);
            set_no_alloc(vmuscl->avisc,b);
            set_no_alloc(vmuscl->avisc,visc);
            set_no_alloc(vmuscl->avisc,uconM);
            ASSIGN_ARRAY_POINTER(vmuscl->avisc,mdlambda,worksp[0][0]);
        }

        if (vmuscl->avisc != NULL)
        {
            vmuscl->avisc->use_lapidus=use_lapidus_artificial_viscosity(Avisc);
            vmuscl->avisc->use_linear = use_linear_artificial_viscosity(Avisc);
            vmuscl->avisc->use_upwind = use_upwind_artificial_viscosity(Avisc);
        }

        if (use_muscl_slope_flattening(Avisc))
        {
            scalar(&vmuscl->msf,sizeof(Vec_MSF));
            set_alloc(vmuscl->msf,msf);
            VECTOR(vmuscl->msf,chi,vsize,FLOAT);
        }
        else
            vmuscl->msf = NULL;
        vmuscl->monotone_reconstruction = mopts->monotone_reconstruction;
        vmuscl->link_reconstructions = mopts->link_reconstructions;

        vmuscl->max_vsize = vsize;

                /* Set Linear reconstruction data structure */
        ASSIGN_ARRAY_POINTER(vmuscl,backward,worksp[0]);
        ASSIGN_ARRAY_POINTER(vmuscl,central,worksp[1]);
        ASSIGN_ARRAY_POINTER(vmuscl,forward,worksp[2]);
        ASSIGN_ARRAY_POINTER(vmuscl,du,worksp[2]);
        ASSIGN_ARRAY_POINTER(vmuscl,q,worksp[3]);

                /* Set half step calculation data */

        ASSIGN_ARRAY_POINTER(vmuscl,uL,worksp[0]);
        ASSIGN_ARRAY_POINTER(vmuscl,uR,worksp[1]);
        ASSIGN_ARRAY_POINTER(vmuscl,uM,worksp[2]);
        ASSIGN_ARRAY_POINTER(vmuscl,right_wvs,worksp[2]);
        ASSIGN_ARRAY_POINTER(vmuscl,left_wvs,worksp[2]);
        ASSIGN_ARRAY_POINTER(vmuscl,Flux.F,worksp[3]);
        ASSIGN_ARRAY_POINTER(vmuscl,source_sum,worksp[3]);
        ASSIGN_ARRAY_POINTER(vmuscl,awv,worksp[3]);
        ASSIGN_ARRAY_POINTER(vmuscl,dq,worksp[4]);

                /* Set Riemann solver date */
        vmuscl->pL = g_set_vmuscl_vec_gas_storage(vmuscl->uL,&vmuscl->VLst,
                                                  vmuscl, YES);
        vmuscl->pR = g_set_vmuscl_vec_gas_storage(vmuscl->uR,&vmuscl->VRst,
                                                  vmuscl, YES);
        vmuscl->pM = g_set_vmuscl_vec_gas_storage(vmuscl->uM,&vmuscl->VMst,
                                                  vmuscl, NO);
        set_no_alloc(vmuscl,A);
        set_no_alloc(vmuscl,dV);
        return vmuscl;
}   /* end g_unsplit_muscl_alloc_wk_space */ 


LOCAL void g_unsplit_muscl_flux_vectors(
        int        start,
        int        end,
        int        swp_num,
        int        *iperm,
        float      **u,
        float      *p,
        MUSCL_FLUX *Flux,
        Vec_Muscl  *vmuscl)
{
        float *rho = u[vmuscl->index.density];
        float *e = u[vmuscl->index.energy];
        float *v0, *v1, *v2;
        float *F0, *F1, *F2, *F3, *F4;
        float **F = Flux->F;
        int   dim = vmuscl->dim;
        int   j, idirs[MAXD];

        debug_print("mflux","Entered g_unsplit_muscl_flux_vectors()\n");

        for(j = 0; j < dim; j++)
            idirs[j] = iperm[(j+swp_num)%dim]; 

        switch (dim)
        {
        /*  Need to check
        case 1:
            v0 = u[vmuscl->index.v[0]];
            F0  = F[0]; F1 = F[1]; F2 = F[2];
            for (j = start; j < end; ++j)
            {
                // mass 
                F0[j] = rho[j]*v0[j];
                // energy 
                F1[j] = v0[j]*(rho[j]*(0.5*sqr(v0[j]) + e[j]) + p[j]);
                // momentum 
                F2[j] = F0[j]*v0[j] + p[j];
            }
            break;
        */
        case 2:
            v0 = u[vmuscl->index.v[idirs[0]]]; v1 = u[vmuscl->index.v[idirs[1]]];
            F0  = F[vmuscl->index.density];
            F1 = F[vmuscl->index.energy];
            F2 = F[vmuscl->index.v[0]];
            F3 = F[vmuscl->index.v[1]];

            for (j = start; j < end; ++j)
            {
                /* mass */
                F0[j] = rho[j]*v0[j];
                /* energy */
                F1[j] = v0[j]*
                    (rho[j]*(0.5*(sqr(v0[j])+sqr(v1[j]))+e[j])+p[j]);
            }

            /* There is no permutation in mom */
            if(idirs[0] == 0)
            {
                for (j = start; j < end; ++j)
                {
                    /* Sweep component of momentum */
                    F2[j] = F0[j]*v0[j] + p[j];
                    /* Off sweep component of momentum */
                    F3[j] = F0[j]*v1[j];
                }
            }
            else
            {
                for (j = start; j < end; ++j)
                {
                    /* Off sweep component of momentum */
                    F2[j] = F0[j]*v1[j];
                    /* Sweep component of momentum */
                    F3[j] = F0[j]*v0[j] + p[j];
                }
            }
            break;

        case 1:
        case 3:
        default:
            screen("ERROR in g_unsplit_muscl_flux_vectors(), "
                   "invalid dimension %d\n",dim);
            clean_up(ERROR);
            break;
        }

        debug_print("mflux","Left g_unsplit_muscl_flux_vectors()\n");
}

LOCAL void g_2dunsplit_add_art_visc2(
        int             start,
        int             end,
        Vec_Muscl       *vmuscl)
{
        Vec_Avisc       *avisc;
        int             dim, j, k;
        float           *g0;
        float           *uconk, *Fk;
        float           **ucon, **F;
        Vec_Gas         *vst = vmuscl->vst;
        float           uconk1, uconk2;

        if ((avisc = vmuscl->avisc) == NULL)
            return;
        dim = vmuscl->dim;
        F = vmuscl->Flux.F;
        g0 = avisc->g[0];
        for (k = 0; k < dim+2; ++k)
        {
            Fk = F[k];
            for (j = start; j < end; ++j)
            {
                if (k == vmuscl->index.density)
                {
                    uconk2 = vst->rho[j];
                    uconk1 = vst->rho[j-1];
                }
                else if (k == vmuscl->index.energy)
                {
                    uconk2 = vst->en_den[j];
                    uconk1 = vst->en_den[j-1];
                }
                else if (k == vmuscl->index.v[0])
                {
                    uconk2 = vst->m[0][j];
                    uconk1 = vst->m[0][j-1];
                }
                else if (k == vmuscl->index.v[1])
                {
                    uconk2 = vst->m[1][j];
                    uconk1 = vst->m[1][j-1];
                }
                Fk[j] -= 0.5*g0[j-1]*(uconk2 - uconk1);
            }
        }
}

/*ARGSUSED*/
EXPORT  void unsplit_point_FD(
        float           *dh,
        float           dt,
        Locstate        ans,
        const float     *dir,
        int             swp_num,
        int             *iperm,
        int             *index,
        Stencil         *sten)
{
        Front           *fr = sten->fr;
        Front           *newfr = sten->newfr;
        RECT_GRID       *gr = fr->rect_grid;
        Locstate        st, *state;
        Wave            *wave = sten->wave;
        Wave            *newwave = sten->newwave;
        float           *rho, *en_den;
        float           *m[MAXD];
        float           **coords;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        float           *vacuum_dens;
        float           *min_pressure;
        float           *min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
#if defined(ROTATIONAL_SYMMETRY)
        static float    alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
        int             i, j, side;
        int             dim = gr->dim;
        int             idirs[MAXD];
        static Vec_Gas  *vst = NULL;
        static Vec_Src  *src = NULL;
        static Vec_Gas  *Tvst[2] = {NULL, NULL}; 
        static int      endpt, npts;
        static float    **Q = NULL;

        // Set the local debug_flag
        if(sten != NULL)
        {
            if(sten->icoords[0][0] == 35 && 
               sten->icoords[0][1] == 3)
                debug_flag = YES;
        }

        if (is_obstacle_state(sten->st[0]))
        {
            g_obstacle_state(ans,sten->fr->sizest);
            return;
        }
        if (vst == NULL)
        {
            /* Assumes stencil size never changes */

            npts = sten->npts;
            endpt = npts/2;
            // g_unsplit_muscl_alloc_phys_vecs;
            alloc_unsplit_phys_vecs(wave,npts);
            vst = g_wave_vgas(wave);
            src = g_wave_vsrc(wave);
            Tvst[0] = g_wave_Tvgas(wave)[0]; 
            Tvst[1] = g_wave_Tvgas(wave)[1]; 

            g_wave_vgas(wave) = NULL;
            g_wave_vsrc(wave) = NULL;
            g_wave_Tvgas(wave)[0] = NULL;  
            g_wave_Tvgas(wave)[1] = NULL;  
            matrix(&Q,3,3,FLOAT);
            vst->Q = (const float* const*)Q;
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        if (RegionIsFlowSpecified(ans,sten->st[0],Coords(sten->p[0]),
                    sten->newcomp,sten->newcomp,fr))
            return;

        clear_Vec_Gas_set_flags(vst);
        clear_Vec_Gas_set_flags(Tvst[0]);
        clear_Vec_Gas_set_flags(Tvst[1]);
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                Q[i][j] = 0.0;
        for (i = 0; i < dim; ++i)
        {
            idirs[i] = iperm[(i+swp_num)%dim];
            m[i] = vst->m[i];
            Q[i][idirs[i]] = 1.0;
        }
        for (; i < 3; ++i)
            Q[i][i] = 1.0;
        rho = vst->rho;
        state = vst->state;
        coords = vst->coords;
        en_den = vst->en_den;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        vacuum_dens = vst->vacuum_dens;
        min_pressure = vst->min_pressure;
        min_energy = vst->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        for (i = 0; i < npts; ++i)
        {
            state[i] = st = sten->ststore[i];
            coords[i] = Coords(sten->pstore[i]);
            rho[i] = Dens(st);
            en_den[i] = Energy(st);
            for (j = 0; j < dim; ++j)
                m[j][i] = Mom(st)[j];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            vacuum_dens[i] = Vacuum_dens(st);
            min_pressure[i] = Min_pressure(st);
            min_energy[i] = Min_energy(st);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        }
        Vec_Gas_field_set(vst,state) = YES;
        Vec_Gas_field_set(vst,coords) = YES;
        Vec_Gas_field_set(vst,rho) = YES;
        Vec_Gas_field_set(vst,en_den) = YES;
        Vec_Gas_field_set(vst,m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        Vec_Gas_field_set(vst,vacuum_dens) = YES;
        Vec_Gas_field_set(vst,min_pressure) = YES;
        Vec_Gas_field_set(vst,min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        set_params_jumps(vst,0,npts);

#if defined(ROTATIONAL_SYMMETRY)
        /* If using cylindrical coordinates and this is the
         *  r-sweep include radius information for source
         *  computation.
         */

        if (alpha > 0.0 && iperm[swp_num]==0)
        {
            float *radii = src->radii;

            src->rmin = fabs(pos_radius(0.0,gr));
            for (i = 0; i < npts; ++i)
                radii[i] = pos_radius(Coords(sten->p[0])[0]+
                              (i-endpt)*dh[idirs[0]],gr);
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        for(side = 0; side <= 1; side++)
        {
            for (i = 0; i < dim; ++i)
                m[i] = Tvst[side]->m[i];
            rho = Tvst[side]->rho;
            state = Tvst[side]->state;
            en_den = Tvst[side]->en_den;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            vacuum_dens = Tvst[side]->vacuum_dens;
            min_pressure = Tvst[side]->min_pressure;
            min_energy = Tvst[side]->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            for(i = -1; i <= 1; i++)
            {
                state[i+2] = st = sten->Tst[side][i];  
                rho[i+2] = Dens(st);
                en_den[i+2] = Energy(st);
                for (j = 0; j < dim; ++j)
                    m[j][i+2] = Mom(st)[j];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
                vacuum_dens[i+2] = Vacuum_dens(st);
                min_pressure[i+2] = Min_pressure(st);
                min_energy[i+2] = Min_energy(st);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            }
            Vec_Gas_field_set(Tvst[side],state) = YES;
            // Vec_Gas_field_set(Tvst[side],coords) = YES;
            Vec_Gas_field_set(Tvst[side],rho) = YES;
            Vec_Gas_field_set(Tvst[side],en_den) = YES;
            Vec_Gas_field_set(Tvst[side],m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            Vec_Gas_field_set(Tvst[side],vacuum_dens) = YES;
            Vec_Gas_field_set(Tvst[side],min_pressure) = YES;
            Vec_Gas_field_set(Tvst[side],min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            // set_params_jumps(Tvst[side],0,npts);
        }

        /*
        if(debug_flag == YES)
        {
            print_int_vector("icoords",sten->icoords[0],2," "); 
            printf(" in dir[%d] at %d sweep\n", idirs[0], swp_num); 
            for (i = 0; i < npts; ++i)
            {
                (void) printf("state[%d]\n",i);
                g_verbose_print_state(vst->state[i]);
            }
            for(side = 0; side <= 1; side++)
            {
                for(i = 1; i <= 3; i++)
                {
                    (void) printf("Tstate[%d][%d]\n",side, i);
                    g_verbose_print_state(Tvst[side]->state[i]);
                }
            }
        }
        */

        unsplit_2d_MUSCL_in_dir(swp_num,iperm,sten->icoords[0],
                      wave,sten->swap_wv,fr,newfr,sten,NULL,
                      0,npts,vst,Tvst,src,dt,dh,dim,0);

        Dens(ans) = vst->rho[endpt];
        Energy(ans) = vst->en_den[endpt];
        for (i = 0; i < dim; ++i)
            Mom(ans)[i] = vst->m[i][endpt];
        Set_params(ans,vst->state[endpt]);
        set_type_of_state(ans,GAS_STATE);

        if(debug_flag == YES)
            debug_flag = NO; 

#if defined(CHECK_FOR_BAD_STATES)
        if (debugging("bad_state") && 
            is_bad_state(ans,YES,"unsplit_point_FD") &&
            swp_num == 1)
        {
            screen("ERROR in unsplit_point_FD(), bad state found for cell ");
            print_int_vector("icrds",sten->icoords[0],2," "); 
            printf("in dir[%d] at %d sweep\n", idirs[0], swp_num); 
            // fprint_raw_gas_data(stdout,ans,dim);
            g_verbose_print_state(ans);  
            for (i = 0; i < npts; ++i)
            {
                (void) printf("state[%d]\n",i);
                // fprint_raw_gas_data(stdout,vst->state[i],dim);
                g_verbose_print_state(vst->state[i]);  
            }
            /*
            for (i = 0; i < npts; ++i)
            {
                (void) printf("sten store_st[%d]",i);
                fprint_raw_gas_data(stdout,sten->ststore[i],dim);
            }
            */
            clean_up(ERROR);
        }
#endif /* defined(CHECK_FOR_BAD_STATES) */

}  /* End of unsplit_point_FD */

EXPORT void set_unsplit_muscl_debug_flag(
       int  yes_or_no)
{
       debug_flag = yes_or_no; 
}


#endif /* #if defined(CONSERVATIVE_ALG) && defined(TWOD) */
