/*
*                               ghypvol.c
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*/

#include <ghyp/ghyp.h>
#include <gdecs/vecdecs.h>

#if defined(CONSERVATIVE_ALG) && defined(TWOD)

#define DEBUG_STRING "ghypvol"

#define DEBUG  YES

/* Copied from trigrid1.c */
#if !defined(_NTOL)
#if defined(float)
#define _NTOL 1.0e-5
#define _TOL  1.0e-6
#else /* defined(float) */
#define _NTOL 1.0e-3
#define _TOL  1.0e-4
#endif /* defined(float) */

LOCAL const float NTOL     = _NTOL;     /*TOLERANCE*/
LOCAL const float TOL      = _TOL;      /*TOLERANCE*/
LOCAL const float ONEMNTOL = 1.0 - _NTOL; /*TOLERANCE*/
LOCAL const float ONEMTOL  = 1.0 - _TOL;  /*TOLERANCE*/
#endif /* if !defined(_NTOL) */

struct _Face_Stencil{
        int             npts;
        float           nor[3]; /* The normal of faces */
        float           centroid[3];  /* centroid of these faces */
        float           area;   /* Total area of these faces */
        CSG_Face        **f;
        int             nface;  /* Number of the faces on this normal side */ 
        int             alloc_nface; /* The maximum number of faces allocated */
        CSG_Face        *maxf;  /* Face on this side with maximum area */
        Locstate        btm_st; /* Save the btm state of the volume */
        float           b_area; /* Sum of buttom areas */
        float           t_area; /* Sum of top areas */
        float           b_centroid[3]; /* buttom areas centroid */
        float           t_centroid[3]; /* top areas centroid */

        Locstate        *st, *ststore;
                     /* in face normal dir, inner side = 0, */
                     /* outter side = 1, , st[0], st[1], etc */
        Locstate        *tan_st, *tan_ststore;
                     /* tangential state   */
        Locstate        *outtan_st, *outtan_ststore;
        int             ic[MAXD];
        int             outic[MAXD];
        CSG_Solid       *s, *outs;

        bool            is_grav; 
        float           grav[MAXD]; 
        float           scon[MAXD+2]; /* den, mom0, mom1 */
        float           source[MAXD+2]; 
        int             nsrc; 
#if defined(ROTATIONAL_SYMMETRY)
        float           P;   /* pressure sum on the side faces */
        int             np;  /* number of side faces */
#endif /* defined(ROTATIONAL_SYMMETRY) */

        float           **crds;   /* In the normal dir, size [4][3] */
        int             **icrds;  /* In normal dir, crds and icrds are not guranteed 
                                   * inside the computational domain. 
                                   */

        float           **Tcrds;   /* In the Tangent dir, size [2][3], inner side */
        int             **Ticrds;  /* In tangent dir, crds and icrds are not guranteed 
                                   * inside the computational domain. 
                                   */

        float           **oTcrds;   /* In the Tangent dir, size [2][3], outter side */
        int             **oTicrds;  /* In tangent dir, crds and icrds are not guranteed 
                                   * inside the computational domain. 
                                   */
        Wave            *wave;
        Wave            *newwave;
        Front           *front;
        Front           *newfront;  
};
typedef struct _Face_Stencil Face_Stencil;

/* LOCAL GLOBALs */
LOCAL   int     gmax[MAXD], ggmax[MAXD], ggmin[MAXD], vsten_rad, is_vec_method;
LOCAL   int     lbuffed[MAXD], ubuffed[MAXD];
LOCAL   int     dim;
LOCAL   int     lbuf[MAXD], ubuf[MAXD];
LOCAL   float   dh[MAXD];
LOCAL   float   L[MAXD], VL[MAXD], VU[MAXD];
LOCAL   float   nor[6][3] = { 1.0, 0.0, 0.0,
                              0.0, 1.0, 0.0,
                              -1.0, 0.0, 0.0,
                              0.0, -1.0, 0.0,
                              0.0, 0.0, 1.0,
                              0.0, 0.0, -1.0};
LOCAL   byte    **stintfc_st_pool = NULL;
LOCAL   int     alloc_st_n, alloc_st_pool; 
LOCAL   int     st_used, st_pool_used; 

/* LOCAL FUNCTIONs */
LOCAL Face_Stencil   *alloc_face_flux_stencil(Wave*,Front*);
LOCAL void           set_vol_globals(Wave*);
LOCAL int            count_faces_on_this_side(CSG_Solid*,float*);
LOCAL void           collect_sten_faces_on_side(CSG_Solid*,float*,int,Face_Stencil*);  
LOCAL void           g_sten_st_for_inter_face(Face_Stencil*,CSG_Solid*,int*,float*); 
LOCAL void           g_sten_st_for_boundary_face(Face_Stencil*,int*,float*); 

LOCAL CSG_Solid      *find_out_nghbr(CSG_Face*,int*,Wave*,
                        Front*,Wave*,Front*,int*,int*,int*,int*); 
/*** Replaced by is_interior_face().
LOCAL int            is_inter_face(CSG_Face*); 
***/
LOCAL void           btm_state2d(Locstate,CSG_Solid*,Wave*,Front*); 
LOCAL void           verbose_print_btm_state2d(CSG_Solid*);
LOCAL void           g_sten_tangnt_st_for_inter_face(Face_Stencil*,float*);
LOCAL CSG_Face       *large_face_in_dir(CSG_Solid*,float*);
LOCAL void           g_face_centroid_flux(Face_Stencil*,CSG_Solid*,int,float,float*,float*); 
LOCAL int            face_on_boundary( CSG_Face*,int*,int*,int*);
/*** Replaced by find_match_face_in_region4()
LOCAL CSG_Face       *find_match_face_in_region3(CSG_Face*,int*,TRI_GRID*);
***/
LOCAL void           off_face_pt(CSG_Face*,float*,float);
LOCAL void           point_with_rect(float*,RECT_GRID*,int*,int*,int*,int*,int*);  
LOCAL void           find_outter_st2(Locstate,int*,float*,Face_Stencil*); 
LOCAL void           set_face_sten_pt_crds(Face_Stencil*,CSG_Solid*,int*,float*);  
LOCAL void           get_state_in_cell2(float*,int*,COMPONENT,Locstate,Wave*,Wave*,Front*); 
LOCAL void           get_state_in_cell3(float*,int*,COMPONENT,Locstate,Wave*,Wave*,Front*); 
LOCAL CSG_Solid      *pt_on_btm_of_volume(float*,int*,Wave*); 
LOCAL int            point_on_volume_btm(float*,CSG_Solid*); 
LOCAL CRXING         *cross_physics_boundary(float*,Wave*,Front*,int*,int*);  
LOCAL void           ipoint_with_rect(int*,int*,int*,int*,int*); 
LOCAL void           set_cross_physics_boundary_st(float*,COMPONENT,CRXING*,Front*,Wave*,Locstate);
LOCAL void           g_consv_muscl_flux(Face_Stencil*,float*,int,int*,Wave*,Front*,Front*,int,
                         int,Vec_Gas*,Vec_Gas**,Vec_Src*,float,float*,int,float*); 
LOCAL void           g_front_face_flux(Face_Stencil*,CSG_Solid*,int*,float,float*);
LOCAL void           g_tracked_front_face_flux(Face_Stencil*,CSG_Face*,CSG_Solid*,int*,float,float*); 
LOCAL int            front_face(CSG_Face*,int*); 
/*** Replaced with find_match_face_in_region4()
LOCAL CSG_Face       *face_on_other_side(CSG_Face*,CSG_Solid*,int*,Wave*);
***/
LOCAL void           vol_intgel_normal(float,float*,float*); 
LOCAL void           set_sten_vol_geom_par(Face_Stencil*,CSG_Solid*); 
LOCAL void           print_face_sten(Face_Stencil*);
LOCAL void           g_neumann_front_face_state(Face_Stencil*);
LOCAL void           g_neumann_front_face_flux(Face_Stencil*,CSG_Face*,CSG_Solid*,int*,
                            float,float*);
LOCAL void           g_oblique_neumann_front_face_flux(Face_Stencil*,CSG_Face*,CSG_Solid*,
                            int*,float,float*);
LOCAL void           print_face_sten_state(Face_Stencil*);
LOCAL int            oblique_face(CSG_Face*);

/* VERSION 2 code */
LOCAL void           g_face_centroid_state(Face_Stencil*,CSG_Solid*,int,float,float*); 
LOCAL void           g_consv_muscl_soln(Face_Stencil*,float*,int,int*,Wave*,Front*,Front*,
                         int,int,Vec_Gas*,Vec_Gas**,Vec_Src*,float,float*,int); 
LOCAL int            match_sten_face_mid_st(Face_Stencil*); 
LOCAL Locstate       stintfc_st_pool_to_smid(size_t);
LOCAL void           g_front_face_midst(Face_Stencil*,CSG_Solid*,int*,float); 
LOCAL void           g_tracked_front_face_midst(Face_Stencil*,CSG_Face*,
                         CSG_Solid*,int*,float); 
LOCAL void           g_oblique_neumann_front_face_midst(Face_Stencil*,CSG_Face*,
                         CSG_Solid*,int*,float);
LOCAL void           g_face_flux(Face_Stencil*,CSG_Face*,float,float,float*);
LOCAL void           g_h_order_tracked_front_face_midst(Face_Stencil*,CSG_Face*,
                          CSG_Solid*,int*,float);
LOCAL void           gvol_compute_Jacobi(Face_Stencil*,int,int*,int,int,
                          Vec_Muscl*,Wave*,Vec_Gas*,int); 

/******************/

/* Artificial Viscosity code */
LOCAL void           g_add_contrl_vol_art_viscsity(Face_Stencil*,CSG_Face*,
                          float,float*,float*,float);
LOCAL void           g_comp_contrl_vol_art_visc2(Vec_Gas*,
                          Vec_Muscl*,Locstate);
/*****************************/


LOCAL void set_vol_globals(
        Wave      *wave)
{
        RECT_GRID       *r_grid;
        int             i;

        r_grid = wave->rect_grid;
        dim = r_grid->dim;
        for (i = 0; i < dim; ++i)
        {
            gmax[i] = r_grid->gmax[i];
            dh[i] = r_grid->h[i];
            L[i] = r_grid->L[i];
            VL[i] = r_grid->VL[i];
            VU[i] = r_grid->VU[i];
            lbuf[i] = r_grid->lbuf[i];
            ubuf[i] = r_grid->ubuf[i];
            ggmin[i] = 0;
            ggmax[i] = r_grid->gmax[i];
            ggmin[i] -= r_grid->lbuf[i];
            ggmax[i] += r_grid->ubuf[i];
            if(lbuf[i] != 0)
                lbuffed[i] = YES;
            else
                lbuffed[i] = NO;
            if(ubuf[i] != 0)
                ubuffed[i] = YES;
            else
                ubuffed[i] = NO;
        }
        vsten_rad = vsten_radius(wave);
        is_vec_method = is_vector_method(wave);
}


LOCAL Face_Stencil *alloc_face_flux_stencil(
        Wave      *wave,
        Front     *fr)
{
        int           nrad, num_pts;
        Face_Stencil  *f_sten;
        size_t        sizest = fr->sizest;
        int           i;

        num_pts = wave->npts_sten;
        nrad = stencil_radius(wave);

        scalar(&f_sten,sizeof(Face_Stencil));
        vector(&f_sten->ststore,num_pts,sizeof(Locstate));
        vector(&f_sten->tan_ststore,3,sizeof(Locstate));
        vector(&f_sten->outtan_ststore,3,sizeof(Locstate));
        matrix(&f_sten->crds,4,3,sizeof(float));             
        matrix(&f_sten->icrds,4,3,sizeof(int));             

        matrix(&f_sten->Tcrds,2,3,sizeof(float));             
        matrix(&f_sten->Ticrds,2,3,sizeof(int));             
        matrix(&f_sten->oTcrds,2,3,sizeof(float));             
        matrix(&f_sten->oTicrds,2,3,sizeof(int));             

        alloc_state(fr->interf,&f_sten->btm_st,max(sizeof(VGas),sizest));
        for(i = 0; i < num_pts; i++)
            alloc_state(fr->interf,&f_sten->ststore[i],max(sizeof(VGas),sizest));
        for(i = 0; i < 3; i++)
        {
            alloc_state(fr->interf,&f_sten->tan_ststore[i],max(sizeof(VGas),sizest));
            alloc_state(fr->interf,&f_sten->outtan_ststore[i],max(sizeof(VGas),sizest));
        }
        f_sten->st = f_sten->ststore + nrad;
        f_sten->tan_st = f_sten->tan_ststore + 1;
        f_sten->outtan_st = f_sten->outtan_ststore + 1;
        f_sten->f = NULL; 
        f_sten->npts = num_pts;
        f_sten->alloc_nface = 0; 

        return f_sten;
}

EXPORT void g_alloc_stintfc_st_pool(
        Wave           *wave)
{
        int            i, dim, default_alloc;
        int            *gmax; 
        size_t         sizest = wave->sizest; 
        byte           *stintfc_st_storage = NULL;

        dim = wave->rect_grid->dim;
        gmax = wave->rect_grid->gmax; 
        default_alloc = 0;  
        for(i = 0; i < dim; i++)
            default_alloc = max(default_alloc, gmax[i]); 
        
        //alloc_st_n = wave_tri_soln(wave)->tri_grid->n_c_crx*4+20; 
        alloc_st_n = (wave_tri_soln(wave)->tri_grid->n_c_crx != 0 ?
                      wave_tri_soln(wave)->tri_grid->n_c_crx*4 : 
                      default_alloc); 
        alloc_st_pool = 4; 
        vector(&stintfc_st_pool,alloc_st_pool,sizeof(byte*)); 
        vector(&stintfc_st_storage,alloc_st_n,sizest); 
        stintfc_st_pool[0] = stintfc_st_storage;  
        st_pool_used = 0; 
        st_used = 0; 
}

EXPORT void g_free_stintfc_st_pool(void)
{
        int  i;  
        for(i = 0; i < st_pool_used+1; i++)
            free(stintfc_st_pool[i]); 
        free(stintfc_st_pool); 

        stintfc_st_pool = NULL; 
        alloc_st_n = alloc_st_pool =
        st_used = st_pool_used = 0; 
}

EXPORT void contrl_vol_update(
        int             *ic,
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave,
        Wave            *nwave,
        Front           *fr,
        Front           *newfr,
        float           dt,
        float           *dh)
{
        RECT_GRID       *gr = fr->rect_grid;
        size_t          sizest = fr->sizest;
        static Locstate newst = NULL;
        static Face_Stencil *f_sten = NULL;
        CSG_Face        *f;
        int             i, j, where[3];
        float           *dir;
        float           flux[4] = {0.0, 0.0, 0.0, 0.0};
        float           nucon[4] = {0.0, 0.0, 0.0, 0.0};
        float           oucon[4] = {0.0, 0.0, 0.0, 0.0};
        float           Jt, Jb;

#if defined(ROTATIONAL_SYMMETRY)
        static float    alpha;
#endif /* if defined(ROTATIONAL_SYMMETRY) */

        debug_print("ghyp", "Entered contrl_vol_update()\n");

        if(f_sten == NULL)
        {
            scalar(&f_sten,sizeof(Face_Stencil));
            if(is_gravity() == YES)
                f_sten->is_grav = YES;
            else 
                f_sten->is_grav = NO;
            alloc_state(fr->interf,&f_sten->btm_st,
		        max(sizeof(VGas),sizest));
            alloc_state(fr->interf, &newst, sizest);
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        f_sten->s = s;
        f_sten->ic[0] = ic[0]; f_sten->ic[1] = ic[1];
        f_sten->wave = wave; f_sten->newwave = nwave;
        f_sten->front = fr;  f_sten->newfront = newfr;
        f_sten->P = 0.0;
        f_sten->nsrc = f_sten->np = 0;
        for(i = 0; i < dim+2; i++)
            f_sten->scon[i] = f_sten->source[i] = 0.0;  

        btm_state2d(f_sten->btm_st, s, wave, fr);
        set_sten_vol_geom_par(f_sten, s);
        if(f_sten->is_grav == YES)
        {
            float volcent[MAXD], time; 
            time = fr->time + 0.5*dt; 
            for(i = 0; i < dim; i++)
            {
                volcent[i] = 0.5*(f_sten->t_centroid[i] + 
                                  f_sten->b_centroid[i]);  
            }
            for(i = 0; i < dim; i++)
                f_sten->grav[i] = gravity(volcent,time)[i];
        }

        Jt = Jb = 1.0;
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            /***
            Jt = f_sten->t_centroid[0];
            if(fabs(Jt) < 1.0e-04)
                Jt = (Jt < 0.0)? -0.05*dh[0] : 0.05*dh[0];
            Jb = f_sten->b_centroid[0];
            if(fabs(Jb) < 1.0e-04)
                Jb = (Jb < 0.0)? -0.05*dh[0] : 0.05*dh[0];
            ***/
            Jt = f_sten->t_centroid[0];
            Jb = f_sten->b_centroid[0];
            /*
            if(fabs(Jt) < 1.0e-06 && fabs(Jb) < 1.0e-06)
            {
                // Jt and Jb keeps the same sign 
                Jt = (Jt < 0.0)? -0.005*dh[0] : 0.005*dh[0];
                Jb = (Jt < 0.0)? -0.005*dh[0] : 0.005*dh[0];
            }
            else if(fabs(Jt) < 1.0e-06)
                Jt = (Jt < 0.0)? -0.005*dh[0] : 0.005*dh[0];
            else if(fabs(Jb) < 1.0e-06)
                Jb = (Jb < 0.0)? -0.005*dh[0] : 0.005*dh[0];
            */
            if(fabs(Jt) < 0.5*dh[0] && fabs(Jb) < 0.5*dh[0])
            {
                // Jt and Jb keeps the same sign 
                Jt = (Jt < 0.0)? -0.5*dh[0] : 0.5*dh[0];
                Jb = (Jt < 0.0)? -0.5*dh[0] : 0.5*dh[0];
            }
            else if(fabs(Jt) < 0.5*dh[0])
                Jt = (Jt < 0.0)? -0.5*dh[0] : 0.5*dh[0];
            else if(fabs(Jb) < 0.5*dh[0])
                Jb = (Jb < 0.0)? -0.5*dh[0] : 0.5*dh[0];
        }
#endif /* if defined(ROTATIONAL_SYMMETRY) */

        oucon[0] = f_sten->b_area/f_sten->t_area*Dens(f_sten->btm_st)*(Jb/Jt);
        oucon[1] = f_sten->b_area/f_sten->t_area*Mom(f_sten->btm_st)[0]*(Jb/Jt);
        oucon[2] = f_sten->b_area/f_sten->t_area*Mom(f_sten->btm_st)[1]*(Jb/Jt);
        oucon[3] = f_sten->b_area/f_sten->t_area*Energy(f_sten->btm_st)*(Jb/Jt);
        
        f = s->sfaces;
        while(f)
        {
            f_sten->maxf = f; 
            if(YES == is_top_or_bottom_face(f,where))
            {
                f = f->nextf;
                continue; 
            }
            g_face_flux(f_sten,f,dt,Jt,flux); 
            f = f->nextf;
        }

        if(f_sten->is_grav == YES)
        {
            for(i = 0; i < dim+2; i++)
                f_sten->scon[i] /= f_sten->nsrc; 
            f_sten->source[1] = f_sten->scon[0]*
               f_sten->grav[0]*s->vol/f_sten->t_area;
            f_sten->source[2] = f_sten->scon[0]*
               f_sten->grav[1]*s->vol/f_sten->t_area;
            f_sten->source[3] = 
               (f_sten->scon[1]*f_sten->grav[0] + 
                f_sten->scon[2]*f_sten->grav[1])*
               s->vol/f_sten->t_area;
        }

        for(i = 0; i < 4; i++)
            nucon[i] = oucon[i] - flux[i] + f_sten->source[i]/Jt;

#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            if(f_sten->np != 0)
                nucon[1] += f_sten->P/f_sten->np*
			s->vol/(Jt*f_sten->t_area);
            else
            {
                printf("ERROR contrl_vol_update(),"
			" side Pressure is 0:");
	        clean_up(ERROR);
	    }
        }
#endif /* if defined(ROTATIONAL_SYMMETRY) */

#if defined(DEBUG)  
        if(f_sten->s->debug == YES)
        {
            printf("\n----------------\n"); 
            printf("Update the volume[%d] states\n", s->solidno); 
            printf("Source pres[np = %d] = %14.12f Ratiod %14.12f\n", f_sten->np,
                        f_sten->P, f_sten->P/f_sten->np*
                        s->vol/(Jt*f_sten->t_area)); 
            printf("mom[0] flux vary: (-flux[1] + vol_pres) = %14.12f\n",
                        -flux[1] + f_sten->P/f_sten->np*
                        s->vol/(Jt*f_sten->t_area));
            printf("Jacob[%10.8f %10.8f], Vol[%12.10f] T_area[%12.10f], B_area[%12.10f]\n",
                   Jt, Jb, s->vol, f_sten->t_area, f_sten->b_area);
            printf("Ratiod value of Ucon:\n");
            printf("ucon_OLD <%14.12f, %14.12f, %14.12f, %14.12f>\n",
                      oucon[0], oucon[1], oucon[2], oucon[3]);
            printf("flux  <%14.12f, %14.12f, %14.12f, %14.12f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
            printf("ucon_NEW <%14.12f, %14.12f, %14.12f, %14.12f>\n",
                      nucon[0], nucon[1], nucon[2], nucon[3]);
            printf("source <%14.12f, %14.12f, %14.12f, %14.12f>\n",
                        f_sten->source[0], f_sten->source[1], 
                        f_sten->source[2], f_sten->source[3]);
 
        }
#endif /* if defined(DEBUG) */

        Dens(newst) = nucon[0];
        Mom(newst)[0] = nucon[1];
        Mom(newst)[1] = nucon[2];
        Energy(newst) = nucon[3];
        Set_params(newst, f_sten->btm_st);
        set_type_of_state(newst,GAS_STATE);

        if(debugging("bad_state"))
        {
            if(is_bad_state(newst,YES,"contrl_vol_update"))
            {
                printf("ERROR contrl_vol_update(), new state is bad:");
                printf(" For solid %d\n", s->solidno); 
                g_verbose_print_state(newst);
                printf("Old state is:");
                g_verbose_print_state(f_sten->btm_st);
                printf("Jacob[%10.8f %10.8f], Vol[%12.10f] T_area[%12.10f], B_area[%12.10f]\n",
                   Jt, Jb, s->vol, f_sten->t_area, f_sten->b_area);
                printf("Ratioed value:\n");
                printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          oucon[0], oucon[1], oucon[2], oucon[3]);
                printf("ucon_NEW <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          nucon[0], nucon[1], nucon[2], nucon[3]);
                printf("flux  <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
                printf("source <%14.12f, %14.12f, %14.12f, %14.12f>\n",
                        f_sten->source[0], f_sten->source[1], 
                        f_sten->source[2], f_sten->source[3]);
                solidls(s, 2);
                clean_up(ERROR);
            }
        }
        if(!update_top_face_state(f_sten->front->sizest, s, newst))
        {
            printf("ERROR contrl_vol_update()\n");
            printf("ucon_NEW <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          nucon[0], nucon[1], nucon[2], nucon[3]);
            printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          oucon[0], oucon[1], oucon[2], oucon[3]);
            printf("flux     <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
            clean_up(ERROR);
        }

#if defined(DEBUG)
        if(f_sten->s->debug == YES)
        {
            printf("contrl_vol_update(), new state is:");
            g_verbose_print_state(newst);
            solidls(s, 2);
        }
#endif /* if defined(DEBUG) */

        debug_print("ghyp", "Leaving contrl_vol_update()\n");
}

/*** dt used in g_face_flux is only for
 *** estimate the front prop speed limit.
 ***/
LOCAL void g_face_flux(
        Face_Stencil  *f_sten,
        CSG_Face      *f,
        float         dt, 
        float         Jt, 
        float         *flux)
{
        Locstate      smid= NULL; 
        float         cfl_fac, t_area, *fdir, fnor[2], rnor[3];
        float         um = 0.0, E, pM;
        float         fflux[4] = {0.0, 0.0, 0.0, 0.0}; 
        int           wv_type, dim, i;
        float         CFL = Time_step_factor(f_sten->front);
#if defined(ROTATIONAL_SYMMETRY)
        static int    first = YES;
        static float  alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
        float         J, prod;

        smid = f->smid; 
        if(smid == NULL)
        {
            printf("ERROR g_face_flux, smid null\n");
            printf("ON face %d of solid %d\n",
		    f->faceno,f->fsolid->solidno);
            solidls(f->fsolid,2); 
            clean_up(ERROR); 
        } 
        t_area = f_sten->t_area; 
        fdir = f->face_eqn; 
        dim = f_sten->front->rect_grid->dim; 

        if(f_sten->is_grav == YES)
        {
            f_sten->scon[0] += Dens(smid); 
            f_sten->scon[1] += Dens(smid)*Vel(smid)[0]; 
            f_sten->scon[2] += Dens(smid)*Vel(smid)[1]; 
            f_sten->nsrc++; 
        }

        J = 1.0; 
#if defined(ROTATIONAL_SYMMETRY)
        if (first == YES)
        {
            first = NO;
            alpha = rotational_symmetry();
        }
        if(alpha > 0.0)
        {
            J = f->centroid[0];
            /***
            if(fabs(J) < 0.005*dh[0])
                J = (J < 0.0) ? -0.005*dh[0] : 0.005*dh[0];
            if(fabs(J) < 0.5*dh[0])
                J = (J < 0.0) ? -0.5*dh[0] : 0.5*dh[0];
            ***/ 
            if(f_sten->s->debug == YES)
            {
                printf("face[%d] Jacobi = %g\n", f->faceno, J); 
            }
            if(is_interior_face(f))
            {
                prod = scalar_product(nor[0],f->face_eqn,3);
                if(fabs(fabs(prod) - 1.0) <  0.0001*TOL)
                {
		    f_sten->P += pressure(smid);  
		    f_sten->np++;  
                }
            }
            else
            {		    
	        f_sten->P += pressure(smid);  
	        f_sten->np++;  
	    }	
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        if(is_interior_face(f))
        {
            E = Dens(smid)*(0.5*(sqr(Vel(smid)[0])+sqr(Vel(smid)[1]))+Energy(smid));
            pM = pressure(smid); 
            fflux[0] = (f->area/t_area)*(Dens(smid)*Vel(smid)[0]*fdir[0]*(J/Jt)
                                + Dens(smid)*Vel(smid)[1]*fdir[1]*(J/Jt));
            fflux[1] = (f->area/t_area)*((Dens(smid)*Vel(smid)[0]*Vel(smid)[0]+pM)*fdir[0]*(J/Jt)
                                + (Dens(smid)*Vel(smid)[0]*Vel(smid)[1])*fdir[1]*(J/Jt));
            fflux[2] = (f->area/t_area)*((Dens(smid)*Vel(smid)[0]*Vel(smid)[1])*fdir[0]*(J/Jt)
                                + (Dens(smid)*Vel(smid)[1]*Vel(smid)[1]+pM)*fdir[1]*(J/Jt));
            fflux[3] = (f->area/t_area)*((pM+E)*Vel(smid)[0]*fdir[0]*(J/Jt)
                                + (pM+E)*Vel(smid)[1]*fdir[1]*(J/Jt));

            /** compute and add the artificial viscosity **/
            g_add_contrl_vol_art_viscsity(f_sten,f,t_area,fdir,fflux,(J/Jt));

            if(f->fsolid->debug == YES)
               printf("Solid[%d] inter face[%d] flux [%10.8f, %10.8f, %10.8f, %10.8f]\n",
                f->fsolid->solidno,f->faceno,fflux[0],fflux[1],fflux[2],fflux[3]); 
        }
        else if(YES == front_face(f, &wv_type))
        {
            if(wv_type >= FIRST_PHYSICS_WAVE_TYPE)
            {
                cfl_fac = dt/dh[0]; 
                fnor[0] = fdir[0]/sqrt(sqr(fdir[0])+sqr(fdir[1]));
                fnor[1] = fdir[1]/sqrt(sqr(fdir[0])+sqr(fdir[1]));

                for ( i = 0; i < 2; i++)
                    um += fnor[i] * Vel(smid)[i];
                // if(fabs(fdir[2]/sqrt(sqr(fdir[0])+sqr(fdir[1]))*cfl_fac) < CFL)
                if(fabs(fdir[2]/sqrt(sqr(fdir[0])+sqr(fdir[1]))/cfl_fac) < CFL)
                    vol_intgel_normal(um, fdir, rnor);
                else
                {
                    rnor[0] = fdir[0];
                    rnor[1] = fdir[1];
                    rnor[2] = fdir[2];
                }
                E = Dens(smid)*(0.5*(sqr(Vel(smid)[0])+sqr(Vel(smid)[1]))+Energy(smid));
                pM = pressure(smid); 

                fflux[0] = (f->area/t_area)*(Dens(smid)*rnor[2]*(J/Jt) + 
                               Dens(smid)*Vel(smid)[0]*rnor[0]*(J/Jt) + 
                               Dens(smid)*Vel(smid)[1]*rnor[1]*(J/Jt));
                fflux[1] = (f->area/t_area)*(Dens(smid)*Vel(smid)[0]*rnor[2]*(J/Jt) + 
                               (Dens(smid)*Vel(smid)[0]*Vel(smid)[0]+pM)*rnor[0]*(J/Jt) + 
                                Dens(smid)*Vel(smid)[0]*Vel(smid)[1]*rnor[1]*(J/Jt));
                fflux[2] = (f->area/t_area)*(Dens(smid)*Vel(smid)[1]*rnor[2]*(J/Jt) + 
                               (Dens(smid)*Vel(smid)[1]*Vel(smid)[1]+pM)*rnor[1]*(J/Jt) + 
                                Dens(smid)*Vel(smid)[0]*Vel(smid)[1]*rnor[0]*(J/Jt));
                fflux[3] = (f->area/t_area)*(E*rnor[2]*(J/Jt) +
                               (E+pM)*Vel(smid)[0]*rnor[0]*(J/Jt) + 
                               (E+pM)*Vel(smid)[1]*rnor[1]*(J/Jt));

                if(f->fsolid->debug == YES)
                {
                   /***
                   printf("Solid[%d] front face[%d] um %10.8f, u[%10.8f %10.8f] pM %10.8f\n",
                     f->fsolid->solidno, f->faceno, um, Vel(smid)[0], Vel(smid)[1], pM); 
                   ***/
                   printf("Solid[%d] front face[%d] flux [%10.8f, %10.8f, %10.8f, %10.8f]\n",
                     f->fsolid->solidno,f->faceno,fflux[0],fflux[1],fflux[2],fflux[3]); 
                } 
            }
            else
            {
                printf("ERROR implement g_face_flux\n");
                printf("wave_type = %d\n", wv_type);
                clean_up(ERROR); 
            }
        }
        else
        {
            printf("ERROR in g_face_flux\n");
            printf("unknow face type\n");
            clean_up(ERROR); 
        }

        flux[0] += fflux[0];
        flux[1] += fflux[1];
        flux[2] += fflux[2];
        flux[3] += fflux[3];
}

EXPORT void contrl_vol_FD_ver2(
        int             *ic,
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave,
        Wave            *nwave,
        Front           *fr,
        Front           *newfr,
        float           dt,
        float           *dh)
{
        RECT_GRID       *gr = fr->rect_grid;
        size_t          sizest = fr->sizest;
        static Face_Stencil *f_sten = NULL;
        CSG_Face        *f;
        int             nface, i, j, idir, where[3];  
        float           *dir; 


        debug_print("hyp", "Entered contrl_vol_FD_ver2()\n"); 

        if(f_sten == NULL)
        {
            set_vol_globals(wave);  
            f_sten = alloc_face_flux_stencil(wave,fr);  
        }
 
        f_sten->s = s; 
        f_sten->ic[0] = ic[0]; f_sten->ic[1] = ic[1];
        f_sten->wave = wave; f_sten->newwave = nwave; 
        f_sten->front = fr;  f_sten->newfront = newfr; 
        f_sten->P = 0.0;
        f_sten->np = 0; 

        btm_state2d(f_sten->btm_st, s, wave, fr); 
        set_sten_vol_geom_par(f_sten, s); 

        if(s->debug)
        {
            printf("Check ic[%d %d] solid[%d]\n",
              ic[0], ic[1], s->solidno); 
            g_verbose_print_state(f_sten->btm_st);
            // verbose_print_btm_state2d(s);
            /*
            solidls(s,2);
            printf("The OLD time step interior state of"
                   " VOL %d, btm_area = %17.12f:\n",
                   s->solidno, f_sten->b_area);
            verbose_print_btm_state2d(s);
            g_verbose_print_state(f_sten->btm_st);
            printf("End of print The OLD time step interior state\n");
            */
        }

        /* First compute the interior face mid state */
        for(idir = 0; idir < 4; idir++)
        {
            dir = nor[idir]; 
            nface = count_faces_on_this_side(s,dir); 
            if(nface != 0)
            {
                collect_sten_faces_on_side(s,dir,nface,f_sten);  
                if(0 != match_sten_face_mid_st(f_sten)) 
                {
                    set_face_sten_pt_crds(f_sten,s,ic,dir); 
                    g_sten_st_for_inter_face(f_sten,s,ic,dir); 
#if defined(DEBUG)
                    if(s->debug)
                    {
                        print_face_sten(f_sten); 
                        // print_face_sten_state(f_sten); 
                    }
#endif /* if defined(DEBUG) */
                    // change dt to face->centroid[2]
                    g_face_centroid_state(f_sten, s, idir, 
                               f_sten->centroid[2], dir);
                 
                    if(f_sten->f != NULL)
                    {
                        for(j = 0; j < f_sten->alloc_nface; j++)
                            f_sten->f[j] = NULL; 
                    }
                }
            }
        }
        /* Then compute the front face mid state */ 
        g_front_face_midst(f_sten, s, ic, dt);  
 
        debug_print("hyp", "Left contrl_vol_FD()_ver2\n"); 
}

LOCAL void g_front_face_midst(
        Face_Stencil *f_sten,
        CSG_Solid    *s,
        int          *ic,
        float        dt)
{
        CSG_Face     *f;
        int          wv_type;

        debug_print("ghyp", "Entered g_front_face_midst\n");

        f = s->sfaces;
        while(f)
        {
            if(YES == front_face(f, &wv_type))
            {
                if(wv_type >= FIRST_PHYSICS_WAVE_TYPE)
                {
                    // g_h_order_tracked_front_face_midst(f_sten, f, s, ic, dt);
                    g_tracked_front_face_midst(f_sten, f, s, ic, dt);
                }
                else if(wv_type == NEUMANN_BOUNDARY)
                {
                    if(YES == oblique_face(f))
                    {
                        if(s->debug == YES)
                            printf("face in dir[%g %g %g] is oblique NEUMANN\n",
                                f->face_eqn[0],f->face_eqn[1],f->face_eqn[2]);
                        g_oblique_neumann_front_face_midst(f_sten,f,s,ic,dt);
                    }
                    else
                    {
                        if(s->debug == YES)
                            printf("face in dir[%g %g %g] is NEUMANN\n", f->face_eqn[0],
                                   f->face_eqn[1], f->face_eqn[2]);
                        g_oblique_neumann_front_face_midst(f_sten,f,s,ic,dt);
                        /* This g_neumann_front_face_flux function is not right yet */
                        // g_neumann_front_face_flux(f_sten, f, s, ic, dt, flux);
                    }
                }
                else
                {
                    printf("ERROR g_front_face_midst\n");
                    printf("Implement boundary front case wv_type = %d\n", wv_type);
                    facels(f, 2);
                    solidls(s, 2);
                    clean_up(ERROR);
                }
            }
            f = f->nextf;
        }

        debug_print("ghyp", "Left g_front_face_midst\n");
}

// This is a zero order implementation.
LOCAL void g_oblique_neumann_front_face_midst(
        Face_Stencil *f_sten,
        CSG_Face     *f,
        CSG_Solid    *s,
        int          *ic,
        float        dt)
{
        float        fnor[2], *fdir; /* fnor is fdir's projection in x-y plane */
        Locstate     ans = NULL;
        int          i, j;
        size_t       sizest = f_sten->front->sizest;

        f_sten->maxf = f;
        for(i = 0; i < 3; i++)
            f_sten->centroid[i] = f->centroid[i];
        g_neumann_front_face_state(f_sten);

        ans = stintfc_st_pool_to_smid(sizest);
        set_type_of_state(ans,GAS_STATE);

        fdir = f->face_eqn;
        fnor[0] = fdir[0]/sqrt(sqr(fdir[0])+sqr(fdir[1]));
        fnor[1] = fdir[1]/sqrt(sqr(fdir[0])+sqr(fdir[1]));

        riemann_solution(0.0,fnor,f_sten->btm_st,f_sten->st[1],ans,EGAS_STATE);
        f->smid = ans; 
}

LOCAL void g_h_order_tracked_front_face_midst(
        Face_Stencil *f_sten,
        CSG_Face     *f,
        CSG_Solid    *s,
        int          *ic,
        float        dt)
{
        Solid_nghbr     *nghbrs; 
        int             num;
        
        num = neighbors_for_solid(f->fsolid,
                wave_tri_soln(f_sten->wave)->tri_grid,
                &nghbrs);
        if(num != 0)
            free(nghbrs); 
}

LOCAL void g_tracked_front_face_midst(
        Face_Stencil *f_sten,
        CSG_Face     *f,
        CSG_Solid    *s,
        int          *ic,
        float        dt)
{
        CSG_Solid    *outs = NULL;
        CSG_Face     *fother;
        float        fnor[2], *fdir; /* fnor is fdir's projection in x-y plane */
        static Locstate     outbtm_st = NULL, Tist, Tost;
        Locstate     ansi;
        int          i, j;
        size_t       sizest = f_sten->front->sizest;
        float        vtani[MAXD], vtano[MAXD];     /* velocities */
        float        ui,uo;
        float        pjump = 0.0;
        float        pml, pmr, pl, pr;    /* pressures */
        float        mr,ml;            /* miscellaneous variables */
        RIEMANN_SOLVER_WAVE_TYPE  l_wave,r_wave;
        float        uml,umr;    /* normal velocities */
        float        ri,ro;          /* densities */
        float        ux, uy;
        float        rarea, rnor[3], t_area;

        ansi = stintfc_st_pool_to_smid(sizest);
        if(outbtm_st == NULL)
        {
            alloc_state(f_sten->front->interf,&outbtm_st,
                     max(sizeof(VGas),sizest));
            alloc_state(f_sten->front->interf,&Tost,sizest);
            alloc_state(f_sten->front->interf,&Tist,sizest);
        }

        f_sten->maxf = f;
        for(i = 0; i < 3; i++)
            f_sten->centroid[i] = f->centroid[i];

        //fother = face_on_other_side(f, s, ic, f_sten->wave);
        fother = find_match_face_in_region4(f, ic, 
                   wave_tri_soln(f_sten->wave)->tri_grid);
        if(fother == NULL)
        {
            printf("ERROR g_tracked_front_face_midst\n");
            printf("face does not have matched face from other side\n");
            facels(f, 2);
            solidls(s, 2);
            clean_up(ERROR);
        }
        if(fother->fsolid->comp == f->fsolid->comp) 
        {
            printf("ERROR  g_tracked_front_face_midst()\n");
            printf("face %d of solid %d eqn<%f, %f, %f>\n",
                   f->faceno, f->fsolid->solidno,
               f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
            printf("match face %d of solid %d eqn<%f, %f, %f>\n",
                  fother->faceno, fother->fsolid->solidno,
                 fother->face_eqn[0], fother->face_eqn[1],
                 fother->face_eqn[2]);
            printf("f, match_f Solid comp<%d, %d>\n",
                f->fsolid->comp , fother->fsolid->comp);
            printf("Print f->fsolid\n");
            solidls(f->fsolid,2);
            printf("Print match_f->fsolid\n");
            solidls(fother->fsolid,2);
            clean_up(ERROR);
        }
        btm_state2d(outbtm_st, fother->fsolid, f_sten->wave, f_sten->front);

        fdir = f->face_eqn;
        fnor[0] = fdir[0]/sqrt(sqr(fdir[0])+sqr(fdir[1]));
        fnor[1] = fdir[1]/sqrt(sqr(fdir[0])+sqr(fdir[1]));

        set_state_for_find_mid_state(Tist, f_sten->btm_st);
        set_state_for_find_mid_state(Tost, outbtm_st);

        ui = uo = 0.0;
        for ( i = 0; i < 2; i++)
        {
             ui += fnor[i] * Vel(Tist)[i];
             uo += fnor[i] * Vel(Tost)[i];
         }

         for ( i = 0; i < dim; i++)
         {
              vtani[i] = Vel(Tist)[i] - fnor[i] * ui;
              vtano[i] = Vel(Tost)[i] - fnor[i] * uo;
         }

         Vel(Tist)[0] = ui;     Vel(Tost)[0] = uo;
         for (i = 1; i < 2; i++)
              Vel(Tist)[i] = Vel(Tost)[i] = 0.0;

#if defined(DEBUG)
        if(debugging("bad_state"))
        {
            if(is_bad_state(Tist,YES,"g_tracked_front_face_midst"))
            {
                printf("ERROR in g_tracked_front_face_flux(),\n");
                printf("Tist is bad\n");
                clean_up(ERROR); 
            }
            if(is_bad_state(Tost,YES,"g_tracked_front_face_midst"))
            {
                printf("ERROR in g_tracked_front_face_flux(),\n");
                printf("Tost is bad\n");
                printf("Outter Volume state:"); 
                solidls(fother->fsolid, 2); 
                verbose_print_btm_state2d(fother->fsolid);
                clean_up(ERROR); 
            }
        }
#endif /* if defined(DEBUG) */

        if (! find_mid_state(Tist,Tost,pjump,&pml,
                &pmr,&uml,&umr,&ml,&mr, &l_wave,&r_wave))
        {
             printf("ERROR in g_tracked_front_face_flux(),\n");
             printf("find_mid_state()did not converge");
             printf("for left/right RP\n");
        }
        /* contact only */
        midstate(Tist,ansi,ml,uml,pml,EGAS_STATE,l_wave,LEFT_FAMILY);

        ri = Dens(ansi);
        for(i = dim -1; i >= 0; i--)
            Vel(ansi)[i] = vtani[i] + fnor[i]*Vel(ansi)[0];
        f->smid = ansi; 

        if(s->debug == YES)
        {
            printf("\nFront face[%d] eqn[%g %g %g] um = %g\n", 
                  f->faceno, fdir[0], fdir[1], fdir[2], uml); 
            printf("mid state rho, v, inter_e, p"
                   " [%10.8f, %10.8f, %10.8f, %10.8f, %10.8f]\n",
             Dens(ansi), Vel(ansi)[0], Vel(ansi)[1], Energy(ansi), pressure(ansi));
            printf("Volume state:"); 
            g_print_state(f_sten->btm_st);
            printf("Outter Volume state:"); 
            g_print_state(outbtm_st); 
            // verbose_print_btm_state2d(fother->fsolid);
            printf("End print front face smid\n"); 
        }
}

/* return the # of faces do not have mid state computed */
LOCAL int  match_sten_face_mid_st(
        Face_Stencil *f_sten)
{
        CSG_Face    *f, *match_f; 
        int         nface, i; 
        int         dim, ic[MAXD]; 
        Wave        *wave = f_sten->wave; 
        TRI_GRID    *ntg = wave_tri_soln(wave)->tri_grid; 
        int         not_assigned = 0; 

#if defined(DEBUG)
        if(f_sten->s->debug == YES)
            return 1; 
#endif /* if defined(DEBUG) */
 
        nface = f_sten->nface; 
        dim = f_sten->wave->rect_grid->dim;
        for(i = 0; i < dim; i++)
            ic[i] = f_sten->ic[i]; 

        for(i = 0; i < nface; i++)
        {
            if(YES == is_interior_face(f_sten->f[i]) && 
               f_sten->f[i]->smid == NULL)
            {
                match_f = find_match_face_in_region4(f_sten->f[i],ic,ntg); 
                if(match_f == NULL)
                    not_assigned++; 
                else
                {
                    if(match_f->fsolid->comp != f_sten->f[i]->fsolid->comp)
                    {
                        printf("ERROR  match_sten_face_mid_st()\n");
                        printf("face %d of solid %d eqn<%f, %f, %f>\n",
                            f_sten->f[i]->faceno, f_sten->f[i]->fsolid->solidno,
                            f_sten->f[i]->face_eqn[0], f_sten->f[i]->face_eqn[1],
                            f_sten->f[i]->face_eqn[2]);
                        printf("match face %d of solid %d eqn<%f, %f, %f>\n",
                         match_f->faceno, match_f->fsolid->solidno,
                         match_f->face_eqn[0], match_f->face_eqn[1],
                         match_f->face_eqn[2]);
                        printf("f, match_f Solid comp<%d, %d>\n",
                                f_sten->f[i]->fsolid->comp , match_f->fsolid->comp);
                        printf("Print f->fsolid\n");
                        solidls(f_sten->f[i]->fsolid,2);
                        printf("Print match_f->fsolid\n");
                        solidls(match_f->fsolid,2);
                        clean_up(ERROR);
                    }
                    if(match_f->smid != NULL)
                    {
                        f_sten->f[i]->smid = match_f->smid;  
                        /* copy the artificial viscosity terms */
                        if(match_f->art_visc_st_x != NULL)
                            f_sten->f[i]->art_visc_st_x = match_f->art_visc_st_x;
                        if(match_f->art_visc_st_y != NULL)
                            f_sten->f[i]->art_visc_st_y = match_f->art_visc_st_y;

                        if(f_sten->s->debug == YES)
                        {
                            printf("match inter face[%d] eqn[%g %g %g]\n",
                              f_sten->f[i]->faceno, f_sten->f[i]->face_eqn[0], 
                              f_sten->f[i]->face_eqn[1],f_sten->f[i]->face_eqn[2]);
                            printf("mid state rho, v, inter_e, p"
                                   " [%g, %g, %g, %g, %g]\n",
                             Dens(f_sten->f[i]->smid), Vel(f_sten->f[i]->smid)[0], 
                             Vel(f_sten->f[i]->smid)[1], Energy(f_sten->f[i]->smid), 
                             pressure(f_sten->f[i]->smid));                            
                        }
                    }
                    else
                        not_assigned++; 
                }
            }    
        }

        return not_assigned; 
}

LOCAL void g_face_centroid_state(
        Face_Stencil *f_sten,
        CSG_Solid    *s,
        int          idir, 
        float        dt, 
        float        *fdir)
{
        Wave           *wave = f_sten->wave; 
        static Vec_Gas *vst = NULL;
        static Vec_Src *src;
        static Vec_Gas *Tvst[2] = {NULL, NULL}; 
        static float   **Q = NULL;
        static int     *pseudo_iperm; 

        int            i, j, side, swp_num; 
        float          **coords; 
        float          *rho, *en_den;
        float          *m[MAXD];
        Locstate       st, *state, *tstate0, *tstate1; 
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        float           *vacuum_dens;
        float           *min_pressure;
        float           *min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
#if defined(ROTATIONAL_SYMMETRY)
        static float    alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
        int             idirs[MAXD];
        RECT_GRID       *gr = f_sten->front->rect_grid;
 
        if (vst == NULL)
        {
            /* Assumes stencil size never changes */

            // g_unsplit_muscl_alloc_phys_vecs;
            alloc_unsplit_phys_vecs(wave,vsten_rad*2);
            vst = g_wave_vgas(wave);
            src = g_wave_vsrc(wave);
            Tvst[0] = g_wave_Tvgas(wave)[0];
            Tvst[1] = g_wave_Tvgas(wave)[1];

            g_wave_vgas(wave) = NULL;
            g_wave_vsrc(wave) = NULL;
            g_wave_Tvgas(wave)[0] = NULL;
            g_wave_Tvgas(wave)[1] = NULL;
            matrix(&Q,3,3,FLOAT);
            vst->Q = (const float* const*)Q;
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
            vector(&pseudo_iperm, 2, sizeof(int)); 
            for(i = 0; i < dim; i++)
                pseudo_iperm[i] = i; 
        }

        clear_Vec_Gas_set_flags(vst);
        clear_Vec_Gas_set_flags(Tvst[0]);
        clear_Vec_Gas_set_flags(Tvst[1]);
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                Q[i][j] = 0.0;
        if(idir == 0 || idir == 2)
            swp_num = 0;
        else
            swp_num = 1; 
        idirs[0] = swp_num; 
        idirs[1] = (swp_num+1)%2;


        for (i = 0; i < dim; ++i)
            Q[i][idirs[i]] = 1.0;
        for (; i < 3; ++i)
            Q[i][i] = 1.0;

        state = vst->state; 
        coords = vst->coords; 
        switch(idir)
        {
        case 0:
        case 1:
            for(i = 0; i < 4; i++)
            {
                state[i] = f_sten->st[i-1]; 
                coords[i] = f_sten->crds[i]; 
            }
        break; 
        case 2:
        case 3:
            for(i = 0; i < 4; i++)
            {
                state[i] = f_sten->st[2-i]; 
                coords[i] = f_sten->crds[3-i]; 
            }
        break; 
        }

        tstate1 = Tvst[1]->state; 
        tstate0 = Tvst[0]->state; 
        switch(idir)
        {
        case 0:
            tstate1[1] = f_sten->tan_st[-1]; 
            tstate1[2] = f_sten->outtan_st[-1]; 
            tstate0[1] = f_sten->tan_st[1]; 
            tstate0[2] = f_sten->outtan_st[1]; 
        break; 
        case 1:
            tstate1[1] = f_sten->tan_st[1]; 
            tstate1[2] = f_sten->outtan_st[1]; 
            tstate0[1] = f_sten->tan_st[-1]; 
            tstate0[2] = f_sten->outtan_st[-1]; 
        break;  
        case 2:
            tstate1[1] = f_sten->outtan_st[1]; 
            tstate1[2] = f_sten->tan_st[1]; 
            tstate0[1] = f_sten->outtan_st[-1]; 
            tstate0[2] = f_sten->tan_st[-1]; 
        break; 
        case 3:
            tstate1[1] = f_sten->outtan_st[-1]; 
            tstate1[2] = f_sten->tan_st[-1]; 
            tstate0[1] = f_sten->outtan_st[1]; 
            tstate0[2] = f_sten->tan_st[1]; 
        break; 
        }

        rho = vst->rho;
        state = vst->state;
        en_den = vst->en_den;
        for (i = 0; i < dim; ++i)
            m[i] = vst->m[i];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        vacuum_dens = vst->vacuum_dens;
        min_pressure = vst->min_pressure;
        min_energy = vst->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        for(i = 0; i < 4; i++)
        {
            rho[i] = Dens(state[i]);
            en_den[i] = Energy(state[i]);
            for (j = 0; j < dim; ++j)
                m[j][i] = Mom(state[i])[j];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            vacuum_dens[i] = Vacuum_dens(state[i]);
            min_pressure[i] = Min_pressure(state[i]);
            min_energy[i] = Min_energy(state[i]);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        }
        Vec_Gas_field_set(vst,state) = YES;
        Vec_Gas_field_set(vst,coords) = YES;
        Vec_Gas_field_set(vst,rho) = YES;
        Vec_Gas_field_set(vst,en_den) = YES;
        Vec_Gas_field_set(vst,m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        Vec_Gas_field_set(vst,vacuum_dens) = YES;
        Vec_Gas_field_set(vst,min_pressure) = YES;
        Vec_Gas_field_set(vst,min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        set_params_jumps(vst,0,4);

#if defined(ROTATIONAL_SYMMETRY)
        /* If using cylindrical coordinates and this is the
         *  r-sweep include radius information for source
         *  computation.
         */

        if (alpha > 0.0 && idirs[0] == 0)
        {
            float *radii = src->radii;

            src->rmin = fabs(pos_radius(0.0,gr));
            for (i = 0; i < 4; ++i)
                radii[i] = pos_radius(coords[i][0],gr);
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        for(side = 0; side <= 1; side++)
        {
            rho = Tvst[side]->rho;
            state = Tvst[side]->state;
            en_den = Tvst[side]->en_den;
            for (i = 0; i < dim; ++i)
                m[i] = Tvst[side]->m[i];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            vacuum_dens = Tvst[side]->vacuum_dens;
            min_pressure = Tvst[side]->min_pressure;
            min_energy = Tvst[side]->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            for(i = 1; i <= 2; i++)
            {
                rho[i] = Dens(state[i]);
                en_den[i] = Energy(state[i]);
                for (j = 0; j < dim; ++j)
                    m[j][i] = Mom(state[i])[j];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
                vacuum_dens[i] = Vacuum_dens(state[i]);
                min_pressure[i] = Min_pressure(state[i]);
                min_energy[i] = Min_energy(state[i]);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            }
            Vec_Gas_field_set(Tvst[side],state) = YES;
            // Vec_Gas_field_set(Tvst[side],coords) = YES;
            Vec_Gas_field_set(Tvst[side],rho) = YES;
            Vec_Gas_field_set(Tvst[side],en_den) = YES;
            Vec_Gas_field_set(Tvst[side],m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            Vec_Gas_field_set(Tvst[side],vacuum_dens) = YES;
            Vec_Gas_field_set(Tvst[side],min_pressure) = YES;
            Vec_Gas_field_set(Tvst[side],min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            // set_params_jumps(Tvst[side],0,npts);
        }

        if(f_sten->s->debug == YES)
            add_to_debug("gvol_compute_Jacobi");

        g_consv_muscl_soln(f_sten,fdir,swp_num,pseudo_iperm,
             f_sten->wave,f_sten->front,f_sten->newfront,
             0,4,vst,Tvst,src,dt,dh,0);

        if(f_sten->s->debug == YES)
            remove_from_debug("gvol_compute_Jacobi");
}

LOCAL void  gvol_compute_Jacobi(
        Face_Stencil    *f_sten,
        int             swp_num,
        int             *iperm,
        int             start,
        int             end,
        Vec_Muscl       *vmuscl,
        Wave            *wv,
        Vec_Gas         *vst,
        int             offset)
{
        int             j;
        float           *J = vmuscl->J + offset;
        float           *J_half = vmuscl->J_half + offset;
        float           **coords = vst->coords + offset;
        RECT_GRID       *r_grid;
        float           dr, J0;
        int             sten_rad;
        int             vsize;
                                                                                                           
        static bool     first = YES;
#if defined(ROTATIONAL_SYMMETRY)
        static  float   alpha;
#endif // defined(ROTATIONAL_SYMMETRY)
                                                                                                           
        if (first == YES)
        {
            first = NO;
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif // defined(ROTATIONAL_SYMMETRY)
        }

        sten_rad = vmuscl->sten_rad;
        vsize = end - start;
        r_grid = wv->rect_grid;
                                                                                                           
        dr = r_grid->h[0];
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            J0 = f_sten->centroid[0]+f_sten->maxf->face_eqn[0]*dr*(-0.5);
            for (j = start; j < end; ++j)
            {
                if(iperm[swp_num] == 0)
                    J[j] = J0+(j-1)*dr; 
                else
                    J[j] = J0; 
                if(fabs(J[j]) < 0.5*dr)
                    J[j] = (J[j] < 0.0) ? -0.5*dr : 0.5*dr;
            }
            for (j = start; j < end; ++j)
            {
                if(iperm[swp_num] == 0)
                {
                    J_half[j] = J[j] - dr/2.0;
                    if(fabs(J_half[j]) < 0.5*dr)
                        J_half[j] = (J[j] < 0.0) ? -0.5*dr : 0.5*dr;
                }
                else
                    J_half[j] = J[j]; 
            }
        }
        else
#endif // #if defined(ROTATIONAL_SYMMETRY)
        {
            for (j = start; j < end; ++j)
                J[j] = J_half[j] = 1.0;
        }

        if(debugging("gvol_compute_Jacobi"))
        {
            printf("gvol_compute_Jacobi: idir[%d]\n", iperm[swp_num]);
            for (j = start; j < end; ++j)
                printf("Jacobi[%d] = %g\n", j, J[j]);
            for (j = start; j < end; ++j)
                printf("J_half_Jacobi[%d] = %g\n", j, J_half[j]);
        }
}
                                                                                                     
LOCAL void g_consv_muscl_soln(
        Face_Stencil    *f_sten,
        float           *fdir, 
        int             swp_num,
        int             *iperm,
        Wave            *wave,
        Front           *fr,
        Front           *newfr,
        int             offset,
        int             vsize,
        Vec_Gas         *vst,
        Vec_Gas         **Tvst,
        Vec_Src         *src,
        float           dt,
        float           *dn,
        int             pbuf)
{
        Vec_Muscl       *vmuscl;
        float           dtdni;
        float           t_area; 
        int             idirs[MAXD];
        int             start, end;
        int             i, j, k, kmax;
        Muscl_Opts      *MOpts;
        float           radl;
        RECT_GRID       *r_grid = wave->rect_grid;
        float           *pM, E;
        Locstate        smid = NULL;
        Locstate        art_visc_st = NULL; /*artificial viscosity state*/
        size_t          sizest;
        bool            is_src; 

        MOpts = muscl_options();
        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];
        dtdni = dt/dn[idirs[0]];

        /***
        if(f_sten->s->debug == YES)
        {
            add_to_debug("N_half_step");  
            add_to_debug("load_Vgas");  
            printf("Debugging: g_consv_muscl_flux\n");  
            printf("In dir[%g, %g], swp_num = %d side area = %g,"
                   " top_area = %g area ratio = %g\n",
                  fdir[0], fdir[1], swp_num, f_sten->area, f_sten->t_area, 
                  f_sten->area/f_sten->t_area);
            printf("Idirs[%d, %d]\n", idirs[0], idirs[1]); 
            for(j = 0; j < 4; j++)
                print_vector("Coords",dim, vst->coords[j], "%g ");  
            for(j = 0; j < 4; j++)
                g_verbose_print_state(vst->state[j]);  
        }
        ***/

        vmuscl = g_init_unsplit_muscl_state_data(MOpts, swp_num, iperm,
                    fr, wave, NULL, NULL, vst, Tvst, src, offset,
                     vsize, dim, dt, dn[idirs[0]]);
        gvol_compute_Jacobi(f_sten,swp_num,iperm,0,vsize,vmuscl,
                    wave,vst,offset); 
        is_src = g_unsplit_muscl_load_source_vectors(swp_num,iperm,NULL,
                 vmuscl->gravity,vst,src,vmuscl->J,offset,vsize,fr->rect_grid);

        /***
        vmuscl = g_load_unsplit_muscl_state_data(MOpts,swp_num,iperm,fr,wave,
                      NULL,NULL,vst,Tvst,src,offset,vsize,dim,dt,dn[idirs[0]]);
        ***/
 
        /***
        if(f_sten->s->debug == YES)
        {
            printf("The internal energy :\n"); 
            for(j = 0; j < 4; j++)
                printf("e[%d]= %g    ", j, vst->e[j]);
            printf("\n");  
        }
        ***/

        /* compute the eigenvalues, eigenvectors and max wavespeed*/
        g_compute_2dunsplit_eigens(0,vsize,swp_num,iperm,dim,vmuscl);

        /* compute the linear reconstruction of the state variables */
        g_2dunsplit_bct_linear_reconstructor(1,3,vmuscl);

#if defined(DEBUG)
        if(f_sten->s->debug == YES)
        {
            // add_to_debug("N_half_step");  
            // add_to_debug("T_half_step");  
            // set_unsplit_muscl_debug_flag(YES); 
        }
#endif /* if defined(DEBUG) */

        /* USing face centroid here, so dt multiples 2.0 here.
         * Inside N_half_step and T_half_step, there is a factor 0.5
         */
        /* Half time step expansion in normal dir. */
        g_2dunsplit_N_half_step(2,3,2.0*dt,dn,swp_num,iperm,dim,vmuscl);

        /* Add the constribution from another direction */
        g_2dunsplit_T_half_step(1,3,2.0*dt,dn,swp_num,iperm,dim,vmuscl);

#if defined(DEBUG)
        if(f_sten->s->debug == YES)
        {
            // remove_from_debug("N_half_step");  
            // remove_from_debug("T_half_step");  
            // set_unsplit_muscl_debug_flag(NO); 
        }
#endif /* if defined(DEBUG) */

        g_unsplit_muscl_exact_rsolver(2,3,swp_num,iperm,vmuscl->uL,
             &vmuscl->VLst,vmuscl->uR,&vmuscl->VRst,vmuscl->uM,
             &vmuscl->VMst,&vmuscl->Flux,vmuscl);

        /* Do not need to set conservative source terms */
        // g_unsplit_cons_src(2,2,swp_num,iperm,NULL,vmuscl);

        sizest = vmuscl->sizest;
        smid = stintfc_st_pool_to_smid(sizest);
        if (vmuscl->avisc)
            art_visc_st = stintfc_st_pool_to_smid(sizest);

        Dens(smid) = vmuscl->VMst.rho[2];
        Energy(smid) = vmuscl->VMst.e[2];
        for (i = 0; i < dim; ++i)
            Vel(smid)[i] = vmuscl->VMst.v[i][2];
        Set_params(smid,vmuscl->vst->state[1]);
        set_type_of_state(smid,EGAS_STATE);

        if(f_sten->s->debug == YES)
        {
            t_area = f_sten->t_area; 
            pM = vmuscl->VMst.p; 
            E = Dens(smid)*( 0.5*(sqr(Vel(smid)[0])+sqr(Vel(smid)[1]))
               +Energy(smid));
            printf("Solid[%d] faces in dir[%g, %g, %g]\n", 
                 f_sten->s->solidno, fdir[0], fdir[1], fdir[2]); 
            /*
            for(i = 0; i < f_sten->nface; i++)
            {
                if(f_sten->f[i]->smid == NULL)
                    printf("Face[%d] ", f_sten->f[i]->faceno);
            } 
            */
            printf("mid state rho, v, inter_e, p"
                   " [%g, %g, %g, %g, %g]\n",
              Dens(smid),Vel(smid)[0],Vel(smid)[1],Energy(smid),pM[2]); 
        }

        if (vmuscl->avisc)
            g_comp_contrl_vol_art_visc2(vst,vmuscl,art_visc_st);

        for(i = 0; i < f_sten->nface; i++)
        {
            if(f_sten->f[i]->smid == NULL)
            {
                f_sten->f[i]->smid = smid; 
                if (vmuscl->avisc)
                { 
                    if (iperm[swp_num] == 0)
                        f_sten->f[i]->art_visc_st_x = art_visc_st;
                    else
                        f_sten->f[i]->art_visc_st_y = art_visc_st;
                }
            }
        }
}

LOCAL Locstate stintfc_st_pool_to_smid(
        size_t      sizest)
{
        int  tmp_pool_used, tmp_st_used;
        int  tmp_alloc_pool, tmp_alloc_st;
        byte  **tmp_stintfc_st_pool = NULL;
        byte  *stintfc_st_storage = NULL;
        int  i, j;

        if(st_used == alloc_st_n)
        {
            st_pool_used++; 
            if(st_pool_used == alloc_st_pool)
            {
                tmp_alloc_pool = alloc_st_pool + 4; 
                vector(&tmp_stintfc_st_pool,tmp_alloc_pool,sizeof(byte*));
                for(i = 0; i < st_pool_used; i++)
                    tmp_stintfc_st_pool[i] = stintfc_st_pool[i]; 
                free(stintfc_st_pool);   
                stintfc_st_pool = tmp_stintfc_st_pool;   
                alloc_st_pool = tmp_alloc_pool; 
            }
            vector(&stintfc_st_storage,alloc_st_n,sizest);  
            stintfc_st_pool[st_pool_used] = stintfc_st_storage; 
            st_used = 0; 
        }

        stintfc_st_storage = stintfc_st_pool[st_pool_used];
        tmp_st_used = st_used; 
        st_used++;  
        return &(stintfc_st_storage[sizest*tmp_st_used]); 
}


EXPORT void contrl_vol_FD(
        int             *ic,
        CSG_BLK_CRX     *blk,
        CSG_Solid       *s,
        Wave            *wave,
        Wave            *nwave,
        Front           *fr,
        Front           *newfr,
        float           dt,
        float           *dh)
{
        RECT_GRID       *gr = fr->rect_grid;
        size_t          sizest = fr->sizest;
        static Locstate newst = NULL;
        static Face_Stencil *f_sten = NULL;
        CSG_Face        *f;
        int             nface, i, j, where[3];  
        float           *dir; 
        float           flux[4] = {0.0, 0.0, 0.0, 0.0}; 
        float           nucon[4] = {0.0, 0.0, 0.0, 0.0};
        float           oucon[4] = {0.0, 0.0, 0.0, 0.0};
        float           Jt, Jb; 

#if defined(ROTATIONAL_SYMMETRY)
        static float    alpha; 
#endif /* if defined(ROTATIONAL_SYMMETRY) */         

        debug_print("hyp", "Entered contrl_vol_FD()\n"); 

        if(f_sten == NULL)
        {
            set_vol_globals(wave);  
            f_sten = alloc_face_flux_stencil(wave,fr);  
            alloc_state(fr->interf, &newst, sizest);            
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }
 
        f_sten->s = s; 
        f_sten->ic[0] = ic[0]; f_sten->ic[1] = ic[1];
        f_sten->wave = wave; f_sten->newwave = nwave; 
        f_sten->front = fr;  f_sten->newfront = newfr; 
        f_sten->P = 0.0;
        f_sten->np = 0; 

        btm_state2d(f_sten->btm_st, s, wave, fr); 
        set_sten_vol_geom_par(f_sten, s); 

        Jt = Jb = 1.0; 
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            Jt = f_sten->t_centroid[0]; 
            if(fabs(Jt) < 1.0e-04)
                Jt = (Jt < 0.0)? -0.05*dh[0] : 0.05*dh[0];  
            Jb = f_sten->b_centroid[0]; 
        }
#endif /* if defined(ROTATIONAL_SYMMETRY) */

        oucon[0] = f_sten->b_area/f_sten->t_area*Dens(f_sten->btm_st)*Jb; 
        oucon[1] = f_sten->b_area/f_sten->t_area*Mom(f_sten->btm_st)[0]*Jb; 
        oucon[2] = f_sten->b_area/f_sten->t_area*Mom(f_sten->btm_st)[1]*Jb; 
        oucon[3] = f_sten->b_area/f_sten->t_area*Energy(f_sten->btm_st)*Jb; 

        if(s->debug)
        {
            printf("Check ic[%d %d]\n", ic[0], ic[1]); 
            /*
            solidls(s,2);
            printf("The OLD time step interior state of"
                   " VOL %d, btm_area = %17.12f:\n",
                   s->solidno, f_sten->b_area);
            verbose_print_btm_state2d(s);
            g_verbose_print_state(f_sten->btm_st);
            printf("Ratioed value Jb = %g Jt = %g:\n", Jb, Jt); 
            printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                      oucon[0], oucon[1], oucon[2], oucon[3]);
            printf("End of print The OLD time step interior state\n");
            */
        }

        /* First compute the interior face flux */
        for(i = 0; i < 4; i++)
        {
            dir = nor[i]; 
            nface = count_faces_on_this_side(s,dir); 
            if(nface != 0)
            {
                collect_sten_faces_on_side(s,dir,nface,f_sten);  
                set_face_sten_pt_crds(f_sten,s,ic,dir); 
                g_sten_st_for_inter_face(f_sten,s,ic,dir); 
                // change dt to face->centroid[2]
                //g_face_centroid_flux(f_sten, s, i, dt, dir, flux);
           
                if(s->debug == YES)
                {
                   printf("In dir[%g, %g, %g]\n", dir[0], dir[1], dir[2]);
                }
                g_face_centroid_flux(f_sten,s,i,f_sten->centroid[2],dir,flux);
                 
                if(f_sten->f != NULL)
                {
                    for(j = 0; j < f_sten->alloc_nface; j++)
                        f_sten->f[j] = NULL; 
                }
            }
        }
 
        /* Then compute the front face flux */
        g_front_face_flux(f_sten, s, ic, dt, flux); 

        for(i = 0; i < 4; i++) 
            nucon[i] = (oucon[i] - flux[i])/Jt;  
        if(s->debug == YES)
            printf("old dens %17.15f, dens_flux %17.15f, Jt %17.15f\n",
                 oucon[0], flux[0], Jt); 

#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            if(f_sten->P/f_sten->np != 0)
                nucon[1] += f_sten->P/f_sten->np*s->vol/(Jt*f_sten->t_area);  
            else
            {
                printf("ERROR contrl_vol_FD(), side Pressure is 0:");
                clean_up(ERROR);  
            }
        }
#endif /* if defined(ROTATIONAL_SYMMETRY) */

        Dens(newst) = nucon[0];
        Mom(newst)[0] = nucon[1];
        Mom(newst)[1] = nucon[2];
        Energy(newst) = nucon[3];
        Set_params(newst, f_sten->btm_st);
        set_type_of_state(newst,GAS_STATE);

        if(debugging("bad_state"))
        {
            if(is_bad_state(newst,YES,"contrl_vol_FD")) 
            {
                printf("ERROR contrl_vol_FD(), new state is bad:");
                g_verbose_print_state(newst);
                printf("Old state is:");
                g_verbose_print_state(f_sten->btm_st); 

                printf("Ratioed value:\n"); 
                printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          oucon[0], oucon[1], oucon[2], oucon[3]);
                printf("flux  <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
                solidls(s, 2);
                clean_up(ERROR);   
            }
            /*
            if(pressure(newst) < 0.001)
            {
                printf("ERROR contrl_vol_FD(), new state is bad:");
                g_verbose_print_state(newst);
                printf("Old state is:");
                g_verbose_print_state(f_sten->btm_st); 

                printf("Ratioed value:\n"); 
                printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          oucon[0], oucon[1], oucon[2], oucon[3]);
                printf("flux  <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
                solidls(s, 2);
                clean_up(ERROR);   
            }
            */
        }
        
        if(nucon[0] < 0.0)
        {
            printf("ERROR contrl_vol_FD(), new dens < 0.0\n");
            printf("ucon_NEW <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          nucon[0], nucon[1], nucon[2], nucon[3]);
            printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          oucon[0], oucon[1], oucon[2], oucon[3]);
            printf("flux     <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
            clean_up(ERROR);
        }

        if(s->debug == YES)
        {
            printf("Solid %d new state average:", s->solidno);
            g_print_state(newst); 
        }
        if(!update_top_face_state(f_sten->front->sizest, s, newst))
        {
            printf("ERROR contrl_vol_FD()\n");
            printf("ucon_NEW <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          nucon[0], nucon[1], nucon[2], nucon[3]);
            printf("ucon_OLD <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          oucon[0], oucon[1], oucon[2], oucon[3]);
            printf("flux     <%17.14f, %17.14f, %17.14f, %17.14f>\n",
                          flux[0], flux[1], flux[2], flux[3]);
            clean_up(ERROR);
        }

        debug_print("hyp", "Left contrl_vol_FD()\n"); 
}

EXPORT int update_top_face_state(
        size_t       sizest,
        CSG_Solid    *s,
        Locstate     newst)
{
        CSG_Face     *f; 
        int          where[3];  
        

        f = s->sfaces;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 1)
                {
                    assign(f->i_stat, newst, sizest); 
                    /*
                    Dens(f->i_stat) = nucon[0];
                    Mom(f->i_stat)[0] = nucon[1];
                    Mom(f->i_stat)[1] = nucon[2];
                    Energy(f->i_stat) = nucon[3];
                    Set_params(f->i_stat, f_sten->btm_st);
                    set_type_of_state(f->i_stat,GAS_STATE);
                    */
                }
            }
            f = f->nextf;
        }

        return YES; 
}

LOCAL void set_sten_vol_geom_par(
        Face_Stencil *f_sten,
        CSG_Solid    *s)
{
        CSG_Face     *f; 
        float        t_area, b_area;
        int          i, where[3], b, t;

        b = t = 0; 
        t_area = b_area = 0.0;
        for(i = 0; i < 3; i++)
           f_sten->t_centroid[i] = f_sten->b_centroid[i] = 0.0;

        f = s->sfaces;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 0)
                {
                    for(i = 0; i < 3; i++)
                        f_sten->b_centroid[i] += f->centroid[i]*f->area;
                    b_area += f->area;
                    b++; 
                }
                else if(where[0] == 1)
                {
                    for(i = 0; i < 3; i++)
                        f_sten->t_centroid[i] += f->centroid[i]*f->area;
                    t_area += f->area;
                    t++; 
                }
            }
            f = f->nextf;
        }

        f_sten->b_area = b_area; 
        if(b != 0)
        {
            for(i = 0; i < 3; i++)
                f_sten->b_centroid[i] /= b_area;
        }

        f_sten->t_area = t_area; 
        if(t != 0)
        {
            for(i = 0; i < 3; i++)
                f_sten->t_centroid[i] /= t_area;
        }
}

LOCAL void g_front_face_flux(
        Face_Stencil *f_sten,
        CSG_Solid    *s,
        int          *ic, 
        float        dt, 
        float        *flux)
{
        CSG_Face     *f;
        int          wv_type; 

        debug_print("ghyp", "Entered g_front_face_flux\n"); 
        DEBUG_ENTER(g_front_face_flux) 

        
        f = s->sfaces; 
        while(f)
        {
            if(YES == front_face(f, &wv_type))
            {
                if(wv_type >= FIRST_PHYSICS_WAVE_TYPE)
                {
                    g_tracked_front_face_flux(f_sten, f, s, ic, dt, flux); 
                }
                else if(wv_type == NEUMANN_BOUNDARY)
                {
                    if(YES == oblique_face(f))
                    {
                        if(s->debug == YES)
                            printf("face in dir[%g %g %g] is oblique NEUMANN\n",
                              f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                        g_oblique_neumann_front_face_flux(f_sten,f,s,ic,dt,flux); 
                    }
                    else
                    { 
                        if(s->debug == YES)
                            printf("face in dir[%g %g %g] is NEUMANN\n", f->face_eqn[0],
                                   f->face_eqn[1], f->face_eqn[2]);
                        g_oblique_neumann_front_face_flux(f_sten,f,s,ic,dt,flux); 
                        /* This g_neumann_front_face_flux function is not right yet */
                        // g_neumann_front_face_flux(f_sten, f, s, ic, dt, flux); 
                    }
                    /*
                    printf("ERROR g_front_face_flux\n");
                    printf("Implement boundary front case wv_type NEUMANN_B\n");
                    facels(f, 2);
                    solidls(s, 2);
                    clean_up(ERROR); 
                    */
                }
                else 
                {
                    printf("ERROR g_front_face_flux\n");
                    printf("Implement boundary front case wv_type = %d\n", wv_type);
                    facels(f, 2);
                    solidls(s, 2);
                    clean_up(ERROR); 
                }
            }
            f = f->nextf; 
        }

        DEBUG_LEAVE(g_front_face_flux) 
        debug_print("ghyp", "Left g_front_face_flux\n"); 
}

LOCAL int oblique_face(
        CSG_Face     *f)
{
        int          i; 
        float        prod, *fn;

        for(i = 0; i < 4; i++)
        {
            fn = nor[i];
            prod = scalar_product(fn,f->face_eqn,3);
            if(fabs(prod - 1.0) <  0.0001*TOL)
                break;
        }
        if(i >= 4)
            return YES;
        else
            return NO; 
}

// This is a zero order implementation.
LOCAL void g_oblique_neumann_front_face_flux(
        Face_Stencil *f_sten,
        CSG_Face     *f,
        CSG_Solid    *s,
        int          *ic, 
        float        dt, 
        float        *flux)
{
        float        fnor[2], *fdir; /* fnor is fdir's projection in x-y plane */
        static Locstate     ans = NULL;
        int          i, j;
        size_t       sizest = f_sten->front->sizest;
        float        fflux[4] = {0.0, 0.0, 0.0, 0.0};
        float        rm, em, um[MAXD], pM; 
        float        t_area, area; 
#if defined(ROTATIONAL_SYMMETRY)
        static float alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
        float        J;

        f_sten->maxf = f; 
        for(i = 0; i < 3; i++)
            f_sten->centroid[i] = f->centroid[i];
        g_neumann_front_face_state(f_sten);

        if(ans == NULL)
        {
            alloc_state(f_sten->front->interf,&ans,max(sizeof(VGas),sizest));
            set_type_of_state(ans,GAS_STATE); 
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        J = 1.0;
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
            J = f_sten->centroid[0];
#endif /* defined(ROTATIONAL_SYMMETRY) */

        fdir = f->face_eqn;
        fnor[0] = fdir[0]/sqrt(sqr(fdir[0])+sqr(fdir[1]));
        fnor[1] = fdir[1]/sqrt(sqr(fdir[0])+sqr(fdir[1]));

        riemann_solution(0.0,fnor,f_sten->btm_st,f_sten->st[1],ans,GAS_STATE);
        rm = Dens(ans);
        em = Energy(ans); 
        pM = pressure(ans); 
        for(i = 0; i < dim; i++)
            um[i] = Mom(ans)[i]/rm;

        t_area = f_sten->t_area;
        area = f->area; 

        fflux[0] = (area/t_area)*(rm*um[0]*fnor[0]*J + rm*um[1]*fnor[1]*J);
        fflux[1] = (area/t_area)*((rm*um[0]*um[0]+pM)*fnor[0]*J + (rm*um[0]*um[1])*fnor[1]*J);
        fflux[2] = (area/t_area)*((rm*um[0]*um[1])*fnor[0]*J + (rm*um[1]*um[1]+pM)*fnor[1]*J);
        fflux[3] = (area/t_area)*((pM+em)*um[0]*fnor[0]*J + (pM+em)*um[1]*fnor[1]*J);

        for(i = 0; i < 4; i++)
            flux[i] += fflux[i];

#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            f_sten->P += pM;
            f_sten->np++;
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        if(f_sten->s->debug == YES)
        {
            printf("g_oblique_neumann_front_face_flux(), in dir[%g, %g, %g]\n", 
                     fdir[0], fdir[1], fdir[2]);
            printf("flux[%g, %g, %g, %g]\n",
                     fflux[0], fflux[1], fflux[2], fflux[3]);
        }
        

}

LOCAL void g_neumann_front_face_flux(
        Face_Stencil *f_sten,
        CSG_Face     *f,
        CSG_Solid    *s,
        int          *ic, 
        float        dt, 
        float        *flux)
{
        CSG_HalfEdge  *lh;
        CRXING        *crxing = NULL;
        float         dtt = HUGE_VAL;
        int           i, idir; 
        float         *fn, prod; 

        lh = f->floops->ledg;
        do
        {
            if(lh->vtx->pt->csg_pt.crx->hs  != NULL)
            {
                if(wave_type(lh->vtx->pt->csg_pt.crx->hs) == NEUMANN_BOUNDARY)
                {
                    if(dtt > Coords(lh->vtx->pt->csg_pt.crx->pt)[2])
                    {
                        crxing = lh->vtx->pt->csg_pt.crx;
                        dtt = Coords(lh->vtx->pt->csg_pt.crx->pt)[2]; 
                    }
                }
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        if(crxing == NULL)
        {
            printf("ERROR g_neumann_front_face_flux, crx null\n");
            clean_up(ERROR);
        }

        f_sten->maxf = f; 
        for(i = 0; i < 3; i++)
            f_sten->centroid[i] = f->centroid[i];  
        set_face_sten_pt_crds(f_sten,s,ic,f->face_eqn); 
        hyp_solution(f_sten->crds[1],s->comp,crxing->hs,UNKNOWN_SIDE,
                     f_sten->front,f_sten->wave,f_sten->st[0],NULL);

        g_sten_st_for_boundary_face(f_sten, ic, f->face_eqn); 

        if(f_sten->s->debug == YES)
        {
            printf("IN g_neumann_front_face_flux\n"); 
            print_face_sten_state(f_sten); 
        }

        for(i = 0; i < 4; i++)
        {
            fn = nor[i];
            prod = scalar_product(fn,f->face_eqn,3);
            if(fabs(prod - 1.0) <  0.0001*TOL)
                break;
        }        
        idir = i; 
        g_face_centroid_flux(f_sten, s, idir, f->centroid[2], f->face_eqn, flux);
}

LOCAL void g_neumann_front_face_state(
        Face_Stencil *f_sten)
{
        float        b_centroid[3]; 
        CSG_HalfEdge  *lh;
        CRXING       *crxing = NULL; 
        float        dt = HUGE_VAL; 
        Wave            *wave = f_sten->wave;
        float           nor[MAXD];
        float           coords_ref[MAXD];
        float           coords_tmp[MAXD];
        int             i;
        float           tmp_vel[MAXD], vn;

        lh = f_sten->maxf->floops->ledg;
        do
        {
            if(lh->vtx->pt->csg_pt.crx->hs  != NULL)
            {
                if(wave_type(lh->vtx->pt->csg_pt.crx->hs) == NEUMANN_BOUNDARY)
                {
                    if(dt > Coords(lh->vtx->pt->csg_pt.crx->pt)[2])
                    {
                        crxing = lh->vtx->pt->csg_pt.crx; 
                        dt = Coords(lh->vtx->pt->csg_pt.crx->pt)[2]; 
                    }
                }
            }
        }while((lh = lh->nxt) != f_sten->maxf->floops->ledg);
 
        if(crxing == NULL)
        {
            printf("ERROR g_neumann_front_face_state, crx null\n");
            clean_up(ERROR);  
        } 

        if(!reflect_pt_about_Nbdry(f_sten->b_centroid,coords_ref,nor,
                      f_sten->s->comp,crxing->hs,f_sten->front))
        {
            printf("ERROR g_neumann_front_face_state, not reflect pt\n");
            printf("coords[%g, %g]\n", f_sten->b_centroid[0], f_sten->b_centroid[1]); 
            clean_up(ERROR);  
        }
        assign(f_sten->st[1], f_sten->btm_st, f_sten->front->sizest); 
        set_state(f_sten->st[1],TGAS_STATE,f_sten->st[1]);
        vn = 0.0;
        for (i = 0; i < dim; i ++)
        {
            tmp_vel[i] = Vel(f_sten->st[1])[i];
            vn += tmp_vel[i] * nor[i];
        }

        if (no_slip(crxing->hs))
        {
            float alpha = adherence_coeff(crxing->hs);
            for (i = 0; i < dim; i ++)
                Vel(f_sten->st[1])[i] = (1-2*alpha)*Vel(f_sten->st[1])[i];
        }

        zero_normal_velocity(f_sten->st[1],nor,dim);
        for (i = 0; i < dim; i ++)
            Vel(f_sten->st[1])[i] += - vn * nor[i];
        set_state(f_sten->st[1],GAS_STATE,f_sten->st[1]);
}

LOCAL void g_tracked_front_face_flux(
        Face_Stencil *f_sten,
        CSG_Face     *f,
        CSG_Solid    *s,
        int          *ic, 
        float        dt, 
        float        *flux)
{
        CSG_Solid    *outs = NULL;  
        CSG_Face     *fother; 
        float        fnor[2], *fdir; /* fnor is fdir's projection in x-y plane */
        static Locstate     outbtm_st = NULL, Tist, Tost, ansi;
        int          i, j;  
        size_t       sizest = f_sten->front->sizest; 
        float        vtani[MAXD], vtano[MAXD];     /* velocities */ 
        float        ui,uo; 
        float        pjump = 0.0; 
        float        pml, pmr, pl, pr;    /* pressures */
        float        mr,ml;            /* miscellaneous variables */
        RIEMANN_SOLVER_WAVE_TYPE      l_wave,r_wave;
        float        uml,umr;    /* normal velocities */
        float        ri,ro;          /* densities */
        float        ux, uy; 
        float        cfl_fac; 
        float        en_den;
        float        CFL = Time_step_factor(f_sten->front);  
        float        rarea, rnor[3], t_area; 
        float        fflux[4] = {0.0, 0.0, 0.0, 0.0};  
#if defined(ROTATIONAL_SYMMETRY)
        static float alpha;      
#endif /* defined(ROTATIONAL_SYMMETRY) */
        float        J;

        if(outbtm_st == NULL)
        {
            alloc_state(f_sten->front->interf,&outbtm_st,
                     max(sizeof(VGas),sizest)); 
            alloc_state(f_sten->front->interf,&Tost,sizest); 
            alloc_state(f_sten->front->interf,&Tist,sizest); 
            alloc_state(f_sten->front->interf,&ansi,sizest); 
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry(); 
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        f_sten->maxf = f;
        for(i = 0; i < 3; i++)
            f_sten->centroid[i] = f->centroid[i];
        J = 1.0;
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
            J = f_sten->centroid[0]; 
#endif /* defined(ROTATIONAL_SYMMETRY) */

        fother = find_match_face_in_region4(f, ic, 
                   wave_tri_soln(f_sten->wave)->tri_grid); 
        if(fother == NULL)
        {
            printf("ERROR g_tracked_front_face_flux\n"); 
            printf("face does not have matched face from other side\n");
            facels(f, 2);
            solidls(s, 2);
            clean_up(ERROR); 
        }
        if(fother->fsolid->comp == f->fsolid->comp)
        {
            printf("ERROR  g_tracked_front_face_midst()\n");
            printf("face %d of solid %d eqn<%f, %f, %f>\n",
                   f->faceno, f->fsolid->solidno,
               f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
            printf("match face %d of solid %d eqn<%f, %f, %f>\n",
                  fother->faceno, fother->fsolid->solidno,
                 fother->face_eqn[0], fother->face_eqn[1],
                 fother->face_eqn[2]);
            printf("f, match_f Solid comp<%d, %d>\n",
                f->fsolid->comp , fother->fsolid->comp);
            printf("Print f->fsolid\n");
            solidls(f->fsolid,2);
            printf("Print match_f->fsolid\n");
            solidls(fother->fsolid,2);
            clean_up(ERROR);
        }

        btm_state2d(outbtm_st, fother->fsolid, f_sten->wave, f_sten->front); 

        fdir = f->face_eqn;
        fnor[0] = fdir[0]/sqrt(sqr(fdir[0])+sqr(fdir[1]));
        fnor[1] = fdir[1]/sqrt(sqr(fdir[0])+sqr(fdir[1]));
 
        set_state_for_find_mid_state(Tist, f_sten->btm_st);
        set_state_for_find_mid_state(Tost, outbtm_st);
       
        ui = uo = 0.0;
        for ( i = 0; i < 2; i++)
        {
             ui += fnor[i] * Vel(Tist)[i];
             uo += fnor[i] * Vel(Tost)[i];
         }

         for ( i = 0; i < dim; i++)
         {
              vtani[i] = Vel(Tist)[i] - fnor[i] * ui;
              vtano[i] = Vel(Tost)[i] - fnor[i] * uo;
         }

         Vel(Tist)[0] = ui;     Vel(Tost)[0] = uo;
         for (i = 1; i < 2; i++)
              Vel(Tist)[i] = Vel(Tost)[i] = 0.0;

        if (! find_mid_state(Tist,Tost,pjump,&pml,
                &pmr,&uml,&umr,&ml,&mr, &l_wave,&r_wave))
        {
             printf("ERROR in g_tracked_front_face_flux(),\n");
             printf("find_mid_state()did not converge");
             printf("for left/right RP\n");
        }
        /* contact only */
        midstate(Tist,ansi,ml,uml,pml,EGAS_STATE,l_wave,LEFT_FAMILY); 

        ri = Dens(ansi); 
        for(i = dim -1; i >= 0; i--)
        {
            Vel(ansi)[i] = vtani[i] + fnor[i]*Vel(ansi)[0]; 
        }

        cfl_fac = dt/dh[0]; 
        rarea = f->area;
        if(fabs(fdir[2]/sqrt(sqr(fdir[0])+sqr(fdir[1]))*cfl_fac) < CFL)
        {
            vol_intgel_normal(uml, fdir, rnor);
        }
        else
        {
            rnor[0] = fdir[0];
            rnor[1] = fdir[1];
            rnor[2] = fdir[2];
        }

        // ux = uml*fnor[0]; uy = uml*fnor[1]; 
        ux = Vel(ansi)[0]; uy = Vel(ansi)[1]; 
        en_den = ri*(0.5*(sqr(ux) + sqr(uy)) + Energy(ansi));
        pml = pressure(ansi); 
        t_area = f_sten->t_area; 
         
        fflux[0] = (rarea/t_area)*(ri*rnor[2]*J + ri*ux*rnor[0]*J + ri*uy*rnor[1]*J);
        fflux[1] = (rarea/t_area)*(ri*ux*rnor[2]*J + (ri*ux*ux+pml)*rnor[0]*J
                            + ri*ux*uy*rnor[1]*J);
        fflux[2] = (rarea/t_area)*(ri*uy*rnor[2]*J + (ri*uy*uy+pml)*rnor[1]*J
                             + ri*ux*uy*rnor[0]*J);
        fflux[3] = (rarea/t_area)*(en_den*rnor[2]*J +
                    (en_den+pml)*ux*rnor[0]*J + (en_den+pml)*uy*rnor[1]*J);  

#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            f_sten->P += pml;
            f_sten->np++;   
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        flux[0] += fflux[0];
        flux[1] += fflux[1];
        flux[2] += fflux[2];
        flux[3] += fflux[3];

        if(f_sten->s->debug == YES)
        {
            printf("g_tracked_front_face_flux(), in dir[%g, %g, %g]\n",
                   fdir[0], fdir[1], fdir[2]);
            printf("front face[%d] flux[%g, %g, %g, %g]\n", 
                  f->faceno, fflux[0], fflux[1], fflux[2], fflux[3]);
            printf("rarea %g, t_area %g, pml %g, uml %g, u[%g, %g] J %g\n",
                   rarea, t_area, pml, uml, ux, uy, J);
            printf("correction nor[[%g, %g, %g]\n", rnor[0], rnor[1], rnor[2]); 
        }

        /* Front face conservation check routine */
        if(s->comp == 2)
        {
            static int first = YES, step; 
            static float front_dens_flux = 0.0; 

            if(first)
            {
                first = NO;
                step = f_sten->front->step; 
            } 
            if(step != f_sten->front->step)
            {
                printf("Step[%d] front face dens conservation error %g\n",
                   step, front_dens_flux); 
                step = f_sten->front->step;
                front_dens_flux = 0.0; 
            }
            front_dens_flux += fflux[0]*t_area; 
        }
}

LOCAL void  vol_intgel_normal(
        float           um,
        float           *f_eq,  /* face equation */
        float           *int_normal)
{
        float  nx, ny, nt, len;

        nx = f_eq[0], ny = f_eq[1];
        nt = -um*sqrt(sqr(nx)+sqr(ny));
        len = sqrt(sqr(nx)+sqr(ny)+sqr(nt));
        int_normal[0] = nx/len;
        int_normal[1] = ny/len;
        int_normal[2] = nt/len;
}

/*** Replaced with find_match_face_in_region4()
 *** Calling functions do compoenent check.
LOCAL CSG_Face *face_on_other_side(
        CSG_Face     *f,
        CSG_Solid    *soff,
        int          *ic, 
        Wave         *wave)
{
        TRI_GRID      *ntg = wave_tri_soln(wave)->tri_grid; 
        int         i, j;
        CSG_Face    *match_f;
        int         icrds[MAXD];
        CSG_Solid   *s;
        CSG_BLK_CRX   *blk;

        for(i = ic[0]-3; i <= ic[0]+3; i++)
        {
            for(j = ic[1]-3; j <= ic[1]+3; j++)
            {
                icrds[0] = i; icrds[1] = j;
                if(icrds[0] < ggmin[0] || icrds[0] >= ggmax[0])
                    continue;
                if(icrds[1] < ggmin[1] || icrds[1] >= ggmax[1])
                    continue;
                if((blk = Comp_blk(icrds,ntg->Volume.blk,ntg)) == NULL)
                    continue;
                s = blk->s;
                while(s)
                {
                    if(s == f->fsolid)
                    {
                        s = s->nexts;
                        continue;
                    }
                    match_f = match_face_in_other(f, s);
                    if(match_f == NULL)
                        s = s->nexts;
                    else
                    {
                        if(f->fsolid->comp == match_f->fsolid->comp)
                        {
                            printf("ERROR  volume_on_other_side()\n");
                            printf("face %d of solid %d eqn<%f, %f, %f>\n",
                                    f->faceno, f->fsolid->solidno,
                                    f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                            printf("match face %d of solid %d eqn<%f, %f, %f>\n",
                             match_f->faceno, match_f->fsolid->solidno,
                             match_f->face_eqn[0], match_f->face_eqn[1],
                             match_f->face_eqn[2]);
                            printf("f, match_f Solid comp<%d, %d>\n",
                                    f->fsolid->comp , match_f->fsolid->comp);
                            printf("Print f->fsolid\n");
                            solidls(f->fsolid,2);
                            printf("Print match_f->fsolid\n");
                            solidls(match_f->fsolid,2);
                            clean_up(ERROR);
                        }
                        return  match_f;
                    }
                 }
             }  // End of for loop 
         }

         return NULL;
}
***/

/* The tracked front and the 
 * boundary front faces are only composed 
 * of crxing points.
 */

LOCAL int front_face(
        CSG_Face     *f,
        int          *wv_type)
{
        CSG_HalfEdge  *lh;

        *wv_type = UNKNOWN_BOUNDARY_TYPE; 

        lh = f->floops->ledg;
        do
        {
            if(lh->vtx->pt->type == IS_ICRDS)
            {
                return NO;  
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        lh = f->floops->ledg;
        do
        {
            if(lh->vtx->pt->csg_pt.crx->hs  != NULL)
            {
                // if(f->faceno == 1582) 
                //    print_hypersurface(lh->vtx->pt->csg_pt.crx->hs); 
                if(wave_type(lh->vtx->pt->csg_pt.crx->hs) > (*wv_type))
                    *wv_type = wave_type(lh->vtx->pt->csg_pt.crx->hs); 
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        return YES;  
}

LOCAL void set_face_sten_pt_crds(
        Face_Stencil *f_sten,
        CSG_Solid    *s,
        int          *ic,  
        float        *fdir)
{
        int          i, j, icrds[MAXD];  
        int          idir, sign, Tidir;  
        Wave         *wave = f_sten->wave;  
        RECT_GRID    *rect = wave->rect_grid; 
        float        prod, *fn, **crds2 = f_sten->crds; 
        int          **icrds2 = f_sten->icrds; 
        float        crds[3], factor; 
        float        Tfdir[3];  

        for(i = 0; i < 4; i++)
        {
            fn = nor[i];
            prod = scalar_product(fn,fdir,3);
            if(fabs(prod - 1.0) <  0.0001*TOL)
                break;
        }
        sign = 1; 
        if(i == 0 || i == 2)
            idir = 0; 
        else 
            idir = 1; 
        if(i == 2 || i == 3)
            sign = -1; 

        /* Get the cell center of ic */
        // off_face_pt(f_sten->maxf, crds, -1.5);
        for(i = 0; i < rect->dim; i++)
            crds[i] = f_sten->centroid[i]+f_sten->maxf->face_eqn[i]*dh[i]*(-1.5);
 
        /* The closest cell center to this crds */ 
        for(i = 0; i < rect->dim; i++)
            icrds[i] = cell_index(crds[i],i,rect);
        for(i = 0; i < rect->dim; i++)
            crds[i] = rect->L[i] + (icrds[i]+0.5)*rect->h[i]; 

        for(i = 0; i < rect->dim; i++)
        {
            factor = 0.0; 
            for(j = 0; j < 4; j++)
            {
                crds2[j][i] = crds[i]; 
                icrds2[j][i] = icrds[i]; 
                if(i == idir)
                {
                    crds2[j][i] += factor*rect->h[i]; 
                    icrds2[j][i] += sign*j; 
                    factor += 1.0*sign;  
                }
            }
        }

        /* tangential = ( n2,-n1)  */
        Tfdir[0] = fdir[1];
        Tfdir[1] = -1.0*fdir[0];

        for(i = 0; i < 4; i++)
        {
            fn = nor[i];
            prod = scalar_product(fn,Tfdir,3);
            if(fabs(prod - 1.0) <  0.0001*TOL)
                break;
        }

        if(i == 0 || i == 2)
            Tidir = 0; 
        else 
            Tidir = 1; 
        sign = 1; 
        if(i == 2 || i == 3)
            sign = -1; 
 
        /* point crds in tangential direction */
        for(i = 0; i < rect->dim; i++)
        {
            for(j = 0; j <= 1; j++)
            {
                f_sten->Tcrds[j][i] = crds2[1][i]; 
                f_sten->Ticrds[j][i] = icrds2[1][i]; 
                f_sten->oTcrds[j][i] = crds2[2][i]; 
                f_sten->oTicrds[j][i] = icrds2[2][i]; 
                if(i == Tidir)
                {
                    if(j == 0)
                    {
                        f_sten->Tcrds[j][i] += -1.0*sign*rect->h[i];
                        f_sten->oTcrds[j][i] += -1.0*sign*rect->h[i];
                        f_sten->Ticrds[j][i] += -1*sign; 
                        f_sten->oTicrds[j][i] += -1*sign; 
                    }
                    else
                    {
                        f_sten->Tcrds[j][i] += sign*rect->h[i];
                        f_sten->oTcrds[j][i] += sign*rect->h[i];
                        f_sten->Ticrds[j][i] += sign; 
                        f_sten->oTicrds[j][i] += sign; 
                    }
                }
            }
        }
}

LOCAL void g_face_centroid_flux(
        Face_Stencil *f_sten,
        CSG_Solid    *s,
        int          idir, 
        float        dt, 
        float        *fdir,
        float        *flux)
{
        Wave           *wave = f_sten->wave; 
        static Vec_Gas *vst = NULL;
        static Vec_Src *src;
        static Vec_Gas *Tvst[2] = {NULL, NULL}; 
        static float   **Q = NULL;
        static int     *pseudo_iperm; 

        int            i, j, side, swp_num; 
        float          **coords; 
        float          *rho, *en_den;
        float          *m[MAXD];
        Locstate       st, *state, *tstate0, *tstate1; 
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        float           *vacuum_dens;
        float           *min_pressure;
        float           *min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
#if defined(ROTATIONAL_SYMMETRY)
        static float    alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
        int             idirs[MAXD];
        RECT_GRID       *gr = f_sten->front->rect_grid;
 
        if (vst == NULL)
        {
            /* Assumes stencil size never changes */

            // g_unsplit_muscl_alloc_phys_vecs;
            alloc_unsplit_phys_vecs(wave,vsten_rad*2);
            vst = g_wave_vgas(wave);
            src = g_wave_vsrc(wave);
            Tvst[0] = g_wave_Tvgas(wave)[0];
            Tvst[1] = g_wave_Tvgas(wave)[1];

            g_wave_vgas(wave) = NULL;
            g_wave_vsrc(wave) = NULL;
            g_wave_Tvgas(wave)[0] = NULL;
            g_wave_Tvgas(wave)[1] = NULL;
            matrix(&Q,3,3,FLOAT);
            vst->Q = (const float* const*)Q;
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
            vector(&pseudo_iperm, 2, sizeof(int)); 
            for(i = 0; i < dim; i++)
                pseudo_iperm[i] = i; 
        }

        clear_Vec_Gas_set_flags(vst);
        clear_Vec_Gas_set_flags(Tvst[0]);
        clear_Vec_Gas_set_flags(Tvst[1]);
        for (i = 0; i < 3; ++i)
            for (j = 0; j < 3; ++j)
                Q[i][j] = 0.0;
        if(idir == 0 || idir == 2)
            swp_num = 0;
        else
            swp_num = 1; 
        idirs[0] = swp_num; 
        idirs[1] = (swp_num+1)%2;


        for (i = 0; i < dim; ++i)
            Q[i][idirs[i]] = 1.0;
        for (; i < 3; ++i)
            Q[i][i] = 1.0;

        state = vst->state; 
        coords = vst->coords; 
        switch(idir)
        {
        case 0:
        case 1:
            for(i = 0; i < 4; i++)
            {
                state[i] = f_sten->st[i-1]; 
                coords[i] = f_sten->crds[i]; 
            }
        break; 
        case 2:
        case 3:
            for(i = 0; i < 4; i++)
            {
                state[i] = f_sten->st[2-i]; 
                coords[i] = f_sten->crds[3-i]; 
            }
        break; 
        }

        tstate1 = Tvst[1]->state; 
        tstate0 = Tvst[0]->state; 
        switch(idir)
        {
        case 0:
            tstate1[1] = f_sten->tan_st[-1]; 
            tstate1[2] = f_sten->outtan_st[-1]; 
            tstate0[1] = f_sten->tan_st[1]; 
            tstate0[2] = f_sten->outtan_st[1]; 
        break; 
        case 1:
            tstate1[1] = f_sten->tan_st[1]; 
            tstate1[2] = f_sten->outtan_st[1]; 
            tstate0[1] = f_sten->tan_st[-1]; 
            tstate0[2] = f_sten->outtan_st[-1]; 
        break;  
        case 2:
            tstate1[1] = f_sten->outtan_st[1]; 
            tstate1[2] = f_sten->tan_st[1]; 
            tstate0[1] = f_sten->outtan_st[-1]; 
            tstate0[2] = f_sten->tan_st[-1]; 
        break; 
        case 3:
            tstate1[1] = f_sten->outtan_st[-1]; 
            tstate1[2] = f_sten->tan_st[-1]; 
            tstate0[1] = f_sten->outtan_st[1]; 
            tstate0[2] = f_sten->tan_st[1]; 
        break; 
        }

        rho = vst->rho;
        state = vst->state;
        en_den = vst->en_den;
        for (i = 0; i < dim; ++i)
            m[i] = vst->m[i];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        vacuum_dens = vst->vacuum_dens;
        min_pressure = vst->min_pressure;
        min_energy = vst->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        for(i = 0; i < 4; i++)
        {
            rho[i] = Dens(state[i]);
            en_den[i] = Energy(state[i]);
            for (j = 0; j < dim; ++j)
                m[j][i] = Mom(state[i])[j];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            vacuum_dens[i] = Vacuum_dens(state[i]);
            min_pressure[i] = Min_pressure(state[i]);
            min_energy[i] = Min_energy(state[i]);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        }
        Vec_Gas_field_set(vst,state) = YES;
        Vec_Gas_field_set(vst,coords) = YES;
        Vec_Gas_field_set(vst,rho) = YES;
        Vec_Gas_field_set(vst,en_den) = YES;
        Vec_Gas_field_set(vst,m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
        Vec_Gas_field_set(vst,vacuum_dens) = YES;
        Vec_Gas_field_set(vst,min_pressure) = YES;
        Vec_Gas_field_set(vst,min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
        set_params_jumps(vst,0,4);

#if defined(ROTATIONAL_SYMMETRY)
        /* If using cylindrical coordinates and this is the
         *  r-sweep include radius information for source
         *  computation.
         */

        if (alpha > 0.0 && idirs[0] == 0)
        {
            float *radii = src->radii;

            src->rmin = fabs(pos_radius(0.0,gr));
            for (i = 0; i < 4; ++i)
                radii[i] = pos_radius(coords[i][0],gr);
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        for(side = 0; side <= 1; side++)
        {
            rho = Tvst[side]->rho;
            state = Tvst[side]->state;
            en_den = Tvst[side]->en_den;
            for (i = 0; i < dim; ++i)
                m[i] = Tvst[side]->m[i];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            vacuum_dens = Tvst[side]->vacuum_dens;
            min_pressure = Tvst[side]->min_pressure;
            min_energy = Tvst[side]->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            for(i = 1; i <= 2; i++)
            {
                rho[i] = Dens(state[i]);
                en_den[i] = Energy(state[i]);
                for (j = 0; j < dim; ++j)
                    m[j][i] = Mom(state[i])[j];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
                vacuum_dens[i] = Vacuum_dens(state[i]);
                min_pressure[i] = Min_pressure(state[i]);
                min_energy[i] = Min_energy(state[i]);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            }
            Vec_Gas_field_set(Tvst[side],state) = YES;
            // Vec_Gas_field_set(Tvst[side],coords) = YES;
            Vec_Gas_field_set(Tvst[side],rho) = YES;
            Vec_Gas_field_set(Tvst[side],en_den) = YES;
            Vec_Gas_field_set(Tvst[side],m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
            Vec_Gas_field_set(Tvst[side],vacuum_dens) = YES;
            Vec_Gas_field_set(Tvst[side],min_pressure) = YES;
            Vec_Gas_field_set(Tvst[side],min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
            // set_params_jumps(Tvst[side],0,npts);
        }

        /*
        if(s->debug == YES)
        {
            printf("solid face flux state\n");
            for (i = 0; i < 4; ++i)
            {
                (void) printf("state[%d]\n",i);
                g_print_state(vst->state[i]);
            }
            for(side = 0; side <= 1; side++)
            {
                for(i = 1; i <= 2; i++)
                {
                    (void) printf("Tstate[%d][%d]\n",side, i);
                    g_print_state(Tvst[side]->state[i]);
                }
            }
        }
        */

        g_consv_muscl_flux(f_sten,fdir,swp_num,pseudo_iperm,
            f_sten->wave,f_sten->front,f_sten->newfront,
            0,4,vst,Tvst,src,dt,dh,0,flux);
}

LOCAL void g_consv_muscl_flux(
        Face_Stencil    *f_sten,
        float           *fdir, 
        int             swp_num,
        int             *iperm,
        Wave            *wave,
        Front           *fr,
        Front           *newfr,
        int             offset,
        int             vsize,
        Vec_Gas         *vst,
        Vec_Gas         **Tvst,
        Vec_Src         *src,
        float           dt,
        float           *dn,
        int             pbuf,
        float           *flux)
{
        Vec_Muscl       *vmuscl;
        float           dtdni;
        float           t_area; 
        int             idirs[MAXD];
        int             start, end;
        int             i, j, k, kmax;
        Muscl_Opts      *MOpts;
        float           radl;
        RECT_GRID       *r_grid = wave->rect_grid;
        float           fflux[4]; 
        float           *pM, E;
        static Locstate smid = NULL;

        static int     first = YES; 
#if defined(ROTATIONAL_SYMMETRY)
        static float   alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
        float          Jx, Jy; 

        if (first == YES)
        {
            first = NO;
#if defined(ROTATIONAL_SYMMETRY)
            alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
        }

        MOpts = muscl_options();
        for (j = 0; j < dim; ++j)
            idirs[j] = iperm[(j+swp_num)%dim];
        dtdni = dt/dn[idirs[0]];

        Jx = Jy = 1.0; 
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0)
        {
            if(idirs[0] == 0)
            {
                Jx = f_sten->centroid[0]; 
                if(fabs(Jx) < 0.005*dh[0])  
                    Jx = (Jx < 0.0) ? -0.005*dh[0] : 0.005*dh[0];
            }
            else
            {
                Jy = f_sten->centroid[0]; 
            }
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */

        /*
        if(f_sten->s->debug == YES)
        {
            add_to_debug("N_half_step");  
            add_to_debug("load_Vgas");  
            printf("Debugging: g_consv_muscl_flux\n");  
            printf("In dir[%g, %g], swp_num = %d side area = %g, top_area = %g area ratio = %g\n",
                  fdir[0], fdir[1], swp_num, f_sten->area, f_sten->t_area, 
                  f_sten->area/f_sten->t_area);
            printf("Idirs[%d, %d]\n", idirs[0], idirs[1]); 
            for(j = 0; j < 4; j++)
                print_vector("Coords",dim, vst->coords[j], "%g ");  
            for(j = 0; j < 4; j++)
                g_verbose_print_state(vst->state[j]);  
        }
        */

        vmuscl = g_load_unsplit_muscl_state_data(MOpts,swp_num,iperm,fr,wave,
                      NULL,NULL,vst,Tvst,src,offset,vsize,dim,dt,dn[idirs[0]]);
        /*
        if(f_sten->s->debug == YES)
        {
            printf("The internal energy :\n"); 
            for(j = 0; j < 4; j++)
                printf("e[%d]= %g    ", j, vst->e[j]);
            printf("\n");  
        }
        */

        /* compute the eigenvalues, eigenvectors and max wavespeed*/
        g_compute_2dunsplit_eigens(0,vsize,swp_num,iperm,dim,vmuscl);

        /* compute the linear reconstruction of the state variables */
        g_2dunsplit_bct_linear_reconstructor(1,3,vmuscl);

        if(f_sten->s->debug == YES)
        {
            // add_to_debug("N_half_step");  
            // add_to_debug("T_half_step");  
        }

        /* USing face centroid here, so dt multiples 2.0 here.
         * Inside N_half_step and T_half_step, there is a factor 0.5
         */
        /* Half time step expansion in normal dir. */
        g_2dunsplit_N_half_step(2,3,2.0*dt,dn,swp_num,iperm,dim,vmuscl);

        /* Add the constribution from another direction */
        g_2dunsplit_T_half_step(1,3,2.0*dt,dn,swp_num,iperm,dim,vmuscl);

        if(f_sten->s->debug == YES)
        {
            // remove_from_debug("N_half_step");  
            // remove_from_debug("T_half_step");  
        }

        g_unsplit_muscl_exact_rsolver(2,3,swp_num,iperm,vmuscl->uL,
              &vmuscl->VLst,vmuscl->uR,&vmuscl->VRst,vmuscl->uM,
              &vmuscl->VMst,&vmuscl->Flux,vmuscl);

        /* Set conservative source terms */
        // g_test_cons_src(2,2,swp_num,iperm,NULL,vmuscl);

        if (smid == NULL)
        {
            size_t      sizest = vmuscl->sizest;
            g_alloc_state(&smid,sizest);
        }        
        Dens(smid) = vmuscl->VMst.rho[2];
        Energy(smid) = vmuscl->VMst.e[2];
        for (i = 0; i < dim; ++i)
            Vel(smid)[i] = vmuscl->VMst.v[i][2];
        Set_params(smid,vmuscl->vst->state[1]);
        set_type_of_state(smid,EGAS_STATE);

        t_area = f_sten->t_area; 
        pM = vmuscl->VMst.p; 
        E = Dens(smid)*( 0.5*(sqr(Vel(smid)[0])+sqr(Vel(smid)[1]))
              +Energy(smid));

        /*
        if(f_sten->s->debug == YES)
        {
            printf("The middle state is:");
            g_verbose_print_state(smid);    
        }
        */
 
        if(debugging("bad_state"))
        {
            if(is_bad_state(smid,YES,"g_consv_muscl_flux"))
            {
                printf("ERROR g_consv_muscl_flux(), middle state is bad:");
                g_verbose_print_state(smid);

                solidls(f_sten->s, 2);
                clean_up(ERROR);
            }
        }
        
        fflux[0] = (f_sten->area/t_area)*(Dens(smid)*Vel(smid)[0]*fdir[0]*Jx + 
                                 Dens(smid)*Vel(smid)[1]*fdir[1]*Jy);
        fflux[1] = (f_sten->area/t_area)*((Dens(smid)*Vel(smid)[0]*Vel(smid)[0]+pM[2])*fdir[0]*Jx + 
                                 (Dens(smid)*Vel(smid)[0]*Vel(smid)[1])*fdir[1]*Jy);
        fflux[2] = (f_sten->area/t_area)*((Dens(smid)*Vel(smid)[0]*Vel(smid)[1])*fdir[0]*Jx + 
                                 (Dens(smid)*Vel(smid)[1]*Vel(smid)[1]+pM[2])*fdir[1]*Jy); 
        fflux[3] = (f_sten->area/t_area)*((pM[2]+E)*Vel(smid)[0]*fdir[0]*Jx +
                                  (pM[2]+E)*Vel(smid)[1]*fdir[1]*Jy); 
#if defined(ROTATIONAL_SYMMETRY)
        if(alpha > 0.0 && idirs[0] == 0)
        // if(alpha > 0.0)
        {
            f_sten->P += pM[2];
            f_sten->np++; 
        }
#endif /* defined(ROTATIONAL_SYMMETRY) */
       
        for(i = 0; i < 4; i++)
            flux[i] += fflux[i]; 

        if(f_sten->s->debug == YES)
        {
            static int loop = 0;
            static float dens_flux_sum = 0.0; 

            printf("g_consv_muscl_flux(), in dir[%g, %g, %g]\n",fdir[0],fdir[1],fdir[2]); 
            printf("mid state rho, v, e, p [%g, %g, %g, %g, %g]\n",
                        Dens(smid), Vel(smid)[0], Vel(smid)[1], Energy(smid), pM[2]); 
            /*
            printf("flux[%17.15f, %17.15f, %17.15f, %17.15f]\n", 
                      fflux[0]/ (f_sten->area/t_area), 
                      fflux[1]/ (f_sten->area/t_area), 
                      fflux[2]/ (f_sten->area/t_area), 
                      fflux[3]/ (f_sten->area/t_area)); 
            */
            printf("inter face[%d] flux[%g, %g, %g, %g]\n",
               f_sten->maxf->faceno,fflux[0],fflux[1],fflux[2],fflux[3]); 
            dens_flux_sum += fflux[0]/ (f_sten->area/t_area);
            loop++;  
            if(loop == 4)
            {
                loop = 0; 
                printf("Face dens_flux sum = %17.15f\n", dens_flux_sum);
                dens_flux_sum = 0.0; 
            }    
        }
        /* 
        if(f_sten->s->debug == YES)
        {
            remove_from_debug("N_half_step");  
            remove_from_debug("load_Vgas"); 
        }
        */
}


LOCAL void g_sten_st_for_boundary_face(
        Face_Stencil *f_sten,
        int          *ic,
        float        *dir)
{
        int           i, j;
        Wave         *wave = f_sten->wave;
        Wave         *nwave = f_sten->newwave;
        Front        *fr = f_sten->front;
        Front        *nfr = f_sten->newfront;
        TRI_GRID     *ntg = wave_tri_soln(f_sten->wave)->tri_grid;
        size_t        sizest = fr->sizest;
        CSG_Solid     *outs;
        int           icrds[MAXD], icrds1[MAXD], ic1[MAXD], outic[MAXD];
        float         crds[3], p[3];
        int           outdomain, side, ddir, buffered;
        CRXING        *crxing, *first_crxing = NULL; 

        for( i = -vsten_rad; i < 0; i++)
           assign(f_sten->st[i], f_sten->st[0], sizest);
        for( i = 1; i <= vsten_rad; i++)
           assign(f_sten->st[i], f_sten->st[0], sizest);

        point_with_rect(f_sten->crds[2], &ntg->comp_grid,
                outic,&outdomain,&ddir,&side,&buffered);

        /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
        if((crxing = cross_physics_boundary(dir, wave, fr,
                 f_sten->icrds[1], f_sten->icrds[2])) != NULL)
        {
            first_crxing = crxing;
            set_cross_physics_boundary_st(f_sten->crds[1],f_sten->s->comp,
                  crxing, f_sten->front, f_sten->wave, f_sten->st[1]);
        }
        else
        {
            printf("ERROR g_sten_st_for_boundary_face, face not on physics boundary\n");
            print_face_sten(f_sten);
            clean_up(ERROR); 
        }

        for(i = 2; i <= vsten_rad; i++)
            assign(f_sten->st[i], f_sten->st[1], sizest);
        for(i = -1; i <= 1; i++)
        {
            assign(f_sten->tan_st[i], f_sten->st[0], sizest);
            assign(f_sten->outtan_st[i], f_sten->st[1], sizest);
        }
       
        /* Assume the interior front normal is pointing to the postive dir */
        /* First find the left far state st[-1], st[-2], */
        /* We only need to use st[-1], st[-2] is set equal to st[-1] */

        point_with_rect(f_sten->crds[0], &ntg->comp_grid,
                     outic,&outdomain,&ddir,&side,&buffered);

        if(outdomain == YES)
        {
            /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
            if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                  f_sten->icrds[0], f_sten->icrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->crds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->st[-1]);
            }
        }
        else
        {
            if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                  f_sten->icrds[0], f_sten->icrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->crds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->st[-1]);
            }
            else
            {
                get_state_in_cell3(f_sten->crds[0],f_sten->icrds[0],f_sten->s->comp,
                 f_sten->st[-1], f_sten->wave, f_sten->newwave, f_sten->front);
            }
        }
        assign(f_sten->st[-2], f_sten->st[-1], sizest);
        /* FIND THE FAR RIGHT STATE st[2] */

        if(first_crxing != NULL)
        {
            set_cross_physics_boundary_st(f_sten->crds[3],f_sten->s->comp,
                  first_crxing, f_sten->front, f_sten->wave, f_sten->st[2]);
        }

        /* SET THE TANGENTIAL DIRECTION STATE */
        g_sten_tangnt_st_for_inter_face(f_sten, dir);        
}


LOCAL void g_sten_st_for_inter_face(
       Face_Stencil *f_sten,
       CSG_Solid    *s,
       int          *ic,
       float        *dir)
{
        int           i, j;
        Wave         *wave = f_sten->wave; 
        Wave         *nwave = f_sten->newwave;
        Front        *fr = f_sten->front;
        Front        *nfr = f_sten->newfront; 
        TRI_GRID     *ntg = wave_tri_soln(f_sten->wave)->tri_grid;
        size_t        sizest = fr->sizest;

        CSG_Solid     *outs;  
        int           icrds[MAXD], icrds1[MAXD], ic1[MAXD], outic[MAXD];
        float         crds[3], p[3];
        int           outdomain, side, ddir, buffered; 
        CRXING        *crxing, *first_crxing = NULL;  
        int           deb_far_st = NO, deb_far_st_t = NO;
 
        debug_print("ghyp","Entered g_sten_st_for_inter_face\n");  

        assign(f_sten->st[0], f_sten->btm_st, sizest);

        f_sten->maxf->i_stat = f_sten->st[0];
        for( i = -vsten_rad; i < 0; i++)
           assign(f_sten->st[i], f_sten->st[0], sizest);
        for( i = 1; i <= vsten_rad; i++)
           assign(f_sten->st[i], f_sten->st[0], sizest);

        /* Due to the shifting of the crxing points, the physical boundary
         * curves are outside the computational domain. So there is the
         * case the interior face(no crxing points) is on the physical boundary.
         */  
        if(YES == face_on_boundary(f_sten->maxf, &buffered, &ddir, &side))
        {
            g_sten_st_for_boundary_face(f_sten, ic, dir); 
            return; 
        }

        outs = find_out_nghbr(f_sten->maxf,ic,wave,fr,nwave,nfr,
                  outic,&outdomain,&ddir, &side);         
        if(outs != NULL)
        {
             f_sten->outs = outs;
             btm_state2d(f_sten->st[1], outs, wave, fr);
             // f_sten->maxf->o_stat = f_sten->st[1];
             f_sten->outic[0] = outs->icrds[0]; f_sten->outic[1] = outs->icrds[1];
        }
        else
        { 
             /* in the case there is no outter nghbr, use cell center states */
             /* Or might need to call the interpolation function if the volume
              * is not constructed.
              * This is a rather complicated situation, the 
              * volume might be on the boundary of the cells which has bifurcated front
              * or other degenerated cases. 
              */
             point_with_rect(f_sten->crds[2], &ntg->comp_grid, 
                     outic,&outdomain,&ddir,&side,&buffered); 
             if(outdomain == YES)
             {
                /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
                if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                      f_sten->icrds[1], f_sten->icrds[2])) != NULL)
                {
                    first_crxing = crxing; 
                    set_cross_physics_boundary_st(f_sten->crds[1],f_sten->s->comp,
                          crxing, f_sten->front, f_sten->wave, f_sten->st[1]); 
                }
             }
             else
             {
                 get_state_in_cell2(f_sten->crds[2],f_sten->icrds[2],f_sten->s->comp,
                     f_sten->st[1], f_sten->wave, f_sten->newwave, f_sten->front);  
                 f_sten->outic[0] = f_sten->icrds[2][0];
                 f_sten->outic[1] = f_sten->icrds[2][1];
             }

             f_sten->outs = NULL;
             // f_sten->maxf->o_stat = f_sten->st[1];
        }
        for(i = 2; i <= vsten_rad; i++)
            assign(f_sten->st[i], f_sten->st[1], sizest);  
        for(i = -1; i <= 1; i++)
        {
            assign(f_sten->tan_st[i], f_sten->st[0], sizest);
            assign(f_sten->outtan_st[i], f_sten->st[1], sizest);
        }
        
        /* SET THE TANGENTIAL DIRECTION STATE */       
        g_sten_tangnt_st_for_inter_face(f_sten, dir); 

        /* Assume the interior front normal is pointing to the postive dir */
        /* First find the left far state st[-1], st[-2], */
        /* We only need to use st[-1], st[-2] is set equal to st[-1] */

        point_with_rect(f_sten->crds[0], &ntg->comp_grid, 
                     outic,&outdomain,&ddir,&side,&buffered); 
 
        if(outdomain == YES)
        {
            /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
            if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                  f_sten->icrds[0], f_sten->icrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->crds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->st[-1]); 
            }
        }
        else
        {
            if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                  f_sten->icrds[0], f_sten->icrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->crds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->st[-1]); 
            }
            else
            {
                get_state_in_cell3(f_sten->crds[0],f_sten->icrds[0],f_sten->s->comp,
                 f_sten->st[-1], f_sten->wave, f_sten->newwave, f_sten->front);  
            }
        }       
        assign(f_sten->st[-2], f_sten->st[-1], sizest);

        /* FIND THE FAR RIGHT STATE st[2] */
 
        if(first_crxing != NULL)
        {
            set_cross_physics_boundary_st(f_sten->crds[3],f_sten->s->comp,
                  first_crxing, f_sten->front, f_sten->wave, f_sten->st[2]); 
            return; 
        }
 
        point_with_rect(f_sten->crds[3], &ntg->comp_grid, 
                     outic,&outdomain,&ddir,&side,&buffered); 
        if(outdomain == YES)
        {
            /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
            if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                  f_sten->icrds[2], f_sten->icrds[3])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->crds[3],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->st[2]); 
            }
        }
        else
        {
            if((crxing = cross_physics_boundary(dir, f_sten->wave, f_sten->front,
                  f_sten->icrds[2], f_sten->icrds[3])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->crds[3],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->st[2]); 
            }
            else
            {
                get_state_in_cell3(f_sten->crds[3],f_sten->icrds[3],f_sten->s->comp,
                     f_sten->st[2], f_sten->wave, f_sten->newwave, f_sten->front);  
            }
        }

        /*
        point_with_rect(p, &ntg->comp_grid, icrds1,&outdomain,&ddir,&side,&buffered);
        find_outter_st2(f_sten->st[2],icrds1, dir, f_sten);  
        */

        debug_print("ghyp","Left g_sten_st_for_inter_face\n");  
}

LOCAL void set_cross_physics_boundary_st(
        float         *crds,
        COMPONENT     comp,
        CRXING        *crx,
        Front         *fr,
        Wave          *wv,
        Locstate      st)
{

        if ((wave_type(crx->hs) == NEUMANN_BOUNDARY) &&
            (fr->neumann_bdry_state))
        {
            if (!(*(fr->neumann_bdry_state))(crds,comp,
                    crx->pt,crx->hs,fr,(POINTER)(wv),st))
            {
                printf("ERROR set_cross_physics_boundary_st, not set Neumann\n");
                clean_up(ERROR);
            }
        }
        else if(wave_type(crx->hs) == DIRICHLET_BOUNDARY)
        {
            evaluate_dirichlet_boundary_state(crds,crx->hs,fr,wv,st);
        }
}

LOCAL CRXING *cross_physics_boundary(
        float         *fdir,
        Wave          *wave,
        Front         *front,
        int           *pic,   /* pic and nic are order in the fdir direction */
        int           *nic)
{
        int           idir, i;
        float         *fn, prod; 
        int           ic1[MAXD], ic2[MAXD]; 
        int           outdomain1, outdomain2; 
        int           ddir, side, buffered;
        CRXING        *crx[20];
        int           nc; 
        GRID_DIRECTION  pgdir, gdir[2][2] = {SOUTH, NORTH,
                                      WEST, EAST};

        for(i = 0; i < 4; i++)
        {
            fn = nor[i];
            prod = scalar_product(fn,fdir,3);
            if(fabs(prod - 1.0) <  0.0001*TOL)
                break; 
        }
        idir = i;  
        /* Save the pic and nic in the x-y coords direction */
        if(idir == 0 || idir == 1)
        {
            memcpy(ic1, pic, sizeof(int)*2); 
            memcpy(ic2, nic, sizeof(int)*2); 
        }
        else
        {
            memcpy(ic1, nic, sizeof(int)*2); 
            memcpy(ic2, pic, sizeof(int)*2); 
        }
        
        ipoint_with_rect(ic1, &outdomain1, &ddir, &side, &buffered); 
        ipoint_with_rect(ic2, &outdomain2, &ddir, &side, &buffered); 

        if(outdomain1 == YES && outdomain2 == YES)
            return NULL; 
 
        if(outdomain1 == YES)
        {
            /* use ic2 to do the check, */
            if(idir == 0 || idir == 2)
                pgdir = gdir[1][0];
            else
                pgdir = gdir[0][0];

            nc = crossings_in_direction(crx, ic2,pgdir,
                     wave_tri_soln(wave)->tri_grid);
            if(crx[0] != NULL)
            {
                if (wave_type(crx[0]->hs) == NEUMANN_BOUNDARY ||
                    wave_type(crx[0]->hs) == DIRICHLET_BOUNDARY)
                {
                     return crx[0]; 
                }
            }
            return NULL; 
        }
       
        /* use ic1 to do the check, */
        if(idir == 0 || idir == 2)
            pgdir = gdir[1][1];
        else
            pgdir = gdir[0][1];
        
        nc = crossings_in_direction(crx, ic1,pgdir,
                 wave_tri_soln(wave)->tri_grid);
        if(crx[0] != NULL)
        {
            if (wave_type(crx[0]->hs) == NEUMANN_BOUNDARY ||
                wave_type(crx[0]->hs) == DIRICHLET_BOUNDARY)
            {
                return crx[0]; 
            }
        }
        return NULL; 
}  

/* This function is build for f_sten->st[2] */
LOCAL void find_outter_st2(
        Locstate      st,
        int           *ic, /* cell center on the face normal side */
        float         *fdir,
        Face_Stencil  *f_sten) 
{
        int           idir, i;
        float         *fn, prod; 
        CRXING        *btcrx[20], *ncrx[20]; 
        HYPER_SURF    *hs;
        int           btnc, nnc; 
        GRID_DIRECTION  pgdir, gdir[2][2] = {SOUTH, NORTH,
                                      WEST, EAST};
        float         crds[3]; 
        int           newic[3]; 
        Wave          *wave = f_sten->wave; 
        int           outdomain, ddir, side, buffered; 

        /* Get the cell center of ic */
        off_face_pt(f_sten->maxf, crds, -0.5);
        if(rect_in_which(crds,newic,wave->rect_grid) == FUNCTION_FAILED)
        {
            printf("ERROR in find_outter_st2(), "
                   "rect_in_which() failed\n");
            printf("crds[%g, %g], icrds[%d,%d]\n", crds[0], crds[1],
                     newic[0], newic[1]);
            clean_up(ERROR);        
        }
        for(i = 0; i < 2; i++)
            crds[i] = Rect_coords(newic,wave)[i] + dh[i]*fdir[i]; 

        for(i = 0; i < 4; i++)
        {
            fn = nor[i];
            prod = scalar_product(fn,fdir,3);
            if(fabs(prod - 1.0) <  0.0001*TOL)
                break; 
        }
        idir = i; 

        if(idir == 0) 
            pgdir = gdir[1][0]; 
        else if(idir == 2)
            pgdir = gdir[1][1]; 
        else if(idir == 1)
            pgdir = gdir[0][0]; 
        else
            pgdir = gdir[0][1]; 
  
         /* First check two pts which are on two sides of maxf
          * respectively.
          */
         btnc = crossings_in_direction(btcrx, ic,pgdir,
                     wave_tri_soln(f_sten->wave)->tri_grid);        

         /* At this moment, we only take care of physics boundary */
         if(btcrx[0] != NULL)
         {
             hs = btcrx[0]->hs;
             if ((wave_type(hs) == NEUMANN_BOUNDARY) &&
              (f_sten->front->neumann_bdry_state))             
             {
                if (!(*(f_sten->front->neumann_bdry_state))(crds,f_sten->s->comp,
                        btcrx[0]->pt,hs,f_sten->front,
                    (POINTER)(f_sten->wave),f_sten->st[2]))
                {
                    printf("ERROR find_outter_st2, the state is not set\n");
                    clean_up(ERROR);                    
                } 
             }
             else if(wave_type(hs) == DIRICHLET_BOUNDARY)
             {
                 evaluate_dirichlet_boundary_state(crds,hs,f_sten->front,
                              wave,f_sten->st[2]);
             }
             else
             {
             }
             return; 
         }

         /* Reverse the direction */
         if(idir == 0) 
             pgdir = gdir[1][1]; 
         else if(idir == 2)
             pgdir = gdir[1][0]; 
         else if(idir == 1)
             pgdir = gdir[0][1]; 
         else
             pgdir = gdir[0][0]; 

         nnc = crossings_in_direction(ncrx, ic,pgdir,
                     wave_tri_soln(wave)->tri_grid);        
         if(ncrx[0] != NULL)
         {
            /* Get the correponding postion of st[2] */
            for(i = 0; i < 2; i++)
                crds[i] = crds[i] + dh[i]*fdir[i]; 

             hs = ncrx[0]->hs;
             if ((wave_type(hs) == NEUMANN_BOUNDARY) &&
              (f_sten->front->neumann_bdry_state))             
             {
                if (!(*(f_sten->front->neumann_bdry_state))(crds,f_sten->s->comp,
                        ncrx[0]->pt,hs,f_sten->front,
                    (POINTER)(f_sten->wave),f_sten->st[2]))
                {
                    printf("ERROR find_outter_st2, 2 the state is not set\n");
                    clean_up(ERROR);                    
                } 
             }
             else if(wave_type(hs) == DIRICHLET_BOUNDARY)
             {
                 evaluate_dirichlet_boundary_state(crds,hs,f_sten->front,
                              wave,f_sten->st[2]);
             }
             return; 
         }

         /* Now check whether st[2] is located inside
          * the domain cell which is not ocuppied by
          * any front face */

        off_face_pt(f_sten->maxf,crds,1.5);
        point_with_rect(crds,wave->rect_grid,newic,&outdomain,&ddir,&side,&buffered);
        if(outdomain == YES)
            return;
        /*
        get_state_in_cell(crds, f_sten->s->comp, f_sten->st[2], NO, NO, 
            ddir, side, wave, f_sten->front,f_sten->newfront,NO) ; 
        */
}

/* 
 *  This function only takes care of the 
 *  cell interpolation inside the computational domain.
 *  If this cell contains bifurcated front,
 *  do interplation, otherwise, do not interplate.  
 * 
 */
LOCAL void get_state_in_cell2(
        float         *crds,
        int           *ic,
        COMPONENT     comp,
        Locstate      st,
        Wave          *wave,
        Wave          *newwave,
        Front         *fr)
{
        TRI_GRID      *otg = wave_tri_soln(wave)->tri_grid;
        TRI_GRID      *ntg = wave_tri_soln(newwave)->tri_grid;
        size_t        sizest = fr->sizest; 
        int           ic_end[MAXD]; 
        static CRXING   **crx[4] = {NULL,NULL,NULL,NULL};
        int             i, nc[4]; /* number of CRXING */
        COMPONENT     ccomp; 

        if(NULL == crx[0])
        {
            for(i = 0; i < 4; i++)
                 vector(&crx[i],4,sizeof(CRXING *));
        }

        ccomp = Rect_comp(ic,wave); 
        ic_end[0] = ic[0] + 1;
        ic_end[1] = ic[1] + 1;
        nc[0] = comp_crossings_in_direction(crx[0],ic,EAST,otg);
        nc[2] = comp_crossings_in_direction(crx[2],ic,NORTH,otg);
        nc[1] = comp_crossings_in_direction(crx[1],ic_end,WEST,otg);
        nc[3] = comp_crossings_in_direction(crx[3],ic_end,SOUTH,otg);

        // if(Comp_blk(ic,ntg->blk_type,otg) == F_NO_VOL ||
        //    Comp_blk(ic,ntg->blk_type,ntg) == F_NO_VOL)
        if(is_complex_blk(Comp_blk(ic,otg->blk_type,otg)) ||
           is_complex_blk(Comp_blk(ic,ntg->blk_type,ntg)))
        {
            hyp_solution(crds,comp, NULL,UNKNOWN_SIDE,fr,wave,st,st);
            return; 
        }

        if(nc[0] == 0 && nc[1] == 0 && nc[2] == 0 && nc[3] == 0)
        {
            if(ccomp == comp)
                assign(st, Rect_state(ic, wave), sizest);
        }
        else
        {
           /* What should be done here ?? */
        }
}

/* 
 *  This function only takes care of the 
 *  cell interpolation inside the computational domain.
 *  If this cell contains bifurcated front,
 *  do interplation, otherwise, do not interplate.  
 *  If the crds in some volume with same compoent,
 *  assign volume state to st. This function applies
 *  to state st[-1], st[2], and state in the other
 *  direction of the corresponding points are inside
 *  the domain.  
 */
LOCAL void get_state_in_cell3(
        float         *crds,
        int           *ic,
        COMPONENT     comp,
        Locstate      st,
        Wave          *wave,
        Wave          *newwave,
        Front         *fr)
{
        TRI_GRID      *otg = wave_tri_soln(wave)->tri_grid;
        TRI_GRID      *ntg = wave_tri_soln(newwave)->tri_grid;
        size_t        sizest = fr->sizest; 
        int           ic_end[MAXD]; 
        static CRXING   **crx[4] = {NULL,NULL,NULL,NULL};
        int             i, nc[4]; /* number of CRXING */
        CSG_Solid     *s;  
        COMPONENT     ccomp; 

        if(NULL == crx[0])
        {
            for(i = 0; i < 4; i++)
                 vector(&crx[i],4,sizeof(CRXING *));
        }

        ccomp = Rect_comp(ic,wave); 
        ic_end[0] = ic[0] + 1;
        ic_end[1] = ic[1] + 1;
        nc[0] = comp_crossings_in_direction(crx[0],ic,EAST,otg);
        nc[2] = comp_crossings_in_direction(crx[2],ic,NORTH,otg);
        nc[1] = comp_crossings_in_direction(crx[1],ic_end,WEST,otg);
        nc[3] = comp_crossings_in_direction(crx[3],ic_end,SOUTH,otg);

        // if(Comp_blk(ic,ntg->blk_type,otg) == F_NO_VOL ||
        //    Comp_blk(ic,ntg->blk_type,ntg) == F_NO_VOL)
        if(is_complex_blk(Comp_blk(ic,otg->blk_type,otg)) ||
           is_complex_blk(Comp_blk(ic,ntg->blk_type,ntg)))
        {
            hyp_solution(crds,comp, NULL,UNKNOWN_SIDE,fr,wave,st,st);
            return; 
        }

        if(nc[0] == 0 && nc[1] == 0 && nc[2] == 0 && nc[3] == 0)
        {
            if(ccomp == comp)
                assign(st, Rect_state(ic, wave), sizest);
        }
        else
        {   
             if((s = pt_on_btm_of_volume(crds, ic, wave)) != NULL)   
             {
                 if(s->comp == comp)
                 {
                     btm_state2d(st, s, wave, fr);  
                 }
             }
        }
}

LOCAL CSG_Solid *pt_on_btm_of_volume(
        float       *crds,
        int         *ic,
        Wave        *wave)
{
        TRI_GRID    *otg = wave_tri_soln(wave)->tri_grid;
        int         i, j;
        int         icrds[MAXD];
        CSG_Solid   *s;
        CSG_BLK_CRX *blk;
 
        for(i = ic[0]-3; i <= ic[0]+3; i++)
        {
            for(j = ic[1]-3; j <= ic[1]+3; j++)
            {
                icrds[0] = i; icrds[1] = j;
                if(icrds[0] < ggmin[0] || icrds[0] >= ggmax[0])
                    continue;
                if(icrds[1] < ggmin[1] || icrds[1] >= ggmax[1])
                    continue;
                if((blk = Comp_blk(icrds,otg->Volume.blk,otg)) == NULL)
                    continue;
                s = blk->s;
                while(s)
                {
                     if(YES == point_on_volume_btm(crds, s))
                     {
                         return s; 
                     }
                     s = s->nexts;
                }
            }
        }
        return NULL;  
}

LOCAL int point_on_volume_btm(
        float      *crds,
        CSG_Solid  *s)
{
        CSG_Face   *f;  
        int        where; 

        f = s->sfaces; 
        while(f) 
        {
            if(YES == is_top_or_bottom_face(f,&where))
            {
                if(where == 0)
                {
                    if(YES == point_in_face2d(f->floops, crds)) 
                        return YES; 
                }
            }
            f = f->nextf; 
        }

        return NO; 
}


LOCAL void g_sten_tangnt_st_for_inter_face(
        Face_Stencil  *f_sten,
        float         *fdir)  /* The normal direction */
{
        int           i, j;
        Wave         *wave = f_sten->wave;
        Wave         *nwave = f_sten->newwave;
        Front        *fr = f_sten->front;
        Front        *nfr = f_sten->newfront;
        TRI_GRID     *ntg = wave_tri_soln(f_sten->wave)->tri_grid;
        size_t        sizest = fr->sizest;
        float         Tfdir[3];  
      
        int           icrds[MAXD], icrds1[MAXD], ic1[MAXD], outic[MAXD];
        CSG_BLK_CRX   *blk;
        float         crds[3], p[3];
        int           outdomain, side, ddir, buffered;
        CRXING        *crxing;
        int           deb_far_st = NO, deb_far_st_t = NO;

        debug_print("ghyp", "Entered g_sten_tangnt_st_for_inter_face\n"); 

        /* tangential = ( n2,-n1) */
        Tfdir[0] = fdir[1];
        Tfdir[1] = -1.0*fdir[0];
        Tfdir[2] = 0.0;

        /* Set Tst[1] */
        point_with_rect(f_sten->Tcrds[1], &ntg->comp_grid,
                     outic,&outdomain,&ddir,&side,&buffered);
 
        if(outdomain == YES)
        {
            /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
            if((crxing = cross_physics_boundary(Tfdir, f_sten->wave, f_sten->front,
                  f_sten->icrds[1], f_sten->Ticrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->Tcrds[1],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->tan_st[1]);
            }
        }
        else
        {
            if((crxing = cross_physics_boundary(Tfdir, f_sten->wave, f_sten->front,
                  f_sten->icrds[1], f_sten->Ticrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->Tcrds[1],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->tan_st[1]);
            }
            else
            {
                get_state_in_cell3(f_sten->Tcrds[1],f_sten->Ticrds[1],f_sten->s->comp,
                 f_sten->tan_st[1], f_sten->wave, f_sten->newwave, f_sten->front);
            }
        }

        /* Set Tst[-1] */
        point_with_rect(f_sten->Tcrds[0], &ntg->comp_grid,
                     outic,&outdomain,&ddir,&side,&buffered);
 
        if(outdomain == YES)
        {
            /* SHOULD CHECK WHETHER THIS FACE IS ON THE PHYSICS BOUNDARY */
            if((crxing = cross_physics_boundary(Tfdir, f_sten->wave, f_sten->front,
                  f_sten->Ticrds[0], f_sten->icrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->Tcrds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->tan_st[-1]);
            }
        }
        else
        {
            if((crxing = cross_physics_boundary(Tfdir, f_sten->wave, f_sten->front,
                  f_sten->Ticrds[0], f_sten->icrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->Tcrds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->tan_st[-1]);
            }
            else
            {
                get_state_in_cell3(f_sten->Tcrds[0],f_sten->Ticrds[0],f_sten->s->comp,
                 f_sten->tan_st[-1], f_sten->wave, f_sten->newwave, f_sten->front);
            }
        }

        /* TO SET THE OUTTER SIDE TANGENTIAL STATE IS MORE COMPLICATED, 
         * SINCE THE OUTTER COORDS MIGHT BE ALREADY OUTSIDE THE COMPUTATIONAL DOMAIN. 
         */

        /* Set outter_Tst[1] */
        point_with_rect(f_sten->oTcrds[1], &ntg->comp_grid,
                     outic,&outdomain,&ddir,&side,&buffered);

        if(outdomain == YES)
        {
            /* Should check if there is boundary between Tst[1] and out_Tst[1] */

            if((crxing = cross_physics_boundary(fdir, f_sten->wave, f_sten->front,
                  f_sten->Ticrds[1], f_sten->oTicrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->oTcrds[1],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->outtan_st[1]);
            }
        }
        else
        {
            /* Should do double checks, first check Tst[1] and out_Tst[1],
             * then check out_Tst[0] and out_Tst[1] */

            if((crxing = cross_physics_boundary(fdir, f_sten->wave, f_sten->front,
                  f_sten->Ticrds[1], f_sten->oTicrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->oTcrds[1],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->outtan_st[1]);
            }
            else if((crxing = cross_physics_boundary(Tfdir, f_sten->wave, f_sten->front,
                  f_sten->icrds[2], f_sten->oTicrds[1])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->oTcrds[1],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->outtan_st[1]);
            }
            else
            {
                get_state_in_cell3(f_sten->oTcrds[1],f_sten->oTicrds[1],f_sten->s->comp,
                 f_sten->outtan_st[1], f_sten->wave, f_sten->newwave, f_sten->front);
            }
        }

        /* Set outter_Tst[-1] */
        point_with_rect(f_sten->oTcrds[0], &ntg->comp_grid,
                     outic,&outdomain,&ddir,&side,&buffered);

        if(outdomain == YES)
        {
            /* Should check if there is boundary between Tst[1] and out_Tst[1] */

            if((crxing = cross_physics_boundary(fdir, f_sten->wave, f_sten->front,
                  f_sten->Ticrds[0], f_sten->oTicrds[0])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->oTcrds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->outtan_st[-1]);
            }
        }
        else
        {
            /* Should do double checks, first check Tst[-1] and out_Tst[-1],
             * then check out_Tst[-1] and out_Tst[0] */
            if((crxing = cross_physics_boundary(fdir, f_sten->wave, f_sten->front,
                  f_sten->Ticrds[0], f_sten->oTicrds[0])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->oTcrds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->outtan_st[-1]);
            }
            else if((crxing = cross_physics_boundary(Tfdir, f_sten->wave, f_sten->front,
                  f_sten->oTicrds[0], f_sten->icrds[2])) != NULL)
            {
                set_cross_physics_boundary_st(f_sten->oTcrds[0],f_sten->s->comp,
                      crxing, f_sten->front, f_sten->wave, f_sten->outtan_st[-1]);
            }
            else
            {
                get_state_in_cell3(f_sten->oTcrds[0],f_sten->oTicrds[0],f_sten->s->comp,
                 f_sten->outtan_st[-1], f_sten->wave, f_sten->newwave, f_sten->front);
            }
        }


        debug_print("ghyp", "Left g_sten_tangnt_st_for_inter_face\n"); 
}

LOCAL void ipoint_with_rect(
        int           *icrds, 
        int           *outdomain,
        int           *ddir,
        int           *side,
        int           *buffered)
{
        *outdomain = NO; 
        *buffered = 100; 

        if(icrds[0] < ggmin[0])
        {
            *outdomain = YES;
            *ddir = 0; *side = 0;
            *buffered = lbuffed[0];
        }
        if(icrds[1] < ggmin[1])
        {
            *outdomain = YES;
            *ddir = 1; *side = 0;
            *buffered = lbuffed[1];
        }
        if(icrds[0] >= ggmax[0])
        {
            *outdomain = YES;
            *ddir = 0; *side = 1;
            *buffered = ubuffed[0];
        }
        if(icrds[1] >= ggmax[1])
        {
            *outdomain = YES;
            *ddir = 1; *side = 1;
            *buffered = ubuffed[1];
        }

        return; 
}

LOCAL void point_with_rect(
        float         *crds,
        RECT_GRID     *rect,
        int           *icrds, 
        int           *outdomain,
        int           *ddir,
        int           *side,
        int           *buffered)
{
        if(rect_in_which(crds,icrds, rect) == FUNCTION_FAILED)
        {
            if(icrds[0] <= ggmin[0])
            {
                *ddir = 0; *side = 0;
                *buffered = lbuffed[0];
            }
            if(icrds[1] <= ggmin[1])
            {
                *ddir = 1; *side = 0;
                *buffered = lbuffed[1];
            }
            if(icrds[0] >= ggmax[0])
            {
                *ddir = 0; *side = 1;
                *buffered = ubuffed[0];
            }
            if(icrds[1] >= ggmax[1])
            {
                *ddir = 1; *side = 1;
                *buffered = ubuffed[1];
            }
            *outdomain = YES;
            return; 
        }         

        *buffered = 100; 
        *outdomain = NO; 
        return; 
}

LOCAL void off_face_pt(
        CSG_Face      *f,
        float         *crds,
        float         factor)
{
        int   i; 
        for(i = 0; i < 2; i++)
            crds[i] = f->centroid[i];
        crds[0] = crds[0]+f->face_eqn[0]*dh[0]*factor;
        crds[1] = crds[1]+f->face_eqn[1]*dh[1]*factor;
}

LOCAL CSG_Face *large_face_in_dir(
        CSG_Solid     *s,
        float         *nord)
{
        float         area = 0.0, dot_product;
        CSG_Face      *f, *opp_f = NULL;
        int           i;

        f = s->sfaces;
        while(f)
        {
            dot_product = scalar_product(nord,f->face_eqn,3);
            if(fabs(dot_product-1.0) < 0.0001*TOL)
            {
                if(f->area >= area)
                {
                    opp_f = f;
                    area = f->area;
                }
            }
            f = f->nextf;
        }
        return opp_f;
}


/* If the outter neighbor of this face is outside the
 * computational domain, outdomain is YES.
 */
LOCAL CSG_Solid *find_out_nghbr(
        CSG_Face      *f,
        int           *ic,
        Wave          *wave,
        Front         *fr,
        Wave          *newwv,
        Front         *newfr, 
        int           *outic,
        int           *outdomain,
        int           *dir,
        int           *side)
{
        int           icrds[MAXD], ic1[MAXD];
        int           i, j;
        CSG_BLK_CRX   *blk;
        CSG_Face      *match_f = NULL;
        CSG_Solid     *s; 
        CSG_POINT     *pt;
        CSG_HalfEdge  *lh;
        int           found = NO, buffered;
        float         crds[3], p[3];
        TRI_GRID      *ntg = wave_tri_soln(wave)->tri_grid;
        TRI_GRID      *nntg = wave_tri_soln(newwv)->tri_grid;

        *outdomain = NO; 
        if(YES == face_on_boundary(f, &buffered, dir, side))
        {
            *outdomain = YES;
            return NULL;  
        }

        for(i = 0; i < 2; i++)
            ic1[i] = ic[i]; 

        lh = f->floops->ledg; 
        do
        {
            if(lh->vtx->pt->type == IS_ICRDS)
            {
                ic1[0] = lh->vtx->pt->csg_pt.icrds[0];
                ic1[1] = lh->vtx->pt->csg_pt.icrds[1];
                break;
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        match_f = find_match_face_in_region4(f,ic1,ntg); 
        if(match_f != NULL)
        {
            if(f->fsolid->comp != match_f->fsolid->comp)
            {
                printf("ERROR  find_out_nghbr()\n");
                printf("face %d of solid %d eqn<%f, %f, %f>\n",
                         f->faceno, f->fsolid->solidno,
                   f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                printf("match face %d of solid %d eqn<%f, %f, %f>\n",
                   match_f->faceno, match_f->fsolid->solidno,
                   match_f->face_eqn[0], match_f->face_eqn[1],
                   match_f->face_eqn[2]);
                printf("f, match_f Solid comp<%d, %d>\n",
                    f->fsolid->comp , match_f->fsolid->comp);
                printf("Print f->fsolid\n");
                solidls(f->fsolid,2);
                printf("Print match_f->fsolid\n");
                solidls(match_f->fsolid,2);
                clean_up(ERROR);
            }
            return match_f->fsolid; 
        }

        /* in the case there is no outter nghbr, use cell center states */
        /* Or might need to call the interpolation function if the volume 
         * is not constructed.
         */
 
         /* Should replace this point with centroid ??? */
         if(is_interior_face(f)) 
         {
            for(i = 0; i < 2; i++)
                crds[i] = f->centroid[i]; 
            crds[0] = crds[0]+f->face_eqn[0]*dh[0]/2;
            crds[1] = crds[1]+f->face_eqn[1]*dh[1]/2;
            if(rect_in_which(crds,icrds, &ntg->comp_grid) == FUNCTION_FAILED)
            {
                printf("WARNING in find_out_nghbr(), "
                       "rect_in_which() failed\n");
                printf("crds[%g, %g], icrds[%d,%d]\n", crds[0], crds[1], 
                         icrds[0], icrds[1]);  
                facels(f, 2);
                solidls(f->fsolid,2);
                clean_up(ERROR);  
            }

            if(icrds[0] < ggmin[0] || icrds[0] >= ggmax[0])
            {
                printf("ERROR in find_out_nghbr(), "
                       "icrds[0] < ggmin[0] <%d, %d>\n",icrds[0],icrds[1]);
                clean_up(ERROR);

            }
            if(icrds[1] < ggmin[1] || icrds[1] >= ggmax[1])
            {
                printf("ERROR in find_out_nghbr(), "
                       "icrds[1] < ggmin[1] <%d, %d>\n",icrds[0],icrds[1]);
                clean_up(ERROR);
            }
            if((blk = Comp_blk(icrds,ntg->Volume.blk,ntg)) == NULL)
            {
                // if(Comp_blk(icrds,ntg->blk_type,ntg) == F_NO_VOL) 
                if(is_complex_blk(Comp_blk(icrds,ntg->blk_type,ntg)) ||
                   is_complex_blk(Comp_blk(icrds,nntg->blk_type,nntg)) ) 
                {
                    outic[0] = icrds[0];
                    outic[1] = icrds[1];
                    return NULL;
                }
                if(f->fsolid->comp != Regular_grid_comp(icrds,ntg))
                {
                    printf("ERROR in find_out_nghbr(), \n");
                    printf("Face has no outter neighbr, f->comp[%d] != grid_comp[%d]\n",
                       f->fsolid->comp, Regular_grid_comp(icrds,ntg));
                    printf("CEll icoords[%d,%d] coords[%g %g]\n",
                       icrds[0], icrds[1], crds[0], crds[1]);
                    print_vector("Comp G_Node Coords:", dim, 
                           Rect_compute_coords(icrds,wave), "%g ");  
                    print_vector("Cell center Coords:", dim, 
                           Rect_coords(icrds,wave), "%g ");  
                    printf("This cell type %d\n", Comp_blk(icrds,ntg->blk_type,ntg)); 
                    printf("type %d\n", Comp_blk(icrds,ntg->blk_type,ntg)); 
                    facels(f, 2); 
                    solidls(f->fsolid,2);
                    clean_up(ERROR);  
                }
                outic[0] = icrds[0];
                outic[1] = icrds[1];
                return NULL;
            }
         }


        (void)printf("ERROR In find_out_nghbr()\n");
        (void)printf("Face does not find outter\n");
        (void)printf("face-eqn<%f, %f, %f> ic<%d, %d> \n",
                f->face_eqn[0], f->face_eqn[1], f->face_eqn[2],
                ic[0], ic[1]);
        facels(f, 2);
        solidls(f->fsolid,2);
        printf("\n\n Print out surrand blocks\n");
        for(i = ic1[0]-3; i < ic1[0]+3; i++)
        {
            for(j = ic1[1]-3; j < ic1[1]+3; j++)
            {
                icrds[0] = i; icrds[1] = j;
                if(icrds[0] < ggmin[0] || icrds[0] >= ggmax[0])
                    continue;
                if(icrds[1] < ggmin[1] || icrds[1] >= ggmax[1])
                    continue;
                if((blk = Comp_blk(icrds,ntg->Volume.blk,ntg)) == NULL)
                    continue;
                print_solid(blk);
            }
        }
        clean_up(ERROR);
}

/**** Replace find_match_face_in_region3() with
 **** find_match_face_in_region4(). 
 find_match_face_in_region3() only 
 allows interior side face
 to find its match 
LOCAL CSG_Face *find_match_face_in_region3(
        CSG_Face    *f,
        int         *ic,
        TRI_GRID    *ntg)
{
        int         i, j; 
        CSG_Face    *match_f; 
        int         icrds[MAXD]; 
        CSG_Solid   *s;
        CSG_BLK_CRX   *blk;

        for(i = ic[0]-3; i <= ic[0]+3; i++)
        {
            for(j = ic[1]-3; j <= ic[1]+3; j++)
            {
                icrds[0] = i; icrds[1] = j;
                if(icrds[0] < ggmin[0] || icrds[0] >= ggmax[0])
                    continue;
                if(icrds[1] < ggmin[1] || icrds[1] >= ggmax[1])
                    continue;
                if((blk = Comp_blk(icrds,ntg->Volume.blk,ntg)) == NULL)
                    continue;
                s = blk->s;
                while(s)
                {
                    if(s == f->fsolid)
                    {
                        s = s->nexts;
                        continue;
                    }
                    match_f = match_face_in_other(f, s);
                    if(match_f == NULL)
                        s = s->nexts;
                    else
                    {
                        if(f->fsolid->comp != match_f->fsolid->comp)
                        {
                            printf("ERROR  find_match_face_in_region3()\n");
                            printf("face %d of solid %d eqn<%f, %f, %f>\n",
                                    f->faceno, f->fsolid->solidno,
                                    f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                            printf("match face %d of solid %d eqn<%f, %f, %f>\n",
                             match_f->faceno, match_f->fsolid->solidno,
                             match_f->face_eqn[0], match_f->face_eqn[1],
                             match_f->face_eqn[2]);
                            printf("f, match_f Solid comp<%d, %d>\n",
                                    f->fsolid->comp , match_f->fsolid->comp);
                            printf("Print f->fsolid\n");
                            solidls(f->fsolid,2);
                            printf("Print match_f->fsolid\n");
                            solidls(match_f->fsolid,2);
                            clean_up(ERROR);
                        }
                        return  match_f;
                    }
                 }
             }  // End of for loop 
         }

         return NULL;  
}
****/

/* Replaced with is_interior_face()
// The front face must only be composed of crxing points 
LOCAL int is_inter_face(
       CSG_Face    *f)
{
        CSG_HalfEdge  *lh;
        int           i;
        float         *dir, prod;
        int           all_crx = YES; 

        // for( i = 4; i < 6; i++)
        // {
        //    dir = nor[i];  
        //    prod = scalar_product(dir,f->face_eqn,3);  
        //    if(fabs(prod - 1.0) <  0.0001*TOL)
        //        return NO; 
        //    if(fabs(prod + 1.0) <  0.0001*TOL)
        //        return NO; 
        // }
        // if(f->faceno == 109)
        // {
        //     lh = f->floops->ledg;
        //     do
        //     {
        //         if(lh->vtx->pt->type != IS_ICRDS)
        //         {
        //             print_hypersurface(lh->vtx->pt->csg_pt.crx->hs); 
        //         }
        //     }while((lh = lh->nxt) != f->floops->ledg);
        // }

        lh = f->floops->ledg;
        do
        {
            if(lh->vtx->pt->type == IS_ICRDS)
            {
                all_crx = NO; 
                break; 
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        if(all_crx == YES)
            return NO;
        else
            return YES; 
}
*/

/* This function also checks the side
 * face type(whether it is a boundary front face).
 * For the interior front face, it returns no.  
 */
LOCAL int face_on_boundary(
       CSG_Face    *f,
       int         *buffed,
       int         *dir,
       int         *side)
{

        CSG_POINT     *pt;
        CSG_HalfEdge  *lh;
        int           on_buf = NO;
        int           all_crx = YES; 
        int           i;
        float         *fdir, prod;

        /* The top and bottom face returns NO */
        for( i = 4; i < 6; i++)
        {
           fdir = nor[i];
           prod = scalar_product(fdir,f->face_eqn,3);
           if(fabs(prod - 1.0) <  0.0001*TOL ||
              fabs(prod + 1.0) <  0.0001*TOL)
               return NO;
        }

        lh = f->floops->ledg;
        do
        {
            if(lh->vtx->pt->type == IS_ICRDS)
            {
                all_crx = NO;
                break;
            }
        }while((lh = lh->nxt) != f->floops->ledg);

        /* Front face */
        if(all_crx == YES)
        {
            lh = f->floops->ledg;
            do
            {
                /* only the interior front face can have time-line
                 * crxings */
                pt = lh->vtx->pt; 
                if(pt->csg_pt.crx->hs == NULL)
                    return NO; 
                if(wave_type(pt->csg_pt.crx->hs) 
                   >= FIRST_PHYSICS_WAVE_TYPE)
                    return NO; 
            }while((lh = lh->nxt) != f->floops->ledg);
            /* The face is a boundary front face */
            *buffed = NO; 
            return YES; 
        }

        /* Interior side face */
        for( i = 0; i < 4; i++)
        {
           fdir = nor[i];
           prod = scalar_product(fdir,f->face_eqn,3);
           if(fabs(prod - 1.0) <  0.0001*TOL)
               break;  
        }

        lh = f->floops->ledg;
        do
        {
            pt = lh->vtx->pt;
            if(pt->type == IS_ICRDS)
                break; 
        }while((lh = lh->nxt) != f->floops->ledg);

        switch(i)
        {
        case 0:
            if(pt->csg_pt.icrds[0] == ggmax[0])
            {
                *dir = 0; 
                *side = 1; 
                *buffed = ubuffed[0]; 
                return YES; 
            }     
            return NO; 
        break; 
        case 1:
            if(pt->csg_pt.icrds[1] == ggmax[1])
            {
                *dir = 1; 
                *side = 1; 
                *buffed = ubuffed[1]; 
                return YES; 
            }
            return NO; 
        break;
        case 2:
            if(pt->csg_pt.icrds[0] == ggmin[0])
            {
                *dir = 0; 
                *side = 0; 
                *buffed = lbuffed[0]; 
                return YES; 
            }
            return NO; 
        break;
        case 3:
            if(pt->csg_pt.icrds[1] == ggmin[1])
            {
                *dir = 1; 
                *side = 0; 
                *buffed = lbuffed[1]; 
                return YES; 
            }
            return NO; 
        break; 
        }

        printf("ERROR: face_on_boundary\n");
        printf("Unknow face case\n");
        facels(f, 2);
        solidls(f->fsolid, 2);
        clean_up(ERROR);  
}

LOCAL int count_faces_on_this_side(
       CSG_Solid   *s,
       float       *dir)
{
       int         i = 0;
       CSG_Face    *f;
       float       prod; 
     
       f = s->sfaces; 
       while(f)
       {
           if(! is_interior_face(f))
           {
               f = f->nextf; 
               continue;  
           }
           prod = scalar_product(dir,f->face_eqn,3);  
           if(fabs(prod - 1.0) <  0.0001*TOL)
               i++; 
           f = f->nextf; 
       }
       return i; 
}

LOCAL void collect_sten_faces_on_side(
       CSG_Solid    *s,
       float        *dir,
       int          nface,
       Face_Stencil *f_sten)
{
       CSG_Face    *f;
       float       prod, centroid[3]; 
       int         i,j; 
       float       area = 0.0;
       
  
       for(i = 0; i < 3; i++)
           f_sten->centroid[i] = 0.0;  
       f_sten->nface = nface;   
       f_sten->area = 0.0;  
       if(f_sten->f == NULL || f_sten->alloc_nface < nface)
       { 
           if(f_sten->f != NULL) 
               free(f_sten->f); 
           vector(&(f_sten->f), nface, sizeof(CSG_Face*));  
           f_sten->alloc_nface = nface;  
       }
       
       i = 0; 
       f = s->sfaces; 
       while(f)
       {
           if(! is_interior_face(f))
           {
               f = f->nextf; 
               continue;  
           }
           prod = scalar_product(dir,f->face_eqn,3);  
           if(fabs(prod - 1.0) <  0.0001*TOL)
           {
               f_sten->f[i] = f; 
               f_sten->area += f->area; 
               for(j = 0; j < 3; j++)
                   f_sten->centroid[j] += f->centroid[j]*f->area; 
               if(f->area > area)
               {
                   f_sten->maxf = f;
                   area = f->area;  
               }
               i++; 
           }
           f = f->nextf; 
       }

       if(i != 0)
       {
           for(j = 0; j < 3; j++)
               f_sten->centroid[j] /= f_sten->area; 
       }
}

LOCAL void btm_state2d(
        Locstate        btm_st,
        CSG_Solid       *s,
        Wave            *wv,
        Front           *fr)
{
        int             where[3],i;
        CSG_Face        *f;
        float           ucon[4] = {0.0, 0.0, 0.0, 0.0};
        float           area = 0.0;
        int             first = YES;
        Locstate        st;
        f = s->sfaces;
        i = 0;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 0)
                {
                    i++;
                    ucon[0] += Dens(f->i_stat)*f->area;
                    ucon[1] += Mom(f->i_stat)[0]*f->area;
                    ucon[2] += Mom(f->i_stat)[1]*f->area;
                    ucon[3] += Energy(f->i_stat)*f->area;
                    if(first)
                    {
                        st = f->i_stat;
                        first = NO;
                    }
                    area += f->area;
                    if (is_obstacle_state(f->i_stat) && 
                        (!is_excluded_comp(f->fsolid->comp,fr->interf)))
                    {
                        hyp_solution(f->centroid, f->fsolid->comp, 
                          NULL,UNKNOWN_SIDE,fr,wv,f->i_stat,f->i_stat);
                    }
                    /* Check again */
                    if (is_obstacle_state(f->i_stat) && 
                        (!is_excluded_comp(f->fsolid->comp,fr->interf)))
                    {
                         printf("ERROR in btm_state2d(), NO excluded comp volume\n");
                         printf("face %d is OBSTACLE_STATE %d\n",
                              f->faceno, f->i_stat);
                         (void)printf("face-eqn<%f, %f, %f> \n",
                                f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                         facels(f, 2);
                         printf("blk id<%d, %d>\n",
                          s->blk->icrds[0][0][0].csg_pt.icrds[0],
                          s->blk->icrds[0][0][0].csg_pt.icrds[1]);
                         solidls(f->fsolid,2);
                         clean_up(ERROR);
                    }
                    if(Dens(f->i_stat) < 0.0)
                    {
                        printf("ERROR: in  btm_state2d() LOW Dens found\n");
                        facels(f,2);
                        solidls(f->fsolid,2);
                        g_verbose_print_state(f->i_stat);
                        verbose_print_btm_state2d(s);
                        clean_up(ERROR);
                    }
                }
            }
            f = f->nextf;
        }

        /* there might be the volume with trivial bottom face */
        if(i == 0)
        {
            printf("ERROR: In btm_state2d(), non btm vol found\n");
            solidls(s, 2);
            clean_up(ERROR);
        }
        if(i != 0)
        {
            ucon[0] = ucon[0]/area;
            ucon[1] = ucon[1]/area;
            ucon[2] = ucon[2]/area;
            ucon[3] = ucon[3]/area;
        }
        Dens(btm_st) = ucon[0];
        Mom(btm_st)[0] = ucon[1];
        Mom(btm_st)[1] = ucon[2];
        Energy(btm_st) = ucon[3];

        Set_params(btm_st, st);
        set_type_of_state(btm_st,GAS_STATE);
        if(i != s->n_btm)
        {
            printf("ERROR btm_state2d()\n"); 
            printf("ERROR number of btm face from i = %d, s->n_btm = %d\n",
                    i, s->n_btm);
            g_verbose_print_state(btm_st);
            solidls(s, 2);
            clean_up(ERROR);  
        }
}

LOCAL   void    verbose_print_btm_state2d(
        CSG_Solid       *s)
{
        int             where[3],i;
        CSG_Face        *f;
        float           ucon[4] = {0.0, 0.0, 0.0, 0.0};
        float           area = 0.0;
        Locstate        st;

        debug_print("ghyp", "Entered verbose_print_btm_state2d()\n");

        printf("Print btm states of solid[%d]\n",s->solidno);
        f = s->sfaces;
        i = 0;
        while(f)
        {
            if(YES == is_top_or_bottom_face(f,where))
            {
                if(where[0] == 0)
                {
                    i++;
                    ucon[0] += Dens(f->i_stat)*f->area;
                    ucon[1] += Energy(f->i_stat)*f->area;
                    ucon[2] += Mom(f->i_stat)[0]*f->area;
                    ucon[3] += Mom(f->i_stat)[1]*f->area;
                    st = f->i_stat;
                    area += f->area;
                    if (is_obstacle_state(f->i_stat))
                    {
                         printf("ERROR in verbose_print_btm_state2d()\n");
                         printf("face %d is OBSTACLE_STATE\n", f);
                         (void)printf("face-eqn<%f, %f, %f> \n",
                                f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                         facels(f, 2);
                         solidls(f->fsolid,2);
                         clean_up(ERROR);
                    }
                    g_print_state(f->i_stat);
                    printf("face<%d>'s areas = %e ", f->faceno, f->area);
                    (void)printf("face-eqn<%f, %f, %f> \n",
                                f->face_eqn[0], f->face_eqn[1], f->face_eqn[2]);
                }
            }
            f = f->nextf;
        }

        /* there might be the volume with trivial bottom face */
        if(i == 0 || i != s->n_btm)
        {
            printf("ERROR verbose_print_btm_state2d()\n");
            printf("i = 0\n");
            solidls(s,2);
            clean_up(ERROR);  
        }
        if(i != 0)
        {
            ucon[0] = ucon[0]/area;
            ucon[1] = ucon[1]/area;
            ucon[2] = ucon[2]/area;
            ucon[3] = ucon[3]/area;
        }

        printf("ucon<rho, E, mom> <%15.13f, %17.13f, %15.13f, %15.13f>\n",
               ucon[0], ucon[1], ucon[2], ucon[3]);
        printf("End of print solid[%d] btm state\n", s->solidno); 
        debug_print("ghyp", "Left verbose_print_btm_state2d()\n");
}

LOCAL void print_face_sten(
       Face_Stencil *f_sten)
{
       int i, j;

       printf("Face centroid[%g %g %g], normal[%g %g %g] maxfcenter[%g %g]\n", 
            f_sten->centroid[0], f_sten->centroid[1], f_sten->centroid[2],
            f_sten->maxf->face_eqn[0], f_sten->maxf->face_eqn[1], 
            f_sten->maxf->face_eqn[2], f_sten->maxf->centroid[0],
            f_sten->maxf->centroid[1]); 

       for(j = 0; j < 4; j++)
       {
           if(j %2 == 0 && j != 0)
               printf("\n"); 
           printf("Cell[%d] crds:", j);
           printf("[%g %g] ", f_sten->crds[j][0], f_sten->crds[j][1]);
       }       
       printf("\n"); 
       for(j = 0; j < 4; j++)
       {
           if(j %2 == 0 && j != 0)
               printf("\n"); 
           printf("Cell[%d] icrds:", j);
           printf("[%d %d] ", f_sten->icrds[j][0], f_sten->icrds[j][1]);
       }       
       printf("\n"); 

       printf("Inner side tangential:\n");   
       for(j = 0; j < 2; j++)
       {
           printf("Tang Cell[%d] crds:", j);
           printf("[%g %g]", f_sten->Tcrds[j][0], f_sten->Tcrds[j][1]);
       }
       printf("\n");
       for(j = 0; j < 2; j++)
       {
           printf("Tang Cell[%d] icrds:", j);
           printf("[%d %d]", f_sten->Ticrds[j][0], f_sten->Ticrds[j][1]);
       }
       printf("\n");

       printf("Outter side tangential:\n");   
       for(j = 0; j < 2; j++)
       {
           printf("outTang Cell[%d] crds:", j);
           printf("[%g %g]", f_sten->oTcrds[j][0], f_sten->oTcrds[j][1]);
       }
       printf("\n");
       for(j = 0; j < 2; j++)
       {
           printf("outTang Cell[%d] icrds:", j);
           printf("[%d %d]", f_sten->oTicrds[j][0], f_sten->oTicrds[j][1]);
       }
       printf("\n");

       // facels(f_sten->maxf, 2);
       // solidls(f_sten->s, 2);
}

LOCAL void print_face_sten_state(
       Face_Stencil   *f_sten)   
{
       int  j;
       
       printf("Face sten state:\n"); 
       for(j = -1; j <=2; j++)
       {
           printf("Normal pt[%d] stat:", j);
           g_print_state(f_sten->st[j]); 
       }

       for(j = -1; j <=1; j+=2)
       {
           printf("Inner tangent pt[%d] stat:", j);
           g_print_state(f_sten->tan_st[j]); 
       }

       for(j = -1; j <=1; j+=2)
       {
           printf("Outter tangent pt[%d] stat:", j);
           g_print_state(f_sten->outtan_st[j]); 
       }
}

LOCAL void g_add_contrl_vol_art_viscsity(
        Face_Stencil    *f_sten,
        CSG_Face        *f,
        float           t_area,
        float           *fdir,
        float           *fflux,
        float           Jac)
{
        if(f->art_visc_st_x != NULL)
        {
            fflux[0] -= (f->area/t_area)*Dens(f->art_visc_st_x)*fdir[0]*Jac;
            fflux[1] -= (f->area/t_area)*Mom(f->art_visc_st_x)[0]*fdir[0]*Jac;
            fflux[2] -= (f->area/t_area)*Mom(f->art_visc_st_x)[1]*fdir[0]*Jac;
            fflux[3] -= (f->area/t_area)*Energy(f->art_visc_st_x)*fdir[0]*Jac;
        }
        if(f->art_visc_st_y != NULL)
        {
            fflux[0] -= (f->area/t_area)*Dens(f->art_visc_st_y)*fdir[1]*Jac;
            fflux[1] -= (f->area/t_area)*Mom(f->art_visc_st_y)[0]*fdir[1]*Jac;
            fflux[2] -= (f->area/t_area)*Mom(f->art_visc_st_y)[1]*fdir[1]*Jac;
            fflux[3] -= (f->area/t_area)*Energy(f->art_visc_st_y)*fdir[1]*Jac;
        }
}

LOCAL void g_comp_contrl_vol_art_visc2(
        Vec_Gas         *vst,
        Vec_Muscl       *vmuscl,
        Locstate        art_visc_st)
{
        Locstate        s1,s2;
        float           g0 = vmuscl->avisc->g[0][1]*0.5;

        s1 = vst->state[1];
        s2 = vst->state[2];

        Dens(art_visc_st) = g0*(Dens(s2) - Dens(s1));
        Energy(art_visc_st) = g0*(Energy(s2) - Energy(s1));
        Mom(art_visc_st)[0] = g0*(Mom(s2)[0] - Mom(s1)[0]);
        Mom(art_visc_st)[1] = g0*(Mom(s2)[1] - Mom(s1)[1]);
        Set_params(art_visc_st,vst->state[1]);
        set_type_of_state(art_visc_st,GAS_STATE);

        /*** 
        if (iperm[swp_num] == 0)
            f->art_visc_st_x = art_visc_st;
        else
            f->art_visc_st_y = art_visc_st;
        ***/
}

#endif /* if defined(CONSERVATIVE_ALG) && defined(TWOD) */

