/*
*				ghyperbolic.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains g_tri_integral() and g_quad_integral, set to funciton
*		pointers in the wave structure.
*
*	Contains routines used in the hyperbolic step.
*
*/

#include <ghyp/ghyp.h>

	/* LOCAL Function Declarations */
LOCAL	bool	too_many_bad_gas_states(const char*,int,Front*);


/*ARGSUSED*/
EXPORT	void point_FD(
	float		dh,
	float		dt,
	Locstate	ans,
	const float	*dir,
	int		swp_num,
	int		*iperm,
	int		*index,
	Stencil		*sten)
{
	Front		*fr = sten->fr;
	Front		*newfr = sten->newfr;
	RECT_GRID	*gr = fr->rect_grid;
	Locstate	st, *state;
	Wave 		*wave = sten->wave;
	Wave 		*newwave = sten->newwave;
	float		*rho, *en_den;
	float		*m[MAXD];
	float		**coords;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	float		*vacuum_dens;
	float		*min_pressure;
	float		*min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
#if defined(ROTATIONAL_SYMMETRY)
	static float	alpha;
#endif /* defined(ROTATIONAL_SYMMETRY) */
	int		i, j;
	int		dim = gr->dim;
	int		idirs[MAXD];
	static Vec_Gas	*vst = NULL;
	static Vec_Src  *src = NULL;
	static int	endpt, npts;
	static float    **Q = NULL;

	if (is_obstacle_state(sten->st[0])) 
	{
	    g_obstacle_state(ans,sten->fr->sizest);
	    return;
	}
	if (vst == NULL) 
	{
	    /* Assumes stencil size never changes */

	    npts = sten->npts;
	    endpt = npts/2;
            alloc_phys_vecs(wave,npts);
	    vst = g_wave_vgas(wave);
	    src = g_wave_vsrc(wave);
	    g_wave_vgas(wave) = NULL;
	    g_wave_vsrc(wave) = NULL;
	    matrix(&Q,3,3,FLOAT);
	    vst->Q = (const float* const*)Q;
#if defined(ROTATIONAL_SYMMETRY)
	    alpha = rotational_symmetry();
#endif /* defined(ROTATIONAL_SYMMETRY) */
	}

	if (RegionIsFlowSpecified(ans,sten->st[0],Coords(sten->p[0]),
				  sten->newcomp,sten->newcomp,fr))
	    return;

	clear_Vec_Gas_set_flags(vst);
	for (i = 0; i < 3; ++i)
	    for (j = 0; j < 3; ++j)
	        Q[i][j] = 0.0;
	for (i = 0; i < dim; ++i)
	{
	    idirs[i] = iperm[(i+swp_num)%dim];
	    m[i] = vst->m[i];
	    Q[i][idirs[i]] = 1.0;
	}
	for (; i < 3; ++i)
	    Q[i][i] = 1.0;
	rho = vst->rho;
	state = vst->state;
	coords = vst->coords;
	en_den = vst->en_den;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	vacuum_dens = vst->vacuum_dens;
	min_pressure = vst->min_pressure;
	min_energy = vst->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	for (i = 0; i < npts; ++i)
	{
	    state[i] = st = sten->ststore[i];
	    coords[i] = Coords(sten->pstore[i]);
	    rho[i] = Dens(st);
	    en_den[i] = Energy(st);
	    for (j = 0; j < dim; ++j)
	    	m[j][i] = Mom(st)[idirs[j]];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	    vacuum_dens[i] = Vacuum_dens(st);
	    min_pressure[i] = Min_pressure(st);
	    min_energy[i] = Min_energy(st);
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	}
	Vec_Gas_field_set(vst,state) = YES;
	Vec_Gas_field_set(vst,coords) = YES;
	Vec_Gas_field_set(vst,rho) = YES;
	Vec_Gas_field_set(vst,en_den) = YES;
	Vec_Gas_field_set(vst,m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	Vec_Gas_field_set(vst,vacuum_dens) = YES;
	Vec_Gas_field_set(vst,min_pressure) = YES;
	Vec_Gas_field_set(vst,min_energy) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	set_params_jumps(vst,0,npts);

#if defined(ROTATIONAL_SYMMETRY)
	/* If using cylindrical coordinates and this is the
	 *  r-sweep include radius information for source
	 *  computation.  
	 */

	if (alpha > 0.0 && iperm[swp_num]==0)   
	{
	    float *radii = src->radii;

	    src->rmin = fabs(pos_radius(0.0,gr));
	    for (i = 0; i < npts; ++i)
	    	radii[i] = pos_radius(Coords(sten->p[0])[0]+(i-endpt)*dh,gr);
	}
#endif /* defined(ROTATIONAL_SYMMETRY) */

	oned_interior_scheme(swp_num,iperm,sten->icoords[0],wave,newwave,
		             fr,newfr,sten,0,npts,vst,src,dt,dh,dim);

	Dens(ans) = vst->rho[endpt];
	Energy(ans) = vst->en_den[endpt];
	for (i = 0; i < dim; ++i)
	    Mom(ans)[idirs[i]] = vst->m[i][endpt];
	Set_params(ans,vst->state[endpt]);
	set_type_of_state(ans,GAS_STATE);
#if defined(CHECK_FOR_BAD_STATES)
	if (debugging("bad_state") && is_bad_state(ans,YES,"point_FD"))
	{
	    screen("ERROR in point_FD(), bad state found\n");
	    fprint_raw_gas_data(stdout,ans,dim);
	    for (i = 0; i < npts; ++i)
	    {
		(void) printf("state[%d]\n",i);
	        fprint_raw_gas_data(stdout,state[i],dim);
	    }
	    for (i = 0; i < npts; ++i)
	    {
		(void) printf("state[%d]",i);
	        verbose_print_state("",state[i]);
	    }
	    clean_up(ERROR);
	}
#endif /* defined(CHECK_FOR_BAD_STATES) */

}		/*end point_FD*/


/*
*			g_two_side_npt_tang():
*	
*	Calls one_side_npt_tang_solver() first on the left side and then on the
*	right side of the curve curve.
*/


EXPORT void g_two_side_npt_tang_solver(
	float		ds,
	float		dt,
	Tan_stencil	*sten,
	Locstate	ansl,
	Locstate	ansr,
	Front		*fr)
{
	INTERFACE	*intfc;
	COMPONENT	lcomp, rcomp;
	HYPER_SURF	*hs = sten->newhs;
	float		vn, v[MAXD], n[MAXD];
	int		i;
	static	size_t	sizest = 0;
	static	int	dim = 0;

	if (hs == NULL)
	    return;
	intfc = hs->interface;
	lcomp = negative_component(hs);
	rcomp = positive_component(hs);
	if (dim == 0)
	{
	    sizest = fr->sizest;
	    dim = intfc->dim;
	}

	if ((dim==2) && (sten->hs[0] != NULL))
	{
	    Locstate	sl, sr;

	    slsr(sten->p[0],sten->hse[0],sten->hs[0],&sl,&sr);
	    copy_state(sten->leftst[0],sl);
	    copy_state(sten->rightst[0],sr);
	}

	switch (wave_type(hs))
	{
	case SUBDOMAIN_BOUNDARY:
	case PASSIVE_BOUNDARY:
	    g_obstacle_state(ansl,sizest);
	    g_obstacle_state(ansr,sizest);
	    break;

	/* Should this be applied in general? */
	case	NEUMANN_BOUNDARY:
	    dim = fr->rect_grid->dim;
	    normal(sten->p[0],sten->hse[0],sten->hs[0],n,fr);
	    if (is_excluded_comp(lcomp,intfc) == YES)
	    {
		sten->comp = rcomp;
		sten->states = sten->rightst;
	    	one_side_npt_tang_solver(ds,dt,sten,ansr,fr);
		if (no_slip(hs))
		{
		    float alpha = 1.0 - adherence_coeff(hs);
		    alpha_state_velocity(alpha,ansr,dim);
		}
	    	zero_normal_velocity(ansr,n,dim);
	    	g_obstacle_state(ansl,sizest);
	    }
	    else
	    {
		sten->comp = lcomp;
		sten->states = sten->leftst;
	    	one_side_npt_tang_solver(ds,dt,sten,ansl,fr);
		if (no_slip(hs))
		{
		    float alpha = 1.0 - adherence_coeff(hs);
		    alpha_state_velocity(alpha,ansl,dim);
		}
	    	zero_normal_velocity(ansl,n,dim);
	    	g_obstacle_state(ansr,sizest);
	    }
	    break;

	case	DIRICHLET_BOUNDARY:
	    if (is_excluded_comp(lcomp,intfc) == YES)
	    	g_obstacle_state(ansl,sizest);
	    else
	    {
		sten->comp = lcomp;
		sten->states = sten->leftst;
	    	one_side_npt_tang_solver(ds,dt,sten,ansl,fr);
	    }
	    if (is_excluded_comp(rcomp,intfc) == YES)
	    	g_obstacle_state(ansr,sizest);
	    else
	    {
		sten->comp = rcomp;
		sten->states = sten->rightst;
	    	one_side_npt_tang_solver(ds,dt,sten,ansr,fr);
	    }
	    break;

	default:
	    sten->comp = lcomp;
	    sten->states = sten->leftst;
	    one_side_npt_tang_solver(ds,dt,sten,ansl,fr);
	    sten->comp = rcomp;
	    sten->states = sten->rightst;
	    one_side_npt_tang_solver(ds,dt,sten,ansr,fr);
	    break;
	}
        /**** Woh, change made in here ****/
        /***
        if(is_scalar_wave(wave_type(hs)) && Params(ansl)!=Params(ansr))
        {   // TO enfore the pressure match across the contact
            float V[3];
            float pjump;
            // verbose_print_state("before correction ansl",ansl);
            // verbose_print_state("before correction ansr",ansr);
            normal(sten->p[0],sten->hse[0],sten->hs[0],n,fr);

            pjump=set_pjump_at_wave(sten->p[0],sten->hse[0],sten->hs[0],fr,n);
            w_speed(Coords(sten->p[0]),ansl,ansr,ansl,ansr,V,pjump,
                    n,wave_type(hs),fr);
            // verbose_print_state("after correction ansl",ansl);
            // verbose_print_state("after correction ansr",ansr);
        }
        ***/
#if defined(CHECK_FOR_BAD_STATES)
	if (debugging("bad_state") && 
	    ((is_bad_state(ansl,YES,"g_two_side_npt_tang_solver")) ||
	     (is_bad_state(ansr,YES,"g_two_side_npt_tang_solver"))))
	{
	    int	    i, nrad = sten->npts/2;
	    char    s[80];
	    screen("ERROR in g_two_side_npt_tang_solver(), bad state found\n");
	    (void) printf("ansl - ");
	    fprint_raw_gas_data(stdout,ansl,current_interface()->dim);
	    (void) printf("ansr - ");
	    fprint_raw_gas_data(stdout,ansr,current_interface()->dim);

	    (void) printf("ds = %g, dt = %g\n",ds,dt);
	    print_general_vector("dir = ",sten->dir,dim,"\n");
	    print_Tan_stencil(fr,sten);

	    for (i = -nrad; i <= nrad; ++i)
	    {
	        (void) sprintf(s,"sten->leftst[%d]\n",i);
	        verbose_print_state(s,sten->leftst[i]);
	        (void) sprintf(s,"sten->rightst[%d]\n",i);
	        verbose_print_state(s,sten->rightst[i]);
	    }
	    verbose_print_state("ansl",ansl);
	    verbose_print_state("ansr",ansr);
	    (void) printf("Input hypersurface\n");
	    print_hypersurface(hs);
	    print_interface(intfc);
	    clean_up(ERROR);
	}
#endif /* defined(CHECK_FOR_BAD_STATES) */
}		/*end g_two_side_npt_tang_solver*/


/*
*			check_ans():
*
*	Used in this file and ggodunov.c only.
*	Returns YES if states are physical,  NO otherwise.
*/

EXPORT	bool check_ans(
	const char	*function,
	float		ds,
	float		dt,
	Locstate	ans,
	COMPONENT	comp,
	Stencil		*sten,
	int		increment_count)
{
	bool		bad;
	int		nrad = sten->npts/2;
	int		*icoords = sten->icoords[0];
	int		i, dim = sten->fr->interf->dim;

	bad = is_bad_state(ans,YES,function);
	if (bad == NO)
	{
	    for (i = -nrad; (bad==NO) && i <= nrad; ++i)
	    	bad = is_bad_state(sten->st[i],YES,function);
	}
	if (bad == YES)
	{
	    if (debugging("bad_gas"))
	    {
 	        screen("WARNING - check_ans() detects bad gas state in %s\n",
		       function);
	        print_general_vector("Position = ",Coords(sten->p[0]),dim,"");
		print_int_vector(", icoords = ",icoords,dim,"\n");
		(void) printf("ds = %g, dt = %g, comp = %d\n",ds,dt,comp);
		print_Stencil(sten);
		for (i = -nrad; i <= nrad; ++i)
		{
		    (void) printf("state[%d]\n",i);
		    (*sten->fr->print_state)(sten->st[i]);
		}
		(void) printf("ANSWER\n");
		(*sten->fr->print_state)(ans);
		(void) printf("\n");
	    }

	    if (too_many_bad_gas_states("check_ans()",increment_count,sten->fr))
		clean_up(ERROR);
	    return NO;
	}
	return YES;
}		/*end check_ans*/

/*
*			check_gas():
*
*	Returns YES if states are physical,  NO otherwise.
*/

EXPORT bool check_gas(
	const char      *function,
	Locstate	*sts,
	Locstate	ans,
	Tan_stencil	*sten,
	int		increment_count,
	Front		*fr)
{
	bool		bad;
	int		i, j, nrad = sten->npts/2;
	int		dim = fr->rect_grid->dim;
	char		title[20];

	bad = is_bad_state(ans,YES,function);
	if (bad == NO)
	{
	    for (i = -nrad; (bad==NO) && i <= nrad; ++i)
	    	bad = is_bad_state(sts[i],YES,function);
	}
	if (bad == YES)
	{
	    if (debugging("bad_gas"))
	    {
	        (void) printf("WARNING in check_gas(),  bad gas state\n");
	        for (i = -nrad; i <= nrad; ++i)
	        {
	            (void) printf("curve[%d] = %llu, point[%d] = %llu",i,
	    		          curve_number(Curve_of_hs(sten->hs[i])),i,
	                          point_number(sten->p[i]));
	            if (sten->p[i] != NULL)
	            {
	                for (j = 0; j < dim; ++j)
	                    (void) printf(" %g",Coords(sten->p[i])[j]);
	            }
	            (void) printf("\n");
	            (void) sprintf(title,"state[%d]",i);
	            verbose_print_state(title,sts[i]);
	        }
	        verbose_print_state("NEW",ans);
	        (void) printf("\n");
	    }

	    if (too_many_bad_gas_states("check_gas()",increment_count,fr))
	    	clean_up(ERROR);
		
	    return NO;
	}
	return YES;
}		/*end check_gas*/


LOCAL	bool too_many_bad_gas_states(
	const char	*mesg,
	int		increment_count,
	Front		*front)
{
	static const  int MAX_WARNINGS = 20;/*TOLERANCE*/
	static int total_warnings = 0;
	static int warnings_per_step = 0;
	static int timestep_of_last_warning = -1;

	if (!increment_count)
	    return NO;
	++total_warnings;
	if (front->step != timestep_of_last_warning)
	{
	    timestep_of_last_warning = front->step;
	    warnings_per_step = 0;
	}
	if (++warnings_per_step > MAX_WARNINGS)
	{
	    screen("Fatal ERROR in %s\nERROR - too many (%d) bad gas states "
	           "in a single time step\n",
		   mesg,warnings_per_step);
	    screen("ERROR - Total warnings = %d\n",total_warnings);
	    return YES;
	}
	return NO;
}		/*end too_many_bad_gas_states*/

/*
*			g_load_state_vectors():
*
*	This function loads the conservative state variables from the wave into
*	a Vec_Gas suitable for use by the vector solvers.  This function is
*	currently used only by the vector solvers, thus we make the assumption
*	that imin == 0.	 This means there are offset artificial states 
*	on either end of the Vec_Gas -- 0 to (offset-1) and (imax-offset) to
*	(imax-1).  
*
*	To simply load a Vec_Gas from arbitray imin to arbitray imax,
*	just pass offset = 0.
*/

EXPORT TIME_DYNAMICS g_load_state_vectors(
	int		swp_num,
	int		*iperm,
	Vec_Gas		*vst,
	int		imin,
	int		imax,
	Wave		*wv,
	Wave		*newwv,
	int		*icoords,
	int		pbuf)
{
	Locstate	rst;
	Locstate	*state = vst->state;
	float		**coords = vst->coords;
	float		*rho = vst->rho;
	float		*en_den = vst->en_den;
	float		*m[MAXD];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	float		*vacuum_dens = vst->vacuum_dens;
	float		*min_pressure = vst->min_pressure;
	float		*min_energy = vst->min_energy;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	int		i, j;
	int		dim = wv->rect_grid->dim;
	int		idirs[MAXD];
	static float    **Q = NULL;

	clear_Vec_Gas_set_flags(vst);
	for (j = 0; j < dim; ++j)
	{
	    idirs[j] = iperm[(j+swp_num)%dim];
	    m[j] = vst->m[j];
	}
	if (Q == NULL)
	    matrix(&Q,3,3,FLOAT);
	Q[0][0] = Q[0][1] = Q[0][2] = 0.0;
	Q[1][0] = Q[1][1] = Q[1][2] = 0.0;
	Q[2][0] = Q[2][1] = Q[2][2] = 0.0;
	for (i = 0; i < dim; ++i)
	    Q[i][idirs[i]] = 1.0;
	for (; i < 3; ++i)
	    Q[i][i] = 1.0;
	vst->Q = (const float* const*)Q;
	for (i = imin; i < imax; ++i)
	{
	    icoords[idirs[0]] = i + pbuf;
	    state[i] = rst = Rect_state(icoords,wv);
	    coords[i] = Rect_coords(icoords,wv);
	    rho[i] = Dens(rst);
	    en_den[i] = Energy(rst);
	    for (j = 0; j < dim; ++j)
		m[j][i] = Mom(rst)[idirs[j]];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	    if (Params(rst) != NULL)
	    {
	    	vacuum_dens[i] = Vacuum_dens(rst);
	    	min_pressure[i] = Min_pressure(rst);
	    	min_energy[i] = Min_energy(rst);
	    }
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	}
	Vec_Gas_field_set(vst,state) = YES;
	Vec_Gas_field_set(vst,coords) = YES;
	Vec_Gas_field_set(vst,rho) = YES;
	Vec_Gas_field_set(vst,en_den) = YES;
	Vec_Gas_field_set(vst,m) = YES;
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	Vec_Gas_field_set(vst,min_pressure) = YES;
	Vec_Gas_field_set(vst,min_energy) = YES;
	Vec_Gas_field_set(vst,vacuum_dens) = YES;
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	set_params_jumps(vst,imin,imax);
	if ((Params(state[0]) == NULL) && (vst->nprms <= 1))
	{
	    int	nrad = vsten_radius(wv);
	    copy_states_to_new_time_level(idirs,wv,newwv,imin+nrad,
				          imax-nrad,icoords,pbuf);
	    return CONSTANT_IN_TIME;
	}
	g_init_obstacle_states(vst,imin,dim);
	return DYNAMIC_IN_TIME;
}		/*end g_load_state_vectors*/

/*
*			g_init_obstacle_states():
*
*	This function copies a valid state into any index which is an
*	obstacle state.	 This allows the interior solver to ignore 
*	obstacle states completely.  See assign_wave_state_vectors().
*/

EXPORT void g_init_obstacle_states(
	Vec_Gas		*vst,
	int		imin,
	int		dim)
{
	int		start, end;     /*indices of obstacle state*/
	int		non_obst;     	/*index of state to be copied*/
	int		*prms_jmp = vst->prms_jmp;
	int		i, j, k;

	for (i = 0; i < vst->nprms; ++i)
	{
	    if (is_obstacle_state(vst->state[prms_jmp[i]]))
	    {
	        start = prms_jmp[i];
	        end = prms_jmp[i + 1];

	        non_obst = (start == imin) ? end : start - 1;

	        for (j = start; j < end; ++j)
	        {
	            vst->rho[j] = vst->rho[non_obst];
	            vst->en_den[j] = vst->en_den[non_obst];
	            for (k = 0; k < dim; ++k)
	            	vst->m[k][j] = vst->m[k][non_obst];
	            vst->state[j] = vst->state[non_obst];
#if !defined(UNRESTRICTED_THERMODYNAMICS)
	            vst->vacuum_dens[j] = vst->vacuum_dens[non_obst];
	            vst->min_pressure[j] = vst->min_pressure[non_obst];
	            vst->min_energy[j] = vst->min_energy[non_obst];
#endif /* !defined(UNRESTRICTED_THERMODYNAMICS) */
	        }
	    }
	}
}		/*end g_init_obstacle_states*/

EXPORT	void	set_rotation(
	float	    **Q,
	const float *dir,
	int	    dim)
{
	float	dmin, mag;
	int	i, imin;

	for (i = 0; i < dim; ++i)
	    Q[0][i] = dir[i];
	switch (dim)
	{
	case 1:
	    break;
	case 2:
	    Q[1][0] = -Q[0][1];
	    Q[1][1] =  Q[0][0];
	    break;
	case 3:
	    dmin = fabs(dir[dim-1]);	imin = dim-1;
	    for (i = 0; i < (dim-1); ++i)
	    {
	    	if (fabs(dir[i]) < dmin)
	    	{
	    	    dmin = fabs(dir[i]);
	    	    imin = i;
	    	}
	    }
	    Q[1][imin] = 0.0;
	    Q[1][(imin+1)%3] = -Q[0][(imin+2)%3];
	    Q[1][(imin+2)%3] =  Q[0][(imin+1)%3];
	    mag = mag_vector(Q[1],dim);
	    Q[1][0] /= mag; Q[1][1] /= mag; Q[1][2] /= mag;
	    mag = vector_product(Q[0],Q[1],Q[2],dim);
	    Q[2][0] /= mag; Q[2][1] /= mag; Q[2][2] /= mag;
	    break;
	}
}		/*end set_rotation*/
