/*
*				funtan2d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	The routines in this file untangle a interface.
*/

#if defined(TWOD)

#define DEBUG_STRING	"hypunr"
#include <front/fdecs.h>

#define Cross_pro(p0,p1,p2)						   \
	((Coords(p1)[0] - Coords(p0)[0])*(Coords(p2)[1] - Coords(p0)[1]) - \
	 (Coords(p1)[1] - Coords(p0)[1])*(Coords(p2)[0] - Coords(p0)[0]))

#define Dot_pro(p0,p1,p2)						   \
	((Coords(p1)[0] - Coords(p0)[0])*(Coords(p2)[0] - Coords(p0)[0]) + \
	 (Coords(p1)[1] - Coords(p0)[1])*(Coords(p2)[1] - Coords(p0)[1]))

struct _Snlist {
	CURVE		**nc;
	NODE    	**nopp, *sn;
	POINT		**pt;
	float		*ang;
	ORIENTATION	*orient;
	int		num_c;
	int		nc_set, nopp_set, pt_set, ang_set, orient_set;
} ;
typedef struct _Snlist Snlist;

	/* LOCAL Function Declarations */
LOCAL	bool	check_phys_loops_on_snlist(Snlist*);
LOCAL	bool	correct_and_identify_node_types(Front*);
LOCAL	bool	is_node_in_node_list(NODE*,NNLIST*);
LOCAL	bool	pass1_delete_unphys_curves(Front*,NNLIST**,float,int);
LOCAL	bool	pass2_delete_unphys_curves(Front*,NNLIST**,float,int);
LOCAL	float	Pangle(POINT*,POINT*,POINT*);
LOCAL	float	area_of_loop(NNLIST*,int);
LOCAL	int	total_number_curves_at_node(NODE*);
LOCAL	void	alloc_sn_list(Snlist*);
LOCAL	void	check_physical_loops_at_sink_nodes(INTERFACE*);
LOCAL	void	delete_from_nnlist(NNLIST*,NNLIST**);
LOCAL	void	free_sn_list(Snlist*);
LOCAL	void	print_NNLIST_struct(NNLIST*);
LOCAL	void	print_Snlist(Snlist*);
LOCAL	void	set_curve_info_in_new_node_list(NNLIST*);
LOCAL	void	set_curve_list_for_node(Snlist*);
LOCAL	void	set_index_for_phys_sectors(NNLIST*);


/*
*			is_vector_vector_cross():
*/

EXPORT	bool	is_vector_vector_cross(
	CROSS		*cr)
{
	if (	wave_type(cr->c1) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE &&
		wave_type(cr->c2) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE)
		return YES;
	return NO;
}		/*end is_vector_vector_cross*/

/*
*			is_scalar_vector_cross():
*/

EXPORT	bool	is_scalar_vector_cross(
	CROSS		*cr)
{
	if (is_vector_vector_cross(cr) == YES)
		return NO;
	if (	wave_type(cr->c1) < FIRST_PHYSICS_WAVE_TYPE ||
		wave_type(cr->c2) < FIRST_PHYSICS_WAVE_TYPE)
		return NO;
	if (	wave_type(cr->c1) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE ||
		wave_type(cr->c2) >= FIRST_VECTOR_PHYSICS_WAVE_TYPE)
		return YES;
	return NO;
}		/*end is_scalar_vector_cross*/


/*
*			scalar_unravel():
*
*	The main routine for untangling a scalar interface.
*	This routine assumes that all node types on the incoming
*	interface are known and thus after the crossing points
*	are inserted they are the only nodes with unknown type.
*/


EXPORT	int scalar_unravel(
	Front		*fr,
	CROSS		**crx,
	int		flag)
{
	INTERFACE	*intfc = fr->interf;
	INTERFACE	*hold_intfc;
	CROSS		*cross;
	NNLIST		*new_node_list, *nnlist, *nl;
	CROSS		*cr, *crb;
	CURVE		**c;
	CURVE		*curves1[2], *curves2[2];
        NODE		*m, **n;
	float		*h = fr->rect_grid->h;
	float		hx = h[0], hy = h[1];
	float		min_area = 2.0*hx*hy;
	int		dim = fr->rect_grid->dim;
	int		j, num;

	DEBUG_ENTER(scalar_unravel)
	cross = *crx;

	if (cross == NULL)
	{
	    DEBUG_LEAVE(scalar_unravel)
	    return CURVES_UNTANGLED;
	}

	hold_intfc = current_interface();
	set_current_interface(intfc);
	if (DEBUG)
	{
	    (void) printf("In scalar_unravel(), intfc %llu cross %p\n",
			  interface_number(intfc),(POINTER)cross);
	    for (c = intfc->curves; c && *c;  c++)
		    print_bond_list(*c);

	    /* output debugging information */

	    for (num = 0, cr = cross;  cr;  num++, cr = cr->next)
		    ;

	    (void) printf("\nThe hyp interface is tangled at %d Point(s)\n",
			  num);
	    (void) printf("Here is the cross list of intersections:\n\n");

	    for (cr = cross;  cr;  cr = cr->next)
		    print_cross(cr);

	    (void) output();
	    (void) printf("\t\tHERE IS THE HYP INTERFACE TO BE UNTANGLED\n\n");
	    if (debugging("states"))
		    show_intfc_states(intfc);
	    else
		    print_interface(intfc);
	}

/* PREPROCESS the cross_list to spot crossings very close to each other */

label_1:
	if (cross == NULL)
	{
	    set_current_interface(hold_intfc);
	    DEBUG_LEAVE(scalar_unravel)
	    return CURVES_UNTANGLED;
	}
	for  (cr = cross; cr->next; cr = cr->next)
	{
	    for (crb = cr->next; crb; crb = crb->next)
	    {
		float	sclen;

		sclen = scaled_separation(cr->p,crb->p,h,dim);
		if (DEBUG)
		{
		    (void) printf("Testing crosses %p %p for close crosses, "
		                  "Scaled separation = %g\n",
				  (POINTER)cr,(POINTER)crb,sclen);
		}

#define are_redundant_crosses(cr1,cr2,dim)				\
	((separation((cr1)->p,(cr2)->p,(dim)) <				\
		 			EPSILON*bond_length((cr1)->b1))	\
			 	&&					\
	 (((cr1)->b1 == (cr2)->b1) || ((cr1)->b1 == (cr2)->b2) ||	\
	  ((cr1)->b2 == (cr2)->b1) || ((cr1)->b2 == (cr2)->b2)))

		if (are_redundant_crosses(cr,crb,dim))
		{
		    (void) printf("WARNING in scalar_unravel(), "
		                  "redundant crosses detected\n");
		    set_current_interface(hold_intfc);
		    DEBUG_LEAVE(scalar_unravel)
		    return ERROR_IN_UNTANGLE;
		}
		else if (sclen < MIN_SC_SEP(intfc))
		{
		    BOND	*cb1, *cb2, *cbb1, *cbb2,
		    		*cbb1n, *cbb1p, *cbb2n, *cbb2p;

				/* RESOLVE THE TWO CROSSES */

		    if (DEBUG)
			(void) printf("eliminate two close crosses %p, %p \n",
				      (POINTER)cr,(POINTER)crb);


		    cb1   = cr->b1;		cbb1  = crb->b1;
		    cb2   = cr->b2;		cbb2  = crb->b2;


		    if (cr->c1 == crb->c1 && is_closed_curve(cr->c1))
		    {
			if ((cb1->prev == NULL) && (cbb1->next == NULL))
			    (void) move_closed_loop_node(cr->c1,cb1->next);
			if ((cb1->next == NULL) && (cbb1->prev == NULL))
			    (void) move_closed_loop_node(cr->c1,cbb1->next);
		    }
		    if (cr->c1 == crb->c2 && is_closed_curve(cr->c1))
		    {
			if ((cb1->prev == NULL) && (cbb2->next == NULL))
			    (void) move_closed_loop_node(cr->c1,cb1->next);
			if ((cb1->next == NULL) && (cbb2->prev == NULL))
			    (void) move_closed_loop_node(cr->c1,cbb2->next);
		    }
		    if (cr->c2 == crb->c1 && is_closed_curve(cr->c2))
		    {
			if ((cb2->prev == NULL) && (cbb1->next == NULL))
			    (void) move_closed_loop_node(cr->c2,cb2->next);
			if ((cb2->next == NULL) && (cbb1->prev == NULL))
			    (void) move_closed_loop_node(cr->c2,cbb1->next);
		    }
		    if (cr->c2 == crb->c2 && is_closed_curve(cr->c2))
		    {
			if ((cb2->prev == NULL) && (cbb2->next == NULL))
			    (void) move_closed_loop_node(cr->c2,cb2->next);
			if ((cb2->next == NULL) && (cbb2->prev == NULL))
			    (void) move_closed_loop_node(cr->c2,cbb2->next);
		    }

		    cbb1p = cbb1->prev;		cbb2p = cbb2->prev;
		    cbb1n = cbb1->next;		cbb2n = cbb2->next;

		    if     (cb1 == cbb1n)
			(void) delete_start_of_bond(cb1, cr->c1);
		    else if (cb1 == cbb1p)
			(void) delete_start_of_bond(cbb1,crb->c1);
		    else if (cb2 == cbb2n)
			(void) delete_start_of_bond(cb2, cr->c2);
		    else if (cb2 == cbb2p)
			(void) delete_start_of_bond(cbb2,crb->c2);
		    else if (cb2 == cbb1n)
			(void) delete_start_of_bond(cb2, cr->c2);
		    else if (cb2 == cbb1p)
			(void) delete_start_of_bond(cbb1,crb->c1);
		    else if (cb1 == cbb2n)
			(void) delete_start_of_bond(cb1, cr->c1);
		    else if (cb1 == cbb2p)
			(void) delete_start_of_bond(cbb2,crb->c2);
		    else
		    {
			(void) printf("WARNING in scalar_unravel(), "
			              "Nothing done\n");
			set_current_interface(hold_intfc);
			DEBUG_LEAVE(scalar_unravel)
			return ERROR_IN_UNTANGLE;
		    }
					
		    if (DEBUG)
		    {
			(void) printf("Near crosses found and "
			              "resolution attempted\n");
			(void) printf("Recalling intersections\n");
		    }

		    if (intersections(intfc,crx,NO) == FUNCTION_FAILED)
		    {
			(void) printf("WARNING in scalar_unravel(), "
			              "intersections() failed\n");
			set_current_interface(hold_intfc);
			DEBUG_LEAVE(scalar_unravel)
			return ERROR_IN_UNTANGLE;
		    }
		    cross = *crx;
		    goto label_1;
		}
	    }
	}
/* END OF PREPROCESSING */

	if (cross == NULL)
	{
	    set_current_interface(hold_intfc);
	    DEBUG_LEAVE(scalar_unravel)
	    return CURVES_UNTANGLED;
	}

	new_node_list = (NNLIST *)Store(sizeof(NNLIST));
	new_node_list->prev = NULL;
	new_node_list->next = NULL;
	nnlist = new_node_list;
	nnlist->num_phys = 0;
	for (j = 0; j < 4; j++)
	    nnlist->area[j] = HUGE_VAL;

/* 
* PRIMARY LOOP ON CROSS LIST, in which the cross points are replaced by nodes 
*/
	
	if (DEBUG)
	    (void) printf("CROSS LOOP\n");
	for (cr = cross;  cr != NULL;  cr = cr->next)
	{
	    if (DEBUG)
	    	(void) printf("cr %p\n",(POINTER)cr);
		
	    split_curves_at_cross(cr,fr,&m,curves1,NULL,NULL,curves2,NULL,
			          NULL,MIN_SC_SEP(intfc),NULL);
	    nnlist->m = m;


	    if (cr->next != NULL)
	    {
	    	nnlist->next = (NNLIST *)Store(sizeof(NNLIST));
	    	nnlist->next->prev = nnlist;
	    	nnlist = nnlist->next;
	    	nnlist->next = NULL;
	    	nnlist->num_phys = 0;
	    	for (j = 0; j < 4; j++)
	    	    nnlist->area[j] = HUGE_VAL;
	    }
		/* New_node list is now generated */

	}

/* END OF PRIMARY LOOP */

		/*  Eliminate redundant CLOSED nodes */
redundant_delete:
	for (n = intfc->nodes; *n; n++)
	{
	    if (DEBUG)
	       (void) printf("Check node %llu to eliminate\n",node_number(*n));
	    if (node_type(*n) != CLOSED_NODE)
		continue;
	    if (delete_redundant_node(*n,NULL,NULL,fr))
		goto redundant_delete;
	}

		/* Insert curve info in new_node_list */

	set_curve_info_in_new_node_list(new_node_list);

	if (DEBUG)
	{
	    (void) printf("HYP INTERFACE AFTER CROSS LIST LOOP\n");
	    if (debugging("states"))
	    	show_intfc_states(intfc);
	    else
	    	print_interface(intfc);
	}

		/* First pass at identifying physical sectors */

	set_index_for_phys_sectors(new_node_list);

	if (DEBUG)
	{
	    (void) printf("Node list after first set index\n");
	    for (nl = new_node_list; nl != NULL; nl = nl->next)
	    	print_NNLIST_struct(nl);
	}

	/* loop over new_node_list; identify and delete unphysical loops */

	if (pass1_delete_unphys_curves(fr,&new_node_list,min_area,flag) == NO)
	{
	    (void) printf("WARNING in scalar_unravel(), "
	                  "pass1_delete_unphys_curves() failed\n");
	    set_current_interface(hold_intfc);
	    DEBUG_LEAVE(scalar_unravel)
	    return ERROR_IN_UNTANGLE;
	}

	if (DEBUG)
	{
	    (void) printf("Node list after first delete "
	                  "unphysical curve loop\n");
	    for (nl = new_node_list; nl != NULL; nl = nl->next)
		print_NNLIST_struct(nl);
	}


	/* loop over new_node_list again to identify and delete unphysical
	   loops, resolve if possible which of the two unphysical loops
	   should be deleted if this fails delete the one with smallest area */

	if (pass2_delete_unphys_curves(fr,&new_node_list,min_area,flag) == NO)
	{
	    (void) printf("WARNING in scalar_unravel(), "
	                  "pass2_delete_unphys_curves() failed\n");
	    set_current_interface(hold_intfc);
	    DEBUG_LEAVE(scalar_unravel)
	    return ERROR_IN_UNTANGLE;
	}

	if (DEBUG)
	{
	    (void) printf("before corr \n");
	    if (debugging("states"))
	    	show_intfc_states(intfc);
	    else
	    	print_interface(intfc);
	}

	if (correct_and_identify_node_types(fr) == NO)
	{
	    (void) printf("WARNING in scalar_unravel(), "
	                  "correct_and_identify_node_types() failed\n");
	    set_current_interface(hold_intfc);
	    DEBUG_LEAVE(scalar_unravel)
	    return ERROR_IN_UNTANGLE;
	}
	if (DEBUG)
	{
	    (void) printf("After correct_and_identify_nodes\n");
	    if (debugging("states"))
	    	show_intfc_states(intfc);
	    else
	    	print_interface(intfc);
	}

		/* Check all loops remaining at sink nodes are physical */

	check_physical_loops_at_sink_nodes(intfc);

		/* update points_on_curve count -- should be unnecessary */
	
	update_num_points(intfc);

	if (DEBUG)
	{
	    (void) output();
	    (void) printf("UNTANGLED HYP INTERFACE:\n");
	    if (debugging("states"))
	    	show_intfc_states(intfc);
	    else
	    	print_interface(intfc);
	}

	set_current_interface(hold_intfc);
	DEBUG_LEAVE(scalar_unravel)
	return CURVES_UNTANGLED;
}		/*end scalar_unravel*/


/*
*			set_curve_info_in_new_node_list():
*
*	Insert curve information in node list
*/

LOCAL	void set_curve_info_in_new_node_list(
	NNLIST		*new_node_list)
{
	NNLIST		*nl;
	CURVE		**ci, **co, *c1;
	POINT 		*pt[4];
        NODE    	*m;
	int		i, j;
	ORIENTATION	orient;
	float		a, an[4];

	DEBUG_ENTER(set_curve_info_in_new_node_list)
	for (nl = new_node_list; nl != NULL; nl = nl->next)
	{
		if (DEBUG)
			(void) printf("Inserting curve info in node_list\n");

		/* check if necessary to flip c1 and c3 */

		m = nl->m;
		ci = m->in_curves;	co = m->out_curves;
		nl->nc[0] = *ci;	nl->orient[0] = NEGATIVE_ORIENTATION;
		nl->nc[2] = *co;	nl->orient[2] = POSITIVE_ORIENTATION;
		ci++;			co++;
		nl->nc[1] = *ci;	nl->orient[1] = NEGATIVE_ORIENTATION;
		nl->nc[3] = *co;	nl->orient[3] = POSITIVE_ORIENTATION;

		for (i = 0; i < 4; i++)
		{
			nl->nn[i] = Node_of(nl->nc[i],
					Opposite_orient(nl->orient[i]));
			pt[i] = Point_adjacent_to_node(nl->nc[i],
						       nl->orient[i]);
		}

		if (DEBUG)
		{
		    (void) printf("Ordering the curves here ");
		    (void) printf("c[i] = %llu %llu %llu %llu\n",
			       curve_number(nl->nc[0]),curve_number(nl->nc[1]),
			       curve_number(nl->nc[2]),curve_number(nl->nc[3]));
		}

		for (i = 0; i < 3; i++)
			an[i] = Pangle(m->posn,pt[0],pt[i+1]);

		if (DEBUG)
			(void) printf("angles %g %g %g\n",an[0],an[1],an[2]);

			/* Order curves by angle */

		for (i = 0; i < 2; i++)
		{
			for (j = i+1; j < 3; j++)
			{
				if (an[j] < an[i])
				{
					a = an[i];
					m = nl->nn[i+1];
					c1 = nl->nc[i+1];
					orient = nl->orient[i+1];
					an[i] = an[j];	
					nl->nc[i+1] = nl->nc[j+1];
					nl->nn[i+1] = nl->nn[j+1];
					nl->orient[i+1] = nl->orient[j+1];
					an[j] = a;
					nl->nn[j+1] = m;
					nl->nc[j+1] = c1;
					nl->orient[j+1] = orient;
				}
			}
		}
		for (i = 0; i < 4; i++)
		{
		    nl->comp[i] = (nl->orient[i] == POSITIVE_ORIENTATION)?
			    negative_component(nl->nc[i]) :
				    positive_component(nl->nc[i]);
		    if (DEBUG)
		    {
		      (void) printf("Setting components for nl\n");
		      (void) printf("i %d nc %llu comps: l %d r %d  comp[] %d\n",
				    i,curve_number(nl->nc[i]),
				    negative_component(nl->nc[i]),
				    positive_component(nl->nc[i]),nl->comp[i]);
	            }
		}

		if (DEBUG)
			print_NNLIST_struct(nl);
	}
	DEBUG_LEAVE(set_curve_info_in_new_node_list)
}		/*end set_curve_info_in_new_node_list*/

/*
*			set_index_for_phys_sectors():
*
*	Set index for physical sectors using algorithm described in
*	Glimm, Grove, Lindquist, Mcbryan, Tryggvasson.
*/

LOCAL	void set_index_for_phys_sectors(
	NNLIST		*new_node_list)
{
	NNLIST		*nl;
	int		i;

	DEBUG_ENTER(set_index_for_phys_sectors)
	for (nl = new_node_list; nl != NULL; nl = nl->next)
	{
	    for (i = 0; i < 4; i++)
		nl->ni[i] = UNPHYSICAL;

	    for (i = 0; i < 4; i++)
	    {
	    	if (nl->nn[i] != nl->nn[(i+1)%4])
	    	    nl->ni[i] = PHYSICAL;
	    	else
	    	{
	    	    if ((nl->nn[i] == nl->m) && (nl->nc[i] == nl->nc[(i+5)%4]))
	    	    {
	    	    	nl->ni[(i+3)%4] = PHYSICAL;
	    	    	nl->ni[(i+1)%4] = PHYSICAL;
	    	    }
	    	}
	    }
	}
	DEBUG_LEAVE(set_index_for_phys_sectors)
}		/*end set_index_for_phys_sectors*/


/*
*			pass1_delete_unphys_curves():
*
*	First loop over new_node_list - identify and delete unphysical loops
*	First loop algorithm described in Glimm, Grove, Lindquist, McBryan,
*	Tryggvasson.
*/

LOCAL	bool pass1_delete_unphys_curves(
	Front		*fr,
	NNLIST		**new_node_list,
	float		min_area,
	int		flag)
{
	NNLIST		*nl, *ml;
	CURVE		**c, *newc;
	int		j, i, count;

	DEBUG_ENTER(pass1_delete_unphys_curves)
restart_loop:

	for (nl = (*new_node_list); nl != NULL; nl = nl->next)
	{
		/* Node with less than 4 curves. Delete from new node list */
restart_pass:
	    count = 0;
	    for (c = nl->m->in_curves;  c && *c; c++) count++;
	    for (c = nl->m->out_curves; c && *c; c++) count++;

	    if  (count < 4)
	    {
	    	ml = nl;
	    	nl = nl->next;
	    	delete_from_nnlist(ml,new_node_list);
	    	if (nl == NULL) break;
	    	goto restart_pass;
	    }

	    nl->num_phys = 0;
	    j = 0;
	    for (i = 0; i < 4; i++)
	    {
	    	if (nl->ni[i] == PHYSICAL)
		    nl->num_phys++;
	    	else
		    j = i;
	    }

	    if (nl->num_phys == 3)     /* one unphys. loop, (loop j) */
	    {
	    	nl->area[j] = area_of_loop(nl,j);
	    	if (replace_unphys_loop(nl,new_node_list,&newc,fr,
	    			j,min_area,flag) == FUNCTION_FAILED)
	    	{
	    	    DEBUG_LEAVE(pass1_delete_unphys_curves)
	    	    return NO;
	    	}
	    	goto restart_loop;
	    }
	    if (nl->num_phys == 4)
	    {
	        (void) printf("WARNING in pass1_delete_unphys_curves(), ");
	        (void) printf("all sectors physical\n");
	        return NO;
	    }
	}
	DEBUG_LEAVE(pass1_delete_unphys_curves)
	return YES;
}		/*end pass1_delete_unphys_curves*/


/*
*			pass2_delete_unphys_curves():
*
*	Second loop over new_node_list - identify and delete unphysical loops
*	Second loop algorithm described in Glimm, Grove, Lindquist, Mcbryan,
*	Tryggvasson.
*/

LOCAL	bool pass2_delete_unphys_curves(
	Front		*fr,
	NNLIST		**new_node_list,
	float		min_area,
	int		flag)
{
	CURVE		*newc, *cc;
	INTERFACE	*intfc = fr->interf;
	NNLIST		*nl, *ml;
	int		i, j, stat;
	float		area, ai[4];

	DEBUG_ENTER(pass2_delete_unphys_curves)
	nl = *new_node_list;

	while (nl != NULL)
	{
	    /* if the node has only two in and/or out curves, */
	    /* 	  delete it from the new node list 	  */

	    if (total_number_curves_at_node(nl->m) < 4)
	    {
		if (DEBUG)
			(void) printf("go to next_node_list\n");
		goto next_node_list;
	    }
		

	    for (i = 0; i < 4; i++) ai[i] = 0.0;

	    if (DEBUG)
		(void) printf("after set ai = 0\n");

	    for (i = 0; i < 4; i++)
	    {
		if (DEBUG)
			(void) printf("find ai\n");

		if (nl->ni[i] == UNPHYSICAL)
		{
			ai[i] = nl->area[i] = area_of_loop(nl,i);
		}
		else	ai[i] = HUGE_VAL;

		if (ai[i] < 0.0)
		{
			if (nl->nc[i]->start == nl->nc[i]->end)
				ai[i] = -ai[i];
			else
				ai[i] = HUGE_VAL;
		}
	    }

	    if (DEBUG)
	    {
		(void) printf(" the ai's found:\n");
		for (j = 0; j < 4; j++)
		{
			if (ai[j] == HUGE_VAL)
				(void) printf("\tai[%d] = UNASSIGNED\n",j);
			else
				(void) printf("\tai[%d] = %g\n",j,ai[j]);
		}
	    }

	    if (nl->num_phys == 2)
	    {
		INTERFACE *sav_intfc, *tmp_intfc;
		COMPONENT interior_comp[2], comp0, comp1, ext_comp;
		NODE	  *corr_m;
		bool	  sav_interpolate;
		int	  unphys[2], k0, k1, k0n, k1n;

		for (i = 0, j = 0; i < 4; i++)
		{
			if (nl->ni[i] == UNPHYSICAL) unphys[j++] = i;
		}

			/*
			*	check for and handle this case
			*
			*	  -------------------------
			*    P	/  U          a          U  \  P
			*   ---------------------------------------
			*    P	\  U          b          U  /  P
			*	  -------------------------
			*  
			*	where loops 'a' and 'b' are both unphysical
			*	by deleting loop 'a' ('b') if its area is
			*	much smaller than loop 'b' ('a').
			*/

		k0 = unphys[0];		k0n = (k0 + 1) % 4;
		k1 = unphys[1];		k1n = (k1 + 1) % 4;
		if ((nl->nn[k0] == nl->nn[k0n]) && (nl->nn[k1] == nl->nn[k1n]))
		{
		    if (DEBUG)
		    {
			(void) printf("\nchecking bananna loops ");
			(void) printf("k0 %d k0n %d  k1 %d k1n %d\n",
				      k0,k0n,k1,k1n);
		    }
		    if (fabs(ai[k0]) < 0.01*fabs(ai[k1]))
		    {
			stat = replace_unphys_loop(nl,new_node_list,&newc,
						   fr,k0,min_area,flag);
			if (stat == FUNCTION_FAILED)
			{
			    DEBUG_LEAVE(pass2_delete_unphys_curves)
			    return NO;
			}
			if (DEBUG)
				(void) printf("deleting loop k0 k0n\n");
			goto next_node_list;
		    }
		    else if (fabs(ai[k1]) < 0.01*fabs(ai[k0]))
		    {
			stat = replace_unphys_loop(nl,new_node_list,&newc,
						   fr,k1,min_area,flag);
			if (stat == FUNCTION_FAILED)
			{
			    DEBUG_LEAVE(pass2_delete_unphys_curves)
			    return NO;
			}
			if (DEBUG)
				(void) printf("deleting loop k1 k1n\n");
			goto next_node_list;
		    }
		}

		for (j = 0; j < 2; j++)
		{
		    k0 = unphys[j];		k1 = (k0 + 1) % 4;

		    if ((nl->nn[k0] == nl->nn[k1])
		            &&
			(is_node_in_node_list(nl->nn[k0],*new_node_list) == NO)
		            &&
			comps_consistent_at_node(nl->nn[k0]))
		    {
			stat = replace_unphys_loop(nl,new_node_list,&newc,
					fr,unphys[(j+1)%2],min_area,flag);
			if (stat == FUNCTION_FAILED)
			{
			    DEBUG_LEAVE(pass2_delete_unphys_curves)
			    return NO;
			}
			goto next_node_list;
		    }
		    if (nl->orient[k0] == POSITIVE_ORIENTATION)
		    {
			comp0 = negative_component(nl->nc[k0]);
		    }
		    else
		    {
			comp0 = positive_component(nl->nc[k0]);
		    }
		    if (nl->orient[k1] == POSITIVE_ORIENTATION)
		    {
			comp1 = positive_component(nl->nc[k1]);
		    }
		    else
		    {
			comp1 = negative_component(nl->nc[k1]);
		    }
		    if (comp0 != comp1)
		    {
			if (DEBUG)
			{
			    (void) printf("Inconsistent components ");
			    (void) printf("inside loop %d of nl %p\n",
					  k0,(POINTER)nl);
			    stat = replace_unphys_loop(nl,new_node_list,&newc,
						       fr,k0,min_area,flag);
			    if (stat == FUNCTION_FAILED)
			    {
				DEBUG_LEAVE(pass2_delete_unphys_curves)
				return NO;
			    }
			    goto next_node_list;
			}
		    }
		    interior_comp[j] = comp0;
		}
		sav_intfc = current_interface();
		sav_interpolate = interpolate_intfc_states(intfc);
		interpolate_intfc_states(intfc) = NO;
		set_copy_intfc_states(NO);
		tmp_intfc = copy_interface(intfc);
		cc = correspond_curve(nl->nc[0]);
		corr_m = Node_of(cc,nl->orient[0]);
		for (i = 0; i < 4; i++)
		{
		    for (j = 0; j < i; j++)
		    {
			if (nl->nc[i] == nl->nc[j])
			    goto already_deleted;
		    }
		    (void) delete_curve(correspond_curve(nl->nc[i]));

		    already_deleted:
		    	;
		}
		(void) delete_node(corr_m);
		ext_comp = long_component(Coords(nl->m->posn),tmp_intfc);
		(void) delete_interface(tmp_intfc);
		set_current_interface(sav_intfc);
		set_copy_intfc_states(YES);
		interpolate_intfc_states(intfc) = sav_interpolate;
		set_correspond_hyper_surfaces_to_NULL(intfc);

		if ((interior_comp[0] == ext_comp) &&
		    (interior_comp[1] != ext_comp))
		{
		    stat = replace_unphys_loop(nl,new_node_list,&newc,
					       fr,unphys[0],min_area,flag);
		    if (stat == FUNCTION_FAILED)
		    {
			DEBUG_LEAVE(pass2_delete_unphys_curves)
			return NO;
		    }
		    goto next_node_list;
		}
		else if ((interior_comp[0] != ext_comp) &&
			 (interior_comp[1] == ext_comp))
		{
		    stat = replace_unphys_loop(nl,new_node_list,&newc,
					       fr,unphys[1],min_area,flag);
		    if (stat == FUNCTION_FAILED)
		    {
			DEBUG_LEAVE(pass2_delete_unphys_curves)
			return NO;
		    }
		    goto next_node_list;
		}
	    }
		/* find the loop with the smallest area */

	    if (DEBUG)
		(void) printf("find the smallest area\n");

	    area = HUGE_VAL;
	    i = -1;
	    for (j = 0; j < 4; j++)
	    {
		    if (ai[j] < area)
		    {
			    area = ai[j];
			    i = j;
		    }
	    }

	    if (i == -1)
	    {
		(void) printf("WARNING in pass2_delete_unphys_curves(), ");
		(void) printf("all areas HUGE_VAL\n");
		DEBUG_LEAVE(pass2_delete_unphys_curves)
		return NO;
	    }
	    if (DEBUG)
		(void) printf("smallest area for i %d\n",i);

	    stat = replace_unphys_loop(nl,new_node_list,&newc,
				       fr,i,min_area,flag);
	    if (stat == FUNCTION_FAILED)
	    {
		DEBUG_LEAVE(pass2_delete_unphys_curves)
		return NO;
	    }

	    if (DEBUG)
		(void) printf("after delete\n");

next_node_list:
	    if (new_node_list == NULL) break;
	    ml = nl;
	    nl = nl->next;
	    delete_from_nnlist(ml,new_node_list);
	    if (DEBUG)
		(void) printf("after delete from nnl\n");
	}
	DEBUG_LEAVE(pass2_delete_unphys_curves)
	return YES;
}		/*end pass2_delete_unphys_curves*/

LOCAL	int total_number_curves_at_node(
	NODE *node)
{
	CURVE **c;
	int count = 0;

	for (c = node->in_curves;  c && *c; c++) count++;
	for (c = node->out_curves; c && *c; c++) count++;

	if (DEBUG)
	      (void) printf("Curve count %d for node %llu \n",
			    count,node_number(node));
	return count;
}		/*end total_number_curves_at_node*/


/*ARGSUSED*/
EXPORT	bool f_delete_loop(
	NNLIST		*nl,
	NNLIST		**new_node_list,
	CURVE		**newc,
	Front		*fr,
	int		i,
	float		min_area,
	int		flag)
{
	int		j; 

	DEBUG_ENTER(f_delete_loop)
	*newc = NULL;
	if (DEBUG)
	{
	    (void) printf("f_delete_loop() for sector i %d\n",i);
	    print_NNLIST_struct(nl);
	}

	if (fabs(nl->area[i]) > min_area)
	{
	    (void) printf("WARNING in f_delete_loop(), "
	                  "large loop unphysical\n");
	    if (flag != LAST_ATTEMPT_TO_UNTANGLE)
	    {
	        DEBUG_LEAVE(f_delete_loop)
	        return FUNCTION_FAILED;
	    }
	    else
	    {
	        (void) printf("Deleting large unphysical loop on third "
	                      "attempt to untangle front\n");
	    }
	}
	j = (i + 1) % 4;
	(void) delete_curve(nl->nc[j]);
	if (nl->nc[j] != nl->nc[i])
	    (void) delete_curve(nl->nc[i]);
	if (DEBUG)
	    (void) printf("Leaving f_delete_loop()\n");
	
	DEBUG_LEAVE(f_delete_loop)
	return FUNCTION_SUCCEEDED;
}		/*end f_delete_loop*/

EXPORT	bool f_replace_unphys_loop(
	NNLIST		*nl,
	NNLIST		**new_node_list,
	CURVE		**newc,
	Front		*fr,
	int		i,
	float		min_area,
	int		flag)
{

	NNLIST		*nlopp;
	Locstate	st;
	int		im1, ip1, ip2;
	int		k, km1, kp2;
	size_t		szst = fr->sizest;

	DEBUG_ENTER(f_replace_unphys_loop)
	*newc = NULL;
	if (DEBUG)
	{
	    (void) printf("f_replace_unphys_loop() for sector i %d\n",i);
	    print_NNLIST_struct(nl);
	}

	if (fabs(nl->area[i]) > min_area)
	{
	    (void) printf("WARNING in f_replace_unphys_loop(), "
	                  "large loop unphysical\n");
	    if (flag != LAST_ATTEMPT_TO_UNTANGLE)
	    {
	        DEBUG_LEAVE(f_replace_unphys_loop)
	        return FUNCTION_FAILED;
	    }
	    else
	    {
	        (void) printf("Replacing large unphysical loop on third "
	                      "attempt to untangle front\n");
	    }
	}
		/* Test whether to replace loop by single curve */

	im1 = (i+3)%4;	ip1 = (i+1)%4;	ip2 = (i+2)%4;

	if (DEBUG)
	{
	    (void) printf("test for replacement by single curve\n");
	    (void) printf("im1 %d  i %d  ip1 %d  ip2 %d\n",im1,i,ip1,ip2);
	}

		/* WARNING: tests on nl->comp[] are not reliable     */
		/*	    enough for the replacement decision      */
		/*	    OR for deciding on component assignments */
		/*	    Note how nl->comp[] was assigned !!	  */

	nl->comp[ip1] = (nl->orient[ip2] == POSITIVE_ORIENTATION)?
			positive_component(nl->nc[ip2]) :
			negative_component(nl->nc[ip2]);
	nl->comp[im1] = (nl->orient[im1] == POSITIVE_ORIENTATION)?
			negative_component(nl->nc[im1]) :
			positive_component(nl->nc[im1]);

	if (((nl->comp[ip1] != (nl->comp[im1]))) &&
	    (nl->nc[ip1] != nl->nc[i]) &&
	    (wave_type(nl->nc[ip1]) == wave_type(nl->nc[i])))
	{
	    for (nlopp = *new_node_list; nlopp != NULL; nlopp = nlopp->next)
	    	if (nlopp->m == nl->nn[i]) break;

	    if (nlopp == NULL)
	    {
	    	(void) printf("WARNING in f_replace_unphys_loop(), "
	    	              "Unable to find opposite node list\n");
	    	DEBUG_LEAVE(f_replace_unphys_loop)
	    	return f_delete_loop(nl,new_node_list,newc,
	    			     fr,i,min_area,flag);
	    }
	    if (total_number_curves_at_node(nlopp->m) < 4)
	    {
	    	(void) printf("WARNING in f_replace_unphys_loop(), "
		              "Opposite node list has < 4 curves\n");
		DEBUG_LEAVE(f_replace_unphys_loop)
		return FUNCTION_FAILED;
	    }

	    for (k = 0; k < 4; k++)
	    	if (nlopp->nc[k] == nl->nc[ip1]) break;

	    km1 = (k+3) % 4;	kp2 = (k + 2) % 4;

	    *newc = make_curve(nl->comp[ip1],nl->comp[im1],nl->m,nl->nn[i]);
	    wave_type(*newc) = wave_type(nl->nc[ip1]);

	    st = (nl->orient[im1] == POSITIVE_ORIENTATION) ?
					left_start_state(nl->nc[im1]) :
					right_end_state(nl->nc[im1]);
	    assign(right_start_state(*newc),st,szst);

	    st = (nl->orient[ip2] == POSITIVE_ORIENTATION) ?
					right_start_state(nl->nc[i]) :
					left_end_state(nl->nc[i]);
	    assign(left_start_state(*newc),st,szst);

	    st = (nlopp->orient[kp2] == POSITIVE_ORIENTATION) ?
					right_start_state(nlopp->nc[kp2]) :
					left_end_state(nlopp->nc[kp2]);
	    assign(right_end_state(*newc),st,szst);

	    st = (nlopp->orient[km1] == POSITIVE_ORIENTATION) ?
					left_start_state(nlopp->nc[km1]) :
					right_end_state(nlopp->nc[km1]);
	    assign(left_end_state(*newc),st,szst);

	    delete_from_nnlist(nlopp,new_node_list);
	}
	(void) delete_curve(nl->nc[ip1]);
	if (nl->nc[ip1] != nl->nc[i])
	    (void) delete_curve(nl->nc[i]);
	
	DEBUG_LEAVE(f_replace_unphys_loop)
	return FUNCTION_SUCCEEDED;
}		/*end f_replace_unphys_loop*/


LOCAL   void delete_from_nnlist(
	NNLIST		*nl,
	NNLIST		**new_node_list)
{
		/* NNLIST members created by calls to Store(). */
		/* All storage will be freed when appropriate  */
		/* interface is deleted. Merely unlink NNLIST  */
		/* 		    element		       */

	DEBUG_ENTER(delete_from_nnlist)
	if ((nl->next == NULL) && (nl->prev == NULL))
	{
		*new_node_list = NULL;
		DEBUG_LEAVE(delete_from_nnlist)
		return;
	}
	if (nl->next == NULL)
	{
		nl->prev->next = NULL;
	} 
 	else if (nl->prev == NULL)
	{
		nl->next->prev = NULL;
		*new_node_list = nl->next;
	}
	else
	{
		nl->next->prev = nl->prev;
		nl->prev->next = nl->next;
	}

	DEBUG_LEAVE(delete_from_nnlist)
}		/*end delete_from_nnlist*/

LOCAL	bool is_node_in_node_list(
	NODE		*node,
	NNLIST		*nnlist)
{
	NNLIST		*nl;

	for (nl = nnlist; nl != NULL; nl = nl->next)
	    if (nl->m == node)
		    return YES;
	
	return NO;
}		/*end is_node_in_node_list*/

LOCAL	void print_NNLIST_struct(
	NNLIST		*nl)
{
	int		i;

	(void) printf("NNLIST %p:  prev %p next %p\n",(POINTER)nl,
		      (POINTER)nl->prev,(POINTER)nl->next);
	(void) printf("\tnode m %llu  pos: %g %g\n\n",
		      node_number(nl->m),
		      Coords(nl->m->posn)[0],Coords(nl->m->posn)[1]);
	for (i = 0; i < 4; i++)
	{
		(void) printf("\ti %d   nc %llu   nn %llu   comp %d\n",
			      i,curve_number(nl->nc[i]),
			      node_number(nl->nn[i]),nl->comp[i]);
		(void) printf("\torient %s   ni %s\n",
			      (nl->orient[i] == POSITIVE_ORIENTATION) ?
			      "POSITIVE ORIENTATION" : "NEGATIVE ORIENTATION",
			      (nl->ni[i] == PHYSICAL) ?
			      		"PHYSICAL" : "UNPHYSICAL");
		if (nl->area[i] == HUGE_VAL)
			(void) printf("\tArea of loop  UNASSIGNED\n\n");
		else
			(void) printf("\tArea of loop %g\n\n",nl->area[i]);
	}
	(void) printf("\tNumber of physical regions %d\n",nl->num_phys);
	(void) printf("\n");
}		/*end print_NNLIST_struct*/

LOCAL 	float area_of_loop(
	NNLIST		*nl,
	int		i)
{
	float		area, a;
	int		j;
	BOND		*b;
	POINT		*p0;

	DEBUG_ENTER(area_of_loop)

	j = (i+1)%4;
	p0 = nl->m->posn;
	area = 0.0;
	a = 0.0;

	for (b = nl->nc[i]->first; b; b = b->next)
		area = area + Cross_pro(p0,b->start,b->end);
	if (nl->nc[i]->first->start != nl->m->posn)
		area = -area;


	if (DEBUG)
		(void) printf("Area %g  curve %llu\n",
			      area,curve_number(nl->nc[i]));

	if (nl->nc[i] != nl->nc[j])
	{
		for (b = nl->nc[j]->first; b; b = b->next)
			a = a + Cross_pro(p0,b->start,b->end);
		if (nl->nc[j]->last->end != nl->m->posn)
			a = -a;

		if (DEBUG)
			(void) printf("A %g  curve %llu\n",
				      a,curve_number(nl->nc[j]));
	}
	area = area + a;

	if (DEBUG)
	 	(void) printf("leaving area_of_loop  area %g\n",area);

	DEBUG_LEAVE(area_of_loop)
	return area;
}		/*end area_of_loop*/



LOCAL 	float Pangle(
	POINT		*p0,
	POINT		*p1,
	POINT		*p2)
{
	float		x,y,a;

	DEBUG_ENTER(Pangle)
	if (DEBUG)
	{
		(void) printf("Entered Pangle\n");
		(void) printf("p0 %llu  x %g  y %g\n",
			      point_number(p0),Coords(p0)[0],Coords(p0)[1]);
		(void) printf("p1 %llu  x %g  y %g\n",
			      point_number(p1),Coords(p1)[0],Coords(p1)[1]);
		(void) printf("p2 %llu  x %g  y %g\n",
			      point_number(p2),Coords(p2)[0],Coords(p2)[1]);
	}

	y = Dot_pro(p0,p1,p2);
	x = Cross_pro(p0,p1,p2);
	a = atan2(x,y);


	if (DEBUG)
		(void) printf("Dot(x): %g  Cross(y): %g  a: %g\n",x,y,a);

	if (a < 0.0)
		a = 2.*PI + a;

	if (DEBUG)
		(void) printf("Return %g\n",a);

	DEBUG_LEAVE(Pangle)
	return a;
}		/*end Pangle*/




LOCAL	bool correct_and_identify_node_types(
	Front		*fr)
{
	INTERFACE	*intfc = fr->interf;
	NODE		**n;
	CURVE 		*c1, *c2, *cur;
	bool		sav_interp;

	DEBUG_ENTER(correct_and_identify_node_types)
restart_loop:
	for (n = intfc->nodes;  n && *n;  n++)
	{
		if (DEBUG)
		{
			(void) printf("Resolving node type at node %llu\n",
				      node_number(*n));
			print_node(*n);
		}

		if ((node_type(*n) != ERROR) &&
		    (node_type(*n) != CLOSED_NODE) &&
		    (node_type(*n) != UNKNOWN_NODE_TYPE))
			continue;
		
			/* identify probable cases */

			/* closed loops */
		if (is_node_of_closed_curve_only(*n))
			node_type(*n) = CLOSED_NODE;

		else if (((*n)->in_curves== NULL) &&
			((*n)->out_curves == NULL))
		{
			(void) delete_node(*n);
			goto restart_loop;
		} 

			/* two shock nodes */
		else if (    (*n)->in_curves && (*n)->out_curves &&
			(*n)->in_curves[0]  && (!(*n)->in_curves[1])
		 && (*n)->out_curves[0] && (!(*n)->out_curves[1]))
		{
		    c1 = (*n)->in_curves[0];
		    c2 = (*n)->out_curves[0];

		    if ((negative_component(c1) != negative_component(c2))
			    			||
			(positive_component(c1) != positive_component(c2)))
		    {
			(void) printf("WARNING in ");
			(void) printf("correct_and_identify_node_types(), ");
			(void) printf("Inconsistent components\n");
			if (DEBUG)
				print_interface(intfc);
			DEBUG_LEAVE(correct_and_identify_node_types)
			return NO;
		    }
		    sav_interp = interpolate_intfc_states(intfc);
		    interpolate_intfc_states(intfc) = YES;
		    cur = join_curves(c1,c2,negative_component(c1),
				      positive_component(c2),(BOND **)NULL);
		    interpolate_intfc_states(intfc) = sav_interp;
		    if (cur == NULL)
		    {
			(void) printf("WARNING in ");
			(void) printf("correct_and_identify_node_types(), ");
			(void) printf("join_curves() returns NULL\n");
		        DEBUG_LEAVE(correct_and_identify_node_types)
			return NO;
		    }
		    (void) delete_node(*n);
		    goto restart_loop;
		}
		else if ((!(*n)->in_curves) &&
			 ((*n)->out_curves[0] && (*n)->out_curves[1]
					 && (!(*n)->out_curves[2])))
		{
		    c1 = (*n)->out_curves[0];
		    c2 = (*n)->out_curves[1];
		    invert_curve(c1);

		    if ((negative_component(c1) != negative_component(c2))
			    			||
			(positive_component(c1) != positive_component(c2)))
		    {
			(void) printf("WARNING in ");
			(void) printf("correct_and_identify_node_types(), ");
			(void) printf("Inconsistent components\n");
			if (DEBUG)
				print_interface(intfc);
			DEBUG_LEAVE(correct_and_identify_node_types)
			return NO;
		    }
		    sav_interp = interpolate_intfc_states(intfc);
		    interpolate_intfc_states(intfc) = YES;
		    cur = join_curves(c1,c2,negative_component(c1),
				      positive_component(c1),(BOND **)NULL);
		    interpolate_intfc_states(intfc) = sav_interp;
		    if (cur == NULL)
		    {
			(void) printf("WARNING in ");
			(void) printf("correct_and_identify_node_types(), ");
			(void) printf("join_curves() returns NULL\n");
		        DEBUG_LEAVE(correct_and_identify_node_types)
			return NO;
		    }
		    (void) delete_node(*n);
		    goto restart_loop;
		}
		else if ((!(*n)->out_curves) &&
			 ((*n)->in_curves[0] && (*n)->in_curves[1]
					 && (!(*n)->in_curves[2])))
		{
		    c1 = (*n)->in_curves[0];
		    c2 = (*n)->in_curves[1];
		    invert_curve(c2);

		    if ((negative_component(c1) != negative_component(c2))
			    			||
			(positive_component(c1) != positive_component(c2)))
		    {
			(void) printf("WARNING in ");
			(void) printf("correct_and_identify_node_types(), ");
			(void) printf("Inconsistent components\n");
			if (DEBUG)
				print_interface(intfc);
			DEBUG_LEAVE(correct_and_identify_node_types)
			return NO;
		    }
		    sav_interp = interpolate_intfc_states(intfc);
		    interpolate_intfc_states(intfc) = YES;
		    cur = join_curves(c1,c2,negative_component(c1),
				      positive_component(c1),(BOND **)NULL);
		    interpolate_intfc_states(intfc) = sav_interp;
		    if (cur == NULL)
		    {
			(void) printf("WARNING in ");
			(void) printf("correct_and_identify_node_types(), ");
			(void) printf("join_curves() returns NULL\n");
		        DEBUG_LEAVE(correct_and_identify_node_types)
			return NO;
		    }
		    (void) delete_node(*n);
		    goto restart_loop;
		}
		else
			(*fr->identify_physical_node)(*n);
	}
	DEBUG_LEAVE(correct_and_identify_node_types)
	return YES;
}		/*end correct_and_identify_node_types*/


/* 
*			eliminate_small_loops():
*
*	This routine finds and eliminates small loops generated by a self-
*	intersecting loop. It is quite crude, and slow, and calls
*	intersections()	after each loop has been eliminated. This routine
*	can be called in the beginning of untangle and reduces the
*	possibility of double crossings. If double crossings don't occur
*	it is a slow and painful way of doing only a part of what the
*	unravel routine can do.
*/

EXPORT	void eliminate_small_loops(
	INTERFACE	*intfc,
	float		hx,
	float		hy,
	CROSS		**cross)
{
	CURVE		*c;
	BOND		*b1, *b2, *bb;
	POINT		*p0;
	CROSS		*cr;
	float		area, min_area = hx*hy;

	DEBUG_ENTER(eliminate_small_loops)
restart:
	if (*cross == NULL)
	{
	    DEBUG_LEAVE(eliminate_small_loops)
	    return;
	}

	for (cr = *cross;  cr->next;  cr = cr->next)
	{
	    if (cr->c1 == cr->c2)
	    {
	    	c = cr->c1;
		p0 = cr->p;

		/* Make sure b2 follows b1 in bond list of curve */

		for (bb = c->first;  bb != c->last;  bb = bb->next)
		{
		    if (bb == cr->b1)
		    {
		    	b1 = cr->b1;	b2 = cr->b2;
		    	break;
		    }
		    if (bb == cr->b2)
		    {
		    	b1 = cr->b2;	b2 = cr->b1;
		    	break;
		    }
		}

		/* Compute area of loop */

		for (area = 0.0, bb = b1->next; bb != b2; bb = bb->next)
		    area += Cross_pro(p0,bb->start,bb->end);
		area = fabs(area);
		if (area > min_area)
		    continue;
			
		/* Delete loop with small area */

		interpolate_intfc_states(c->interface) = YES;
		if (insert_point_in_bond(p0,b2,c) != FUNCTION_SUCCEEDED)
	        {
	            screen("ERROR in eliminate_small_loops(), "
		           "insert_point_in_bond() failed\n");
	            clean_up(ERROR);
	        }
		for (bb = b1->next; bb != b2->next; bb = bb->next)
		    (void) delete_start_of_bond(bb,c);
		if (intersections(intfc,cross,NO) == FUNCTION_FAILED)
		{
		    screen("ERROR in eliminate_small_loops(), "
		           "intersections() failed\n");
		    clean_up(ERROR);
		}
		goto restart;
	    }
	} 
	DEBUG_LEAVE(eliminate_small_loops)
}		/*end eliminate_small_loops*/


LOCAL	void alloc_sn_list(
	Snlist	*snlist)
{
	vector(&snlist->nc    ,snlist->num_c,sizeof(POINTER));
	vector(&snlist->nopp  ,snlist->num_c,sizeof(POINTER));
	vector(&snlist->pt    ,snlist->num_c,sizeof(POINTER));
	vector(&snlist->ang   ,snlist->num_c,FLOAT);
	vector(&snlist->orient,snlist->num_c,INT);
	snlist->nc_set = 0;	snlist->nopp_set = 0;
	snlist->pt_set = 0;	snlist->ang_set = 0;
	snlist->orient_set = 0;
}		/*end alloc_sn_list*/

LOCAL	void free_sn_list(
	Snlist	*snlist)
{
	free(snlist->nc);
	free(snlist->nopp);
	free(snlist->pt);
	free(snlist->ang);
	free(snlist->orient);
	snlist->sn = NULL;
	snlist->num_c = 0;
	snlist->nc_set = 0;	snlist->nopp_set = 0;
	snlist->pt_set = 0;	snlist->ang_set = 0;
	snlist->orient_set = 0;
}		/*end free_sn_list*/

LOCAL	void print_Snlist(
	Snlist		*snlist)
{
	POINT		*p;
	int		i;

	if (snlist == NULL)
	{
		(void) printf("Snlist unallocated\n");
		return;
	}
	if (snlist->num_c <= 0)
	{
		(void) printf("Snlist empty\n");
		return;
	}
	(void) printf("Snlist at node %llu\n",node_number(snlist->sn));
	print_node(snlist->sn);
	for (i = 0;  i < snlist->num_c;  i++)
	{
		if (snlist->nc_set)
			(void) printf("curve %llu ",curve_number(snlist->nc[i]));
		else
			(void) printf("curve NULL ");
		if (snlist->orient_set)
			(void) printf("orient %d ",snlist->orient[i]);
		else
			(void) printf("orient NULL ");
		if (snlist->ang_set)
			(void) printf("ang %g ",snlist->ang[i]);
		else
			(void) printf("ang NULL ");
		if (snlist->nopp_set)
		{
			p = (snlist->nopp[i])->posn;
			(void) printf("opp_node %g %g ",
				      Coords(p)[0],Coords(p)[1]);
		}
		else
			(void) printf("nopp NULL ");
		if (snlist->pt_set)
		{
			p = snlist->pt[i];
			(void) printf("pt %g %g\n",Coords(p)[0],Coords(p)[1]);
		}
		else
			(void) printf("pt NULL\n");
	}
}		/*end print_Snlist*/


LOCAL	void set_curve_list_for_node(
	Snlist		*snl)
{
	CURVE		**ci, **co, *c1;
        NODE		*sn, *m;
	POINT		*p;
	int		i, j;
	ORIENTATION	orient;
	float		a;

	DEBUG_ENTER(set_curve_list_for_node)
	sn = snl->sn;		snl->num_c = 0;
	for (ci = sn->in_curves;  ci && *ci;  ci++)
		(snl->num_c)++;
	for (co = sn->out_curves;  co && *co;  co++)
		(snl->num_c)++;

	if (DEBUG)
	{
		(void) printf("Inserting curve info in Snlist for node -\n");
		print_node(sn);
		(void) printf("Num curves %d\n",snl->num_c);
	}

	alloc_sn_list(snl);

	i = 0;
	for (ci = sn->in_curves;  ci && *ci;  ci++)
	{
		snl->nc[i] = *ci;
		snl->orient[i] = NEGATIVE_ORIENTATION;
		snl->nopp[i] = (*ci)->start;
		snl->pt[i] = (*ci)->last->start;
		i++;
	}
	for (co = sn->out_curves;  co && *co;  co++)
	{
		snl->nc[i] = *co;
		snl->orient[i] = POSITIVE_ORIENTATION;
		snl->nopp[i] = (*co)->end;
		snl->pt[i] = (*co)->first->end;
		i++;
	}
	snl->nc_set = 1;	snl->orient_set = 1;
	snl->nopp_set = 1;	snl->pt_set = 1;
	if (DEBUG)
	{
		(void) printf("Initial Snlist load\n");
		print_Snlist(snl);
	}

	snl->ang[0] = 0.0;
	for (i = 1;  i < snl->num_c;  i++)
		snl->ang[i] = Pangle(sn->posn,snl->pt[0],snl->pt[i]);
	snl->ang_set = 1;

	if (DEBUG)
	{
		(void) printf("After setting angles\n");
		print_Snlist(snl);
	}

		/* Order curves by angle */

	for (i = 1;  i < snl->num_c-1;  i++)
	{
		for (j = i+1;  j < snl->num_c;  j++)
		{
			if (snl->ang[j] < snl->ang[i])
			{
				a      = snl->ang[i];
				m      = snl->nopp[i];
				p      = snl->pt[i];
				c1     = snl->nc[i];
				orient = snl->orient[i];

				snl->ang[i]    = snl->ang[j];	
				snl->nopp[i]   = snl->nopp[j];
				snl->pt[i]     = snl->pt[j];
				snl->nc[i]     = snl->nc[j];
				snl->orient[i] = snl->orient[j];

				snl->ang[j]    = a;
				snl->nopp[j]   = m;
				snl->pt[j]     = p;
				snl->nc[j]     = c1;
				snl->orient[j] = orient;
			}
		}
	}

	if (DEBUG)
	{
		(void) printf("After ordering angles\n");
		print_Snlist(snl);
	}
	DEBUG_LEAVE(set_curve_list_for_node)
}		/*end set_curve_list_for_node*/


/*
*			check_phys_loops_on_snlist():
*
*	Returns YES if all loops physical. Returns NO if unphysical
*	loop found. Also deletes unphysical loop.
*/

LOCAL	bool check_phys_loops_on_snlist(
	Snlist		*snl)
{
	int		i, j, k;

	DEBUG_ENTER(check_phys_loops_on_snlist)
	if (snl->num_c <= 2)
	{
		if (DEBUG)
		{
			(void) printf(" <= 2 curves attached to S_node. ");
			(void) printf("Nothing checked\n");
		}
		DEBUG_LEAVE(check_phys_loops_on_snlist)
		return YES;
	}
	for (i = 0;  i < snl->num_c - 1;  i++)
	{
	    j = (i+1) % snl->num_c;
	    if (DEBUG)
		    (void) printf("i %d j %d ",i,j);

	    if (snl->nc[i] != snl->nc[j])
	    {
		    if (DEBUG)
			    (void) printf("no loop\n");
		    continue;
	    }
	    k = (i+2) % snl->num_c;
	    if (DEBUG)
	    {
		(void) printf("k %d\n",k);
		(void) printf("curve %llu ",curve_number(snl->nc[j]));
		(void) printf("negative component %d positive component %d\n",
			      negative_component((snl->nc[j])),
			      positive_component((snl->nc[j])));
		(void) printf("curve %llu ",curve_number(snl->nc[k]));
		(void) printf("negative component %d positive component %d\n",
			      negative_component((snl->nc[k])),
			      positive_component((snl->nc[k])));
	    }
	    if (snl->orient[j] == POSITIVE_ORIENTATION)
	    {
		if (snl->orient[k] == POSITIVE_ORIENTATION)
		{
		    if (negative_component((snl->nc[j]))
			!= positive_component((snl->nc[k])))
		    {
			(void) delete_curve(snl->nc[i]);
			if (DEBUG)
				(void) printf("Unphys loop %d deleted\n",i);
			DEBUG_LEAVE(check_phys_loops_on_snlist)
			return NO;
		    }
	        }
		else
		{
		    if (negative_component((snl->nc[j]))
			!= negative_component((snl->nc[k])))
		    {
			(void) delete_curve(snl->nc[i]);
			if (DEBUG)
				(void) printf("Unphys loop %d deleted\n",i);
			DEBUG_LEAVE(check_phys_loops_on_snlist)
			return NO;
		    }
	        }
	    }
	    else
	    {
		if (snl->orient[k] == POSITIVE_ORIENTATION)
		{
		    if (positive_component((snl->nc[j]))
			!= positive_component((snl->nc[k])))
		    {
			(void) delete_curve(snl->nc[i]);
			if (DEBUG)
				(void) printf("Unphys loop %d deleted\n",i);
			DEBUG_LEAVE(check_phys_loops_on_snlist)
			return NO;
		    }
	        }
		else
		{
		    if (positive_component((snl->nc[j]))
			!= negative_component((snl->nc[k])))
		    {
			(void) delete_curve(snl->nc[i]);
			if (DEBUG)
				(void) printf("Unphys loop %d deleted\n",i);
			DEBUG_LEAVE(check_phys_loops_on_snlist)
			return NO;
		    }
	        }
	    }
	}
	if (DEBUG)
	   (void) printf("All loops physical\n");
	DEBUG_LEAVE(check_phys_loops_on_snlist)
	return YES;
}		/*end check_phys_loops_on_snlist*/



LOCAL	void check_physical_loops_at_sink_nodes(
	INTERFACE	*intfc)
{
	Snlist		Snlist_store;
	Snlist		*snl = &Snlist_store;
	NODE		**n;

	DEBUG_ENTER(check_physical_loops_at_sink_nodes)
	for (n = intfc->nodes;  n && *n;  n++)
	{
		if (node_type(*n) != SINK_NODE)
			continue;

	reloop_node:
		snl->sn = *n;
		set_curve_list_for_node(snl);
		if (check_phys_loops_on_snlist(snl) == NO)
		{
			free_sn_list(snl);
			goto reloop_node;
		}
		else
			free_sn_list(snl);
	}
	DEBUG_LEAVE(check_physical_loops_at_sink_nodes)
}		/*end check_physical_loops_at_sink_nodes*/




#define GEN_CURVE
#if defined(GEN_CURVE)

/*
*	A generalized curve is a linked group of CURVES satisfying the 
*	following conditions:
*		1) each CURVE has the same wave type
*		2) each interior node has the same node type
*		3) each interior node has exactly 2 (not necessarily distinct)
*		   CURVEs attached to it - the previous CURVE of the
*		   generalized curve and the next one.  (PASSIVE_BOUNDARY
*		   curves at a node are not counted in this total unless
*		   the generalized curve is a PASSIVE_BOUNDARY.)
*		4) the CURVEs have a consistent orientation - i.e.
*		      c1->start => c1->end = c2->start => c2->end ...  or
*		      c1->end <= c1->start = c2->end <= c2->start ... 
*		   Usually the left and right components of the generalized
*		   curve are just the left and right components, respectively,
*		   of any CURVE within it.  This may not be true when 
*		   PASSIVE_BOUNDARYs are present at interior nodes.
*
*	The idea is that a generalized curve would be a single CURVE except
*	that simple NODEs had to be added for some reason (e.g. to preserve the
*	shape of the curve).  Hopefully the above conditions will 
*	identify all such curves and not any unwanted curves.  
*
*	In the future it may be desirable to require consistency of other
*	fields in a CURVE (e.g. start and end states, boundary function,...)
*	Consistency of the boundary flag of CURVEs is intentionally not
*	included since situations like the following should be considered
*	generalized curves:    bdry-->-x  - passive bdry -  x->-bdry
*				       |                    |   
*				       |		    ^
*				       -->--internal bdry ->-
*	
*/

/*
*			opp_node_of_gen_curve():
*
*	Given a generalized curve which has orientation 'orient'
*	relative to one end node (the "beginning") and with "beginning" curve
*	'c', this routine returns the node at the opposite end (the "end") of 
*	the generalized curve.
*	(For a generalized curve containing a single curve, it is
*	equivalent to Node_of(c,Opposite_orient(orient))).
*
*	On error the routine returns NULL.
*/

EXPORT NODE *opp_node_of_gen_curve(
	CURVE		*c,
	ORIENTATION	orient)
{
	int		n_type;	/* node type of interior nodes of gen curve */
	NODE		*endn=NULL;/* node at "end" of each component curve */
	CURVE		*nextc,*prevc;

		/* preliminary error checking */

	DEBUG_ENTER(opp_node_of_gen_curve)
	if (DEBUG)
	{
		(void) printf("Entered opp_node_of_gen_curve()\n");
		(void) printf("c %llu orient %d\n",curve_number(c),orient);
	}
	if (!c) goto Exit;

	/**** check if c really is at beginning?
	      If c is not at the beginning, this routine may find the end
	      of the wrong generalized curve.  Problems can be prevented
	      by initializing n_type properly (not to UNKOWN) when c is not at
	      the beginning.  
	n_type = node_type(Node_of(c,orient));
	prevc = next_curve_of_gen_curve(c,Opposite_orient(orient),
					&n_type,&endn);
	if (!prevc) n_type = UNKNOWN_NODE_TYPE;
	*******/

		/* loop through the curves in the generalized curve 	*/
		/*   until the curve ends or closes back on itself 	*/

	prevc = c;
	nextc = NULL;
	n_type = UNKNOWN_NODE_TYPE;
	while (nextc != c)
	{			/* until gen curve closes */

		nextc = next_curve_of_gen_curve(prevc,orient,&n_type,&endn);
		if (nextc == NULL) break;	/* reached end of gen curve */
		prevc = nextc;
	}

Exit:
	if (DEBUG)
		(void) printf("Leaving opp_node_of_gen_curve() -  endn %llu\n",
			      node_number(endn));
	DEBUG_LEAVE(opp_node_of_gen_curve)
	return endn;

}		/*end opp_node_of_gen_curve*/


/*
*			next_curve_of_gen_curve():
*
*	This routine attempts to find the next curve after curve 'c' in
*	a generalized curve which has orientation 'orient' relative
*	to the "beginning" and has interior node type 'n_type'.
*	The node type at the "beginning" of the continuation curve will
*	not be checked if n_type = UNKNOWN_NODE_TYPE.
*
*	ON EXIT:
*		n_type	- node type of next_n, UNKNOWN_NODE_TYPE if c is NULL
*		next_n	- node at "end" of 'c', NULL if c is NULL
*		Returns the continuation if it is found, NULL otherwise.
*				  (NOTE: c closed => no continuation)
*/

EXPORT CURVE *next_curve_of_gen_curve(
	CURVE		*c,
	ORIENTATION	orient,
	int		*n_type,    /* node type of interior nodes on curve */
	NODE		**next_n) 		/* node at "end" of c */
{
	CURVE		*next_c = NULL;
	CURVE		**cp,**c_array;
	int		nn_type;
	int		nw_type,w_type;
	int		ignore_passive=YES;

		/* preliminary error checking */

	DEBUG_ENTER(next_curve_of_gen_curve)

	*next_n = NULL;
	if (DEBUG)
		(void) printf("c %llu orient %d n_type %d\n",
			      curve_number(c),orient,*n_type);
	if (!c)
	{
		*n_type = UNKNOWN_NODE_TYPE;
		DEBUG_LEAVE(next_curve_of_gen_curve)
		return NULL;
	}

		/* find the end node of c and check (or set) the node type */

	*next_n = Node_of(c,Opposite_orient(orient));
	nn_type = node_type(*next_n);
	if (DEBUG)
		(void) printf("next_n %llu nn_type %d\n",
			      node_number(*next_n),nn_type);
	if (*n_type != UNKNOWN_NODE_TYPE && *n_type != nn_type)
	{
		DEBUG_LEAVE(next_curve_of_gen_curve)
		return NULL;
	}
	*n_type = nn_type;

		/* check for closed curve */

	if (is_closed_curve(c))
	{
		if (DEBUG)
			(void) printf("curve is closed - returning NO\n");
		DEBUG_LEAVE(next_curve_of_gen_curve)
		return NULL;
	}
		
		/* loop through the in_curves or out_curves (depending on */
		/*         orient) to find the next valid curve  	  */

	w_type = wave_type(c);
	if (w_type == PASSIVE_BOUNDARY) ignore_passive=NO;
	if (DEBUG)
		(void) printf("w_type %d ignore_passive %d\n",
			      w_type,ignore_passive);
	if (orient == POSITIVE_ORIENTATION)
	{
		c_array = (*next_n)->out_curves;
		if (DEBUG)
		    (void) printf("searching through out_curves for next_c\n");
	}
	else
	{
		c_array = (*next_n)->in_curves;
		if (DEBUG)
		    (void) printf("searching through in_curves for next_c\n");
	}
	for (cp = c_array; cp && *cp; cp++)
	{
			/* ignore c and (conditionally) passive boundaries */

		if (c == *cp) continue;	/* really unneeded since c not closed*/
		nw_type = wave_type(*cp);
		if (nw_type == PASSIVE_BOUNDARY && ignore_passive) continue;

			/* found a curve - check that it is the only	*/ 
			/* curve at the node and check wave type	*/

		if (DEBUG)
			(void) printf("found curve %llu wave type %d\n",
				      curve_number(*cp),nw_type);
		if (next_c)
		{
			DEBUG_LEAVE(next_curve_of_gen_curve)
			return NULL;
		}
		next_c = *cp;
		if (nw_type != w_type)
		{
			DEBUG_LEAVE(next_curve_of_gen_curve)
			return NULL;
		}
	}
	if (!next_c) 	/* no curves found with proper orient*/
	{
		DEBUG_LEAVE(next_curve_of_gen_curve)
		return NULL;
	}

		/* look through remaining curves at node to make sure there
		*  are no more (non-ignored) curves
		*/

	if (orient == POSITIVE_ORIENTATION)
	{
		c_array = (*next_n)->in_curves;
		if (DEBUG)
			(void) printf("now checking in_curves for extras\n");
	}
	else
	{
		c_array = (*next_n)->out_curves;
		if (DEBUG)
			(void) printf("now checking out_curves for extras\n");
	}
	for (cp = c_array; cp && *cp; cp++)
	{

			/* ignore c and (conditionally) passive boundaries */

		if (*cp == c) continue;
		nw_type = wave_type(*cp);
		if (nw_type == PASSIVE_BOUNDARY && ignore_passive) continue;

		if (DEBUG)
			(void) printf("found extra curve %llu w_type %d\n",
				      curve_number(*cp),nw_type);
		DEBUG_LEAVE(next_curve_of_gen_curve)
		return NULL;
	}
		/* continuation curve found */

	if (DEBUG)
	{
		(void) printf("next_n %llu *n_type %d next_c %llu\n",
			     node_number(*next_n),*n_type,curve_number(next_c));
	}
	DEBUG_LEAVE(next_curve_of_gen_curve)
	return next_c;
}		/*end next_curve_of_gen_curve*/

#else /* defined(GEN_CURVE) */

	/*
	*	The following are versions of the above routines
	*	when "generalized" curves are not allowed.  That is,
	*	when generalized curves are just single CURVEs.
	*/

EXPORT NODE *opp_node_of_gen_curve(
	CURVE		*c,
	ORIENTATION	orient)
{
	return Node_of(c,Opposite_orient(orient));
}		/*end opp_node_of_gen_curve*/

/*ARGSUSED*/
EXPORT CURVE *next_curve_of_gen_curve(
	CURVE		*c,
	ORIENTATION	orient,
	int		*n_type,    /* node type of interior nodes on curve */
	NODE		**next_n)		/* node at "end" of c */
{
	/* no continuation if generalized curve is a single curve */

	*next_n = NULL;
	n_type = UNKNOWN_NODE_TYPE;
	return NULL;
}		/*end next_curve_of_gen_curve*/

#endif /* defined(GEN_CURVE) */
#endif /* defined(TWOD) */
