/*
*			fscat3d2.c
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/

#if defined(THREED)
#define DEBUG_STRING    "fscatter"
#include <front/flocaldecs.h>

	/* LOCAL Function Declarations */
LOCAL	INTERFACE *cut_buf_interface2(INTERFACE*,int,int,int*,int*);
LOCAL	bool      append_adj_intfc_to_buffer2(INTERFACE*,INTERFACE*,RECT_GRID*,
					      RECT_GRID*,int,int);
LOCAL	bool      append_buffer_surface2(SURFACE*,SURFACE*,RECT_GRID*,RECT_GRID*,
				         int,int,P_LINK*,int);
LOCAL	bool      buffer_extension3d2(INTERFACE*,INTERFACE*,int,int,bool);
LOCAL	bool      tri_in_matching_strip(TRI*,float,float,int);
LOCAL	bool      append_reflected_surface(SURFACE*,SURFACE*,RECT_GRID*,
				           RECT_GRID*,int,int,P_LINK*,int);
LOCAL	bool      append_rfl_intfc_to_buffer(INTERFACE*,INTERFACE*,
				    	     RECT_GRID*,RECT_GRID*,int,int);
LOCAL	bool      is_reflected_tri(TRI*,TRI*,int,int,RECT_GRID*,int,int*);
LOCAL	bool      match_tris_in_block(SURFACE*,SURFACE*,RECT_GRID*,TRI**,TRI**,
				      int,int,P_LINK*,int);
LOCAL	bool      matching_tris(TRI*,TRI*,int*,float*);
LOCAL	bool      point_on_same_edge(POINT*,POINT*,float*);
LOCAL	bool      reconstruct_tris_in_rfl_block(SURFACE*,TRI**,TRI**,int*,
	 				        RECT_GRID*,int,int);
LOCAL	bool      reflect_buffer_interface(INTERFACE*,int,int,bool);
LOCAL	bool      tri_on_bound(TRI*,float,int);
LOCAL	bool      tri_out_domain2(TRI*,float*,float*,int,int);
LOCAL	bool      tri_side_on_bound(TRI*,float,int,int*);
LOCAL	float     dist2_between_tris(TRI*,TRI*,int*);
LOCAL	void      open_null_sides2(INTERFACE*,float*,float*,int,int);
LOCAL	void      set_floating_point_tolerance2(RECT_GRID*);
LOCAL	void      detach_tri_side(TRI*,int);
LOCAL	void      merge_block_tris(SURFACE*,TRI**,TRI**,int);
LOCAL	void      stitch_blocks(TRI**,int,TRI**,int);
LOCAL   bool      is_reflected_point_pair(POINT*,POINT*,float,int,float);

LOCAL	float	ltol[3];/*LINE TOLERANCE*/

/*ARGSUSED*/
EXPORT bool f_intfc_communication3d2(
	Front		*fr)
{
	INTERFACE	*intfc = fr->interf;
	INTERFACE	*adj_intfc[2], *sav_intfc, *buf_intfc;
	PP_GRID		*pp_grid = fr->pp_grid;
	int		me[MAXD], him[MAXD];
	int		myid, dst_id;
	int		*G;
	int		i,j,k;
	int		dim = intfc->dim;
	bool		sav_copy;
	bool            status = FUNCTION_SUCCEEDED;

	DEBUG_ENTER(f_intfc_communication3d2)

	set_floating_point_tolerance2(fr->rect_grid);
	sav_copy = copy_intfc_states();
	sav_intfc = current_interface();
	set_copy_intfc_states(YES);

	myid = pp_mynode();
	G = pp_grid->gmax;
	find_Cartesian_coordinates(myid,pp_grid,me);

	if (DEBUG)
	{
	    (void) printf("myid = %d, ",myid);
	    print_int_vector("me = ",me,dim,"\n");
	    print_PP_GRID_structure(pp_grid);
	    (void) printf("Input interface:\n");
	    print_interface(intfc);
	}

		/* Extend interface in three directions */
	clip_intfc_at_grid_bdry2(intfc);

	if (DEBUG && debugging("consistency"))
	{
	    null_sides_are_consistent();
	    if (!consistent_interface(intfc))
	    {
		screen("ERROR in f_intfc_communication3d2(), "
		       "grid boundary clipped interface is inconsistent\n");
		clean_up(ERROR);
	    }
	}

	for (i = 0; i < dim; ++i)
	{
	    for (j = 0; j < 2; ++j)
	    {
	    	pp_gsync();
		for (k = 0; k < dim; ++k)
		    him[k] = me[k];

		if (rect_boundary_type(intfc,i,j) == SUBDOMAIN_BOUNDARY)
	        {
		    him[i] = me[i] + 2*j - 1;
		    him[i] = (him[i]+G[i])%G[i];
	        }
		else if (rect_boundary_type(intfc,i,j) == REFLECTION_BOUNDARY)
		{
		    status = reflect_buffer_interface(intfc,i,j,status);
		    set_current_interface(intfc);
		}

	        /*Send buffer region to adjacent domain if necessary*/
                if (rect_boundary_type(intfc,i,j) == SUBDOMAIN_BOUNDARY)
                {
		    dst_id = domain_id(him,G,dim);
		    if (!Is_outside(him,G,i))
		    {
			buf_intfc = cut_buf_interface2(intfc,i,j,me,him);
			if (me[i] == him[i])
			    adj_intfc[(j+1)%2] = buf_intfc;
		    	else
			{
			    send_interface(buf_intfc,dst_id);
			    (void) delete_interface(buf_intfc);
			}
			if (DEBUG)
			{
			    (void) printf("Interface sent to dst_id = %d,",
					  dst_id);
			    print_int_vector("him = ",him,dim,"\n");
			}
		    }
	        }

	        /*Receive adjacent buffer region if necessary*/
                if (rect_boundary_type(intfc,i,(j+1)%2) == SUBDOMAIN_BOUNDARY)
                {
		    him[i] = me[i] - 2*j + 1;
		    him[i] = (him[i]+G[i])%G[i];
		    dst_id = domain_id(him,G,dim);
		    if (!Is_outside(him,G,i))
		    {
		        if (me[i] != him[i])
			    adj_intfc[(j+1)%2] = receive_interface(dst_id);
    
		        if (DEBUG)
		        {
			    (void) printf("Interface received from "
					  "dst_id = %d,",dst_id);
			    print_int_vector("him = ",him,dim,"\n");
		        }
		    }
	        }
	    }
	    for (j = 0; j < 2; ++j)
	    {
		if (rect_boundary_type(intfc,i,j) == SUBDOMAIN_BOUNDARY)
	        {
		    him[i] = me[i] - 2*j + 1;
		    him[i] = (him[i]+G[i])%G[i];
		    if (!Is_outside(him,G,i))
		    {
		        status = buffer_extension3d2(intfc,adj_intfc[j],
						     i,j,status);
			if (!status)
			{
			  (void) printf("WARNING in "
					"f_intfc_communication3d2 "
					"buffer_extension3d2 failed for "
					"i = %d, j = %d\n",i,j);
			}
		        (void) delete_interface(adj_intfc[j]);
		        set_current_interface(intfc);
		    }
	        }
	    }
	    reset_intfc_num_points(intfc);
	}

	if (status == FUNCTION_SUCCEEDED)
	{
	    install_subdomain_bdry_curves(intfc);
	    reset_intfc_num_points(intfc);
	    if (DEBUG)
	    {
	    	(void) printf("Final intfc:\n");
	    	print_interface(intfc);
	    }
	}
	else
	{
	    (void) printf("WARNING in f_intfc_communication3d2(), "
			  "failure status produced\n");
	}

	set_copy_intfc_states(sav_copy);
	set_current_interface(sav_intfc);
	DEBUG_LEAVE(f_intfc_communication3d2)
	return status;
}	/*end f_intfc_communication3d2 */

EXPORT void clip_front_for_output(
        Front           *fr,
	RECT_GRID       *zoom_grid)
{
	INTERFACE    *intfc = fr->interf;
        INTERFACE    *sav_intfc;
        bool         sav_copy;
	float	     *VL = zoom_grid->VL;
	float	     *VU = zoom_grid->VU;
	float	     *h = zoom_grid->h;
        float        l[MAXD],u[MAXD];
        int          nb,dir, dim = intfc->dim;

	sav_copy = copy_intfc_states();
        set_copy_intfc_states(YES);
        sav_intfc = current_interface();
        strip_subdomain_bdry_curves(intfc);
        set_current_interface(intfc);

	if (!interface_reconstructed(intfc))
	{
	    set_floating_point_tolerance1(fr->rect_grid->h);
	    for (dir = 0; dir < dim; ++dir)
            {
            	for (nb = 0; nb < 2; ++nb)
                    open_null_sides1(intfc,VL,VU,dir,nb);
            }
	}
	else
	{
	    set_floating_point_tolerance2(fr->rect_grid);
	    for (dir = 0; dir < dim; ++dir)
	    {
	    	l[dir] = zoom_grid->VL[dir] - 0.5*h[dir];
	    	u[dir] = zoom_grid->VU[dir] + 0.5*h[dir];
	    }
	    for (dir = 0; dir < dim; ++dir)
            {
            	for (nb = 0; nb < 2; ++nb)
                    open_null_sides2(intfc,l,u,dir,nb);
            }
	}
	cut_out_curves_in_buffer(intfc);
	reset_intfc_num_points(intfc);
	set_current_interface(sav_intfc);
	install_subdomain_bdry_curves(intfc);
        reset_intfc_num_points(intfc);
	set_copy_intfc_states(sav_copy);
}	/* end clip_front_for_output */


LOCAL bool buffer_extension3d2(
	INTERFACE	*intfc,
	INTERFACE	*adj_intfc,
	int		dir,
	int		nb,
	bool		status)
{
	BOND		*b;
	NODE		**n;
	CURVE		**c;
	HYPER_SURF	*hs;
	HYPER_SURF_ELEMENT *hse;
	POINT		*p;
	RECT_GRID	*gr = computational_grid(intfc);
	RECT_GRID	*adj_gr = computational_grid(adj_intfc);
	RECT_GRID	dual_gr;
	float		T[MAXD];
	int		i;
	int		dim = intfc->dim;

	DEBUG_ENTER(buffer_extension3d2)

	set_current_interface(intfc);

		/* Set periodic shift */

	for (i = 0; i < dim; ++i)
	    T[i] = 0;
	if (nb == 0)				/* Lower neighbor */
	    T[dir] = gr->L[dir] - adj_gr->U[dir];
	else					/* Upper neighbor */
	    T[dir] = gr->U[dir] - adj_gr->L[dir];

		/* Shift points on interface */

	for (n = adj_intfc->nodes; n && *n; ++n)
	    Coords((*n)->posn)[dir] += T[dir];
	for (c = adj_intfc->curves; c && *c; ++c)
	    for (b = (*c)->first; b != (*c)->last; b = b->next)
	    	Coords(b->end)[dir] += T[dir];
	(void) next_point(adj_intfc,NULL,NULL,NULL);
	while (next_point(adj_intfc,&p,&hse,&hs))
	    Coords(p)[dir] += T[dir];

	set_dual_grid(&dual_gr,gr);

		/* Patch tris from adj_intfc to intfc */
	if(!append_adj_intfc_to_buffer2(intfc,adj_intfc,gr,&dual_gr,dir,nb))
	{
	    status = FUNCTION_FAILED;
	    (void) printf("WARNING: in buffer_extension3d2(), "
	    	          "append_adj_intfc_to_buffer2() failed\n");
	}
	DEBUG_LEAVE(buffer_extension3d2)
	return status;
}	/*end buffer_extension3d2*/

LOCAL 	bool append_adj_intfc_to_buffer2(
	INTERFACE	*intfc,		/* my interface 	       */
	INTERFACE	*adj_intfc,	/* received interface	       */
	RECT_GRID	*grid,		/* Rectangular grid for region */
	RECT_GRID	*dual_gr,	/* Dual grid for region        */
	int		dir,
	int		nb)
{
	INTERFACE *cur_intfc;
	P_LINK	  *p_table;	/* Table of matching points on intfc
					 * and adj_intfc*/
	SURFACE	  **s, **as;
	int	  p_size;		/*Size of space allocated for p_table*/
	bool      status;
	bool	  corr_surf_found;

	DEBUG_ENTER(append_adj_intfc_to_buffer2)

	cur_intfc = current_interface();
	set_current_interface(intfc);

	p_size = 4*(adj_intfc->num_points) + 1;
	vector(&p_table,p_size,sizeof(P_LINK));
	reset_hash_table(p_table,p_size);

	/* Begin patching adj_intfc to current interface */
	status = YES;
	for (as = adj_intfc->surfaces; as && *as; ++as)
	{
	    corr_surf_found = NO;
	    for (s = intfc->surfaces; s && *s; ++s)
	    {
		/*
		*  COMMENT -
		*  The Hyper_surf_index() function is not
		*  fully supported.  This will fail in the
		*  presence of interface changes in topology
		*  TODO: FULLY SUPPORT THIS OBJECT
		*/
		if (Hyper_surf_index(*s) == Hyper_surf_index(*as))
		{
		    corr_surf_found = YES;
		    if (!append_buffer_surface2(*s,*as,grid,dual_gr,dir,nb,
						p_table,p_size))
		    {
			(void) printf("WARNING in "
			              "append_adj_intfc_to_buffer2(), "
			              "append surface failed\n");
			status = NO;
		    }
		}
	    }
	    if (!corr_surf_found)
	    {
		SURFACE *surf;
		surf = copy_buffer_surface(*as,p_table,p_size);
		Hyper_surf_index(surf) = Hyper_surf_index((*as));
	    }
	}
	free(p_table);
	set_current_interface(cur_intfc);
	if (debugging("consistency"))
	{
	    null_sides_are_consistent();
	    if (!consistent_interface(intfc))
	    {
		screen("ERROR in append_adj_intfc_to_buffer2(), "
		       "intfc is inconsistent\n");
		clean_up(ERROR);
	    }
	}
	DEBUG_LEAVE(append_adj_intfc_to_buffer2)
	return status;
}		/*end append_adj_intfc_to_buffer2*/

LOCAL void set_floating_point_tolerance2(
	RECT_GRID *gr)
{
	int   i;
	float crx_tol;

	crx_tol = line_cross_tolerance(gr);
	for (i = 0; i < 3; ++i)
	    ltol[i] = crx_tol; /*TOLERANCE*/
}		/*end set_floating_point_tolerance2*/

EXPORT	float	line_cross_tolerance(
	RECT_GRID *gr)
{
	float		hmin;
	int		i;

	for (hmin = gr->h[0], i = 1; i < 3; ++i)
	    if (hmin > gr->h[i])
		hmin = gr->h[i];
	return 1.0e-6*hmin; /*TOLERANCE*/
}		/*end line_cross_tolerance*/

LOCAL bool append_buffer_surface2(
	SURFACE	  *surf,
	SURFACE	  *adj_surf,
	RECT_GRID *grid,
	RECT_GRID *dual_gr,
	int	  dir,
	int	  nb,
	P_LINK	  *p_table,
	int	  p_size)
{
	TRI	**tris_s,**tris_a;
	TRI	*tri;
	int	i, ns, na, gmax[MAXD];
	float	crx_l, crx_u;
        int     idir1, idir2;
        int     i1, i2;
        int     **nbt_s, **nbt_a;
        int     *lbuf = dual_gr->lbuf;
        int     *ubuf = dual_gr->ubuf;
        TRI     **tri_store_s, **tri_store_a;
        TRI     ****blk_tri_s, ****blk_tri_a;

	DEBUG_ENTER(append_buffer_surface2)

	idir1 = (dir+1)%3;
        idir2 = (dir+2)%3;
        gmax[dir] = dual_gr->gmax[dir];
        gmax[idir1] = dual_gr->gmax[idir1] + lbuf[idir1] + ubuf[idir1] + 2;
        gmax[idir2] = dual_gr->gmax[idir2] + lbuf[idir2] + ubuf[idir2] + 2;

	vector(&tris_s,surf->num_tri,sizeof(TRI *));
	vector(&tris_a,adj_surf->num_tri,sizeof(TRI *));
	matrix(&nbt_s,gmax[idir1],gmax[idir2],INT);
        matrix(&nbt_a,gmax[idir1],gmax[idir2],INT);
        matrix(&blk_tri_s,gmax[idir1],gmax[idir2],sizeof(TRI**));
        matrix(&blk_tri_a,gmax[idir1],gmax[idir2],sizeof(TRI**));

	crx_l = (nb == 0) ? grid->L[dir] - 0.5*grid->h[dir] : 
			    grid->U[dir] - 0.5*grid->h[dir];
	crx_u = (nb == 0) ? grid->L[dir] + 0.5*grid->h[dir] : 
			    grid->U[dir] + 0.5*grid->h[dir];

	ns = 0;
	for (tri = first_tri(surf); !at_end_of_tri_list(tri,surf);
		tri = tri->next)
	{
	    if (tri_in_matching_strip(tri,crx_l,crx_u,dir))
	    {
	    	tris_s[ns++] = tri;
		assign_tri_icoords(dual_gr,tri);
                i1 = Tri_icoords(tri)[idir1] + lbuf[idir1] + 1;
                i2 = Tri_icoords(tri)[idir2] + lbuf[idir2] + 1;
                ++nbt_s[i1][i2];
	    }
	}

	na = 0;
	for (tri = first_tri(adj_surf); !at_end_of_tri_list(tri,adj_surf);
		tri = tri->next)
	{
	    if (tri_in_matching_strip(tri,crx_l,crx_u,dir))
	    {
	    	tris_a[na++] = tri;
		assign_tri_icoords(dual_gr,tri);
                i1 = Tri_icoords(tri)[idir1] + lbuf[idir1] + 1;
                i2 = Tri_icoords(tri)[idir2] + lbuf[idir2] + 1;
                ++nbt_a[i1][i2];
	    }
	}

	vector(&tri_store_s,ns,sizeof(TRI*));
        vector(&tri_store_a,na,sizeof(TRI*));

	ns = na = 0;
        for (i1 = 0; i1 < gmax[idir1]; ++i1)
        {
            for (i2 = 0; i2 < gmax[idir2]; ++i2)
            {
                if (nbt_s[i1][i2] != 0)
                {
                    blk_tri_s[i1][i2] = tri_store_s+ns;
                    ns += nbt_s[i1][i2];
                    nbt_s[i1][i2] = 0;
                }
                if (nbt_a[i1][i2] != 0)
                {
                    blk_tri_a[i1][i2] = tri_store_a+na;
                    na += nbt_a[i1][i2];
                    nbt_a[i1][i2] = 0;
                }
            }
        }
	for (i = 0; i < ns; ++i)
        {
            i1 = Tri_icoords(tris_s[i])[idir1] + lbuf[idir1] + 1;
            i2 = Tri_icoords(tris_s[i])[idir2] + lbuf[idir2] + 1;
            blk_tri_s[i1][i2][nbt_s[i1][i2]] = tris_s[i];
            ++nbt_s[i1][i2];
        }
        for (i = 0; i < na; ++i)
        {
            i1 = Tri_icoords(tris_a[i])[idir1] + lbuf[idir1] + 1;
            i2 = Tri_icoords(tris_a[i])[idir2] + lbuf[idir2] + 1;
            blk_tri_a[i1][i2][nbt_a[i1][i2]] = tris_a[i];
            ++nbt_a[i1][i2];
        }

	for (i1 = 0; i1 < gmax[idir1]; ++i1)
        {
            for (i2 = 0; i2 < gmax[idir2]; ++i2)
            {
                if (nbt_s[i1][i2] != nbt_a[i1][i2])
                {
		    POINT *p;
		    int j;
                    (void) printf("WARNING in append_buffer_surface2(), "
                                  "local and adjacent sides have "
                                  "different number of tris\n");
		    (void) printf("i1 = %d  i2 = %d\n",i1,i2);
		    (void) printf("Local number of tris: %d\n",
					nbt_s[i1][i2]);
		    for (i = 0; i < nbt_s[i1][i2]; ++i)
		    {
			for (j = 0; j < 3; ++j)
			{
			    p = Point_of_tri(blk_tri_s[i1][i2][i])[j];
			    (void) printf("%g %g %g\n",Coords(p)[0],
						       Coords(p)[1],
						       Coords(p)[2]);
			}
			(void) printf("\n");
		    }
		    (void) printf("Adjacent number of tris: %d\n",
					nbt_a[i1][i2]);
		    for (i = 0; i < nbt_a[i1][i2]; ++i)
		    {
			for (j = 0; j < 3; ++j)
			{
			    p = Point_of_tri(blk_tri_a[i1][i2][i])[j];
			    (void) printf("%g %g %g\n",Coords(p)[0],
						       Coords(p)[1],
						       Coords(p)[2]);
			}
			(void) printf("\n");
		    }
	            free_these(8,nbt_s,nbt_a,blk_tri_s,blk_tri_a,tris_s,tris_a,
                               tri_store_s,tri_store_a);
		    DEBUG_LEAVE(append_buffer_surface2)
                    return FUNCTION_FAILED;
                }
                if (nbt_s[i1][i2] == 0)
		    continue;
                if (!match_tris_in_block(surf,adj_surf,grid,
					 blk_tri_s[i1][i2],blk_tri_a[i1][i2],
					 nbt_s[i1][i2],nb,p_table,p_size))
                {
                    (void) printf("WARNING in append_buffer_surface2(), "
                                  "match_tris_in_block() failed\n");
	            free_these(8,nbt_s,nbt_a,blk_tri_s,blk_tri_a,tris_s,tris_a,
                               tri_store_s,tri_store_a);
		    DEBUG_LEAVE(append_buffer_surface2)
                    return FUNCTION_FAILED;
                }
            }
        }

	adj_surf = copy_buffer_surface(adj_surf,p_table,p_size);

        for (i1 = 0; i1 < gmax[idir1]; ++i1)
        {
            for (i2 = 0; i2 < gmax[idir2]; ++i2)
            {
                nbt_s[i1][i2] = 0;
                nbt_a[i1][i2] = 0;
	    }
	}
	na = 0;
	for (tri = first_tri(adj_surf); !at_end_of_tri_list(tri,adj_surf);
		tri = tri->next)
	{
	    if (tri_in_matching_strip(tri,crx_l,crx_u,dir))
	    {
	    	tris_a[na++] = tri;
		assign_tri_icoords(dual_gr,tri);
	    }
	}
	for (i = 0; i < ns; ++i)
        {
            i1 = Tri_icoords(tris_s[i])[idir1] + lbuf[idir1] + 1;
            i2 = Tri_icoords(tris_s[i])[idir2] + lbuf[idir2] + 1;
            blk_tri_s[i1][i2][nbt_s[i1][i2]] = tris_s[i];
            ++nbt_s[i1][i2];
        }
        for (i = 0; i < na; ++i)
        {
            i1 = Tri_icoords(tris_a[i])[idir1] + lbuf[idir1] + 1;
            i2 = Tri_icoords(tris_a[i])[idir2] + lbuf[idir2] + 1;
            blk_tri_a[i1][i2][nbt_a[i1][i2]] = tris_a[i];
            ++nbt_a[i1][i2];
        }

	/*adjoin adj_surf tri list to surf tri list */
	last_tri(surf)->next = first_tri(adj_surf);
	first_tri(adj_surf)->prev = last_tri(surf);
	link_tri_list_to_surface(first_tri(surf),last_tri(adj_surf),surf);
	adj_surf->pos_curves = adj_surf->neg_curves = NULL;
	(void) delete_surface(adj_surf);
	adj_surf = NULL;

	for (i1 = 0; i1 < gmax[idir1]; ++i1)
        {
            for (i2 = 0; i2 < gmax[idir2]; ++i2)
            {
                if (nbt_s[i1][i2] == 0)
		    continue;
		merge_block_tris(surf,blk_tri_s[i1][i2],
			         blk_tri_a[i1][i2],nbt_s[i1][i2]);
	    }
	}

	free_these(8,nbt_s,nbt_a,blk_tri_s,blk_tri_a,tris_s,tris_a,
                   tri_store_s,tri_store_a);
	DEBUG_LEAVE(append_buffer_surface2)
	return YES;
}		/*end append_buffer_surface2*/


LOCAL	bool tri_in_matching_strip(
	TRI   *tri,
	float crx_l,
	float crx_u,
	int   dir)
{
	float tri_center;
	int   i;

	tri_center = 0.0;
	for (i = 0; i < 3; ++i)
	    tri_center += Coords(Point_of_tri(tri)[i])[dir];
	tri_center /= 3.0;
	if (((crx_l - ltol[dir]) <= tri_center        ) && 
	    ((        tri_center <= (crx_u + ltol[dir]))))
	    return YES;
	else
	    return NO;
}		/*end tri_in_matching_strip*/

LOCAL void open_null_sides2(
	INTERFACE	*intfc,
	float		*L,
	float		*U,
	int		dir,
	int		nb)
{
	TRI		*tri;
	SURFACE 	**s;

	DEBUG_ENTER(open_null_sides2)
	if (DEBUG)
	{
	    static const char *xyz[] = { "x", "y", "z"};
	    static const char *sname[] = { "lower", "upper"};
	    (void) printf("Clipping interface in at %s %s side\n",
			  sname[nb],xyz[dir]);
	}

	for (s = intfc->surfaces; s && *s; ++s)
	{
	    for (tri=first_tri(*s); !at_end_of_tri_list(tri,*s); tri=tri->next)
	    {
	    	if (tri_out_domain2(tri,L,U,dir,nb))
	    	{
	    	    remove_out_domain_tri(tri,*s);
	    	}
	    }
	}
	for (s = intfc->surfaces; s && *s; ++s)
	{
	    if (no_tris_on_surface(*s))
	    {
	    	(void) delete_surface(*s);
		--s;
	    }
	}
	DEBUG_LEAVE(open_null_sides2)
}		/*end open_null_sides2*/

LOCAL bool tri_out_domain2(
	TRI		*tri,
	float		*L,
	float		*U,
	int		dir,
	int		nb)
{
	float		tri_center, LB, UB;
	int		i;

	tri_center = 0.0;
	for (i = 0; i < 3; ++i)
	    tri_center += Coords(Point_of_tri(tri)[i])[dir];
	tri_center /= 3.0;
	if (nb == 0)
	{
	    LB = L[dir] - ltol[dir];
	    if (tri_center <= LB)
		return YES;
	}
	else
	{
	    UB = U[dir] + ltol[dir];
	    if (tri_center >= UB)
		return YES;
	}
	return NO;
}	/* end tri_out_domain2 */

LIB_LOCAL void clip_intfc_at_grid_bdry2(
	INTERFACE	*intfc)
{
	INTERFACE	*cur_intfc = current_interface();
	RECT_GRID	*gr = computational_grid(intfc);
	int		dim = intfc->dim;
	int		dir,nb;
	float		L[3],U[3];

	DEBUG_ENTER(clip_intfc_at_grid_bdry2)
	strip_subdomain_bdry_curves(intfc);
	set_current_interface(intfc);
	for (dir = 0; dir < dim; ++dir)
	{
	    L[dir] = gr->L[dir] - 0.5*gr->h[dir];
	    U[dir] = gr->U[dir] + 0.5*gr->h[dir];
	}
	for (dir = 0; dir < dim; ++dir)
	{
	    for (nb = 0; nb < 2; ++nb)
	    {
	    	open_null_sides2(intfc,L,U,dir,nb);
	    }
	}
	reset_intfc_num_points(intfc);
	set_current_interface(cur_intfc);
	DEBUG_LEAVE(clip_intfc_at_grid_bdry2)
}		/*end clip_intfc_at_grid_bdry2*/

LOCAL	INTERFACE  *cut_buf_interface2(
	INTERFACE	*intfc,
	int		dir,
	int		nb,
	int		*me,
	int		*him)
{
	INTERFACE	*sav_intfc, *tmp_intfc, *buf_intfc;
	RECT_GRID	*gr = computational_grid(intfc);
	RECT_GRID	dual_gr;
	float		L[3], U[3];
	bool		sav_copy = copy_intfc_states();

	set_copy_intfc_states(YES);
	sav_intfc = current_interface();

	set_size_of_intfc_state(size_of_state(intfc));
	tmp_intfc = copy_interface(intfc);
	set_dual_grid(&dual_gr,gr);

	if (nb == 0)
	{
	    L[dir] = gr->L[dir] - 0.5*gr->h[dir];
	    U[dir] = gr->L[dir] + (gr->L[dir] - dual_gr.VL[dir]) + gr->h[dir];
	}
	else
	{
	    L[dir] = gr->U[dir] - (dual_gr.VU[dir] - gr->U[dir]) - gr->h[dir];
	    U[dir] = gr->U[dir] + 0.5*gr->h[dir];
	}

	open_null_sides2(tmp_intfc,L,U,dir,(nb+1)%2);

	cut_out_curves_in_buffer(tmp_intfc);

	reset_intfc_num_points(tmp_intfc);

	if (me[dir] == him[dir])
	{
	    buf_intfc = tmp_intfc;
	}
	else
	{
	    set_size_of_intfc_state(size_of_state(intfc));
	    buf_intfc = copy_interface(tmp_intfc);
	    (void) delete_interface(tmp_intfc);
	}
	set_copy_intfc_states(sav_copy);
	set_current_interface(sav_intfc);

	if (debugging("consistency"))
	{
	    null_sides_are_consistent();
	    if (!consistent_interface(buf_intfc))
	    {
		screen("ERROR in cut_buf_interface2(), "
		       "cut interface is inconsistent\n");
		clean_up(ERROR);
	    }
	}

	return buf_intfc;

}		/*end cut_buf_interface2*/

LOCAL	bool  reflect_buffer_interface(
	INTERFACE *intfc,
	int dir,
	int nb,
	bool status)
{
	INTERFACE	*rfl_intfc,*sav_intfc;
	RECT_GRID       *gr = computational_grid(intfc);
	RECT_GRID       dual_gr;
	float           L[3],U[3];
	bool         sav_copy = copy_intfc_states();
	float		nor[3],posn[3];
	int		i,dim = intfc->dim;

	DEBUG_ENTER(reflect_buffer_interface)

	set_copy_intfc_states(YES);
	sav_intfc = current_interface();
	set_size_of_intfc_state(size_of_state(intfc));
	set_dual_grid(&dual_gr,gr);
	rfl_intfc = copy_interface(intfc);
	if (nb == 0)
	{
	    L[dir] = dual_gr.L[dir];
	    U[dir] = dual_gr.L[dir] + (dual_gr.ubuf[dir] + 2)*dual_gr.h[dir];
	}
	else
	{
	    L[dir] = dual_gr.U[dir] - (dual_gr.lbuf[dir] + 2)*dual_gr.h[dir];
	    U[dir] = dual_gr.U[dir];
	}
	open_null_sides2(rfl_intfc,L,U,dir,(nb+1)%2);
	if (rfl_intfc->surfaces == NULL)
	{
	    delete_interface(rfl_intfc);
	    set_copy_intfc_states(sav_copy);
	    set_current_interface(sav_intfc);
	    DEBUG_LEAVE(reflect_buffer_interface);
	    return FUNCTION_SUCCEEDED;
	}

	reset_intfc_num_points(rfl_intfc);

	/* Reflect buffer interface */

	for (i = 0; i < dim; ++i)
	    nor[i] = posn[i] = 0.0;

	/* Reflect buffer interface */

	nor[dir] = 1.0;
	posn[dir] = (nb == 0) ? gr->L[dir] : gr->U[dir];
	reflect_interface(rfl_intfc,posn,nor);

	if (!append_rfl_intfc_to_buffer(intfc,rfl_intfc,gr,&dual_gr,dir,nb))
	{
	    status = FUNCTION_FAILED;
	    (void) printf("WARNING in reflect_buffer_interface(), "
	                  "append_rfl_intfc_to_buffer() failed\n");
	    return status;
	}
	delete_interface(rfl_intfc);
	set_copy_intfc_states(sav_copy);
	set_current_interface(sav_intfc);

	DEBUG_LEAVE(reflect_buffer_interface)
	return status;
}	/* end reflect_buffer_interface */

LOCAL	bool append_rfl_intfc_to_buffer(
	INTERFACE *intfc,
	INTERFACE *rfl_intfc,
	RECT_GRID *gr,
	RECT_GRID *dual_gr,
	int	  dir,
	int	  nb)
{
	INTERFACE	*cur_intfc;
	P_LINK		*p_table;	/* Table of matching points on intfc
					 * and rfl_intfc*/
	SURFACE		**s, **rs;
	int		p_size;		/*Size of space allocated for p_table*/

	DEBUG_ENTER(append_rfl_intfc_to_buffer)

	cur_intfc = current_interface();
	set_current_interface(intfc);

	p_size = 4*(rfl_intfc->num_points) + 1;
	vector(&p_table,p_size,sizeof(P_LINK));
	reset_hash_table(p_table,p_size);

	/* Begin patching rfl_intfc to current interface */
	for (s = intfc->surfaces; s && *s; ++s)
	{
	    for (rs = rfl_intfc->surfaces; rs && *rs; ++rs)
	    {
		/*
		*	COMMENT -
		*	The Hyper_surf_index() function is not
		*	fully supported.  This will fail in the
		*	presences of interface changes in topology
		*	TODO: FULLY SUPPORT THIS OBJECT
		*/
		if (Hyper_surf_index(*s) == Hyper_surf_index(*rs))
		{
		    if (!append_reflected_surface(*s,*rs,gr,dual_gr,dir,nb,
					p_table,p_size))
		    {
			set_current_interface(cur_intfc);
			(void) printf("WARNING in ");
			(void) printf("append_rfl_intfc_to_buffer(), ");
			(void) printf("append surface failed\n");
			DEBUG_LEAVE(append_rfl_intfc_to_buffer)
			return FUNCTION_FAILED;
		    }
		}
	    }
	}
	free(p_table);
	set_current_interface(cur_intfc);
	DEBUG_LEAVE(append_rfl_intfc_to_buffer)
	return FUNCTION_SUCCEEDED;
}	/* end append_rfl_intfc_to_buffer */

LOCAL	bool append_reflected_surface(
	SURFACE		*surf,
	SURFACE         *rfl_surf,
	RECT_GRID       *gr,
	RECT_GRID       *dual_gr,
	int             dir,
	int             nb,
	P_LINK          *p_table,
	int             p_size)
{
	TRI             **tris_s,**tris_r;
	TRI             *tri;
	int             i,ns,nr,gmax[MAXD];
	float           crx_l,crx_u;
	int		idir1,idir2;
	int		i1,i2;
	int		**nbt_s,**nbt_r;
	int		*lbuf = dual_gr->lbuf;
	int		*ubuf = dual_gr->ubuf;
	TRI		**tri_store_s,**tri_store_r;
	TRI		****blk_tri_s,****blk_tri_r;
	TRI             *tris1[8],*tris2[8];

	DEBUG_ENTER(append_reflected_surface)

	idir1 = (dir+1)%3;
	idir2 = (dir+2)%3;
	gmax[dir] = dual_gr->gmax[dir];
	gmax[idir1] = dual_gr->gmax[idir1] + lbuf[idir1] + ubuf[idir1] + 2;
	gmax[idir2] = dual_gr->gmax[idir2] + lbuf[idir2] + ubuf[idir2] + 2;

	vector(&tris_s,rfl_surf->num_tri,sizeof(TRI *));
	vector(&tris_r,rfl_surf->num_tri,sizeof(TRI *));
	matrix(&nbt_s,gmax[idir1],gmax[idir2],INT);
	matrix(&nbt_r,gmax[idir1],gmax[idir2],INT);
	matrix(&blk_tri_s,gmax[idir1],gmax[idir2],sizeof(TRI**));
	matrix(&blk_tri_r,gmax[idir1],gmax[idir2],sizeof(TRI**));

	crx_l = (nb == 0) ? gr->L[dir] - 0.5*gr->h[dir] :
			gr->U[dir] - 0.5*gr->h[dir];
	crx_u = (nb == 0) ? gr->L[dir] + 0.5*gr->h[dir] :
			gr->U[dir] + 0.5*gr->h[dir];

	rfl_surf = copy_buffer_surface(rfl_surf,p_table,p_size);

	ns = 0;
	for (tri = first_tri(surf); !at_end_of_tri_list(tri,surf);
			tri = tri->next)
	{
	    if (tri_in_matching_strip(tri,crx_l,crx_u,dir))
	    {
		tris_s[ns++] = tri;
		assign_tri_icoords(dual_gr,tri);
		i1 = Tri_icoords(tri)[idir1] + lbuf[idir1] + 1;
		i2 = Tri_icoords(tri)[idir2] + lbuf[idir2] + 1;
		++nbt_s[i1][i2];
	    }
	}
	nr = 0;
	for (tri = first_tri(rfl_surf); !at_end_of_tri_list(tri,rfl_surf);
			tri = tri->next)
	{
	    if (tri_in_matching_strip(tri,crx_l,crx_u,dir))
	    {
		tris_r[nr++] = tri;
		assign_tri_icoords(dual_gr,tri);
		i1 = Tri_icoords(tri)[idir1] + lbuf[idir1] + 1;
		i2 = Tri_icoords(tri)[idir2] + lbuf[idir2] + 1;
		++nbt_r[i1][i2];
	    }
	}
	vector(&tri_store_s,ns,sizeof(TRI*));
	vector(&tri_store_r,nr,sizeof(TRI*));

	/* adjoin rfl_surf tri list to surf tri list */

	last_tri(surf)->next = first_tri(rfl_surf);
	first_tri(rfl_surf)->prev = last_tri(surf);
	link_tri_list_to_surface(first_tri(surf),last_tri(rfl_surf),surf);
	rfl_surf->pos_curves = rfl_surf->neg_curves = NULL;
	(void) delete_surface(rfl_surf);
	rfl_surf = NULL;

	ns = nr = 0;
	for (i1 = 0; i1 < gmax[idir1]; ++i1)
	{
	    for (i2 = 0; i2 < gmax[idir2]; ++i2)
	    {
		if (nbt_s[i1][i2] != 0)
		{
		    blk_tri_s[i1][i2] = &tri_store_s[ns];
		    ns += nbt_s[i1][i2];
		    nbt_s[i1][i2] = 0;
		}
		if (nbt_r[i1][i2] != 0)
		{
		    blk_tri_r[i1][i2] = &tri_store_r[nr];
		    nr += nbt_r[i1][i2];
		    nbt_r[i1][i2] = 0;
		}
	    }
	}
	for (i = 0; i < ns; ++i)
	{
	    i1 = Tri_icoords(tris_s[i])[idir1] + lbuf[idir1] + 1;
	    i2 = Tri_icoords(tris_s[i])[idir2] + lbuf[idir2] + 1;
	    blk_tri_s[i1][i2][nbt_s[i1][i2]] = tris_s[i];
	    ++nbt_s[i1][i2];
	}
	for (i = 0; i < nr; ++i)
	{
	    i1 = Tri_icoords(tris_r[i])[idir1] + lbuf[idir1] + 1;
	    i2 = Tri_icoords(tris_r[i])[idir2] + lbuf[idir2] + 1;
	    blk_tri_r[i1][i2][nbt_r[i1][i2]] = tris_r[i];
	    ++nbt_r[i1][i2];
	}

	for (i1 = 0; i1 < gmax[idir1]; ++i1)
	{
	    for (i2 = 0; i2 < gmax[idir2]; ++i2)
	    {
		if (nbt_s[i1][i2] != nbt_r[i1][i2])
		{
		    (void) printf("WARNING in append_reflected_surface(), "
				  "local and reflected sides have "
		                  "different number of tris\n");
		    return FUNCTION_FAILED;
		}
		if (nbt_s[i1][i2] == 0) continue;
		if (!reconstruct_tris_in_rfl_block(surf,blk_tri_s[i1][i2],
			blk_tri_r[i1][i2],&nbt_s[i1][i2],gr,dir,nb))
		{
		    (void) printf("WARNING in append_reflected_surface(), "
				  "reconstruct_tris_in_rfl_block() failed\n");
		    return FUNCTION_FAILED;
		}
	    }
	}
	for (i1 = 0; i1 < gmax[idir1]; ++i1)
	{
	    for (i2 = 0; i2 < gmax[idir2]; ++i2)
	    {
		if (nbt_s[i1][i2] == 0) continue;
		if ((i1 != gmax[idir1] -1) && (nbt_s[i1+1][i2] != 0))
		{
		    ns = 0;
		    for (i = 0; i < nbt_s[i1][i2]; ++i)
		    {
			tris1[ns++] = blk_tri_s[i1][i2][i];
			tris1[ns++] = blk_tri_r[i1][i2][i];
		    }
		    nr = 0;
		    for (i = 0; i < nbt_s[i1+1][i2]; ++i)
		    {
			tris2[nr++] = blk_tri_s[i1+1][i2][i];
			tris2[nr++] = blk_tri_r[i1+1][i2][i];
		    }
		    stitch_blocks(tris1,ns,tris2,nr);
		}
		if ((i2 != gmax[idir2] -1) && (nbt_s[i1][i2+1] != 0))
		{
		    ns = 0;
		    for (i = 0; i < nbt_s[i1][i2]; ++i)
		    {
			tris1[ns++] = blk_tri_s[i1][i2][i];
			tris1[ns++] = blk_tri_r[i1][i2][i];
		    }
		    nr = 0;
		    for (i = 0; i < nbt_s[i1][i2+1]; ++i)
		    {
			tris2[nr++] = blk_tri_s[i1][i2+1][i];
			tris2[nr++] = blk_tri_r[i1][i2+1][i];
		    }
		    stitch_blocks(tris1,ns,tris2,nr);
		}
	    }
	}

	free_these(8,nbt_s,nbt_r,blk_tri_s,blk_tri_r,tris_s,tris_r,
			tri_store_s,tri_store_r);

	DEBUG_LEAVE(append_reflected_surface)
	return FUNCTION_SUCCEEDED;
}	/* end append_reflected_surface */

LOCAL	bool reconstruct_tris_in_rfl_block(
	SURFACE	*surf,
	TRI **tris_s,
	TRI **tris_r,
	int *nt,
	RECT_GRID *gr,
	int dir,
	int nb)
{
	TRI   *ts, *tr;
	POINT **ps, **pr;
	float bs;
	int   side, rside;
	int   i, j, j1, j2;

	bs = (nb == 0) ? gr->L[dir] + 0.5*gr->h[dir] : 
			 gr->U[dir] - 0.5*gr->h[dir];

	for (i = 0; i < *nt; ++i)
	{
	    ts = tris_s[i];
	    if (ts == NULL)
		continue;
	    ps = Point_of_tri(ts);
	    if (tri_on_bound(ts,bs,dir))
		continue;
	    else if (tri_side_on_bound(ts,bs,dir,&side))
	    {
		for (j = 0; j < *nt; ++j)
		{
		    tr = tris_r[j];
		    if (tr == NULL)
			continue;
	            pr = Point_of_tri(tr);
		    if (is_reflected_tri(ts,tr,dir,nb,gr,side,&rside))
		    {
			ps[(side+2)%3]  = pr[(rside+1)%3];
			pr[(rside+2)%3] = ps[(side+1)%3];
			Tri_on_side(ts,( side+1)%3) = tr;
			Tri_on_side(tr,(rside+1)%3) = ts;
			detach_tri_side(ts,(side+2)%3);
			detach_tri_side(tr,(rside+2)%3);
			set_normal_of_tri(ts);
			set_normal_of_tri(tr);
			break;
		    }
		}
	    }
	    else
	    {
		for (j = 0; j < *nt; ++j)
		{
		    tr = tris_r[j];
		    if (tr == NULL) continue;
		    if (is_reflected_tri(ts,tr,dir,nb,gr,side,&rside))
		    {
			remove_tri_from_surface(tr,surf,NO);
			tris_r[j] = NULL;
			break;
		    }
		}
		remove_tri_from_surface(ts,surf,NO);
		tris_s[i] = NULL;
	    }
	}
	j1 = 0;
	for (i = 0; i < *nt; ++i)
	{
	    if (tris_s[i] != NULL)
	    {
		tris_s[j1++] = tris_s[i];
	    }
	}
	j2 = 0;
	for (i = 0; i < *nt; ++i)
	{
	    if (tris_r[i] != NULL)
	    {
	    	tris_r[j2++] = tris_r[i];
	    }
	}
	if (j1 != j2)
	{
	    (void) printf("WARNING in reconstruct_tris_in_rfl_block(), "
	                  "Final number of triangles do not match "
	                  "j1 = %d  j2 = %d\n",j1,j2);
	    return FUNCTION_FAILED;
	}
	*nt = j1;
	return FUNCTION_SUCCEEDED;
}	/* end reconstruct_tris_in_rfl_block */

LOCAL	bool tri_on_bound(
	TRI *tri,
	float line,
	int dir)
{
	if (fabs(Coords(Point_of_tri(tri)[0])[dir] - line) < ltol[dir] &&
	    fabs(Coords(Point_of_tri(tri)[1])[dir] - line) < ltol[dir] &&
	    fabs(Coords(Point_of_tri(tri)[2])[dir] - line) < ltol[dir])
		return YES;
	else
		return NO;
}	/* end tri_on_bound */

LOCAL	bool tri_side_on_bound(
	TRI *tri,
	float line,
	int dir,
	int *side)
{
	POINT *p[3];
	int i;

	p[0] = Point_of_tri(tri)[0];
	p[1] = Point_of_tri(tri)[1];
	p[2] = Point_of_tri(tri)[2];
	for (i = 0; i < 3; ++i)
	{
	    if (fabs(Coords(p[i])[dir] - line) < ltol[dir] &&
		fabs(Coords(p[(i+1)%3])[dir]-line) < ltol[dir])
	    {
		if (fabs(Coords(p[(i+2)%3])[dir]-line) < ltol[dir])
		    return NO;	/* parallel triangle */
		else
		{
		    *side = i;
		    return YES;
		}
	    }
	}
	*side = 0;
	return NO;
}	/* end tri_side_on_bound */

LOCAL	bool is_reflected_tri(
	TRI *ts,
	TRI *tr,
	int dir,
	int nb,
	RECT_GRID *gr,
	int side,
	int *rside)
{
	float ml = (nb == 0) ? 2.0*gr->L[dir] : 2.0*gr->U[dir];
	POINT *sp[3];
	POINT *rp[3];
	float h = gr->h[dir];
	int i;

	for (i = 0; i < 3; ++i)
	{
	    sp[i] = Point_of_tri(ts)[(i+side)%3];
	    rp[i] = Point_of_tri(tr)[i];
	}
	for (i = 0; i < 3; ++i)
	{
	    if (is_reflected_point_pair(sp[0],rp[(i+1)%3],ml,dir,h) &&
		is_reflected_point_pair(sp[1],rp[i],ml,dir,h) &&
		is_reflected_point_pair(sp[2],rp[(i+2)%3],ml,dir,h))
	    {
		*rside = i;
		return YES;
	    }
	}
	return NO;
}	/* end is_reflected_tri */

LOCAL	bool is_reflected_point_pair(
	POINT *ps,
	POINT *pr,
	float ml,
	int dir,
	float h)
{
	if (fabs(ml - Coords(ps)[dir] - Coords(pr)[dir]) < 0.00001*h &&
	    fabs(Coords(ps)[(dir+1)%3] - Coords(pr)[(dir+1)%3]) < 0.00001*h &&
	    fabs(Coords(ps)[(dir+2)%3] - Coords(pr)[(dir+2)%3]) < 0.00001*h)
	    return YES;
	else return NO;
}	/* end is_reflected_point_pair */

LOCAL	void stitch_blocks(
	TRI **tris1,
	int nt1,
	TRI **tris2,
	int nt2)
{
	TRI *tri1, *tri2;
	int i,j;

	for (i = 0; i < nt1; ++i)
	{
	    tri1 = tris1[i];
	    for (j = 0; j < nt2; ++j)
	    {
	    	tri2 = tris2[j];
		link_neighbor_tris(tri1,tri2);
	    }
	}
}	/* end stitch_blocks */

LOCAL	void detach_tri_side(
	TRI *tri,
	int side)
{
	if (!is_side_bdry(tri,side))
	{
	    TRI *nbtri = Tri_on_side(tri,side);
	    if (nbtri == NULL) return;
	    else if (Tri_on_side01(nbtri) == tri)
		Tri_on_side01(nbtri) = NULL;
	    else if (Tri_on_side12(nbtri) == tri)
		Tri_on_side12(nbtri) = NULL;
	    else if (Tri_on_side20(nbtri) == tri)
		Tri_on_side20(nbtri) = NULL;
	}
	else
	{
	    BOND_TRI *bt = Bond_tri_on_side(tri,side);
	    (void) delete_from_pointers(bt,&Btris(bt->bond));
	}
	Neighbor_on_side(tri,side) = NULL;
}	/* end detach_tri_side */

LOCAL   bool match_tris_in_block(
        SURFACE   *ss,
        SURFACE   *sa,
	RECT_GRID *gr,
        TRI       **tris_s,
        TRI       **tris_a,
        int       nt,
        int       nb,
	P_LINK    *p_table,
	int       p_size)
{
	TRI                *ts, *ta;
	int                i, j, jmin, k, i_rot;
	HYPER_SURF_ELEMENT *hse, *hae;
        HYPER_SURF         *hs, *ha;
	POINT              *ps, *pa,*pp;
	float              min_d2;
	static float       **dist = NULL;
	static int         **rot = NULL;
	static int         n_alloc = 0;

	DEBUG_ENTER(match_tris_in_block)
	if (nt > n_alloc)
	{
	    if (dist)
		free_these(2,dist,rot);
	    n_alloc = 2*nt;
	    matrix(&dist,n_alloc,n_alloc,FLOAT);
	    matrix(&rot,n_alloc,n_alloc,INT);
	}
	for (i = 0; i < nt; ++i)
	    for (j = 0; j < nt; ++j)
		dist[i][j] = dist2_between_tris(tris_s[i],tris_a[j],&rot[i][j]);
	for (i = 0; i < nt; ++i)
	{
	    ts = tris_s[i];
	    jmin = -1;
	    min_d2 = HUGE_VAL;
	    for (j = 0; j < nt; ++j)
	    {
		if (tris_a[j] != NULL)
		{
		    if (dist[i][j] < min_d2)
		    {
			min_d2 = dist[i][j];
			jmin = j;
		    }
		}
	    }
	    if (jmin == -1)
	    {
		screen("ERROR in match_tris_in_block(), "
		       "can't find closest tri to tri %llu\n",
		       tri_number(ts,ss->interface));
		(void) printf("ltol = %g %g %g\n",ltol[0],ltol[1],ltol[2]);
		print_tri(ts,ss->interface);
	        for (k = 0, j = 0; j < nt; ++j)
		    if (tris_a[j])
			++k;
		(void) printf("%d Remaining candidates\n",k);
	        for (j = 0; j < nt; ++j)
		{
		    if (tris_a[j])
			print_tri(tris_a[j],sa->interface);
		}
		clean_up(ERROR);
	    }
	    ta = tris_a[jmin];
	    if (matching_tris(ts,ta,&i_rot,gr->h))
	    {
		if (i_rot != rot[i][jmin])
		{
		    screen("ERROR in match_tris_in_block(), "
		           "inconsistent rotatations for tri ts %llu "
			   "and tri ta %llu\n",
		           tri_number(ts,ss->interface),
		           tri_number(ta,sa->interface));
		    (void) printf("i_rot = %d, rot[%d][%d] = %d\n",
				  i_rot,i,jmin,rot[i][jmin]);
		    (void) printf("dist[%d][%d] = %g\n",i,jmin,dist[i][jmin]);
		    (void) printf("ltol = %g %g %g\n",ltol[0],ltol[1],ltol[2]);
		    (void) printf("tri ts\n");
		    print_tri(ts,ss->interface);
		    (void) printf("tri ta\n");
		    print_tri(ta,sa->interface);
		    clean_up(ERROR);
		}
		if (i_rot != 0)
		    rotate_triangle(ts,(nb==0)?3-i_rot:i_rot);
		for (k = 0; k < 3; ++k)
		{
		    ps = Point_of_tri(ts)[k];
                    pa = Point_of_tri(ta)[k];
		    pp = (POINT*)find_from_hash_table((POINTER)pa,
                                                      p_table,p_size);
		    if (pp == NULL)
		    {
			(void) add_to_hash_table((POINTER)pa,(POINTER)ps,
                                                 p_table,p_size);
			hse = Hyper_surf_element(ts);
                        hae = Hyper_surf_element(ta);
			hs = Hyper_surf(ss);
			ha = Hyper_surf(sa);
			(void) average_points(NO,ps,hse,hs,pa,hae,ha);
		    }
		}
		set_normal_of_tri(ts);
		tris_s[i] = NULL;
		tris_a[jmin] = NULL;
	    }
	    else
	    {
		(void) printf("WARNING in match_tris_in_block(), "
			      "unmatched tri found in match_tris_in_block\n");
		(void) printf("Unmatched tri %llu\n",
			      tri_number(ts,ss->interface));
		(void) printf("rot[%d][%d] = %d\n",i,jmin,rot[i][jmin]);
		(void) printf("dist[%d][%d] = %g\n",i,jmin,dist[i][jmin]);
		(void) printf("ltol = %g %g %g\n",ltol[0],ltol[1],ltol[2]);
		print_tri(ts,ss->interface);
	        for (k = 0, j = 0; j < nt; ++j)
		    if (tris_a[j])
			++k;
		(void) printf("%d Remaining candidates\n",k);
	        for (j = 0; j < nt; ++j)
		{
		    if (tris_a[j])
			print_tri(tris_a[j],sa->interface);
		}
		(void) printf("Aborting matching\n");
	        DEBUG_LEAVE(match_tris_in_block)
		return NO;
	    }
	}
	DEBUG_LEAVE(match_tris_in_block)
	return YES;
	
}	/* end match_tris_in_block */

LOCAL	float dist2_between_tris(
	TRI *tri1,
	TRI *tri2,
	int *rot)
{
	float *p1[3], *p2[3];
	float d2, min_d2;

	if ((tri1 == NULL) || (tri2 == NULL))
	{
	    *rot = 0;
	    return HUGE_VAL;
	}

	p1[0] = Coords(Point_of_tri(tri1)[0]);
	p1[1] = Coords(Point_of_tri(tri1)[1]);
	p1[2] = Coords(Point_of_tri(tri1)[2]);

	p2[0] = Coords(Point_of_tri(tri2)[0]);
	p2[1] = Coords(Point_of_tri(tri2)[1]);
	p2[2] = Coords(Point_of_tri(tri2)[2]);

	min_d2 = sqr(p1[0][0]-p2[0][0])+sqr(p1[0][1]-p2[0][1])+sqr(p1[0][2]-p2[0][2]) +
	         sqr(p1[1][0]-p2[1][0])+sqr(p1[1][1]-p2[1][1])+sqr(p1[1][2]-p2[1][2]) +
	         sqr(p1[2][0]-p2[2][0])+sqr(p1[2][1]-p2[2][1])+sqr(p1[2][2]-p2[2][2]);
	*rot = 0;

	d2 = sqr(p1[0][0]-p2[1][0])+sqr(p1[0][1]-p2[1][1])+sqr(p1[0][2]-p2[1][2]) +
	     sqr(p1[1][0]-p2[2][0])+sqr(p1[1][1]-p2[2][1])+sqr(p1[1][2]-p2[2][2]) +
	     sqr(p1[2][0]-p2[0][0])+sqr(p1[2][1]-p2[0][1])+sqr(p1[2][2]-p2[0][2]);
	if (d2 < min_d2)
	{
	    min_d2 = d2;
	    *rot = 1;
	}
	d2 = sqr(p1[0][0]-p2[2][0])+sqr(p1[0][1]-p2[2][1])+sqr(p1[0][2]-p2[2][2]) +
	     sqr(p1[1][0]-p2[0][0])+sqr(p1[1][1]-p2[0][1])+sqr(p1[1][2]-p2[0][2]) +
	     sqr(p1[2][0]-p2[1][0])+sqr(p1[2][1]-p2[1][1])+sqr(p1[2][2]-p2[1][2]);
	if (d2 < min_d2)
	{
	    min_d2 = d2;
	    *rot = 2;
	}
	return min_d2;
}		/*end dist2_between_tris*/

LOCAL	bool matching_tris(
	TRI   *tri1,
	TRI   *tri2,
	int   *i_rot,
	float *h)
{
	POINT *p1[3], *p2[3];
	int i;

	p1[0] = Point_of_tri(tri1)[0];
	p1[1] = Point_of_tri(tri1)[1];
	p1[2] = Point_of_tri(tri1)[2];

	p2[0] = Point_of_tri(tri2)[0];
	p2[1] = Point_of_tri(tri2)[1];
	p2[2] = Point_of_tri(tri2)[2];

	for (i = 0; i < 3; ++i)
	{
	    if (point_on_same_edge(p1[0],p2[i],h) &&
		point_on_same_edge(p1[1],p2[(i+1)%3],h) &&
		point_on_same_edge(p1[2],p2[(i+2)%3],h))
	    {
		*i_rot = i;
		return YES;
	    }
		
	}
	return NO;
}	/* end matching_tris */

LOCAL	bool point_on_same_edge(
	POINT *p1,
	POINT *p2,
	float *h)
{
	int i, num_same_coords;

	num_same_coords = 0;
	for (i = 0; i < 3; ++i)
	{
	    if (fabs(Coords(p1)[i] - Coords(p2)[i]) > 0.5*h[i])
		return NO;
	    else if (fabs(Coords(p1)[i] - Coords(p2)[i]) < ltol[i])
		++num_same_coords;
	}
	return (num_same_coords >= 2) ? YES : NO;
}

LOCAL	void merge_block_tris(
	SURFACE *s,
	TRI **tris_s,
	TRI **tris_a,
	int nt)
{
	int i,j;
	TRI *ts,*ta;

	for (i = 0; i < nt; ++i)
	{
	    ts = tris_s[i];
	    for (j = 0; j < nt; ++j)
	    {
		ta = tris_a[j];
		if (ta == NULL)
		    continue;
		if (Point_of_tri(ts)[0] == Point_of_tri(ta)[0] &&
		    Point_of_tri(ts)[1] == Point_of_tri(ta)[1] &&
		    Point_of_tri(ts)[2] == Point_of_tri(ta)[2])
		{
		    merge_two_tris(ts,ta,s,s);
		    tris_a[j] = NULL;
		}
	    }
	}
}	/* end merge_block_tris */
#endif /* defined(THREED) */
