/*
*			fscat3d1.c
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*/

#if defined(THREED)

#define DEBUG_STRING    "fscatter"
#include <front/fdecs.h>

struct _POINT_LIST {
	POINT              *p;
	HYPER_SURF         *hs;
	HYPER_SURF_ELEMENT *hse;
	struct _POINT_LIST *prev, *next;
};
typedef struct _POINT_LIST POINT_LIST;

struct _POINT_LIST_STORE {
	POINT_LIST *pl;
	int        len;
};
typedef struct _POINT_LIST_STORE POINT_LIST_STORE;


	/* LOCAL Function Declarations */
LOCAL	POINT_LIST	*set_point_list(TRI**,int,HYPER_SURF*,
					POINT_LIST_STORE*);
LOCAL	TRI*	find_following_null_tri(TRI*,POINT**,int*,ANGLE_DIRECTION);
LOCAL	bool	add_matching_pt_to_hash_table(TRI**,TRI**,int,int,SURFACE*,
					      SURFACE*,P_LINK*,int);
LOCAL	bool	buffer_extension3d1(INTERFACE*,INTERFACE*,int,int,bool);
LOCAL	bool    f_intfc_communication3d1(Front*);
LOCAL	bool	match_tris_at_subdomain_bdry(SURFACE*,SURFACE*,TRI**,TRI**,
	                                     int,int);
LOCAL	bool	match_two_tris(TRI*,TRI*);
LOCAL	bool	null_side_on_surface(SURFACE*,TRI**,int*);
LOCAL	bool	tri_cross_line(TRI*,float,int);
LOCAL	bool	tri_out_domain1(TRI*,float*,float*,int,int);
LOCAL	int	append_adj_intfc_to_buffer1(INTERFACE*,INTERFACE*,
					   RECT_GRID*,int,int);
LOCAL	int	append_buffer_surface1(SURFACE*,SURFACE*,RECT_GRID*,int,int,
				      P_LINK*,int);
LOCAL	void	clip_intfc_at_grid_bdry1(INTERFACE*);
LOCAL	void	copy_tri_state_to_btri(BOND_TRI*,BOND*,ORIENTATION,size_t);
LOCAL	void	merge_point_pointers_at_subdomain_bdry(TRI**,TRI**,
						       int,P_LINK*,int);
LOCAL	void	shift_interface(INTERFACE*,float,int);
LOCAL	void	strip_curve_from_surf(CURVE*,SURFACE*,ORIENTATION);
LOCAL	void	synchronize_tris_at_subdomain_bdry(TRI**,TRI**,int,P_LINK*,int);
LOCAL	INTERFACE	*cut_buf_interface1(INTERFACE*,int,int,int*,int*);

LOCAL	float	tol1[MAXD]; /*TOLERANCE*/

#if defined(USE_OVERTURE)
LOCAL   bool    f_form_patch_subintfc_via_cut3d1(Front*);
LOCAL   bool    f_form_patch_subintfc_via_cut3d2(Front*);
#endif /* if defined(USE_OVERTURE) */

/*ARGSUSED*/
EXPORT bool f_intfc_communication3d(
	Front		*fr)
{
	if (interface_reconstructed(fr->interf))
	    return f_intfc_communication3d2(fr);
	else
	    return f_intfc_communication3d1(fr);

}	/* end f_intfc_communication3d */


LOCAL	bool f_intfc_communication3d1(
	Front		*fr)
{
	INTERFACE    *intfc = fr->interf;
	INTERFACE    *adj_intfc[2], *sav_intfc, *buf_intfc;
	PP_GRID	     *pp_grid = fr->pp_grid;
	RECT_GRID    *gr = fr->rect_grid;
	bool	     sav_copy;
	bool      status = FUNCTION_SUCCEEDED;
	float        *U = gr->U, *L = gr->L;
	float        *nor, p[3];
	float        T;
	int	     me[MAXD], him[MAXD];
	int	     myid, dst_id;
	int	     *G;
	int	     i, j, k;
	int	     dim = intfc->dim;
	static float nors[] = {  1.0,  0.0,  0.0,
			         0.0,  1.0,  0.0,
			         0.0,  0.0,  1.0,
			        -1.0,  0.0,  0.0,
			         0.0, -1.0,  0.0,
			         0.0,  0.0, -1.0};

	DEBUG_ENTER(f_intfc_communication3d1)

	set_floating_point_tolerance1(fr->rect_grid->h);
	sav_copy = copy_intfc_states();
	sav_intfc = current_interface();
	set_copy_intfc_states(YES);

	myid = pp_mynode();
	G = pp_grid->gmax;
	find_Cartesian_coordinates(myid,pp_grid,me);

	if (DEBUG)
	{
	    (void) printf("myid = %d, ",myid);
	    print_int_vector("me = ",me,dim,"\n");
	    print_PP_GRID_structure(pp_grid);
	    (void) printf("Input interface:\n");
	    print_interface(intfc);
	}

		/* Extend interface in three directions */
	clip_intfc_at_grid_bdry1(intfc);

	for (i = 0; i < dim; ++i)
	{
	    adj_intfc[0] = adj_intfc[1] = NULL;
	    for (j = 0; j < 2; ++j)
	    {
	    	pp_gsync();

		for (k = 0; k < dim; ++k)
		    him[k] = me[k];

		if (rect_boundary_type(intfc,i,j) == SUBDOMAIN_BOUNDARY)
		{
		    him[i] = (me[i] + 2*j - 1 + G[i])%G[i];
		    dst_id = domain_id(him,G,dim);
		    buf_intfc = cut_buf_interface1(intfc,i,j,me,him);
		    if ((j == 0) && (me[i] == 0))
		    {
			T = gr->GU[i] - gr->GL[i];
	                shift_interface(buf_intfc,T,i);
		    }
		    else if ((j == 1) && (me[i] == (G[i]-1)))
		    {
			T = gr->GL[i] - gr->GU[i];
	                shift_interface(buf_intfc,T,i);
		    }
		    if (dst_id != myid)
		    {
		    	send_interface(buf_intfc,dst_id);
		        (void) delete_interface(buf_intfc);
		    }
		    else
		        adj_intfc[(j+1)%2] = buf_intfc;
		}
		else if (rect_boundary_type(intfc,i,j) == REFLECTION_BOUNDARY)
		{
		    nor = nors + 3*i + 9*j;
		    buf_intfc = cut_buf_interface1(intfc,i,j,me,him);
		    p[i] = (j > 0) ? U[i] : L[i];
		    for (k = 1; k < dim; ++k)
			p[(k+i)%dim] = 0.5*(U[(k+i)%dim] + L[(k+i)%dim]);
		    reflect_interface(buf_intfc,p,nor);
		    adj_intfc[j] = buf_intfc;
		}
		if (rect_boundary_type(intfc,i,(j+1)%2) == SUBDOMAIN_BOUNDARY)
		{
		    him[i] = (me[i] - 2*j + 1 + G[i])%G[i];
		    dst_id = domain_id(him,G,dim);
		    if (dst_id != myid)
			adj_intfc[(j+1)%2] = receive_interface(dst_id);
		}
	    }
	    for (j = 0; j < 2; ++j)
	    {
		if (adj_intfc[j] != NULL)
		{
		    status = buffer_extension3d1(intfc,adj_intfc[j],i,j,status);
		    (void) delete_interface(adj_intfc[j]);
		    set_current_interface(intfc);
		}
	    }
	    reset_intfc_num_points(intfc);
	}

	if (status == FUNCTION_SUCCEEDED)
	{
	    install_subdomain_bdry_curves(intfc);
	    reset_intfc_num_points(intfc);
	    if (DEBUG)
	    {
	    	(void) printf("Final intfc:\n");
	    	print_interface(intfc);
	    }
	}

	set_copy_intfc_states(sav_copy);
	set_current_interface(sav_intfc);
	DEBUG_LEAVE(f_intfc_communication3d1)
	return status;
}	/*end f_intfc_communication3d1 */

EXPORT void install_subdomain_bdry_curves(
	INTERFACE	*intfc)
{
	BOND	 *b;
	BOND_TRI *btri, *bte;
	CURVE	 *curve;
	NODE	 *ns, *ne, **n;
	POINT	 *p, *ps, *pe;
	TRI	 *tri_start, *tri_end;
	SURFACE	 **s;
	bool  sav_intrp;
	int	 side_start, side_end;
	int	 dim = intfc->dim;
	size_t	 sizest = size_of_state(intfc);

	DEBUG_ENTER(install_subdomain_bdry_curves)
	sav_intrp = interpolate_intfc_states(intfc);
	interpolate_intfc_states(intfc) = NO;
	for (s = intfc->surfaces; s && *s; ++s)
	{
	    while (null_side_on_surface(*s,&tri_start,&side_start))
	    {
		p = ps = Point_of_tri(tri_start)[side_start];
		if ((ns = node_of_point(ps,intfc)) == NULL)
	    	    ns = make_node(ps);
		pe = Point_of_tri(tri_start)[Next_m3(side_start)];
		if ((ne = node_of_point(pe,intfc)) == NULL)
		    ne = make_node(pe);
	    	curve = make_curve(NO_COMP,NO_COMP,ns,ne);
	    	hsbdry_type(curve) = SUBDOMAIN_HSBDRY;
	    	install_curve_in_surface_bdry(*s,curve,POSITIVE_ORIENTATION);
	    	btri = link_tri_to_bond(NULL,tri_start,*s,curve->first,curve);
	    	copy_tri_state_to_btri(btri,curve->first,POSITIVE_ORIENTATION,
				       sizest);
	    	copy_tri_state_to_btri(btri,curve->first,NEGATIVE_ORIENTATION,
				       sizest);

	    	tri_end = tri_start;
	    	side_end = side_start;
	    	b = curve->first;
	    	while ((tri_end=find_following_null_tri(tri_end,&p,&side_end,
							COUNTER_CLOCK)) != NULL)
	    	{
		    if (insert_point_in_bond(ne->posn,b,curve) != 
			FUNCTION_SUCCEEDED)
	            {
	                screen("ERROR in install_subdomain_bdry_curves(), "
		               "insert_point_in_bond() failed\n");
	                clean_up(ERROR);
	            }
		    b = b->next;
	            ne->posn = b->end = p;
		    set_bond_length(b,dim);
	    	    btri = link_tri_to_bond(NULL,tri_end,*s,b,curve);
	            copy_tri_state_to_btri(btri,b,NEGATIVE_ORIENTATION,sizest);
	            copy_tri_state_to_btri(btri,b,POSITIVE_ORIENTATION,sizest);
		    if (ns->posn == ne->posn) /*Closed loop formed*/
		    {
			change_node_of_curve(curve,NEGATIVE_ORIENTATION,ns);
			(void) delete_node(ne);
			break;
		    }
	        }
	    	if (is_closed_curve(curve)) 
			continue;

	    	b = curve->first;
		tri_end = tri_start;
		side_end = side_start;
	    	while ((tri_start = find_following_null_tri(tri_start,&p,
					&side_start,CLOCKWISE)) != NULL)
	    	{
		    bte = Bond_tri_on_side(tri_end,side_end);
		    if (insert_point_in_bond(ns->posn,b,curve) !=
			FUNCTION_SUCCEEDED)
	            {
	                screen("ERROR in install_subdomain_bdry_curves(), "
		               "insert_point_in_bond() failed\n");
	                clean_up(ERROR);
	            }
	            ns->posn = b->start = p;
		    set_bond_length(b,dim);
	            btri = link_tri_to_bond(NULL,tri_end,*s,b->next,curve);
	            copy_tri_state_to_btri(btri,b->next,NEGATIVE_ORIENTATION,
				sizest);
	            copy_tri_state_to_btri(btri,b->next,POSITIVE_ORIENTATION,
				sizest);
		    btri = bte;
		    (void) link_tri_to_bond(btri,tri_start,*s,b,curve);
	            copy_tri_state_to_btri(btri,b,NEGATIVE_ORIENTATION,sizest);
	            copy_tri_state_to_btri(btri,b,POSITIVE_ORIENTATION,sizest);
		    tri_end = tri_start;
		    side_end = side_start;
	        }

		/*
		*       At this point the surface has run into a boundary
		*       This means there should be existing nodes with
		*       positions ns->posn and ne->posn
		*/

	    	for (n = intfc->nodes; n && *n; ++n)
	    	{
		    if (*n == ns)
			continue;
		    if ((*n)->posn == ns->posn) 
		    {
			change_node_of_curve(curve,POSITIVE_ORIENTATION,*n);
			(void) delete_node(ns);
			ns = *n;
			break;
		    }
	        }
	        for (n = intfc->nodes; n && *n; ++n)
	        {
		    if (*n == ne)
			continue;
		    if ((*n)->posn == ne->posn) 
		    {
			change_node_of_curve(curve,NEGATIVE_ORIENTATION,*n);
			(void) delete_node(ne);
			ne = *n;
			break;
		    }
	        }
	    }
	}
	if (debugging("consistency"))
	{
	    if (!consistent_interface(intfc))
	    {
		screen("ERROR in install_subdomain_bdry_curves(), "
		       "intfc is inconsistent\n");
		clean_up(ERROR);
	    }
	}
	interpolate_intfc_states(intfc) = sav_intrp;
	DEBUG_LEAVE(install_subdomain_bdry_curves)
}		/*end install_subdomain_bdry_curves*/

LOCAL bool null_side_on_surface(
	SURFACE *s,
	TRI     **tri_start,
	int     *side_start)
{
	TRI *tri;
	int i;

	for (tri = first_tri(s); !at_end_of_tri_list(tri,s); tri = tri->next)
	{
	    for (i = 0; i < 3; ++i)
	    {
		if (Tri_on_side(tri,i) == NULL)
		{
		    *side_start = i;
		    *tri_start = tri;
		    return YES;
		}
	    }
	}
	return NO;
}	/* end null_side_on_surface */

LOCAL bool buffer_extension3d1(
	INTERFACE	*intfc,
	INTERFACE	*adj_intfc,
	int		dir,
	int		nb,
	bool		status)
{
	RECT_GRID	*gr = computational_grid(intfc);

	DEBUG_ENTER(buffer_extension3d1)

	set_current_interface(intfc);

		/* Patch tris from adj_intfc to intfc */
	if (!append_adj_intfc_to_buffer1(intfc,adj_intfc,gr,dir,nb))
	{
	    status = FUNCTION_FAILED;
	    (void) printf("WARNING in buffer_extension3d1(), "
	                  "append_adj_intfc_to_buffer1() failed\n");
	    DEBUG_LEAVE(buffer_extension3d1)
	    return status;
	}
	DEBUG_LEAVE(buffer_extension3d1)
	return status;
}	/*end buffer_extension3d1*/

LOCAL	void	shift_interface(
	INTERFACE *intfc,
	float     T,
	int       dir)
{
	POINT   *p;
	SURFACE **s;
	TRI     *t;
	int     i;

	DEBUG_ENTER(shift_interface)

	if (DEBUG)
	{
	    char dname[1024];
	    static int ntimes[3];
	    (void) printf("Shifting interface %llu by %g in direction %d\n",
			  interface_number(intfc),T,dir);
	    print_interface(intfc);
	    (void) sprintf(dname,"fscatter/shift_interface/Into%d.%d.%s%g",
			   ntimes[dir],dir,(T>0.0) ? "+" : "",T);
	    ++ntimes[dir];
	    gview_plot_interface(dname,intfc);
	}

		/* Shift points on interface */
	(void) next_point(intfc,NULL,NULL,NULL);/*reset sort status*/
	for (s = intfc->surfaces; s && *s; ++s)
	{
	    for (t = first_tri(*s); !at_end_of_tri_list(t,*s); t = t->next)
	    {
		for (i = 0; i < 3; ++i)
		{
		    p = Point_of_tri(t)[i];
		    if (sorted(p) == NO)
	                Coords(p)[dir] += T;
		    sorted(p) = YES;
		}
	    }
	}

	if (DEBUG)
	{
	    char dname[1024];
	    static int ntimes[3];
	    (void) printf("Interface %llu after shift by %g in direction %d\n",
			  interface_number(intfc),T,dir);
	    print_interface(intfc);
	    (void) sprintf(dname,"fscatter/shift_interface/Outof%d.%d.%s%g",
			   ntimes[dir],dir,(T>0.0)?"+":"",T);
	    ++ntimes[dir];
	    gview_plot_interface(dname,intfc);
	}

	DEBUG_LEAVE(shift_interface)
}		/*end periodic_shift_interface*/


LOCAL 	int append_adj_intfc_to_buffer1(
	INTERFACE	*intfc,		/* my interface 	       */
	INTERFACE	*adj_intfc,	/* received interface	       */
	RECT_GRID	*grid,		/* Rectangular grid for region */
	int		dir,
	int		nb)
{
	INTERFACE	*cur_intfc;
	SURFACE		**s, **as;
	int		p_size;		/*Size of space allocated for p_table*/
	static P_LINK	*p_table = NULL;/* Table of matching points on intfc
					 * and adj_intfc*/
	static int      len_p_table = 0;
	bool      	corr_surf_found;

	DEBUG_ENTER(append_adj_intfc_to_buffer1)

	if (DEBUG)
	{
	    char dname[256];
	    static int ntimes[3][2];
	    static const char *strdir[3] = { "X", "Y", "Z" };
	    static const char *strnb[2] = { "LOWER", "UPPER" };

	    (void) sprintf(dname,"fscatter/"
				 "append_adj_intfc_to_buffer1/Data%d-%s_%s/%s",
			         ntimes[dir][nb],strdir[dir],strnb[nb],
				 "intfc_gv");
	    gview_plot_interface(dname,intfc);
	    (void) sprintf(dname,"fscatter/"
				 "append_adj_intfc_to_buffer1/Data%d-%s_%s/%s",
			         ntimes[dir][nb],strdir[dir],strnb[nb],
				 "adj_intfc_gv");
	    gview_plot_interface(dname,adj_intfc);

	    ++ntimes[dir][nb];
	}

	cur_intfc = current_interface();
	set_current_interface(intfc);

	p_size = 4*(adj_intfc->num_points) + 1;
	if (len_p_table < p_size)
	{
	    len_p_table = p_size;
	    if (p_table != NULL)
		free(p_table);
	    vector(&p_table,len_p_table,sizeof(P_LINK));
	}

	/* Begin patching adj_intfc to current interface */
	for (s = intfc->surfaces; s && *s; ++s)
	{
	    corr_surf_found = NO;
	    for (as = adj_intfc->surfaces; as && *as; ++as)
	    {
		/*
		*  COMMENT -
		*  The Hyper_surf_index() function is not
		*  fully supported.  This will fail in the
		*  presence of interface changes in topology
		*  TODO: FULLY SUPPORT THIS OBJECT
		*/
		if (Hyper_surf_index(*s) == Hyper_surf_index(*as))
		{
		    corr_surf_found = YES;
		    if (!append_buffer_surface1(*s,*as,grid,dir,nb,p_table,
						   p_size))
		    {
			set_current_interface(cur_intfc);
			(void) printf("WARNING in "
			              "append_adj_intfc_to_buffer1(), "
			              "append surface failed\n");
			(void) printf("Surface\n");
			print_surface(*s);
			(void) printf("Adjacent surface\n");
			print_surface(*as);
			(void) printf("intfc\n");
			print_interface(intfc);
			(void) printf("adj_intfc\n");
			print_interface(adj_intfc);
			DEBUG_LEAVE(append_adj_intfc_to_buffer1)
			return NO;
		    }
		}
	    }
	    if (!corr_surf_found)
	    {
	    	SURFACE *surf;
		surf = copy_buffer_surface(*as,p_table,p_size);
		Hyper_surf_index(surf) = Hyper_surf_index((*as));
	    }
	}
	set_current_interface(cur_intfc);
	DEBUG_LEAVE(append_adj_intfc_to_buffer1)
	return YES;
}		/*end append_adj_intfc_to_buffer1*/

EXPORT void set_floating_point_tolerance1(
	float		*h)
{
	float eps;
	static const float mfac = 10.0;/*TOLERANCE*/
	static const float gfac = 0.001;/*TOLERANCE*/
	int   i;

	eps = mfac*MACH_EPS;
	for (i = 0; i < 3; ++i)
	{
	    tol1[i] = gfac*h[i];/*TOLERANCE*/
	    tol1[i] = max(tol1[i],eps);
	}
}		/*end set_floating_point_tolerance1*/

LOCAL int append_buffer_surface1(
	SURFACE		*surf,
	SURFACE		*adj_surf,
	RECT_GRID	*grid,
	int		dir,
	int		nb,
	P_LINK		*p_table,
	int		p_size)
{
	SURFACE    *new_adj_surf;
	TRI	   *tri;
	bool    done;
	int	   ns, na, new_na, ntimes;
	float	   crx_coord;
	static TRI **tris_s = NULL, **tris_a = NULL;
	static int len_tris_s = 0, len_tris_a = 0;

	DEBUG_ENTER(append_buffer_surface1)
	if (DEBUG)
	{
	    (void) printf("dir = %d,nb = %d\n",dir,nb);
	    (void) printf("My surface\n");
	    print_surface(surf);
	    (void) printf("Buffer surface\n");
	    print_surface(adj_surf);
	}

	if (len_tris_s < surf->num_tri)
	{
	    len_tris_s = surf->num_tri;
	    if (tris_s != NULL)
		free(tris_s);
	    vector(&tris_s,len_tris_s,sizeof(TRI *));
	}
	if (len_tris_a < adj_surf->num_tri)
	{
	    len_tris_a = adj_surf->num_tri;
	    if (tris_a != NULL)
		free(tris_a);
	    vector(&tris_a,len_tris_a,sizeof(TRI *));
	}

	crx_coord = (nb == 0) ? grid->L[dir] : grid->U[dir];

	done = NO;
	ntimes = 0;
	do
	{
	    ns = na = 0;
	    for (tri=first_tri(surf); !at_end_of_tri_list(tri,surf);
		 tri=tri->next)
	    {
	        if (tri_cross_line(tri,crx_coord,dir) == YES)
	    	    tris_s[ns++] = tri;
	    }
	    for (tri = first_tri(adj_surf); !at_end_of_tri_list(tri,adj_surf);
		 tri = tri->next)
	    {
	        if (tri_cross_line(tri,crx_coord,dir) == YES)
	    	    tris_a[na++] = tri;
	    }

	    /* Add matching points to the hashing table p_table */

	    if (add_matching_pt_to_hash_table(tris_s,tris_a,ns,na,surf,
					      adj_surf,p_table,p_size))
	    {
	        new_adj_surf = copy_buffer_surface(adj_surf,p_table,p_size);
	        new_na = 0;
	        for (tri=first_tri(new_adj_surf);
		     !at_end_of_tri_list(tri,new_adj_surf);
	             tri = tri->next)
	        {
	            if (tri_cross_line(tri,crx_coord,dir) == YES)
	    	        tris_a[new_na++] = tri;
	        }
	        if (new_na == na)
	        {
	            if (add_matching_pt_to_hash_table(tris_s,tris_a,ns,na,surf,
					              new_adj_surf,
						      p_table,p_size))
		    {
		        adj_surf = new_adj_surf;
		        done = YES;
		    }
	        }
	        else if (!delete_surface(new_adj_surf))
	        {
		    screen("ERROR in append_buffer_surface1(), can't delete "
		           "new_adj_surf\n");
		    clean_up(ERROR);
	        }
	    }
	    if (ntimes++ > 10)
	    {
		screen("ERROR in append_buffer_surface1(), can't match "
		       "tris on with buffer surface\n");
		clean_up(ERROR);
	    }
	} while (!done);

	synchronize_tris_at_subdomain_bdry(tris_a,tris_s,ns,p_table,p_size);
	merge_point_pointers_at_subdomain_bdry(tris_a,tris_s,ns,p_table,p_size);

	/*adjoin adj_surf tri list to surf tri list */
	last_tri(surf)->next = first_tri(adj_surf);
	first_tri(adj_surf)->prev = last_tri(surf);
	link_tri_list_to_surface(first_tri(surf),last_tri(adj_surf),surf);
	adj_surf->pos_curves = adj_surf->neg_curves = NULL;
	(void) delete_surface(adj_surf);
	adj_surf = NULL;

	if (!match_tris_at_subdomain_bdry(surf,adj_surf,tris_s,tris_a,ns,na))
	{
	    (void) printf("WARNING in append_buffer_surface1(), "
	                  "no match of tris at subdomain\n");
	    (void) printf("dir = %d, nb = %d\n",dir,nb);
	    print_rectangular_grid(grid);
	    DEBUG_LEAVE(append_buffer_surface1)
	    return NO;
	}

	DEBUG_LEAVE(append_buffer_surface1)
	return YES;
}		/*end append_buffer_surface1*/


LOCAL	void	synchronize_tris_at_subdomain_bdry(
	TRI    **tris_a,
	TRI    **tris_s,
	int    nt,
	P_LINK *p_table,
	int    p_size)
{
	POINT **ps, **pa, *p0, *p1, *p2;
	TRI   *ts, *ta;
	int   i, j, id, idp, idn;

	for (i = 0; i < nt; ++i)
	{
	    ts = tris_s[i];
	    ps = Point_of_tri(ts);
	    for (j = 0; j < nt; ++j)
	    {
		ta = tris_a[j];
		pa = Point_of_tri(ta);
		for (id = 0; id < 3; ++id)
		{
		    p0 = (POINT*) find_from_hash_table((POINTER)pa[id],
						       p_table,p_size);
		    if (p0 == ps[0])
		    {
		        idn = Next_m3(id);
		        p1 = (POINT*) find_from_hash_table((POINTER)pa[idn],
						           p_table,p_size);
		        if (p1 == ps[1])
		        {
		            idp = Prev_m3(id);
		            p2 = (POINT*) find_from_hash_table((POINTER)pa[idp],
						               p_table,p_size);
			    if (p2 == ps[2])
			    {
			        rotate_triangle(ts,id);
			        set_normal_of_tri(ts);
			        set_normal_of_tri(ta);
			        break;
			    }
		        }
		    }
		}
		if (id < 3)
		    break;
	    }
	}
}		/*end synchronize_tris_at_subdomain_bdry*/

LOCAL	bool	tri_cross_line(
	TRI		*tri,
	float		crx_coord,
	int		dir)
{
	float	min_coord, max_coord;
	float	crx_tol = tol1[dir];

	min_coord = max_coord = Coords(Point_of_tri(tri)[0])[dir];

	if (min_coord > Coords(Point_of_tri(tri)[1])[dir])
	    min_coord = Coords(Point_of_tri(tri)[1])[dir];
	if (min_coord > Coords(Point_of_tri(tri)[2])[dir])
	    min_coord = Coords(Point_of_tri(tri)[2])[dir];

	if (max_coord < Coords(Point_of_tri(tri)[1])[dir])
	    max_coord = Coords(Point_of_tri(tri)[1])[dir];
	if (max_coord < Coords(Point_of_tri(tri)[2])[dir])
	    max_coord = Coords(Point_of_tri(tri)[2])[dir];
	
	return (((min_coord - crx_coord) <= crx_tol) && 
	        ((crx_coord - max_coord) <= crx_tol)) ? YES : NO;
}		/*end tri_cross_line*/

LOCAL bool match_tris_at_subdomain_bdry(
	SURFACE		*surf,
	SURFACE		*adj_surf,
	TRI		**tri_s,
	TRI		**tri_a,
	int		ns,
	int		na)
{
	TRI		*ta, *ts;
	int		i, j;
	int		ums,uma;
	static bool	*ms = NULL, *ma = NULL;
	static int      ms_len = 0, ma_len = 0;

	DEBUG_ENTER(match_tris_at_subdomain_bdry)
	if (DEBUG)
	{
	    (void) printf("Entered match_tris_at_subdomain_bdry()\n");
	    (void) printf("tri_a: na=%d\n",na);
	    for (i = 0; i < na; ++i)
		print_tri(tri_a[i],adj_surf->interface);
	    (void) printf("tri_s: ns=%d\n",ns);
	    for (i = 0; i < ns; ++i)
		print_tri(tri_s[i],surf->interface);
	}

	if (ms_len < ns)
	{
	    ms_len = ns;
	    if (ms != NULL)
		free(ms);
	    vector(&ms,ms_len,sizeof(bool));
	}
	if (ma_len < na)
	{
	    ma_len = na;
	    if (ma != NULL)
		free(ma);
	    vector(&ma,ma_len,sizeof(bool));
	}

	for (i = 0; i < ns; ++i)
	    ms[i] = NO;
	for (i = 0; i < na; ++i)
	    ma[i] = NO;

	for (i = 0; i < na; ++i)
	{
	    ta = tri_a[i];
	    for (j = 0; j < ns; ++j)
	    {
	        ts = tri_s[j];
	        if (match_two_tris(ts,ta))
		{
		    ma[i] = ms[j] = YES;
	    	    merge_two_tris(ts,ta,surf,adj_surf);
		    break;
		}
	    }
	}
	ums = uma = 0;
	for (i = 0; i < ns; ++i)
	    if (ms[i] == NO)
		++ums;
	for (i = 0; i < na; ++i)
	    if (ma[i] == NO)
		++uma;
	if (ums != 0 || uma != 0)
	{
	    (void) printf("WARNING in match_tris_at_subdomain_bdry(), "
	                  "unmatched local tris\n"
	                  "na = %d ns = %d\n"
	                  "ums = %d uma = %d\n",na,ns,ums,uma);
	    for (i = 0; i < ns; ++i)
	    	if (ms[i] == NO)
		    print_tri(tri_s[i],surf->interface);
	    (void) printf("unmatched adj tris:\n");
	    for (i = 0; i < na; ++i)
	    	if (ma[i] == NO)
		    print_tri(tri_a[i],adj_surf->interface);
	    
	    DEBUG_LEAVE(match_tris_at_subdomain_bdry)
	    return NO;
	}

	DEBUG_LEAVE(match_tris_at_subdomain_bdry)
	return YES;
}		/*end match_tris_at_subdomain_bdry*/

/*
*			match_two_tris():
*
*	Determines whether two triangles are the same triangle up to a rotation
*	of the vertex indices.  Returns YES if the tris are the same,
*	otherwise returns NO.
*/

LOCAL bool match_two_tris(
	TRI		*tri,
	TRI		*atri)
{
	int		j;
	POINT		**p, **ap;

	p = Point_of_tri(tri);
	ap = Point_of_tri(atri);
	for (j = 0; j < 3; ++j)
	    if (p[0] == ap[j])
		break;
	if (j == 3)
	    return NO;
	return ((p[1]==ap[Next_m3(j)]) && (p[2]==ap[Prev_m3(j)])) ? YES : NO;
}		/*end match_two_tris*/

/*
*			merge_two_tris():
*
*	Merges two triangles ts and ta by replacing all linking
*	information to and from ta by linking with ts  while preserving
*	any nontrivial linking information in ts.
*/

EXPORT	void merge_two_tris(
	TRI	*ts,
	TRI	*ta,
	SURFACE	*s,
	SURFACE *as)
{
	TRI		*tri;
	int		i, j;

	DEBUG_ENTER(merge_two_tris)
	if (DEBUG)
	{
	    print_tri(ts,s->interface);
	    print_tri(ta,as->interface);
	}

	/* Set the links on the null side of ts by corresponding links to ta*/
	for (i = 0; i < 3; ++i)
	{
	    if (Tri_on_side(ts,i) == NULL)
	    {
	    	if (Tri_on_side(ta,i) == NULL)
	    	    continue;
 
	    	/* set tri_neighbor */
	    	Tri_on_side(ts,i) = tri = Tri_on_side(ta,i);
	    	for (j = 0; j < 3; ++j)
	    	{
	    	    if (Tri_on_side(tri,j) == ta)
	    	    	Tri_on_side(tri,j) = ts;
	    	}
	    }
	}
	/*remove ta from s tri list*/
	remove_tri_from_surface(ta,s,NO);
	if (DEBUG)
	{
	    (void) printf("after merge_two_tris\n");
	    print_tri(ts,s->interface);
	}
	DEBUG_LEAVE(merge_two_tris)
}		/*end merge_two_tris*/



/*
*			add_matching_pt_to_hash_table():
*
*	Creates a hashed list of matching points on intfc and adj_intfc,
*	where two points match if their positions are within a prescribed
*	floating point tolerance.  This is an extremely expensive function
*	and should be a candidate for an improved version.
*/

LOCAL bool add_matching_pt_to_hash_table(
	TRI 	**tris_s,
	TRI	**tris_a,
	int	ns,
	int 	na,
	SURFACE	*ss,
	SURFACE	*sa,
	P_LINK	*p_table,
	int	p_size)
{
	bool                  status;
	int 	                 i;
	POINT_LIST               *plists, *plista, *pls, *pla;
	static POINT_LIST_STORE  Pslist_store, Palist_store;

	reset_hash_table(p_table,p_size);
	plists = set_point_list(tris_s,ns,Hyper_surf(ss),&Pslist_store);
	plista = set_point_list(tris_a,na,Hyper_surf(sa),&Palist_store);

	status = YES;
	for (pls = plists; pls != NULL; pls = pls->next)
	{
	    for (pla = plista; pla != NULL; pla = pla->next)
	    {
		for (i = 0; i < 3; ++i) /*Floating point TOLERANCE test*/
	            if (fabs(Coords(pla->p)[i]-Coords(pls->p)[i]) > tol1[i])
			break;
		if (i == 3)
		{
	            (void) add_to_hash_table((POINTER)pla->p,(POINTER)pls->p,
				             p_table,p_size);
		    (void) average_points(NO,pla->p,pla->hse,pla->hs,
				          pls->p,pls->hse,pls->hs);
		    if (pla->prev == NULL)
			plista = pla->next;
		    else
			pla->prev->next = pla->next;
		    if (pla->next != NULL)
			pla->next->prev = pla->prev;
		    break;
		}
	    }
	    if (pla == NULL)
	        status = NO;
	}
	if (plista != NULL)
	    status = NO;
	return (ns == na) ? status : NO;
}		/*end add_matching_pt_to_hash_table*/

LOCAL	POINT_LIST	*set_point_list(
	TRI	         **tris,
	int              nt,
	HYPER_SURF       *hs,
	POINT_LIST_STORE *plist_store)
{
	POINT      **p;
	POINT_LIST *plist, *pl, Plist;
	TRI        *tri;
	int        i, j, max_np;

	if (nt == 0)
	    return NULL;
	max_np = 3*nt + 1;
	if (max_np > plist_store->len)
	{
	    if (plist_store->pl != NULL)
		free(plist_store->pl);
	    plist_store->len = max_np;
	    vector(&plist_store->pl,sizeof(POINT_LIST),plist_store->len);
	}
	zero_scalar(plist_store->pl,sizeof(POINT_LIST)*plist_store->len);
	for (i = 0; i < nt; ++i)
	{
	    p = Point_of_tri(tris[i]);
	    for (j = 0; j < 3; ++j)
	        sorted(p[j]) = NO;
	}

	plist = plist_store->pl;
	pl = &Plist;
	for (i = 0; i < nt; ++i)
	{
	    tri = tris[i];
	    p = Point_of_tri(tri);
	    for (j = 0; j < 3; ++j)
	    {
		if (!sorted(p[j]))
		{
		    pl->next = plist++;
		    pl->next->prev = pl;
		    pl = pl->next;
	            pl->p = p[j];
	            pl->hs = hs;
	            pl->hse = Hyper_surf_element(tri);
	            sorted(pl->p) = YES;
		}
	    }
	}
	Plist.next->prev = NULL;
	return Plist.next;
}		/*end set_point_list*/

EXPORT	SURFACE *copy_buffer_surface(
	SURFACE		*as,
	P_LINK		*p_table,
	int		p_size)
{
	SURFACE		*s;
	POINT		*p, *np, *ap;
	TRI		*tri, *atri;
	CURVE		**c, **pos_curves = NULL, **neg_curves = NULL;
	int		i;

	DEBUG_ENTER(copy_buffer_surface)

	/* First identify and copy nodes on as */

	for (c = as->pos_curves; c && *c; ++c)
	{
	    if (!add_to_pointers(matching_curve(*c,p_table,p_size),
				    &pos_curves))
	    {
	        screen("ERROR in copy_buffer_surface(), "
	               "add_to_pointers() failed\n");
	        clean_up(ERROR);
	    }
	}
	for (c = as->neg_curves; c && *c; ++c)
	{
	    if (!add_to_pointers(matching_curve(*c,p_table,p_size),
				    &neg_curves))
	    {
	        screen("ERROR in copy_buffer_surface(), "
	               "add_to_pointers() failed\n");
	        clean_up(ERROR);
	    }
	}

	/* set tri_array_numbers */
	for(i=0, tri=first_tri(as); !at_end_of_tri_list(tri,as); tri=tri->next)
	    Tri_index(tri) = i++;

	s = copy_surface(as,pos_curves,neg_curves,YES);

	/* null_tri_array_numbers */
	for(i=0, tri=first_tri(as); !at_end_of_tri_list(tri,as); tri=tri->next)
	    Tri_index(tri) = 0;

	for (tri = first_tri(s); !at_end_of_tri_list(tri,s); tri = tri->next)
	{
	    sorted(Point_of_tri(tri)[0]) = NO;
	    sorted(Point_of_tri(tri)[1]) = NO;
	    sorted(Point_of_tri(tri)[2]) = NO;
	}
	for (tri = first_tri(s), atri = first_tri(as);
	     !at_end_of_tri_list(tri,s); tri = tri->next, atri = atri->next)
	{
	    for (i = 0; i < 3; ++i)
	    {
	    	p = Point_of_tri(tri)[i];
	    	if (sorted(p) == NO)
	    	{
	    	    if (!vertex_on_bond(tri,i))
		    {
			ap = Point_of_tri(atri)[i];
			np = (POINT*)find_from_hash_table((POINTER)ap,
					                  p_table,p_size);
			if (np == NULL)
			{
			    (void) add_to_hash_table((POINTER)ap,(POINTER)p,
						     p_table,p_size);
			}
			else if (p != np)
			{
			    p = np;
			    Point_of_tri(tri)[i] = p;
			}
		    }
		    sorted(p) = YES;
		}
	    }
	}
	DEBUG_LEAVE(copy_buffer_surface)
	return s;
}		/*end copy_buffer_surface*/

EXPORT	CURVE *matching_curve(
	CURVE		*ac,
	P_LINK		*p_table,
	int		p_size)
{
	BOND		*b, *ab;
	CURVE		**cc, *c;
	NODE		*ns, *ne;
	POINT		*ap, *p;

	ns = matching_node(ac->start,p_table,p_size);
	ne = matching_node(ac->end,p_table,p_size);
	for (cc = ns->out_curves; cc && *cc; ++cc)
	    if (curves_match(*cc,ac,p_table,p_size))
	        return *cc;
	c = copy_curve(ac,ns,ne);
	for (b = c->first, ab = ac->first; b != c->last && ab != ac->last;
		b = b->next, ab = ab->next)
	{
	    ap = ab->end;
	    p = (POINT*) find_from_hash_table((POINTER)ap,p_table,p_size);
	    if (p != NULL)
	    	b->end = p;
	    else
	    	(void) add_to_hash_table((POINTER)ap,(POINTER)b->end,
					 p_table,p_size);
	}
	return c;
}		/*end matching_curve*/

EXPORT	bool curves_match(
	CURVE		*c,
	CURVE		*ac,
	P_LINK		*p_table,
	int		p_size)
{
	BOND		*b, *ab;
	POINT		*p, *ap;

	ap = ac->start->posn;
	p = (POINT*)find_from_hash_table((POINTER)ap,p_table,p_size);
	if (p != c->start->posn)
	    return NO;
	for (b = c->first, ab = ac->first; b && ab; b = b->next, ab = ab->next)
	{
	    ap = ab->end;
	    p = (POINT*)find_from_hash_table((POINTER)ap,p_table,p_size);
	    if (p != b->end)
		return NO;
	}
	return YES;
}		/*end curves_match*/

EXPORT	NODE *matching_node(
	NODE		*an,
	P_LINK		*p_table,
	int		p_size)
{
	INTERFACE	*intfc = current_interface();
	NODE		*newn;
	POINT		*p, *ap = an->posn;

	p = (POINT*)find_from_hash_table((POINTER)ap,p_table,p_size);
	if (p == NULL)
	{
	    p = copy_point(ap);
	    (void) add_to_hash_table((POINTER)ap,(POINTER)p,p_table,p_size);
	}
	newn = node_of_point(p,intfc);
	if (newn == NULL)
	    newn = make_node(p);
	return newn;
}		/*end matching_node*/

EXPORT void open_null_sides1(
	INTERFACE	*intfc,
	float		*L,
	float		*U,
	int		dir,
	int		nb)
{
	TRI		*tri;
	SURFACE 	**s;
	DEBUG_ENTER(open_null_sides1)

	if (DEBUG)
	{
	    char	      dname[1024];
	    static const char *xyz[] = { "x", "y", "z"};
	    static const char *sname[] = { "lower", "upper"};
	    static const char *strdir[3] = { "X", "Y", "Z" };
	    static const char *strnb[2] = { "LOWER", "UPPER" };
	    static int into[3][2];
	    (void) printf("Clipping interface in at %s %s side\n",
			  sname[nb],xyz[dir]);
	    (void) printf("U = ( %g %g %g )\n",U[0],U[1],U[2]);
	    (void) printf("L = ( %g %g %g )\n",L[0],L[1],L[2]);
	    (void) sprintf(dname,"fscatter/open_null_sides1/Into%d-%s_%s",
			   into[dir][nb],strdir[dir],strnb[nb]);
	    ++into[dir][nb];
	    summarize_interface(dname,"before",intfc,XY_PLANE,
				"open_null_sides1","before");
	}
	if (!buffered_boundary_type(rect_boundary_type(intfc,dir,nb)))
	{
	    DEBUG_LEAVE(open_null_sides1)
	    return;
	}

	for (s = intfc->surfaces; s && *s; ++s)
	{
	    TRI *ntri = NULL;
	    for (tri=first_tri(*s); !at_end_of_tri_list(tri,*s); tri=ntri)
	    {
		ntri = tri->next;
	    	if (tri_out_domain1(tri,L,U,dir,nb))
	    	    remove_out_domain_tri(tri,*s);
	    }
	}
	for (s = intfc->surfaces; s && *s; ++s)
	{
	    if (no_tris_on_surface(*s))
	    	(void) delete_surface(*s);
	}
	DEBUG_LEAVE(open_null_sides1)
}		/*end open_null_sides1*/

EXPORT void remove_out_domain_tri(
	TRI		*tri,
	SURFACE		*s)
{
	TRI		*nbtri;
	int		i;

	for (i = 0; i < 3; ++i)
	{
	    if (!is_side_bdry(tri,i))
	    {
		nbtri = Tri_on_side(tri,i);
		if (nbtri != NULL)
		{
		    if (Tri_on_side01(nbtri) == tri)
		    	Tri_on_side01(nbtri) = NULL;
		    else if (Tri_on_side12(nbtri) == tri)
		    	Tri_on_side12(nbtri) = NULL;
		    else if (Tri_on_side20(nbtri) == tri)
		    	Tri_on_side20(nbtri) = NULL;
		}
	    }
	    else
	    {
		BOND_TRI *bt = Bond_tri_on_side(tri,i);
		if (bt != NULL)
		    (void) delete_from_pointers(bt,&Btris(bt->bond));
	    }
	}
	remove_tri_from_surface(tri,s,NO);
}		/*end remove_out_domain_tri*/


LOCAL bool tri_out_domain1(
	TRI		*tri,
	float		*L,
	float		*U,
	int		dir,
	int		nb)
{
	POINT **p;
	int   i;

	p = Point_of_tri(tri);
	if (nb == 0)
	{
	    for (i = 0; i < 3; ++i)
	    {
		if ((L[dir] - Coords(p[i])[dir]) <= tol1[dir])
	    	    return NO;
	    }
	}
	else
	{
	    for (i = 0; i < 3; ++i)
	    {
		if ((Coords(p[i])[dir] - U[dir]) <= tol1[dir])
	    	    return NO;
	    }
	}
	return YES;
}	/* end tri_out_domain1 */

LOCAL void clip_intfc_at_grid_bdry1(
	INTERFACE	*intfc)
{
	INTERFACE	*cur_intfc = current_interface();
	RECT_GRID	*gr = computational_grid(intfc);
	int		dim = intfc->dim;
	int		dir, nb;

	DEBUG_ENTER(clip_intfc_at_grid_bdry1)
	strip_subdomain_bdry_curves(intfc);
	set_current_interface(intfc);
	for (dir = 0; dir < dim; ++dir)
	{
	    for (nb = 0; nb < 2; ++nb)
	    	open_null_sides1(intfc,gr->L,gr->U,dir,nb);
	}
	cut_out_curves_in_buffer(intfc);
	reset_intfc_num_points(intfc);
	set_current_interface(cur_intfc);
	DEBUG_LEAVE(clip_intfc_at_grid_bdry1)
}		/*end clip_intfc_at_grid_bdry1*/

LOCAL void merge_point_pointers_at_subdomain_bdry(
	TRI             **tris_a,
	TRI		**tris_s,
	int		nt,
	P_LINK		*p_table,
	int		p_size)
{
	POINT		*p, *ap;
	int		i, j;

        for (i = 0; i < nt; ++i)
	{
	    sorted(Point_of_tri(tris_s[i])[0]) = NO; 
	    sorted(Point_of_tri(tris_s[i])[1]) = NO;
	    sorted(Point_of_tri(tris_s[i])[2]) = NO;
	    sorted(Point_of_tri(tris_a[i])[0]) = NO; 
	    sorted(Point_of_tri(tris_a[i])[1]) = NO;
	    sorted(Point_of_tri(tris_a[i])[2]) = NO;
	}
	for (i = 0; i < nt; ++i)
	{
	    for (j = 0; j < 3; ++j)
	    {
		ap = Point_of_tri(tris_a[i])[j];
		if (sorted(ap) == NO)
		{
	            p = (POINT*)find_from_hash_table((POINTER)ap,
						     p_table,p_size);
		    Point_of_tri(tris_a[i])[j] = p;
		    sorted(p) = YES;
		}
	    }
	}
}	/*end merge_point_pointers_at_subdomain_bdry*/


LOCAL	void	copy_tri_state_to_btri(
	BOND_TRI	*btri,
	BOND		*b,
	ORIENTATION	orient,
	size_t		sizest)
{
	POINT		*p;
	Locstate	sl, sr;

	if (orient == POSITIVE_ORIENTATION)
	{
	    p = b->start;
	    sl = left_start_btri_state(btri);
	    sr = right_start_btri_state(btri);
	}
	else
	{
	    p = b->end;
	    sl = left_end_btri_state(btri);
	    sr = right_end_btri_state(btri);
	}
	assign(sl,left_state(p),sizest);
	assign(sr,right_state(p),sizest);
}		/*end copy_tri_state_to_btri*/

LOCAL TRI *find_following_null_tri(
	TRI		*tri,
	POINT		**np,
	int		*side,
	ANGLE_DIRECTION dir)
{
	POINT *p;
	TRI   *newtri;
	int   nside;

	p = (dir == COUNTER_CLOCK) ? Point_of_tri(tri)[Next_m3(*side)] :
			             Point_of_tri(tri)[*side];

	newtri = tri;
	while (newtri != NULL)
	{
	    nside = Following_side_at_vertex(newtri,p,dir);
	    if (nside == ERROR)
	    {
	    	*side = -1;
	    	return NULL;
	    }
	    if (is_side_bdry(newtri,nside))
	    {
	    	if (Bond_tri_on_side(newtri,nside) == NULL)
	    	{
	    	    *side = nside;
	    	    *np = (dir == COUNTER_CLOCK) ?
	    			Point_of_tri(newtri)[Next_m3(*side)] :
	    			Point_of_tri(newtri)[*side];
	    	    return newtri;
	    	}
	    	return NULL;
	    }
	    else if (Tri_on_side(newtri,nside) == NULL)
	    {
	    	*side = nside;
	    	*np = (dir == COUNTER_CLOCK) ?
	    		Point_of_tri(newtri)[Next_m3(*side)] :
	    		Point_of_tri(newtri)[*side];
	    	return newtri;
	    }
	    newtri = Tri_on_side(newtri,nside);
	}
	return NULL;
}		/*end find_following_null_tri*/


EXPORT void strip_subdomain_bdry_curves(
	INTERFACE	*intfc)
{
	CURVE		**c;
	CURVE		**curves_to_delete = NULL;
	NODE		**nodes_to_delete = NULL;
	SURFACE		**s;
	NODE		**n;

	DEBUG_ENTER(strip_subdomain_bdry_curves)
	for (c = intfc->curves; c && *c; ++c)
	{
	    if (hsbdry_type(*c) != SUBDOMAIN_HSBDRY)
		continue;
	    for (s = (*c)->pos_surfaces; s && *s; ++s)
	    	strip_curve_from_surf(*c,*s,POSITIVE_ORIENTATION);
	    for (s = (*c)->neg_surfaces; s && *s; ++s)
	    	strip_curve_from_surf(*c,*s,NEGATIVE_ORIENTATION);
	    if (!add_to_pointers(*c,&curves_to_delete))
	    {
	    	screen("ERROR in strip_subdomain_bdry_curves(), "
	    	       "add_to_pointers() failed\n");
	    	clean_up(ERROR);
	    }
	}
	for (c = curves_to_delete; c && *c; ++c)
	    (void) delete_curve(*c);
	for (n = intfc->nodes; n && *n; ++n)
	{
	    if (!add_to_pointers(*n,&nodes_to_delete))
	    {
	    	screen("ERROR in strip_subdomain_bdry_curves(), "
	    	       "add_to_pointers() failed\n");
	    	clean_up(ERROR);
	    }
	}
	for (n = nodes_to_delete; n && *n; ++n)
	    (void) delete_node(*n);
	DEBUG_LEAVE(strip_subdomain_bdry_curves)
}		/*end strip_subdomain_bdry_curves*/

LOCAL void strip_curve_from_surf(
	CURVE		*curve,
	SURFACE		*surf,
	ORIENTATION	orient)
{
	BOND		*b;
	BOND_TRI	**btris;
	TRI		*tri;
	int		i;

	DEBUG_ENTER(strip_curve_from_surf)
	for (b = curve->first; b != NULL; b = b->next)
	{
	    Boundary_point(b->start) = Boundary_point(b->end) = 0;
	    for (btris = Btris(b); btris && *btris; ++btris)
	    {
		tri = (*btris)->tri;
		for (i = 0; i < 3; ++i)
		{
		    if (Bond_tri_on_side(tri,i) == *btris)
		    {
		    	Bond_tri_on_side(tri,i) = NULL;
		    	set_side_bdry(Boundary_tri(tri),i,NO);
		    	break;
		    }
		}
		if (!delete_from_pointers(*btris,&Btris(b)))
		{
		    screen("ERROR in strip_curve_from_surf(), "
		           "delete_from_pointers() failed\n");
		    clean_up(ERROR);
		}
	    }
	}
	if (orient == POSITIVE_ORIENTATION)
	{
	    if (!(delete_from_pointers(curve,&surf->pos_curves) &&
		     delete_from_pointers(surf,&curve->pos_surfaces)))
	    {
		screen("ERROR in strip_curve_from_surf(), "
		       "delete_from_pointers() failed\n");
		clean_up(ERROR);
	    }
	}
	else
	{
	    if (!(delete_from_pointers(curve,&surf->neg_curves) &&
	    	     delete_from_pointers(surf,&curve->neg_surfaces)))
	    {
	    	screen("ERROR in strip_curve_from_surf(), "
		       "delete_from_pointers() failed\n");
		clean_up(ERROR);
	    }
	}
	DEBUG_LEAVE(strip_curve_from_surf)
}		/*end strip_curve_from_surf*/


LOCAL	INTERFACE  *cut_buf_interface1(
	INTERFACE	*intfc,
	int		dir,
	int		nb,
	int		*me,
	int		*him)
{
	INTERFACE	  *sav_intfc, *tmp_intfc, *buf_intfc;
	RECT_GRID	  *gr = computational_grid(intfc);
	RECT_GRID	  dual_gr;
	float		  L[3], U[3];
	bool		  sav_copy = copy_intfc_states();
	char              dname[1024]; 
	static const char *strdir[3] = { "X", "Y", "Z" };
	static const char *strnb[2] = { "LOWER", "UPPER" };

	DEBUG_ENTER(cut_buf_interface1)

	if (DEBUG)
	{
	    static int into[3][2];
	    (void) sprintf(dname,"fscatter/cut_buf_interface1/Into%d-%s_%s",
			   into[dir][nb],strdir[dir],strnb[nb]);
	    (void) printf("cut_buf_interface1() at ENTRY\n"
			  "  dir = %s  nb = %s  me = ( %d, %d, %d )  "
			  "him = ( %d, %d, %d )\n",
			  strdir[dir],strnb[nb],me[0],me[1],me[2],
			  him[0],him[1],him[2]);
	    summarize_interface(dname,"intfc",intfc,XY_PLANE,
				"cut_buf_interface1","intfc at ENTER");
	    ++into[dir][nb];
	}

	set_copy_intfc_states(YES);
	sav_intfc = current_interface();

	set_size_of_intfc_state(size_of_state(intfc));
	tmp_intfc = copy_interface(intfc);
	set_dual_grid(&dual_gr,gr);

	if (nb == 0)
	{
	    L[dir] = gr->L[dir];
	    U[dir] = gr->L[dir]+(gr->L[dir]-dual_gr.VL[dir])+0.5*gr->h[dir];
	}
	else
	{
	    L[dir] = gr->U[dir]-(dual_gr.VU[dir]-gr->U[dir])-0.5*gr->h[dir];
	    U[dir] = gr->U[dir];
	}

	open_null_sides1(tmp_intfc,L,U,dir,(nb+1)%2);

	/*
	 * Split curves that have been disconnected from surfaces
	 * and delete those sections that lie entirely within the
	 * subdomain boundary.
	 */

	cut_out_curves_in_buffer(tmp_intfc);

	reset_intfc_num_points(tmp_intfc);

	if (me[dir] == him[dir])
	{
	    buf_intfc = tmp_intfc;
	}
	else
	{
	    set_size_of_intfc_state(size_of_state(intfc));
	    buf_intfc = copy_interface(tmp_intfc);
	    (void) delete_interface(tmp_intfc);
	}
	set_copy_intfc_states(sav_copy);
	set_current_interface(sav_intfc);

	if (DEBUG)
	{
	    static int outof[3][2];
	    (void) sprintf(dname,"fscatter/cut_buf_interface1/Outof%d-%s_%s",
			   outof[dir][nb],strdir[dir],strnb[nb]);
	    summarize_interface(dname,"intfc",intfc,XY_PLANE,
				"cut_buf_interface1","intfc at EXIT");
	    summarize_interface(dname,"buf_intfc",buf_intfc,XY_PLANE,
				"cut_buf_interface1","buf_intfc at EXIT");
	    ++outof[dir][nb];
	}
	DEBUG_LEAVE(cut_buf_interface1)
	return buf_intfc;
}		/*end cut_buf_interface1*/

EXPORT	void cut_out_curves_in_buffer(
	INTERFACE	*intfc)
{
	BOND  *bs, *be;
	CURVE **c;
	NODE  **n;
	NODE  **nodes_to_delete;

	for (c = intfc->curves; c && *c; ++c)
	{
	    for (bs = (*c)->first; bs != NULL; bs = bs->next)
	        if (Btris(bs) == NULL)
		    break;
	    if (bs == NULL)
		continue;
	    for (be = bs; be->next != NULL; be = be->next)
		if (Btris(be->next) != NULL)
		    break;
	    if ((bs == (*c)->first) && (be == (*c)->last))
	    {
	        /*Entire curve is in buffer region*/
		if (!delete_curve(*c))
		{
		    screen("ERROR in cut_out_curves_in_buffer(), "
		           "can't delete curve %llu\n",curve_number(*c));
		    print_curve(*c);
		    print_interface(intfc);
		    clean_up(ERROR);
		}
		if (intfc->curves == NULL) /*No curves left*/
		    break;
		c = intfc->curves - 1;/*Start over on curve list*/
	    }
	    else if (be != (*c)->last)
	    {
		if (split_curve(be->end,be,*c,NO_COMP,NO_COMP,NO_COMP,NO_COMP)
		    == NULL)
		{
		    screen("ERROR in cut_out_curves_in_buffer(), can't split "
			   "start section of curve %llu\n",curve_number(*c));
		    print_curve(*c);
		    print_interface(intfc);
		    clean_up(ERROR);
		}
		c = intfc->curves - 1;/*Start over on curve list*/
	    }
	    else if (bs != (*c)->first)
	    {
		if (split_curve(bs->start,bs,*c,NO_COMP,NO_COMP,NO_COMP,NO_COMP)
		    == NULL)
		{
		    screen("ERROR in cut_out_curves_in_buffer(), can't split "
			   "end section of curve %llu\n",curve_number(*c));
		    print_curve(*c);
		    print_interface(intfc);
		    clean_up(ERROR);
		}
		c = intfc->curves - 1;/*Start over on curve list*/
	    }
	}
	nodes_to_delete = NULL;
	for (n = intfc->nodes; n && *n; ++n)
	{
	    if (!add_to_pointers(*n,&nodes_to_delete))
	    {
	    	screen("ERROR in cut_out_curves_in_buffer(), "
	    	       "add_to_pointers() failed\n");
	    	clean_up(ERROR);
	    }
	}
	for (n = nodes_to_delete; n && *n; ++n)
	    (void) delete_node(*n);
}		/*end cut_out_curves_in_buffer*/


#if defined(USE_OVERTURE)

LOCAL bool f_form_patch_subintfc_via_cut3d1(
        Front           *fr)
{
        INTERFACE    *intfc = fr->interf;
        INTERFACE    *adj_intfc[2], *sav_intfc, *buf_intfc;
        PP_GRID      *pp_grid = fr->pp_grid;
        RECT_GRID    *gr = fr->rect_grid;
        RECT_GRID    dual_gr; 
        bool         sav_copy;
        bool         status = FUNCTION_SUCCEEDED;
        float        *U = gr->U, *L = gr->L;
        float        *nor, p[3];
        float        T;
        int          *G;
        int          i, j, k, nb;
        int          dir, dim = intfc->dim;

        DEBUG_ENTER(f_form_patch_subintfc_via_cut3d1)

        set_floating_point_tolerance1(fr->rect_grid->h);
        sav_copy = copy_intfc_states();
        sav_intfc = current_interface();
        set_copy_intfc_states(YES);
        set_current_interface(intfc);

        /* Extend interface in three directions */
        /* 
        clip_intfc_at_grid_bdry1(intfc);
        */

        set_dual_grid(&dual_gr,gr);
        strip_subdomain_bdry_curves(intfc);
        for (dir = 0; dir < dim; ++dir)
        {
            for (nb = 0; nb < 2; ++nb)
                open_null_sides1(intfc,gr->VL,gr->VU,dir,nb);
        }
        install_subdomain_bdry_curves(intfc);
        reset_intfc_num_points(intfc);

        /* 
        printf("g_form_patch_subintfc_via_cut3d called1\n");
        printf("interface is reconstructed = %d\n", 
               interface_reconstructed(fr->interf)); 
        print_rectangular_grid(gr); 
        print_rectangular_grid(&dual_gr); 
        print_interface(intfc);  
        printf("EXIT in f_form_patch_subintfc_via_cut3d1\n");
        clean_up(0); 
        */

        DEBUG_LEAVE(f_form_patch_subintfc_via_cut3d1)
        set_copy_intfc_states(sav_copy);
        set_current_interface(sav_intfc);
        return status;
}

LOCAL bool f_form_patch_subintfc_via_cut3d2(
        Front           *fr)
{
        bool         status = FUNCTION_SUCCEEDED;

        DEBUG_ENTER(f_form_patch_subintfc_via_cut3d2)

        DEBUG_LEAVE(f_form_patch_subintfc_via_cut3d2)
        return status; 
}

EXPORT bool f_form_patch_subintfc_via_cut3d(
        Front           *fr)
{
        if (interface_reconstructed(fr->interf))
            return f_form_patch_subintfc_via_cut3d2(fr);
        else
            return f_form_patch_subintfc_via_cut3d1(fr);
}
#endif /* if defined(USE_OVERTURE) */

#endif /* defined(THREED) */
