/*
*				fprop3d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*/

#if defined(THREED)

#define DEBUG_STRING	"tprop3d"

#include <front/fdecs.h>              /* includes int.h, table.h */

typedef struct {
	TRI     *tri; 
	BOND    *b; 
	bool    is_vertex;
	int     side;
	int     iv;
	float   pc[3];
} TN;

typedef struct {
	HYPER_SURF *hs;
	POINT      *p;
	float      *nor;
	float      tan[3]; 
	float      plane[4]; 
	float      ds;
	float      dt;
	TN         tnl, tnr;
} Tparams;


	/* LOCAL Function Declarations */
LOCAL	bool 	plane_side_intersection(const float*,TRI*,int,float*,int*);
LOCAL	bool	set_tangent_space_projection(Front*,Tparams*,TRI**,int);
LOCAL	bool	set_up_tangent_params(Front*,POINT*,HYPER_SURF_ELEMENT*,
	                              HYPER_SURF*,Tparams*);
LOCAL	bool	tn_common_vertex(int,int,TRI*,const TN*,bool);
LOCAL 	bool 	update_state_in_tan_direction(Front*,const Tparams*,POINT*,
                                              float);
LOCAL	void 	fill_stencil_in_direction(Front*,Tan_stencil*,int,int,
	                                  const Tparams*);
LOCAL	void	copy_stencil_in_direction(Tan_stencil*,int,Front*);
LOCAL	void	set_tn_on_tri_side(TN*,TRI*,int);
LOCAL	void	set_tn_on_tri_vertex(TN*,TRI*,int);
LOCAL	int	set_posn_and_states(Tan_stencil*,TRI*,HYPER_SURF*,float*,
				    int,int,Front*);
LOCAL	int	set_rest_of_states(Tan_stencil*,TRI*,HYPER_SURF*,float*,
				    int,int,Front*);
LOCAL	bool is_forward_pn(float*,float*,float*);
LOCAL	bool	check_record_tri(TRI*,TRI**,int*);


#define copy_posn(p1,p2)	(p1)[0] = (p2)[0]; 		\
				(p1)[1] = (p2)[1]; 		\
				(p1)[2] = (p2)[2]


/*ARGSUSED*/
EXPORT	bool f_tan_point_propagate(
	Front              *fr,
	POINT              *p,
	POINT              *newp,
	HYPER_SURF_ELEMENT *hse,
	HYPER_SURF         *hs,
	float dt)
{
	float kappa;
	static 	Tparams tp[2];

	if (Boundary_point(p)) 
	    return YES;
	
	tp[0].dt = tp[1].dt = dt;
	if (!set_up_tangent_params(fr,p,hse,hs,tp))
	    return NO;
	kappa = mean_curvature_at_point(p,hse,hs,fr);

	if (!update_state_in_tan_direction(fr,tp,newp,kappa) ||
	    !update_state_in_tan_direction(fr,tp+1,newp,kappa))
	    return NO;
	return YES;

}		/*end f_tan_point_propagate*/

LOCAL	bool set_up_tangent_params(
	Front			*fr,
	POINT			*p,
	HYPER_SURF_ELEMENT	*hse,
	HYPER_SURF		*hs,
	Tparams			*tp)
{
	INTERFACE *intfc = hs->interface;
	TRI	  **tris;
	float	  *t1 = tp[0].tan;	/* float[3] */
	float	  *t2 = tp[1].tan;	/* float[3] */
	float	  nor[3];
	bool      folded;
	int	  nt;

	normal(p,hse,hs,nor,fr);
	nt = tri_list_computed_by_normal(p,Tri_of_hse(hse),&tris,intfc);

	tp[0].hs  = tp[1].hs  = hs;
	tp[0].p   = tp[1].p   = p;
	tp[0].nor = tp[1].nor = nor;

	principal_tangent(fr,p,hse,hs,nor,t1);
	Cross3d(nor,t1,t2);
	folded = set_tangent_space_projection(fr,tp,tris,nt);

	if (folded ||
	     ( ((tp[0].tnr.tri == NULL) && (tp[0].tnl.tri == NULL)) ||
	       ((tp[1].tnr.tri == NULL) && (tp[1].tnl.tri == NULL)) ) )
	{
	    /* A cusped geometry has been detected, recompute the normal   */
	    /* vector so that the local triangles project in a regular way */
	    /* onto the tangent space                                      */
	    omit_vertex_in_plane_fit();
	    plane_fit_normal3d(p,hse,hs,nor);
	    nt = tri_list_computed_by_normal(p,Tri_of_hse(hse),&tris,intfc);
	    principal_tangent(fr,p,hse,hs,nor,t1);
	    Cross3d(nor,t1,t2);
	    folded = set_tangent_space_projection(fr,tp,tris,nt);
	}

	if (folded ||
	     ((tp[0].tnr.tri == NULL) && (tp[0].tnl.tri == NULL)) ||
	     ((tp[1].tnr.tri == NULL) && (tp[1].tnl.tri == NULL)) )
	{
	    if (debugging("tparams"))
	    {
	    	(void) printf("WARNING in set_up_tangent_params(), "
			  "didn't set tris, nt = %d\n",nt);
	    	(void) printf("p = %llu\n",point_number(p));
	    	(void) printf("Tri_of_hse(hse)\n");
	    	print_tri(Tri_of_hse(hse),intfc);
		clean_up(ERROR);
	    }
	    return NO;
	}
	else
	    return YES;
}		/* end set_up_tangent_params */

LOCAL	bool set_tangent_space_projection(
	Front		   *fr,
	Tparams		   *tp,
	TRI	           **tris,
	int                num_tris)
{
	POINT *p;
    	bool  folded;
	float *t[2], *plane[2];
	float *p0;
	float *h = fr->rect_grid->h;
	float pc[3], pdir[3];
	int   iv, iside, ivtx;
	int   i, j;


	p = tp[0].p;
	t[0] = tp[0].tan;	/* float[3] */
	t[1] = tp[1].tan;	/* float[3] */
	plane[0] = tp[0].plane;	/* float[4] */
	plane[1] = tp[1].plane;	/* float[4] */
	for (i = 0; i < 3; ++i)
	{
	    plane[0][i] =  t[1][i];
	    plane[1][i] = -t[0][i];
	}

	p0 = Coords(p);
	for (i = 0; i < 2; ++i)
	{
	  plane[i][3] = plane[i][0]*p0[0]+plane[i][1]*p0[1]+plane[i][2]*p0[2];
	  tp[i].ds = sqr(t[i][0]/h[0]) + sqr(t[i][1]/h[1]) + sqr(t[i][2]/h[2]);
	  tp[i].ds = 1/sqrt(tp[i].ds);
	  zero_scalar(&tp[i].tnl,sizeof(TN));
	  zero_scalar(&tp[i].tnr,sizeof(TN));
	}
	folded = NO;
	for (i = 0; i < num_tris; ++i)
	{
	  iv = Vertex_of_point(tris[i],p);
	  iside = Next_m3(iv);
	  for (j = 0; j < 2; ++j)
	  {
	    if (plane_side_intersection(plane[j],tris[i],iside,pc,&ivtx))
	    {
	      difference(pc,Coords(p),pdir,3);
	      if (Dot3d(pdir,t[j]) < 0.0)
	      {
	        if (tp[j].tnl.tri != NULL)
	          folded = tn_common_vertex(iv,ivtx,tris[i],&tp[j].tnl,folded);
	        else
	        {
	          copy_posn(tp[j].tnl.pc,pc);
		  if (ivtx == -1)
		    set_tn_on_tri_side(&tp[j].tnl,tris[i],iside);
		  else
		    set_tn_on_tri_vertex(&tp[j].tnl,tris[i],ivtx);
	        }
	      }
	      else
	      {
	        if (tp[j].tnr.tri != NULL)
	          folded = tn_common_vertex(iv,ivtx,tris[i],&tp[j].tnr,folded);
	        else
	        {
	          copy_posn(tp[j].tnr.pc,pc);
	          if (ivtx == -1)
	            set_tn_on_tri_side(&tp[j].tnr,tris[i],iside);
		  else
		    set_tn_on_tri_vertex(&tp[j].tnr,tris[i],ivtx);
	        }
	      }
	    }
	  }
	}
	if ( (tp[0].tnr.tri == NULL) || (tp[0].tnl.tri == NULL) ||
	     (tp[1].tnr.tri == NULL) || (tp[1].tnl.tri == NULL) ) 
	{
	  int iv0, ivl;
	  int side0, sidel;
	  /* Check if p is on a boundary curve */
	  iv0 = Vertex_of_point(tris[0],p);
	  ivl = Vertex_of_point(tris[num_tris-1],p);
	  side0 = sidel = -1;
	  if (num_tris == 1)
	  {
	      if (is_side_bdry(tris[0],iv0) ||
		      (Tri_on_side(tris[0],iv0) == NULL))
	          side0 = iv0;
	      if (is_side_bdry(tris[0],Prev_m3(iv0)) ||
		   (Tri_on_side(tris[0],Prev_m3(iv0)) == NULL))
	          sidel = Prev_m3(iv0);
	  }
	  else if (is_side_bdry(tris[0],iv0) ||
		  (Tri_on_side(tris[0],iv0) == NULL))
	      side0 = iv0;
	  else if (is_side_bdry(tris[0],Prev_m3(iv0)) ||
		   (Tri_on_side(tris[0],Prev_m3(iv0)) == NULL))
	      side0 = Prev_m3(iv0);
	  if (is_side_bdry(tris[num_tris-1],ivl) ||
		  (Tri_on_side(tris[num_tris-1],ivl) == NULL))
	      sidel = ivl;
	  else if (is_side_bdry(tris[num_tris-1],Prev_m3(ivl)) ||
		   (Tri_on_side(tris[num_tris-1],Prev_m3(ivl)) == NULL))
	      sidel = Prev_m3(ivl);
	  if ((side0 != -1) && (sidel != -1))
	  {
	    for (i = 0; i < 2; ++i)
	    {
	      if ((tp[i].tnr.tri == NULL) && (tp[i].tnl.tri != NULL))
	      {
	        tp[i].tnr.tri = tp[i].tnl.tri;
	        tp[i].tnr.is_vertex = YES;
	        tp[i].tnr.iv = Vertex_of_point(tp[i].tnr.tri,p);
	      }
	      else if ((tp[i].tnr.tri != NULL) && (tp[i].tnl.tri == NULL))
	      {
	        tp[i].tnl.tri = tp[i].tnr.tri;
	        tp[i].tnl.is_vertex = YES;
	        tp[i].tnl.iv = Vertex_of_point(tp[i].tnl.tri,p);
	      }
	      else if ((tp[i].tnr.tri == NULL) && (tp[i].tnl.tri == NULL))
	      {
		POINT *popp;
		float d0, dl;
		/* all tris are on one side of the plane */
		popp = (side0 == iv0) ?  Point_of_tri(tris[0])[Next_m3(iv0)] :
		                         Point_of_tri(tris[0])[Prev_m3(iv0)];
	        difference(Coords(popp),Coords(p),pdir,3);
		d0 = Dot3d(pdir,t[i]);
		popp = (sidel == ivl) ?  Point_of_tri(tris[num_tris-1])
			                              [Next_m3(ivl)] :
		                         Point_of_tri(tris[num_tris-1])
					              [Prev_m3(ivl)];
	        difference(Coords(popp),Coords(p),pdir,3);
		dl = Dot3d(pdir,t[i]);
		if (d0 > dl)
		{
	            tp[i].tnr.tri = tris[0];
	            tp[i].tnl.tri = tris[num_tris-1];
		}
		else
		{
	            tp[i].tnl.tri = tris[0];
	            tp[i].tnr.tri = tris[num_tris-1];
		}

	        tp[i].tnr.is_vertex = YES;
	        tp[i].tnr.iv = Vertex_of_point(tp[i].tnr.tri,p);

	        tp[i].tnl.is_vertex = YES;
	        tp[i].tnl.iv = Vertex_of_point(tp[i].tnl.tri,p);
	      }
	    }
	  }
	}
	if (folded ||
	     ( ((tp[0].tnr.tri == NULL) && (tp[0].tnl.tri == NULL)) ||
	       ((tp[1].tnr.tri == NULL) && (tp[1].tnl.tri == NULL)) ) )
	{
	    /* A cusped geometry has been detected, recompute the normal   */
	    /* vector so that the local triangles project in a regular way */
	    /* onto the tangent space                                      */
	    if (debugging("tparams"))
	    {
		POINT      *p;
		float      BBL[3], BBU[3];
		float      unor[3], ut1[3], ut2[3];
		float      len;
		char       s[256];
		const char *scnt;
		static int cnt = 0;

		len = Mag3d(tp[0].nor);
		unor[0] = tp[0].nor[0]/len;
		unor[1] = tp[0].nor[1]/len;
		unor[2] = tp[0].nor[2]/len;
		len = Mag3d(t[0]);
		ut1[0] = t[0][0]/len;
		ut1[1] = t[0][1]/len;
		ut1[2] = t[0][2]/len;
		len = Mag3d(t[1]);
		ut2[0] = t[1][0]/len;
		ut2[1] = t[1][1]/len;
		ut2[2] = t[1][2]/len;

		(void) printf("WARNING in set_tangent_space_projection(), "
			      "Interface folded\n");
	        set_tri_list_bounding_box(tris,num_tris,BBL,BBU,NO,YES);
	        len = sqrt(sqr(BBU[0]-BBL[0]) +
			   sqr(BBU[1]-BBL[1]) +
			   sqr(BBU[2]-BBL[2]));
		unor[0] *= len; unor[1] *= len; unor[2] *= len;
		 ut1[0] *= len;  ut1[1] *= len;  ut1[2] *= len;
		 ut2[0] *= len;  ut2[1] *= len;  ut2[2] *= len;
	        p = tp[0].p;
		set_vector_bounding_box(Coords(p),unor,-1.0,BBL,BBU,YES,YES);
		set_vector_bounding_box(Coords(p),unor,1.0,BBL,BBU,YES,YES);
		set_vector_bounding_box(Coords(p),ut1,-1.0,BBL,BBU,YES,YES);
		set_vector_bounding_box(Coords(p),ut1,1.0,BBL,BBU,YES,YES);
		set_vector_bounding_box(Coords(p),ut2,-1.0,BBL,BBU,YES,YES);
		set_vector_bounding_box(Coords(p),ut2,1.0,BBL,BBU,YES,YES);
		scnt = right_flush(cnt,6);
		(void) sprintf(s,"tan_params-axes.%s",scnt);
		gview_plot_axes("",s,BBL,BBU,BBL,BBU);
		(void) sprintf(s,"tan_params-coords-axes.%s",scnt);
		gview_plot_coord_sys("",s,Coords(p),
			             unor,ut1,ut2,BBL,BBU);
		(void) sprintf(s,"tan_params-tris.%s",scnt);
		gview_plot_triangle_list("",s,tris,num_tris,
			                 0.1,0.0,0.0,0.9,0.0,0.0,0.5,BBL,BBU);
		++cnt;
	    }
	}
	return folded;
}		/*end set_tangent_space_projection*/

LOCAL bool tn_common_vertex(
	int      iv,
	int      ivtx,
	TRI      *tri,
	const TN *tn,
	bool     folded)
{
	if (tn->is_vertex && (ivtx != -1))
	{
	    /*
	     * Both instances occur at a vertex, are the
	     * triangles neighbor's ?
	     */
	    if (ivtx == Next_m3(iv))
	    {
		if (Tri_on_side(tri,iv) != tn->tri)
		    folded = YES;
	    }
	    else if (ivtx == Prev_m3(iv))
	    {
		if (Tri_on_side(tri,ivtx) != tn->tri)
		    folded = YES;
	    }
	    else
	    {
		screen("ERROR in to_common_vertex(), inconsistent indices\n");
		clean_up(ERROR);
	    }
	}
	else
	    folded = YES;
	return folded;
}		/*end tn_common_vertex*/

LOCAL	bool plane_side_intersection(
	const float *plane,
	TRI         *tri,
	int         side,
	float       *pi,
	int         *iv)
{
	float *p1, *p2, dp[3];
	float D1, D2, t;
	float tol = 1.0e-12;/*TOLERANCE*/

	p1 = Coords(Point_of_tri(tri)[side]);
	p2 = Coords(Point_of_tri(tri)[Next_m3(side)]);

	difference(p2,p1,dp,3);

	D1 = Dot3d(plane,p1);
	D2 = Dot3d(plane,dp);

	if (fabs(plane[3] - D1) >= (1.0 + tol)*fabs(D2))
	    return NO;

	t = (plane[3] - D1)/D2;

	if (fabs(t - 1.0) < tol)
	{
	    copy_posn(pi,p2);
	    *iv = Next_m3(side);
	    return YES;
	}
	else if (fabs(t) < tol)
	{
	    copy_posn(pi,p1);
	    *iv = side;
	    return YES;
	}
	else if (tol < t && t < (1.0 - tol)) 
	{
	    pi[0] = p1[0] + t*dp[0];
	    pi[1] = p1[1] + t*dp[1];
	    pi[2] = p1[2] + t*dp[2];
	    *iv = -1;
	    return YES;
	}
	else
	    return NO;
}

LOCAL	bool update_state_in_tan_direction(
	Front         *fr,
	const Tparams *tp,
	POINT         *newp,
	float         kappa)
{
	Locstate 	    ansl, ansr;
	int                 il, ir;
	Locstate            sl,sr;
	static  Tan_stencil *sten = NULL;

	if (sten == NULL)
	    sten = alloc_tan_stencil(fr,fr->npts_tan_sten/2);

	ir =  sten->npts/2 + 1;
	il = -sten->npts/2 - 1;

	if (tp->tnl.tri != NULL)
	    sten->hse[0] = Hyper_surf_element(tp->tnl.tri);
	else if (tp->tnr.tri != NULL)
	    sten->hse[0] = Hyper_surf_element(tp->tnr.tri);
	else
	{
	    (void) printf("WARNING in update_state_in_tan_direction(), "
			  "can't set sten->hse[0]\n");
	    return NO;
	}
	sten->hs[0] = tp->hs;
	sten->p[0] = tp->p;

	slsr(tp->p,Hyper_surf_element(tp->tnl.tri),tp->hs,&sl,&sr);
	assign(sten->leftst[0],sl,fr->sizest);
	assign(sten->rightst[0],sr,fr->sizest);

	fill_stencil_in_direction(fr,sten,-1,il,tp);
	fill_stencil_in_direction(fr,sten,1,ir,tp);

	ansl = left_state(newp);
	ansr = right_state(newp);
	sten->newhs = tp->hs;
	sten->dir = tp->tan;
	sten->curvature = kappa;
	npt_tang_solver(tp->ds,tp->dt,sten,ansl,ansr,fr);
	return YES;
}		/*end update_state_in_tan_direction*/


/*
*
*
*                   set_weight_for_tri_interpolation():
*
*       This function takes a point p in the interior of a triangle TRI 
*	and computes the coefficients of the linear combination
*
*		p = f[0]*p0 + f[1]*p1 + f[2]*p2
*
*	where p0, p1, and p3 are the verticies of tri, and
*	f[0] + f[1] + f[2] = 1.
*/

/*ARGSUSED*/
EXPORT	void set_weight_for_tri_interpolation(
	float	  *p,
	TRI	  *tri,
	float	  *f,
	float	  *h,
	INTERFACE *intfc)
{
	static const float WEIGHT_TOL = 1.0e-5;	/*TOLERANCE*/
	float	           *p0 = Coords(Point_of_tri(tri)[0]);
	float	           *p1 = Coords(Point_of_tri(tri)[1]);
	float	           *p2 = Coords(Point_of_tri(tri)[2]);
	float	           p0p[3], p1p[3], p2p[3], x[3];
	float	           den;
	bool	           renormalize;
	const float        *nor = Tri_normal(tri);

	den = Dot3d(nor,nor); /* NOTE: by construction tri->nor 
			      * equals (p1-p0) x (p2-p0) */
	if (den == 0.0)
	{
	    f[0] = 1.0/3.0;
	    f[1] = 1.0/3.0;
	    f[2] = 1.0/3.0;

	    (void) printf("WARNING in "
	                  "set_weight_for_tri_interpolation(), "
	                  "degenerate tri found\n");
	    print_tri(tri,intfc);
	    return;
	}

	difference(p0,p,p0p,3);   /* Note: difference(a,b,c) is c=a-b*/
	difference(p1,p,p1p,3);
	difference(p2,p,p2p,3);

	Cross3d(p0p,nor,x);     /* Note: Cross3d(a,b,c) is c=axb*/

	f[1] =  Dot3d(p2p,x);
	f[2] = -Dot3d(p1p,x);
	f[0] = den - (f[1] + f[2]);

	if (f[0] < 0.0)
	{
	    Cross3d(p1p,p2p,x);
	    f[0] = Dot3d(nor,x);
	    den = f[0] + f[1] + f[2];
	}
	f[0] /= den;
	f[1] /= den;
	f[2] /= den;

	renormalize = NO;
	if (f[0] < 0.0 && f[0] > -WEIGHT_TOL)
	{
	    f[0] = 0.0;
	    renormalize = YES;
	}
	if (f[1] < 0.0 && f[1] > -WEIGHT_TOL)
	{
	    f[1] = 0.0;
	    renormalize = YES;
	}
	if (f[2] < 0.0 && f[2] > -WEIGHT_TOL)
	{
	    f[2] = 0.0;
	    renormalize = YES;
	}
	if (f[0] < 0.0 || f[1] < 0.0 || f[2] < 0.0)
	{
	    screen("ERROR in set_weight_for_tri_interpolation(), "
	           "negative weight beyond tolerance\n");

	    (void) printf("f = (%g %g %g)\n",f[0],f[1],f[2]);
	    (void) printf("p = (%g %g %g)\n",p[0],p[1],p[2]);
	    (void) printf("p0 = (%g %g %g)\n",p0[0],p0[1],p0[2]);
	    (void) printf("p1 = (%g %g %g)\n",p1[0],p1[1],p1[2]);
	    (void) printf("p2 = (%g %g %g)\n",p2[0],p2[1],p2[2]);
	    (void) printf("Tri_normal(tri) = (%g %g %g)\n",nor[0],nor[1],nor[2]);
	    (void) printf("sqr_norm(tri) = %g\n",sqr_norm(tri));
	    (void) printf("p0 - p = (%g %g %g)\n",
	    	          p0[0]-p[0],p0[1]-p[0],p0[2]-p[2]);
	    (void) printf("p1 - p = (%g %g %g)\n",
			  p1[0]-p[0],p1[1]-p[0],p1[2]-p[2]);
	    (void) printf("p2 - p = (%g %g %g)\n",
	    	          p2[0]-p[0],p2[1]-p[0],p2[2]-p[2]);
	    clean_up(ERROR);
	}
	if (renormalize)
	{
	    den = f[0] + f[1] + f[2];
	    f[0] /= den;
	    f[1] /= den;
	    f[2] /= den;
	}

}		/*end set_weight_for_tri_interpolation*/


LOCAL	void fill_stencil_in_direction(
	Front*		fr,
	Tan_stencil*	sten,
	int		index,
	int		limit,
	const Tparams	*tp)
{
	TRI 		*n_t, *o_t;
	POINT 		*ptmp;
	TN              tn;
	float		l,dist;
	float		po[3], pn[3], posn[3],dir[3];
	int		i, iv, iside, loop_index;
	bool            flag;
	const float	*plane = tp->plane;
	float		ds = tp->ds;
	HYPER_SURF	*hs = tp->hs;
	static	TRI	*tris[50];
	int		num_tris = 0;

	tn = (limit < 0) ? tp->tnl : tp->tnr;

	n_t = tn.tri;
	copy_posn(po,Coords(tp->p));
	copy_posn(pn,tn.pc);
	if (n_t == NULL)
	{
	    copy_stencil_in_direction(sten,limit,fr);
	    return;
	}
	num_tris = 0;

	loop_index = 0;
	dist = ds;
	while (index != limit)
	{
	    ++loop_index;
	    if (loop_index >= 50) 
	    {
		screen("ERROR in fill_stencil_in_direction(), "
		       "loop_index exceeds 50\n");
		print_general_vector("point - ",Coords(sten->p[0]),
				3,"\n");
		clean_up(ERROR);
	    }

	    l = distance_between_positions(po,pn,3);
	    difference(pn,po,dir,3);
            while (index != limit && l >= dist)
            {
		posn[0] = po[0] + dist*dir[0]/l;
		posn[1] = po[1] + dist*dir[1]/l;
		posn[2] = po[2] + dist*dir[2]/l;

		index = set_posn_and_states(sten,n_t,hs,posn,index,limit,fr);
		dist += ds;
	    }

	    dist -= l;
	    l = 0.0;

	    if (((!tn.is_vertex) && (is_side_bdry(n_t,tn.side)))
		||
	        ((tn.is_vertex) && (Boundary_point(Point_of_tri(n_t)[tn.iv]))) 
		||
	        ((check_record_tri(n_t,tris,&num_tris)) && (loop_index != 1)))
	    {
		index = set_rest_of_states(sten,n_t,hs,pn,index,limit,fr);

		return;
	    }

	    copy_posn(po,pn);

	    if (!tn.is_vertex)
	    {
		int is[2];

		o_t = n_t;
		n_t = Tri_on_side(o_t,tn.side);

		/*if (n_t == NULL)	*/				/*TEST*/
		/*{			*/				/*TEST*/
		/*    index = set_rest_of_states(sten,o_t,hs,	*/	/*TEST*/
		/*			       po,index,limit,fr);*/	/*TEST*/
		/*    return;		*/				/*TEST*/
		/*}		*/					/*TEST*/

		ptmp = Point_of_tri(o_t)[tn.side];
		is[0] = Vertex_of_point(n_t,ptmp);
		is[1] = Next_m3(is[0]);

		for (i = 0; i < 2; ++i)
		{
		    flag = plane_side_intersection(plane,n_t,is[i],pn,&iv);
		    if (flag)
	            {
			if (iv == -1)
			    set_tn_on_tri_side(&tn,n_t,is[i]);
			else
			    set_tn_on_tri_vertex(&tn,n_t,iv);
			break;
		    }
		}
	    }
	    else
	    {
		o_t = n_t;
		ptmp = Point_of_tri(n_t)[tn.iv];
		n_t = Next_tri_at_vertex(n_t,ptmp);
		while (n_t != o_t)
		{
		    iv = Vertex_of_point(n_t,ptmp);
		    iside = Next_m3(iv);
		    flag = plane_side_intersection(plane,n_t,iside,pn,&iv);
		    if (flag && is_forward_pn(po,pn,dir))
		    {
			if (iv == -1)
			    set_tn_on_tri_side(&tn,n_t,iside);
			else
			    set_tn_on_tri_vertex(&tn,n_t,iv);
			break;
		    }
		    n_t = Next_tri_at_vertex(n_t,ptmp);
		}
		if (o_t == n_t)
		{
		    index = set_rest_of_states(sten,o_t,hs,po,index,limit,fr);

		    return;
		}
	    }
	}

}		/*end fill_stencil_in_direction*/

LOCAL	bool is_forward_pn(
	float *po,
	float *pn,
	float *t)
{
	float pdir[3];

	difference(pn,po,pdir,3);
	return (Dot3d(pdir,t) > 0.0) ? YES : NO;
}		/* end is_forward_pn */


LOCAL	void copy_stencil_in_direction(
	Tan_stencil *sten,
	int limit,
	Front *fr)
{
	int di,N,i,j;
	Locstate sl, sr, s;


	if (limit > 0)
	{
	    N = limit;
	    di = 1;
	}
	else
	{
	    N = -limit;
	    di = -1;
	}
	sl = sten->leftst[0];
	sr = sten->rightst[0];
	for (i = 0; i < N; ++i)
	{
	    j = di*i;
	    sten->hse[j] = sten->hse[0];
	    sten->hs[j] = sten->hs[0];
	    Coords((sten->p)[j])[0] = Coords((sten->p)[0])[0];
	    Coords((sten->p)[j])[1] = Coords((sten->p)[0])[1];
	    Coords((sten->p)[j])[2] = Coords((sten->p)[0])[2];
	    s = sten->leftst[j];
	    assign(s,sl,fr->sizest);
	    s = sten->rightst[j];
	    assign(s,sr,fr->sizest);
	}
}	/* copy_stencil_in_direction */


LOCAL	int set_rest_of_states(
	Tan_stencil*	sten,
	TRI*		t,
	HYPER_SURF*	hs,
	float*		p,
	int		index,
	int		limit,
	Front*		fr)
{
	int	i, j;
	int	N, di, pindex;
	Locstate sl, sr, s;

	if (limit > 0)
	{
	    N = limit - index;
	    di = 1;
	    pindex = index - 1;
	}
	else
	{
	    N = index - limit;
	    di = -1;
	    pindex = index + 1;
	}
	sl = sten->leftst[pindex];
	sr = sten->rightst[pindex];
	for (i = 0; i < N; ++i)
	{
	    j = index + di*i;
	    sten->hse[j] = Hyper_surf_element(t);
	    sten->hs[j] = hs;
	    Coords((sten->p)[j])[0] = p[0];
	    Coords((sten->p)[j])[1] = p[1];
	    Coords((sten->p)[j])[2] = p[2];
	    s = sten->leftst[j];
	    assign(s,sl,fr->sizest);
	    s = sten->rightst[j];
	    assign(s,sr,fr->sizest);
	}

	return limit;
}		/*end set_rest_of_states*/

LOCAL	int set_posn_and_states(
	Tan_stencil*	sten,
	TRI*		t,
	HYPER_SURF*	hs,
	float*		p,
	int		index,
	int		limit,
	Front*		fr)
{
	Locstate	sl1, sl2, sl3;
	Locstate	sr1, sr2, sr3;
	POINT		*p1,*p2,*p3;
	float		f[3];


	set_weight_for_tri_interpolation(p,t,f,fr->rect_grid->h,hs->interface);

	Coords(sten->p[index])[0] = p[0];
	Coords(sten->p[index])[1] = p[1];
	Coords(sten->p[index])[2] = p[2];
	sten->hse[index] = Hyper_surf_element(t);
	sten->hs[index] = hs;

	p1 = Point_of_tri(t)[0];
	p2 = Point_of_tri(t)[1];
	p3 = Point_of_tri(t)[2];
	slsr(p1,Hyper_surf_element(t),hs,&sl1,&sr1);
	slsr(p2,Hyper_surf_element(t),hs,&sl2,&sr2);
	slsr(p3,Hyper_surf_element(t),hs,&sl3,&sr3);
	if ((tri_interpolate_states(fr,f[0],f[1],f[2],Coords(p1),sl1,
		                    Coords(p2),sl2,Coords(p3),sl3,
				    sten->leftst[index])
					!= FUNCTION_SUCCEEDED)
	    ||
	    (tri_interpolate_states(fr,f[0],f[1],f[2],Coords(p1),sr1,
		                    Coords(p2),sr2,Coords(p3),sr3,
				    sten->rightst[index])
					!= FUNCTION_SUCCEEDED))
	{
	    screen("ERROR in set_posn_and_states(), "
		   "tri_interpolate_states() failed\n");
	}

	return (limit < 0) ? --index : ++index;
}		/*end set_posn_and_states*/

LOCAL	void	set_tn_on_tri_side(
	TN  *tn,
	TRI *tri,
	int is)
{
	tn->tri = tri;
	tn->is_vertex = NO;
	tn->side = is;
}		/*end set_tn_on_tri_side*/

LOCAL	void	set_tn_on_tri_vertex(
	TN  *tn,
	TRI *tri,
	int iv)
{
	tn->tri = tri;
	tn->is_vertex = YES;
	tn->iv = iv;
}

LOCAL	bool check_record_tri(
	TRI *tri,
	TRI **tris,
	int *num_tris)
{
	int i;

	for (i = 0; i < *num_tris; ++i)
	{
	    if (tris[i] == tri)
		return YES;
	}
	tris[(*num_tris)++] = tri;
	return NO;
}

#endif /* defined(THREED) */
