/*
*				fprop2d.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*/

#if defined(TWOD)

#include <front/fdecs.h>

/*
*			f_tan_curve_propagate():
*
*	Performs the tangential sweep update of states on the front.
*	All points with t_pt_propagated(p) == NO are updated.
*/

/* ARGSUSED */
EXPORT	void f_tan_curve_propagate(
	Front		*fr,
	Front		*newfr,
	INTERFACE	*tempintfc,
	CURVE		*tempc,
	CURVE		*newc,
	float		dt)
{
	BOND		    *tempb, *newb;
	Locstate	    ansl, ansr;
	bool		    curveIsClosed;
	float		    *h = fr->rect_grid->h;
	float		    tngt[MAXD], ds, sbl;
	int		    i, dim = fr->rect_grid->dim;
	static	int	    nrad = 0;
	static	Tan_stencil *sten = NULL;

	debug_print("f_tan_prop","Entered f_tan_curve_propagate()\n");
	if (debugging("f_tan_prop"))
	{
	    (void) printf("tempc %llu  newc %llu\n",curve_number(tempc),
	    	          curve_number(newc));
	    (void) printf("tempc\n");	print_curve(tempc);
	    (void) printf("\nnewc\n");	print_curve(newc);
	}

	if (sten == NULL) 
	{
	    nrad = fr->npts_tan_sten/2;
	    sten = alloc_tan_stencil(fr,nrad);
	}

	switch (wave_type(tempc))
	{
	case PASSIVE_BOUNDARY:
	case SUBDOMAIN_BOUNDARY:
	    return;
	default:
	    break;
	}

	curveIsClosed = (is_closed_node(newc->end)) ? YES : NO;

	tempb = tempc->first;		newb  = newc->first;

		/* Check if zero length curve */

	if (tempc->first == tempc->last)
	{
	    sbl = scaled_bond_length(tempc->first,h,dim);
	    if (sbl < MIN_SC_SEP(tempintfc))
	    {
	    	debug_print("f_tan_prop","Left f_tan_curve_propagate()\n");
	    	return;
	    }
	}

	for (;  newb;  tempb = tempb->next, newb = newb->next)
	{
	    if (t_pt_propagated(newb->end))
	        continue;

	    /* stop at tempc->last if no continuation */

	    if ((tempb == tempc->last) &&
	        !curveIsClosed && !is_fixed_node(newc->end))
	    {
	    	break;
	    }

	    /* calculate tangential displacement */

	    /*
	     *  TODO:  the upgrade of this function
	     *  to 3 dimensions is non-trivial.
	     *  There will need to be either two
	     *  operator splitting sweeps,  or one
	     *  unsplit solver.  There is arbitrary
	     *  choice of tangent directions and this
	     *  will have to be resolved.
	     */

	    tangent(newb->end,newb,newc,tngt,newfr);

	    ds = grid_size_in_direction(tngt,h,dim);

	    /* find the stencil states */

	    states_at_distance_along_curve(tempb->end,tempb,tempc,
			                   NEGATIVE_ORIENTATION,ds,nrad,
			                   sten->leftst-1,sten->rightst-1,
			                   sten->hs-1,sten->hse-1,sten->t-1,
					   sten->p-1,newfr);

	    if (tempb->next != NULL)
	    {
	    	ansl  = left_state(newb->end);
	    	ansr  = right_state(newb->end);
	    }
	    else
	    {
	    	ansl  = left_end_state(newc);
	    	ansr  = right_end_state(newc);
	    }

	    states_at_distance_along_curve(tempb->end,tempb,tempc,
			                   POSITIVE_ORIENTATION,ds,nrad,
			                   sten->leftst+1,sten->rightst+1,
			                   sten->hs+1,sten->hse+1,sten->t+1,
					   sten->p+1,newfr);

	    sten->p[0] = tempb->end;
	    sten->hse[0] = Hyper_surf_element(tempb);
	    sten->hs[0] = Hyper_surf(tempc);
	    sten->t[0] = 1.0;
	    sten->curvature = mean_curvature_at_point(sten->p[0],sten->hse[0],
	                                              sten->hs[0],fr);

	    if (debugging("f_tan_prop")) 
	    {
	    	int        j;
	    	static const char *xyz[3] = { "x", "y", "z" };

	        (void) printf("state locations\n");
		(void) printf("%-8s"," ");
		for (i = 0; i < dim; ++i)
		    (void) printf("%-14s",xyz[i]);
		(void) printf("\n");
		for (j = -nrad; j <= nrad; ++j)
		{
		    for (i = 0; i < dim; ++i)
			(void) printf("%-14g",Coords(sten->p[j])[i]);
		    (void) printf("\n");
		}
		(void) printf("\n");
		(void) printf("State values\n");
		for (j = -nrad; j <= nrad; ++j)
		{
		    (void) printf("left state[%d] at ",j);
		    print_general_vector("",Coords(sten->p[j]),dim,"\n");
		    (*fr->print_state)(
			left_state_at_point_on_curve(sten->p[j],
						     Bond_of_hse(sten->hse[j]),
					             Curve_of_hs(sten->hs[j])));
		    (void) printf("right state[%d] at ",j);
		    print_general_vector("",Coords(sten->p[j]),dim,"\n");
		    (*fr->print_state)(
	                right_state_at_point_on_curve(sten->p[j],
						      Bond_of_hse(sten->hse[j]),
					              Curve_of_hs(sten->hs[j])));
		    (void) printf("\n");
		}
	    }

	    /* update using n-point stencil tangential op */

	    sten->newhs = Hyper_surf(newc);
	    sten->dir = tngt;
	    npt_tang_solver(ds,dt,sten,ansl,ansr,fr);
	    t_pt_propagated(newb->end) = YES;

	    if (debugging("f_tan_prop"))
	    {
		(void) printf("answers: left right\n");
		(*newfr->print_state)(ansl);
		(*newfr->print_state)(ansr);
		(void) printf("\n");
	    }
	}

	if (curveIsClosed)
	{
	    /* assign start states to end states */
	    assign(left_start_state(newc),left_end_state(newc),fr->sizest);
	    assign(right_start_state(newc),right_end_state(newc),fr->sizest);
	}
	debug_print("f_tan_prop","Left f_tan_curve_propagate()\n");
}		/*end f_tan_curve_propagate*/








/*
*			oblique_propagate_at_node():
*/

EXPORT void oblique_propagate_at_node(
	Front		*fr,
	POINTER		wave,
	POINT		*newp,
	O_CURVE		*oldc,
	O_CURVE		*newc,
	float		*nor,
	float		dt)
{
	BOND		*oldb;
	POINT		*oldn_posn, *oldp;
	float		save[MAXD], dposn[MAXD];
	float		len, t[MAXD];
	float		V[MAXD];
	int		i, dim = fr->rect_grid->dim;
	void		(*save_impose_bc)(POINT*,BOND*,CURVE*,float*,Front*,
					  bool,bool);

	oldb = Bond_at_node_of_o_curve(oldc);
	oldn_posn = Node_of_o_curve(oldc)->posn;
	oldp = Point_adjacent_to_node(oldc->curve,oldc->orient);
	for (i = 0; i < dim; ++i)
	{
		save[i] = Coords(oldp)[i];
		dposn[i] = save[i] - Coords(oldn_posn)[i];
	}
	t[0] = -nor[1];		t[1] =  nor[0];
	len = mag_vector(t,dim);
	for (i = 0; i < dim; ++i) t[i] /= len;
	if (scalar_product(t,dposn,dim) < 0.0)
	{
		for (i = 0; i < dim; ++i) t[i] = -t[i];
	}
	len = 0.0;
	for (i = 0; i < dim; ++i) len += fabs(t[i]*fr->rect_grid->h[i]);
	for (i = 0; i < dim; ++i)
		Coords(oldp)[i] = Coords(oldn_posn)[i] + len * t[i];
	save_impose_bc = fr->impose_bc;
	fr->impose_bc = NULL;
	point_propagate(fr,wave,oldn_posn,newp,oldb,oldc->curve,dt,V);
	if (newc->orient != oldc->orient) reverse_states_at_point(newp,fr);
	for (i = 0; i < dim; ++i) Coords(oldp)[i] = save[i];
	fr->impose_bc = save_impose_bc;
}		/*end oblique_propagate_at_node*/

/*
*			f_curve_propagate2d():
*
*	Propagates a curve by propagating each point from
*	oldc->first->end to oldc->last->start,
*	through a call to point_propagate.
*	and propagates each bond from old->first
*	to oldc->last through a call to fr->bond_propagate.
*/

EXPORT void f_curve_propagate2d(
	Front		*fr,
	POINTER		wave,
	CURVE		*oldc,
	CURVE		*newc,
	float		dt)
{
	BOND		*oldb = oldc->first;
	BOND		*newb = newc->first;
	float		V[MAXD];
	int		dim = fr->interf->dim;	

	debug_print("f_curve_propagate","Entered f_curve_propagate2d\n");

	if ((fr->_point_propagate == NULL)   ||
	    (oldc == NULL)                   ||
	    (newc == NULL)                   ||
	    (correspond_curve(oldc) != newc) ||
	    (correspond_curve(newc) != oldc))
	    return;


	while (oldb) 
	{
	    if ((oldb != oldc->last) && (!n_pt_propagated(newb->end)))
	    {
	    	point_propagate(fr,wave,oldb->end,newb->end,oldb->next,
				oldc,dt,V);
	    	n_pt_propagated(newb->end) = YES;
	    }
	    if (fr->bond_propagate != NULL)
	    	(*fr->bond_propagate)(fr,wave,oldb,newb,oldc,dt);
	    else
	    	set_bond_length(newb,dim); /* Update new bond length */
	    if (oldb == oldc->last)
		break;
	    oldb = oldb->next;
	    newb = newb->next;
	}
	debug_print("f_curve_propagate","Leaving f_curve_propagate2d\n");
}		/*end f_curve_propagate2d*/

EXPORT	void	set_no_tan_propagate(
	CURVE	*c)
{
	BOND	*b;

	t_pt_propagated(c->first->start) = YES;
	for (b = c->first; b != NULL; b = b->next)
		t_pt_propagated(b->end) = YES;
}		/*end set_no_tan_propagate*/
#endif /* defined(TWOD) */
