/*
*				fprint.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains printing routines for front debugging and for all front
*	structures.
*
*			debug_front(debug_string,title,front)
*			fshow_curve_states(file,curve)
*			show_curve_states(curve)
*			fshow_surface_states(file,surface)
*			show_surface_states(surface)
*			print_bond_and_states(b,c,fr)
*			f_print_front(fr,file)
*			f_print_Front_structure(fr)
*			print_AFLIN(aflin)
*			f_print_rproblem(rp)
*			f_print_rp_node(rpn,rp)
*
*/


#include <front/fdecs.h>		/* includes int.h, table.h */


	/* LOCAL Function Prototypes */
LOCAL	void	   fprint_FlowSpecifiedRegion_list(FILE*,Front*);
#if defined(TWOD)
LOCAL	void	   show_phys_curve_states(INTERFACE*);
#endif /* defined(TWOD) */
#if defined(THREED)
LOCAL	void	   show_states_at_point_on_tri(FILE*,int,TRI*,SURFACE*);
#endif /* defined(THREED) */




/*******************************************************************************
*									       *
*		    Printout for Structures in fdecs.h			       *
*									       *
*******************************************************************************/



EXPORT	void	f_fprint_front(
	Front	*front,
	FILE	*file)
{
	(void) fprintf(file,"  ");/*PRINT VERSION = NUMBER OF BLANKS*/
	fprint_interface(file,front->interf);

	if (debugging("grfrsts"))
	    fgraph_front_states(file,front);
	
	(void) fprintf(file,"\n\n");
	(void) foutput(file);
	(void) fprintf(file,"\t\t\tREDISTRIBUTION INFORMATION:\n");
	(void) fprintf(file,"redistribution count = %d\n",
			     Redistribution_count(front));

	(void) foutput(file);
	(void) fprintf(file,"\t\t\tFRONT SPEEDS:\n");
	print_max_front_speed_info(file,front);
	fprint_FlowSpecifiedRegion_list(file,front);
}		/*end f_fprint_front*/


/*
*			f_print_Front_structure():
*/

EXPORT	void f_print_Front_structure(
	Front		*fr)
{
	(void) printf("\n\n\n\t\tFront %p structure\n",fr);
	if (fr == NULL)
	{
	    (void) printf("\t\tstructure not yet allocated\n");
	    (void) printf("\n\t\tEnd Front %p structure\n\n",fr);
	    return;
	}

	print_RECT_GRID_structure(fr->rect_grid);

	(void) printf("Redistribution parameters:\n\t");
	(void) printf("init_redistribute() %p, curve_redist_func() %p\n\t",
		      Init_redistribution_function(fr),
		      Curve_redistribution_function(fr));
	(void) printf("forward_cur_redist() %p backward_cur_redist() %p\n\t",
		      Forward_curve_redistribute_function(fr),
		      Backward_curve_redistribute_function(fr));
	(void) printf("frequency (general %d vector %d node %d) ",
		      Frequency_of_redistribution(fr,GENERAL_WAVE),
		      Frequency_of_redistribution(fr,VECTOR_WAVE),
		      Frequency_of_redistribution(fr,GENERAL_NODE));
	(void) printf("\tnode_redistribute() %p\n",
		      Node_redistribute_function(fr));
	(void) printf("count %d\n\t",Redistribution_count(fr));
	(void) printf("spacing (general %g vector %g) ",
		      Front_spacing(fr,GENERAL_WAVE),
		      Front_spacing(fr,VECTOR_WAVE));
	(void) printf("cos_big_angle (general %g vector %g)\n\t",
		      Cosine_big_angle(fr,GENERAL_WAVE),
		      Cosine_big_angle(fr,VECTOR_WAVE));
	(void) printf("length %g\n",Front_length(fr));

	(void) printf("timestep control:\ttime_step_factor %g\n",
		      Time_step_factor(fr));
	(void) printf("                :\tapply_cfl_at_nodes %s\n",
		      y_or_n(Apply_CFL_at_nodes(fr)));
	(void) printf("                :\tmax_sep %g\n",
		      Max_new_node_separation(fr));
	(void) printf("                :\tcfl_fudge %g\n",
		      Time_step_increase_factor(fr));
	(void) printf("                :\tfrac_floor %g\n",
		      Min_time_step_modification_factor(fr));
	(void) printf("                :\tfrac_ceil %g\n",
		      Max_time_step_modification_factor(fr));
	(void) printf("\n");
	print_max_front_speed_info(stdout,fr);
	(void) printf("\n");

	(void) printf("\nfront movement:\thyperbolic %d\n",fr->hyperbolic);
	(void) printf("\t\t[NO_STATES %d FULL_STATES %d",NO_STATES,FULL_STATES);
	(void) printf("]\n");
	(void) printf("\tnode_propagate() %p\tcurve_propagate() %p\n",
		      fr->node_propagate,fr->curve_propagate);
	(void) printf("\tpoint_propagate() %p\tbond_propagate() %p\n",
		      fr->_point_propagate,fr->bond_propagate);
	(void) printf("\tsnd_node_propagate() %p\ttan_curve_propagate() %p\n",
		      fr->snd_node_propagate,fr->tan_curve_propagate);
	(void) printf("\t_npt_tang_solver() %p\t\n",fr->_npt_tang_solver);
	(void) printf("\t_one_side_npt_tang_solver() %p\t\n",
		      fr->_one_side_npt_tang_solver);
	(void) printf("\timpose_bc() %p\n",fr->impose_bc);

	(void) printf("\nstate variables:\tsizest %d\tnfloats %d\n",
		      (int)fr->sizest,fr->nfloats);
	(void) printf("_state_interpolator() %p  print_state() %p\n",
		      fr->_state_interpolator,fr->print_state);
	(void) printf("_fgraph_front_states() %p\n",
		      fr->_fgraph_front_states);
	(void) printf("_fprint_header_for_graph_curve_states() %p\n",
		      fr->_fprint_header_for_graph_curve_states);
	(void) printf("_fgraph_curve_states() %p\n",
		      fr->_fgraph_curve_states);
	(void) printf("transform_state() %p\n",fr->transform_state);

	(void) printf("interactions:\tfr_bdry_untangle() %p"
		      "\tuntangle_front() %p\n",
		      fr->fr_bdry_untangle,fr->untangle_front);
	(void) printf("\tB_node_bifurcation() %p\ttwodrproblem() %p\n",
		      fr->B_node_bifurcation,fr->twodrproblem);

	(void) printf("\ninterface:\tinterf %llu\n",
		      interface_number(fr->interf));

	(void) printf("\n\t\tEnd Front %p structure\n\n",fr);

}		/*end f_print_Front_structure*/



EXPORT	void print_AFLIN(
	FILE		*file,
	AFLIN		*aflin,
	int		dim)
{
	int		i, j;

	(void) fprintf(file,"\n\n\n\t\tAFLIN %p structure\n",aflin);
	if (aflin == NULL)
	{
	    (void) fprintf(file,"\t\tstructure not yet allocated\n");
	    (void) fprintf(file,"\n\t\tEnd AFLIN %p structure\n\n",aflin);
	    return;
	}
	for (i = 0; i < dim; ++i)
	{
	    for (j = 0; j < dim; ++j)
	    	(void) fprintf(file,"a[%d][%d] %g ",i,j,aflin->a[i][j]);
	    (void) fprintf(file,"\n");
	}
	(void) fprintf(file,"\n");
	for (i = 0; i < dim; ++i)
	    (void) fprintf(file,"b[%d] %g ",i,aflin->b[i]);
	(void) fprintf(file,"det %g\n",aflin->det);
	(void) fprintf(file,"\n\t\tEnd AFLIN %p structure\n\n",aflin);
}		/*end print_AFLIN*/

EXPORT	void	print_time_step_status(
	const char *mesg,
	int	   status,
	const char *end)
{
	(void) printf("%s%s%s",mesg,time_step_status_as_string(status),end);
}		/*end print_time_step_status*/

EXPORT	const char* time_step_status_as_string(
	int	   status)
{
	static char s[120];
	switch (status)
	{
	case GOOD_STEP:
	    return "GOOD_STEP";
	case MODIFY_TIME_STEP:
	    return "MODIFY_TIME_STEP";
	case REPEAT_TIME_STEP:
	    return "REPEAT_TIME_STEP";
	case ERROR_IN_STEP:
	    return "ERROR_IN_STEP";
	default:
	    (void) sprintf(s,"UNKNOWN %d",status);
	    return s;
	}
}		/*end time_step_status_as_string*/

EXPORT	void	f_fprint_max_front_speed_info(
	FILE	*file,
	Front	*front)
{
	int	i, j, dim = front->rect_grid->dim;

	(void) fprintf(file,"Maximum Front Speed Information\n");
	(void) fprintf(file,"Spfr = ");
	if (is_binary_output() == YES)
	{
	    (void) fprintf(file,"\f%c",dim+1);
	    (void) fwrite((const void *) Spfr(front),sizeof(float),dim+1,file);
	}
	else
	{
	    for (i = 0; i < dim; ++i)
	    	(void) fprintf(file,"%"FFMT", ",Spfr(front)[i]);
	    (void) fprintf(file,"%"FFMT,Spfr(front)[dim]);
	}
	(void) fprintf(file,"\n");
	for (i = 0; i <= dim; ++i)
	{
	    (void) fprintf(file,"MaxFrontSpeedCoords[%d] = ",i);
	    if (is_binary_output() == YES)
	    {
	        (void) fprintf(file,"\f%c",dim);
	    	(void) fwrite((const void *) MaxFrontSpeedCoords(front)[i],
			      sizeof(float),dim,file);
	    }
	    else
	    {
	    	(void) fprintf(file,"%"FFMT,MaxFrontSpeedCoords(front)[i][0]);
		for (j = 1; j < dim; ++j)
		    (void) fprintf(file,", %"FFMT,
				   MaxFrontSpeedCoords(front)[i][j]);
	    }
	    (void) fprintf(file,"\n");
	}
	for (i = 0; i <= dim; ++i)
	{
	    (void) fprintf(file,"MaxFrontSpeedState[%d] = ",i);
	    fprint_state_data(file,MaxFrontSpeedState(front)[i],front->interf);
	}
}		/*end f_fprint_max_front_speed_info*/


#include <sys/types.h>
#include <sys/stat.h>

/*
*			debug_front():
*
*	Prints the interface of a front if debugging is requested.
*	There is a dual control structure.  The first involves 
*	debug_string, which controls whether any debugging is printed
*	based on the location of the call to debug_front().  The 
*	second determines what kind of information to print out.
*
*	The current choices for debug_string from advance_front() are:
*		old_front	initial front into advance_front
*		cp_front	after normal curve propagation
*		2drp_front	after rp loop
*		node_front	after node loop
*		ztc_front	before/after zero timestep correction
*		dec_front	after delete_exterior_curves()
*		dvsb_front	after delete_very_short_bonds()
*		dfbb_front	after delete_fold_back_bonds()
*		snd_front	after snd_node_prop()
*		tcp_front	after tan curve propagate
*		dspr_front	after delete_phys_remn_on_bdry()
*		redist_front	after redistribution
*		dsloop_front	after delete_small_loops()
*		ERROR_front	after any error in the front propagate
*	Debugging on front basically turns on all of these.
*
*	The current choices for type of output are:
*		front_intfc	call print_interface()
*		graph_front_states	call graph_front_states()
*		front_states	call show_intfc_states()
*		phys_fr_states	call show_phys_curve_states()
*		plot_front	call plot_interface()
*
*	NOTE:  to anyone who adds a call(s) to this function, PLEASE
*		update these lists.
*
*	NOTE 2: It is not sufficient to debug on a string from the first
*		list.  You must also debug on at least one string from
*		the second list in order to get any output.
*/

EXPORT void debug_front(
	const char	*debug_string,
	const char	*title,
	Front		*fr)
{
	bool deb = (debugging(debug_string) || debugging("front")) ? YES : NO;

	if (debugging("time_step"))
	{
	    (void) printf("Spfr(fr) %s = ",title);
	    print_general_vector("",Spfr(fr),fr->rect_grid->dim+1,"\n");
	}

	if (!((strcmp(debug_string,"ERROR_front")==0) || deb))
	    return;

	if (deb && debugging("front_intfc"))
	{
	    (void) output();
	    (void) printf("\t\tFront %s\n",title);
	    print_interface(fr->interf);
	}
	if (deb && debugging("graph_fr")) 
	{
	    (void) output();
	    (void) printf("\t\tGraph of front states %s\n",title);
	    graph_front_states(fr);
	}
	if (deb && debugging("front_states")) 
	{
	    (void) output();
	    (void) printf("\t\tFront states %s\n",title);
	    show_intfc_states(fr->interf);
	}
#if defined(TWOD)
	if (deb && debugging("phys_fr_states") && fr->rect_grid->dim==2) 
	{
	    (void) output();
	    (void) printf("\t\tPhysical curve states %s\n",title);
	    show_phys_curve_states(fr->interf);
	}
#endif /* defined(TWOD) */
	if (deb && debugging("plot_front"))
	{
	    static int     ctr = 0;
	    static int     step = -1;
	    static bool first = YES;
	    char           fname[1024];
	    static const char     *dname = "debug_front";
	    static const char     *funcname = "debug_front()";

	    if (first == YES)
	    {
	    	first = NO;
		if (create_directory(dname,YES) == FUNCTION_FAILED)
	    	{
	    	    screen("ERROR in %s directory %s doesn't exist ",
		           "and can't be made\n",funcname,dname);
		    clean_up(ERROR);
		}
	    }
	    if (step < 0 || fr->step != step)
	    {
	    	step = fr->step;
	    	ctr = 0;
	    }
	    (void) sprintf(fname,"%s/plt",dname);
	    plot_interface(fr->interf,fname,&step,&ctr,title);
	    ++ctr;
	}
}		/*end debug_front*/


LOCAL	void	fprint_FlowSpecifiedRegion_list(
	FILE	*file,
	Front	*fr)
{
	FlowSpecifiedRegion *fsr;
	int n;

	(void) foutput(file);
	(void) fprintf(file,"FLOW SPECIFIED REGIONS DATA LIST\n");
	for (n = 0, fsr = Fsr_list(fr); fsr != NULL; ++n, fsr = fsr->next)
	    ;
	(void) fprintf(file,"Number of regions = %d\n",n);

	for (fsr = Fsr_list(fr); fsr != NULL; fsr = fsr->next)
	    fprint_FlowSpecifiedRegion_data(file,fsr,fr);

	(void) foutput(file);
	(void) fprintf(file,"END FLOW SPECIFIED REGIONS DATA LIST\n");
}		/*end fprint_FlowSpecifiedRegion_list*/

/*ARGSUSED*/
EXPORT	void	f_fprint_FlowSpecifiedRegion_data(
	FILE			*file,
	FlowSpecifiedRegion	*fsr,
	Front			*fr)
{
	(void) fprintf(file,"\nData for FlowSpecifiedRegion structure %p\n",
		       fsr);
	(void) fprintf(file,"comp = %d\n",fsr->comp);
	(void) fprintf(file,"type = %s\n",fsr->type);
}		/*end f_fprint_FlowSpecifiedRegion_data*/


EXPORT	void	fprint_Tan_stencil(
	FILE		*file,
	Front		*front,
	Tan_stencil	*sten)
{
	int		i;
	int		dim = front->rect_grid->dim;
	int		nrad = sten->npts/2;
	char		s[80];

	(void) fprintf(file,"Data for Tan_stencil %p\n",sten);
	(void) fprintf(file,"npts = %d\n",sten->npts);
	for (i = -nrad; i <= nrad; ++i)
	{
	    (void) sprintf(s,"Coords(sten->p[%d]) = ",i);
	    fprint_general_vector(file,s,Coords(sten->p[i]),dim,"\n");
	    (void) fprintf(file,"sten->t[%d] = %g\n",i,sten->t[i]);
	    (void) fprintf(file,"sten->leftst[%d]\n",i);
	    fprint_state_data(file,sten->leftst[i],front->interf);
	    (void) fprintf(file,"sten->rightst[%d]\n",i);
	    fprint_state_data(file,sten->rightst[i],front->interf);
	}
}		/*end fprint_Tan_stencil*/

EXPORT	void	print_Tan_stencil(
	Front		*front,
	Tan_stencil	*sten)
{
	fprint_Tan_stencil(stdout,front,sten);
}		/*end print_Tan_stencil*/


#if defined(TWOD)

/*
*			f_print_rproblem():
*/

EXPORT void f_print_rproblem(
	RPROBLEM	*rp)
{
	RP_NODE		*rpn;

	(void) printf("\n\t\tRPROBLEM %p:\n",rp);
	if (!rp)
	{
	    (void) printf("\tEND OF RPROBLEM %p:\n\n",rp);
	    return;
	}

	(void) printf("prev %p  next %p\n",rp->prev,rp->next);
	(void) printf("new_intfc %llu  old_intfc %llu  ",
		      interface_number(rp->new_intfc),
		      interface_number(rp->old_intfc));
	(void) printf("dt %g  dt_frac %g\n\n",rp->dt,rp->dt_frac);

	/* 
	*	Even though these sets of interacting nodes are
	*	reprinted in the rp_node printout below, it is
	*	convenient to print the node list separately in 
	*	order to see the list of interacting nodes as
	*	a single object.  The problem is that the
	*	print_rp_node() function prints so much that
	*	it is difficult to remember what are the other
	*	interacting nodes in the list.
	*/

	(void) printf("Old Interacting Nodes\n");
	for (rpn = rp->first_rp_node; rpn; rpn = rpn->next)
	    print_node(rpn->old_node);
	(void) printf("End Old Interacting Nodes\n");

	(void) printf("New Interacting Nodes\n");
	for (rpn = rp->first_rp_node; rpn; rpn = rpn->next)
	{
	    print_node(rpn->node);
	    print_propagation_status(rpn->node);
	    (void) printf("\n\n");
	}
	(void) printf("End New Interacting Nodes\n");

	(void) printf("\t\tRp_nodes:\n");
	for (rpn = rp->first_rp_node; rpn; rpn = rpn->next)
	    print_rp_node(rpn,rp);
	(void) printf("\tEnd Rp_nodes\n\n");

	(void) printf("\tAngle ordered curves:\n");
	print_o_curve_family(rp->ang_ordered_curves);
	(void) printf("\tEnd Angle ordered curves\n\n");

	(void) printf("\tOld Angle ordered curves:\n");
	print_o_curve_family(rp->old_ang_ordered_curves);
	(void) printf("\tEnd Old Angle ordered curves:\n\n");

	(void) printf("\t\tBoundary curves:\n");
	print_o_curve_family(rp->bdry_curves);
	(void) printf("\tEnd Boundary curves\n\n");

	(void) printf("\tOld Boundary curves:\n");
	print_o_curve_family(rp->old_bdry_curves);
	(void) printf("\tEnd Old Boundary curves\n\n");

	(void) printf("\tStatistical information\n");
	(void) printf("bdry_type1 %d  bdry_type2 %d\n",
		      rp->bdry_type1,rp->bdry_type2);
	(void) printf("num_nd %d  num_fxd %d  num_srce %d\n",
		      rp->num_nd,rp->num_fxd,rp->num_srce);
	(void) printf("num_nod %d  num_bdry_nod %d  num_phys %d\n",
		      rp->num_nod,rp->num_bdry_nod,rp->num_phys);
	(void) printf("\tEnd Statistical information\n\n");

	user_print_rproblem(rp);
	(void) printf("\tEND OF RPROBLEM %p:\n\n",rp);
}		/*end f_print_rproblem*/


/*
*			f_print_rp_node():
*/

/*ARGSUSED*/
EXPORT void f_print_rp_node(
	RP_NODE		*rpn,
	RPROBLEM	*rp)
{
	(void) printf("\t\tRp_node %p:\n",rpn);
	if (!rpn)
	{
	    (void) printf("\tEnd of Rp_node %p:\n\n",rpn);
	    return;
	}

	(void) printf("prev %p  next %p\n\n",rpn->prev,rpn->next);

	(void) printf("(new) node - "); print_node(rpn->node);
	(void) printf("old node ");	 print_node(rpn->old_node);
	(void) printf("states_assigned_at_node %s\n",
		      (rpn->states_assigned_at_node) ? "YES" : "NO");

	(void) printf("\t\tBoundary curves:\n");
	(void) printf("\tNeumann1:\n");	print_o_curve(rpn->neumann1);
	(void) printf("\tNeumann2:\n");	print_o_curve(rpn->neumann2);
	(void) printf("\tDirichlet1:\n");	print_o_curve(rpn->dirichlet1);
	(void) printf("\tDirichlet2:\n");	print_o_curve(rpn->dirichlet2);
	(void) printf("\tSubdomain1:\n");	print_o_curve(rpn->subdomain1);
	(void) printf("\tSubdomain2:\n");	print_o_curve(rpn->subdomain2);
	(void) printf("\tEnd Boundary curves\n\n");
	
	user_print_rp_node(rpn,rp);
	(void) printf("\tEnd of Rp_node %p:\n\n",rpn);
}		/*end f_print_rp_node*/


EXPORT	void	print_untangle_status(
	int		status)
{
	(void) printf("status = %s\n",untangle_status_as_string(status));
}		/*end print_untangle_status*/

EXPORT	void print_propagation_status(
	NODE		*node)
{
	(void) printf("Propagation status of node %llu = ",node_number(node));
	if (node == NULL)
	{
	    (void) printf("NULL NODE\n");
	    return;
	}
	(void) printf("%s\n",
		      propagation_status_as_string(propagation_status(node)));
}		/*end print_propagation_status*/


EXPORT	const char *untangle_status_as_string(
	int		status)
{
	static char s[120];
	switch (status)
	{
	case CURVES_UNTANGLED:
	    return "CURVES_UNTANGLED";
	case MODIFY_TIME_STEP_TO_UNTANGLE:
	    return "MODIFY_TIME_STEP_TO_UNTANGLE";
	case ERROR_IN_UNTANGLE:
	    return "ERROR_IN_UNTANGLE";
	default:
	    (void) sprintf(s,"UNKNOWN UNTANGLE STATUS %d",status);
	    return s;
	}
}		/*end untangle_status_as_string*/

EXPORT	const char *propagation_status_as_string(
	NODE_PROPAGATION_STATUS	status)
{
	static char s[120];
	switch (status)
	{
	case UNPROPAGATED_NODE:
	    return "UNPROPAGATED_NODE";
	case VEL_COMPUTED_NODE:
	    return "VEL_COMPUTED_NODE";
	case PROPAGATED_NODE:
	    return "PROPAGATED_NODE";
	case DELETED_NODE:
	    return "DELETED_NODE";
	default:
	    (void) sprintf(s,"UNKNOWN PROPAGATION STATUS %d",status);
	    return s;
	}
}		/*end propagation_status_as_string*/


LOCAL	void show_phys_curve_states(
	INTERFACE	*intfc)
{
	CURVE		**c;

	if (intfc->dim != 2) return;

	if (size_of_state(intfc) == 0)
	{
	    (void) printf("No states assigned on intfc\n");
	    return;
	}

	(void) printf("\t\tSTATES ON THE FRONT\n\n");
	for (c = intfc->curves;  *c != NULL;  ++c)
	{
	    if (wave_type(*c) < FIRST_PHYSICS_WAVE_TYPE)
	        continue;
	    show_curve_states(*c);
	}
	(void) printf("\n\n");
	(void) printf("\t\tEND OF STATES ON THE FRONT\n\n");
}		/*end show_phys_curve_states*/


/*
*		show_curve_states():
*		fshow_curve_states():
*
*	Prints states on a curve using print_intfc_state().
*/

EXPORT	void	show_curve_states(
	CURVE		*c)
{
	fshow_curve_states(stdout,c);
}		/*end show_curve_states*/

EXPORT	void fshow_curve_states(
	FILE		*file,
	CURVE		*c)
{
	INTERFACE	*intfc;
	Locstate	sl, sr;
	BOND		*b;
	size_t		sizest;
	const char	**Dnm = computational_grid(c->interface)->Remap.Dnm;

	if (c == NULL)
	    return;
	intfc = c->interface;

	sizest = size_of_state(intfc);

	if (sizest == 0)
	{
	    (void) fprintf(file,"No states assigned on curve\n");
	    return;
	}

	(void) fprintf(file,"\t\tSTATES ON CURVE\n\n");
	(void) fprintf(file,"    %-10s %-10s %-10s %llu ",Dnm[0],Dnm[1],
		       "STATES on CURVE_",curve_number(c));
	(void) fprintf(file,"%s\n",(is_bdry(c)) ? "Boundary curve" :
						  "Interior curve");

	if (sizest == FLOAT)
	{
	    (void) fprintf(file,"\t\t\t    l_st\tr_st\n");

	    sl = left_start_state(c);       sr = right_start_state(c);
	    (void) fprintf(file,"start curve states ");
	    (void) fprintf(file,"%-11g %-11g\n",
	    	           ((float *)sl)[0],((float *)sr)[0]);
 
	    sl =  left_state(c->start->posn);
	    sr = right_state(c->start->posn);
	    (void) fprintf(file,"start posn states ");
	    (void) fprintf(file,"%-11g %-11g\n",
			   ((float *)sl)[0],((float *)sr)[0]);
	    (void) fprintf(file,"\n");

	    b = c->first;
	    slsr(b->start,Hyper_surf_element(b),Hyper_surf(c),&sl,&sr);

	    (void) fprintf(file,"%-11g %-11g  ",
	    	           Coords(b->start)[0],Coords(b->start)[1]);
	    (void) fprintf(file,"%-11g %-11g\n",
	    	           ((float *)sl)[0],((float *)sr)[0]);

	    while (b != NULL)
	    {
	    	slsr(b->end,Hyper_surf_element(b),Hyper_surf(c),&sl,&sr);

		(void) fprintf(file,"%-11g %-11g  ",
				    Coords(b->end)[0],Coords(b->end)[1]);
		(void) fprintf(file,"%-11g %-11g\n",
				    ((float *)sl)[0],((float *)sr)[0]);

		b = b->next;
	    }
	    (void) fprintf(file,"\n");

	    sl = left_end_state(c);         sr = right_end_state(c);
	    (void) fprintf(file,"end curve states ");
	    (void) fprintf(file,"%-11g %-11g\n",
			   ((float *)sl)[0],((float *)sr)[0]);

	    sl =  left_state(c->end->posn);
	    sr = right_state(c->end->posn);
	    (void) fprintf(file,"end posn states ");
	    (void) fprintf(file,"%-11g %-11g\n",
			   ((float *)sl)[0],((float *)sr)[0]);
	    (void) fprintf(file,"\n");
	}
	else if (sizest == 2 * FLOAT)
	{
	    (void) fprintf(file,"\t\t\t    l_st\t\tr_st\n");
	    b = c->first;
	    slsr(b->start,Hyper_surf_element(b),Hyper_surf(c),&sl,&sr);

	    (void) fprintf(file,"%-11g %-11g  ",
	    	                Coords(b->start)[0],Coords(b->start)[1]);
	    (void) fprintf(file,"%-11g %-11g " ,
			        ((float *)sl)[0],((float *)sl)[1]);
	    (void) fprintf(file,"%-11g %-11g\n",
			        ((float *)sr)[0],((float *)sr)[1]);

	    while (b != NULL)
	    {
	    	slsr(b->end,Hyper_surf_element(b),Hyper_surf(c),&sl,&sr);

		(void) fprintf(file,"%-11g %-11g  ",
				    Coords(b->end)[0],Coords(b->end)[1]);
		(void) fprintf(file,"%-11g %-11g ",
				    ((float *)sl)[0],((float *)sl)[1]);
		(void) fprintf(file,"%-11g %-11g\n",
				    ((float *)sr)[0],((float *)sr)[1]);

		b = b->next;
	    }
	    (void) fprintf(file,"\n");
	}
	else
	{
	    b = c->first;
	    sl = left_start_state(c);
	    sr = right_start_state(c);

	    (void) fprintf(file,"%g %g\n",
			        Coords(b->start)[0],Coords(b->start)[1]);
	    (void) fprintf(file,"\t\tl_st ");
	    fprint_intfc_state(file,sl,intfc);
	    (void) fprintf(file,"\t\tr_st ");
	    fprint_intfc_state(file,sr,intfc);

	    for (; b->next != NULL; b = b->next)
	    {
	    	slsr(b->end,Hyper_surf_element(b),Hyper_surf(c),&sl,&sr);

		(void) fprintf(file,"%g %g\n",
				    Coords(b->end)[0],Coords(b->end)[1]);
		(void) fprintf(file,"\t\tl_st ");
		fprint_intfc_state(file,sl,intfc);
		(void) fprintf(file,"\t\tr_st ");
		fprint_intfc_state(file,sr,intfc);
	    }

	    sl = left_end_state(c);
	    sr = right_end_state(c);

	    (void) fprintf(file,"%g %g\n",Coords(b->end)[0],Coords(b->end)[1]);
	    (void) fprintf(file,"\t\tl_st ");
	    fprint_intfc_state(file,sl,intfc);
	    (void) fprintf(file,"\t\tr_st ");
	    fprint_intfc_state(file,sr,intfc);

	    (void) fprintf(file,"\n");
	}
	(void) fprintf(file,"\n");
	(void) fprintf(file,"\n");
	(void) fprintf(file,"\t\tEND OF STATES ON CURVE %llu\n\n",
		       curve_number(c));
}		/*end fshow_curve_states*/


EXPORT void print_bond_and_states(
	BOND		*b,
	CURVE		*c,
	Front		*fr)
{
	if (b == NULL)
	{
	    (void) printf("bond = NULL\n");
	    return;
	}
	(void) printf("bond (%llu):  %g %g -> %g %g\n",
		      bond_number(b,fr->interf),
		      Coords(b->start)[0],Coords(b->start)[1],
		      Coords(b->end)[0],Coords(b->end)[1]);
	(void) printf("prev = %llu next = %llu\n",bond_number(b->prev,fr->interf),
		      bond_number(b->next,fr->interf));
	(void) printf("left/right states at b->start:\n");
	(*fr->print_state)(left_state_at_point_on_curve(b->start,b,c));
	(*fr->print_state)(right_state_at_point_on_curve(b->start,b,c));
	(void) printf("left/right states at b->end:\n");
	(*fr->print_state)(left_state_at_point_on_curve(b->end,b,c));
	(*fr->print_state)(right_state_at_point_on_curve(b->end,b,c));
}		/*end print_bond_and_states*/

EXPORT	void	fprint_redistribution_direction(
	FILE                     *file,
	const char               *mesg,
	REDISTRIBUTION_DIRECTION dir,
	const char               *end)
{
	(void) fprintf(file,"%s%s%s",mesg,
				     redistribution_direction_as_string(dir),
				     end);
}		/*end fprint_redistribution_direction*/

EXPORT	const char *redistribution_direction_as_string(
	REDISTRIBUTION_DIRECTION direction)
{
	static char s[120];
	switch (direction)
	{
	case FORWARD_REDISTRIBUTION:
	    return "FORWARD_REDISTRIBUTION";
	case BACKWARD_REDISTRIBUTION:
	    return "BACKWARD_REDISTRIBUTION";
	default:		    
	    (void) sprintf(s,"UNKNOWN REDISTRIBUTION DIRECTION %d",direction);
	    return s;
	}
}		/*end redistribution_direction_as_string*/
#endif /* defined(TWOD) */

#if defined(THREED)
/*
*		show_surface_states():
*		fshow_surface_states():
*
*	Prints states on a surface using print_intfc_state().
*/

EXPORT	void show_surface_states(
	SURFACE		*s)
{
	fshow_surface_states(stdout,s);
}		/*end show_surface_states*/

EXPORT	void fshow_surface_states(
	FILE		*file,
	SURFACE		*s)
{
	INTERFACE	*intfc;
	TRI		*tri;
	size_t		sizest;
	int		i;

	if (s == NULL)
	    return;
	intfc = s->interface;

	sizest = size_of_state(intfc);

	if (sizest == 0)
	{
	    (void) fprintf(file,"No states assigned on surface\n");
	    return;
	}

	(void) fprintf(file,"\t\tSTATES ON %s SURFACE %llu, ",
		     (is_bdry(s)) ? "BOUNDARY" : "INTERIOR",surface_number(s));
	fprint_wave_type(file,"WAVE TYPE = ",wave_type(s),"\n",intfc);
	(void) fprintf(file,
		"\tPositive Component = %-4d   Negative Component = %-4d    ",
		positive_component(s),negative_component(s));
	(void) fprintf(file,"%s\n\n",
		       is_bdry(s) ? "Boundary Surface" : "Interior Surface");

	(void) fprintf(file,"States on Tris\n");
	for (i = 0, tri = first_tri(s); !at_end_of_tri_list(tri,s);
	     					++i, tri = tri->next)
	{
	    (void) fprintf(file,"States on Tri %d, Boundary = %d\n",
	    	                i,Boundary_tri(tri));

	    show_states_at_point_on_tri(file,0,tri,s);
	    show_states_at_point_on_tri(file,1,tri,s);
	    show_states_at_point_on_tri(file,2,tri,s);

	    (void) fprintf(file,"End States on Tri %d\n",i);
	}
	(void) fprintf(file,"\n");
	(void) fprintf(file,"\t\tEND OF STATES ON SURFACE\n\n");

}		/*end fshow_surface_states*/


LOCAL	void	show_states_at_point_on_tri(
	FILE	*file,
	int	i,
	TRI	*tri,
	SURFACE	*s)
{
	INTERFACE	*intfc = s->interface;
	Locstate	sl, sr;
	POINT		*p = Point_of_tri(tri)[i];

	(void) fprintf(file,"tri->p%d %g %g %g, Boundary = %d\n",i+1,
		      Coords(p)[0],Coords(p)[1],Coords(p)[2],Boundary_point(p));
	slsr(p,Hyper_surf_element(tri),Hyper_surf(s),&sl,&sr);
	(void) fprintf(file,"\t\tl_st ");
	fprint_intfc_state(file,sl,intfc);
	(void) fprintf(file,"\t\tr_st ");
	fprint_intfc_state(file,sr,intfc);
}		/*end show_states_at_point_on_tri*/

EXPORT	void print_tri_states(
	TRI                *tri,
	HYPER_SURF	   *hs)
{
	INTERFACE *intfc = hs->interface;
	HYPER_SURF_ELEMENT *hse = Hyper_surf_element(tri);
	Locstate  sl, sr;
	POINT     *p;
	int       i;

	for (i = 0; i < 3; ++i)
	{
	    p = Point_of_tri(tri)[i];
            slsr(p,hse,hs,&sl,&sr);
	    (void) printf("left state at p[%d]\n",i);
	    print_state_data(sl,intfc);
	    (void) printf("right state at p[%d]\n",i);
	    print_state_data(sr,intfc);
	}
}		/*end print_tri_states*/

EXPORT  void    f_gview_plot_interface(
	const char *dname,
	INTERFACE  *intfc)
{
	RECT_GRID *gr = computational_grid(intfc);
	geomview_interface_plot(dname,intfc,gr);
}               /*end f_default_gview_plot_interface*/
#endif /* defined(THREED) */

#if defined(TWOD) || defined(THREED)
EXPORT	void print_node_status(
	const char *mesg,
	int	   status,
	const char *end)
{
	(void) printf("%s%s%s",mesg,node_status_as_string(status),end);
}		/*end print_node_status*/

EXPORT	const char	*node_status_as_string(
	int	status)
{
	static char s[120];
	switch (status)
	{
	case GOOD_NODE:
	    return "GOOD_NODE";
	case PSEUDOCROSS_NODE_NODE:
	    return "PSEUDOCROSS_NODE_NODE";
	case CROSS_NODE_NODE:
	    return "CROSS_NODE_NODE";
	case CROSS_PAST_CURVE_NODE:
	    return "CROSS_PAST_CURVE_NODE";
	case PSEUDOCROSS_CURVE_NODE:
	    return "PSEUDOCROSS_CURVE_NODE";
	case CROSS_CURVE_NODE:
	    return "CROSS_CURVE_NODE";
	case NO_CROSS_NODE:	 
	    return "NO_CROSS_NODE";
	case BIFURCATION_NODE:
	    return "BIFURCATION_NODE";
	case ERROR_NODE:	
	    return "ERROR_NODE";
	case NO_STORAGE_NODE:	
	    return "NO_STORAGE_NODE";
	case MODIFY_TIME_STEP_NODE:
	    return "MODIFY_TIME_STEP_NODE";
	case REPEAT_TIME_STEP_NODE: 
	    return "REPEAT_TIME_STEP_NODE";
	default:		    
	    (void) sprintf(s,"UNKNOWN NODE STATUS %d",status);
	    return s;
	}
}		/*end node_status_as_string*/

#endif /* defined(TWOD) || defined(THREED) */
