/*
*                               fpatch3d.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*       this file includes functions set_patch_front, set_patch_topo_grid,
*                       and set_patch_comput_grid
*/

#define DEBUG_STRING    "fpatch3d"

#include <front/flocaldecs.h>

#if defined(THREED)
#if defined(USE_OVERTURE)

        /* The bounding box for merging interface is
         * defined as the following: in the direction two
         * boxes are attached together, the cell is taken
         * as (+1) and (-1) centered on the attached plane.
         * In the other directions, the box size is the actuall
         * overlapping area.
         */
struct _Merge_Box {	
	int        num;      /* number of boxes */
        int        *nbids;   /* The adj patches to a specific one */
        int        *dirs;    /* Adj. on dir. */
        int        *sides;   /* Adj. on side (viewing from this specific one) */
        Rect_ComponentGrid  *boxes;  /* The boxes are made according to the above info. */
};
typedef struct _Merge_Box Merge_Box; 

LOCAL  void      clip_patch_fronts_to_interior3d(Front**,int,COMPONENT*);
LOCAL  void      delete_interior_surfs(INTERFACE*);  
LOCAL  int       merge_fronts3d(Front*,Front**,int);
LOCAL  bool      stitch_adj_intfc(INTERFACE*,INTERFACE*,RECT_GRID*,
                    RECT_GRID*,Merge_Box,Front**,int);
LOCAL  int       neighbor_to_merged_patch_in_dirs(CompositeGrid*,int*,
                     int,RECT_GRID*,int,int,int*,int*,int*,int*,Merge_Box*);
LOCAL  bool      append_patch_surface(SURFACE*,SURFACE*,RECT_GRID*,
                     RECT_GRID*,Merge_Box,P_LINK*,int);

/* This implementation is for grid-based case only */
LIB_LOCAL  int  assembly_distribute_patch_fronts3d(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        all_level)
{
        int        source, numnodes, myid;
        int        dist;
        int        patch_id, i;
        Front      **tmpfrs, *front;
        Front      *glue_fr;
        int        total_patch, levels;
        bool       sav_copy, status = FUNCTION_SUCCEEDED;
        INTERFACE  *current_intfc;
        COMPONENT  dummy = -1,  i_comp, *i_comps;
        float      coords[MAXD];
        RECT_GRID  *cgr, *tgr;
        RECT_GRID  *rgr;

        DEBUG_ENTER(assembly_distribute_patch_fronts3d)

        numnodes = pp_numnodes();
        myid = pp_mynode();
        sav_copy = copy_intfc_states();

        vector(&i_comps, num_patches, sizeof(COMPONENT));
        clip_patch_fronts_to_interior3d(frs, num_patches, i_comps);

        total_patch = set_copy_proc_frs_ver2(&tmpfrs,frs,num_patches,
                         redistr_table,max_n_patch);
        levels = tmpfrs[0]->NumberOfLevels;
        set_current_interface(frs[0]->interf);

        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                patch_id = -100;
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;

                if(myid == redistr_table[source][i].pc_id)
                {
                    patch_id = redistr_table[source][i].wv_id;
                    front = redistr_table[source][i].front;
                    if(exists_interface(front->interf) != YES)
                        printf("WARNING: Invalid Interface send in "
                          " assembly_distribute_patch_fronts, "
                              "Probably already deleted\n");
                    pp_send(0,(POINTER)(&patch_id),sizeof(int),source);
                    send_front_misc(front,&dummy,source);
                }
                if(source == myid)
                {
                    pp_recv(0,redistr_table[source][i].pc_id,
                            (POINTER)(&patch_id),sizeof(int));
                    recv_front_misc(tmpfrs[patch_id],
                        &dummy, redistr_table[source][i].pc_id);
                    front = tmpfrs[patch_id];
                    front->interf->modified = YES;
                    rgr = front->rect_grid;
                    cgr = computational_grid(front->interf);
                    copy_rect_grid(cgr,rgr);
                    tgr = &topological_grid(front->interf);
                    tgr->Remap.remap = rgr->Remap.remap;
                    set_patch_topo_grid(rgr,tgr);
                }
                pp_gsync();
            }
        }
        glue_fr = copy_front(tmpfrs[0]);

        status = assembly_fine_patch_fronts_to_one(tmpfrs, glue_fr);
        
        DEBUG_LEAVE(assembly_distribute_patch_fronts3d)
        return status;  
}

LOCAL void clip_patch_fronts_to_interior3d(
        Front      **frs,
        int        num_patches,
        COMPONENT  *i_comp)    /* number of patches computed in the proc */
{
        INTERFACE  *sav_intfc;
        bool       sav_copy, save_intrp;
        int        *lbuf,*ubuf;
        int        i, j, dim;
        float      coords[MAXD];
        RECT_GRID  *gr;

        sav_intfc = current_interface();
        for(i = 0; i < num_patches; i++)
        {
            if(frs[i]->patch_level == frs[i]->NumberOfLevels-1)
            {
                lbuf = frs[i]->rect_grid->lbuf;
                ubuf = frs[i]->rect_grid->ubuf;
                set_current_interface(frs[i]->interf);
                save_intrp = interpolate_intfc_states(frs[i]->interf);
                interpolate_intfc_states(frs[i]->interf) = NO;
                for(j = 0; j < frs[i]->rect_grid->dim; j++)
                    coords[j] = grid_center_coord(j,frs[i]->rect_grid);
                i_comp[i] = (frs[i]->interf->modified) ?
                    long_component(coords,frs[i]->interf) :
                    component(coords,frs[i]->interf);
                clip_intfc_at_grid_bdry2(frs[i]->interf); 
                interpolate_intfc_states(frs[i]->interf) = save_intrp;
            }
        }
        set_current_interface(sav_intfc);
}

EXPORT int assembly_fine_patch_fronts_to_one3d(
        Front    **oldfrs,
        Front    *newfr)
{
        int          status = FUNCTION_SUCCEEDED;
        int          i, j, levels, num_fine_frs;
        int          total_patch;
        int          dim, dir, side;
        INTERFACE    *sav_intfc, *bdy_intfc;
        bool         sav_copy;
        Front        **frs, *front = NULL;
        Front        *basefront = NULL;

        DEBUG_ENTER(assembly_fine_patch_fronts_to_one3d)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        levels = oldfrs[0]->NumberOfLevels;
        total_patch = oldfrs[0]->totalNumberOfPatches;
        num_fine_frs = 0; j = 0;
        for (i = 0; i < total_patch; i++)
        {
            if( oldfrs[i]->patch_level == levels-1)
                num_fine_frs++;
            if( oldfrs[i]->patch_level == 0 &&
                oldfrs[i]->patch_number == 0)
                basefront = oldfrs[i];
        }

        set_current_interface(basefront->interf);
        set_size_of_intfc_state(size_of_state(oldfrs[0]->interf));
        set_copy_intfc_states(YES);
        if (( newfr->interf = copy_interface(basefront->interf)) == NULL)
        {
            screen("ERROR assembly_fine_patch_fronts_to_one3d()",
                 "copy_interface() failed\n");
            clean_up(ERROR);
        }
        if(0 == num_fine_frs)
        {
            set_current_interface(newfr->interf);
            set_copy_intfc_states(sav_copy);
            DEBUG_LEAVE(assembly_fine_patch_fronts_to_one2d_ver2)
            return FUNCTION_SUCCEEDED;
        }

        vector(&frs,num_fine_frs,sizeof(Front*));

        for (i = 0; i < total_patch; i++)
        {
            if (oldfrs[i]->patch_level == levels-1)
            {
                /* 
                printf("Copied fine grids [%d]\n", oldfrs[i]->patch_number);
                print_rectangular_grid(oldfrs[i]->rect_grid);
                */
                frs[j] = copy_front(oldfrs[i]);
                set_size_of_intfc_state(size_of_state(oldfrs[i]->interf));
                set_copy_intfc_states(YES);
                if (( frs[j]->interf = copy_interface(oldfrs[i]->interf)) == NULL)
                {
                    screen("ERROR assembly_fine_patch_fronts_to_one2d_ver2()",
                         "copy_interface() failed\n");
                    clean_up(ERROR);
                }
                j++;
            }
        }
        if(j!=num_fine_frs)
        {
            printf("Error: assembly_fine_patch_fronts_to_one2d_ver2()\n");
            printf("did not find finest level front, exit j = %d\n", j);
            clean_up(ERROR);
        }
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        status = merge_fronts3d(newfr,frs,num_fine_frs);
        set_current_interface(newfr->interf);

        for(i = 0 ; i < num_fine_frs; i++)
            free_front(frs[i]);
        free_these(1,frs);

        if(debugging("assembly_fine_patch_fronts_to_one"))
        {
            printf("In assembly_patch_fronts_to_one3d()\n");
            printf("Print the assembled interface with states\n");
            print_interface(newfr->interf);
            show_intfc_states(newfr->interf);
        }

        DEBUG_LEAVE(assembly_fine_patch_fronts_to_one3d)
        return status; 
}

LOCAL  int merge_fronts3d(
        Front *newfront,
        Front **frs,         /* Fronts all come from finest level. */
        int num_frs)
{
        int             i, j, k, dir, dim = frs[0]->rect_grid->dim;
        int             *lbuf,*ubuf;
        int             patch_num;
        INTERFACE       *sav_intfc;
        RECT_GRID       *gr = computational_grid(newfront->interf);
        int             *merged, mnum = 0;
        int             nbid, side; 
        int             *nbids, *sides, *dirs, nbnum; 
        bool            sav_copy;
        bool            save_intrp;
        bool            status;
        CompositeGrid   *cg = (CompositeGrid*)newfront->cg_over; 
        RECT_GRID       dual_gr;  
        int             refine = 1; 
        Merge_Box       mergebox;  

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(newfront->interf);

        /* Adjust base grid resolution to fine level's */
        for(i = 1; i < newfront->NumberOfLevels; i++)
            refine *= 2;  /* should be adjustable */
        for(i = 0; i < gr->dim; i++)
        {
            gr->gmax[i] = refine*gr->gmax[i];
            gr->lbuf[i] = refine*gr->lbuf[i];
            gr->ubuf[i] = refine*gr->ubuf[i];
        }
        set_rect_grid(gr->L, gr->U, gr->GL, gr->GU,
            gr->lbuf, gr->ubuf, gr->gmax, 
            gr->dim, &gr->Remap, gr);
        set_dual_grid(&dual_gr,gr);
        printf("\nrefined base patch is\n");
        print_rectangular_grid(gr); 
         
        /* remove subdomain bdry curves and physical curves */
        delete_interior_surfs(newfront->interf);

        copy_interface_into(frs[0]->interf,newfront->interf);
        vector(&merged, newfront->totalNumberOfPatches, sizeof(int)); 
        vector(&nbids, 6, sizeof(int)); 
        vector(&sides, 6, sizeof(int)); 
        vector(&dirs, 6, sizeof(int)); 
        merged[0] = frs[0]->patch_number;
        mnum = 1; 

        printf("\nDUAL patch is\n"); 
        print_rectangular_grid(&dual_gr);  

loop_again: 
        for (j = 1; j < num_frs; j++)
        {
            printf("THe unmerged neighbor patch[%d] is :\n", 
                 frs[j]->patch_number);
            print_rectangular_grid(frs[j]->rect_grid);
            printf("Neighboring to::::::::::::\n");  

            if(YES == neighbor_to_merged_patch_in_dirs(cg,
                 merged, mnum, frs[j]->rect_grid, frs[j]->patch_number, 
                 frs[j]->patch_level, &nbnum, nbids, dirs, sides, &mergebox))
            {
                for(i = 0; i < nbnum; i++)
                {
                    printf("Merged patch[%d] in dir[%d] on  side[%d]\n", 
                        nbids[i], dirs[i], sides[i]);
                    for(k = 0; k < num_frs; k++)
                        if(nbids[i] == frs[k]->patch_number)
                            print_rectangular_grid(frs[k]->rect_grid);
                    printf("Merged area is:\n"); 
                    print_rect_componentgrid(mergebox.boxes[i]); 
                } 
                merged[mnum] = frs[j]->patch_number;
                mnum++; 
        
                if(!stitch_adj_intfc(newfront->interf,
                       frs[j]->interf, gr, &dual_gr, 
                       mergebox, frs, num_frs))
                {
                    status = FUNCTION_FAILED;
                    (void) printf("WARNING: in merge_fronts3d(), "
                           "stitch_adj_intfc() failed\n");
                }
            } 
            if((j == num_frs - 1) && mnum < num_frs)
            {
                j = 1;
                goto loop_again; 
            }
        }

        free(merged); free(nbids); 
        free(sides); free(dirs); 

        return status; 
}

LOCAL bool stitch_adj_intfc(
        INTERFACE       *intfc,         // my interface                
        INTERFACE       *adj_intfc,     // received interface          
        RECT_GRID       *grid,          // Rectangular grid for region 
        RECT_GRID       *dual_gr,       // Dual grid for region        
        Merge_Box       mbox, 
        Front           **frs,         // Fronts all come from finest level. 
        int             num_frs)
{
        INTERFACE *cur_intfc;
        P_LINK    *p_table;     /* Table of matching points on intfc
                                         * and adj_intfc*/
        SURFACE   **s, **as;
        int       p_size;               /*Size of space allocated for p_table*/
        bool      status;
        bool      corr_surf_found;

        DEBUG_ENTER(stitch_adj_intfc)

        cur_intfc = current_interface();
        set_current_interface(intfc);

        p_size = 4*(adj_intfc->num_points) + 1;
        vector(&p_table,p_size,sizeof(P_LINK));
        reset_hash_table(p_table,p_size);

        /* Begin patching adj_intfc to current interface */
        status = YES;
        for (as = adj_intfc->surfaces; as && *as; ++as)
        {
            corr_surf_found = NO;
            for (s = intfc->surfaces; s && *s; ++s)
            {
                /*
                *  COMMENT -
                *  The Hyper_surf_index() function is not
                *  fully supported.  This will fail in the
                *  presence of interface changes in topology
                *  TODO: FULLY SUPPORT THIS OBJECT
                */
                if (Hyper_surf_index(*s) == Hyper_surf_index(*as))
                {
                    corr_surf_found = YES;
                    if (!append_patch_surface(*s,*as,grid,dual_gr,mbox,
                                                p_table,p_size))
                    {
                        (void) printf("WARNING in "
                                      "stitch_adj_intfc(), "
                                      "append surface failed\n");
                        status = NO;
                    }
                }
            }
            if (!corr_surf_found)
            {
                SURFACE *surf;
                surf = copy_buffer_surface(*as,p_table,p_size);
                Hyper_surf_index(surf) = Hyper_surf_index((*as));
            }
        }
        free(p_table);
        set_current_interface(cur_intfc);


        DEBUG_LEAVE(stitch_adj_intfc)
        return status; 
}

LOCAL bool append_patch_surface(
        SURFACE   *surf,
        SURFACE   *adj_surf,
        RECT_GRID *grid,
        RECT_GRID *dual_gr,
        Merge_Box mbox, 
        P_LINK    *p_table,
        int       p_size)
{
        TRI     **tris_s,**tris_a;
        TRI     *tri;
        int     i, ns, na, gmax[MAXD];
        float   crx_l, crx_u;
        int     idir1, idir2;
        int     i1, i2;
        int     **nbt_s, **nbt_a;
        int     *lbuf = dual_gr->lbuf;
        int     *ubuf = dual_gr->ubuf;
        TRI     **tri_store_s, **tri_store_a;
        TRI     ****blk_tri_s, ****blk_tri_a;

        return YES;  
}

/* patch "me" on neighbor patches (*nbids) in direction dir */
/* Neighbor patches on me side (*sides)*/
/* The neighbor patches (*nbids) are already merged */

LOCAL int neighbor_to_merged_patch_in_dirs(
        CompositeGrid   *cg, 
        int        *mergedid,
        int        mnum, 
        RECT_GRID  *mygr, 
	int        me,
        int        mylevel, 
        int        *nbnum,
        int        *nbids,
        int        *dirs,
        int        *sides,
        Merge_Box  *mbox)
{
        int      grid, i, j;
        int      dim = 3; 
        int      dir, dir2, dir3; 
        int      nb_n = 0; 
        int      meimin[MAXD], meimax[MAXD];
        int      imin[MAXD], imax[MAXD];
        Index    I1,I2,I3;
        int      L[MAXD], U[MAXD];

        getIndex((*cg)[me].indexRange(),I1,I2,I3);
        meimin[0] = I1.getBase();
        meimin[1] = I2.getBase();
        meimin[2] = I3.getBase();
        meimax[0] = I1.getBound();
        meimax[1] = I2.getBound();
        meimax[2] = I3.getBound();

        if(axis1 != 0)
        {
            printf("ERROR: in neighbor_to_merged_patch_in_dirs\n");
            printf("Overture changed axis1 value to %d\n", axis1);
            clean_up(ERROR);  
        }

        /*
        printf("ME icords L[%d, %d, %d] U[%d, %d, %d]\n",
          meimin[0], meimin[1], meimin[2], meimax[0], meimax[1], meimax[2]);
        */
        for(grid = 0; grid < mnum; grid++)
        {
            if(mergedid[grid] == me) 
            {
                printf("ERROR in neighbor_to_merged_patch_in_dirs\n");
                printf("Patch[%d] is already in the merged array\n", me); 
                clean_up(ERROR);  
            }
            if((cg)->refinementLevelNumber(mergedid[grid]) != mylevel) continue;
            getIndex((*cg)[mergedid[grid]].indexRange(),I1,I2,I3);
            imin[0] = I1.getBase();
            imin[1] = I2.getBase();
            imin[2] = I3.getBase();
            imax[0] = I1.getBound();
            imax[1] = I2.getBound();
            imax[2] = I3.getBound();
       
            /*
            printf("Test icords L[%d, %d, %d] U[%d, %d, %d]\n",
                 imin[0], imin[1], imin[2], imax[0], imax[1], imax[2]);
            */
            for(i = 0; i < dim; i++)
            {
                dir2 = (i+1)%3;
                dir3 = (i+2)%3;
                L[i] = max(imin[i], meimin[i]);
                U[i] = min(imax[i], meimax[i]);
                /*
                printf("dir0[%d], LU[%d, %d]\n", i, L[i], U[i]);
                */
                if(L[i] == U[i])
                {
                    /* Test if the boxes overlap in the i's projection plane */
                    L[dir2] = max(imin[dir2], meimin[dir2]);
                    U[dir2] = min(imax[dir2], meimax[dir2]);
                    /*
                    printf("dir2[%d], LU[%d, %d]\n", dir2, L[dir2], U[dir2]);
                    */
                    if(L[dir2] >= U[dir2])
                        L[dir2] = U[dir2] = 0;
                    L[dir3] = max(imin[dir3], meimin[dir3]);
                    U[dir3] = min(imax[dir3], meimax[dir3]);
                    /*
                    printf("dir3[%d], LU[%d, %d]\n", dir3, L[dir3], U[dir3]);
                    */
                    if(L[dir3] >= U[dir3])
                        L[dir3] = U[dir3] = 0;
                    if((U[dir2]-L[dir2])*(U[dir3]-L[dir3]) != 0)
                    {
                        if(imin[i] == meimax[i])
                        {
                            nbids[nb_n] = mergedid[grid];
                            dirs[nb_n] = i;
                            sides[nb_n] = 1;
                            nb_n++;
                            break; 
                        }
                        if(imax[i] == meimin[i])
                        {
                            nbids[nb_n] = mergedid[grid];
                            dirs[nb_n] = i;
                            sides[nb_n] = 0;
                            nb_n++;
                            break; 
                        }
                    }
                }
            }
        }
        *nbnum = nb_n;

        /* set bounding boxes for the merging.
         * See the notes at the top of the file.
         */
        if(nb_n != 0)
        {
            mbox->num = nb_n;  
            mbox->nbids = nbids;
            mbox->dirs = dirs;
            mbox->sides = sides; 
            vector(&(mbox->boxes), nb_n, sizeof(Rect_ComponentGrid));
            for(j = 0; j < nb_n; j++)
            {
                mbox->boxes[j].dim = mygr->dim;
                mbox->boxes[j].grid_id = j; 
                mbox->boxes[j].grid_level = mylevel; 
                getIndex((*cg)[nbids[j]].indexRange(),I1,I2,I3);
                imin[0] = I1.getBase();
                imin[1] = I2.getBase();
                imin[2] = I3.getBase();
                imax[0] = I1.getBound();
                imax[1] = I2.getBound();
                imax[2] = I3.getBound();
                dir = dirs[j];
                dir2 = (dir+1)%3;
                dir3 = (dir+2)%3;
                L[dir] = max(imin[dir], meimin[dir]);
                U[dir] = min(imax[dir], meimax[dir]);
                L[dir2] = max(imin[dir2], meimin[dir2]);
                U[dir2] = min(imax[dir2], meimax[dir2]);
                L[dir3] = max(imin[dir3], meimin[dir3]);
                U[dir3] = min(imax[dir3], meimax[dir3]);
                for(i = 0; i < dim; i++)
                {
                    mbox->boxes[j].base[i] = L[i];
                    mbox->boxes[j].bound[i] = U[i];
                    if(i == dir)
                    {
                        mbox->boxes[j].base[i] -= 1;
                        mbox->boxes[j].bound[i] += 1;
                    }
                }
                for(i = 0; i < dim; i++)
                {
                    mbox->boxes[j].h[i] = mygr->h[i]; 
                    if(i == dir)
                    {
                        if(sides[j] == 0)
                        {
                            mbox->boxes[j].L[i] = mygr->L[i] - mygr->h[i]; 
                            mbox->boxes[j].U[i] = mygr->L[i] + mygr->h[i]; 
                        }
                        else
                        {
                            mbox->boxes[j].L[i] = mygr->U[i] - mygr->h[i]; 
                            mbox->boxes[j].U[i] = mygr->U[i] + mygr->h[i]; 
                        }
                    }
                    else
                    {
                        mbox->boxes[j].L[i] = (*cg)[me].vertex()(mbox->boxes[j].base[0], 
                               mbox->boxes[j].base[1], mbox->boxes[j].base[2], i);
                        mbox->boxes[j].U[i] = (*cg)[me].vertex()(mbox->boxes[j].bound[0], 
                               mbox->boxes[j].bound[1], mbox->boxes[j].bound[2], i);
                    }
                }
            }
            return YES;
        }
        mbox->num = 0;  
        return NO; 
}


LOCAL void delete_interior_surfs(
        INTERFACE       *intfc)
{
        INTERFACE       *sav_intfc;
        NODE            **n;
        SURFACE         **s, **delete_surfs = NULL;
        CURVE           **c;
        CURVE           **delete_curves = NULL;
        NODE            **delete_nodes = NULL;
        bool            sav_copy;

        DEBUG_ENTER(delete_interior_surfs)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_current_interface(intfc);

        for(s = intfc->surfaces; s && *s; s++)
        {
            if(wave_type(*s) >= FIRST_PHYSICS_WAVE_TYPE)
            {
                if (not add_to_pointers((POINTER)*s,(POINTER**)&delete_surfs))
                {
                    screen("ERROR in delete_interior_curves(), "
                        "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (s = delete_surfs; s and *s; s++)
            (void) delete_surface(*s);

        /* 
        delete_nodes = NULL;
        for (n = intfc->nodes; n and *n; n++)
        {
            if (((*n)->in_curves == NULL) && ((*n)->out_curves == NULL))
            {
                if (not add_to_pointers((POINTER)*n,(POINTER**)&delete_nodes))
                {
                    screen("ERROR in delete_interior_curves(), "
                              "add_to_pointers() failed\n");
                    clean_up(ERROR);
                }
            }
        }
        for (n = delete_nodes; n && *n; n++)
            (void) delete_node(*n);
        */

        if(debugging("delete_interior_surfs"))
        {
            printf("In delete_interior_surfs\n");
            printf("Print left tmp interface:::\n");
            print_interface(intfc);
            show_intfc_states(intfc);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(delete_interior_surfs)
}

#endif /* if defined(USE_OVERTURE) */
#endif /* if defined(THREED) */

