/*
*                               fpatch.c:
*
*       Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*
*       this file includes functions set_patch_front, set_patch_topo_grid,
*                       and set_patch_comput_grid
*/

#define DEBUG_STRING    "fpatch"

#include <front/flocaldecs.h>

#if defined(USE_OVERTURE)

LOCAL   int     front_in_array(Front*,Front**,int);
LOCAL   byte    *bundle_send_front_misc(Front*,COMPONENT*,int*);


/* front_on_redistr_table()
 * find out the front hooked on the 
 * redistribute_table item specified by
 * node and patch_number.
 * in_node is used to speed up the searching.
 * if in_node is set, only this node is searched.
 */
EXPORT Front *front_on_redistr_table(
	Wv_on_pc        **redistr_table,
        int             max_n_patch,
        int             node,
        int             patch_number,
        int             in_node)
{
        int             numnodes, source;
        int             i;

        numnodes = pp_numnodes();

        if(in_node != -1)
            source = in_node; 
        else
            source = 0; 
        for(; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(redistr_table[source][i].pc_id == node &&
                   redistr_table[source][i].wv_id == patch_number)
                {
                    if(redistr_table[source][i].front == NULL)
                    {
                        printf("ERROR in front_on_redistr_table()\n");
                        printf("node[%d] patch_num[%d] does not"
                          " have front on redistr_table[%d][%d]\n",
                           node,patch_number,source,i); 
                        printf("_____________________________\n");
                        printf("The patch is scattered as:\n");
                        for(source = 0; source < numnodes; source++)
                        {
                            printf("Proc[%d]: ", source);
                            for(i = 0; i < max_n_patch; i++)
                            {
                                if(-1 != redistr_table[source][i].wv_id)
                                    printf(" %2d %2d, ", redistr_table[source][i].pc_id,
                                                 redistr_table[source][i].wv_id);
                                else
                                   printf("        ");
                            }
                            printf("\n");
                        }
                        printf("_____________________________\n");
                        clean_up(ERROR);
                    }
                    return redistr_table[source][i].front;
                }
            }

            if(in_node != -1) break; 
        } 

        printf("ERROR in_node[%d] in front_on_redistr_table()\n", in_node);
        printf("node[%d] patch_num[%d] does not"
               " have front on redistr_table\n",
                node,patch_number); 
        printf("_____________________________\n");
        printf("The patch is scattered as:\n");
        for(source = 0; source < numnodes; source++)
        {
            printf("Proc[%d]: ", source);
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 != redistr_table[source][i].wv_id)
                    printf(" %2d %2d, ", redistr_table[source][i].pc_id,
                                 redistr_table[source][i].wv_id);
                else
                   printf("        ");
            }
            printf("\n");
        }
        printf("_____________________________\n");
        clean_up(ERROR); 
        return NULL; 
}

EXPORT void newfront_to_distri_table(
        Front           *fr,
        Front           *nfr,
        int             num_patches,
        Wv_on_pc        **redistr_table,
        int             max_n_patch)
{
        int             numnodes, source;
        int             i;
        int             myid;
        int             found = NO;

        numnodes = pp_numnodes();
        myid = pp_mynode();
        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(redistr_table[source][i].front == fr)
                {
                    redistr_table[source][i].front = nfr;
                    found = YES;
                    break;
                }
            }
            if(found)
                break;
        }

        if(NO == found)
        {
            for(source = 0; source < numnodes; source++)
            {
                printf("Proc[%d]: ", source);
                for(i = 0; i < max_n_patch; i++)
                {
                    printf(" %p ", redistr_table[source][i].front);
                }
                printf("\n");
            }
            printf("ERROR: newfront_to_distri_table()\n");
            printf("newfr is not instored into redistr_table\n");
            printf("fr = %p, nfr = %p\n", fr, nfr); 
            clean_up(ERROR);
        }
}

EXPORT void set_patch_front(
        Front           *mother_front,
        Front           *patch_front,
        RECT_GRID       *newrgr,
        int             patch_num)
{
        Front                   *front;
        INTERFACE               *sav_intfc;
        int                     i, dir, dim, *lbuf, *ubuf;
        bool                    sav_copy; 
        RECT_GRID               *cgr;
        RECT_GRID               *tgr;
        Patch_bdry_flag         *pd_flag = patch_front->pd_flag;

        DEBUG_ENTER(set_patch_front)
        front = mother_front;

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_size_of_intfc_state(size_of_state(front->interf));
        set_copy_intfc_states(YES);

        if(NULL == (patch_front->interf = copy_interface(front->interf)))
        {
            printf("ERROR set_patch_front\n");
            printf("copy interface failed\n");
            clean_up(ERROR);
        }

        set_current_interface(patch_front->interf);
        patch_front->interf->modified = YES;
        patch_front->rect_grid = newrgr;

        cgr = computational_grid(patch_front->interf);
        copy_rect_grid(cgr,newrgr);  

        dim = cgr->dim;
        lbuf = cgr->lbuf;
        ubuf = cgr->ubuf;

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        tgr = &topological_grid(patch_front->interf);
        tgr->Remap.remap = newrgr->Remap.remap;
        set_patch_topo_grid(newrgr,tgr);
        
        if ( patch_num != 0 )
        {
            for ( dir = 0; dir < dim; dir++)
            {
                for ( i = 0; i < 2; i++)
                {
                    if (pd_flag->bdry_side[dir][i] == 0)
                        rect_boundary_type(patch_front->interf,dir,i)
                        = AMR_SUBDOMAIN_BOUNDARY;
                }
            }
        }

        DEBUG_LEAVE(set_patch_front)
}

EXPORT void set_patch_topo_grid(
        RECT_GRID       *pgr,
        RECT_GRID       *tgr)
{
        float           *L = pgr->L;
        float           *U = pgr->U;
        float           *GL = pgr->GL;
        float           *GU = pgr->GU;
        float           *h = pgr->h;
        int             *lbuf = pgr->lbuf;
        int             *ubuf = pgr->ubuf;
        int             gmax[MAXD];
        int             dim = pgr->dim;
        int             i, dir;
        RECT_GRID       dual_grid;
        RECT_GRID       expanded_dual_grid;

        DEBUG_ENTER(set_patch_topo_grid)

        set_dual_grid(&dual_grid, pgr);
        for (i = 0; i < dim; i++)
            gmax[i] = pgr->gmax[i]+pgr->lbuf[i]+pgr->ubuf[i];
        for (i = 0; i < dim; i++)
            gmax[i] = gmax[i]/2;

        expanded_dual_grid.Remap = pgr->Remap;  
        set_rect_grid(pgr->VL,pgr->VU,pgr->GL,pgr->GU,
                      NOBUF,NOBUF,gmax,dim,&expanded_dual_grid.Remap,
                      &expanded_dual_grid);
        for (i = 0; i < dim; i++)
        {
            tgr->gmax[i] = expanded_dual_grid.gmax[i];
            tgr->L[i] = expanded_dual_grid.L[i];
            tgr->U[i] = expanded_dual_grid.U[i];
        }
        set_rect_grid(tgr->L,tgr->U, expanded_dual_grid.GL,expanded_dual_grid.GU,
            NOBUF,NOBUF,tgr->gmax,dim,&tgr->Remap,tgr);

        DEBUG_LEAVE(set_patch_topo_grid)
}


/* this function should be called after the assembling
 * process, it cuts a piece from global interface.
 * the input front contains a global interface.
 */

EXPORT bool clip_patch_front(
        Front           *front,
        int             zoom_rect_grids)
{
        COMPONENT       max_comp;
        INTERFACE       *intfc = front->interf;
        RECT_GRID       *gr = front->rect_grid;
        bool            status;
        bool            sav_copy = copy_intfc_states();
        bool            sav_intrp = interpolate_intfc_states(intfc);
        int             i, dim = gr->dim;

        DEBUG_ENTER(clip_patch_front)

        for (i = 0; i < dim; i++)
            if ((gr->lbuf[i] > 0) || (gr->ubuf[i] > 0)) break;
        if (i == dim)
        {
            status = FUNCTION_SUCCEEDED; /* No subdomains to process */
        }
        else
        {
            set_copy_intfc_states(YES);
            interpolate_intfc_states(intfc) = NO;
            status = form_patch_subintfc_via_cut(front);
            /*  
            max_comp = max_component(intfc);
            pp_global_imax(&max_comp,1L);
            max_component(intfc) = max_comp;
            */
            interpolate_intfc_states(intfc) = sav_intrp;
            set_copy_intfc_states(sav_copy);
        }
#if defined(TWOD)
        if(debugging("clip_patch_front"))
        {
            CURVE **c;
            BOND  *b;
            if(front->patch_number != 4)
                return status;
            for(c = front->interf->curves; c and *c;  c++)
            {
                if(is_bdry(*c)) continue;

                b = (*c)->first;
                while( b != NULL)
                {
                    printf("point <%g, %g> state\n",
                            Coords(b->start)[0], Coords(b->start)[1]);
                    (*front->print_state)(left_state(b->start));
                    if(b == (*c)->last)
                    {
                        printf("point <%g, %g> state\n",
                          Coords(b->end)[0], Coords(b->end)[1]);
                        (*front->print_state)(left_state(b->end));
                    }
                    b = b->next;
                }
            }
        }
#endif  /* defined(TWOD) */
/*
#if defined(THREED)
        if (debugging("consistency"))
        {
            (void) printf("Check consistency of interface ");
            (void) printf("after scatter_front()\n");
            check_consistency_of_tris_on_intfc(intfc);
        }
#endif */  /* defined(THREED) */

        DEBUG_LEAVE(clip_patch_front)
        return status;
}

/* NOTE: assembly_fine_patch_fronts_to_one()
 * only assembly the interfaces. The
 * boundary is not considered in this function.
 * To complete the boundary, 
 * set_amr_subdomain_boundary() or
 * set_subdomain_boundary() or should be called. 
 */
EXPORT int assembly_fine_patch_fronts_to_one(
        Front   **frs,
        Front   *front)
{
        if (front == NULL or front->interf == NULL)
        {
            printf("ERROR: assembly_fine_patch_fronts_to_one\n");
            printf("function failed because of front = %p\n", front);
            clean_up(ERROR);
        }
        return (*f_user_interface(front->interf)._assembly_fine_patch_fronts_to_one)(frs,front);
        /* 
         *  Hooked to g_assembly_fine_patch_fronts_to_one().
         *  assembly_fine_patch_fronts_to_one2d_ver2().
         *  assembly_fine_patch_fronts_to_one3d().
         */ 
}

LIB_LOCAL  bool form_patch_subintfc_via_cut(
        Front   *front)
{
        if (front == NULL or front->interf == NULL)
            return FUNCTION_FAILED;
        /* hooked with g_form_patch_subintfc_via_cut2d */
        /* hooked with g_form_patch_subintfc_via_cut3d */
        return 
          (*f_user_interface(front->interf)._form_patch_subintfc_via_cut)(front);
        /* 
        return f_form_patch_subintfc_via_cut2d(front);  		
        */ 
}               /*end form_patch_subintfc_via_communication*/

/* This function is built based on the assumption that
 * the base patch interface is assembled from the fine
 * patch interface. AND the base interface performs the
 * real interface buffer communication. Then the
 * patch interface first copies the rebuilt base interface,
 * and the patch interface only needs to do the cut to
 * fit the interface to the domain.
 */

/* nn is the number of patches which are not transfered from other procs */ 
/* Collect fronts which are originally on this proc and make             */
/* storages for the fronts which are transfer back to this proc          */  
/* The difference between set_copy_proc_frs() and set_copy_proc_frs_ver2() */
/* is that set_copy_proc_frs() DOES NOT make front copy and                */
/* set_copy_proc_frs_ver2() DOES make front copy                           */
EXPORT  int     set_copy_proc_frs(
	Front      ***tmpfrs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        *nn)
{
        int        source, numnodes, myid;
        int        patch_id, i;
        Front      *basefront = NULL;   
        Front      *front; 
        int        total_patch; 
        COMPONENT  dummy = -1; 
        RECT_GRID  *gr;  
        INTERFACE  *sav_intfc;
        bool       sav_copy;    

        DEBUG_ENTER(set_copy_proc_frs)

        numnodes = pp_numnodes();
        myid = pp_mynode(); 
  
        for(i = 0; i < max_n_patch; i++)
        {
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid == redistr_table[myid][i].pc_id &&
               redistr_table[myid][i].wv_id == 0) 
            {
                basefront = redistr_table[myid][i].front;  
                total_patch = basefront->totalNumberOfPatches;
                break;  
            } 
        } 
        if((basefront == NULL) ||
           (basefront->patch_number != 0 && basefront->patch_level != 0))
        {
            printf("set_copy_proc_frs() "); 
            printf("did not find out the basefront\n");
            clean_up(ERROR);  
        }  

        vector(tmpfrs,total_patch,sizeof(Front*));
        for(i = 0; i < total_patch; i++) (*tmpfrs)[i] = NULL;

        /* Save local patches which are not transfered to the other procs */ 
        *nn = 0; 
        for(i = 0; i < max_n_patch; i++)
        {
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid == redistr_table[myid][i].pc_id)
            {
                patch_id = redistr_table[myid][i].wv_id;
                (*tmpfrs)[patch_id] = redistr_table[myid][i].front;
                if(patch_id != redistr_table[myid][i].front->patch_number)
                {
                    printf("ERROR: set_copy_proc_frs()\n");
                    printf("resid Front[%d] NOT same in redistr_table and frs\n",i);
                    clean_up(ERROR);
                }
                (*nn)++;
            }
        }

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        set_size_of_intfc_state(size_of_state(basefront->interf));
        set_copy_intfc_states(YES);

        /* make storage for recving patches from other procs */  
        for(i = 0; i < total_patch; i++)
        {
            if((*tmpfrs)[i] != NULL)
                continue;
            (*tmpfrs)[i] = deep_copy_front(basefront);
            set_size_of_intfc_state(size_of_state(basefront->interf));
            (*tmpfrs)[i]->interf = copy_interface(basefront->interf);
            delete_patch_all_curves((*tmpfrs)[i]);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(set_copy_proc_frs)
        return  total_patch;  
}

/*  set_copy_proc_frs_ver2().
 * nn is the number of patches which are not transfered 
 * from other procs. Collect fronts (by making a copy) 
 * which are originally on this proc and make storages 
 * for the fronts which are transfer back to this proc.
 */  
LIB_LOCAL int set_copy_proc_frs_ver2(
	Front      ***tmpfrs,
        Front      **frs,
        int        num_patches,
        Wv_on_pc   **redistr_table,
        int        max_n_patch)
{
        int        source, numnodes, myid;
        int        patch_id, i;
        Front      *basefront = NULL;   
        Front      *front; 
        int        total_patch; 
        INTERFACE  *sav_intfc;
        bool       sav_copy;    

        DEBUG_ENTER(set_copy_proc_frs_ver2)

        numnodes = pp_numnodes();
        myid = pp_mynode(); 
  
        for(i = 0; i < max_n_patch; i++)
        {
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid == redistr_table[myid][i].pc_id &&
               redistr_table[myid][i].wv_id == 0) 
            {
                basefront = redistr_table[myid][i].front;  
                total_patch = basefront->totalNumberOfPatches;
                break;  
            } 
        } 
        if((basefront == NULL) || 
           (basefront->patch_number != 0 && basefront->patch_level != 0))
        {
            printf("ERROR in assembly_distribute_patch_fronts()\n");  
            printf("set_copy_proc_frs_ver2() "); 
            printf("did not find out the basefront\n");
            clean_up(ERROR);  
        }  

        vector(tmpfrs,total_patch,sizeof(Front*));
        for(i = 0; i < total_patch; i++) (*tmpfrs)[i] = NULL; 

        /* Save local grids which are not transfered to the other procs */ 
        /* redistr_table[myid][i].front = frs[i], they are set to be equal */
        for(i = 0; i < max_n_patch; i++)
        {
            int found_same; 
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(myid == redistr_table[myid][i].pc_id)
            {
                patch_id = redistr_table[myid][i].wv_id; 
                (*tmpfrs)[patch_id] = copy_front(redistr_table[myid][i].front);
                if( patch_id != redistr_table[myid][i].front->patch_number)
                {
                    printf("ERROR: set_copy_proc_frs_ver2()\n");
                    printf("resid Front[%d] NOT same in redistr_table[%d][%d] and frs\n",
                          i, myid, i);
                    printf("patch_id [%d] from redistr_table\n", patch_id);
                    clean_up(ERROR);
                }
                found_same = NO; 
                for(int tmpjj = 0; tmpjj < num_patches; tmpjj++)
                {
                    if(frs[tmpjj] == redistr_table[myid][i].front)
                    {
                        found_same = YES;
                        break; 
                    }
                }
                if(found_same == NO)
                {
                    printf("ERROR: set_copy_proc_frs_ver2()\n");
                    printf("Fronts NOT in redistr_table[%d][%d] and frs\n",
                          myid, i);
                    printf("patch_id [%d] from redistr_table\n", patch_id);
                    clean_up(ERROR);
                }
            }
        }

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        set_size_of_intfc_state(size_of_state(basefront->interf));
        set_copy_intfc_states(YES);

        /* make storage for recving patches from other procs */  
        for(i = 0; i < total_patch; i++)
        {
            if((*tmpfrs)[i] != NULL) 
                continue;  
            (*tmpfrs)[i] = deep_copy_front(basefront);
            set_size_of_intfc_state(size_of_state(basefront->interf));
            (*tmpfrs)[i]->interf = copy_interface(basefront->interf);
            delete_patch_all_curves((*tmpfrs)[i]);
        }

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(set_copy_proc_frs_ver2)
        return  total_patch;  
}

LIB_LOCAL void clear_copy_frs(
	Front      **cpyfrs,
        Wv_on_pc   **redistr_table,
        int        max_n_patch)
{
        int        myid;
        int        i, j;
        int        total_patch; 
        int        is_local; 

        myid = pp_mynode(); 
        total_patch = cpyfrs[0]->totalNumberOfPatches;
        for(i = 0; i < total_patch; i++)
        {
            is_local = NO;  
            for(j = 0; j < max_n_patch; j++)
            {
                if(-1 == redistr_table[myid][j].wv_id) 
                    continue;
                if(myid == redistr_table[myid][j].pc_id)
                {
                    if(redistr_table[myid][j].wv_id == i)
                        is_local = YES; 
                }
            }
            if(YES == is_local)
               free_front(cpyfrs[i]); 
            else
                deep_free_front(cpyfrs[i]); 
        }
}


/* assembly the coarse level patches from fine  only*/
EXPORT  int    assembly_coarse_patch_fronts(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */ 
        Wv_on_pc   **redistr_table,
        int        max_n_patch)
{
        int        dim, status; 
        INTERFACE  *current_intfc; 

        DEBUG_ENTER(assembly_coarse_patch_fronts)  

        if(NULL == frs[0]->interf) 
        {
            printf("ERROR: assembly_coarse_patch_fronts()\n");  
            printf("frs[0] interface is NULL\n");  
            clean_up(ERROR);  
        } 

        if(frs[0]->NumberOfLevels == 1 && max_n_patch == 1)
        {
            current_intfc = current_interface();  
            set_current_interface(frs[0]->interf); 
            if(! (status = scatter_front(frs[0])))
            {
                printf("WARNING in assembly_coarse_patch_fronts(), "
                  "scatter_front() failed for the base front in 1 level case\n");
                set_current_interface(current_intfc); 
                return status; 
            }    
            set_current_interface(current_intfc); 
            DEBUG_LEAVE(assembly_coarse_patch_fronts) 
            return FUNCTION_SUCCEEDED;
        }

        dim = frs[0]->rect_grid->dim;
        switch(dim)
        {
        case 2:
#if defined(TWOD) 
            status = assembly_coarse_patch_fronts2d(frs, num_patches, 
                 redistr_table, max_n_patch);
#endif /* if defined(TWOD) */
        break;  
        case 3:
#if defined(THREED)
#endif /* if defined(THREED) */
        default:
            printf("ERROR: %D case not implemented\n", dim);
            printf("for assembly_coarse_patch_fronts\n");
            clean_up(ERROR);  
        } 
        DEBUG_LEAVE(assembly_coarse_patch_fronts)  
        return status;
}

/*   assembly_distribute_patch_fronts()
 * 1. assembly the coarse level patches from fine. 
 * 2. The fine patch buffer zone interfaces are re-assemblied.
 */
EXPORT  int    assembly_distribute_patch_fronts(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */ 
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        all_level)
{
        int        dim, status; 
        INTERFACE  *current_intfc; 

        DEBUG_ENTER(assembly_distribute_patch_fronts)  

        if(NULL == frs[0]->interf) 
        {
            printf("ERROR: assembly_distribute_patch_fronts()\n");  
            printf("frs[0] interface is NULL\n");  
            clean_up(ERROR);  
        } 

        if(frs[0]->NumberOfLevels == 1 && max_n_patch == 1)
        {
            current_intfc = current_interface();  
            set_current_interface(frs[0]->interf); 
            if(! (status = scatter_front(frs[0])))
            {
                printf("WARNING in assembly_distribute_patch_fronts(), "
                     "scatter_front() failed for the base front in 1 level case\n");
                set_current_interface(current_intfc); 
                return status; 
            }    
            set_current_interface(current_intfc); 
            DEBUG_LEAVE(assembly_distribute_patch_fronts) 
            return FUNCTION_SUCCEEDED;
        }

        if(DEBUG)
        {
            printf("assembly_distribute_patch_fronts() Entering : ");
            printf("STORAGE: %-d \n", get_vmalloc_storage_use());
        }

        dim = frs[0]->rect_grid->dim;
        switch(dim)
        {
        case 2:
#if defined(TWOD) 
            status = assembly_distribute_patch_fronts2d(frs, num_patches, 
                 redistr_table, max_n_patch, all_level);
#endif /* if defined(TWOD) */
        break;  
        case 3:
#if defined(THREED)
            status = assembly_distribute_patch_fronts3d(frs, num_patches, 
                 redistr_table, max_n_patch, all_level);
#endif /* if defined(THREED) */
        break;  
        default:
            printf("ERROR: %D case not implemented\n", dim);
            printf("for assembly_distribute_patch_fronts\n");
            clean_up(ERROR);  
        } 

        DEBUG_LEAVE(assembly_distribute_patch_fronts)  
        return status; 
}

EXPORT void send_front_misc(
        Front      *fr,
        COMPONENT  *comp,  
        int        dist)
{
        int        i, j;
        int        len; 
        byte       *storage = NULL;

        DEBUG_ENTER(send_front_misc)
#if defined(__MPI__)

        send_interface(fr->interf, dist);
        storage = bundle_send_front_misc(fr, comp, &len);
        pp_send(0, (POINTER)storage, len, dist);
        free(storage);

#endif /* defined(__MPI__) */
        DEBUG_LEAVE(send_front_misc)
}

EXPORT void recv_front_misc(
        Front      *fr,
        COMPONENT  *comp,  
        int        source)
{
        INTERFACE  *recv_intfc;
        INTERFACE  *sav_intfc;
        bool       sav_copy;
        bool       save_intrp;

        int        i, j;
        int        len;
        byte       *storage = NULL;

        DEBUG_ENTER(recv_front_misc)
#if defined(__MPI__)

        recv_intfc = receive_interface(source);

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        save_intrp = interpolate_intfc_states(fr->interf);
        set_current_interface(fr->interf);
        interpolate_intfc_states(fr->interf) = NO;

        copy_interface_into(recv_intfc,fr->interf);
        (void) delete_interface(recv_intfc);

        interpolate_intfc_states(fr->interf) = save_intrp;
        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        /* fr->rect_grid */
        /* wave->center_comp */
        /* patch_number */
        /* patch_level  */
        /* rect_boundary_type[MAXD][2] */
        /* Patch_bdry_flag             */ 

        len = sizeof(RECT_GRID) + sizeof(COMPONENT) + sizeof(int) + sizeof(int)
              + sizeof(int)*(fr->interf->dim)*2 + sizeof(Patch_bdry_flag);
        scalar(&storage, len);

        pp_recv(0, source, (POINTER)(storage), len);

        unbundle_recv_front_misc(fr,comp,storage); 

#endif /* defined(__MPI__) */
        DEBUG_LEAVE(recv_front_misc)
}


EXPORT void send_mini_front_misc(
        Front      *fr,
        COMPONENT  *comp,  
        int        dist)
{
        int        i, j;
        int        len; 
        byte       *storage = NULL;

        DEBUG_ENTER(send_front_misc)
#if defined(__MPI__)

        storage = bundle_send_front_misc(fr, comp, &len); 
        pp_send(0, (POINTER)storage, len, dist);
        free(storage);

#endif /* defined(__MPI__) */
        DEBUG_LEAVE(send_front_misc)
}

EXPORT void recv_mini_front_misc(
        Front      *fr,
        COMPONENT  *comp,  
        int        source)
{
        bool       sav_copy;
        bool       save_intrp;

        int        i, j;
        int        len;
        byte       *storage = NULL;

        DEBUG_ENTER(recv_mini_front_misc)
#if defined(__MPI__)

        /* fr->rect_grid */
        /* wave->center_comp */
        /* patch_number */
        /* patch_level  */
        /* rect_boundary_type[MAXD][2] */
        /* Patch_bdry_flag             */ 

        len = sizeof(RECT_GRID) + sizeof(COMPONENT) + sizeof(int) + sizeof(int)
              + sizeof(int)*(fr->interf->dim)*2 + sizeof(Patch_bdry_flag);
        scalar(&storage, len);

        pp_recv(0, source, (POINTER)(storage), len);
        unbundle_recv_front_misc(fr,comp, storage); 
 
#endif /* defined(__MPI__) */
        DEBUG_LEAVE(recv_mini_front_misc)
}

LOCAL byte *bundle_send_front_misc(
        Front      *fr,
        COMPONENT  *comp,
        int        *len)
{
        int        i, j;
        byte       *storage = NULL;
        byte       *buf;
        POINTER    info;

        /* fr->rect_grid */
        /* wave->center_comp */
        /* patch_number */
        /* patch_level  */
        /* rect_bdry_type[MAXD][2] */
        /* Patch_bdry_flag         */  
        /* patch's pc_ic[MAXD]     */

        (*len) = sizeof(RECT_GRID) + sizeof(COMPONENT) + sizeof(int) +sizeof(int)
             + sizeof(int)*(fr->interf->dim)*2 + sizeof(Patch_bdry_flag)
             + sizeof(int)*MAXD;

        scalar(&storage, (*len));
        buf = storage;

        info = (POINTER) buf;
        assign(info, fr->rect_grid, sizeof(RECT_GRID));
        buf += sizeof(RECT_GRID);

        info = (POINTER) buf;
        assign(info, comp, sizeof(COMPONENT));
        buf += sizeof(COMPONENT);

        info = (POINTER) buf;
        assign(info, &(fr->patch_number), sizeof(int));
        buf += sizeof(int);

        info = (POINTER) buf;
        assign(info, &(fr->patch_level), sizeof(int));
        buf += sizeof(int);

        for(i = 0; i < fr->interf->dim; i++)
        {
            info = (POINTER) buf;
            assign(info, &(fr->interf->rect_bdry_type[i][0]), sizeof(int));
            buf += sizeof(int);
            info = (POINTER) buf;
            assign(info, &(fr->interf->rect_bdry_type[i][1]), sizeof(int));
            buf += sizeof(int);
        }

        info = (POINTER) buf;
        assign(info, fr->pd_flag, sizeof(Patch_bdry_flag));
        buf += sizeof(Patch_bdry_flag);

        //091404 added
        info = (POINTER) buf;
        assign(info, fr->pc_ic, sizeof(int)*MAXD);
        buf += sizeof(int)*MAXD;

        return storage; 
}

EXPORT void unbundle_recv_front_misc(
        Front      *fr,
        COMPONENT  *comp,  
        byte       *storage)
{
        int        i, j;
        int        len;
        byte       *buf;
        POINTER    info;

        DEBUG_ENTER(unbundle_recv_front_misc)

        /* fr->rect_grid */
        /* wave->center_comp */
        /* patch_number */
        /* patch_level  */
        /* rect_boundary_type[MAXD][2] */
        /* Patch_bdry_flag             */ 
        /* patch's pc_ic[MAXD]     */

        len = sizeof(RECT_GRID) + sizeof(COMPONENT) + sizeof(int) + sizeof(int)
              + sizeof(int)*(fr->interf->dim)*2 + sizeof(Patch_bdry_flag)
              + sizeof(int)*MAXD;

        buf = storage;
        info = (POINTER) buf;
        assign(fr->rect_grid, info, sizeof(RECT_GRID));
        buf += sizeof(RECT_GRID);
  
        info = (POINTER) buf;
        assign(comp,info,sizeof(COMPONENT));
        buf += sizeof(COMPONENT);
  
        info = (POINTER) buf;
        assign(&(fr->patch_number),info,sizeof(int));
        buf += sizeof(int);

        info = (POINTER) buf;
        assign(&(fr->patch_level),info,sizeof(int));
        buf += sizeof(int);

        for(i = 0; i < fr->rect_grid->dim; i++)
        {
            info = (POINTER) buf;
            assign(&(fr->interf->rect_bdry_type[i][0]), info, sizeof(int));
            buf += sizeof(int);
            info = (POINTER) buf;
            assign(&(fr->interf->rect_bdry_type[i][1]), info, sizeof(int));
            buf += sizeof(int);
        }

        info = (POINTER) buf;
        assign(fr->pd_flag, info, sizeof(Patch_bdry_flag));
        buf += sizeof(Patch_bdry_flag);

        //091404 added
        info = (POINTER) buf;
        assign(fr->pc_ic, info, sizeof(int)*MAXD);
        buf += sizeof(int)*MAXD;
 
        free(storage);
        DEBUG_LEAVE(unbundle_recv_front_misc)
}


/* which_level = 2, finest level only */
/* which_level = 1, coarse levels (do not include level 0) */
/* which_level = 0, all the levels (do not include level 0) */
EXPORT void fast_ireinstall_undistribute_fronts(
        Front      **newfrs,
        Front      **frs,
        Wv_on_pc   **redistr_table,
        int        num_patches,    /* number of patches compute in this proc */
        int        max_n_patch,
        int        which_level)
{
        int        levels; 
        int        source, numnodes, myid;
        int        max_n_chunks, patch_id;
        Front      *front;
        int         nsend, nrecv;
        int         isend_tags, irecv_tags;
        int         i, ii, jj;
        int         *isend_chunks;
#if defined(__MPI__)
        MPI_Request **isend_reqs, **irecv_reqs;
        MPI_Status  **isend_stat, **irecv_stat;
#endif /* if defined(__MPI__) */
        POINTER     **out_ncaddr, **out_ocaddr, **top_addr;
        struct Table **o_tbl;
        INTERFACE   **recv_intfc, *intfc;
        Front       **recv_fr;

        INTERFACE  *sav_intfc;
        bool       sav_copy;
        bool       save_intrp;

        DEBUG_ENTER(fast_ireinstall_undistribute_fronts)

        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_copy_intfc_states(YES);
        numnodes = pp_numnodes();
        myid = pp_mynode();
        levels = frs[0]->NumberOfLevels;

        max_n_chunks = 0;
        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;
                if(which_level == 2)
                {
                    if(redistr_table[source][i].wv_level != (levels - 1))
                        continue;
                }
                if(myid == redistr_table[source][i].pc_id)
                {
                    patch_id = redistr_table[source][i].wv_id;
                    front = redistr_table[source][i].front;
                    if(! front_in_array(front, frs, num_patches))
                    {
                        printf("ERROR fast_ireinstall_undistribute_fronts\n");
                        printf("front from item[%d][%d] not in front array\n",
                                  source, i);
                        clean_up(ERROR);
                    }
                    /*
                    printf("Item[%d][%d] send chunk = %d\n",
                        source, i, front->interf->table->num_chunks);
                    */
                    if(front->interf->table->num_chunks > max_n_chunks)
                        max_n_chunks = front->interf->table->num_chunks;
                }
            }
        }

        pp_global_imax(&max_n_chunks, 1);
        max_n_chunks += 2;

        nsend = 0;
        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;
                if(which_level == 2)
                {
                    if(redistr_table[source][i].wv_level != (levels - 1))
                        continue;
                }
                if(myid == redistr_table[source][i].pc_id)
                    nsend++;
            }
        }

        nrecv = 0;
        for(i = 0; i < max_n_patch; i++)
        {
            if(-1 == redistr_table[myid][i].wv_id) continue;
            if(which_level == 2)
            {
                if(redistr_table[myid][i].wv_level != (levels - 1))
                    continue;
            }
            if(redistr_table[myid][i].pc_id != myid)
                nrecv++;
        }
#if !defined(__MPI__)
        if(numnodes != 1 || nrecv != 0 || nsend != 0)
        {
            printf("ERROR in fast_ireinstall_undistribute_fronts\n");
            printf("In serial mode, nrecv = %d, nsend = %d, not  zero\n",
                     nrecv, nsend);
            clean_up(ERROR);
        }
#endif /* if !defined(__MPI__) */


#if defined(__MPI__)
        if(nsend != 0)
        {
            vector(&isend_chunks, nsend, sizeof(int));
            vector(&isend_reqs, nsend, sizeof(MPI_Request*));
            vector(&isend_stat, nsend, sizeof(MPI_Status*));
            vector(&top_addr, nsend, sizeof(POINTER*));

            ii = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;
                    if(which_level == 2)
                    {
                        if(redistr_table[source][i].wv_level != (levels - 1))
                            continue;
                    }
                    if(myid == redistr_table[source][i].pc_id)
                    {
                        front = redistr_table[source][i].front;
                        isend_tags = trans_item_offset(redistr_table,source,i,max_n_patch)
                                           *max_n_chunks;
                        isend_interface(front->interf, source, isend_tags,
                                         &isend_reqs[ii], &top_addr[ii]);
                        isend_chunks[ii] = front->interf->table->num_chunks+2;
                        vector(&isend_stat[ii], isend_chunks[ii], sizeof(MPI_Status));
                        ii++;
                    }
                }
            }
        }

        if(nrecv != 0)
        {
            vector(&irecv_reqs, nrecv, sizeof(MPI_Request*));
            vector(&irecv_stat, nrecv, sizeof(MPI_Status*));
            vector(&out_ncaddr, nrecv, sizeof(POINTER*));
            vector(&out_ocaddr, nrecv, sizeof(POINTER*));
            vector(&o_tbl,nrecv, sizeof(struct Table*));
            vector(&recv_intfc,nrecv, sizeof(INTERFACE*));
            vector(&recv_fr,nrecv, sizeof(Front*));
            jj = 0;
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[myid][i].wv_id) continue;
                if(which_level == 2)
                {
                    if(redistr_table[myid][i].wv_level != (levels - 1))
                        continue;
                }
                if(redistr_table[myid][i].pc_id != myid)
                {
                    irecv_tags = trans_item_offset(redistr_table,myid,i,max_n_patch)
                                      *max_n_chunks;
                    recv_fr[jj] = newfrs[redistr_table[myid][i].wv_id];
                    recv_intfc[jj] = ireceive_interface(redistr_table[myid][i].pc_id,
                         irecv_tags,&irecv_reqs[jj],&out_ncaddr[jj],&out_ocaddr[jj],&o_tbl[jj]);
                    vector(&irecv_stat[jj], o_tbl[jj]->num_chunks+1, sizeof(MPI_Status));
                    jj++;
                }
            }
        }
        if(nrecv != 0)
        {
            for(i = 0; i < nrecv; i++)
            {
                MPI_Waitall(o_tbl[i]->num_chunks+1, irecv_reqs[i], irecv_stat[i]);
                ireconstruct_interface_pointers(recv_intfc[i],o_tbl[i],out_ocaddr[i],out_ncaddr[i]);
                free_these(2, irecv_reqs[i], irecv_stat[i]);

                save_intrp = interpolate_intfc_states(recv_fr[i]->interf);
                set_current_interface(recv_fr[i]->interf);
                interpolate_intfc_states(recv_fr[i]->interf) = NO;
                copy_interface_into(recv_intfc[i],recv_fr[i]->interf);
                (void) delete_interface(recv_intfc[i]);
                interpolate_intfc_states(recv_fr[i]->interf) = save_intrp;
            }
            free_these(5,irecv_reqs,irecv_stat,out_ncaddr,out_ocaddr,o_tbl);
            free_these(2, recv_intfc, recv_fr);
        }

        if(nsend != 0)
        {
            for(i = 0; i < nsend; i++)
            {
                MPI_Waitall(isend_chunks[i], isend_reqs[i], isend_stat[i]);
                free_these(3, isend_reqs[i], isend_stat[i], top_addr[i]);
            }
            free_these(4, isend_chunks, isend_reqs, isend_stat, top_addr);
        }
#endif /* if defined(__MPI__) */

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(fast_ireinstall_undistribute_fronts)
}


EXPORT int trans_item_offset(
        Wv_on_pc  **redistr_table,
        int       pc,
        int       item,
        int       max_n_patch)
{
        int       numnodes, src, i;
        int       offset = 0;

        numnodes = pp_numnodes();
        for(src = 0; src < numnodes; src++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[src][i].wv_id) continue;
                // if(src == redistr_table[src][i].pc_id) continue;

                if(src == pc && i == item)
                    return offset;
                offset++;
            }
        }
        printf("ERROR in trans_item_offset\n");
        printf("item[%d][%d] not fund\n", pc, item);
        clean_up(ERROR); 
}

#if defined(__MPI__)
EXPORT void irecv_front_misc(
        Front      *fr,
        int        source,
        byte       **out_storage,  
        int        tag,
        MPI_Request *req)
{
        int        len;
        byte       *storage = NULL;

        /* fr->rect_grid */
        /* wave->center_comp = comp */
        /* patch_number */
        /* patch_level  */
        /* rect_boundary_type[MAXD][2] */
        /* Patch_bdry_flag             */ 
        /* patch's pc_ic[MAXD]         */

        len = sizeof(RECT_GRID) + sizeof(COMPONENT) + sizeof(int) + sizeof(int)
              + sizeof(int)*(fr->interf->dim)*2 + sizeof(Patch_bdry_flag)
              + sizeof(int)*MAXD;
        scalar(&storage, len);

        pp_irecv(tag, source, (POINTER)(storage), len, req);
        *out_storage = storage;
}

EXPORT void isend_front_misc(
        Front      *fr,
        COMPONENT  *comp,
        int        dist,
        byte       **out_storage,
        int        tag,
        MPI_Request *req)
{
        int        i, j;
        int        len; 
        byte       *storage = NULL;

        storage = bundle_send_front_misc(fr, comp, &len);
        pp_isend(tag, (POINTER)storage, len, dist, req);

        *out_storage = storage;
}
#endif /* defined(__MPI__) */

LIB_LOCAL void reinstall_undistribute_fronts_and_misc(
        Front      **newfrs,
        Front      **frs,
        Wv_on_pc   **redistr_table,
        int        total_patch,    /* number of patches originally genereated */
        int        num_patches,    /* number of patches compute in this proc */
        int        max_n_patch)
{
        int        source, numnodes, myid;
        int        dist;
        int        patch_id, i, j;
        COMPONENT  dummy = -1;
        RECT_GRID  *cgr, *tgr;
        RECT_GRID  *rgr;
        int         nsend, nrecv;
        bool        not_all_recved;
        bool        *irecv_flags;
        int         *irecv_indices;
        int         isend_tags, irecv_tags;
#if defined(__MPI__)
        MPI_Request *isend_reqs, *irecv_reqs;
        MPI_Status  *isend_stat, *irecv_stat;
#endif /* if defined(__MPI__) */
        int         ii, jj;
        int         *fly_patch_id;
        byte        **irecv_stor, **isend_stor;
        Front       **irecv_fr, *front;

        numnodes = pp_numnodes();
        myid = pp_mynode();

        vector(&fly_patch_id, total_patch, sizeof(int));
        for(i = 0; i < total_patch; i++)
            fly_patch_id[i] = -1;

        nsend = 0;
        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;
                if(myid == redistr_table[source][i].pc_id)
                    nsend++;
            }
        }
        nrecv = 0;
        for(i = 0; i < max_n_patch; i++)
        {
            if(redistr_table[myid][i].pc_id != myid &&
               redistr_table[myid][i].wv_id != -1)
                nrecv++;
        }

        fast_ireinstall_undistribute_fronts(newfrs, frs,
                 redistr_table, num_patches, max_n_patch, 2);

#if defined(__MPI__)
        if(nrecv != 0)
        {
            vector(&irecv_flags, nrecv, sizeof(bool));
            vector(&irecv_indices, nrecv, sizeof(int));
            vector(&irecv_reqs, nrecv, sizeof(MPI_Request));
            vector(&irecv_stat, nrecv, sizeof(MPI_Status));
            vector(&irecv_stor, nrecv, sizeof(byte*));
            vector(&irecv_fr, nrecv, sizeof(Front*));
            ii = 0;
            for(i = 0; i < max_n_patch; i++)
            {
                if(redistr_table[myid][i].pc_id != myid &&
                   redistr_table[myid][i].wv_id != -1)
                {
                    irecv_tags = STATE_ID + redistr_table[myid][i].wv_id + 1;
                    irecv_fr[ii] = front = newfrs[redistr_table[myid][i].wv_id];
                    fly_patch_id[redistr_table[myid][i].wv_id] = redistr_table[myid][i].wv_id;

                    irecv_front_misc(front, redistr_table[myid][i].pc_id,
                          &irecv_stor[ii], irecv_tags, &irecv_reqs[ii]);
                    irecv_flags[ii] = NO;
                    ii++;
                }
            }
        }
        if(nsend != 0)
        {
            vector(&isend_reqs, nsend, sizeof(MPI_Request));
            vector(&isend_stat, nsend, sizeof(MPI_Status));
            vector(&isend_stor, nsend, sizeof(byte*));
            ii = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;
                    if(myid == redistr_table[source][i].pc_id)
                    {
                        front = redistr_table[source][i].front;
                        dummy = front->center_comp;
                        isend_tags = STATE_ID + redistr_table[source][i].wv_id + 1;
                        isend_front_misc(front, &dummy, source, &isend_stor[ii],
                                isend_tags, &isend_reqs[ii]);
                        ii++; 
                    }
                }
            }
        }
        if(nrecv != 0)
        {
            int ndone;
            not_all_recved = YES;
            while(not_all_recved == YES)
            {
                MPI_Waitsome(nrecv, irecv_reqs, &ndone, irecv_indices, irecv_stat);
                for(i = 0; i < ndone; i++)
                {
                    ii = irecv_indices[i];
                    if(irecv_flags[ii] != YES)
                    {
                        /*
                        unbundle_recv_front_misc(irecv_fr[ii],
                         &dummy,irecv_stor[ii]);
                        */
                        unbundle_recv_front_misc(irecv_fr[ii],
                         &irecv_fr[ii]->center_comp,irecv_stor[ii]);
                        irecv_flags[ii] = YES;
                    }
                }
                not_all_recved = NO;
                for(j = 0; j < nrecv; j++)
                {
                    if(irecv_flags[j] == NO)
                    {
                        not_all_recved = YES;
                        break;
                    }
                }
            }
            free_these(6, irecv_stor, irecv_fr, irecv_flags, irecv_indices,
                   irecv_reqs, irecv_stat);
        }
        if(nsend != 0)
        {
            MPI_Waitall(nsend, isend_reqs, isend_stat);
            for(j = 0; j < nsend; j++)
                free(isend_stor[j]);
            free_these(3, isend_stor, isend_reqs, isend_stat);
        }
        /* Set recving patches */
        for(i = 0; i < total_patch; i++)
        {
            if(fly_patch_id[i] == -1) continue;

            newfrs[i]->interf->modified = YES;
            rgr = newfrs[i]->rect_grid;
            cgr = computational_grid(newfrs[i]->interf);
            copy_rect_grid(cgr,rgr);
            tgr = &topological_grid(newfrs[i]->interf);
            tgr->Remap.remap = rgr->Remap.remap;
            set_patch_topo_grid(rgr,tgr);
        }
        pp_gsync();
#endif /* if defined(__MPI__) */

        free(fly_patch_id); 
}

LOCAL int front_in_array(
       Front   *front,
       Front   **frs,
       int     num_patches)
{
       int     i;
       for(i = 0; i < num_patches; i++)
       {
           if(front == frs[i])
               return YES; 
       } 
       return NO; 
} 

/* which_level = 2, finest level only */
/* which_level = 1, coarse levels (do not include level 0) */
/* which_level = 0, all the levels (do not include level 0) */
/* THIS IS A EXTENSION of fast_patch_front_trans(). 
 * consider to replace fast_patch_front_trans() with 
 * patch_front_distribute_to_nodes()
 */
EXPORT void patch_front_distribute_to_nodes(
        Front      **tmpfrs,
        Front      **frs,
        Wv_on_pc   **redistr_table,
        int        num_patches,    /* number of patches compute in this proc */
        int        max_n_patch,
        int        which_level)
{
        int        levels;
        int        source, numnodes, myid;
        int        max_n_chunks, patch_id;
        Front      *front;
        int         nsend, nrecv;
        int         isend_tags, irecv_tags;
        int         i, ii, jj;
        int         *isend_chunks;
#if defined(__MPI__)
        MPI_Request **isend_reqs, **irecv_reqs;
        MPI_Status  **isend_stat, **irecv_stat;
#endif /* if defined(__MPI__) */
        POINTER     **out_ncaddr, **out_ocaddr, **top_addr;
        struct Table **o_tbl;
        INTERFACE   **recv_intfc, *intfc;
        Front       **recv_frs;

        INTERFACE  *sav_intfc;
        bool       sav_copy, save_intrp;

        DEBUG_ENTER(patch_front_distribute_to_nodes)

        numnodes = pp_numnodes();
        myid = pp_mynode();
        levels = tmpfrs[0]->NumberOfLevels;
        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();
        set_copy_intfc_states(YES);

        max_n_chunks = nsend = nrecv = 0;
        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;

                if(which_level == 2)
                {
                    if(redistr_table[source][i].wv_level != (levels - 1))
                        continue;
                }
                else if(which_level == 1)
                {
                    if(redistr_table[source][i].wv_level == (levels - 1))
                        continue;
                }
                if(myid == source)
                {
                    patch_id = redistr_table[source][i].wv_id;
                    front = tmpfrs[patch_id];
                    if(front->interf->table->num_chunks > max_n_chunks)
                        max_n_chunks = front->interf->table->num_chunks;
                    nsend++;
                }
                if(myid == redistr_table[source][i].pc_id)
                    nrecv++;
            }
        }

        pp_global_imax(&max_n_chunks, 1);
        max_n_chunks += 2;

#if defined(__MPI__)
        if(nsend != 0)
        {
            vector(&isend_chunks, nsend, sizeof(int));
            vector(&isend_reqs, nsend, sizeof(MPI_Request*));
            vector(&isend_stat, nsend, sizeof(MPI_Status*));
            vector(&top_addr, nsend, sizeof(POINTER*));
            ii = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;
                    if(which_level == 2)
                    {
                        if(redistr_table[source][i].wv_level != (levels - 1))
                            continue;
                    }
                    else if(which_level == 1)
                    {
                        if(redistr_table[source][i].wv_level == (levels - 1))
                            continue;
                    }
                    if(myid == source)
                    {
                        patch_id = redistr_table[source][i].wv_id;
                        front = tmpfrs[patch_id];
                        isend_tags = trans_item_offset(redistr_table,source,i,max_n_patch)
                                           *max_n_chunks;
                        isend_interface(front->interf, redistr_table[source][i].pc_id,
                                        isend_tags, &isend_reqs[ii], &top_addr[ii]);
                        isend_chunks[ii] = front->interf->table->num_chunks+2;
                        vector(&isend_stat[ii], isend_chunks[ii], sizeof(MPI_Status));
                        /*
                        printf("redist[%d][%d] should be sent to : ",source,i);
                        printf("tag_offset[%d]: Proc [%d] Send to proc[%d] chunks %d\n",
                                isend_tags[ii], myid, redistr_table[source][i].pc_id,
                                front->interf->table->num_chunks);
                        */
                        ii++;
                    }
                }
            }
        }
        if(nrecv != 0)
        {
            vector(&irecv_reqs, nrecv, sizeof(MPI_Request*));
            vector(&irecv_stat, nrecv, sizeof(MPI_Status*));
            vector(&out_ncaddr,nrecv, sizeof(POINTER*));
            vector(&out_ocaddr,nrecv, sizeof(POINTER*));
            vector(&o_tbl,nrecv, sizeof(struct Table*));
            vector(&recv_intfc,nrecv, sizeof(INTERFACE*));
            vector(&recv_frs,nrecv, sizeof(Front*));
            jj = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;

                    if(which_level == 2)
                    {
                        if(redistr_table[source][i].wv_level != (levels - 1))
                            continue;
                    }
                    else if(which_level == 1)
                    {
                        if(redistr_table[source][i].wv_level == (levels - 1))
                            continue;
                    }

                    if(myid == redistr_table[source][i].pc_id)
                    {
                        patch_id = redistr_table[source][i].wv_id;

                        recv_frs[jj] = front = front_on_redistr_table(redistr_table,
                              max_n_patch, myid, patch_id, source);
                        irecv_tags = trans_item_offset(redistr_table,source,i,max_n_patch)
                                      *max_n_chunks;
                        recv_intfc[jj] = ireceive_interface(source, irecv_tags,
                             &irecv_reqs[jj],&out_ncaddr[jj],&out_ocaddr[jj],&o_tbl[jj]);
                        vector(&irecv_stat[jj], o_tbl[jj]->num_chunks+1, sizeof(MPI_Status));

                        /*
                        printf("redist[%d][%d] should get : ",source,i);
                        printf("Proc[%d] received patch[%d] from Proc[%d]\n",
                            myid, patch_id, source);
                        */
                        jj++;
                    }
                }
            }
        }

        if(nrecv != 0)
        {
            for(i = 0; i < nrecv; i++)
            {
                MPI_Waitall(o_tbl[i]->num_chunks+1, irecv_reqs[i], irecv_stat[i]);
                ireconstruct_interface_pointers(recv_intfc[i],o_tbl[i],out_ocaddr[i],out_ncaddr[i]);
                free_these(2, irecv_reqs[i], irecv_stat[i]);

                save_intrp = interpolate_intfc_states(recv_frs[i]->interf);
                set_current_interface(recv_frs[i]->interf);
                interpolate_intfc_states(recv_frs[i]->interf) = NO;
                copy_interface_into(recv_intfc[i],recv_frs[i]->interf);
                (void) delete_interface(recv_intfc[i]);
                interpolate_intfc_states(recv_frs[i]->interf) = save_intrp;
            }
            free_these(5,irecv_reqs,irecv_stat,out_ncaddr,out_ocaddr,o_tbl);
            free_these(2, recv_intfc, recv_frs);
        }

        if(nsend != 0)
        {
            for(i = 0; i < nsend; i++)
            {
                MPI_Waitall(isend_chunks[i], isend_reqs[i], isend_stat[i]);
                free_these(3, isend_reqs[i], isend_stat[i], top_addr[i]);
            }
            free_these(4, isend_chunks, isend_reqs, isend_stat, top_addr);
        }

#endif /* if defined(__MPI__) */

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(patch_front_distribute_to_nodes)
}

EXPORT void fast_patch_front_trans(
        Front      **tmpfrs,
        Front      **frs,
        Wv_on_pc   **redistr_table,
        int        num_patches,    /* number of patches compute in this proc */
        int        max_n_patch,
        int        all_level)
{
        int        levels; 
        int        source, numnodes, myid;
        int        max_n_chunks, patch_id;
        Front      *front;
        int         nsend, nrecv;
        int         isend_tags, irecv_tags;
        int         i, ii, jj;
        int         *isend_chunks;
#if defined(__MPI__)
        MPI_Request **isend_reqs, **irecv_reqs;
        MPI_Status  **isend_stat, **irecv_stat;
#endif /* if defined(__MPI__) */
        POINTER     **out_ncaddr, **out_ocaddr, **top_addr;
        struct Table **o_tbl;
        INTERFACE   **recv_intfc, *intfc;
        Front       **recv_frs; 

        INTERFACE  *sav_intfc;
        bool       sav_copy;
        bool       save_intrp;

        DEBUG_ENTER(fast_patch_front_trans)

        numnodes = pp_numnodes();
        myid = pp_mynode();
        levels = tmpfrs[0]->NumberOfLevels; 
        sav_intfc = current_interface();
        sav_copy = copy_intfc_states();

        max_n_chunks = nsend = nrecv = 0; 
        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;
                if(all_level != YES)
                {
                    if(redistr_table[source][i].wv_level != (levels - 1))
                        continue;
                }
                if(myid == source)
                {
                    patch_id = redistr_table[source][i].wv_id;
                    front = tmpfrs[patch_id]; 
                    if(front->interf->table->num_chunks > max_n_chunks)
                        max_n_chunks = front->interf->table->num_chunks;
                    nsend++; 
                }
                if(myid == redistr_table[source][i].pc_id)
                    nrecv++; 
            }
        }

        pp_global_imax(&max_n_chunks, 1);
        max_n_chunks += 2;

#if defined(__MPI__)
        if(nsend != 0)
        {
            vector(&isend_chunks, nsend, sizeof(int));
            vector(&isend_reqs, nsend, sizeof(MPI_Request*));
            vector(&isend_stat, nsend, sizeof(MPI_Status*));
            vector(&top_addr, nsend, sizeof(POINTER*));
            ii = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;
                    if(all_level != YES)
                    {
                        if(redistr_table[source][i].wv_level != (levels - 1))
                            continue;
                    }
                    if(myid == source)
                    {
                        patch_id = redistr_table[source][i].wv_id;
                        front = tmpfrs[patch_id]; 
                        isend_tags = trans_item_offset(redistr_table,source,i,max_n_patch)
                                           *max_n_chunks;
                        isend_interface(front->interf, redistr_table[source][i].pc_id, 
                                        isend_tags, &isend_reqs[ii], &top_addr[ii]);
                        isend_chunks[ii] = front->interf->table->num_chunks+2;
                        vector(&isend_stat[ii], isend_chunks[ii], sizeof(MPI_Status));
                        /*
                        printf("redist[%d][%d] should be sent to : ",source,i);
                        printf("tag_offset[%d]: Proc [%d] Send to proc[%d] chunks %d\n",
                                isend_tags[ii], myid, redistr_table[source][i].pc_id, 
                                front->interf->table->num_chunks);
                        */
                        // delete this interface later 
                        ii++;
                    }
                }
            } 
        } 
        if(nrecv != 0)
        {
            vector(&irecv_reqs, nrecv, sizeof(MPI_Request*));
            vector(&irecv_stat, nrecv, sizeof(MPI_Status*));
            vector(&out_ncaddr,nrecv, sizeof(POINTER*));
            vector(&out_ocaddr,nrecv, sizeof(POINTER*));
            vector(&o_tbl,nrecv, sizeof(struct Table*));
            vector(&recv_intfc,nrecv, sizeof(INTERFACE*));
            vector(&recv_frs,nrecv, sizeof(Front*));
            jj = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;
                    if(all_level != YES)
                    {
                        if(redistr_table[source][i].wv_level != (levels - 1))
                            continue;
                    }
                    if(myid == redistr_table[source][i].pc_id)
                    {
                        patch_id = redistr_table[source][i].wv_id; 

                        recv_frs[jj] = front = front_on_redistr_table(redistr_table,
                              max_n_patch, myid, patch_id, source);
                        // front = front_on_redistr_table(redistr_table,
                        //       max_n_patch, myid, patch_id, source);

                        // done outside this function
                        // if(front->patch_level != (levels-1))
                        //     delete_patch_all_curves(front);

                        irecv_tags = trans_item_offset(redistr_table,source,i,max_n_patch)
                                      *max_n_chunks;
                        /*
                        intfc = ireceive_interface(source, irecv_tags,
                             NULL,&out_ncaddr[jj],&out_ocaddr[jj],&o_tbl[jj]);
                        ireconstruct_interface_pointers(intfc,o_tbl[jj],out_ocaddr[jj],out_ncaddr[jj]);

                        save_intrp = interpolate_intfc_states(front->interf);
                        interpolate_intfc_states(front->interf) = NO;
                        copy_interface_into(intfc,front->interf);
                        (void) delete_interface(intfc);
                        interpolate_intfc_states(front->interf) = save_intrp;
                        */

                        recv_intfc[jj] = ireceive_interface(source, irecv_tags,
                             &irecv_reqs[jj],&out_ncaddr[jj],&out_ocaddr[jj],&o_tbl[jj]);
                        vector(&irecv_stat[jj], o_tbl[jj]->num_chunks+1, sizeof(MPI_Status));

                        /*
                        printf("redist[%d][%d] should get : ",source,i);
                        printf("Proc[%d] received patch[%d] from Proc[%d]\n",
                            myid, patch_id, source);
                        */
                        jj++;
                    }
                }
            }
        }
        if(nrecv != 0)
        {
            for(i = 0; i < nrecv; i++)
            {
                MPI_Waitall(o_tbl[i]->num_chunks+1, irecv_reqs[i], irecv_stat[i]);
                ireconstruct_interface_pointers(recv_intfc[i],o_tbl[i],out_ocaddr[i],out_ncaddr[i]);
                free_these(2, irecv_reqs[i], irecv_stat[i]);

                save_intrp = interpolate_intfc_states(recv_frs[i]->interf);
                set_current_interface(recv_frs[i]->interf);
                interpolate_intfc_states(recv_frs[i]->interf) = NO;
                copy_interface_into(recv_intfc[i],recv_frs[i]->interf);
                (void) delete_interface(recv_intfc[i]);
                interpolate_intfc_states(recv_frs[i]->interf) = save_intrp;
            }
            free_these(5,irecv_reqs,irecv_stat,out_ncaddr,out_ocaddr,o_tbl);
            free_these(2, recv_intfc, recv_frs);
            /*
            jj = 0;
            for(source = 0; source < numnodes; source++)
            {
                for(i = 0; i < max_n_patch; i++)
                {
                    if(-1 == redistr_table[source][i].wv_id) continue;
                    if(source == redistr_table[source][i].pc_id) continue;
                    if(all_level != YES)
                    {
                        if(redistr_table[source][i].wv_level != (levels - 1))
                            continue;
                    }
                    if(myid == redistr_table[source][i].pc_id)
                    {
                        front = recv_frs[jj];
                        save_intrp = interpolate_intfc_states(front->interf);
                        set_current_interface(front->interf);
                        interpolate_intfc_states(front->interf) = NO;
                        copy_interface_into(recv_intfc[jj],front->interf);
                        (void) delete_interface(recv_intfc[jj]);
                        interpolate_intfc_states(front->interf) = save_intrp;
                        // print_interface(front->interf);
                        jj++;
                    }
                }
            } 
            */
        }  
        if(nsend != 0)
        {
            for(i = 0; i < nsend; i++)
            {
                MPI_Waitall(isend_chunks[i], isend_reqs[i], isend_stat[i]);
                free_these(3, isend_reqs[i], isend_stat[i], top_addr[i]);
            }
            free_these(4, isend_chunks, isend_reqs, isend_stat, top_addr);
        }
        // pp_gsync(); 

#endif /* if defined(__MPI__) */

        set_current_interface(sav_intfc);
        set_copy_intfc_states(sav_copy);

        DEBUG_LEAVE(fast_patch_front_trans)
}

/*
LIB_LOCAL void delete_tmp_fronts_after_distr(
        Front      **tmpfrs,
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        int        all_level)
{
        int        source, numnodes, myid;
        int        patch_id, levels;
        int        i, j; 
        Front      *front;

        INTERFACE  *sav_intfc;
        numnodes = pp_numnodes();
        myid = pp_mynode();
        levels = tmpfrs[0]->NumberOfLevels;

        sav_intfc = current_interface();

        for(source = 0; source < numnodes; source++)
        {
            for(i = 0; i < max_n_patch; i++)
            {
                if(-1 == redistr_table[source][i].wv_id) continue;
                if(source == redistr_table[source][i].pc_id) continue;
                if(all_level != YES)
                {
                    if(redistr_table[source][i].wv_level != (levels - 1))
                        continue;
                }

                if(myid == source)
                {
                    patch_id = redistr_table[source][i].wv_id;
                    delete_interface(tmpfrs[patch_id]->interf);
                    tmpfrs[patch_id]->interf = NULL;
                }
            }
        }

        set_current_interface(sav_intfc);
}
*/


EXPORT  int    scatter_patch_fronts(
        Front      **frs,
        int        num_patches,    /* number of patches computed in the proc */
        Wv_on_pc   **redistr_table,
        int        max_n_patch,
        Overparam  *overparam, 
        int        all_level)
{
        int        dim, status;
        INTERFACE  *current_intfc;

        DEBUG_ENTER(scatter_patch_fronts)

        if(NULL == frs[0]->interf)
        {
            printf("ERROR: scatter_patch_fronts()\n");
            printf("frs[0] interface is NULL\n");
            clean_up(ERROR);
        }

        if(frs[0]->NumberOfLevels == 1 && max_n_patch == 1)
        {
            current_intfc = current_interface();
            set_current_interface(frs[0]->interf);
            if(! (status = scatter_front(frs[0])))
            {
                printf("WARNING in scatter_patch_fronts(), "
                     "scatter_front() failed for the base front in 1 level case\n");
                set_current_interface(current_intfc);
                return status;
            }
            set_current_interface(current_intfc);
            DEBUG_LEAVE(scatter_patch_fronts)
            return FUNCTION_SUCCEEDED;
        }
        if(DEBUG)
        {
            printf("scatter_patch_fronts() Entering : ");
            printf("STORAGE: %-d \n", get_vmalloc_storage_use());
        }

        dim = frs[0]->rect_grid->dim;
        switch(dim)
        {
        case 2:
#if defined(TWOD)
            status = scatter_patch_fronts2d(frs, num_patches,
                 redistr_table, max_n_patch, overparam, all_level);
#endif /* if defined(TWOD) */
        break;
        case 3:
#if defined(THREED)
#endif /* if defined(THREED) */
        default:
            printf("ERROR: %D case not implemented\n", dim);
            printf("for scatter_patch_fronts\n");
            clean_up(ERROR);
        }

        DEBUG_LEAVE(scatter_patch_fronts)
        return status;
}

#endif  /* if defined(USE_OVERTURE) */

