/*
*				finit.c:
*
*	Copyright 1999 by The University at Stony Brook, All rights reserved.
*
*	Contains the initialization routines
*
*		init_front()
*		init_front_states()
*/

#include <front/fdecs.h>		/* includes int.h, table.h */


	/* LOCAL Function Declarations */
LOCAL	bool	f_read_print_FlowSpecifiedRegion_data(INIT_DATA*,const IO_TYPE*,
						      FlowSpecifiedRegion**);
LOCAL	bool	read_print_ConstantFlowRegion(INIT_DATA*,const IO_TYPE*,
                                              INTERFACE*,
					      FlowSpecifiedRegion*,
					      FlowSpecifiedRegion**);
LOCAL	void	init_front_time_step_control(INIT_DATA*);
LOCAL	void	prompt_for_front_spacing(INIT_DATA*);
LOCAL	void	read_print_FlowSpecifiedRegion_list(INIT_DATA*,const IO_TYPE*);

#if defined(TWOD) || defined(THREED)
LOCAL	void	clear_curve_redistribution_info(CURVE_REDISTRIBUTE*);
LOCAL	void	init_redistribution_frequency(INIT_DATA*);
LOCAL	void	set_redistribution_frequency(INIT_DATA*,Front*);
#endif /* defined(TWOD) || defined(THREED) */

#if defined(ONED)
LOCAL void	init_1d_front_states(Front*,INIT_DATA*,
				     void(*)(POINT*,HYPER_SURF_ELEMENT*,
					     HYPER_SURF*,Locstate,Locstate,
					     INIT_DATA*));
#endif /* defined(ONED) */

#if defined(TWOD)
LOCAL	void	f_init_curve_redistribution_parameters(INIT_DATA*,Front*);
LOCAL	void	f_prompt_for_curve_redist_options(INIT_DATA*);
LOCAL void	init_2d_front_states(Front*,INIT_DATA*,
				     void (*)(POINT*,HYPER_SURF_ELEMENT*,
					      HYPER_SURF*,Locstate,Locstate,
					      INIT_DATA*));
#endif /* defined(TWOD) */

#if defined(THREED)
LOCAL	void	f_init_surface_redistribution_parameters(INIT_DATA*,Front*);
LOCAL	void    init_3d_front_states(Front*,INIT_DATA*,
				     void (*)(POINT*,HYPER_SURF_ELEMENT*,
					      HYPER_SURF*,Locstate,Locstate,
					      INIT_DATA*));
LOCAL	void	f_prompt_for_surface_redist_options(INIT_DATA*);
#endif /* defined(THREED) */

/*
*			prompt_for_front_options():
*
*	Prompts for user selected front options.
*
*/

EXPORT void f_prompt_for_front_options(
	INIT_DATA *init,
	Front     *front)
{
	char	s[Gets_BUF_SIZE];
	int	dim = Comp_grid(init).dim;

	debug_print("init","Entered f_prompt_for_front_options()\n");

	InitialMaxFrontSpeed(init) =
	    alloc_MaxFrontSpeed(InitialMaxFrontSpeed(init),
	                        i_intfc(init),StateSize(init));

	if (restart_io_type(init) != NULL)
	    read_print_front_options(init,front);

	set_redistribution_defaults(init);
	prompt_for_redistribute(init);

	    /* Choose Constant to Increase/Decrease Time-Steps */

	init_front_time_step_control(init);

	screen("\n\t\tflow specified state enforcement at fronts\n\n");

	enforce_flow_specified_states(init) = YES;
	screen("Enforce flow specified states at fronts (dflt=%s): ",
	       (enforce_flow_specified_states(init)==YES)?"yes":"no");
	(void) Gets(s);
	if (s[0] == 'y' || s[0] == 'Y')
	    enforce_flow_specified_states(init) = YES;
	else if (s[0] == 'n' || s[0] == 'N')
	    enforce_flow_specified_states(init) = NO;

	(void) printf("\n");

	movingframe(init) = NO;
	screen("Type yes to propagate front in moving frame (dflt = %s): ",
	       (movingframe(init)==YES)?"yes":"no");
	(void) Gets(s);
	if (s[0] == 'y' || s[0] == 'Y')
	    movingframe(init) = YES;
	else if (s[0] == 'n' || s[0] == 'N')
	    movingframe(init) = NO;

	(void) printf("\n");

	tangent_method(init) = LINEAR_SECANT;
	if (restart_io_type(init) != NULL)
	    tangent_method(init) = TANGENT_METHOD_FROM_RESTART;
#if defined(THREED)
	normal3d_method(init) = AREA_WEIGHTED_NORMAL;
	if (restart_io_type(init) != NULL)
	    normal3d_method(init) = NORMAL_METHOD_FROM_RESTART;
#endif /* defined(THREED) */

#if defined(TWOD)
	if (dim == 2) /*Only 2D currently supports multiple tangent algorithms*/
	{
	    static const char *dflt = ", default";
	    screen("Select tangent computation algorithm, choices are\n"
	           "\tLinear centered SECANT vectors (SECANT%s)\n"
	           "\tFourth order LANGRANGIAN interpolation (LANGRANGIAN%s)\n"
	           "\tCubic SPLINE fit (SPLINE%s)\n",
		   (tangent_method(init)==LINEAR_SECANT) ? dflt : "",
		   (tangent_method(init)==LANGRANGIAN_INTERPOLANT) ? dflt : "",
		   (tangent_method(init)==CUBIC_SPLINE) ? dflt : "");
	    if (restart_io_type(init) != NULL)
	        screen("\tTangent method from restart file (restart%s)\n",
		       (tangent_method(init)==TANGENT_METHOD_FROM_RESTART) ?
		           dflt : "");
	    screen("Enter choice: ");
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
		if (strcasecmp(s,"secant")==0)
	            tangent_method(init) = LINEAR_SECANT;
		if (strcasecmp(s,"langrangian")==0)
	            tangent_method(init) = LANGRANGIAN_INTERPOLANT;
		if (strcasecmp(s,"spline")==0)
	            tangent_method(init) = CUBIC_SPLINE;
		if (strcasecmp(s,"restart")==0)
	            tangent_method(init) = TANGENT_METHOD_FROM_RESTART;
	    }
	    if (restart_io_type(init) != NULL)
	    {
	        F_USER_INTERFACE *fuh;
	        set_tangent_operator(tangent_method(init),dim);
		fuh = f_user_hook(dim);
		interface_tangent_function(restart_intfc(init)) =
		    fuh->_interface_tangent_function;
	    }
	}
#endif /* defined(TWOD) */
#if defined(THREED)
	if (dim == 3)
	{
	    static const char *dflt = ", default";
	    screen("Select normal computation algorithm, choices are\n"
	           "\tArea weighted normals (AREA%s)\n"
	           "\tLeast squares plane fit (PLANE%s)\n",
		   (normal3d_method(init)==AREA_WEIGHTED_NORMAL) ? dflt : "",
		   (normal3d_method(init)==PLANE_FIT_NORMAL) ? dflt : "");
	    if (restart_io_type(init) != NULL)
	        screen("\tNormal method from restart file (restart%s)\n",
		       (normal3d_method(init)==NORMAL_METHOD_FROM_RESTART) ?
		           dflt : "");
	    screen("Enter choice: ");
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
		if (strcasecmp(s,"area")==0)
	            normal3d_method(init) = AREA_WEIGHTED_NORMAL;
		if (strcasecmp(s,"plane")==0)
	            normal3d_method(init) = PLANE_FIT_NORMAL;
		if (strcasecmp(s,"restart")==0)
	            normal3d_method(init) = NORMAL_METHOD_FROM_RESTART;
	    }
	}
#endif /* defined(THREED) */

	debug_print("init","Left f_prompt_for_front_options()\n");
}		/*end f_prompt_for_front_options*/


EXPORT	void	set_front_hooks(
	INIT_DATA	*init)
{
	F_INIT_DATA	*f_init = f_init_data(init);

	f_init->_set_redistribution_defaults = f_set_redistribution_defaults;
	f_init->_copy_redistribution_values = f_copy_redistribution_values;
	f_init->_prompt_for_redistribute = f_prompt_for_redistribute;
	f_init->_read_print_FlowSpecifiedRegion_data = 
	    f_read_print_FlowSpecifiedRegion_data;
	f_init->_prompt_for_front_options = f_prompt_for_front_options;
	f_init->_read_print_front_options = f_read_print_front_options;
}		/*end set_front_hooks*/

EXPORT	void	f_read_print_front_options(
	INIT_DATA *init,
	Front     *front)
{
	const IO_TYPE *io_type = restart_io_type(init);
	FILE    *file = io_type->file;
	front->interf = restart_intfc(init);
	if (next_output_line_containing_string(file,
					 "REDISTRIBUTION INFORMATION:") != NULL)
	{
	    (void) fgetstring(file,"redistribution count = ");
	    (void) fscanf(file,"%d",&redistribution_count(init));
	}
	else
	{
	    redistribution_count(init) = 0;
	}

	if (next_output_line_containing_string(file,"FRONT SPEEDS:"))
	{
	    if (!initial_read_print_max_front_speed_info(init,front))
	    {
		MAX_FRONT_SPEED *mfs = InitialMaxFrontSpeed(init);
	        int dim = Comp_grid(init).dim;

		/*Old Style output*/
	        if (fgetstring(file,"front->spfr ="))
	        {
	            int		i, c;
		    if ((c = getc(file)) != '\f')
		    {		/* NOBINARY */
			(void) ungetc(c,file);
			for (i = 0; i <= dim; ++i)
		            (void) fscan_float(file,mfs->_spfr+i);
		    }
		    else
		    {		/* BINARY */
			(void) getc(file);
			(void) read_binary_real_array(mfs->_spfr,dim+1,
			                              io_type);
		    }
	        }
	    }
	}
	read_print_FlowSpecifiedRegion_list(init,io_type);
}		/*end f_read_print_front_options*/

/*
*			     init_front():
*
*	Initialization of a front involves setting various parameters 
*	such as the type of redistribute algorithm to be used, 
*	the front spacing, the time step factor.
*/


EXPORT void init_front(
	INIT_DATA     *init,
	Front	      *front)
{
	debug_print("init","Entered init_front()\n");

	Clear_redistribution_parameters(front);
	if (restart_io_type(init) != NULL)
	    read_print_front(init,front);

		/* Choose Redistribution Method */

	Init_redistribution(init,front);

	    /* Choose Constant to Increase/Decrease Time-Steps */

	front->Tstep = time_step_control_options(init);

	debug_print("init","Left init_front()\n");
}		/*end init_front*/


/*
*		initial_front_redistribute():
*
*	Also, the redistribution
*	of the initial interface front->interf is performed.
*
*	IMPORTANT NOTE:  For restarts,  the interface is never 
*	redistributed here.
*/

/*ARGSUSED*/
EXPORT void initial_front_redistribute(
	Front	      *front,
	const IO_TYPE *restart_io_type)
{
		/* Redistribute Initial Front */
	debug_print("init","Entered initial_front_redistribute()\n");

	if (front->rect_grid->dim == 1)
	{
	    debug_print("init","Left initial_front_redistribute()\n");
	    return;
	}

	if (debugging("init"))
	{
	    (void) printf("Interface before redistribution\n");
	    print_interface(front->interf);
	}

#if defined(TWOD) || defined(THREED)
 	if (restart_io_type == NULL)
	{
            if(Tracking_algorithm(front) == GRID_BASED_TRACKING &&
               front->rect_grid->dim == 2)
            {
                printf("WARNING: in initial_front_redistribute()\n");
                printf("The Tracking_algorithm is GRID_BASED_TRACKING\n");
                printf("But the 1st redistribution is not done by GRID_BASED reconstruction\n");
            }
	    if (redistribute(front,YES,NO) != GOOD_REDISTRIBUTION)
	    {
	    	screen("ERROR in initial_front_redistribute(), "
	    	       "redistribute failed\n");
	    	clean_up(ERROR);
	    }
	}
#endif /* defined(TWOD) || defined(THREED) */

	measure_front(front);

	if (debugging("Front")) 
	    print_Front_structure(front);
	debug_print("init","Left initial_front_redistribute()\n");
}		/*end initial_front_redistribute*/


LOCAL	void	init_front_time_step_control(
	INIT_DATA *init)
{
	char	s[Gets_BUF_SIZE];
	int	dim = Comp_grid(init).dim;
	TSTEP_CONTROL	*tstep;

	debug_print("init","Entered init_front_step_control()\n");
	screen("\n\t\ttime step size control\n\n");

	tstep = &time_step_control_options(init);
	tstep->time_step_factor   = 0.75; /*DEFAULT TOLERANCE*/
	tstep->apply_cfl_at_nodes = YES;  /*DEFAULT*/
	tstep->apply_cfl_at_nodes = (debugging("NoNdCFL")) ? NO : YES;
	tstep->max_sep            = 2.0;  /*DEFAULT TOLERANCE*/
	tstep->cfl_fudge          = 1.1;  /*DEFAULT TOLERANCE*/
	tstep->frac_floor         = 0.75; /*DEFAULT TOLERANCE*/
	tstep->frac_ceil          = 1.25; /*DEFAULT TOLERANCE*/

	screen("\nThe current defaults for the front time step control are\n");
	screen("\tTime step factor = %g\n",tstep->time_step_factor);
	screen("\tApply CFL at nodes = %s\n",y_or_n(tstep->apply_cfl_at_nodes));
	screen("\tMaximum node separation at untangle = %g\n",tstep->max_sep);
	screen("\tCFL increase factor = %g\n",tstep->cfl_fudge);
	screen("\tMinimum time step modification factor = %g\n",
	       tstep->frac_floor);
	screen("\tMaximum time step modification factor = %g\n",
	       tstep->frac_ceil);

	screen("Use defaults for front time step control (default = y): ");
	(void) Gets(s);

	if (s[0] != 'n' && s[0] != 'N')
	    return;

	screen("Enter the time step factor (fraction of CFL condition - "
	       "default %g): ",tstep->time_step_factor);
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    int n;
	    n = sscan_float(s,&tstep->time_step_factor);
	    if (n != 1)
	    {
		screen("ERROR in init_front_time_step_control(), "
		       "couldn't scan Time_step_factor\n");
		clean_up(ERROR);
	    }
	}

	if (dim == 2)
	{
	    screen("Use node velocity to restrict CFL condition "
	           "(default %s): ",y_or_n(tstep->apply_cfl_at_nodes));
	    (void) Gets(s);
	    if (s[0] != '\0')
	    {
	    	if (tstep->apply_cfl_at_nodes == YES)
	    	{
	   	    if (s[0] == 'N' || s[0] == 'n')
		        tstep->apply_cfl_at_nodes = NO;
		}
		else
		{
		    if (s[0] == 'Y' || s[0] == 'y')
		        tstep->apply_cfl_at_nodes = YES;
		}
	    }
	    if (tstep->apply_cfl_at_nodes == YES)
	    {
		screen("Enter the maximum node separation at "
		       "tangles (default %g): ",tstep->max_sep);
		(void) Gets(s);
		if (s[0] != '\0')
		    (void) sscan_float(s,&tstep->max_sep);
	    }
	}

	screen("Enter the CFL increase factor (default %g): ",tstep->cfl_fudge);
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscan_float(s,&tstep->cfl_fudge);

	screen("Enter the minimum time step modification factor "
	       "(default %g): ",tstep->frac_floor);
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscan_float(s,&tstep->frac_floor);

	screen("Enter the maximum time step modification factor "
	       "(default %g): ",tstep->frac_ceil);
	(void) Gets(s);
	if (s[0] != '\0')
	    (void) sscan_float(s,&tstep->frac_ceil);
	debug_print("init","Left init_front_step_control()\n");
}		/*end init_front_time_step_control*/

EXPORT	void	f_set_redistribution_defaults(
	INIT_DATA *init)
{
	static const float STANDARD_SCALED_LENGTH = 0.75;/*TOLERANCE*/
	int dim = Comp_grid(init).dim;

	/*Set default values*/
	front_redist_mode(init) =             FULL_REDIST;
	full_curve_redist_version(init) =     EQUI_BOND_REDISTRIBUTE;
	use_rect_bdry_redistribution(init) =  NO;
	cosine_big_angle(init,GENERAL_WAVE) = 1.0;
	cosine_big_angle(init,VECTOR_WAVE) =  1.0;
	
	redistribution_frequency(init,GENERAL_WAVE) = 20; /*DEFAULT*/
	redistribution_frequency(init,VECTOR_WAVE) = 5;   /*DEFAULT*/
	redistribution_frequency(init,GENERAL_NODE) = 10; /*DEFAULT*/
	reconstruction_frequency(init) = 10;              /*DEFAULT*/

	tracking_algorithm(init) = (dim == 3) ? MIXED_TRACKING :
						GRID_FREE_TRACKING;

	front_spacing(init,GENERAL_WAVE) = STANDARD_SCALED_LENGTH;
	front_spacing(init,VECTOR_WAVE) = STANDARD_SCALED_LENGTH;

#if defined(THREED)
	/*The default values are all TOLERANCES*/
	maximum_triangle_area_factor(init,GENERAL_WAVE) = 2.0;
	minimum_triangle_area_factor(init,GENERAL_WAVE) = 0.5;
	maximum_triangle_area_factor(init,VECTOR_WAVE) = 2.0;
	minimum_triangle_area_factor(init,VECTOR_WAVE) = 0.5;
	minimum_angle_at_triangle_vertex(init,GENERAL_WAVE) = radians(15.0);
	minimum_angle_at_triangle_vertex(init,VECTOR_WAVE) = radians(15.0);
	maximum_scaled_triangle_side_length(init) = 1.3;
#endif /* defined(THREED) */
}		/*end f_set_redistribution_defaults*/

EXPORT void f_copy_redistribution_values(
	INIT_DATA	*init,
	Front		*front)
{
	set_redistribution_defaults(init);

	front_redist_mode(init) = Redistribution_mode(front);
	curve_redist_options(init) = CurveRedistributionOptions(front);

	redistribution_frequency(init,GENERAL_WAVE) =
	    Frequency_of_redistribution(front,GENERAL_WAVE);
	redistribution_frequency(init,VECTOR_WAVE) =
	    Frequency_of_redistribution(front,VECTOR_WAVE);
	redistribution_frequency(init,GENERAL_NODE) =
	    Frequency_of_redistribution(front,GENERAL_NODE);

	reconstruction_frequency(init) = Frequency_of_reconstruction(front);

	redistribution_count(init) = Redistribution_count(front);

	tracking_algorithm(init) = Tracking_algorithm(front);

	front_spacing(init,GENERAL_WAVE) = Front_spacing(front,GENERAL_WAVE);
	front_spacing(init,VECTOR_WAVE) = Front_spacing(front,VECTOR_WAVE);

#if defined(THREED)
	surface_redist_options(init) = SurfaceRedistributionOptions(front);
#endif /* defined(THREED) */
}		/*end f_copy_redistribution_values*/

/*
*			f_prompt_for_redistribute():
*
*	Prompt for redistribution parameters.
*/

EXPORT void f_prompt_for_redistribute(
	INIT_DATA   *init)
{
	int     dim = Comp_grid(init).dim;

	debug_print("init","Entered f_prompt_for_redistribute()\n");
	if (dim == 1)
	    return;

#if defined(TWOD)
	f_prompt_for_curve_redist_options(init);   
#endif /* defined(TWOD) */

	/* Choose Approximate Spacing of Points on Front */
	prompt_for_front_spacing(init);

#if defined(THREED)
	if (dim == 3)
	    f_prompt_for_surface_redist_options(init);
#endif /* defined(THREED) */

	debug_print("init","Left f_prompt_for_redistribute()\n");
}		/* end to f_prompt_for_redistribute() */


/*
*			f_init_redistribute():
*
*	Provides for the initialization of redistribution parameters
* 	in a physics independent manner.  It is assumed that the calling
*	routine decides whether to call Clear_redistribution_parameters().
*	This is to preserve any values that are set during pause or restart.
*/

EXPORT void f_init_redistribute(
	INIT_DATA	*init,
	Front		*front)
{
	debug_print("init","Entered f_init_redistribute()\n");

	if (front->rect_grid->dim == 1)
	{
	    debug_print("init","Left f_init_redistribute()\n");
	    return;
	}

	Redistribution_mode(front) = front_redist_mode(init);
#if defined(TWOD)
	f_init_curve_redistribution_parameters(init,front);
#endif /* defined(TWOD) */

	Front_spacing(front,GENERAL_WAVE) = front_spacing(init,GENERAL_WAVE);
	Front_spacing(front,VECTOR_WAVE) = front_spacing(init,VECTOR_WAVE);

#if defined(THREED)
	if (front->rect_grid->dim == 3)
	    f_init_surface_redistribution_parameters(init,front);
#endif /* defined(THREED) */
	
	debug_print("init","Left f_init_redistribute()\n");
}		/*end f_init_redistribute*/

#if defined(TWOD)
EXPORT	void	set_dflt_cur_redist_params(
	Front	*front)
{
	Curve_redistribution_function(front) = full_redistribute;
	Forward_curve_redistribute_function(front) = equi_curve_redistribute;
	Backward_curve_redistribute_function(front) =
	    backward_equi_curve_redistribute;
	Node_redistribute_function(front) = NULL;
	Delete_small_loops_function(front) = f_delete_small_loops;
	Delete_fold_back_bonds_function(front) = f_delete_fold_back_bonds;
	Delete_exterior_curves_function(front) = f_delete_exterior_curves;
	Delete_phys_remn_on_bdry_function(front) = f_delete_phys_remn_on_bdry;
	Delete_point_adjacent_to_node_function(front) =
	    f_delete_point_adjacent_to_node;
	Use_rect_boundary_redistribution(front) = NO;
	Cosine_big_angle(front,GENERAL_WAVE) =
	    Cosine_big_angle(front,VECTOR_WAVE) = 1.0;
	Front_length(front) = -HUGE_VAL;
}		/*end set_dflt_cur_redist_params*/

/*
*		f_prompt_for_curve_redist_options():
*
*	Prompt for curve redistribution options.
*/

LOCAL void f_prompt_for_curve_redist_options(
	INIT_DATA   *init)
{
	char		s[Gets_BUF_SIZE];
	float		big_angle;
	bool		use_big_angle = NO;
	int		dim = Comp_grid(init).dim;	

	debug_print("init","Entered f_prompt_for_curve_redist_oprtions()\n");

#if defined(CONSERVATIVE_ALG)
        if(dim == 2)
        {
            screen("\n\t\t2D front redistribution control\n\n"); 
            screen("Enter 2D front tracking algorithm, choices are:\n");
            screen("\tGrid free tracking(F%s),\n",
                    (tracking_algorithm(init)==GRID_FREE_TRACKING)?", dflt":"");
            screen("\tGrid based tracking (G%s),\n",
                    (tracking_algorithm(init)==GRID_BASED_TRACKING)?", dflt":"");
            screen("Enter choice: ");
            (void) Gets(s);
            switch (s[0])
            {
            case 'F':
            case 'f':
                tracking_algorithm(init) = GRID_FREE_TRACKING;
            break;
            case 'G':
            case 'g':
                tracking_algorithm(init) = GRID_BASED_TRACKING;
            break;
            default:
            break;
            }
            if (tracking_algorithm(init) == GRID_BASED_TRACKING)
            {
                /* Use the default values, for the purpose of redistribution 
                 * during the initial_front_redistribute().  
                 * The values are set in f_set_redistribution_defaults()
	    	init_redistribution_frequency(init);
                use_rect_bdry_redistribution(init) = NO;
                */
                return;
            }
        }
#endif /* if defined(CONSERVATIVE_ALG) */

	screen("\n\t\tCurve Redistribution Control\n\n");
	screen("Enter the mode of curve redistribution --\n\t");
	screen("`none'%s, `expansion'%s, or `full'%s: ",
	    (front_redist_mode(init) == NO_REDIST) ? " (dflt)" : "",
	    (front_redist_mode(init) == EXPANSION_REDIST) ? " (dflt)" : "",
	    (front_redist_mode(init) == FULL_REDIST) ? " (dflt)" : "");
	(void) Gets(s);
	switch (s[0])
	{
	case 'n':					/* None */
	case 'N':
	    front_redist_mode(init) = NO_REDIST;
	    break;

	case 'e':					/* Expansion */
	case 'E':
	    front_redist_mode(init) = EXPANSION_REDIST;
	    break;

	case 'f':					/* Full */
	case 'F':
	    front_redist_mode(init) = FULL_REDIST;
	    break;

	default:
	    break;
	}

	switch (front_redist_mode(init))
	{
	case EXPANSION_REDIST:
	    if (dim == 2)
	    	init_redistribution_frequency(init);
	    break;

	case FULL_REDIST:
	    screen("Enter version of full curve redistribution\n");
	    screen("\tordinary full curve redistribution [o%s]\n",
	    	   (full_curve_redist_version(init) == ORDINARY_REDISTRIBUTE) ?
	           "(default)" : "");
	    screen("\tequi-bond curve redistribution [e%s]\n",
	    	   (full_curve_redist_version(init) == EQUI_BOND_REDISTRIBUTE) ?
	               "(default)" : "");
	    screen("Enter choice: ");
	    (void) Gets(s);
	    switch (s[0])
	    {
	    case 'o':
	    case 'O':
	    	use_big_angle = YES;
		full_curve_redist_version(init) = ORDINARY_REDISTRIBUTE;
            break;
	    case 'e':
	    case 'E':
	    	full_curve_redist_version(init) = EQUI_BOND_REDISTRIBUTE;
		break;
	    default:
		break;
	    }

	    if (dim == 2)
	    	init_redistribution_frequency(init);

	    if (use_big_angle == YES)
	    {
	    	big_angle = acos(cosine_big_angle(init,GENERAL_WAVE));
	    	screen("Enter the hyperbolic big angle for general "
	    	       "curves (in degrees) (dflt = %g): ",degrees(big_angle));
		(void) Gets(s);
		if (s[0] != '\0')
		{
		    (void) sscan_float(s,&big_angle);
		    big_angle = radians(big_angle);
		    cosine_big_angle(init,GENERAL_WAVE) = cos(big_angle);
	        }

	        big_angle = acos(cosine_big_angle(init,VECTOR_WAVE));
	        screen("Enter the hyperbolic big angle for vector  "
		       "curves (in degrees) (dflt = %g): ",degrees(big_angle));
		(void) Gets(s);
		if (s[0] != '\0')
		{
		    (void) sscan_float(s,&big_angle);
		    big_angle = radians(big_angle);
		    cosine_big_angle(init,VECTOR_WAVE) = cos(big_angle);
		}
	    }
	    break;

	case NO_REDIST:
	default:
	    break;
	}

	if ((dim == 2) && (front_redist_mode(init) != NO_REDIST))
	{
	    screen("Type 'y' for rect grid based redistribution "
	           "of rectangular boundaries: ");
	    (void) Gets(s);
	    if (s[0] == 'y' || s[0] == 'Y')
	    	use_rect_bdry_redistribution(init) = YES;
	}
	debug_print("init","Left f_prompt_for_curve_redist_oprtions()\n");
}		/* end to f_prompt_for_curve_redist_options() */									

/*
*		f_init_curve_redistribution_parameters():
*
*	Provides for the initialization of redistribution parameters
* 	in a physics independent manner.
*/

LOCAL void f_init_curve_redistribution_parameters(
	INIT_DATA *init,
	Front	  *front)
{
	int	       dim = front->rect_grid->dim;
	CURVE_CLEANERS Sav_cleaners;

	debug_print("init","Entered f_init_curve_redistribution_parameters()\n");

		/* DEFAULT values of redistribution parameters */
        Tracking_algorithm(front) = tracking_algorithm(init);
	CurveRedistributionOptions(front) = curve_redist_options(init);
	clear_curve_redistribution_info(&Curve_redistribution_info(front));
	if (dim == 1)
	    return;

	set_dflt_cur_redist_params(front);
	Sav_cleaners = Curve_redistribution_info(front).Cleaners;

        if (Tracking_algorithm(front) == GRID_BASED_TRACKING)
        {
            /* The default Curve_redistribution_function()
             * is set in set_dflt_cur_redist_params()
             * The default redistribution_frequency() 
             * and use_rect_bdry_redistribution() is set in 
             * f_set_redistribution_defaults()
             */
	    Curve_redistribution_info(front).Cleaners = Sav_cleaners;
	    set_redistribution_frequency(init,front);
            Use_rect_boundary_redistribution(front) =
                use_rect_bdry_redistribution(init);
            return; 
        }

	switch (front_redist_mode(init))
	{
	case NO_REDIST:
	    clear_curve_redistribution_info(&Curve_redistribution_info(front));
	    Curve_redistribution_info(front).Cleaners = Sav_cleaners;
	    break;

	case EXPANSION_REDIST:
	    clear_curve_redistribution_info(&Curve_redistribution_info(front));
	    Curve_redistribution_function(front) = expansion_redistribute;
	    Curve_redistribution_info(front).Cleaners = Sav_cleaners;
	    break;

	case FULL_REDIST:
	    Curve_redistribution_function(front) = full_redistribute;
	    switch (full_curve_redist_version(init))
	    {
	    case ORDINARY_REDISTRIBUTE:
	        Forward_curve_redistribute_function(front) =
	            full_inc_redist_cur;
	        Backward_curve_redistribute_function(front) =
	            full_dec_redist_cur;
	        break;

	    case EQUI_BOND_REDISTRIBUTE:
	        Forward_curve_redistribute_function(front) =
	            equi_curve_redistribute;
	        Backward_curve_redistribute_function(front) =
	            backward_equi_curve_redistribute;
	        break;

	    default:
	        screen("ERROR in f_init_curve_redistribution_parameters(), "
	               "invalid curve redistribution version%d\n",
	               full_curve_redist_version(init));
	        clean_up(ERROR);
	    }
	    Cosine_big_angle(front,GENERAL_WAVE) =
	        cosine_big_angle(init,GENERAL_WAVE);
	    Cosine_big_angle(front,VECTOR_WAVE) =
	        cosine_big_angle(init,VECTOR_WAVE);
	    break;
	    
	default:
	    screen("ERROR in f_init_curve_redistribution_parameters(), "
	           "invalid redistribution mode %d\n",front_redist_mode(init));
	    clean_up(ERROR);
	}

	if ((dim == 2) && (Curve_redistribution_function(front) != NULL))
	{
	    set_redistribution_frequency(init,front);
	    Use_rect_boundary_redistribution(front) =
	        use_rect_bdry_redistribution(init);
	}
	
	debug_print("init","Left f_init_curve_redistribution_parameters()\n");
}		/*end f_init_curve_redistribution_parameters*/

LOCAL	void	clear_curve_redistribution_info(
	CURVE_REDISTRIBUTE *cdi)
{
	zero_scalar(cdi,sizeof(CURVE_REDISTRIBUTE));
}		/*end clear_curve_redistribution_info*/
#endif /* defined(TWOD) */


LOCAL	void	prompt_for_front_spacing(
	INIT_DATA *init)
{
	char	    s[Gets_BUF_SIZE];
	int	    dim = Comp_grid(init).dim;
	static const char  *name[] = {"", "", "curves", "surfaces"};


	if (dim == 1)
	    return;

	screen("\n\t\tfront spacing control\n\n");

	screen("Enter the spacing for general %s ",name[dim]);
	screen("in dimensionless\n\tlength/mesh units (dflt = %g): ",
	       front_spacing(init,GENERAL_WAVE));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&front_spacing(init,GENERAL_WAVE));
	    if (front_spacing(init,GENERAL_WAVE) >= 1.0)	/*TOLERANCE*/
	    	front_spacing(init,GENERAL_WAVE) = 0.999;	/*TOLERANCE*/
	}

	screen("Enter the spacing for vector type %s ",name[dim]);
	screen("in dimensionless\n\tlength/mesh units (dflt = %g): ",
		front_spacing(init,VECTOR_WAVE));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&front_spacing(init,VECTOR_WAVE));
	    if (front_spacing(init,VECTOR_WAVE) >= 1.0)		/*TOLERANCE*/
	    	front_spacing(init,VECTOR_WAVE) = 0.999;	/*TOLERANCE*/
	}
}		/*end prompt_for_front_spacing*/


#if defined(TWOD) || defined(THREED)
LOCAL	void	init_redistribution_frequency(
	INIT_DATA *init)
{
	char	s[Gets_BUF_SIZE];
	int	dim = Comp_grid(init).dim;

	debug_print("init","Entered init_redistribution_frequency()\n");

	if (tracking_algorithm(init) == GRID_BASED_TRACKING)
	{
	  redistribution_frequency(init,GENERAL_WAVE) = INT_MAX;
	  redistribution_frequency(init,VECTOR_WAVE) = INT_MAX;
	  redistribution_frequency(init,GENERAL_NODE) = INT_MAX;
	  reconstruction_frequency(init) = INT_MAX;
	  redistribution_count(init) = 0;
	}
	else
	{
	  screen("\n\t\tRedistribution Frequency Control\n\n");

 	  screen("Enter the frequency of redistribution for general curves "
	         "(dflt = %d): ",redistribution_frequency(init,GENERAL_WAVE));
	  (void) Gets(s);
	  if (s[0] != '\0')
 	    (void) sscanf(s,"%d",&redistribution_frequency(init,GENERAL_WAVE));

 	  screen("Enter the frequency of redistribution for vector curves "
	         "(dlft = %d): ",redistribution_frequency(init,VECTOR_WAVE));
	  (void) Gets(s);
	  if (s[0] != '\0')
 	    (void) sscanf(s,"%d",&redistribution_frequency(init,VECTOR_WAVE));

	  if (dim == 2)
	  {
	      screen("Enter the frequency of node redistribution "
                     "(dflt = %d): ",
		     redistribution_frequency(init,GENERAL_NODE));
	      (void) Gets(s);
	      if (s[0] != '\0')
	      {
	    	(void) sscanf(s,"%d",
	            &redistribution_frequency(init,GENERAL_NODE));
	      }
	  }
	  if (tracking_algorithm(init) == MIXED_TRACKING)
	  {
	    screen("Enter the frequency of reconstruction (dlft = %d): ",
		   reconstruction_frequency(init));
	    (void) Gets(s);
	    if (s[0] != '\0')
		(void) sscanf(s,"%d",&reconstruction_frequency(init));
	  }

		/* Determine new count_redist */

	  screen("Enter the redistribute count (default = %d): ",
		  redistribution_count(init));
	  (void) Gets(s);
	  if (s[0] != '\0')
	      (void) sscanf(s,"%d",&redistribution_count(init));
	}

	debug_print("init","Left init_redistribution_frequency()\n");
}		/*end init_redistribution_frequency*/

LOCAL void  set_redistribution_frequency(
	INIT_DATA *init,
	Front     *front)
{
	debug_print("init","Entered set_redistribution_frequency()\n");
	Frequency_of_redistribution(front,GENERAL_WAVE) =
	    redistribution_frequency(init,GENERAL_WAVE);
	Frequency_of_redistribution(front,VECTOR_WAVE) =
	    redistribution_frequency(init,VECTOR_WAVE);
	Frequency_of_redistribution(front,GENERAL_NODE) =
	    redistribution_frequency(init,GENERAL_NODE);

	if (tracking_algorithm(init) == MIXED_TRACKING)
	    Frequency_of_reconstruction(front) = reconstruction_frequency(init);

	Redistribution_count(front) = redistribution_count(init);
	debug_print("init","Left set_redistribution_frequency()\n");
}		/*end to set_redistribution_frequency()*/
#endif /* defined(TWOD) || defined(THREED) */

EXPORT	void	f_read_print_front(
	INIT_DATA	*init,
	Front		*front)
{
	Redistribution_count(front) = redistribution_count(init);
	MaxFrontSpeed(front) = InitialMaxFrontSpeed(init);
	if (debugging("restart"))
	{
	    (void) printf("Read value of MAX_FRONT_SPEED\n");
	    print_max_front_speed_info(stdout,front);
	}
}		/*end f_read_print_front*/

EXPORT	bool	f_read_print_max_front_speed_info(
	INIT_DATA       *init,
	const IO_TYPE   *io_type,
	Front	        *front,
	MAX_FRONT_SPEED	*mfs)
{
	INTERFACE *intfc = front->interf;
	FILE	  *file = io_type->file;
	int	  i, j, dim = intfc->dim;
	int	  c;
	char	  s[80];

	debug_print("restart","Entered f_read_print_max_front_speed_info()\n");

	if (fgetstring(file,"Maximum Front Speed Information")==FUNCTION_FAILED)
	{
	    (void) printf("WARNING in f_read_print_max_front_speed_info(), "
			  "can't find Maximum Front Speed Information\n");
	    debug_print("restart","Left f_read_print_max_front_speed_info()\n");
	    return FUNCTION_FAILED;
	}
	if (fgetstring(file,"Spfr = ") == FUNCTION_FAILED)
	{
	    (void) printf("WARNING in f_read_print_max_front_speed_info(), "
			  "can't find Spfr = \n");
	    debug_print("restart","Left f_read_print_max_front_speed_info()\n");
	    return FUNCTION_FAILED;
	}
	if ((c = getc(file)) == '\f') /*BINARY*/
	{
	    (void) getc(file);
	    (void) read_binary_real_array(mfs->_spfr,dim+1,io_type);
	}
	else
	{
	    (void) ungetc(c,file);
	    (void) fscan_float(file,mfs->_spfr);
	    for (i = 1; i <= dim; ++i)
	    {
	    	(void) getc(file);/*Grab comma*/
	    	(void) fscan_float(file,mfs->_spfr+i);
	    }
	}
	if (debugging("restart"))
	    (void) print_general_vector("mfs->_spfr = ",mfs->_spfr,dim+1,"\n");
	for (i = 0; i <= dim; ++i)
	{
	    (void) sprintf(s,"MaxFrontSpeedCoords[%d] = ",i);
	    if (fgetstring(file,s) == FUNCTION_FAILED)
	    {
	        (void) printf("WARNING in f_read_print_max_front_speed_info(), "
			      "can't find %s\n",s);
	        debug_print("restart","Left f_read_print_max_front_speed_info()\n");
	        return FUNCTION_FAILED;
	    }
	    if ((c = getc(file)) == '\f') /*BINARY*/
	    {
	    	(void) getc(file);
		(void) read_binary_real_array(mfs->_coords[i],dim,io_type);
	    }
	    else
	    {
	    	(void) ungetc(c,file);
	    	(void) fscan_float(file,mfs->_coords[i]);
	    	for (j = 1; j < dim; ++j)
	    	{
	    	    (void) getc(file);/*Grab comma*/
	    	    (void) fscan_float(file,mfs->_coords[i]+j);
	    	}
	    }
	}
	if (size_of_state(intfc) > 0)
	{
	    for (i = 0; i <= dim; ++i)
	    {
		(void) sprintf(s,"MaxFrontSpeedState[%d] = ",i);
		if (fgetstring(file,s) == FUNCTION_FAILED)
		{
	            (void) printf("WARNING in "
				  "f_read_print_max_front_speed_info(), "
			          "can't find %s\n",s);
	            debug_print("restart",
			  "Left f_read_print_max_front_speed_info()\n");
	            return FUNCTION_FAILED;
		}
		(void) read_print_state_data(init,io_type,
		                             mfs->_mxspst[i],intfc);
	    }
	}
	MaxFrontSpeed(front) = mfs;
	debug_print("restart","Left f_read_print_max_front_speed_info()\n");
	return FUNCTION_SUCCEEDED;
}		/*end f_read_print_max_front_speed_info*/

LOCAL	void	read_print_FlowSpecifiedRegion_list(
	INIT_DATA     *init,
	const IO_TYPE *io_type)
{
	FlowSpecifiedRegion *fsr;
	FILE                *file = io_type->file;
	int                 n, num_regions;

	if (next_output_line_containing_string(file,
			"FLOW SPECIFIED REGIONS DATA LIST") == NULL)
		return;

	(void) fgetstring(file,"Number of regions = ");
	(void) fscanf(file,"%d",&num_regions);

	/* Initialize doubly linked list */
	for (n = 0; n < num_regions; ++n)
	{
	    if (!read_print_FlowSpecifiedRegion_data(init,io_type,&fsr))
	    {
	        screen("ERROR in read_print_FlowSpecifiedRegion_list(), "
	               "can't read data printout\n");
	        clean_up(ERROR);
	    }
	}
}		/*end read_print_FlowSpecifiedRegion_list*/

LOCAL	bool	f_read_print_FlowSpecifiedRegion_data(
	INIT_DATA	    *init,
	const IO_TYPE       *io_type,
	FlowSpecifiedRegion **pfsr)
{
	FILE	                   *file = io_type->file;
	static FlowSpecifiedRegion Fsr;
	char	                   type[180];

	Fsr.type = type;

	(void) fgetstring(file,"comp = ");
	(void) fscanf(file,"%d",&Fsr.comp);
	(void) fgetstring(file,"type = ");
	(void) fscanf(file,"%s",Fsr.type);

	if (strcmp(Fsr.type,"CONSTANT_REGION") == 0)
	    return read_print_ConstantFlowRegion(init,io_type,
	                                         restart_intfc(init),
						 &Fsr,pfsr);
	if (strcmp(Fsr.type,"SKIP_COMPONENT_REGION") == 0)
	{
	    *pfsr = SetSkipComponentRegion(Fsr.comp);
	    return YES;
	}
	if (strcmp(Fsr.type,"SKIP_ALL_COMPONENTS") == 0)
	{
	    *pfsr = SetSkipAllComponents();
	    return YES;
	}

	*pfsr = &Fsr;
	return NO;
}		/*end f_read_print_FlowSpecifiedRegion_data*/

LOCAL	bool	read_print_ConstantFlowRegion(
	INIT_DATA           *init,
	const IO_TYPE       *io_type,
	INTERFACE	    *intfc,
	FlowSpecifiedRegion *fsr,
	FlowSpecifiedRegion **pfsr)
{
	Locstate state = read_print_state_data(init,io_type,NULL,intfc);

	*pfsr = &SetConstantFlowRegion(fsr->comp,state,intfc)->Fsr;

	return YES;
}		/*end read_print_ConstantFlowRegion*/

#if defined(THREED)

LOCAL void f_prompt_for_surface_redist_options(
	INIT_DATA   *init)
{
	char		s[Gets_BUF_SIZE];
	float		msts_len, max_fac, min_fac, a;
#if defined(float)
	static const char	*FMT = "%lf %lf";
#else /* defined(float) */
	static const char	*FMT = "%f %f";
#endif /* defined(float) */

	debug_print("init","Entered f_prompt_for_surface_redist_options()\n");

	screen("\n\t\tsurface redistribution control\n\n");

	screen("Enter tracking algorithm, choices are:\n");
	screen("\tGrid free tracking(F%s),\n",
		(tracking_algorithm(init)==GRID_FREE_TRACKING)?", dflt":"");
	screen("\tGrid based tracking (G%s),\n",
		(tracking_algorithm(init)==GRID_BASED_TRACKING)?", dflt":"");
	screen("\tMixed strategy tracking (M%s),\n",
		(tracking_algorithm(init)==MIXED_TRACKING)?", dflt":"");
	screen("\tHybrid strategy tracking (H%s),\n",
	       (tracking_algorithm(init)==HYBRID_TRACKING)?", dflt":"");
        screen("\tLocally grid based tracking (L%s),\n",
               (tracking_algorithm(init)==LOCALLY_GRID_BASED_TRACKING)?", dflt":"");
	screen("Enter choice: ");
	(void) Gets(s);
	switch (s[0])
	{
	case 'F':
	case 'f':
	    tracking_algorithm(init) = GRID_FREE_TRACKING;
	    break;
	case 'M':
	case 'm':
	    tracking_algorithm(init) = MIXED_TRACKING;
	    break;
	case 'G':
	case 'g':
	    tracking_algorithm(init) = GRID_BASED_TRACKING;
	    break;
	case 'H':
	case 'h':
	    tracking_algorithm(init) = HYBRID_TRACKING;
	    break;
        case 'L':
        case 'l':
            tracking_algorithm(init) = LOCALLY_GRID_BASED_TRACKING;
            break;
	default:
	    break;
	}
	if (tracking_algorithm(init) == GRID_BASED_TRACKING)
	    return;

	if (front_redist_mode(init) != NO_REDIST)
	    front_redist_mode(init) = FULL_REDIST;
	screen("\nEnter the mode of surface redistribution --\n\t"
	       "`none'%s, or `full'%s: ",
	       (front_redist_mode(init) == NO_REDIST) ? " (dflt)" : "",
	       (front_redist_mode(init) == FULL_REDIST) ? " (dflt)" : "");
	(void) Gets(s);
	switch (s[0])
	{
	case 'n':					/* None */
	case 'N':
	    front_redist_mode(init) = NO_REDIST;
		break;
	case 'f':					/* Full */
	case 'F':
	    front_redist_mode(init) = FULL_REDIST;
	    break;
	default:
	    break;
	}


	screen("Enter the maximum and minimum triangle area factors\n\t"
	       "for general waves (dflt = %g %g): ",
	       maximum_triangle_area_factor(init,GENERAL_WAVE),
	       minimum_triangle_area_factor(init,GENERAL_WAVE));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscanf(s,FMT,&max_fac,&min_fac);
	    maximum_triangle_area_factor(init,GENERAL_WAVE) = max_fac;
	    minimum_triangle_area_factor(init,GENERAL_WAVE) = min_fac;
	}

	screen("Enter the maximum and minimum triangle area factors\n\t"
	       "for vector waves (dflt = %g %g): ",
	       maximum_triangle_area_factor(init,VECTOR_WAVE),
	       minimum_triangle_area_factor(init,VECTOR_WAVE));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscanf(s,FMT,&max_fac,&min_fac);
	    maximum_triangle_area_factor(init,VECTOR_WAVE) = max_fac;
	    minimum_triangle_area_factor(init,VECTOR_WAVE) = min_fac;
	}

	screen("Enter the minimum allowed angle (in degrees)\n\t"
	       "at a triangle vertex on a general surface (dflt = %g): ",
	       degrees(minimum_angle_at_triangle_vertex(init,GENERAL_WAVE)));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&a);
	    minimum_angle_at_triangle_vertex(init,GENERAL_WAVE) = radians(a);
	}

	screen("Enter the minimum allowed angle (in degrees)\n\t"
	       "at a triangle vertex on a vector surface (dflt=%g): ",
	       minimum_angle_at_triangle_vertex(init,VECTOR_WAVE));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&a);
	    minimum_angle_at_triangle_vertex(init,VECTOR_WAVE) = radians(a);
	}

	screen("Enter the max_scaled_tri_side_length (dflt = %g): ",
	       maximum_scaled_triangle_side_length(init));
	(void) Gets(s);
	if (s[0] != '\0')
	{
	    (void) sscan_float(s,&msts_len);
	    maximum_scaled_triangle_side_length(init) = msts_len;
	}

	init_redistribution_frequency(init);


	debug_print("init","Left f_prompt_for_surface_redist_options()\n");
}		/* end to f_prompt_for_surface_redist_options() */


LOCAL void f_init_surface_redistribution_parameters(
	INIT_DATA	*init,
	Front		*front)
{
	float		*h = front->rect_grid->h;
	float		std_area, max_fac, min_fac;
	float		a;
	float		sslen;
	float		hmin;

	debug_print("init","Entered f_init_surface_redistribution_parameters()\n");

	hmin = h[0];
	if (hmin > h[1])
	    hmin = h[1];
	if (hmin > h[2])
	    hmin = h[2];

	Tracking_algorithm(front) = tracking_algorithm(init);
	SurfaceRedistributionOptions(front) = surface_redist_options(init);

	Surface_redistribution_function(front) = NULL;

	if (Tracking_algorithm(front) == GRID_BASED_TRACKING)
	    return;

	Surface_redistribution_function(front) = surface_redistribute;
	sslen = Front_spacing(front,GENERAL_WAVE);
	std_area = 0.25*sqrt(3.0)*sqr(sslen)*sqr(hmin); /*TOLERANCE*/

	max_fac = maximum_triangle_area_factor(init,GENERAL_WAVE);
	min_fac = minimum_triangle_area_factor(init,GENERAL_WAVE);
	Max_bond_len(front,GENERAL_WAVE) = sslen*sqrt(max_fac)*hmin;
	Min_bond_len(front,GENERAL_WAVE) = sslen*sqrt(min_fac)*hmin;
	Max_tri_sqr_area(front,GENERAL_WAVE) = sqr(max_fac*std_area);
	Min_tri_sqr_area(front,GENERAL_WAVE) = sqr(min_fac*std_area);

	max_fac = maximum_triangle_area_factor(init,VECTOR_WAVE);
	min_fac = minimum_triangle_area_factor(init,VECTOR_WAVE);
	Max_bond_len(front,VECTOR_WAVE) = sslen*sqrt(max_fac)*hmin;
	Min_bond_len(front,VECTOR_WAVE) = sslen*sqrt(min_fac)*hmin;
	Max_tri_sqr_area(front,VECTOR_WAVE) = sqr(max_fac*std_area);
	Min_tri_sqr_area(front,VECTOR_WAVE) = sqr(min_fac*std_area);

	/*
	* The aspect ratio of a triangle is defined as
	* aspect ratio = area/(sum of squares of sides).
	* The formula used below is the maximum aspect ratio for a
	* triangle with angle a.  Triangles with aspect ratios greater
	* than this value will have all angles greater than a.
	*/
	a = minimum_angle_at_triangle_vertex(init,GENERAL_WAVE);
	Aspect_ratio_tolerance(front,GENERAL_WAVE) = 0.5*sin(a)/(2.0 - cos(a));

	a = minimum_angle_at_triangle_vertex(init,GENERAL_WAVE);
	Aspect_ratio_tolerance(front,VECTOR_WAVE) = 0.5*sin(a)/(2.0 - cos(a));

	/* The following is used in tri_status in fredist3d.c */
	/* TO DO:  reassess 3d redistibution parameters */

	Max_scaled_tri_side_sqr_length(front) =
            maximum_scaled_triangle_side_length(init)
            * maximum_scaled_triangle_side_length(init);

	set_redistribution_frequency(init,front);

	debug_print("init","Left f_init_surface_redistribution_parameters()\n");
}		/*end f_init_surface_redistribution_parameters*/
#endif /* defined(THREED) */


/*
*			init_front_states():
*
*	Initializes the states on the front by looping over all points
*	on front->interf and calling
*
*	(*front_initializer)(point,bond,curve,left_state,right_state,init)
*
*	to fill the Locstate's associated with the point on the left and
*	right of the curve.
*/

EXPORT void init_front_states(
	Front		*front,
	INIT_DATA	*init,
	void		(*front_initializer)(POINT*,HYPER_SURF_ELEMENT*,
					     HYPER_SURF*,Locstate,Locstate,
					     INIT_DATA*))
{
	int dim = front->interf->dim;

	switch (dim)
	{
#if defined(ONED)
	case 1:
	    init_1d_front_states(front,init,front_initializer);
	    break;
#endif /* defined(ONED) */
#if defined(TWOD)
	case 2:
	    init_2d_front_states(front,init,front_initializer);
	    break;
#endif /* defined(TWOD) */
#if defined(THREED)
	case 3:
	    init_3d_front_states(front,init,front_initializer);
	    break;
#endif /* defined(THREED) */
	}
}		/*end init_front_states*/


#if defined(ONED)
LOCAL void init_1d_front_states(
	Front		*front,
	INIT_DATA	*init,
	void		(*front_initializer)(POINT*,HYPER_SURF_ELEMENT*,
					     HYPER_SURF*,Locstate,Locstate,
					     INIT_DATA*))
{
	POINT		**p;
	INTERFACE	*intfc = front->interf;

	debug_print("init","Entered init_1d_front_states()\n");
	if ((front->sizest == 0) || (front_initializer == NULL))
	{
	    debug_print("init","Left init_1d_front_states()\n");
	    return;
	}


	for (p = intfc->points; p && *p; ++p)
	{
	    debug_print("init","Initializing states on point %d\n",p);
	    (*front_initializer)(*p,(HYPER_SURF_ELEMENT *) NULL,
			         Hyper_surf(*p),left_state(*p),
				 right_state(*p),init);
	}
	debug_print("init","Left init_1d_front_states()\n");
}		/*end init_1d_front_states*/
#endif /* defined(ONED) */


#if defined(TWOD)
LOCAL void init_2d_front_states(
	Front		*front,
	INIT_DATA	*init,
	void		(*front_initializer)(POINT*,HYPER_SURF_ELEMENT*,
					     HYPER_SURF*,Locstate,Locstate,
					     INIT_DATA*))
{
	BOND		*b;
	CURVE		*c;
	INTERFACE	*intfc = front->interf;

	debug_print("init","Entered init_2d_front_states()\n");
	if (front->sizest == 0 || front_initializer == NULL)
	{
	    debug_print("init","Left init_2d_front_states()\n");
	    return;
	}

	(void) next_curve(intfc,NULL);
	while (next_curve(intfc,&c))
	{
	    if (debugging("init"))
	    	(void) printf("Initializing states on curve %llu\n",
	   	              curve_number(c));

	    (*front_initializer)(c->first->start,
	   	                 Hyper_surf_element(c->first),Hyper_surf(c),
	   	                 left_start_state(c),right_start_state(c),
				 init);
	    for (b = c->first; b != c->last; b = b->next)
	    	(*front_initializer)(b->end,
				     Hyper_surf_element(b),Hyper_surf(c),
	    		             left_state(b->end),right_state(b->end),
				     init);
	    (*front_initializer)(c->last->end,Hyper_surf_element(c->last),
	   	                 Hyper_surf(c),left_end_state(c),
				 right_end_state(c),init);
	}
	if (debugging("init_front_states"))
	{
	    print_interface(front->interf);
	    show_intfc_states(front->interf);
	}

	debug_print("init","Left init_2d_front_states()\n");
}		/*end init_2d_front_states*/
#endif /* defined(TWOD) */

#if defined(THREED)
LOCAL void init_3d_front_states(
	Front		*front,
	INIT_DATA	*init,
	void		(*front_initializer)(POINT*,HYPER_SURF_ELEMENT*,
					     HYPER_SURF*,Locstate,Locstate,
					     INIT_DATA*))
{
	POINT		*p;
	CURVE		**c;
	HYPER_SURF_ELEMENT *hse;
	HYPER_SURF	*hs;
	Locstate	sl, sr;
	INTERFACE	*intfc = front->interf;
	BOND_TRI 	**btris;
	BOND 		*bond;
	int 		i;

	debug_print("init","Entered init_3d_front_states()\n");
	if (front->sizest == 0 || front_initializer == NULL)
	{
	    debug_print("init","Left init_3d_front_states()\n");
	    return;
	}

	for (c = intfc->curves; c && *c; ++c)
	{
	    i = 0;
	    bond = (*c)->first;
	    if ((btris = Btris(bond)) == NULL) continue;
	    while (btris[i])
	    {
		p = bond->start;
		hse = Hyper_surf_element(btris[i]->tri);
		hs = Hyper_surf(Surface_of_tri(btris[i]->tri));
	    	sl = left_start_btri_state(btris[i]);
	    	sr = right_start_btri_state(btris[i]);
	    	(*front_initializer)(p,hse,hs,sl,sr,init);
		for (; bond; bond = bond->next)
		{
		    btris = Btris(bond);
		    p = bond->end;
		    hse = Hyper_surf_element(btris[i]->tri);
		    sl = left_end_btri_state(btris[i]);
		    sr = right_end_btri_state(btris[i]);
	    	    (*front_initializer)(p,hse,hs,sl,sr,init);
		}
	    	bond = (*c)->first;
		btris = Btris(bond);
		++i;
	    }
	}
	(void) next_point(intfc,NULL,NULL,NULL);
	while (next_point(intfc,&p,&hse,&hs))
	{
	    slsr(p,hse,hs,&sl,&sr); 
	    (*front_initializer)(p,hse,hs,sl,sr,init);
	}

	debug_print("init","Left init_3d_front_states()\n");
}		/*end init_3d_front_states*/
#endif /* defined(THREED) */
