/*
 * *                               doverturepatchmesh3d.c:
 * *
 * *       Copyright 1999 by The University at Stony Brook, All rights reserved.
 * *
 * *       Contains routines for initializing and maintaining nodes in the
 * *       automatic mesh grid level tree.
 * */

#define DEBUG_STRING    "doverturepatchmesh"
#include <driver/dlocaldecs.h>

#if defined(THREED) 
#if defined(USE_OVERTURE) 

/* Declare local variables here. */
LOCAL PP_GRID         *pp_grid;
/* Declare local functions here. */
LOCAL int    pseudo_grid_based_adv_amr_frs3d(Front***,
                float,float*,Overparam*,Wave**,
                Front**,int,Wv_on_pc**,int);

LIB_LOCAL void set_refinement_global3d(
        Front   *front)
{
        pp_grid = front->pp_grid;
}

LIB_LOCAL int perform_overture_init_mesh_refinement3d(
        CompositeGrid                  *rect_over_grid,
        doubleCompositeGridFunction    *cg_function,
        CompositeGrid                  **new_cg,
        Overparam         *overparam,
        int               start_level)
{

        int         i, j;
        int         base[MAXD], bound[MAXD];
        int         baseLevel = overparam->baseLevel;
        int         numberOfRefinementLevels = overparam->numberOfRefinementLevels;
        real        efficiency = overparam->efficiency;
        real        errorThreshold = overparam->errorThreshold;
        int         refinementRatio = overparam->refinementRatio;
        Regrid      regrid;
        Ogen        ogen;
        Range       all;
        Index       I1,I2,I3;
        int         currentGrid = 0, nextGrid;
        realArray   pos(1,3), uval(1,7);

        DEBUG_ENTER(perform_overture_init_mesh_refinement)

        if(debugging("over_init_mesh"))
        {
            printf("--------------------------------------\n");
            printf("Before start init_mesh_refine:\n");
            for(int grid = 0; grid<rect_over_grid->numberOfComponentGrids(); grid++)
            {
                getIndex((*rect_over_grid)[grid].indexRange(),I1,I2,I3);
                printf("grid = %d\n",grid);
                printf("x dir indexRange base, bond[%d,%d]\n",I1.getBase(), I1.getBound());
                printf("y dir indexRange base, bond[%d,%d]\n",I2.getBase(), I2.getBound());
                printf("z dir indexRange base, bond[%d,%d]\n",I3.getBase(), I3.getBound());
                getIndex((*rect_over_grid)[grid].dimension(),I1,I2,I3);
                printf("grid = %d\n",grid);
                printf("x dir dimension base, bond[%d,%d]\n",I1.getBase(), I1.getBound());
                printf("y dir dimension base, bond[%d,%d]\n",I2.getBase(), I2.getBound());
                printf("z dir dimension base, bond[%d,%d]\n",I3.getBase(), I3.getBound());
            }
            printf("--------------------------------------\n");
        }

        CompositeGrid cga[2];
        cga[0] = *rect_over_grid;
        cga[0].update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        cga[1] = *rect_over_grid;
        cga[1].update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );

        InterpolateRefinements interp(cga[0].numberOfDimensions());
        interp.setOrderOfInterpolation(3);  /* was 1 */

        ErrorEstimator errorEstimator(interp);
        /* set the scale factors for the solution (estimate of solution size) */
        RealArray scaleFactor(1);
        scaleFactor=1.;
        errorEstimator.setScaleFactor(scaleFactor);
        /* set the default number of smoothing steps for smoothing the error */
        errorEstimator.setDefaultNumberOfSmooths(overparam->numberOfSmooths);

        Interpolant interpolant;
        interpolant.setImplicitInterpolationMethod(Interpolant::iterateToInterpolate);
        interpolant.setInterpolateRefinements(interp);

        regrid.setEfficiency(efficiency);
        regrid.setRefinementRatio(refinementRatio);

        CompositeGridOperators op(cga[0]);

        doubleCompositeGridFunction error;
        cg_function->updateToMatchGrid(cga[0]);
        cg_function->setOperators(op);


        for( int grid = 0; grid < cga[0].numberOfComponentGrids(); grid++)
        {
            /*
            printf("Original grid[%d] print Dens\n", grid);
            (*cg_function)[grid].display("density");
            */
        }

        for( int level=start_level+1; level<numberOfRefinementLevels; level++ )
        {
            nextGrid = (currentGrid+1) % 2;

            CompositeGrid& cgc = cga[currentGrid];
            CompositeGrid& cgNew = cga[nextGrid];

            error.updateToMatchGrid(cgc);
            for(int grid = 0; grid < cgc.numberOfComponentGrids(); grid++)
            {
                int errorcglevel;
                errorcglevel = (cgc).refinementLevelNumber(grid);
                if(errorcglevel == level - 1)
                {
                    getIndex(cgc[grid].dimension(), I1, I2, I3);
                    error[grid](I1, I2, I3) = 0.0;
                }
            }

            interpolant.updateToMatchGrid(cgc,level-1);
            op.updateToMatchGrid(cgc);
            error.setOperators(op);
            printf("Error computed = %d ",
             errorEstimator.computeAndSmoothErrorFunction(*cg_function, error));
            printf("Error: min=%e, max=%e \n",min(error),max(error));

            regrid.regrid(cgc,cgNew, error, errorThreshold, level, baseLevel);
            ogen.updateRefinement(cgNew);
            cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );

            printf("level[%d], cgNew # of levels [%d], numberOfComponentGrids [%d]\n",
                   level,cgNew.numberOfMultigridLevels(),cgNew.numberOfComponentGrids());
            printf("cgc # of levels [%d],numberOfComponentGrids [%d]\n",
                   cgc.numberOfMultigridLevels(), cgc.numberOfComponentGrids());

            if(YES == stop_adding_amr_grids(cgc, cgNew))
            {
                currentGrid = (currentGrid+1) % 2;
                show_grids_info(cgNew);
                break;
            }

            doubleCompositeGridFunction uu2;
            uu2.updateToMatchGrid(cgNew);
            interp.interpolateRefinements(*cg_function,uu2);

            for( int grid = 1; grid < cgNew.numberOfComponentGrids(); grid++)
            {
                int base1, base2, base3, bound1, bound2, bound3;
                int ix, iy, iz;
                float x0, x1, y0, y1, z0, z1, dx, dy, dz;
                getIndex(cgNew[grid].indexRange(),I1,I2,I3);
                base1 = I1.getBase();
                base2 = I2.getBase();
                base3 = I3.getBase();
                bound1 = I1.getBound();
                bound2 = I2.getBound();
                bound3 = I3.getBound();

                /* Z direction. */
                /* 
                printf("Z direction buffer zone covered\n");
                pos(0,0) = cgNew[grid].vertex()(base1-4,base2-4,base3-4,axis1);
                pos(0,1) = cgNew[grid].vertex()(base1-4,base2-4,base3-4,axis2);
                pos(0,2) = cgNew[grid].vertex()(base1-4,base2-4,base3-4,axis3);
                pos.display("z lower interp pos");
                pos(0,0) = cgNew[grid].vertex()(bound1+4,bound2+4,base3-1,axis1);
                pos(0,1) = cgNew[grid].vertex()(bound1+4,bound2+4,base3-1,axis2);
                pos(0,2) = cgNew[grid].vertex()(bound1+4,bound2+4,base3-1,axis3);
                pos.display("z lower interp pos");

                pos(0,0) = cgNew[grid].vertex()(base1-4,base2-4,bound3,axis1);
                pos(0,1) = cgNew[grid].vertex()(base1-4,base2-4,bound3,axis2);
                pos(0,2) = cgNew[grid].vertex()(base1-4,base2-4,bound3,axis3);
                pos.display("z upper interp pos");
                pos(0,0) = cgNew[grid].vertex()(bound1+4,bound2+4,bound3+4,axis1);
                pos(0,1) = cgNew[grid].vertex()(bound1+4,bound2+4,bound3+4,axis2);
                pos(0,2) = cgNew[grid].vertex()(bound1+4,bound2+4,bound3+4,axis3);
                pos.display("z upper interp pos");
                */

                for (ix = base1-4; ix <= bound1+4; ix++)
                {
                    for (iy = base2-4; iy <= bound2+4; iy++)
                    { 
                        for (iz = base3-4; iz < base3; iz++)
                        {
                            pos(0,0) = cgNew[grid].vertex()(ix,iy,iz,axis1);
                            pos(0,1) = cgNew[grid].vertex()(ix,iy,iz,axis2);
                            pos(0,2) = cgNew[grid].vertex()(ix,iy,iz,axis3);
                            interpolatePoints(pos, *cg_function, uval);
                            uu2[grid](ix,iy,iz) = uval(0,0);
                            /* 
                            pos.display("interp pos");
                            uval.display("interp uval");
                            */
                        }
                        for (iz = bound3; iz <= bound3+4; iz++)
                        {
                            pos(0,0) = cgNew[grid].vertex()(ix,iy,iz,axis1);
                            pos(0,1) = cgNew[grid].vertex()(ix,iy,iz,axis2);
                            pos(0,2) = cgNew[grid].vertex()(ix,iy,iz,axis3);
                            interpolatePoints(pos, *cg_function, uval);
                            uu2[grid](ix,iy,iz) = uval(0,0);
                            /* 
                            pos.display("interp pos");
                            uval.display("interp uval");
                            */
                        }
                    }
                }

                /* Y direction. */
                /* 
                printf("Y direction buffer zone covered\n");
                pos(0,0) = cgNew[grid].vertex()(base1-4,base2-4,base3,axis1);
                pos(0,1) = cgNew[grid].vertex()(base1-4,base2-4,base3,axis2);
                pos(0,2) = cgNew[grid].vertex()(base1-4,base2-4,base3,axis3);
                pos.display("y lower interp pos");
                pos(0,0) = cgNew[grid].vertex()(bound1+4,base2-1,bound3-1,axis1);
                pos(0,1) = cgNew[grid].vertex()(bound1+4,base2-1,bound3-1,axis2);
                pos(0,2) = cgNew[grid].vertex()(bound1+4,base2-1,bound3-1,axis3);
                pos.display("y lower interp pos");

                pos(0,0) = cgNew[grid].vertex()(base1-4,bound2,base3,axis1);
                pos(0,1) = cgNew[grid].vertex()(base1-4,bound2,base3,axis2);
                pos(0,2) = cgNew[grid].vertex()(base1-4,bound2,base3,axis3);
                pos.display("y upper interp pos");
                pos(0,0) = cgNew[grid].vertex()(bound1+4,bound2+4,bound3-1,axis1);
                pos(0,1) = cgNew[grid].vertex()(bound1+4,bound2+4,bound3-1,axis2);
                pos(0,2) = cgNew[grid].vertex()(bound1+4,bound2+4,bound3-1,axis3);
                pos.display("y upper interp pos");
                */

                for (ix = base1-4; ix <= bound1+4; ix++)
                {
                    for (iz = base3; iz < bound3; iz++)
                    { 
                        for (iy = base2-4; iy < base2; iy++)
                        {
                            pos(0,0) = cgNew[grid].vertex()(ix,iy,iz,axis1);
                            pos(0,1) = cgNew[grid].vertex()(ix,iy,iz,axis2);
                            pos(0,2) = cgNew[grid].vertex()(ix,iy,iz,axis3);
                            interpolatePoints(pos, *cg_function, uval);
                            uu2[grid](ix,iy,iz) = uval(0,0);
                        }
                        for (iy = bound2; iy <= bound2+4; iy++)
                        {
                            pos(0,0) = cgNew[grid].vertex()(ix,iy,iz,axis1);
                            pos(0,1) = cgNew[grid].vertex()(ix,iy,iz,axis2);
                            pos(0,2) = cgNew[grid].vertex()(ix,iy,iz,axis3);
                            interpolatePoints(pos, *cg_function, uval);
                            uu2[grid](ix,iy,iz) = uval(0,0);
                        }
                    }
                }
                /* X direction. */
                /* 
                printf("X direction buffer zone covered\n");
                pos(0,0) = cgNew[grid].vertex()(base1-4,base2,base3,axis1);
                pos(0,1) = cgNew[grid].vertex()(base1-4,base2,base3,axis2);
                pos(0,2) = cgNew[grid].vertex()(base1-4,base2,base3,axis3);
                pos.display("x lower interp pos");
                pos(0,0) = cgNew[grid].vertex()(base1-1,bound2-1,bound3-1,axis1);
                pos(0,1) = cgNew[grid].vertex()(base1-1,bound2-1,bound3-1,axis2);
                pos(0,2) = cgNew[grid].vertex()(base1-1,bound2-1,bound3-1,axis3);
                pos.display("x lower interp pos");

                pos(0,0) = cgNew[grid].vertex()(bound1,base2,base3,axis1);
                pos(0,1) = cgNew[grid].vertex()(bound1,base2,base3,axis2);
                pos(0,2) = cgNew[grid].vertex()(bound1,base2,base3,axis3);
                pos.display("x upper interp pos");
                pos(0,0) = cgNew[grid].vertex()(bound1+4,bound2-1,bound3-1,axis1);
                pos(0,1) = cgNew[grid].vertex()(bound1+4,bound2-1,bound3-1,axis2);
                pos(0,2) = cgNew[grid].vertex()(bound1+4,bound2-1,bound3-1,axis3);
                pos.display("x upper interp pos");
                */
 
                for (iy = base2; iy < bound2; iy++)
                {
                    for (iz = base3; iz < bound3; iz++)
                    { 
                        for (ix = base1-4; ix < base1; ix++)
                        {
                            pos(0,0) = cgNew[grid].vertex()(ix,iy,iz,axis1);
                            pos(0,1) = cgNew[grid].vertex()(ix,iy,iz,axis2);
                            pos(0,2) = cgNew[grid].vertex()(ix,iy,iz,axis3);
                            interpolatePoints(pos, *cg_function, uval);
                            uu2[grid](ix,iy,iz) = uval(0,0);
                        }
                        for (ix = bound1; ix <= bound1+4; ix++)
                        {
                            pos(0,0) = cgNew[grid].vertex()(ix,iy,iz,axis1);
                            pos(0,1) = cgNew[grid].vertex()(ix,iy,iz,axis2);
                            pos(0,2) = cgNew[grid].vertex()(ix,iy,iz,axis3);
                            interpolatePoints(pos, *cg_function, uval);
                            uu2[grid](ix,iy,iz) = uval(0,0);
                        }
                    }
                }
            }
            (cg_function)->updateToMatchGrid(cgNew);
            (cg_function)->dataCopy(uu2);

            /*
            for( int grid = cgc.numberOfComponentGrids(); 
                     grid < cgNew.numberOfComponentGrids(); grid++)
            {
                printf("Updated grid[%d] level[%d] print UU Dens\n", grid,level);
                Dens[grid].display("density");
            }
            */
            printf("Regrid: Create AMR Grids on processing level[%d]\n", level);
            show_grids_info(cgNew);

            currentGrid = (currentGrid+1) % 2;
        }

        if(overparam->extra_buf != 0)
            Aug_compositegrid_by_error3d(error, cga[currentGrid], overparam);

        (*new_cg) = new CompositeGrid();
        (**new_cg) = cga[currentGrid];

        DEBUG_LEAVE(perform_overture_init_mesh_refinement)

        return (*new_cg)->numberOfComponentGrids();

}

LIB_LOCAL void  Aug_compositegrid_by_error3d(
        doubleCompositeGridFunction&  error,
        CompositeGrid&      cg,
        Overparam           *overparam)
{

}

LIB_LOCAL  void init_patch_states3d(
        Wave            *wave,
        Front           *front,
        INIT_DATA       *init,
        void            (*initializer)(float*,COMPONENT,Locstate,INTERFACE*,
                                    INIT_DATA*))
{
        COMPONENT       comp;
        Locstate        state;
        float           *coords;
        int             icoords[MAXD];
        int             dim = wave->rect_grid->dim;
        int             i, ix, iy, iz;
        int             imin[MAXD], imax[MAXD];

        debug_print("init","Entered init_patch_states3d()\n");

        if (wave->sizest == 0 || initializer == NULL)
        {
            debug_print("init","Left init_patch_states3d()\n");
            return;
        }

        for (i = 0; i < dim; ++i)
        {
            imin[i] = -wave->rect_grid->lbuf[i];
            imax[i] = wave->rect_grid->gmax[i] + wave->rect_grid->ubuf[i];
        }

        for (iz = imin[2]; iz < imax[2]; ++iz)
        {
            icoords[2] = iz;
            for (iy = imin[1]; iy < imax[1]; ++iy)
            {
                icoords[1] = iy;
                for (ix = imin[0]; ix < imax[0]; ++ix)
                {
                    icoords[0] = ix;
                    coords = Rect_coords(icoords,wave);
                    comp = Rect_comp(icoords,wave);
                    state = Rect_state(icoords,wave);
                    (*initializer)(coords,comp,state,front->interf,init);
                }
            }
        }
        debug_print("init","Left init_patch_states3d()\n");
}

LIB_LOCAL int pseudo_advance_amr_fronts3d(
        Front           ***newfronts,
        float           dt,
        float           *dt_frac,
        Overparam       *overparam,
        Wave            **wvs,
        Front           **frs,
        int             num_patches,
        Wv_on_pc        **redistr_table,
        int             max_n_patch)
{
        int step_status;
        DEBUG_ENTER(pseudo_advance_amr_fronts3d) 

        switch(Tracking_algorithm(frs[0]))
        {
        case GRID_BASED_TRACKING:
            step_status = pseudo_grid_based_adv_amr_frs3d(
                       newfronts, dt, dt_frac, overparam, 
                     wvs, frs, num_patches, redistr_table, max_n_patch);
        break;
        default:
            printf("ERROR 3D tracking trategy %d is not implemented\n",
                 Tracking_algorithm(frs[0]));
            clean_up(ERROR);  
        }

        DEBUG_LEAVE(pseudo_advance_amr_fronts3d) 
        return step_status;  
}

LOCAL int pseudo_grid_based_adv_amr_frs3d(
        Front           ***newfronts,
        float           dt,
        float           *dt_frac,
        Overparam       *overparam,
        Wave            **wvs,
        Front           **frs,
        int             num_patches,
        Wv_on_pc        **redistr_table,
        int             max_n_patch)
{
        int             timesteps;
        int             i, k;
        int             status;
        Front           **newfrs;
        INTERFACE       *sav_intfc; 
        int             dim = frs[0]->interf->dim;
        int             step = frs[0]->step;
        bool            sav_copy;
        int             myid, numnodes;
        bool            reconstruct_status;

        DEBUG_ENTER(pseudo_grid_based_adv_amr_frs3d)
        myid = pp_mynode();
        numnodes = pp_numnodes();

        vector(&newfrs,num_patches,sizeof(Front*));
        /* Normal front sweep */
        status = reconstruct_status = GOOD_STEP;
        for (i = num_patches-1; i >=0; i--)
        {
            if(frs[i]->patch_level != frs[i]->NumberOfLevels-1
               && frs[i]->patch_level != 0)
            {
                sav_intfc = current_interface();
                sav_copy = copy_intfc_states();
                newfrs[i] = copy_front(frs[i]);
                newfrs[i]->interf = copy_interface(frs[i]->interf);
                set_current_interface(sav_intfc);
                set_copy_intfc_states(sav_copy);
                /* 
                delete_patch_all_curves(newfrs[i]);
                */
                newfront_to_distri_table(frs[i],newfrs[i],
                     num_patches,redistr_table,max_n_patch);
                continue;
            }
            if (debugging("hyp_amr"))
                printf("normal_advance_front patch[%d], level[%d]",
                                i,frs[i]->patch_level);
            status = normal_advance_front(dt,dt_frac,frs[i],&newfrs[i],
                                (POINTER)wvs[i]);
            if (debugging("hyp_amr"))
                printf(" status = %d\n",status);
            switch(status)
            {
            case GOOD_STEP:
            break;
            case ERROR_IN_STEP:
            default:
            screen("FT_ERROR: in hyp_amr(),"
                   "  normal_advance_front() failed\n");
            clean_up(ERROR);
            }

            reconstruct_status = reconstruct_front_at_grid_crossing(newfrs[i],
                                    frs[i],ON_DUAL_GRID);
            switch(reconstruct_status)
            {
            case GOOD_STEP:
            break;
            case MODIFY_TIME_STEP:
                goto normal_status;    
            case ERROR_IN_STEP:
            default:
            screen("FT_ERROR: in hyp_amr(),"
                   "  reconstruct_front_at_grid_crossing() failed\n");
            clean_up(ERROR);
            }
            newfront_to_distri_table(frs[i],newfrs[i],
                 num_patches,redistr_table,max_n_patch);
        }
normal_status: 
        reconstruct_status = pp_min_status(reconstruct_status);
        if(!reconstruct_status)
        {
            (void) printf("WARNING in pseudo_grid_based_adv_amr_frs3d(), "
                          "reconstruct_front failed, "
                          "MODIFY_TIME_STEP\n");
            status = MODIFY_TIME_STEP;
            *dt_frac *= TIME_STEP_REDUCTION_FACTOR(frs[0]->interf);
            for (i = num_patches-1; i >=0; i--)
            {
                newfront_to_distri_table(newfrs[i],frs[i],
                   num_patches,redistr_table,max_n_patch);
                free_front(newfrs[i]);
            }
            free(newfrs);
            *newfronts = NULL;
            return status;
        }

        printf("After Normal: assembly_distribute_patch_fronts ");
        status = assembly_distribute_patch_fronts(newfrs,num_patches,
                      redistr_table,max_n_patch, NO);
        printf("status = %d\n", status);
        if (pp_min_status(status) != FUNCTION_SUCCEEDED)
        {
            screen("WARNING overture_hyp_amr(),"
                  " failed after normal: assembly_distribute_patch_fronts()\n");
            clean_up(ERROR); 
        }

        printf("EXIT in pseudo_grid_based_adv_amr_frs3d\n");
        clean_up(0); 
        DEBUG_LEAVE(pseudo_grid_based_adv_amr_frs3d)
        return status; 
}

#endif /* if defined(USE_OVERTURE) */
#endif /* if defined(THREED) */

