/*
* *                               doverturepatchmesh2d.c:
* *
* *       Copyright 1999 by The University at Stony Brook, All rights reserved. * *
* *       Contains routines for initializing and maintaining nodes in the
* *       automatic mesh grid level tree.
* */

#define DEBUG_STRING    "doverturepatchmesh2"
#include <driver/dlocaldecs.h>

#if defined(USE_OVERTURE)

#if defined(TWOD)
/* Declare local variables here. */
LOCAL PP_GRID         *pp_grid = NULL; 
LOCAL INTERFACE       *intfc = NULL; 

LOCAL void  copy_gridfunction_from_uu1(const doubleCompositeGridFunction&,
                    const CompositeGrid&, int, doubleCompositeGridFunction&);

LOCAL int   syncronize_redist_status(Front*,int);
LOCAL void  pp_send_grid_collection(int*,int,int,Rect_ComponentGrid_Collection*);
LOCAL void  pp_receive_grid_collection(int*,int,int,Rect_ComponentGrid_Collection*);
LOCAL void  update_error_by_neighbor2d(doubleCompositeGridFunction&,int,int,int,
                 Rect_ComponentGrid_Collection,Rect_ComponentGrid_Collection,Overparam*);
LOCAL void  perform_update_error2d(doubleCompositeGridFunction&,const CompositeGrid&,Overparam*);
LOCAL void  split_giant_compositegrid2d(CompositeGrid&,Overparam*);
LOCAL void  print_rect_componentgrid_collection(Rect_ComponentGrid_Collection);
LOCAL void  print_rect_componentgrid_collection_on_level(Rect_ComponentGrid_Collection,int);
LOCAL int   perform_split_Grid2d(MappedGrid&, Rect_ComponentGrid_Collection&, int, int, int);

LOCAL void  split_patches_at_level(int,CompositeGrid&,CompositeGrid&,
                  Rect_ComponentGrid_Collection&,int,int,Overparam*);
LOCAL void  split_patches_by_line(int,CompositeGrid&,CompositeGrid&,
                  Rect_ComponentGrid_Collection&,Overparam*);
LOCAL void  copy_rect_grids_collection(Rect_ComponentGrid_Collection&,
                  Rect_ComponentGrid_Collection&);


LIB_LOCAL void set_refinement_global2d(
        Front   *front)
{
        pp_grid = front->pp_grid; 
        intfc = front->interf;
}

LIB_LOCAL int pseudo_advance_amr_fronts2d(
        Front           ***newfronts, 
        float           dt,
        float           *dt_frac,
        Overparam       *overparam,
        Wave            **wvs,
        Front           **frs,
        int             num_patches,
        Wv_on_pc        **redistr_table,
        int             max_n_patch)
{
        int             i, k, changed;
        int             status = GOOD_STEP;
        bool            scatter_normally_propagated_front = YES;
        bool            scatter_tangentially_propagated_front = YES;
        Front           **newfrs;
        INTERFACE       *sav_intfc;  
        bool            sav_copy;
        long            fineintfc_modified = NO;
        int             myid, numnodes;
        char            *msg = NULL;

        DEBUG_ENTER(pseudo_advance_amr_fronts2d)
        myid = pp_mynode();
        numnodes = pp_numnodes();
        
        vector(&newfrs,num_patches,sizeof(Front*));
        /* Normal front sweep */
begin_advance_front2d: 
        status = GOOD_STEP;
        for (i = num_patches-1; i >= 0; i--)
        {
            if(frs[i]->patch_level != frs[i]->NumberOfLevels-1
               && frs[i]->patch_level != 0)
            {
                newfrs[i] = copy_front(frs[i]);

                sav_intfc = current_interface();
                sav_copy = copy_intfc_states();
                set_size_of_intfc_state(size_of_state(frs[i]->interf));
                set_copy_intfc_states(YES);                
                newfrs[i]->interf = copy_interface(frs[i]->interf);
                set_current_interface(sav_intfc);
                set_copy_intfc_states(sav_copy);
                delete_patch_all_curves(newfrs[i]);
                changed = i;
                newfront_to_distri_table(frs[i],newfrs[i],
                     num_patches,redistr_table,max_n_patch);
                continue;
            }
            if (debugging("hyp_amr"))
                printf("normal_advance_front patch[%d], level[%d]",
                                i,frs[i]->patch_level);
            status = normal_advance_front(dt,dt_frac,frs[i],&newfrs[i],
                                (POINTER)wvs[i]);
            if (debugging("hyp_amr"))
                printf(" status = %d\n",status);
            switch(status)
            {
            case GOOD_STEP:
            break;

            case MODIFY_TIME_STEP:
            case REPEAT_TIME_STEP:
                goto normal_status;
            break;
            case ERROR_IN_STEP:
            default:
            screen("ERROR: in pseudo_advance_amr_fronts2d(),"
                   "  normal_advance_front() failed\n");
            clean_up(ERROR);
            }
            changed = i;
            newfront_to_distri_table(frs[i],newfrs[i],
                 num_patches,redistr_table,max_n_patch);
        }

normal_status:
        status = syncronize_time_step_status(status,frs[0]->pp_grid);
        if (status != GOOD_STEP)
        {
            screen("WARNING overture_hyp_amr(),"
               " failed after normal_advance_front, status = %d\n",status);
            for (i = num_patches-1; i >= changed; i--)
            {
                newfront_to_distri_table(newfrs[i],frs[i],
                   num_patches,redistr_table,max_n_patch);
                free_front(newfrs[i]);
            }
            free(newfrs);
            *newfronts = NULL;
            return status;        
        }

        if(YES == scatter_normally_propagated_front)
        {
            start_clock("amr_fr_scat1");
            status = scatter_patch_fronts(newfrs,num_patches,redistr_table,
                         max_n_patch,overparam,NO);
            stop_clock("amr_fr_scat1");
            /*
            status = assembly_distribute_patch_fronts(newfrs,num_patches,
                  redistr_table,max_n_patch, NO);
            */
            if (pp_min_status(status) != FUNCTION_SUCCEEDED)
            {
                screen("WARNING overture_hyp_amr(),"
                  " failed after normal: scatter_patch_fronts()\n");
                screen("Do advance_front again\n"); 
                scatter_normally_propagated_front = NO;
                scatter_tangentially_propagated_front = NO;
                set_current_interface(frs[0]->interf);  
                for (i = num_patches-1; i >=0; i--)
                {
                    newfront_to_distri_table(newfrs[i],frs[i],
                       num_patches,redistr_table,max_n_patch);
                    free_front(newfrs[i]);
                }
                goto begin_advance_front2d;
            }
        }

        /* Tangential front sweep */
        status = GOOD_STEP;
        for (i = num_patches-1; i >= 0; i--)
        {
            if(frs[i]->patch_level != frs[i]->NumberOfLevels-1
               && frs[i]->patch_level != 0)
                continue;

            set_amr_intfc_tol(frs[i]->interf,
                pow(2.0,-(frs[i]->NumberOfLevels-1.0-frs[i]->patch_level)));
            set_amr_intfc_tol(newfrs[i]->interf,
                pow(2.0,-(newfrs[i]->NumberOfLevels-1.0-newfrs[i]->patch_level)));

            if (debugging("hyp_amr"))
                printf("tangnt_advance_front %d's: patch[%d],level[%d]",
                       i,frs[i]->patch_number, frs[i]->patch_level);
            status = tangnt_advance_front(dt,dt_frac,frs[i],&newfrs[i],
                                (POINTER)wvs[i]);

            set_amr_intfc_tol(frs[i]->interf,
                pow(2.0,frs[i]->NumberOfLevels-1.0-frs[i]->patch_level));
            set_amr_intfc_tol(newfrs[i]->interf,
                pow(2.0,newfrs[i]->NumberOfLevels-1.0-newfrs[i]->patch_level));

            if (debugging("hyp_amr"))
                printf(" status = %d\n",status);
            switch(status)
            {
            case GOOD_STEP:
            break;

            case MODIFY_TIME_STEP:
            case REPEAT_TIME_STEP:
                goto tangnt_status;
            break;
            case ERROR_IN_STEP:
            default:
            screen("ERROR: overture_hyp_amr(),"
                   " tangnt_advance_front() failed\n");
            clean_up(ERROR);
            }
        }

tangnt_status:
        status = syncronize_time_step_status(status,frs[0]->pp_grid);
        if (status != GOOD_STEP)
        {
            screen("ERROR overture_hyp_amr(),"
                   " failed after tangnt_advance_front, status = %d\n", status);
            clean_up(ERROR);
        }
        if(YES == scatter_tangentially_propagated_front)
        {
            start_clock("amr_fr_scat2");
            status = scatter_patch_fronts(newfrs,num_patches,redistr_table,
                         max_n_patch,overparam,NO);
            stop_clock("amr_fr_scat2");
            /*
            status = assembly_distribute_patch_fronts(newfrs,num_patches,
                      redistr_table,max_n_patch, NO);
            */
            if (pp_min_status(status) != FUNCTION_SUCCEEDED)
            {
                screen("WARNING overture_hyp_amr(),"
                  " failed after tangnt: scatter_patch_fronts()\n");
                screen("Do advance_front again\n"); 
                scatter_normally_propagated_front = NO;
                scatter_tangentially_propagated_front = NO;
                set_current_interface(frs[0]->interf);
                for (i = num_patches-1; i >=0; i--)
                {
                    newfront_to_distri_table(newfrs[i],frs[i],
                       num_patches,redistr_table,max_n_patch);
                    free_front(newfrs[i]);
                }
                goto begin_advance_front2d;
            }
        }

        status = GOOD_STEP;
        /* Redistribution front sweep */
        for (i = 0; i < num_patches; i++)
        {
            if(frs[i]->patch_level != frs[i]->NumberOfLevels-1
                && frs[i]->patch_level != 0)
            {
                newfrs[i]->step = frs[i]->step + 1;
                newfrs[i]->time = frs[i]->time + dt;
                continue;
            }
            set_amr_intfc_tol(frs[i]->interf,
                pow(2.0,-(frs[i]->NumberOfLevels-1.0-frs[i]->patch_level)));
            set_amr_intfc_tol(newfrs[i]->interf,
                pow(2.0,-(newfrs[i]->NumberOfLevels-1.0-newfrs[i]->patch_level)));
 
            if (debugging("hyp_amr"))
                printf("redistribute patch[%d], level[%d]",
                       i,frs[i]->patch_level);

            /* Do not redistribute base grid interior curves */
            if(frs[i]->patch_level == 0 && 
               frs[i]->NumberOfLevels != 1)
                delete_interior_curves(newfrs[i]->interf); 

            status = redist_advance_front(dt,dt_frac,frs[i],&newfrs[i],
                                (POINTER)wvs[i]);

            set_amr_intfc_tol(frs[i]->interf,
                pow(2.0,frs[i]->NumberOfLevels-1.0-frs[i]->patch_level));
            set_amr_intfc_tol(newfrs[i]->interf,
                pow(2.0,newfrs[i]->NumberOfLevels-1.0-newfrs[i]->patch_level));

            if (debugging("hyp_amr"))
                printf("status = %d\n",status);
            switch(status)
            {
            case GOOD_REDISTRIBUTION:
            break;
            case MODIFY_TIME_STEP_REDISTRIBUTE:
            case BAD_REDISTRIBUTION:
                goto redis_status;
            break; 
            case UNABLE_TO_UNTANGLE:
                goto redis_status;
            break; 
            default:
            screen("ERROR: overture_hyp_amr(),"
                   " redist_advance_front() failed =%d\n", status);
            clean_up(ERROR);
            }
        }

redis_status:
        status = syncronize_redist_status(frs[0], status);
        switch(status)
        { 
        case   GOOD_REDISTRIBUTION:
            status = GOOD_STEP;
        break;
        case   UNABLE_TO_UNTANGLE:
            printf("ERROR in pseudo_advance_amr_fronts2d(), "
                      "redistribution of front failed\n"
                    "UNABLE_TO_UNTANGLE\n");
            status = ERROR_IN_STEP;
            clean_up(ERROR);  
        break; 
        case   MODIFY_TIME_STEP_REDISTRIBUTE:
        case   BAD_REDISTRIBUTION:
            if(status == BAD_REDISTRIBUTION)
                *dt_frac = Min_time_step_modification_factor(frs[0]);
            status = MODIFY_TIME_STEP;
            for (i = num_patches-1; i >=0; i--)
            {
                newfront_to_distri_table(newfrs[i],frs[i],
                   num_patches,redistr_table,max_n_patch);
                free_front(newfrs[i]);
            }
            free(newfrs); 
            *newfronts = NULL;
            printf("WARNING pseudo_advance_amr_fronts2d(), "
                 "redistribution of front return = %d\n", status);
            return status;  
        break; 
        }

        for (i = 1; i < num_patches; i++)
        { 
            Redistribution_count(frs[i]) =
                  Redistribution_count(frs[0]);
        } 

        for (i = 0; i < num_patches; i++)
        {
            if(newfrs[i]->patch_level == frs[i]->NumberOfLevels - 1 &&
               newfrs[i]->interf->modified == YES)
            {
                fineintfc_modified = YES;
                break;
            }
        }
        pp_global_lmax(&fineintfc_modified,1L);
       
        start_clock("amr_fr_scat3");
        if(fineintfc_modified == NO)
        {
            status = assembly_coarse_patch_fronts(newfrs,num_patches,
              redistr_table,max_n_patch);
            msg = "assembly_coarse_patch_fronts()";
        }
        else
        {
            status = assembly_distribute_patch_fronts(newfrs,num_patches,
              redistr_table,max_n_patch, YES);
            msg = "assembly_distribute_patch_fronts()";
        }
        stop_clock("amr_fr_scat3");

        if (pp_min_status(status) != FUNCTION_SUCCEEDED)
        {
            screen("WARNING pseudo_advance_amr_fronts2d(),"
              " failed final assembly: %s\n", msg);
            *dt_frac = Min_time_step_modification_factor(frs[0]);
            status = MODIFY_TIME_STEP;
            for (i = num_patches-1; i >=0; i--)
            {
                newfront_to_distri_table(newfrs[i],frs[i],
                   num_patches,redistr_table,max_n_patch);
                free_front(newfrs[i]);
            }
            free(newfrs);
            *newfronts = NULL;
            printf("WARNING pseudo_advance_amr_fronts2d(), "
                 "final assembly return = %d\n", status);
            DEBUG_LEAVE(pseudo_advance_amr_fronts2d)
            return status;
        }

        *newfronts = newfrs;
        DEBUG_LEAVE(pseudo_advance_amr_fronts2d)
        return GOOD_STEP;
}

LOCAL int syncronize_redist_status(
	Front  *fr,
	int    status)
{
        if (fr->pp_grid->nn > 1)
        {
            long gs;

            gs = (status == BAD_REDISTRIBUTION) ? 1 : 0;
            pp_global_lmax(&gs,1L);

            if (gs == 1)
                return BAD_REDISTRIBUTION;

            gs = (status == MODIFY_TIME_STEP_REDISTRIBUTE) ? 1 : 0;
            pp_global_lmax(&gs,1L);
            if (gs == 1)
                return MODIFY_TIME_STEP_REDISTRIBUTE;

            gs = (status == UNABLE_TO_UNTANGLE) ? 1 : 0;
            pp_global_lmax(&gs,1L);

            if (gs == 1)
                return UNABLE_TO_UNTANGLE;

            gs = (status == GOOD_REDISTRIBUTION) ? 0 : 1;
            pp_global_lmax(&gs,1L);

            if (gs == 1)
                return BAD_REDISTRIBUTION;
        }
        return status;
}

LIB_LOCAL void init_patch_states2d(
        Wave            *wave,
        Front           *front,
        INIT_DATA       *init,
        void            (*initializer)(float*,COMPONENT,Locstate,INTERFACE*,
                                       INIT_DATA*))
{
        COMPONENT       comp;
        Locstate        state;
        float           *coords;
        int             icoords[MAXD];
        int             dim = wave->rect_grid->dim;
        int             i, ix, iy;
        int             imin[MAXD], imax[MAXD];

        debug_print("init","Entered init_patch_states2d()\n");

        if (wave->sizest == 0 || initializer == NULL)
        {
            debug_print("init","Left init_patch_states2d()\n");
            return;
        }

        for (i = 0; i < dim; ++i)
        {
            imin[i] = -wave->rect_grid->lbuf[i];
            imax[i] = wave->rect_grid->gmax[i] + wave->rect_grid->ubuf[i];
        }

        for (iy = imin[1]; iy < imax[1]; ++iy)
        {
            icoords[1] = iy;
            for (ix = imin[0]; ix < imax[0]; ++ix)
            {
                icoords[0] = ix;
                coords = Rect_coords(icoords,wave);
                comp = Rect_comp(icoords,wave);
                state = Rect_state(icoords,wave);
                (*initializer)(coords,comp,state,front->interf,init);
            }
        }
        debug_print("init","Left init_patch_states2d()\n");
}

LIB_LOCAL int perform_overture_init_mesh_refinement2d(
        CompositeGrid                  *rect_over_grid,
        doubleCompositeGridFunction    *cg_function,
        CompositeGrid                  **new_cg,
        Overparam   *overparam,
        int         start_level)  /* This level is already created */
{
        int     i, j;
        int     base[MAXD], bound[MAXD];
        int     baseLevel = overparam->baseLevel;
        int     numberOfRefinementLevels = overparam->numberOfRefinementLevels;
        real    efficiency = overparam->efficiency;
        real    errorThreshold = overparam->errorThreshold;
        int     refinementRatio = overparam->refinementRatio;
        Regrid        regrid;
        Ogen          ogen;
        Range         all;
        Index         I1,I2,I3;
        int           currentGrid = 0, nextGrid;
        realArray     pos(1,3), uval(1,7);

        DEBUG_ENTER(perform_overture_init_mesh_refinement)
        if(debugging("over_init_mesh"))
        {
            printf("--------------------------------------\n");
            printf("Before start init_mesh_refine:\n");
            for(int grid = 0; grid<rect_over_grid->numberOfComponentGrids(); grid++)
            {
                getIndex((*rect_over_grid)[grid].indexRange(),I1,I2,I3);
                printf("grid = %d\n",grid);
                printf("x dir indexRange base, bond[%d,%d]\n",I1.getBase(), I1.getBound());
                printf("y dir indexRange base, bond[%d,%d]\n",I2.getBase(), I2.getBound());
                getIndex((*rect_over_grid)[grid].dimension(),I1,I2,I3);
                printf("grid = %d\n",grid);
                printf("x dir dimension base, bond[%d,%d]\n",I1.getBase(), I1.getBound());
                printf("y dir dimension base, bond[%d,%d]\n",I2.getBase(), I2.getBound());
            }
            printf("--------------------------------------\n");
        }

        CompositeGrid cga[2];
        /*
        CompositeGrid& cg = cga[0];
        cg = *rect_over_grid;
        cga[1] = cg;
        cg.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        */
        cga[0] = *rect_over_grid;
        // cga[0].update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        cga[0].update(MappedGrid::THEvertex | MappedGrid::THEmask );
        /*
        cga[1] = *rect_over_grid;
        cga[1].update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        */ 

        InterpolateRefinements interp(cga[0].numberOfDimensions());
        interp.setOrderOfInterpolation(3);  /* was 1 */

        ErrorEstimator errorEstimator(interp);
        /* set the scale factors for the solution (estimate of solution size) */
        RealArray scaleFactor(1);
        scaleFactor=1.;
        errorEstimator.setScaleFactor(scaleFactor);
        /* set the default number of smoothing steps for smoothing the error */
        errorEstimator.setDefaultNumberOfSmooths(overparam->numberOfSmooths);

        Interpolant interpolant;
        interpolant.setImplicitInterpolationMethod(Interpolant::iterateToInterpolate);
        interpolant.setInterpolateRefinements(interp);

        regrid.setEfficiency(efficiency);
        regrid.setRefinementRatio(refinementRatio);

        CompositeGridOperators op(cga[0]);
        op.updateToMatchGrid(cga[0]);
        doubleCompositeGridFunction error;
        cg_function->updateToMatchGrid(cga[0]);
        cg_function->setOperators(op);

        /* 
        for( int grid = 0; grid < cga[0].numberOfComponentGrids(); grid++)
        {
            printf("Original grid[%d] print Dens\n", grid);
            (*cg_function)[grid].display("density");
        }
        */ 

        if(numberOfRefinementLevels == 1)
        {
            (*new_cg) = new CompositeGrid();
            (**new_cg) = cga[currentGrid];
            // (**new_cg).update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
            (**new_cg).update(MappedGrid::THEvertex | MappedGrid::THEmask );

            show_grids_info(cga[currentGrid]);

            DEBUG_LEAVE(perform_overture_init_mesh_refinement)
            return (*new_cg)->numberOfComponentGrids();
        }
     
        for( int level=start_level+1; level<numberOfRefinementLevels; level++ )
        {
            nextGrid = (currentGrid+1) % 2;

            CompositeGrid& cgc = cga[currentGrid];
            CompositeGrid& cgNew = cga[nextGrid];

            error.updateToMatchGrid(cgc);
            for(int grid = 0; grid < cgc.numberOfComponentGrids(); grid++)
            {
                int errorcglevel;
                errorcglevel = (cgc).refinementLevelNumber(grid);
                if(errorcglevel == level - 1)
                {
                    getIndex(cgc[grid].dimension(), I1, I2, I3);
                    error[grid](I1, I2, I3) = 0.0;
                }  
            }

            interpolant.updateToMatchGrid(cgc,level-1);
            op.updateToMatchGrid(cgc);
            error.setOperators(op);
            printf("Error computed = %d ",
             errorEstimator.computeAndSmoothErrorFunction(*cg_function, error));
            printf("Error: min=%e, max=%e \n",min(error),max(error));

            regrid.regrid(cgc,cgNew, error, errorThreshold, level, baseLevel);
            ogen.updateRefinement(cgNew);
            // cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
            cgNew.update(MappedGrid::THEvertex | MappedGrid::THEmask );

            printf("level[%d], cgNew # of levels [%d], numberOfComponentGrids [%d]\n",
                   level,cgNew.numberOfMultigridLevels(),cgNew.numberOfComponentGrids());
            printf("cgc # of levels [%d],numberOfComponentGrids [%d]\n",
                   cgc.numberOfMultigridLevels(), cgc.numberOfComponentGrids());

            if(YES == stop_adding_amr_grids(cgc, cgNew))
            {
                currentGrid = (currentGrid+1) % 2;
                show_grids_info(cgNew);
                break;
            }

            doubleCompositeGridFunction uu2;
            uu2.updateToMatchGrid(cgNew); 
            interp.interpolateRefinements(*cg_function,uu2);

            for( int grid = 1; grid < cgNew.numberOfComponentGrids(); grid++)
            {
                int base1,base2,bound1,bound2;
                int ix,iy;
                float x0,x1,y0,y1,dx,dy;
                getIndex(cgNew[grid].indexRange(),I1,I2,I3);
                base1 = I1.getBase();
                base2 = I2.getBase();
                bound1 = I1.getBound();
                bound2 = I2.getBound();

                for (ix = base1-4; ix <= bound1+4; ix++)
                {
                    for (iy = base2-4; iy < base2; iy++)
                    {
                        pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                        pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                        pos(0,2) = 0.0;
                        interpolatePoints(pos, *cg_function, uval);
                        uu2[grid](ix,iy,0) = uval(0,0);
                        /*
                        uval.display("interp uval");
                        */
                    }
                    for (iy = bound2; iy <= bound2+4; iy++)
                    {
                        pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                        pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                        pos(0,2) = 0.0;
                        interpolatePoints(pos, *cg_function, uval);
                        uu2[grid](ix,iy,0) = uval(0,0);
                    }
                }
                for (iy = base2-4; iy <= bound2+4; iy++)
                {
                    for (ix = base1-4; ix < base1; ix++)
                    {
                        pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                        pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                        pos(0,2) = 0.0;
                        interpolatePoints(pos, *cg_function, uval);
                        uu2[grid](ix,iy,0) = uval(0,0);
                    }
                    for (ix = bound1; ix <= bound1+4; ix++)
                    {
                        pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                        pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                        pos(0,2) = 0.0;
                        interpolatePoints(pos, *cg_function, uval);
                        uu2[grid](ix,iy,0) = uval(0,0);
                    }
                }
            }
            (cg_function)->updateToMatchGrid(cgNew);
            (cg_function)->dataCopy(uu2);

            /*
            for( int grid = cgc.numberOfComponentGrids(); 
                     grid < cgNew.numberOfComponentGrids(); grid++)
            {
                printf("Updated grid[%d] level[%d] print UU Dens\n", grid,level);
                Dens[grid].display("density");
            }
            */
            printf("Regrid: Create AMR Grids on processing level[%d]\n", level);
            show_grids_info(cgNew);

            currentGrid = (currentGrid+1) % 2;
        }

        if(overparam->extra_buf != 0)
            Aug_compositegrid_by_error2d(error, cga[currentGrid], overparam);
        printf("Augment error size: New generated grid position\n");
        show_grids_info(cga[currentGrid]);

#if defined(__MPI__)
        pp_update_compositegrid2d(cga[currentGrid], error, overparam);
        printf("Parallel update subdomain boundary: New generated grid position\n");
        show_grids_info(cga[currentGrid]);
#endif /* if defined(__MPI__) */

        if(overparam->splitpatch == YES)
        {
            split_giant_compositegrid2d(cga[currentGrid], overparam);
            printf("Split Giant Composite Grid: New generated grid position\n");
            show_grids_info(cga[currentGrid]);
        }

        (*new_cg) = new CompositeGrid();
        (**new_cg) = cga[currentGrid];
        (**new_cg).update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );

        DEBUG_LEAVE(perform_overture_init_mesh_refinement)

        return (*new_cg)->numberOfComponentGrids();
}


LOCAL void split_giant_compositegrid2d(
        CompositeGrid&                cg,
	Overparam                     *overparam)
{
        int            dim = (cg).numberOfDimensions();
        int            level, refine_ratio, fine_level;
        int            i, l;
        int            bbase[MAXD], bbound[MAXD], bsize; 
        int            base[MAXD], bound[MAXD], size; 
        int            gimin[3], gimax[3];
        Rect_ComponentGrid    *r_grid;  
        Rect_ComponentGrid_Collection    r_collect; 
        Range          all;
        Index          I1, I2, I3;
        IntegerArray   range(2, 3);
        CompositeGrid  cga[2];
        int            currentGrid = 0, nextGrid;
        int            bX, bY;

        DEBUG_ENTER(split_giant_compositegrid2d)

        refine_ratio = overparam->refinementRatio; 

        cga[0] = cg;
        cga[0].update(MappedGrid::THEvertex);
        cga[0].update(CompositeGrid::THErefinementLevel);
        cga[1] = cg;
        cga[1].update(MappedGrid::THEvertex);
        cga[1].deleteRefinementLevels(0);
        cga[1].update(MappedGrid::THEvertex);
        cga[1].update(CompositeGrid::THErefinementLevel);


        getIndex((cg)[0].indexRange(),I1,I2,I3);
        bbase[0] = I1.getBase(); bbound[0] = I1.getBound();
        bbase[1] = I2.getBase(); bbound[1] = I2.getBound();
        bsize = (bbound[0] - bbase[0])*(bbound[1] - bbase[1]);
        bX = bbound[0] - bbase[0]; bY = bbound[1] - bbase[1];
        bX = overparam->splitgmax[0]; bY = overparam->splitgmax[1];
        bsize = bX*bY;
        nextGrid = (currentGrid+1) % 2;

        fine_level = (cg).refinementLevelNumber((cg).numberOfComponentGrids()-1);
        for(l = 1; l <= fine_level; l++)
        {
            nextGrid = (currentGrid+1) % 2;

            CompositeGrid& cgc = cga[currentGrid];
            CompositeGrid& cgn = cga[nextGrid];
            r_collect.numberOfComponentGrids = 0;
            r_collect.numberOfRefinementLevels = 0;
            r_collect.rect_componentgrids = NULL;
            split_patches_at_level(l, cgc, cgn, r_collect, bX, bY,overparam);
            if(l != fine_level)
                split_patches_by_line(l+1, cgc, cgn, r_collect,overparam);

            if(r_collect.numberOfComponentGrids != 0)
                free(r_collect.rect_componentgrids);

            cgc = cgn;
            cgc.update(CompositeGrid::THErefinementLevel);
            cgc.deleteRefinementLevels(l);
            cgc.update(CompositeGrid::THErefinementLevel);
            cgc.update(MappedGrid::THEvertex);
            currentGrid = (currentGrid+1) % 2;

            /*
            printf("level %d, middle split result, fine_level = %d\n", l, fine_level);
            show_grids_info(cgc);
            */
        }
        cg = cga[currentGrid];
        cg.update(CompositeGrid::THErefinementLevel);
        cg.update(MappedGrid::THEvertex);

        /*
        printf("SHOW cgNew CompositeGrid\n");
        show_grids_info(cg);
        printf("EXIT in split_giant_compositegrid2d levels = %d\n",
             overparam->numberOfRefinementLevels);
        exit(0);
        */
        DEBUG_LEAVE(split_giant_compositegrid2d)
}

LOCAL void split_patches_at_level(
        int             level,
        CompositeGrid&  cg,
        CompositeGrid&  cgc,
        Rect_ComponentGrid_Collection& r_collect,
        int             bX,
        int             bY,
        Overparam       *overparam)
{
        int            dim = (cg).numberOfDimensions();
        int            refine_ratio, fine_level;
        int            i, l;
        int            bbase[MAXD], bbound[MAXD], bsize;
        int            base[MAXD], bound[MAXD], size;
        int            gimin[3], gimax[3];
        Rect_ComponentGrid    *r_grid;
        Rect_ComponentGrid_Collection r_coltmp;
        Range          all;
        Index          I1, I2, I3;
        IntegerArray   range(2, 3);

        refine_ratio = overparam->refinementRatio;
        bsize = bX*bY;
        for(int grid = 0; grid < (cg).numberOfComponentGrids(); grid++)
        {
            if(level != (cg).refinementLevelNumber(grid))
                continue;
            getIndex((cg)[grid].indexRange(),I1,I2,I3);
            base[0] = I1.getBase(); bound[0] = I1.getBound();
            base[1] = I2.getBase(); bound[1] = I2.getBound();
            size = (bound[0] - base[0])*(bound[1] - base[1]);
            if(size > bsize)
            {
                MappedGrid& mg = cg[grid];

                r_coltmp.numberOfComponentGrids = 0;
                r_coltmp.rect_componentgrids = NULL;
                perform_split_Grid2d(mg, r_coltmp, level, bX, bY);
                r_grid = r_coltmp.rect_componentgrids;
                for(i = 0; i < r_coltmp.numberOfComponentGrids; i++)
                {
                    range(0, 0) = r_grid[i].base[0]/refine_ratio;
                    range(0, 1) = r_grid[i].base[1]/refine_ratio;
                    range(0, 2) = 0;
                    range(1, 0) = r_grid[i].bound[0]/refine_ratio;
                    range(1, 1) = r_grid[i].bound[1]/refine_ratio;
                    range(1, 2) = 0;
                    cgc.addRefinement(range, refine_ratio, level, 0);
                }
                copy_rect_grids_collection(r_collect,r_coltmp);
                free(r_coltmp.rect_componentgrids);
            }
            else
            {
                /* make a copy in cgc */
                getIndex(cg[grid].indexRange(),I1,I2,I3);
                range(0, 0) = I1.getBase()/refine_ratio;
                range(0, 1) = I2.getBase()/refine_ratio;
                range(0, 2) = 0;
                range(1, 0) = I1.getBound()/refine_ratio;
                range(1, 1) = I2.getBound()/refine_ratio;
                range(1, 2) = 0;
                cgc.addRefinement(range, refine_ratio, level, 0);
            }
        }
}

LOCAL void copy_rect_grids_collection(
        Rect_ComponentGrid_Collection& r_colnew,
        Rect_ComponentGrid_Collection& r_colold)
{
        int num, i, ii;
        Rect_ComponentGrid    *r_grid;

        num = r_colnew.numberOfComponentGrids + r_colold.numberOfComponentGrids;
        vector(&r_grid, num, sizeof(Rect_ComponentGrid));

        for(ii = 0, i = 0; i < r_colnew.numberOfComponentGrids; i++, ii++)
            assign(&r_grid[ii], &(r_colnew.rect_componentgrids[i]), sizeof(Rect_ComponentGrid));
        for(i = 0; i < r_colold.numberOfComponentGrids; i++, ii++)
            assign(&r_grid[ii], &(r_colold.rect_componentgrids[i]), sizeof(Rect_ComponentGrid));

        r_colnew.numberOfComponentGrids += r_colold.numberOfComponentGrids;
        free(r_colnew.rect_componentgrids);
        r_colnew.rect_componentgrids = r_grid;
        r_colnew.baseLevel = r_colold.baseLevel;
}

LOCAL void split_patches_by_line(
        int             level,
        CompositeGrid&  cg,
        CompositeGrid&  cgc,
        Rect_ComponentGrid_Collection& r_collect,
        Overparam       *overparam)
{
        int            i, l, fine_level;
        int            dim = (cg).numberOfDimensions();
        int            refine, refine_ratio;
        int            rbase[MAXD], rbound[MAXD];
        int            base[MAXD], bound[MAXD];
        int            ibase[MAXD], ibound[MAXD];
        int            gimin[3], gimax[3];
        Rect_ComponentGrid    *r_grid;
        Range          all;
        Index          I1, I2, I3;
        IntegerArray   range(2, 3);
        int            isect;

        /*
        print_rect_componentgrid_collection(r_collect);
        */
        refine_ratio = overparam->refinementRatio;
        if(r_collect.numberOfComponentGrids == 0)
        {
            for(int grid = 0; grid < (cg).numberOfComponentGrids(); grid++)
            {
                if((fine_level = (cg).refinementLevelNumber(grid)) <level)
                    continue;
                getIndex((cg)[grid].indexRange(),I1,I2,I3);
                base[0] = I1.getBase(); bound[0] = I1.getBound();
                base[1] = I2.getBase(); bound[1] = I2.getBound();
                range(0, 0) = base[0]/refine_ratio;
                range(0, 1) = base[1]/refine_ratio;
                range(0, 2) = 0;
                range(1, 0) = bound[0]/refine_ratio;
                range(1, 1) = bound[1]/refine_ratio;
                range(1, 2) = 0;
                cgc.addRefinement(range, refine_ratio, fine_level, 0);
            }
            return;
        }

        r_grid = r_collect.rect_componentgrids;
        for(int grid = 0; grid < (cg).numberOfComponentGrids(); grid++)
        {
            if((fine_level = (cg).refinementLevelNumber(grid)) <level)
                continue;
            refine = 1;
            for(i = level-1; i < fine_level; i++)
                refine *= overparam->refinementRatio;

            getIndex((cg)[grid].indexRange(),I1,I2,I3);
            base[0] = I1.getBase(); bound[0] = I1.getBound();
            base[1] = I2.getBase(); bound[1] = I2.getBound();
            isect = NO;

            for(i = 0; i < r_collect.numberOfComponentGrids;i++)
            {
                rbase[0] = r_grid[i].base[0]*refine;
                rbase[1] = r_grid[i].base[1]*refine;
                rbound[0] = r_grid[i].bound[0]*refine;
                rbound[1] = r_grid[i].bound[1]*refine;

                if(box_intersect(base,bound,rbase,rbound,dim,ibase,ibound))
                {
                    isect = YES;
                    range(0, 0) = ibase[0]/refine_ratio;
                    range(0, 1) = ibase[1]/refine_ratio;
                    range(0, 2) = 0;
                    range(1, 0) = ibound[0]/refine_ratio;
                    range(1, 1) = ibound[1]/refine_ratio;
                    range(1, 2) = 0;
                    cgc.addRefinement(range, refine_ratio, fine_level, 0);
                }
            }
            if(isect == NO)
            {
                range(0, 0) = base[0]/refine_ratio;
                range(0, 1) = base[1]/refine_ratio;
                range(0, 2) = 0;
                range(1, 0) = bound[0]/refine_ratio;
                range(1, 1) = bound[1]/refine_ratio;
                range(1, 2) = 0;
                cgc.addRefinement(range, refine_ratio, fine_level, 0);
            }
        }
        cgc.update(CompositeGrid::THErefinementLevel);
        cgc.update(MappedGrid::THEvertex);
}

LOCAL int perform_split_Grid2d(
	MappedGrid&    mg,
        Rect_ComponentGrid_Collection& r_collect,
        int            level,
        int            bX,
        int            bY)
{
        int            dim = (mg).numberOfDimensions();
        int            i, j, ii, num;
        int            nx, ny, lenx, leny; 
        int            cut_dir, len;
        int            base[MAXD], bound[MAXD], size; 
        int            base1[MAXD], bound1[MAXD]; 
        int            base2[MAXD], bound2[MAXD]; 
        int            gimin[3], gimax[3];
        int            bsize; 
        Rect_ComponentGrid   *r_grid[2]; 
        Range          all;
        Index          I1, I2, I3;

        bsize = bX*bY; 
        getIndex(mg.indexRange(),I1,I2,I3);
        base[0] = I1.getBase(); bound[0] = I1.getBound();
        base[1] = I2.getBase(); bound[1] = I2.getBound();

        r_collect.dim = dim;
        r_collect.baseLevel = level;
        r_collect.numberOfRefinementLevels = 0;
 
        nx = irint(1.0*(bound[0]-base[0])/bX); 
        ny = irint(1.0*(bound[1]-base[1])/bY); 

        if(nx == 0) 
            nx = 1;
        if(ny == 0) 
            ny = 1;
        
        num = nx*ny; 
        lenx = (int)(1.0*(bound[0] - base[0])/nx); 
        leny = (int)(1.0*(bound[1] - base[1])/ny); 

        if(debugging("split_grids"))
        {
            printf("Entered perform_split_Grid2d, level %d nx, ny[%d, %d]\n",
               level, nx, ny);
            printf("round: %10.8f, %10.8f\n", (1.0*(bound[0]-base[0])/bX),
             (1.0*(bound[1]-base[1])/bY));
            printf("other round: %d, %d\n", (int)(1.0*(bound[0]-base[0])/bX),
             (int)(1.0*(bound[1]-base[1])/bY));
            printf("base[%d, %d] bound[%d, %d], bX bY<%d, %d> lenx_y <%d, %d>\n",
                 base[0], base[1], bound[0], bound[1], bX, bY, lenx, leny);
        }

        vector(&(r_grid[0]), num, sizeof(Rect_ComponentGrid));        
        ii = 0; 
        for(j = 0; j < ny; j++)
        {
            for(i = 0; i < nx; i++)
            {
                base1[0] = base[0] + i*lenx; bound1[0] = base[0] + (i+1)*lenx;
                base1[1] = base[1] + j*leny; bound1[1] = base[1] + (j+1)*leny;
                if(i != 0)
                    base1[0] += base1[0]%2;   
                if(j != 0)
                    base1[1] += base1[1]%2;
                if((i+1) != nx)
                    bound1[0] += bound1[0]%2;
                if((j+1) != ny)
                    bound1[1] += bound1[1]%2;
                if((i+1) == nx)
                    bound1[0] = bound[0]; 
                if((j+1) == ny)
                    bound1[1] = bound[1]; 
                r_grid[0][ii].dim = dim;
                r_grid[0][ii].grid_level = level;
                r_grid[0][ii].base[0] = base1[0];
                r_grid[0][ii].base[1] = base1[1];
                r_grid[0][ii].bound[0] = bound1[0];
                r_grid[0][ii].bound[1] = bound1[1];
                ii++; 
            }
        }

        r_collect.rect_componentgrids = r_grid[0]; 
        r_collect.numberOfComponentGrids = num; 
        return num;
}

LIB_LOCAL void pp_update_compositegrid2d(
        CompositeGrid&                cg,
        doubleCompositeGridFunction&  error,
	Overparam                     *overparam)
{
        int            numnodes, myid;
        int            me[MAXD], i, j;
        int            side, dim = (cg).numberOfDimensions(); 
        int            levels, errorcglevel; 
        int            ix, iy;  
        int            gimin[3], gimax[3];
        Range          all;  
        Index          I1, I2, I3;
        IntegerArray   range(2, 3);  
        Rect_ComponentGrid_Collection  r_collection;
        Rect_ComponentGrid_Collection  r_collect_l[MAXD], r_collect_u[MAXD];
        Rect_ComponentGrid  *component_grid; 
 
        DEBUG_ENTER(pp_update_compositegrid2d)

        if((numnodes = pp_numnodes()) == 1) return;
        myid = pp_mynode();

        r_collection.dim = dim;
        r_collection.pc_id = myid;
        r_collection.baseLevel = overparam->baseLevel;
        r_collection.numberOfComponentGrids = cg.numberOfComponentGrids();
        r_collection.numberOfRefinementLevels = cg.numberOfRefinementLevels();

        vector(&(r_collection.rect_componentgrids), 
               r_collection.numberOfComponentGrids, sizeof(Rect_ComponentGrid));

        component_grid = r_collection.rect_componentgrids;
        for(i = 0; i < r_collection.numberOfComponentGrids; i++)
        {
            component_grid[i].dim =  dim;
            component_grid[i].grid_id = i;
            component_grid[i].grid_level = (cg).refinementLevelNumber(i);
            getIndex((cg)[i].indexRange(),I1,I2,I3); 
            component_grid[i].base[0] = I1.getBase();
            component_grid[i].base[1] = I2.getBase();
            component_grid[i].bound[0] = I1.getBound();
            component_grid[i].bound[1] = I2.getBound();
            component_grid[i].L[0] = (cg)[i].vertex()(I1.getBase(),I2.getBase(),0,axis1);
            component_grid[i].L[1] = (cg)[i].vertex()(I1.getBase(),I2.getBase(),0,axis2);
            component_grid[i].U[0] = (cg)[i].vertex()(I1.getBound(),I2.getBound(),0,axis1);
            component_grid[i].U[1] = (cg)[i].vertex()(I1.getBound(),I2.getBound(),0,axis2);
            component_grid[i].h[0] = 
                       (component_grid[i].U[0] - component_grid[i].L[0])/
                         ((float)(component_grid[i].bound[0] - component_grid[i].base[0])); 
            component_grid[i].h[1] = 
                       (component_grid[i].U[1] - component_grid[i].L[1])/
                         ((float)(component_grid[i].bound[1] - component_grid[i].base[1])); 
        } 

        find_Cartesian_coordinates(myid,pp_grid,me);

        for(i = 0; i < dim; i++)
        {
            zero_scalar(&r_collect_l[i], sizeof(Rect_ComponentGrid_Collection));
            zero_scalar(&r_collect_u[i], sizeof(Rect_ComponentGrid_Collection));
            for(side = 0; side < 2; side++)
            {
                pp_gsync();
                pp_send_grid_collection(me, i, side, &r_collection);

                if(((side+1)%2) == 0)
                    pp_receive_grid_collection(me, i, (side+1)%2, &r_collect_l[i]);
                else
                    pp_receive_grid_collection(me, i, (side+1)%2, &r_collect_u[i]);
            }
        }

        if(debugging("pp_aug_grid"))
        {
            for(i = 0; i < dim; i++)
            {
                printf("print L[%d] Rect_ComponentGrid_Collection\n", i);
                print_rect_componentgrid_collection(r_collect_l[i]); 
                printf("print U[%d] Rect_ComponentGrid_Collection\n", i);
                print_rect_componentgrid_collection(r_collect_u[i]); 
            }
        } 

        Regrid regrid; 
        Ogen   ogen;  
        CompositeGrid cgNew, cgNew2, *errorcg;
        cgNew = cg;
        cgNew.update(MappedGrid::THEvertex | MappedGrid::THEmask);

        InterpolateRefinements interp(cg.numberOfDimensions());
        interp.setOrderOfInterpolation(3);  /* was 1 */

        ErrorEstimator errorEstimator(interp);
        /* set the scale factors for the solution (estimate of solution size) */
        RealArray scaleFactor(1);
        scaleFactor=1.;
        errorEstimator.setScaleFactor(scaleFactor);
        /* set the default number of smoothing steps for smoothing the error */
        errorEstimator.setDefaultNumberOfSmooths(overparam->numberOfSmooths);

        Interpolant interpolant;
        interpolant.setImplicitInterpolationMethod(Interpolant::iterateToInterpolate);
        interpolant.setInterpolateRefinements(interp);
        CompositeGridOperators op(cg);
        doubleCompositeGridFunction errorNew;

        regrid.setEfficiency(overparam->efficiency);
        regrid.setRefinementRatio(overparam->refinementRatio);
        errorcg = error.getCompositeGrid(); 

        cgNew.deleteRefinementLevels(0);
        cgNew.update(MappedGrid::THEvertex | MappedGrid::THEmask );
        cgNew.update(CompositeGrid::THErefinementLevel);

            /* The last error function is defined on (numberOfRefinementLevels - 2) */
        for(i = 0; i < overparam->numberOfRefinementLevels - 1; i++)
        {
            getIndex(cgNew[i].indexRange(),I1,I2,I3);
            range(0, 0) = 0; range(0, 1) = 0; range(0, 2) = 0;
            range(1, 0) = I1.getBound(); 
            range(1, 1) = I2.getBound(); range(1, 2) = 0;
            cgNew.addRefinement(range, overparam->refinementRatio, i+1, 0);  
        }
        cgNew.update(CompositeGrid::THErefinementLevel);
        cgNew.update(MappedGrid::THEvertex | MappedGrid::THEmask );
 
        if(debugging("pp_aug_grid"))
        {
            printf("After manually added levels\n");  
            show_grids_info(cgNew); 
        }

        errorNew.updateToMatchGrid(cgNew);
        interpolant.updateToMatchGrid(cgNew, 1);
        op.updateToMatchGrid(cgNew);  
        errorNew.setOperators(op); 

        for(int grid = 0; grid < cgNew.numberOfComponentGrids(); grid++)
        {   
            getIndex(cgNew[grid].dimension(), I1, I2, I3); 
            errorNew[grid](I1, I2, I3) = 0.0; 
        }

        for( int grid = 0; grid < (*errorcg).numberOfComponentGrids(); grid++)
        {
            errorcglevel = (*errorcg).refinementLevelNumber(grid);
            // if(errorcglevel == overparam->numberOfRefinementLevels-1) continue; 
            // getIndex((*errorcg)[grid].dimension(), I1, I2, I3); 
            getIndex((*errorcg)[grid].indexRange(), I1, I2, I3); 
            gimin[0] = I1.getBase();
            gimin[1] = I2.getBase();
            gimax[0] = I1.getBound();
            gimax[1] = I2.getBound();

            for(iy = gimin[1]; iy <= gimax[1]; iy++)
            {
                for(ix = gimin[0]; ix <= gimax[0]; ix++)
                {
                    (errorNew)[errorcglevel](ix,iy,0) = 
                                     (error)[grid](ix,iy,0);
                }
            }
        }

        for(errorcglevel = 1; 
            errorcglevel < overparam->numberOfRefinementLevels; errorcglevel++)
        {
            for(i = 0; i < dim; i++)
            {
                for(side = 0; side < 2; side++)
                {
                    if(side == 0)
                    {
                        update_error_by_neighbor2d(errorNew, errorcglevel, i, 
                             side, r_collect_l[i], r_collection, overparam);
                    }
                    else
                    {
                        update_error_by_neighbor2d(errorNew, errorcglevel, i, 
                             side, r_collect_u[i], r_collection, overparam); 
                    }
                }
            }
        }
        /* 
        for( int grid = 0; grid < (cgNew).numberOfComponentGrids(); grid++)
        {
            printf("Updated grid[%d] print ERROR\n", grid);
            errorNew[grid].display("density");
        }
        */

        levels = overparam->numberOfRefinementLevels-1;
        regrid.regrid(cgNew,cgNew2, errorNew, overparam->errorThreshold, levels, 0);
        ogen.updateRefinement(cgNew2);
        cgNew2.update(MappedGrid::THEvertex | MappedGrid::THEmask );
        cg = cgNew2;
        cg.update(MappedGrid::THEvertex | MappedGrid::THEmask );
        /* 
        error.updateToMatchGrid(cg);
        printf("After error New update\n");  
        show_grids_info(cgNew2);
        */

        free(r_collection.rect_componentgrids);
        for(i = 0; i < dim; i++) 
        {
            if(r_collect_l[i].numberOfComponentGrids != 0)
                free(r_collect_l[i].rect_componentgrids);
            if(r_collect_u[i].numberOfComponentGrids != 0)
                free(r_collect_u[i].rect_componentgrids);
        } 
        DEBUG_LEAVE(pp_update_compositegrid2d)
}

/* level := neighbour grid level. */
LOCAL void update_error_by_neighbor2d(
        doubleCompositeGridFunction&   errorNew, 
        int                            level,
        int                            dir, 
        int                            side, 
        Rect_ComponentGrid_Collection  nb_r_collect,
        Rect_ComponentGrid_Collection  my_r_collect,  
        Overparam                      *overparam)
{
        int         i, refine, nb_level;
        int         dim, refinebase[MAXD], refinebound[MAXD]; 
        int         myrefinebase[MAXD], myrefinebound[MAXD];  
        int         ix, iy, base[MAXD], bound[MAXD]; 
        int         extra = 4;  
        int         refinementRatio = overparam->refinementRatio;
        Rect_ComponentGrid   *r_grid;
        float       errorThreshold = overparam->errorThreshold; 

        nb_level = nb_r_collect.numberOfRefinementLevels;
        if(nb_level == 1 || nb_r_collect.numberOfComponentGrids == 0) return; 

        if(debugging("pp_aug_grid"))
        {
            printf("Neighbor level[%d] Entered update_error_by_neighbor2d()"
                " dir[%d] side[%d]\n", level, dir, side); 
            /*  
            print_rect_componentgrid_collection_on_level(nb_r_collect,level);
            */
        }        

        refine = 1;
        for(i = 0; i < level; i++)
            refine *= overparam->refinementRatio; 

        dim = nb_r_collect.dim; 
        r_grid = nb_r_collect.rect_componentgrids; 
        for(i = 0; i < dim; i++)
        {
            refinebound[i] = refine * r_grid[0].bound[i];        
            refinebase[i] = refine * r_grid[0].base[i];        
        }

        r_grid = my_r_collect.rect_componentgrids; 
        for(i = 0; i < dim; i++)
        {
            myrefinebound[i] = (refine/refinementRatio) * r_grid[0].bound[i];        
            myrefinebase[i] = (refine/refinementRatio) * r_grid[0].base[i];        
        }
       
        /* for the debugging purpose */
        Range       all;
        Index       I1, I2, I3; 
        CompositeGrid *errorcg = errorNew.getCompositeGrid();

        for(i = 0; i < nb_r_collect.numberOfComponentGrids; i++)
        {
            r_grid = nb_r_collect.rect_componentgrids;
            if(r_grid[i].grid_level != level) continue; 

            /* x lower side */ 
            if(dir == 0 && side == 0)
            {
                if(r_grid[i].bound[0] == refinebound[0])
                {
                    base[0] = 0; bound[0] = extra;
                    base[1] = (int)(r_grid[i].base[1]/refinementRatio); 
                    bound[1] = (int)(r_grid[i].bound[1]/refinementRatio);       
                    for(iy = base[1]; iy < bound[1]; iy++)
                    {
                        for(ix = base[0]; ix < bound[0]; ix++)
                        {
                            errorNew[level-1](ix,iy,0) = 
                                fabs(errorNew[level-1](ix,iy,0)) + errorThreshold * 2.0;
                        } 
                    }
                }
            } 
            /* x upper side */
            if(dir == 0 && side == 1)
            {
                if(r_grid[i].base[0] == refinebase[0])
                {
                    base[0] = myrefinebound[0]-extra; 
                    bound[0] = myrefinebound[0];
                    base[1] = (int)(r_grid[i].base[1]/refinementRatio); 
                    bound[1] = (int)(r_grid[i].bound[1]/refinementRatio);       
                    for(iy = base[1]; iy < bound[1]; iy++)
                    {
                        for(ix = base[0]; ix < bound[0]; ix++)
                        {
                            errorNew[level-1](ix,iy,0) = 
                                fabs(errorNew[level-1](ix,iy,0)) + errorThreshold * 2.0;
                        } 
                    }
                }
            } 
            /* y lower side */
            if(dir == 1 && side == 0)
            {
                if(r_grid[i].bound[1] == refinebound[1])
                {
                    /* 
                    printf("Entered y lower side\n"); 
                    print_rect_componentgrid(r_grid[i]); 
                    */

                    base[1] = 0; bound[1] = extra;
                    base[0] = (int)(r_grid[i].base[0]/refinementRatio); 
                    bound[0] = (int)(r_grid[i].bound[0]/refinementRatio);       

                    /* 
                    printf("add new ERRORs on level[%d], in X<%d, %d>, Y<%d, %d>\n",
                           level-1, base[0], bound[0], base[1], bound[1]); 
                    getIndex((*errorcg)[level-1].indexRange(),I1,I2,I3);
                    printf("error function size X<%d, %d>, Y<%d, %d>, Z<%d, %d>\n", 
                         I1.getBase(),  I1.getBound(), I2.getBase(), I2.getBound(),
                         I3.getBase(),  I3.getBound());
                    */
                    for(iy = base[1]; iy < bound[1]; iy++)
                    {
                        for(ix = base[0]; ix < bound[0]; ix++)
                        {
                            (errorNew)[level-1](ix,iy,0) = 
                                fabs((errorNew)[level-1](ix,iy,0)) + errorThreshold * 2.0;
                        } 
                    }
                }
            } 
            /* y upper side */
            if(dir == 1 && side == 1)
            {
                if(r_grid[i].base[1] == refinebase[1])
                {
                    base[1] = myrefinebound[1]-extra; 
                    bound[1] = myrefinebound[1];
                    base[0] = (int)(r_grid[i].base[0]/refinementRatio); 
                    bound[0] = (int)(r_grid[i].bound[0]/refinementRatio);       

                    for(iy = base[1]; iy < bound[1]; iy++)
                    {
                        for(ix = base[0]; ix < bound[0]; ix++)
                        {
                            errorNew[level-1](ix,iy,0) = 
                                fabs(errorNew[level-1](ix,iy,0)) + errorThreshold * 2.0;
                        } 
                    }
                }
            } 
        }
}

LOCAL void print_rect_componentgrid_collection(
        Rect_ComponentGrid_Collection  r_collection)
{
        int     i; 
        Rect_ComponentGrid   *component_grid;
        component_grid = r_collection.rect_componentgrids;

        if(r_collection.numberOfComponentGrids == 0)
        {
            printf("Rect_ComponentGrid_Collection does not have"
              " Rect_ComponentGrid allocated\n");
            return; 
        } 

        printf("-------------------------------------\n"); 
        printf("NumberOfComponentGrids = %d\n", r_collection.numberOfComponentGrids); 
        printf("NumberOfRefinementLevels = %d\n", r_collection.numberOfRefinementLevels);

        for(i = 0; i < r_collection.numberOfComponentGrids; i++)
            print_rect_componentgrid(component_grid[i]);

        printf("-------------------------------------\n"); 
}

LOCAL void print_rect_componentgrid_collection_on_level(
        Rect_ComponentGrid_Collection  r_collection,
        int   level)
{
        int     i; 
        Rect_ComponentGrid   *component_grid;
        component_grid = r_collection.rect_componentgrids;

        if(r_collection.numberOfComponentGrids == 0)
        {
            printf("Rect_ComponentGrid_Collection does not have"
              " Rect_ComponentGrid allocated\n");
            return; 
        } 

        printf("-------------------------------------\n"); 
        printf("NumberOfComponentGrids = %d\n", r_collection.numberOfComponentGrids); 
        printf("NumberOfRefinementLevels = %d\n", r_collection.numberOfRefinementLevels);

        for(i = 0; i < r_collection.numberOfComponentGrids; i++)
        {
            if(component_grid[i].grid_level != level) continue;   
            print_rect_componentgrid(component_grid[i]);
        }
        printf("-------------------------------------\n"); 
}

LOCAL void pp_send_grid_collection(
        int         *me,
        int         dir,
        int         side,
        Rect_ComponentGrid_Collection  *r_collection)
{
        int           myid, dst_id;   
        int           him[MAXD], i;  
        int           numberOfComponentGrids; 
        size_t        len;
        byte          *buf; 
        byte          *storage = NULL;
        POINTER       info;

        if (rect_boundary_type(intfc,dir,side) != SUBDOMAIN_BOUNDARY)
            return;

        myid = pp_mynode();
        dst_id = neighbor_id(him, me, dir, side, pp_grid);

        if (myid == dst_id) return;

        numberOfComponentGrids = r_collection->numberOfComponentGrids;
        len = sizeof(Rect_ComponentGrid_Collection) +
               numberOfComponentGrids*sizeof(Rect_ComponentGrid);

        /*  
        printf("me[%d] dir[%d] side[%d] Send to dist[%d] numberOfComponentGrids = %d\n",
            myid, dir, side, dst_id, r_collection->numberOfComponentGrids);
        printf("allocated send buffer len(in int) = %d\n", (int)len); 
        */

        scalar(&storage, len);    
        buf = storage; 

        info = (POINTER) buf;
        assign(info, &(r_collection->pc_id), sizeof(int)); 
        buf += sizeof(int); 

        info = (POINTER) buf;
        assign(info, &(r_collection->dim), sizeof(int)); 
        buf += sizeof(int); 
 
        info = (POINTER) buf;
        assign(info, &(r_collection->baseLevel), sizeof(int)); 
        buf += sizeof(int); 

        info = (POINTER) buf;
        assign(info, &(r_collection->numberOfRefinementLevels), sizeof(int)); 
        buf += sizeof(int); 

        info = (POINTER) buf;
        assign(info, &(r_collection->numberOfComponentGrids), sizeof(int)); 
        buf += sizeof(int); 

        buf = storage + sizeof(Rect_ComponentGrid_Collection); 
        for(i = 0; i < r_collection->numberOfComponentGrids; i++) 
        {
            info = (POINTER) buf;
            assign(info, &(r_collection->rect_componentgrids[i]), 
                     sizeof(Rect_ComponentGrid)); 
            buf += sizeof(Rect_ComponentGrid); 
        }

        pp_send(0, (POINTER)(&numberOfComponentGrids), sizeof(int), dst_id); 
        pp_send(0, (POINTER)storage, len, dst_id);

        free(storage); 
}

LOCAL void pp_receive_grid_collection(
        int         *me,
        int         dir,
        int         side,
        Rect_ComponentGrid_Collection  *r_collection)
{
        int           him[MAXD], i;
        int           myid, src_id;
        int           numberOfComponentGrids;
        size_t        len;
        byte          *buf;
        byte          *storage = NULL;
        POINTER       info;

        if (rect_boundary_type(intfc,dir,side) != SUBDOMAIN_BOUNDARY)
        {
            r_collection->numberOfComponentGrids = 0; 
            return;
        }

        myid = pp_mynode();
        src_id = neighbor_id(him,me,dir,side,pp_grid);
        if (myid == src_id)
        {
            r_collection->numberOfComponentGrids = 0; 
            return; 
        }

        pp_recv(0, src_id, (POINTER)(&numberOfComponentGrids),sizeof(int));
        len = sizeof(Rect_ComponentGrid_Collection) +
              numberOfComponentGrids*sizeof(Rect_ComponentGrid);
        scalar(&storage, len);    
        pp_recv(0, src_id, (POINTER)(storage), len);
        
        /* 
        printf("me[%d] dir[%d] side[%d] received from src[%d] numberOfComponentGrids = %d\n",
            myid, dir, side, src_id, numberOfComponentGrids);
        printf("allocated received buffer len(in int) = %d\n", (int)len);        
        */

        r_collection->numberOfComponentGrids = numberOfComponentGrids;
        vector(&(r_collection->rect_componentgrids),
               numberOfComponentGrids, sizeof(Rect_ComponentGrid)); 

        buf = storage; 
        info = (POINTER) buf;
        assign(&(r_collection->pc_id), info, sizeof(int)); 
        buf += sizeof(int); 

        info = (POINTER) buf;
        assign(&(r_collection->dim), info, sizeof(int)); 
        buf += sizeof(int); 
 
        info = (POINTER) buf;
        assign(&(r_collection->baseLevel), info, sizeof(int)); 
        buf += sizeof(int); 

        info = (POINTER) buf;
        assign(&(r_collection->numberOfRefinementLevels), info, sizeof(int)); 
        buf += sizeof(int); 

        info = (POINTER) buf;
        assign(&(r_collection->numberOfComponentGrids), info, sizeof(int)); 
        buf += sizeof(int); 

        buf = storage + sizeof(Rect_ComponentGrid_Collection); 
        for(i = 0; i < numberOfComponentGrids; i++) 
        {
            info = (POINTER) buf;
            assign(&(r_collection->rect_componentgrids[i]), info, 
                     sizeof(Rect_ComponentGrid)); 
            buf += sizeof(Rect_ComponentGrid); 
        }

        free(storage); 
}

LIB_LOCAL void  Aug_compositegrid_by_error2d(
        doubleCompositeGridFunction&  error,
        CompositeGrid&      cg,
        Overparam           *overparam)
{
        int                 gimin[3], gimax[3];
        int                 imin[3], imax[3];
        int                 grid, i, ix, iy;
        int                 errorcglevel, levels; 
        float               LL[3], UU[3];
        Range               all;
        Index               I1,I2,I3;
        IntegerArray        range(2, 3); 
        Regrid              regrid;
        Ogen                ogen; 
        CompositeGrid       cgNew, cgNew2, *errorcg;
        doubleCompositeGridFunction    errorNew; 
        cgNew = cg;
        cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        cgNew2 = cg;
        // cgNew2.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        cgNew2.update(MappedGrid::THEvertex | MappedGrid::THEmask );

        DEBUG_ENTER(Aug_compositegrid_by_error2d)

        if(debugging("aug_by_error"))
        {
            printf("Entering Aug_compositegrid_by_error2d\n");
            printf("The generated CompositeGrid is:\n"); 
            show_grids_info(cg);
            printf("The CompositeGrid associated with ERROR is:\n"); 
            errorcg = error.getCompositeGrid();
            show_grids_info((*errorcg));
            printf("print ERROR function\n"); 

            for(grid = 0; grid < (*errorcg).numberOfComponentGrids(); grid++)
                (error)[grid].display("density");
            printf("End of print ERROR function\n");  
        }

        InterpolateRefinements interp(cg.numberOfDimensions());
        interp.setOrderOfInterpolation(3);  /* was 1 */

        ErrorEstimator errorEstimator(interp);
        /* set the scale factors for the solution (estimate of solution size) */
        RealArray scaleFactor(1);
        scaleFactor=1.;
        errorEstimator.setScaleFactor(scaleFactor);
        /* set the default number of smoothing steps for smoothing the error */
        errorEstimator.setDefaultNumberOfSmooths(overparam->numberOfSmooths);

        Interpolant interpolant;
        interpolant.setImplicitInterpolationMethod(Interpolant::iterateToInterpolate);
        interpolant.setInterpolateRefinements(interp);
        CompositeGridOperators op(cg);

        regrid.setEfficiency(overparam->efficiency);
        regrid.setRefinementRatio(overparam->refinementRatio);
        errorcg = error.getCompositeGrid();

        cgNew.deleteRefinementLevels(0);
        // cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask);
        cgNew.update(MappedGrid::THEvertex | MappedGrid::THEmask);
        cgNew.update(CompositeGrid::THErefinementLevel);

        for(i = 0; i < overparam->numberOfRefinementLevels - 1; i++)
        {
            getIndex(cgNew[i].indexRange(),I1,I2,I3);
            range(0, 0) = 0; range(0, 1) = 0; range(0, 2) = 0;
            range(1, 0) = I1.getBound();
            range(1, 1) = I2.getBound(); range(1, 2) = 0;
            cgNew.addRefinement(range, overparam->refinementRatio, i+1, 0);
        }
        cgNew.update(CompositeGrid::THErefinementLevel);
        // cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask);
        cgNew.update(MappedGrid::THEvertex | MappedGrid::THEmask);

        if(debugging("aug_by_error"))
        {
            printf("After manually added levels\n");
            show_grids_info(cgNew);
        }

        errorNew.updateToMatchGrid(cgNew);
        interpolant.updateToMatchGrid(cgNew, 1);
        op.updateToMatchGrid(cgNew);
        errorNew.setOperators(op);

        for(grid = 0; grid < cgNew.numberOfComponentGrids(); grid++)
        {
            getIndex(cgNew[grid].dimension(), I1, I2, I3);
            errorNew[grid](I1, I2, I3) = 0.0;
        }

        for(grid = 0; grid < (*errorcg).numberOfComponentGrids(); grid++)
        {
            errorcglevel = (*errorcg).refinementLevelNumber(grid);
            getIndex((*errorcg)[grid].dimension(), I1, I2, I3);
            gimin[0] = I1.getBase();
            gimin[1] = I2.getBase();
            gimax[0] = I1.getBound();
            gimax[1] = I2.getBound();
            getIndex((*errorcg)[grid].indexRange(),I1,I2,I3);
            imin[0] = I1.getBase();
            imin[1] = I2.getBase();
            imax[0] = I1.getBound();
            imax[1] = I2.getBound();
            for(iy = imin[1]; iy <= imax[1]; iy++)
            {
                for(ix = imin[0]; ix <= imax[0]; ix++)
                {
                    (errorNew)[errorcglevel](ix,iy,0) =
                                     (error)[grid](ix,iy,0);
                }
            }
        }

        perform_update_error2d(errorNew, cgNew, overparam);

        /* 
        for(grid = 0; grid < (*errorcg).numberOfComponentGrids(); grid++)
        {
            errorcglevel = (*errorcg).refinementLevelNumber(grid);
            getIndex((*errorcg)[grid].dimension(), I1, I2, I3);
            gimin[0] = I1.getBase();
            gimin[1] = I2.getBase();
            gimax[0] = I1.getBound();
            gimax[1] = I2.getBound();
            getIndex((*errorcg)[grid].indexRange(),I1,I2,I3);
            imin[0] = I1.getBase();
            imin[1] = I2.getBase();
            imax[0] = I1.getBound();
            imax[1] = I2.getBound();
            for(iy = gimin[1]; iy <= gimax[1]; iy++)
            {
                for(ix = gimin[0]; ix < imin[0]; ix++)
                {
                    (errorNew)[errorcglevel](ix,iy,0) =
                                     (error)[grid](ix,iy,0);
                }
                for(ix = imax[0]+1; ix <= gimax[0]; ix++)
                {
                    (errorNew)[errorcglevel](ix,iy,0) =
                                     (error)[grid](ix,iy,0);
                }
            }
            for(ix = imin[0]; ix <= imax[1]; ix++)
            {
                for(iy = gimin[1]; iy < imin[1]; iy++)
                {                
                    (errorNew)[errorcglevel](ix,iy,0) =
                                     (error)[grid](ix,iy,0);
                }
                for(iy = gimax[1]+1; iy <= gimax[1]; iy++)
                {                
                    (errorNew)[errorcglevel](ix,iy,0) =
                                     (error)[grid](ix,iy,0);
                }
            }
        }
        */

        levels = overparam->numberOfRefinementLevels-1;
        regrid.regrid(cgNew,cgNew2, errorNew, overparam->errorThreshold, 
                 levels, overparam->baseLevel);
        ogen.updateRefinement(cgNew2);
        // cgNew2.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        cgNew2.update(MappedGrid::THEvertex | MappedGrid::THEmask );
        cg = cgNew2;
        // cg.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask);         
        cg.update(MappedGrid::THEvertex | MappedGrid::THEmask);         

        error.updateToMatchGrid(cg);
        for(grid = 0; grid < (cg).numberOfComponentGrids(); grid++)
        {
            errorcglevel = (cg).refinementLevelNumber(grid);
            getIndex((cg)[grid].dimension(), I1, I2, I3);
            error[grid](I1, I2, I3) = 0.0;
            gimin[0] = I1.getBase();
            gimin[1] = I2.getBase();
            gimax[0] = I1.getBound();
            gimax[1] = I2.getBound();
            for(iy = gimin[1]; iy <= gimax[1]; iy++)
            {
                for(ix = gimin[0]; ix <= gimax[0]; ix++)
                {
                    (error)[grid](ix,iy,0) = 
                       (errorNew)[errorcglevel](ix,iy,0);
                }
            }
        }

        if(debugging("aug_by_error"))
        {
            printf("Leaving Aug_compositegrid_by_error2d\n");
            printf("The generated CompositeGrid is:\n"); 
            show_grids_info(cg);
            printf("The CompositeGrid associated with ERROR is:\n"); 
            errorcg = error.getCompositeGrid();
            show_grids_info((*errorcg));

            printf("print ERROR function\n"); 
            for(grid = 0; grid < (*errorcg).numberOfComponentGrids(); grid++)
                (error)[grid].display("density");
            printf("End of print ERROR function\n");  
        }

        DEBUG_LEAVE(Aug_compositegrid_by_error2d)
}

LOCAL void  perform_update_error2d(
        doubleCompositeGridFunction&  error,
        const CompositeGrid&      cg,
        Overparam           *overparam)
{
        int                 imin[3], imax[3];
        int                 gimin[3], gimax[3];
        int                 grid, i, ix, iy;
        int                 errorcglevel; 
        int                 seg_min, seg_max;
        int                 start, end;
        int                 set_start, set_end;
        float               LL[3], UU[3];
        Index               I1,I2,I3;
        float               tol = overparam->errorThreshold;
        int                 aug = overparam->extra_buf;

        for(grid = 0; grid < (cg).numberOfComponentGrids(); grid++)
        {
            getIndex((cg)[grid].dimension(),I1,I2,I3);
            gimin[0] = I1.getBase();
            gimin[1] = I2.getBase();
            gimax[0] = I1.getBound();
            gimax[1] = I2.getBound();
            getIndex((cg)[grid].indexRange(),I1,I2,I3);
            imin[0] = I1.getBase();
            imin[1] = I2.getBase();
            imax[0] = I1.getBound();
            imax[1] = I2.getBound();

            /* x-sweep, error augment */
            for(iy = imin[1]; iy <= imax[1]; iy++)
            {
                seg_min = imin[0];
                /* The end point is not processed,
                 * guess not important now.
                 */
                while(seg_min < imax[0])
                {
                    set_start = NO;
                    set_end = NO;
                    for(seg_max = seg_min; seg_max <= imax[0]; seg_max++)
                    {
                        if( fabs((error)[grid](seg_max,iy,0)) > tol &&
                            (! set_start))
                        {
                            start = seg_max;
                            set_start = YES;
                        }
                        if(set_start == YES &&
                           fabs((error)[grid](seg_max,iy,0)) < tol)
                        {
                            end = seg_max;
                            set_end = YES;
                            break;
                        }
                    }

                    if(set_start == YES)
                    {
                        if((start - aug) < gimin[0])
                            i = gimin[0];
                        else
                            i = start - aug;
                        for(int tmpi = i; tmpi < start; tmpi++)
                            (error)[grid](tmpi,iy,0) = 
                                fabs((error)[grid](tmpi,iy,0)) + tol * 2.0;
                        /*
                        printf("x start augment is [%d,%d]\n", start, iy);
                        */
                    }
                    if(set_end == YES)
                    {
                        if((end + aug) > gimax[0])
                            i = gimax[0];
                        else
                            i = end + aug;
                        for(int tmpi = end+1; tmpi <= i; tmpi++)
                            (error)[grid](tmpi,iy,0) = 
                                fabs((error)[grid](tmpi,iy,0)) + tol * 2.0;
                        seg_max += (aug + 1);
                        /*
                        printf("x end augment is [%d,%d]\n", end, iy);
                        */
                    }
                    seg_min = seg_max;
                }
            }

            /* y-sweep, error augment */
            for(ix = imin[0]; ix <= imax[0]; ix++)
            {
                seg_min = imin[1];
                while(seg_min < imax[1])
                {
                    set_start = NO;
                    set_end = NO;
                    for(seg_max = seg_min; seg_max <= imax[1]; seg_max++)
                    {
                        if( fabs((error)[grid](ix,seg_max,0)) > tol &&
                            (! set_start))
                        {
                            start = seg_max;
                            set_start = YES;
                        }
                        if(set_start == YES &&
                           fabs((error)[grid](ix,seg_max,0)) < tol)
                        {
                            end = seg_max;
                            set_end = YES;
                            break;
                        }
                    }

                    if(set_start == YES)
                    {
                        if((start - aug) < gimin[1])
                            i = gimin[1];
                        else
                            i = start - aug;
                        for(int tmpi = i; tmpi < start; tmpi++)
                            (error)[grid](ix,tmpi,0) = 
                                fabs((error)[grid](tmpi,iy,0)) + tol * 2.0;
                        /* 
                        printf("grid[%d] y start augment is [%d,%d]\n", grid, ix, start);
                        */
                    }
                    if(set_end == YES)
                    {
                        if((end + aug) > gimax[1])
                            i = gimax[1];
                        else
                            i = end + aug;
                        for(int tmpi = end+1; tmpi <= i; tmpi++)
                            (error)[grid](ix,tmpi,0) = 
                                fabs((error)[grid](tmpi,iy,0)) + tol * 2.0;
                        seg_max += (aug + 1);
                        /* 
                        printf("grid[%d] y end augment is [%d,%d]\n", grid, ix, end);
                        */
                    }
                    seg_min = seg_max;
                }
            }
        }
}


/* We choose to use density as bases where the
 * error estimate builds on.
 */
/* Overture should not be responsible for the
 * interpolation of states between old set of patches and
 * newly created set of patches.
 * THis is because the overture does not know antything about the front
 * the FrontTier should take care of this task.
 * However, to see how the Overture does the interpolation,
 * see before AMR_src.011903.tar.gz.
 */
LIB_LOCAL   int overture_perform_regrid2d(
        CHART           *chart,
        Printplot       *prt,
        CompositeGrid                  *rect_over_grid,
        doubleCompositeGridFunction    *rect_over_function,
        CompositeGrid                  **new_cg,
        doubleCompositeGridFunction    **new_cg_function,
        int                            *level_array)
{
        int                             i, j;
        Range                           all;
        Regrid                          regrid;
        CompositeGrid                   cga[2]; //cga old and new grids(ring buffer)
        Overparam                       *overparam;
        int                             levels, num_float, refined_levels;
        int                             interp_level; 
        size_t                          sizest;
        doubleCompositeGridFunction     uu2, Dens, error;
        Ogen                            ogen;
        Index                           I1,I2,I3;
        int                             currentGrid = 0, nextGrid = 1;

        DEBUG_ENTER(overture_perform_regrid)

        sizest = chart->wave->sizest;
        overparam = chart->overparam;
        levels = overparam->numberOfRefinementLevels;
        real errorThreshold = overparam->errorThreshold; // error tolerance for regridding.
        refined_levels = (rect_over_grid)->refinementLevelNumber(
                          rect_over_grid->numberOfComponentGrids()-1);
    
        printf(" ------------------------------------------------------------ \n");
        printf("  Adaptive Mesh Refinement regrid for FronTier Solver        \n");
        printf(" ------------------------------------------------------------ \n");

        cga[0] = *rect_over_grid;
        cga[0].update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        /*
        cga[1] = *rect_over_grid;
        cga[1].update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        */

        /* interpolator for refinements  */
        InterpolateRefinements interp(cga[0].numberOfDimensions());
        interp.setOrderOfInterpolation(3); /* used to be 1 */

        /* used to compute error estimates */
        ErrorEstimator errorEstimator(interp);
        /* set the scale factors for the solution (estimate of solution size) */
        RealArray scaleFactor(1);
        scaleFactor=1.;
        errorEstimator.setScaleFactor(scaleFactor);
        /* set the default number of smoothing steps for smoothing the error */
        errorEstimator.setDefaultNumberOfSmooths(overparam->numberOfSmooths);

        /* overlapping grid interpolator */
        Interpolant interpolant;
        interpolant.setImplicitInterpolationMethod(Interpolant::iterateToInterpolate);
        interpolant.setInterpolateRefinements(interp);

        /* compute an error estimate of density. */

        CompositeGridOperators op(cga[0]);
        op.updateToMatchGrid(cga[0]);
        error.updateToMatchGrid(cga[0]);
        error.setOperators(op);

        (*rect_over_function).updateToMatchGrid(cga[0]);
        (*rect_over_function).setOperators(op);

        /*
        for( int grid = 0; grid < cga[0].numberOfComponentGrids(); grid++)
        {
            printf("Original grid[%d] print Dens\n", grid);
            (*rect_over_function)[grid].display("density");
        }
        */

        if(levels == 1)
        {
            (*new_cg) = new CompositeGrid();
            (**new_cg) = cga[currentGrid];
            (**new_cg).update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );

            show_grids_info(cga[currentGrid]);
            printf("---------------------------------------------------------- \n");
            printf(" END Adaptive Mesh Refinement regrid for FronTier Solver  \n");
            printf("---------------------------------------------------------- \n");

            DEBUG_LEAVE(perform_overture_init_mesh_refinement)
            return (*new_cg)->numberOfComponentGrids();
        }

        if(refined_levels != 0) 
            interp_level = 1;
        else 
            interp_level = 0; 
        interpolant.updateToMatchGrid(cga[0], interp_level);  // error estimator needs to interpolate

        for(int grid = 0; grid < (cga[0]).numberOfComponentGrids(); grid++)
        {
            getIndex((cga[0])[grid].dimension(), I1, I2, I3);
            error[grid](I1, I2, I3) = 0.0;
        }             
        
        if( refined_levels == (levels - 1))
        {
            if(YES == chart->wave->use_overture_state)
            {
                Dens.updateToMatchGrid(cga[0]);
                Dens.setOperators(op);
                copy_gridfunction_from_uu1(*rect_over_function, cga[0], 0, Dens);
                errorEstimator.computeAndSmoothErrorFunction(Dens, error);
            }
            else
            {
                printf("Error computed = %d ",
                errorEstimator.computeAndSmoothErrorFunction(*rect_over_function, error));
            }
            printf("Error: min=%e, max=%e\n",min(error),max(error));

            if(overparam->extra_buf != 0)
            {
                Aug_compositegrid_by_error2d(error, cga[0], overparam);
                currentGrid = 0;  
                printf("Regrid: Create AMR Grids\n");
                printf("Augment error size: New generated grid position\n");
                show_grids_info(cga[0]);
            }
            else
            {
                CompositeGrid& cgNew = cga[1];

                regrid.setEfficiency(overparam->efficiency);
                regrid.setRefinementRatio(overparam->refinementRatio);
                regrid.regrid(cga[0],cgNew, error, errorThreshold,
                    overparam->numberOfRefinementLevels-1, overparam->baseLevel);
                regrid.printStatistics(cgNew);
                ogen.updateRefinement(cgNew);
                cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask);
                currentGrid = 1; 

                if(debugging("regrid"))
                {
                    /*
                    for( int grid = 0; grid < cgcur.numberOfComponentGrids(); grid++)
                    {
                        printf("old grid[%d] print dens\n", grid);
                        Dens[grid].display("density");
                    }
                    */
                    printf("New generated grid position\n");
                    show_grids_info(cgNew);
                    printf("Old grid position\n");
                    show_grids_info(cga[0]);

                    printf("Generated total new patch numbers =%d\n", cgNew.numberOfComponentGrids());
                    printf("numberOfRefinementLevels =%d\n", overparam->numberOfRefinementLevels);
                }
                printf("Regrid: Create AMR Grids\n");
                show_grids_info(cgNew);
            }
        }
        else /* refined_levels != (levels - 1) */
        {
            realArray pos(1,3), uval(1,7);

            /*
            printf("Show not full levels cgc before regridding level[%d]\n", 
                    refined_levels); 
            show_grids_info(cga[0]);
            */

            for(int level = refined_levels+1; level < levels; level++ )
            {
                nextGrid = (currentGrid+1) % 2;
                CompositeGrid& cgc = cga[currentGrid];
                CompositeGrid& cgNew = cga[nextGrid];

                error.updateToMatchGrid(cgc); 
                for(int grid = 0; grid < cgc.numberOfComponentGrids(); grid++)
                {
                    int errorcglevel;
                    errorcglevel = (cgc).refinementLevelNumber(grid);
                    if(errorcglevel == level - 1)
                    {
                        getIndex(cgc[grid].dimension(), I1, I2, I3);
                        error[grid](I1, I2, I3) = 0.0;
                    }
                }
                interpolant.updateToMatchGrid(cgc, interp_level);
                interp_level = level - 1;  
                op.updateToMatchGrid(cgc);

                error.setOperators(op);
                printf("Error computed = %d ",
                 errorEstimator.computeAndSmoothErrorFunction(*rect_over_function, error));
                printf("Error: min=%e, max=%e \n",min(error),max(error));

                regrid.setEfficiency(overparam->efficiency);
                regrid.setRefinementRatio(overparam->refinementRatio);
                regrid.regrid(cgc,cgNew, error, errorThreshold,
                    level, overparam->baseLevel);
                regrid.printStatistics(cgNew);
                ogen.updateRefinement(cgNew);
                cgNew.update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask);
   
                if(cgNew.refinementLevelNumber(cgNew.numberOfComponentGrids()-1)
                    == levels - 1)
                {
                     currentGrid = (currentGrid+1) % 2;
                     break; 
                } 
                if(YES == stop_adding_amr_grids(cgc, cgNew))
                {
                    currentGrid = (currentGrid+1) % 2; 
                    show_grids_info(cgc);
                    break; 
                }
                else
                {
                    doubleCompositeGridFunction uu2;
                    uu2.updateToMatchGrid(cgNew);
                    show_grids_info(cgNew);

                    // printf("prepare update grid functions for next step regridding 1\n");
                    interp.interpolateRefinements(*rect_over_function, uu2);
                    // printf("update grid functions for next step regridding 1\n");

                    for(int grid = 1; grid < cgNew.numberOfComponentGrids(); grid++)
                    {
                        int base1,base2,bound1,bound2;
                        int ix,iy;
                        float x0,x1,y0,y1,dx,dy;
                        getIndex(cgNew[grid].indexRange(),I1,I2,I3);
                        base1 = I1.getBase();
                        base2 = I2.getBase();
                        bound1 = I1.getBound();
                        bound2 = I2.getBound();

                        for (ix = base1-4; ix <= bound1+4; ix++)
                        {
                            for (iy = base2-4; iy < base2; iy++)
                            {
                                pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                                pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                                pos(0,2) = 0.0;
                                interpolatePoints(pos, *rect_over_function, uval);
                                uu2[grid](ix,iy,0) = uval(0,0);
                            }
                            for (iy = bound2; iy <= bound2+4; iy++)
                            {
                                pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                                pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                                pos(0,2) = 0.0;
                                interpolatePoints(pos, *rect_over_function, uval);
                                uu2[grid](ix,iy,0) = uval(0,0);
                            }
                        }
                        for (iy = base2-4; iy <= bound2+4; iy++)
                        {
                            for (ix = base1-4; ix < base1; ix++)
                            {
                                pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                                pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                                pos(0,2) = 0.0;
                                interpolatePoints(pos, *rect_over_function, uval);
                                uu2[grid](ix,iy,0) = uval(0,0);
                            }
                            for (ix = bound1; ix <= bound1+4; ix++)
                            {
                                pos(0,0) = cgNew[grid].vertex()(ix,iy,0,axis1);
                                pos(0,1) = cgNew[grid].vertex()(ix,iy,0,axis2);
                                pos(0,2) = 0.0;
                                interpolatePoints(pos, *rect_over_function, uval);
                                uu2[grid](ix,iy,0) = uval(0,0);
                            }
                        }
                    }
                    // printf("update grid functions for next step regridding 2\n");
                    (*rect_over_function).updateToMatchGrid(cgNew);
                    (*rect_over_function).dataCopy(uu2);
                    currentGrid = (currentGrid+1) % 2;
                }
            }
            if(overparam->extra_buf != 0)
            {
                Aug_compositegrid_by_error2d(error, cga[currentGrid], overparam);
                printf("Augment error size: New generated grid position\n");
                show_grids_info(cga[currentGrid]);
            }

        } /* end of else refined_levels != (levels - 1) */ 

#if defined(__MPI__)        
        pp_update_compositegrid2d(cga[currentGrid], error, overparam);
        printf("Parallel update subdomain boundary: New generated grid position\n"); 
        show_grids_info(cga[currentGrid]);
#endif /* if defined(__MPI__) */

        if(overparam->splitpatch == YES)
        {
            split_giant_compositegrid2d(cga[currentGrid], overparam);
            printf("Split Giant Composite Grid: New generated grid position\n");
            show_grids_info(cga[currentGrid]);
        }

        *new_cg = new CompositeGrid();
        **new_cg = cga[currentGrid];
        // (*new_cg)->update(MappedGrid::THEvertex | MappedGrid::THEcenter | MappedGrid::THEmask );
        (*new_cg)->update(MappedGrid::THEvertex | MappedGrid::THEmask);

        if(YES==chart->wave->use_overture_state)
        {
            *new_cg_function = new
              doubleCompositeGridFunction(**new_cg,sizest/sizeof(float)+1,all,all,all);
        }
        if(debugging("regrid"))
        {
            printf("In overture_regrid()\n");
            printf("Generated new cg_grid = %d\n", *new_cg);
            printf("Generated new cg_over_function = %d\n",*new_cg_function);
            printf("print new_cg->numberOfComponentGrids() = %d\n",
                (*new_cg)->numberOfComponentGrids());
        }

        /*
        (*new_cg)->refinementLevelNumber.display("test integerarray");
        */

        printf(" ------------------------------------------------------------------- \n");
        printf("  NumberOfComponentGrids = %d, RefinementLevelNumber = %d\n",
                 (*new_cg)->numberOfComponentGrids(),
                 (*new_cg)->refinementLevelNumber((*new_cg)->numberOfComponentGrids()-1)+1);
        for(j = 0; j < (*new_cg)->numberOfComponentGrids(); j++)
            printf("  refine patch [%d], level = %d\n", j,
                    (*new_cg)->refinementLevelNumber(j));

        printf("---------------------------------------------------------- \n");
        printf(" END Adaptive Mesh Refinement regrid for FronTier Solver  \n");
        printf("---------------------------------------------------------- \n");

        DEBUG_LEAVE(overture_perform_regrid)
        return  (*new_cg)->numberOfComponentGrids();
}

LOCAL void copy_gridfunction_from_uu1(
        const doubleCompositeGridFunction&  uu1,
        const CompositeGrid&                cg,
        int                                 R,
        doubleCompositeGridFunction&        u)
{
        Index I1,I2,I3;
        for( int grid=0; grid<cg.numberOfComponentGrids(); grid++)
        {
            int base1,base2,bound1,bound2;
            getIndex(cg[grid].dimension(),I1,I2,I3);
            base1 = I1.getBase();
            base2 = I2.getBase();
            bound1 = I1.getBound();
            bound2 = I2.getBound();
            int ix,iy;
            for (ix = base1; ix <= bound1; ix++)
            {
                for (iy = base2; iy <= bound2; iy++)
                {
                    u[grid](ix,iy,0) = uu1[grid](R,ix,iy,0);
                }
            }
        }
}

#endif /* if defined(TWOD) */

EXPORT void save_overture_show_files(
        Printplot    *prt,  
        Wave         *wave,
        int          step)
{

        char         *runname;
        doubleCompositeGridFunction *uu1;
        CompositeGrid *cg;
        Range        all;
        char         buf[100];
        int          myid; 
        size_t  sizest = wave->sizest; 
    
        debug_print("driver","Entering save_overture_show_files()\n");

        myid = pp_mynode();  
        runname = basename(prt->outfile);  
        sprintf(buf,"%s/%s_%s.%s.%d.ts%d",runname,"over",runname,"den",myid,step);

        if (create_directory(runname,YES) == FUNCTION_FAILED)
        {
            (void) printf("WARNING in save_overture_show_files(), directory "
                         "%s doesn't exist and can't be created\n",runname);
            return;
        }

        cg = (CompositeGrid*)wave->cg_over; 
        Ogshow show(buf);
        show.saveGeneralComment("The Density solution");

        if(wave->use_overture_state)
        {
            uu1 = (doubleCompositeGridFunction*)wave->cg_over_function;

            CompositeGridOperators op(*cg);

            // define solution and operators:
            doubleCompositeGridFunction Dens , Eng;
            doubleCompositeGridFunction Mom0, Mom1, Mom2;
            Dens.updateToMatchGrid(*cg);
            Eng.updateToMatchGrid(*cg);
            Mom0.updateToMatchGrid(*cg);
            Mom1.updateToMatchGrid(*cg);
            Mom2.updateToMatchGrid(*cg);

            copy_gridfunction_from_uu1(*uu1, *cg, 0, Dens);
           // copy_gridfunction_from_uu1(*uu1, *cg, 1, Eng);

            show.startFrame();
            //show.saveSolution(Eng);
            show.saveSolution(Dens);
        }
        else
        {
            doubleCompositeGridFunction Dens;
            Dens.updateToMatchGrid(*cg);  
        }  

        debug_print("driver","Leaving save_overture_show_files()\n");
}   /*  end save_overture_show_files()  */

#endif /* if defined(USE_OVERTURE)  */  
