//hexagon texture shading node.
//partially totally stolen from cellShader (which isn't what you might think)

#include <string>

#include <maya/MPxNode.h> 

#include <maya/MIOStream.h>

#include <maya/MString.h> 
#include <maya/MTypeId.h> 
#include <maya/MPlug.h>
#include <maya/MFloatVector.h>

#include <maya/MFnNumericAttribute.h>

#include <maya/MFnPlugin.h>

#include <maya/MDataBlock.h>
#include <maya/MDataHandle.h>

#include <cmath>

class hexTex : public MPxNode {
public:
	hexTex();
	virtual void postConstructor();
	virtual ~hexTex();

	virtual MStatus compute( const MPlug &plug, MDataBlock &data );

	static void * creator();
	static MStatus initialize();

	static MTypeId id;

private:
	//these are optional inputs
	static MObject scale;
	
	static MObject inColor0;
	static MObject inColor1;
	static MObject inColor2;

	//these are inputs with meaning to maya:
	static MObject uvCoord;

	//these are outputs with meaning to maya:
	static MObject outColor;
};

//unique id.
MTypeId hexTex::id( 0x70b337 );

MObject hexTex::scale;

MObject hexTex::inColor0;
MObject hexTex::inColor1;
MObject hexTex::inColor2;
MObject hexTex::uvCoord;
MObject hexTex::outColor;

hexTex::hexTex() {
}

void hexTex::postConstructor() {
	//tell maya that this node can be run on multiple processors at once.
	setMPSafe(true);
}

hexTex::~hexTex() {
}

void *hexTex::creator() {
	return new hexTex();
}

MStatus hexTex::initialize() {
	MFnNumericAttribute nAttr;
	MStatus stat;

	scale = nAttr.create("scale", "s", MFnNumericData::kFloat, 1.0);
	CHECK_MSTATUS( nAttr.setKeyable(true) );
	CHECK_MSTATUS( nAttr.setStorable(true) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(true) );

	inColor0 = nAttr.createColor("inColor0", "c0");
	CHECK_MSTATUS( nAttr.setDefault(1.0f,1.0f,0.5f) );
	CHECK_MSTATUS( nAttr.setKeyable(true) );
	CHECK_MSTATUS( nAttr.setStorable(true) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(true) );
	
	inColor1 = nAttr.createColor("inColor1", "c1");
	CHECK_MSTATUS( nAttr.setDefault(1.0f,0.5f,1.0f) );
	CHECK_MSTATUS( nAttr.setKeyable(true) );
	CHECK_MSTATUS( nAttr.setStorable(true) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(true) );

	inColor2 = nAttr.createColor("inColor2", "c2");
	CHECK_MSTATUS( nAttr.setDefault(0.5f,0.5f,1.0f) );
	CHECK_MSTATUS( nAttr.setKeyable(true) );
	CHECK_MSTATUS( nAttr.setStorable(true) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(true) );

	//There is no one-shot uv creator, but...
	MObject u = nAttr.create( "uCoord", "u", MFnNumericData::kFloat, 0.0);
	MObject v = nAttr.create( "vCoord", "v", MFnNumericData::kFloat, 0.0);
	
	uvCoord = nAttr.create("uvCoord", "uv", u, v);

	CHECK_MSTATUS( nAttr.setKeyable(true) );
	CHECK_MSTATUS( nAttr.setStorable(true) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(true) );
	CHECK_MSTATUS( nAttr.setHidden(true) );

	//We actually ignore this input. Completely.
	MObject ufx = nAttr.create( "uvFilterSizeX", "ufx", MFnNumericData::kFloat, 0.0);
	MObject ufy = nAttr.create( "uvFilterSizeY", "ufy", MFnNumericData::kFloat, 0.0);

	MObject uvFilterSize = nAttr.create("uvFilterSize", "uf", ufx, ufy);

	CHECK_MSTATUS( nAttr.setKeyable(true) );
	CHECK_MSTATUS( nAttr.setStorable(true) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(true) );
	CHECK_MSTATUS( nAttr.setHidden(true) );

	outColor = nAttr.createColor("outColor", "oc");
	CHECK_MSTATUS( nAttr.setKeyable(false) );
	CHECK_MSTATUS( nAttr.setStorable(false) );
	CHECK_MSTATUS( nAttr.setReadable(true) );
	CHECK_MSTATUS( nAttr.setWritable(false) );

	CHECK_MSTATUS( addAttribute(scale) );
	CHECK_MSTATUS( addAttribute(inColor0) );
	CHECK_MSTATUS( addAttribute(inColor1) );
	CHECK_MSTATUS( addAttribute(inColor2) );
	CHECK_MSTATUS( addAttribute(uvCoord) );
	CHECK_MSTATUS( addAttribute(uvFilterSize) );
	CHECK_MSTATUS( addAttribute(outColor) );

	//everything affects output color. (except filtersize, which is ignored)
	CHECK_MSTATUS( attributeAffects( scale, outColor ) );
	CHECK_MSTATUS( attributeAffects( inColor0, outColor ) );
	CHECK_MSTATUS( attributeAffects( inColor1, outColor ) );
	CHECK_MSTATUS( attributeAffects( inColor2, outColor ) );
	CHECK_MSTATUS( attributeAffects( uvCoord, outColor ) );

	return MS::kSuccess;
}

MStatus hexTex::compute(const MPlug &plug, MDataBlock &data) {
	MStatus stat;
	if (plug != outColor) {
		return MS::kUnknownParameter;
	}

	MDataHandle scaleData = data.inputValue(scale, &stat);
	CHECK_MSTATUS( stat );
	float scale = scaleData.asFloat();
	scale *= 2.0f * sin(M_PI / 3.0f);

	//I use FloatVectors instead of Float3s since FloatVectors are
	//assignable easilly.
	MDataHandle inColor0Data = data.inputValue(inColor0, &stat);
	CHECK_MSTATUS( stat );
	const MFloatVector &color0 = inColor0Data.asFloatVector();
	
	MDataHandle inColor1Data = data.inputValue(inColor1, &stat);
	CHECK_MSTATUS( stat );
	const MFloatVector &color1 = inColor1Data.asFloatVector();
	
	MDataHandle inColor2Data = data.inputValue(inColor2, &stat);
	CHECK_MSTATUS( stat );
	const MFloatVector &color2 = inColor2Data.asFloatVector();
	
	MDataHandle uvCoordData = data.inputValue(uvCoord, &stat);
	CHECK_MSTATUS( stat );
	const float2 &uv = uvCoordData.asFloat2();
	
	MDataHandle outColorData = data.outputValue( outColor, &stat );
	CHECK_MSTATUS( stat );
	MFloatVector &outcolor = outColorData.asFloatVector();


	float half_height = sin(M_PI/3.0f);
	float border = cos(M_PI/3.0f);

	//picture the tiling as columns of hexagons.
	//(including centers and right ends only -- so a stack of boxes and a
	// zigzag bit to the right)
	float col_width = 1.0f + border;

	//we'll translate each column over to column zero and figure out what
	//row it ends up in.

	int col = int(uv[0] * scale / col_width);
	float col_ofs = uv[0] * scale - col * col_width;
	if (col_ofs < 0) {
		col_ofs += col_width;
		col -= 1;
	}

	float translated_height = uv[1] * scale - col * 3 * half_height;
	int row = int(translated_height / (half_height * 2));
	float row_ofs = translated_height - row * half_height * 2;
	if (row_ofs < 0) {
		row_ofs += half_height * 2;
		row -= 1;
	}

	//now we know that we are in the first column at row 'row', offset to
	//the right by col_ofs and offset up by row_ofs.

	//start out with the basic row <-> color thing.
	int color = row;

	//if we are in the zig-zaggy region...
	if (col_ofs > 1.0f) {
		if (row_ofs > half_height) {
			if ( (col_width - col_ofs) * half_height < border * (row_ofs - half_height)) {
				color += 2;
			}
		} else {
			if ( (col_ofs - 1.0f) * half_height > border * row_ofs ) {
				color += 1;
			}
		}
	}

	color = color % 3;
	if (color < 0) color += 3;

	if (color == 0) {
		outcolor = color0;
	} else if (color == 1) {
		outcolor = color1;
	} else {
		outcolor = color2;
	}

	//now we've recomputed the outColor plug:
	outColorData.setClean();
	
	return MS::kSuccess;
}

//Now housekeeping stuff:

MStatus initializePlugin( MObject obj ) {
	const MString UserClassify( "texture/2d" );
	MStatus status;
	MFnPlugin plugin( obj, "TCHOW", "1.0", "Any" );
	status = plugin.registerNode( "hexTex", hexTex::id, hexTex::creator, hexTex::initialize, MPxNode::kDependNode, &UserClassify );
	if (status != MS::kSuccess) {
		status.perror("registerNode");
	}
	return status;
}

MStatus uninitializePlugin( MObject obj ) {
	MStatus status;
	MFnPlugin plugin( obj );
	status = plugin.deregisterNode( hexTex::id );
	if (status != MS::kSuccess) {
		status.perror("deregisterNode");
	}
	return status;
}
