#include "Skeleton.hpp"

namespace Library {

namespace {

void unroll(vector< vector< int > > const &list, vector< int > &target, int row = 0) {
	assert(row >= 0);
	target.push_back(row);
	if (row < (signed)list.size()) {
		for (unsigned int i = 0; i < list[row].size(); ++i) {
			unroll(list, target, list[row][i]+1);
		}
	}
}

Quatd ordered_rotation(string const &order, Vector3d const &rotation);

}

bool Skeleton::check_parse() {
	if (in_bone) {
		cerr << "Bone without closing tag." << endl;
		return false;
	}
	// might as well also set bones up for painless traversal.
	vector< vector< int > > children;
	for (unsigned int b = 0; b < bones.size(); ++b) {
		int list = bones[b].parent + 1; //to make root zero.
		if (list < 0) {
			cerr << "Bone without a valid parent." << endl;
			return false;
		}
		while (children.size() <= (unsigned)list) {
			children.push_back(vector< int >());
		}
		assert(list >= 0 && list < (signed)children.size());
		children[list].push_back(b);
	}

	vector< int > new_inds;
	unroll(children, new_inds);

	vector< int > old_to_new_inds;
	old_to_new_inds.resize(new_inds.size(), -2);
	assert(new_inds.size() == bones.size() + 1);

	old_to_new_inds[0] = -1;

	frame_size = 6; //i.e. just root info.
	vector< Bone > new_bones;
	for (unsigned int i = 1; i < new_inds.size(); ++i) {
		int b = new_inds[i] - 1; //bone for new first index.
		assert(b >= 0 && b < (signed)bones.size());
		old_to_new_inds[b+1] = i - 1;
		new_bones.push_back(bones[b]);
		new_bones.back().frame_offset = frame_size;
		frame_size += new_bones.back().dof.size();

		int op = new_bones.back().parent + 1;
		assert(op >= 0 && op < (signed)old_to_new_inds.size());
		int p = old_to_new_inds[op]; //look up parent.
		assert(p >= -1 && p < (signed)bones.size());
		new_bones.back().parent = p;
	}

	//bones now ordered for easy traversal.
	bones = new_bones;

	for (unsigned int b = 0; b < bones.size(); ++b) {
		bones[b].global_to_local = ordered_rotation(bones[b].offset_order, bones[b].axis_offset);
	}

	return true;
}

namespace {
Quatd ordered_rotation(string const &order, Vector3d const &rot) {
	Quatd ret;
	ret.clear();
	for (unsigned int i = 0; i < order.size(); ++i) {
		switch (order[i]) {
			case 'x':
				ret = multiply(rotation(rot.x * M_PI / 180.0, make_vector(1.0,0.0,0.0)), ret);
				break;
			case 'y':
				ret = multiply(rotation(rot.y * M_PI / 180.0, make_vector(0.0,1.0,0.0)), ret);
				break;
			case 'z':
				ret = multiply(rotation(rot.z * M_PI / 180.0, make_vector(0.0,0.0,1.0)), ret);
				break;
			case 'X':
			case 'Y':
			case 'Z':
			case 'l':
			break;
			default:
				cerr << "Unexpected offset_order character '" << order[i] << "' in '" << order << "'." << endl;
				assert(0);
		}
	}
	return normalize(ret);
}

Vector3d get_dof_trans(string const &dof, double const *info, int start_pos) {
	Vector3d trans;
	trans.x = trans.y = trans.z = 0;
	for (unsigned int i = 0; i != dof.size(); ++i) {
		double d = info[start_pos + i];
		switch (dof[i]) {
			case 'x':
			case 'y':
			case 'z':
				break;
 			case 'X':
				trans.x = d;
				break;
 			case 'Y':
				trans.y = d;
				break;
 			case 'Z':
				trans.z = d;
				break;
 			case 'l':
				cerr << "Currently, we're lazy about length." << endl;
				break;
 			default:
				cerr << "Enountered dof '" << dof[i] << "' we don't know about in '" << dof << "'." << endl;
				assert(0);
 		}
 	}
	return trans;
}

Quatd get_dof_rot(string const &dof, double const *info, int start_pos) {
	Quatd ret;
	ret.clear();
	for (unsigned int i = 0; i != dof.size(); ++i) {
		double d = info[start_pos + i];
		switch (dof[i]) {
			case 'x':
				ret = multiply(rotation(d * M_PI / 180.0, make_vector(1.0,0.0,0.0)), ret);
				break;
			case 'y':
				ret = multiply(rotation(d * M_PI / 180.0, make_vector(0.0,1.0,0.0)), ret);
				break;
			case 'z':
				ret = multiply(rotation(d * M_PI / 180.0, make_vector(0.0,0.0,1.0)), ret);
 			case 'X':
 			case 'Y':
 			case 'Z':
				break;
 			case 'l':
				cerr << "Currently, we're lazy about length." << endl;
				break;
 			default:
				cerr << "Enountered dof '" << dof[i] << "' we don't know about in '" << dof << "'." << endl;
				assert(0);
 		}
 	}
	return normalize(ret);
}

}

//basically, just pull the 'ol root positions from each frame and
//extract the x,z, and yaw delta.
void Skeleton::build_delta(int frame_from, int frame_to, vector< double > const &positions, Character::StateDelta &delta) const {
	delta.clear();
	if (frame_from < 0 || frame_from * frame_size >= (signed)positions.size()
		|| frame_to < 0 || frame_to * frame_size >= (signed)positions.size() ) {
		return;
	}
	Quatd orientation_from, orientation_to;
	orientation_from = orientation_to = ordered_rotation(offset_order, axis_offset);
	Vector3d position_from, position_to;
	position_from = position_to = position;
	position_from += get_dof_trans(order, &(positions[0]), frame_size * frame_from);
	position_to += get_dof_trans(order, &(positions[0]), frame_size * frame_to);

	orientation_from = multiply(orientation_from, get_dof_rot(order, &(positions[0]), frame_size * frame_from));
	orientation_to = multiply(orientation_to, get_dof_rot(order, &(positions[0]), frame_size * frame_to));

	//position delta is given in the starting orientation frame.
	delta.position = rotate(position_to - position_from, conjugate(orientation_from));
	delta.orientation = get_yaw_angle(normalize(multiply(orientation_to, conjugate(orientation_from))));

}

void Skeleton::build_angles(int frame, vector< double > const &data, Character::Angles &angles) const {
	if (frame < 0 || (frame + 1) * frame_size > (signed)data.size()) {
		angles.angles.clear();
		angles.skeleton = NULL;
		assert(0);
		return;
	}
	angles.angles.clear();
	angles.angles.insert(angles.angles.end(), data.begin() + frame * frame_size, data.begin() + (frame + 1) * frame_size);
	angles.skeleton = this;
}

void Skeleton::build_pose(double const *frame_data, Character::Pose &pose) const {
	//clear out the destination pose.
	pose.clear(bones.size());

	//make sure we have the right amount of storage.
	assert(bones.size() == pose.bone_orientations.size());
	
	//also, make sure the pose knows its skeleton.
	pose.skeleton = this;

	pose.root_position = position + get_dof_trans(order, frame_data, 0);

	pose.root_orientation = multiply(ordered_rotation(offset_order, axis_offset), get_dof_rot(order, frame_data, 0));

	for (unsigned int b = 0; b < bones.size(); ++b) {
		Quatd rot;
		rot.clear();
		rot = multiply(conjugate(bones[b].global_to_local), rot);
		Quatd mul = get_dof_rot(bones[b].dof, frame_data, bones[b].frame_offset);
		rot = multiply(mul, rot);
		rot = multiply(bones[b].global_to_local, rot);
		rot = normalize(rot);

		pose.bone_orientations[b] = rot;
	}
}

int Skeleton::get_bone_by_name(string name) const {
	for (unsigned int b = 0; b < bones.size(); ++b) {
		if (name == bones[b].name) {
			return b;
		}
	}
	return -1;
}

} //namespace Library
