#include "Library.hpp"

#include "ReadSkeleton.hpp"

#include <list>
#include <vector>

//linux-specific:
#include <dirent.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fstream>

namespace Library {

using std::list;
using std::vector;
using std::string;
using std::sort;

using std::cout;

namespace {
	list< Skeleton > skeletons;
	list< Motion > motions;
}

unsigned int signature = 0;

void directory_recursion(string base_path) {
	string skeleton_path = "";
	vector< string > motion_paths;
	vector< string > dir_paths;
	DIR *dir = opendir(base_path.c_str());
	if (dir == NULL) {
		cerr << "Cannot open directory '" << base_path << "'" << endl;
	} else {
		struct dirent *ent;
		while ((ent = readdir(dir))) {
			string name = string(ent->d_name);
			if (name.size() > 4 && (name.substr(name.size()-4,4)==".asf"
						|| name.substr(name.size()-4,4)==".ASF")) {
				skeleton_path = base_path + "/" + name;
			} else if (name.size() > 4 && (name.substr(name.size()-4,4)==".amc"
						|| name.substr(name.size()-4,4)==".AMC")) {
				motion_paths.push_back(base_path + "/" + name);
			} else if (name.size() > 4 && (name.substr(name.size()-4,4)==".ann")) {
				// do nothing
			} else if (name[0]!='.') {
				dir_paths.push_back(base_path + "/" + name);
			}
		}
	}

	sort(motion_paths.begin(), motion_paths.end());

	// if no skeleton/motions in dir, that's cool, else read them in!
	if (skeleton_path == "" || motion_paths.size()==0) {
		cerr << "Either no skeleton or motions found in path '" << base_path << "'. Continuing." << endl;
	} else {
		// if skeleton is read successfully, then read motions.
		skeletons.push_back(Skeleton());
		if (!ReadSkeleton(skeleton_path, skeletons.back())) {
			cerr << "Error reading skeleton from " << skeleton_path << "." << endl;
			skeletons.pop_back();
		} else {
			cout << "Read " << skeleton_path << " (" << skeletons.back().bones.size() << " bones)" << endl;
			for (unsigned int i = 0; i < motion_paths.size(); ++i) {
				motions.push_back(Motion());
				motions.back().skeleton = &(skeletons.back());
				motions.back().filename = motion_paths[i];
				if (!ReadAnimation(motion_paths[i], *motions.back().skeleton, motions.back().data)) {
					cerr << "Error reading animation from " << motion_paths[i] << "." << endl;
					motions.pop_back();
				} else {
					cout << "Read " << motion_paths[i] << " (" << motions.back().frames() << " frames)" << endl;
					motions.back().annotations.clear();
					motions.back().annotations.resize(motions.back().frames(), 0);
					motions.back().load_annotations();
				}
			}
			cout << "Read " << motion_paths.size() << " motions in directory '" << base_path << "'." << endl;
		}
	}
	// recurse on directories in current dir
	for (unsigned int i = 0; i<dir_paths.size(); i++) {
		directory_recursion(dir_paths[i]);
	}
}

void init(string base_path) {
	skeletons.clear();
	motions.clear();

	directory_recursion(base_path);

	float length = 0.0;
	unsigned int frames = 0;
	for (typeof(motions.begin()) m = motions.begin(); m != motions.end(); ++m) {
		frames += m->frames();
		length += m->length();
	}
	cout << "Loaded " << length << " seconds of motion (" << frames << " frames)." << endl;


	cout << "Computing signature" << endl;
	long long sig = 0;
	for (typeof(motions.begin()) m = motions.begin(); m != motions.end(); ++m) {
		unsigned char * c = (unsigned char *)(&m->data[0]);
		unsigned char *end = (unsigned char *)(&m->data[m->data.size()-1]+1);
		while (c != end) {
			sig = (sig * 256) % 1610612741LL;
			sig = (sig + (long long)(*c)) % 1610612741LL;
			++c;
		}
	}
	signature = (unsigned int)(sig & 0xffffffff);
	cout << "Signature is " << signature << endl;
	return;
}

unsigned int motion_count() {
	return motions.size();
}

Motion const &motion(unsigned int index) {
	assert(index < motions.size());
	list< Motion >::const_iterator m = motions.begin();
	for (unsigned int i = 0; i < index; ++i) ++m;
	return *m;
}

Motion &motion_nonconst(unsigned int index) {
	assert(index < motions.size());
	list< Motion >::iterator m = motions.begin();
	for (unsigned int i = 0; i < index; ++i) ++m;
	return *m;
}

void Motion::get_delta(unsigned int frame_from, unsigned int frame_to, Character::StateDelta &into) const {
	skeleton->build_delta(frame_from, frame_to, data, into);
}

void Motion::get_state(unsigned int frame, Character::State &into) const {
	//ok, not doing this one for now.
	into.clear();
}

void Motion::get_angles(unsigned int frame, Character::Angles &into) const {
	//simple!
	assert(skeleton);
	skeleton->build_angles(frame, data, into);
}

void Motion::get_pose(unsigned int frame, Character::Pose &into) const {
	//simple!
	assert(skeleton);
	assert(frame < frames());
	assert((frame + 1) * skeleton->frame_size <= data.size());
	skeleton->build_pose(&(data[0]) + frame * skeleton->frame_size, into);
}

int Motion::get_annotation(unsigned int frame) const {
	assert(frame < frames());
	return annotations[frame];
}

void Motion::add_annotation(unsigned int frame, Annotation annotation) {
	assert(frame < frames());
	annotations[frame] |= annotation;
}

void Motion::clear_annotation(unsigned int frame, Annotation annotation) {
	assert(frame < frames());
	int a = annotation;
	annotations[frame] &= ~a;
}

bool Motion::save_annotations() const {
	string save_file = filename;
	save_file[save_file.size()-1] = 'n';
	save_file[save_file.size()-2] = 'n';
	save_file[save_file.size()-3] = 'a';
	std::ofstream annFile(save_file.c_str());
	if ( ! annFile ) return false;
	for (unsigned int f = 0; f < frames(); ++f) {
		annFile << f << " " << annotations[f] << endl;
	}
	annFile.close();
	return true;
}

bool Motion::load_annotations() {
	string load_file = filename;
	load_file[load_file.size()-1] = 'n';
	load_file[load_file.size()-2] = 'n';
	load_file[load_file.size()-3] = 'a';
	std::ifstream annFile(load_file.c_str());
	if ( ! annFile ) {
		cerr << "No such annotation file: " << load_file << "." << std::endl;
		return false;
	}
	unsigned int frame_num;
	for (unsigned int f = 0; f < frames(); ++f) {
		annFile >> frame_num;
		if (frame_num != f) {
			cerr << "Annotation frame mismatch in " << load_file << std::endl;
			return false;
		}
		annFile >> annotations[f];
	}
	annFile.close();
	return true;
}

unsigned int Motion::frames() const {
	assert(skeleton);
	return data.size() / skeleton->frame_size;
}

float Motion::length() const {
	assert(skeleton);
	return frames() * skeleton->timestep;
}

} //namespace Library
