#!/usr/bin/perl -w

#
# Usage: ./gen_extension_loader.pl <path/to/glext.h>
#
# Dumps the code to load (a hell of a lot) of gl extensions.
#

if (@ARGV != 1) {
	print "Yo, give me the path to glext.h\n";
	exit(1);
}

open GLEXT, "<", $ARGV[0];

@ext_names = ();
$current_ext_name = "";
$current_ext_short = "";
%functions = ();

while ($line = <GLEXT>) {
	chomp $line;
	if ($line =~ /^#ifndef GL_VERSION_/) {
		$current_ext_name = "NO";
	} elsif ($line =~ /^#ifndef (GL_([A-Z]+)_[a-z_]+)/) {
		$current_ext_name = $1;
		$current_ext_short = $2;
		push @ext_names, $current_ext_name;
	}
	if ($current_ext_name ne "NO") {
		if ($line =~ /^GLAPI ([^\s]+) APIENTRY ([a-zA-Z0-9]+)/) {
			$current_ext_name ne "" or die "Function outside of extension: $1 $2";
			$name = $2;
			$mangled_name = "PFN" . (uc $name) . "PROC";
			$functions{$current_ext_name}{$mangled_name}{'type'} = $1;
			$functions{$current_ext_name}{$mangled_name}{'name'} = $name;
		}
		if ($line =~ /^typedef ([^\s]+) \(APIENTRYP ([A-Z0-9]+)/) {
			defined $functions{$current_ext_name}{$2} or die "Couldn't locate $1 $2";
			$functions{$current_ext_name}{$2}{'type'} eq $1 or die "Types don't match for $2: $1, ${functions{$current_ext_name}{$2}{'type'}}";
			$functions{$current_ext_name}{$2}{'ok'} = 1;
		}
	}
}

close GLEXT;

print "Writing header...\n";

open HPP, ">", "GLExtensions.hpp";

print HPP <<EOF;
//This header is all about getting your extensions properly managed.
//It is automatically generated. Don't be editing!

#ifdef MACOSX
#include <OpenGL/gl.h>
#include <OpenGL/glext.h>
#else
#ifdef WINDOWS
#include <windows.h>
#endif //WINDOWS
#include <GL/gl.h>
#ifdef WINDOWS
#include "glext.h"
#else
#include <GL/glext.h>
#endif
#endif
#include <string>
#include <set>

using std::string;
using std::set;

class GLExtensions {
public:
	static bool request(string extension_name);
	static void reload();
private:
	static set< string > &get_set();
};

#ifndef MACOSX

//Now all the function pointers you'll be needing...
EOF

foreach my $extension (sort keys %functions) {
	print HPP <<EOF;

//  ...for $extension:
EOF
	foreach my $func (sort keys %{$functions{$extension}}) {
		$functions{$extension}{$func}{'ok'} or die "$func isn't marked as ok.";
		print HPP "extern " . $func . " " . $functions{$extension}{$func}{'name'} . ";\n";
	}
}

print HPP <<EOF;
#endif //MACOSX
EOF

close HPP;

print "Writing source...\n";

open CPP, ">", "GLExtensions.cpp";

print CPP <<EOF;
#include "GLExtensions.hpp"
#include <iostream>
#ifndef MACOSX
#ifdef WINDOWS
//windows use wgl, methinks.
#include <wingdi.h>
#else
//linux use glx
#include <GL/glx.h>
#endif
#endif

using std::cerr;
using std::endl;

//GLExtensions::request is described below, since it is auto-generated.

void GLExtensions::reload() {
	set< string > old_set = get_set();
	get_set().clear();
	for (set< string >::iterator s = old_set.begin(); s != old_set.end(); ++s) {
		if (!GLExtensions::request(*s)) {
			cerr << "Can't reload extension '" << *s << "'. Bailing." << endl;
		}
	}
}

set< string > &GLExtensions::get_set() {
	static set< string > my_set;
	return my_set;
}

typedef void (*VoidFuncPtr)();

#ifndef MACOSX
#ifdef WINDOWS
inline VoidFuncPtr get_proc(string name) {
	return (VoidFuncPtr)wglGetProcAddress( (LPCSTR)name.c_str() );
}
#else
inline VoidFuncPtr get_proc(string name) {
	return (VoidFuncPtr)glXGetProcAddressARB( (const GLubyte *)name.c_str() );
}
#endif //WINDOWS
#endif

bool GLExtensions::request(string extension) {
#ifndef MACOSX
	bool loaded = false;
	//Load proper pointers...
EOF

$do_else = "";

foreach my $extension (sort keys %functions) {
	print CPP <<EOF;

	//  ...for $extension:
	${do_else}if (extension == "$extension") {
		loaded = true
EOF
	foreach my $func (sort keys %{$functions{$extension}}) {
		$name = $functions{$extension}{$func}{'name'};
		print CPP <<EOF;
			&& ($name = ($func)get_proc("$name"))
EOF
	}
	print CPP <<EOF;
			;
	}
EOF
	$do_else = "else ";
}

print CPP <<EOF;
	else {
		cerr << "Extension name '" << extension << "' not found." << endl;
		return false;
	}
	if (loaded) {
		get_set().insert(extension);
		return true;
	} else {
		cerr << "Extension name '" << extension << "' not loaded." << endl;
		return false;
	}
#else
	return true;
#endif
}

#ifndef MACOSX

//Actually define the function pointers...
EOF

foreach my $extension (sort keys %functions) {
	print CPP <<EOF;

//  ...for $extension:
EOF
	foreach my $func (sort keys %{$functions{$extension}}) {
		$functions{$extension}{$func}{'ok'} or die "$func isn't marked as ok.";
		print CPP $func . " " . $functions{$extension}{$func}{'name'} . ";\n";
	}
}

print CPP <<EOF;
#endif //MACOSX
EOF

close CPP;

