#include "BrowseMode.hpp"

#include <Character/Draw.hpp>
#include <Character/pose_utils.hpp>
#include <Library/Library.hpp>

#include <Graphics/Graphics.hpp>
#include <Graphics/Font.hpp>
#include <Vector/VectorGL.hpp>

#include <string>
#include <sstream>
#include <iostream>
#include <fstream>

using std::string;

using std::ostringstream;

using std::cout;
using std::cerr;
using std::endl;

using std::ofstream;

BrowseMode::BrowseMode() {
	camera = make_vector(10.0f, 10.0f, 10.0f);
	target = make_vector(0.0f, 0.0f, 0.0f);
	track = false;
	current_pose.clear();
	converted_pose.clear();
	current_motion = 0;
	time = 0.0f;
	play_speed = 1.0f;
}

BrowseMode::~BrowseMode() {
}

void BrowseMode::update(float const elapsed_time) {
	assert(current_motion < Library::motion_count());
	Library::Motion const &motion = Library::motion(current_motion);
	if (motion.frames() == 0) {
		cerr << "Motion from " << motion.filename << " has zero frames." << endl;
		return;
	}
	time = fmodf(elapsed_time * play_speed + time, motion.length());
	unsigned int frame = (unsigned int)(time / motion.skeleton->timestep);
	if (frame >= motion.frames()) frame = motion.frames() - 1;

	Character::Angles current_angles;
	motion.get_angles(frame, current_angles);
	current_angles.to_pose(current_pose);

	static Library::Skeleton transformer;
	//transformer = *motion.skeleton;
	// copy skeleton by hand
	transformer.bones = motion.skeleton->bones;
	//transformer.order = motion.skeleton->order;
	transformer.position = motion.skeleton->position;
	transformer.offset_order = motion.skeleton->offset_order;
	transformer.axis_offset = motion.skeleton->axis_offset;
	transformer.timestep = motion.skeleton->timestep;
	transformer.ang_is_deg = motion.skeleton->ang_is_deg;
	transformer.rot_is_glob = false;
	transformer.z_is_up = false;
	transformer.frame_size = motion.skeleton->frame_size;
	transformer.filename = "modification_of_" + motion.skeleton->filename;
	transformer.order = "xyzXYZ";

	for (unsigned int b = 0; b < transformer.bones.size(); ++b) {
		transformer.bones[b].dof = "xyz";
		transformer.bones[b].frame_offset = 3*b + 6;

		// change the hinge joint
		/*if (transformer.bones[b].euler_axes.size()==1) {
				transformer.bones[b].dof = "y";
		}*/
		/*if (transformer.bones[b].euler_axes.size()==1) {
			cout << "Bone: " << transformer.bones[b].name << endl;
			cout << "Axis: " << transformer.bones[b].euler_axes[0] << endl;
			Vector3d axis = transformer.bones[b].euler_axes[0];
			Vector3d cross = cross_product(make_vector(1.0, 0.0, 0.0), axis);
			cross = normalize(cross);
			Vector3d perp = cross_product(make_vector(1.0, 0.0, 0.0), cross);
			float costheta = axis * make_vector(1.0, 0.0, 0.0);
			float sintheta = axis * perp;
			transformer.bones[b].global_to_local = rotation(atan2(sintheta, costheta), cross);
			cout << "Global to local: " << transformer.bones[b].global_to_local << endl;
			cout << "Rotated axes1: " << rotate(make_vector(1.0,0.0,0.0), transformer.bones[b].global_to_local) << endl;
			cout << "Rotated axes2: " << rotate(make_vector(0.0,1.0,0.0), transformer.bones[b].global_to_local) << endl;
			cout << "Rotated axes3: " << rotate(make_vector(0.0,0.0,1.0), transformer.bones[b].global_to_local) << endl;
		}*/

	}
	transformer.frame_size = 3*transformer.bones.size() + 6;


	/*
	// for code
	pose.skeleton = &transformer;
	pose.to_angles(angles);
	angles.to_pose(pose);
	*/ 

	vector< double > d(transformer.frame_size);
	transformer.get_angles(current_pose, &d[0]); // takes quaternion tree from pose, sets d based on bone dof format (Euler angles) // called by pose.to_angles
	transformer.build_pose(&d[0], converted_pose); // takes data in format specific to skeleton, converts to quaternion tree // called by angles.to_pose

	/*for (unsigned int b = 0; b < transformer.bones.size(); ++b) {
		if ((1+transformer.bones[b].euler_axes.size()) & 2) {
			unsigned int count = 0;
			for (unsigned int i = 0; i < 3; ++i) {
				if (fabs(d[transformer.bones[b].frame_offset + i]) > 2.0f ) {
					++count;
				}
			}
			if (count != transformer.bones[b].euler_axes.size()) {
				cout << transformer.bones[b].name << "\t"
					<< transformer.bones[b].euler_axes.size() << " ";
				cout << d[transformer.bones[b].frame_offset+0] << " "
					<< d[transformer.bones[b].frame_offset+1] << " "
				     << d[transformer.bones[b].frame_offset+2] << endl;
			}
		}
	}*/

	if (track) {
		Vector3f delta = current_pose.root_position - target;
		target += delta;
		camera += delta;
	}
}

void BrowseMode::handle_event(SDL_Event const &event) {
	if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_TAB) {
		track = !track;
	}
	if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_ESCAPE) {
		quit_flag = true;
	}
	if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_PAGEUP) {
		current_motion += 1;
		if (current_motion >= Library::motion_count()) {
			current_motion = 0;
		}
		time = 0.0f;
	}
	if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_d) {
		for (unsigned int m = 0; m < Library::motion_count(); ++m) {
			Library::Motion const &motion = Library::motion(m);
			ofstream out((motion.filename + ".global").c_str());
			out << "\"position.x\", \"position.z\", \"position.yaw\", \"root.x\", \"root.y\", \"root.z\"";
			for (unsigned int b = 0; b < motion.skeleton->bones.size(); ++b) {
				out << ", ";
				out << '"' << motion.skeleton->bones[b].name << ".x" << '"';
				out << ", ";
				out << '"' << motion.skeleton->bones[b].name << ".y" << '"';
				out << ", ";
				out << '"' << motion.skeleton->bones[b].name << ".z" << '"';
			}
			out << endl;
			for (unsigned int f = 0; f < motion.frames(); ++f) {
				Character::WorldBones w;
				Character::Pose p;
				motion.get_local_pose(f, p);
				Character::get_world_bones(p, w);
				{
					Character::StateDelta delta;
					motion.get_delta(0, f, delta);
					out << delta.position.x << ", " << delta.position.z << ", " << delta.orientation << ", " << p.root_position.x << ", " << p.root_position.y << ", " << p.root_position.z;
				}
				for (unsigned int b = 0; b < w.tips.size(); ++b) {
					for (unsigned int c = 0; c < 3; ++c) {
						out << ", ";
						out << w.tips[b].c[c];
					}
				}
				out << endl;
			}
		}
	}
	if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_PAGEDOWN) {
		current_motion -= 1;
		if (current_motion >= Library::motion_count()) {
			current_motion = Library::motion_count() - 1;
		}
		time = 0.0f;
	}
	if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_SPACE) {
		if (play_speed != 0.0f) play_speed = 0.0f;
		else play_speed = 1.0f;
	}
	//rotate view:
	if (event.type == SDL_MOUSEMOTION && (event.motion.state & SDL_BUTTON(SDL_BUTTON_LEFT))) {
		Vector3f vec = camera - target;
		float len = length(vec);
		float theta_yaw = atan2(vec.z, vec.x);
		float theta_pitch = atan2(vec.y, sqrt(vec.x * vec.x + vec.z * vec.z));
		theta_yaw += event.motion.xrel / float(Graphics::screen_x) / len * 100;
		theta_pitch += event.motion.yrel / float(Graphics::screen_x) / len * 100;
		if (theta_pitch > 0.4 * M_PI) theta_pitch = 0.4 * M_PI;
		if (theta_pitch <-0.4 * M_PI) theta_pitch =-0.4 * M_PI;

		camera = make_vector(cosf(theta_yaw)*cosf(theta_pitch),sinf(theta_pitch),sinf(theta_yaw)*cosf(theta_pitch)) * len + target;
	}
	//pan view:
	if (event.type == SDL_MOUSEMOTION && (event.motion.state & SDL_BUTTON(SDL_BUTTON_MIDDLE))) {
		Vector3f to = normalize(camera - target);
		Vector3f right = normalize(cross_product(to, make_vector< float >(0,1,0)));
		Vector3f up = -cross_product(to, right);
		float len = length(camera - target);
		camera += right * event.motion.xrel / float(Graphics::screen_x) * len;
		camera += up * event.motion.yrel / float(Graphics::screen_x) * len;
		target += right * event.motion.xrel / float(Graphics::screen_x) * len;
		target += up * event.motion.yrel / float(Graphics::screen_x) * len;
	}
	//zoom view:
	if (event.type == SDL_MOUSEMOTION && (event.motion.state & SDL_BUTTON(SDL_BUTTON_RIGHT))) {
		Vector3f vec = camera - target;
		float len = length(vec);
		len *= pow(2, event.motion.yrel / float(Graphics::screen_x) * 10);
		if (len < 1) len = 1;
		if (len > 100) len = 100;
		camera = normalize(camera - target) * len + target;
	}
}

void BrowseMode::draw() {
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();

	gluPerspective(60.0, Graphics::aspect(), 0.1, 100.0);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	{
		float y = camera.y;
		if (y < 0.1f) y = 0.1f;
		gluLookAt(camera.x, y, camera.z, target.x, target.y, target.z, 0, 1, 0);
	}


	for (unsigned int pass = 0; pass <= 2; ++pass) {
		static Vector4f l = make_vector(0.0f, 10.0f, 0.0f, 1.0f);
		static Vector4f amb = make_vector(0.1f, 0.1f, 0.1f, 1.0f);
		static Vector4f dif = make_vector(1.0f, 1.0f, 1.0f, 1.0f);
		static Vector4f zero = make_vector(0.0f, 0.0f, 0.0f, 0.0f);
		glLightfv(GL_LIGHT0, GL_AMBIENT, amb.c);
		glLightfv(GL_LIGHT0, GL_DIFFUSE, dif.c);
		glLightfv(GL_LIGHT0, GL_POSITION, l.c);
		if (pass == 0) { //reflections.
			glEnable(GL_DEPTH_TEST);
			glEnable(GL_LIGHTING);
			glEnable(GL_COLOR_MATERIAL);
			glEnable(GL_LIGHT0);
			l.y *= -1;
			glLightfv(GL_LIGHT0, GL_POSITION, l.c);
			l.y *= -1;
			glPushMatrix();
			glScaled(1.0f,-1.0f, 1.0f); //flip over the floor.
		} else if (pass == 1) {
			//shadows.
			glDisable(GL_DEPTH_TEST);
			glEnable(GL_STENCIL_TEST);
			glStencilFunc(GL_ALWAYS, 1, 0xff);
			glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
			glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
			glPushMatrix();
			double mat[16] = {
				l.y,	0,	0,	0, //x multiplies this
				0,	0,	0,	-1, //y multiplies this
				0,	0,	l.y,	0, //z multiplies this
				-l.x*l.y,	0,	-l.z*l.y,	l.y
			};
			glMultMatrixd(mat);
		} else if (pass == 2) {
			glEnable(GL_DEPTH_TEST);
			glEnable(GL_LIGHTING);
		}
		//state can be used to translate and rotate the character
		//to an arbitrary position, but we don't need that.
		Character::State null;
		null.clear();
		Character::draw(current_pose, null, pass == 2, pass != 1);
		glTranslatef(5.0f,0.0f,0.0f);
		Character::draw(converted_pose, null, pass == 2, pass != 1);
		glTranslatef(-5.0f,0.0f,0.0f);
		if (pass == 0) { //cleanup post-reflections.
			glPopMatrix();
			glDisable(GL_LIGHTING);

			//overwrite anything sticking through the floor:
			glDepthFunc(GL_GEQUAL);
			glBegin(GL_QUADS);
			glColor3f(0,0,0);
			glVertex3f(-10,0,-10);
			glVertex3f(-10,0, 10);
			glVertex3f( 10,0, 10);
			glVertex3f( 10,0,-10);
			glEnd();
			glDepthFunc(GL_LESS);

			glDisable(GL_DEPTH_TEST);

		} else if (pass == 1) {
			glPopMatrix();

			//Now to deal with shadows.
			glEnable(GL_STENCIL_TEST);
			glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
			glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);

			//draw the floor:
			glEnable(GL_LIGHTING);
			glEnable(GL_BLEND);
			glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
			for (int floor_pass = 0; floor_pass < 2; ++floor_pass) {
				if (floor_pass == 0) {
					glLightfv(GL_LIGHT0, GL_AMBIENT, amb.c);
					glLightfv(GL_LIGHT0, GL_DIFFUSE, zero.c);
					glStencilFunc(GL_EQUAL, 1, 0xff);
				} else if (floor_pass == 1) {
					glLightfv(GL_LIGHT0, GL_AMBIENT, amb.c);
					glLightfv(GL_LIGHT0, GL_DIFFUSE, dif.c);
					glStencilFunc(GL_NOTEQUAL, 1, 0xff);
				}
				glNormal3f(0,1,0);
				glBegin(GL_QUADS);
				for (int x = -10; x < 10; ++x) {
					for (int z = -10; z <= 10; ++z) {
						if ((x & 1) ^ (z & 1)) {
							glColor4f(0.5, 0.5, 0.5, 0.1);
						} else {
							glColor4f(0.9, 0.9, 0.9, 0.3);
						}
						glVertex3f(x,0,z);
						glVertex3f(x+1,0,z);
						glVertex3f(x+1,0,z+1);
						glVertex3f(x,0,z+1);
					}
				}
				glEnd();
			}
			glDisable(GL_BLEND);
			glDisable(GL_LIGHTING);
			glDisable(GL_STENCIL_TEST);
		} else if (pass == 2) {
			glDisable(GL_LIGHTING);
			glDisable(GL_DEPTH_TEST);
		}
	}
	glDisable(GL_DEPTH_TEST);

	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	//--- now some info overlay ---
	glPushMatrix();
	glLoadIdentity();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glScaled(1.0 / Graphics::aspect(), 1.0, 1.0);
	glMatrixMode(GL_MODELVIEW);

	{
		Graphics::FontRef gentium = Graphics::get_font("gentium.txf");
		ostringstream info1, info2, info3, info4;
		Library::Motion const &motion = Library::motion(current_motion);
		info1 << "Motion: " << motion.filename;
		info2 << "Skeleton: " << motion.skeleton->filename;
		info3 << "Frame " << (unsigned int)(time / motion.skeleton->timestep) << " of " << motion.frames() << " (" << 1.0f / motion.skeleton->timestep << " fps)";
		info4 << play_speed << "x speed";
		glEnable(GL_BLEND);
		glEnable(GL_TEXTURE_2D);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glColor3f(1.0f, 1.0f, 1.0f);
		Vector2f pos = make_vector(-(float)Graphics::aspect(), 1.0f);
		const float height = 0.07f;
		pos.y -= height;
		gentium.ref->draw(info1.str(), pos, height);
		pos.y -= height;
		gentium.ref->draw(info2.str(), pos, height);
		pos.y -= height;
		gentium.ref->draw(info3.str(), pos, height);
		pos.y -= height;
		gentium.ref->draw(info4.str(), pos, height);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_BLEND);
	}

	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);

	Graphics::gl_errors("BrowseMode::draw");
}
